/// <reference types="node" />
import { EventEmitter } from "events";
import { IStorageProvider } from "./storage/IStorageProvider";
import { IJoinRoomStrategy } from "./strategies/JoinRoomStrategy";
import { UnstableApis } from "./UnstableApis";
import { IPreprocessor } from "./preprocessors/IPreprocessor";
import { MatrixPresence } from "./models/MatrixPresence";
import { Metrics } from "./metrics/Metrics";
import { AdminApis } from "./AdminApis";
/**
 * A client that is capable of interacting with a matrix homeserver.
 */
export declare class MatrixClient extends EventEmitter {
    readonly homeserverUrl: string;
    readonly accessToken: string;
    private storage;
    /**
     * The metrics instance for this client.
     */
    readonly metrics: Metrics;
    /**
     * The presence status to use while syncing. The valid values are "online" to set the account as online,
     * "offline" to set the user as offline, "unavailable" for marking the user away, and null for not setting
     * an explicit presence (the default).
     *
     * Has no effect if the client is not syncing. Does not apply until the next sync request.
     */
    syncingPresence: "online" | "offline" | "unavailable" | null;
    /**
     * The number of milliseconds to wait for new events for on the next sync.
     *
     * Has no effect if the client is not syncing. Does not apply until the next sync request.
     */
    syncingTimeout: number;
    private userId;
    private requestId;
    private filterId;
    private stopSyncing;
    private lastJoinedRoomIds;
    private impersonatedUserId;
    private joinStrategy;
    private eventProcessors;
    /**
     * Creates a new matrix client
     * @param {string} homeserverUrl The homeserver's client-server API URL
     * @param {string} accessToken The access token for the homeserver
     * @param {IStorageProvider} storage The storage provider to use. Defaults to MemoryStorageProvider.
     */
    constructor(homeserverUrl: string, accessToken: string, storage?: IStorageProvider);
    /**
     * Gets the unstable API access class. This is generally not recommended to be
     * used by clients.
     * @return {UnstableApis} The unstable API access class.
     */
    readonly unstableApis: UnstableApis;
    /**
     * Gets the admin API access class.
     * @return {AdminApis} The admin API access class.
     */
    readonly adminApis: AdminApis;
    /**
     * Sets a user ID to impersonate as. This will assume that the access token for this client
     * is for an application service, and that the userId given is within the reach of the
     * application service. Setting this to null will stop future impersonation. The user ID is
     * assumed to already be valid
     * @param {string} userId The user ID to masquerade as
     */
    impersonateUserId(userId: string): void;
    /**
     * Sets the strategy to use for when joinRoom is called on this client
     * @param {IJoinRoomStrategy} strategy The strategy to use, or null to use none
     */
    setJoinStrategy(strategy: IJoinRoomStrategy): void;
    /**
     * Adds a preprocessor to the event pipeline. When this client encounters an event, it
     * will try to run it through the preprocessors it can in the order they were added.
     * @param {IPreprocessor} preprocessor the preprocessor to add
     */
    addPreprocessor(preprocessor: IPreprocessor): void;
    private processEvent;
    /**
     * Retrieves content from account data.
     * @param {string} eventType The type of account data to retrieve.
     * @returns {Promise<*>} Resolves to the content of that account data.
     */
    getAccountData(eventType: string): Promise<any>;
    /**
     * Retrieves content from room account data.
     * @param {string} eventType The type of room account data to retrieve.
     * @param {string} roomId The room to read the account data from
     * @returns {Promise<*>} Resolves to the content of that account data.
     */
    getRoomAccountData(eventType: string, roomId: string): Promise<any>;
    /**
     * Sets account data.
     * @param {string} eventType The type of account data to set
     * @param {*} content The content to set
     * @returns {Promise<*>} Resolves when updated
     */
    setAccountData(eventType: string, content: any): Promise<any>;
    /**
     * Sets room account data.
     * @param {string} eventType The type of room account data to set
     * @param {string} roomId The room to set account data in
     * @param {*} content The content to set
     * @returns {Promise<*>} Resolves when updated
     */
    setRoomAccountData(eventType: string, roomId: string, content: any): Promise<any>;
    /**
     * Gets the presence information for the current user.
     * @returns {Promise<MatrixPresence>} Resolves to the presence status of the user.
     */
    getPresenceStatus(): Promise<MatrixPresence>;
    /**
     * Gets the presence information for a given user.
     * @param {string} userId The user ID to look up the presence of.
     * @returns {Promise<MatrixPresence>} Resolves to the presence status of the user.
     */
    getPresenceStatusFor(userId: string): Promise<MatrixPresence>;
    /**
     * Sets the presence status for the current user.
     * @param {"online"|"offline"|"unavailable"} presence The new presence state for the user.
     * @param {string} statusMessage Optional status message to include with the presence.
     * @returns {Promise<*>} Resolves when complete.
     */
    setPresenceStatus(presence: "online" | "offline" | "unavailable", statusMessage?: string): Promise<any>;
    /**
     * Adds a new room alias to the room directory
     * @param {string} alias The alias to add (eg: "#my-room:matrix.org")
     * @param {string} roomId The room ID to add the alias to
     * @returns {Promise} resolves when the alias has been added
     */
    createRoomAlias(alias: string, roomId: string): Promise<any>;
    /**
     * Removes a room alias from the room directory
     * @param {string} alias The alias to remove
     * @returns {Promise} resolves when the alias has been deleted
     */
    deleteRoomAlias(alias: string): Promise<any>;
    /**
     * Sets the visibility of a room in the directory.
     * @param {string} roomId The room ID to manipulate the visibility of
     * @param {"public" | "private"} visibility The visibility to set for the room
     * @return {Promise} resolves when the visibility has been updated
     */
    setDirectoryVisibility(roomId: string, visibility: "public" | "private"): Promise<any>;
    /**
     * Gets the visibility of a room in the directory.
     * @param {string} roomId The room ID to query the visibility of
     * @return {Promise<"public"|"private">} The visibility of the room
     */
    getDirectoryVisibility(roomId: string): Promise<"public" | "private">;
    /**
     * Resolves a room ID or alias to a room ID. If the given ID or alias looks like a room ID
     * already, it will be returned as-is. If the room ID or alias looks like a room alias, it
     * will be resolved to a room ID if possible. If the room ID or alias is neither, an error
     * will be raised.
     * @param {string} roomIdOrAlias the room ID or alias to resolve to a room ID
     * @returns {Promise<string>} resolves to the room ID
     */
    resolveRoom(roomIdOrAlias: string): Promise<string>;
    /**
     * Does a room directory lookup for a given room alias
     * @param {string} roomAlias the room alias to look up in the room directory
     * @returns {Promise<RoomDirectoryLookupResponse>} resolves to the room's information
     */
    lookupRoomAlias(roomAlias: string): Promise<RoomDirectoryLookupResponse>;
    /**
     * Invites a user to a room.
     * @param {string} userId the user ID to invite
     * @param {string} roomId the room ID to invite the user to
     * @returns {Promise<*>} resolves when completed
     */
    inviteUser(userId: any, roomId: any): Promise<any>;
    /**
     * Kicks a user from a room.
     * @param {string} userId the user ID to kick
     * @param {string} roomId the room ID to kick the user in
     * @param {string?} reason optional reason for the kick
     * @returns {Promise<*>} resolves when completed
     */
    kickUser(userId: any, roomId: any, reason?: any): Promise<any>;
    /**
     * Bans a user from a room.
     * @param {string} userId the user ID to ban
     * @param {string} roomId the room ID to set the ban in
     * @param {string?} reason optional reason for the ban
     * @returns {Promise<*>} resolves when completed
     */
    banUser(userId: any, roomId: any, reason?: any): Promise<any>;
    /**
     * Unbans a user in a room.
     * @param {string} userId the user ID to unban
     * @param {string} roomId the room ID to lift the ban in
     * @returns {Promise<*>} resolves when completed
     */
    unbanUser(userId: any, roomId: any): Promise<any>;
    /**
     * Gets the current user ID for this client
     * @returns {Promise<string>} The user ID of this client
     */
    getUserId(): Promise<string>;
    /**
     * Stops the client from syncing.
     */
    stop(): void;
    /**
     * Starts syncing the client with an optional filter
     * @param {*} filter The filter to use, or null for none
     * @returns {Promise<*>} Resolves when the client has started syncing
     */
    start(filter?: any): Promise<any>;
    private startSync;
    private doSync;
    private processSync;
    /**
     * Gets an event for a room. Returned as a raw event.
     * @param {string} roomId the room ID to get the event in
     * @param {string} eventId the event ID to look up
     * @returns {Promise<*>} resolves to the found event
     */
    getEvent(roomId: string, eventId: string): Promise<any>;
    /**
     * Gets the room state for the given room. Returned as raw events.
     * @param {string} roomId the room ID to get state for
     * @returns {Promise<*[]>} resolves to the room's state
     */
    getRoomState(roomId: string): Promise<any[]>;
    /**
     * Gets the state events for a given room of a given type under the given state key.
     * @param {string} roomId the room ID
     * @param {string} type the event type
     * @param {String} stateKey the state key, falsey if not needed
     * @returns {Promise<*|*[]>} resolves to the state event(s)
     * @deprecated It is not possible to get an array of events - use getRoomStateEvent instead
     */
    getRoomStateEvents(roomId: any, type: any, stateKey: any): Promise<any | any[]>;
    /**
     * Gets a state event for a given room of a given type under the given state key.
     * @param {string} roomId the room ID
     * @param {string} type the event type
     * @param {String} stateKey the state key
     * @returns {Promise<*>} resolves to the state event
     */
    getRoomStateEvent(roomId: any, type: any, stateKey: any): Promise<any>;
    /**
     * Gets the profile for a given user
     * @param {string} userId the user ID to lookup
     * @returns {Promise<*>} the profile of the user
     */
    getUserProfile(userId: string): Promise<any>;
    /**
     * Sets a new display name for the user.
     * @param {string} displayName the new display name for the user, or null to clear
     * @returns {Promise<*>} resolves when complete
     */
    setDisplayName(displayName: string): Promise<any>;
    /**
     * Sets a new avatar url for the user.
     * @param {string} avatarUrl the new avatar URL for the user, in the form of a Matrix Content URI
     * @returns {Promise<*>} resolves when complete
     */
    setAvatarUrl(avatarUrl: string): Promise<any>;
    /**
     * Joins the given room
     * @param {string} roomIdOrAlias the room ID or alias to join
     * @param {string[]} viaServers the server names to try and join through
     * @returns {Promise<string>} resolves to the joined room ID
     */
    joinRoom(roomIdOrAlias: string, viaServers?: string[]): Promise<string>;
    /**
     * Gets a list of joined room IDs
     * @returns {Promise<string[]>} resolves to a list of room IDs the client participates in
     */
    getJoinedRooms(): Promise<string[]>;
    /**
     * Gets the joined members in a room. The client must be in the room to make this request.
     * @param {string} roomId The room ID to get the joined members of.
     * @returns {Promise<string>} The joined user IDs in the room
     */
    getJoinedRoomMembers(roomId: string): Promise<string[]>;
    /**
     * Leaves the given room
     * @param {string} roomId the room ID to leave
     * @returns {Promise<*>} resolves when left
     */
    leaveRoom(roomId: string): Promise<any>;
    /**
     * Sends a read receipt for an event in a room
     * @param {string} roomId the room ID to send the receipt to
     * @param {string} eventId the event ID to set the receipt at
     * @returns {Promise<*>} resolves when the receipt has been sent
     */
    sendReadReceipt(roomId: string, eventId: string): Promise<any>;
    /**
     * Sets the typing status of the current user in a room
     * @param {string} roomId the room ID the user is typing in
     * @param {boolean} typing is the user currently typing
     * @param {number} timeout how long should the server preserve the typing state, in milliseconds
     * @returns {Promise<*>} resolves when the typing state has been set
     */
    setTyping(roomId: string, typing: boolean, timeout?: number): Promise<any>;
    /**
     * Replies to a given event with the given text. The event is sent with a msgtype of m.text.
     * @param {string} roomId the room ID to reply in
     * @param {*} event the event to reply to
     * @param {string} text the text to reply with
     * @param {string} html the HTML to reply with, or falsey to use the `text`
     * @returns {Promise<string>} resolves to the event ID which was sent
     */
    replyText(roomId: string, event: any, text: string, html?: string): Promise<string>;
    /**
     * Replies to a given event with the given text. The event is sent with a msgtype of m.notice.
     * @param {string} roomId the room ID to reply in
     * @param {*} event the event to reply to
     * @param {string} text the text to reply with
     * @param {string} html the HTML to reply with, or falsey to use the `text`
     * @returns {Promise<string>} resolves to the event ID which was sent
     */
    replyNotice(roomId: string, event: any, text: string, html?: string): Promise<string>;
    /**
     * Sends a notice to the given room
     * @param {string} roomId the room ID to send the notice to
     * @param {string} text the text to send
     * @returns {Promise<string>} resolves to the event ID that represents the message
     */
    sendNotice(roomId: string, text: string): Promise<string>;
    /**
     * Sends a text message to the given room
     * @param {string} roomId the room ID to send the text to
     * @param {string} text the text to send
     * @returns {Promise<string>} resolves to the event ID that represents the message
     */
    sendText(roomId: string, text: string): Promise<string>;
    /**
     * Sends a message to the given room
     * @param {string} roomId the room ID to send the message to
     * @param {object} content the event content to send
     * @returns {Promise<string>} resolves to the event ID that represents the message
     */
    sendMessage(roomId: string, content: any): Promise<string>;
    /**
     * Sends an event to the given room
     * @param {string} roomId the room ID to send the event to
     * @param {string} eventType the type of event to send
     * @param {string} content the event body to send
     * @returns {Promise<string>} resolves to the event ID that represents the event
     */
    sendEvent(roomId: string, eventType: string, content: any): Promise<string>;
    /**
     * Sends a state event to the given room
     * @param {string} roomId the room ID to send the event to
     * @param {string} type the event type to send
     * @param {string} stateKey the state key to send, should not be null
     * @param {string} content the event body to send
     * @returns {Promise<string>} resolves to the event ID that represents the message
     */
    sendStateEvent(roomId: string, type: string, stateKey: string, content: any): Promise<string>;
    /**
     * Redact an event in a given room
     * @param {string} roomId the room ID to send the redaction to
     * @param {string} eventId the event ID to redact
     * @param {String} reason an optional reason for redacting the event
     * @returns {Promise<string>} resolves to the event ID that represents the redaction
     */
    redactEvent(roomId: string, eventId: string, reason?: string | null): Promise<string>;
    /**
     * Creates a room. This does not break out the various options for creating a room
     * due to the large number of possibilities. See the /createRoom endpoint in the
     * spec for more information on what to provide for `properties`.
     * @param {*} properties the properties of the room. See the spec for more information
     * @returns {Promise<string>} resolves to the room ID that represents the room
     */
    createRoom(properties?: any): Promise<string>;
    /**
     * Checks if a given user has a required power level
     * @param {string} userId the user ID to check the power level of
     * @param {string} roomId the room ID to check the power level in
     * @param {string} eventType the event type to look for in the `events` property of the power levels
     * @param {boolean} isState true to indicate the event is intended to be a state event
     * @returns {Promise<boolean>} resolves to true if the user has the required power level, resolves to false otherwise
     */
    userHasPowerLevelFor(userId: string, roomId: string, eventType: string, isState: boolean): Promise<boolean>;
    /**
     * Converts a MXC URI to an HTTP URL.
     * @param {string} mxc The MXC URI to convert
     * @returns {string} The HTTP URL for the content.
     */
    mxcToHttp(mxc: string): string;
    /**
     * Converts a MXC URI to an HTTP URL for downsizing the content.
     * @param {string} mxc The MXC URI to convert and downsize.
     * @param {number} width The width, as an integer, for the thumbnail.
     * @param {number} height The height, as an intenger, for the thumbnail.
     * @param {"crop"|"scale"} method Whether to crop or scale (preserve aspect ratio) the content.
     * @returns {string} The HTTP URL for the downsized content.
     */
    mxcToHttpThumbnail(mxc: string, width: number, height: number, method: "crop" | "scale"): string;
    /**
     * Uploads data to the homeserver's media repository.
     * @param {Buffer} data the content to upload.
     * @param {string} contentType the content type of the file. Defaults to application/octet-stream
     * @param {string} filename the name of the file. Optional.
     * @returns {Promise<string>} resolves to the MXC URI of the content
     */
    uploadContent(data: Buffer, contentType?: string, filename?: string): Promise<string>;
    /**
     * Uploads data to the homeserver's media repository after downloading it from the
     * provided URL.
     * @param {string} url The URL to download content from.
     * @returns {Promise<string>} Resolves to the MXC URI of the content
     */
    uploadContentFromUrl(url: string): Promise<string>;
    /**
     * Determines the upgrade history for a given room as a doubly-linked list styled structure. Given
     * a room ID in the history of upgrades, the resulting `previous` array will hold any rooms which
     * are older than the given room. The resulting `newer` array will hold any rooms which are newer
     * versions of the room. Both arrays will be defined, but may be empty individually. Element zero
     * of each will always be the nearest to the given room ID and the last element will be the furthest
     * from the room. The given room will never be in either array.
     * @param {string} roomId the room ID to get the history of
     * @returns {Promise<{previous: RoomReference[], newer: RoomReference[]}>} Resolves to the room's
     * upgrade history
     */
    getRoomUpgradeHistory(roomId: string): Promise<{
        previous: RoomReference[];
        newer: RoomReference[];
        current: RoomReference;
    }>;
    /**
     * Performs a web request to the homeserver, applying appropriate authorization headers for
     * this client.
     * @param {"GET"|"POST"|"PUT"|"DELETE"} method The HTTP method to use in the request
     * @param {string} endpoint The endpoint to call. For example: "/_matrix/client/r0/account/whoami"
     * @param {*} qs The query string to send. Optional.
     * @param {*} body The request body to send. Optional. Will be converted to JSON unless the type is a Buffer.
     * @param {number} timeout The number of milliseconds to wait before timing out.
     * @param {boolean} raw If true, the raw response will be returned instead of the response body.
     * @param {string} contentType The content type to send. Only used if the `body` is a Buffer.
     * @returns {Promise<*>} Resolves to the response (body), rejected if a non-2xx status code was returned.
     */
    doRequest(method: any, endpoint: any, qs?: any, body?: any, timeout?: number, raw?: boolean, contentType?: string): Promise<any>;
    private redactObjectForLogging;
}
export interface RoomDirectoryLookupResponse {
    roomId: string;
    residentServers: string[];
}
export interface RoomReference {
    /**
     * The room ID being referenced
     */
    roomId: string;
    /**
     * The version of the room at the time
     */
    version: string;
    /**
     * If going backwards, the tombstone event ID, otherwise the creation
     * event. If the room can't be verified, this will be null. Will be
     * null if this reference is to the current room.
     */
    refEventId: string;
}
