"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var getSyntaxKindName_1 = require("../utils/compiler/getSyntaxKindName");
var ArgumentError_1 = require("./ArgumentError");
var ArgumentNullOrWhitespaceError_1 = require("./ArgumentNullOrWhitespaceError");
var ArgumentOutOfRangeError_1 = require("./ArgumentOutOfRangeError");
var ArgumentTypeError_1 = require("./ArgumentTypeError");
var FileNotFoundError_1 = require("./FileNotFoundError");
var InvalidOperationError_1 = require("./InvalidOperationError");
var NotImplementedError_1 = require("./NotImplementedError");
/**
 * Thows if not a type.
 * @param value - Value to check the type of.
 * @param expectedType - Expected type.
 * @param argName - Argument name.
 */
function throwIfNotType(value, expectedType, argName) {
    if (typeof value !== expectedType)
        throw new ArgumentTypeError_1.ArgumentTypeError(argName, expectedType, typeof value);
}
exports.throwIfNotType = throwIfNotType;
/**
 * Throws if the value is not a string or is whitespace.
 * @param value - Value to check.
 * @param argName - Arg name.
 */
function throwIfNotStringOrWhitespace(value, argName) {
    if (typeof value !== "string")
        throw new ArgumentTypeError_1.ArgumentTypeError(argName, "string", typeof value);
    if (value.trim().length === 0)
        throw new ArgumentNullOrWhitespaceError_1.ArgumentNullOrWhitespaceError(argName);
}
exports.throwIfNotStringOrWhitespace = throwIfNotStringOrWhitespace;
/**
 * Throws a NotImplementedError if a node doesn't match the expected syntax kind.
 * @param node - Node.
 * @param kind - Syntax kind that's expected.
 * @param message - Optional message to throw.
 */
function throwIfNotSyntaxKind(node, kind, message) {
    if (node.getKind() !== kind)
        throw new NotImplementedError_1.NotImplementedError(message || "Expected node to be syntax kind " + getSyntaxKindName_1.getSyntaxKindName(kind) + ", but was " + node.getKindName() + ".");
}
exports.throwIfNotSyntaxKind = throwIfNotSyntaxKind;
/**
 * Throws an ArgumentOutOfRangeError if an argument's value is out of an inclusive range.
 * @param value - Value.
 * @param range - Range.
 * @param argName - Argument name.
 */
function throwIfOutOfRange(value, range, argName) {
    if (value < range[0] || value > range[1])
        throw new ArgumentOutOfRangeError_1.ArgumentOutOfRangeError(argName, value, range);
}
exports.throwIfOutOfRange = throwIfOutOfRange;
/**
 * Throws an ArgumentOutOfRangeError if an argument's range value is out of an inclusive range.
 *
 * Also throws when the start of the range is greater than the end.
 * @param actualRange - Range to check.
 * @param range - Range to check against.
 * @param argName - Argument name.
 */
function throwIfRangeOutOfRange(actualRange, range, argName) {
    if (actualRange[0] > actualRange[1])
        throw new ArgumentError_1.ArgumentError(argName, "The start of a range must not be greater than the end: [" + actualRange[0] + ", " + actualRange[1] + "]");
    throwIfOutOfRange(actualRange[0], range, argName);
    throwIfOutOfRange(actualRange[1], range, argName);
}
exports.throwIfRangeOutOfRange = throwIfRangeOutOfRange;
/**
 * Gets an error saying that a feature is not implemented for a certain syntax kind.
 * @param kind - Syntax kind that isn't implemented.
 */
function getNotImplementedForSyntaxKindError(kind) {
    return new NotImplementedError_1.NotImplementedError("Not implemented feature for syntax kind '" + getSyntaxKindName_1.getSyntaxKindName(kind) + "'.");
}
exports.getNotImplementedForSyntaxKindError = getNotImplementedForSyntaxKindError;
/**
 * Throws an Argument
 * @param value
 * @param argName
 */
function throwIfNegative(value, argName) {
    if (value < 0)
        throw new ArgumentError_1.ArgumentError(argName, "Expected a non-negative value.");
}
exports.throwIfNegative = throwIfNegative;
/**
 * Throws when the value is null or undefined.
 * @param value - Value to check.
 * @param errorMessage - Error message to throw when not defined.
 */
function throwIfNullOrUndefined(value, errorMessage) {
    if (value == null)
        throw new InvalidOperationError_1.InvalidOperationError(typeof errorMessage === "string" ? errorMessage : errorMessage());
    return value;
}
exports.throwIfNullOrUndefined = throwIfNullOrUndefined;
/**
 * Throw if the value should have been the never type.
 * @param value - Value to check.
 */
function getNotImplementedForNeverValueError(value) {
    return new NotImplementedError_1.NotImplementedError("Not implemented value: " + JSON.stringify(value));
}
exports.getNotImplementedForNeverValueError = getNotImplementedForNeverValueError;
/**
 * Throws an error if the actual value does not equal the expected value.
 * @param actual - Actual value.
 * @param expected - Expected value.
 * @param description - Message to show in the error. Should be a full sentence that doesn't include the actual and expected values.
 */
function throwIfNotEqual(actual, expected, description) {
    if (actual !== expected)
        throw new InvalidOperationError_1.InvalidOperationError("Expected " + actual + " to equal " + expected + ". " + description);
}
exports.throwIfNotEqual = throwIfNotEqual;
/**
 * Throws if true.
 * @param value - Value to check.
 * @param errorMessage - Error message to throw when true.
 */
function throwIfTrue(value, errorMessage) {
    if (value === true)
        throw new InvalidOperationError_1.InvalidOperationError(errorMessage);
}
exports.throwIfTrue = throwIfTrue;
/**
 * Throws if the file does not exist.
 * @param fileSystem - File system host.
 * @param filePath - File path.
 */
function throwIfFileNotExists(fileSystemWrapper, filePath) {
    if (!fileSystemWrapper.fileExistsSync(filePath))
        throw new FileNotFoundError_1.FileNotFoundError(filePath);
}
exports.throwIfFileNotExists = throwIfFileNotExists;
