"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var errors = require("../../errors");
var manipulation_1 = require("../../manipulation");
var typescript_1 = require("../../typescript");
var utils_1 = require("../../utils");
var common_1 = require("../common");
var ExportSpecifier = /** @class */ (function (_super) {
    tslib_1.__extends(ExportSpecifier, _super);
    function ExportSpecifier() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Sets the name of what's being exported.
     */
    ExportSpecifier.prototype.setName = function (name) {
        var nameNode = this.getNameNode();
        if (nameNode.getText() === name)
            return this;
        var start = nameNode.getStart();
        manipulation_1.replaceNodeText({
            sourceFile: this.sourceFile,
            start: start,
            replacingLength: nameNode.getWidth(),
            newText: name
        });
        return this;
    };
    /**
     * Renames the name of what's being exported.
     */
    ExportSpecifier.prototype.renameName = function (name) {
        this.getNameNode().rename(name);
        return this;
    };
    /**
     * Gets the name node of what's being exported.
     */
    ExportSpecifier.prototype.getNameNode = function () {
        return this.getFirstChildByKindOrThrow(typescript_1.SyntaxKind.Identifier);
    };
    /**
     * Sets the alias for the name being exported.
     * @param alias - Alias to set.
     */
    ExportSpecifier.prototype.setAlias = function (alias) {
        var aliasIdentifier = this.getAliasIdentifier();
        if (aliasIdentifier == null) {
            // trick is to insert an alias with the same name, then rename the alias. TS compiler will take care of the rest.
            var nameNode = this.getNameNode();
            manipulation_1.insertIntoParentTextRange({
                insertPos: nameNode.getEnd(),
                parent: this,
                newText: " as " + nameNode.getText()
            });
            aliasIdentifier = this.getAliasIdentifier();
        }
        aliasIdentifier.rename(alias);
        return this;
    };
    /**
     * Gets the alias identifier, if it exists.
     */
    ExportSpecifier.prototype.getAliasIdentifier = function () {
        var asKeyword = this.getFirstChildByKind(typescript_1.SyntaxKind.AsKeyword);
        if (asKeyword == null)
            return undefined;
        var aliasIdentifier = asKeyword.getNextSibling();
        if (aliasIdentifier == null || !(utils_1.TypeGuards.isIdentifier(aliasIdentifier)))
            return undefined;
        return aliasIdentifier;
    };
    /**
     * Gets the export declaration associated with this export specifier.
     */
    ExportSpecifier.prototype.getExportDeclaration = function () {
        return this.getFirstAncestorByKindOrThrow(typescript_1.SyntaxKind.ExportDeclaration);
    };
    /**
     * Gets the local target symbol of the export specifier or throws if it doesn't exist.
     */
    ExportSpecifier.prototype.getLocalTargetSymbolOrThrow = function () {
        return errors.throwIfNullOrUndefined(this.getLocalTargetSymbol(), "The export specifier's local target symbol was expected.");
    };
    /**
     * Gets the local target symbol of the export specifier or undefined if it doesn't exist.
     */
    ExportSpecifier.prototype.getLocalTargetSymbol = function () {
        return this.global.typeChecker.getExportSpecifierLocalTargetSymbol(this);
    };
    /**
     * Gets all the declarations referenced by the export specifier.
     */
    ExportSpecifier.prototype.getLocalTargetDeclarations = function () {
        var symbol = this.getLocalTargetSymbol();
        return symbol == null ? [] : symbol.getDeclarations();
    };
    /**
     * Removes the export specifier.
     */
    ExportSpecifier.prototype.remove = function () {
        var exportDeclaration = this.getExportDeclaration();
        var exports = exportDeclaration.getNamedExports();
        if (exports.length > 1)
            manipulation_1.removeCommaSeparatedChild(this);
        else if (exportDeclaration.hasModuleSpecifier())
            exportDeclaration.toNamespaceExport();
        else
            exportDeclaration.remove();
    };
    return ExportSpecifier;
}(common_1.Node));
exports.ExportSpecifier = ExportSpecifier;
