'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var fs = require('fs-extra');
var LiveServer = require('live-server');
var _ = require('lodash');
var path = require('path');
var Ast = require('ts-simple-ast');
var Ast__default = _interopDefault(Ast);
var Handlebars = require('handlebars');
var semver = require('semver');
var i18next = _interopDefault(require('i18next'));
var JSON5 = require('json5');

var log = require('fancy-log');
var c = require('chalk');
var pkg = require('../package.json');
var LEVEL;
(function (LEVEL) {
    LEVEL[LEVEL["INFO"] = 0] = "INFO";
    LEVEL[LEVEL["DEBUG"] = 1] = "DEBUG";
    LEVEL[LEVEL["ERROR"] = 2] = "ERROR";
    LEVEL[LEVEL["WARN"] = 3] = "WARN";
})(LEVEL || (LEVEL = {}));
var Logger = /** @class */ (function () {
    function Logger() {
        this.name = pkg.name;
        this.version = pkg.version;
        this.logger = log;
        this.silent = true;
    }
    Logger.prototype.info = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.INFO].concat(args)));
    };
    Logger.prototype.error = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.ERROR].concat(args)));
    };
    Logger.prototype.warn = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.WARN].concat(args)));
    };
    Logger.prototype.debug = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.DEBUG].concat(args)));
    };
    Logger.prototype.format = function (level) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var pad = function (s, l, z) {
            if (z === void 0) { z = ''; }
            return s + Array(Math.max(0, l - s.length + 1)).join(z);
        };
        var msg = args.join(' ');
        if (args.length > 1) {
            msg = pad(args.shift(), 15, ' ') + ": " + args.join(' ');
        }
        switch (level) {
            case LEVEL.INFO:
                msg = c.green(msg);
                break;
            case LEVEL.DEBUG:
                msg = c.cyan(msg);
                break;
            case LEVEL.WARN:
                msg = c.yellow(msg);
                break;
            case LEVEL.ERROR:
                msg = c.red(msg);
                break;
        }
        return [msg].join('');
    };
    return Logger;
}());
var logger = new Logger();

var COMPODOC_DEFAULTS = {
    title: 'Application documentation',
    additionalEntryName: 'Additional documentation',
    additionalEntryPath: 'additional-documentation',
    folder: './documentation/',
    hostname: '127.0.0.1',
    port: 8080,
    theme: 'gitbook',
    exportFormat: 'html',
    exportFormatsSupported: ['html', 'json', 'pdf'],
    base: '/',
    defaultCoverageThreshold: 70,
    defaultCoverageMinimumPerFile: 0,
    coverageTestThresholdFail: true,
    toggleMenuItems: ['all'],
    navTabConfig: [],
    disableSourceCode: false,
    disableDomTree: false,
    disableTemplateTab: false,
    disableStyleTab: false,
    disableGraph: false,
    disableMainGraph: false,
    disableCoverage: false,
    disablePrivate: false,
    disableProtected: false,
    disableInternal: false,
    disableLifeCycleHooks: false,
    disableRoutesGraph: false,
    disableDependencies: false,
    PAGE_TYPES: {
        ROOT: 'root',
        INTERNAL: 'internal'
    },
    gaSite: 'auto',
    coverageTestShowOnlyFailed: false,
    language: 'en-US'
};

var Configuration = /** @class */ (function () {
    function Configuration() {
        this._pages = [];
        this._mainData = {
            output: COMPODOC_DEFAULTS.folder,
            theme: COMPODOC_DEFAULTS.theme,
            extTheme: '',
            serve: false,
            hostname: COMPODOC_DEFAULTS.hostname,
            host: '',
            port: COMPODOC_DEFAULTS.port,
            open: false,
            assetsFolder: '',
            documentationMainName: COMPODOC_DEFAULTS.title,
            documentationMainDescription: '',
            base: COMPODOC_DEFAULTS.base,
            hideGenerator: false,
            hasFilesToCoverage: false,
            modules: [],
            readme: false,
            changelog: '',
            contributing: '',
            license: '',
            todo: '',
            markdowns: [],
            additionalPages: [],
            pipes: [],
            classes: [],
            interfaces: [],
            components: [],
            controllers: [],
            directives: [],
            injectables: [],
            interceptors: [],
            guards: [],
            miscellaneous: [],
            routes: [],
            tsconfig: '',
            toggleMenuItems: COMPODOC_DEFAULTS.toggleMenuItems,
            navTabConfig: [],
            templates: '',
            includes: '',
            includesName: COMPODOC_DEFAULTS.additionalEntryName,
            includesFolder: COMPODOC_DEFAULTS.additionalEntryPath,
            disableSourceCode: COMPODOC_DEFAULTS.disableSourceCode,
            disableDomTree: COMPODOC_DEFAULTS.disableDomTree,
            disableTemplateTab: COMPODOC_DEFAULTS.disableTemplateTab,
            disableStyleTab: COMPODOC_DEFAULTS.disableStyleTab,
            disableGraph: COMPODOC_DEFAULTS.disableGraph,
            disableMainGraph: COMPODOC_DEFAULTS.disableMainGraph,
            disableCoverage: COMPODOC_DEFAULTS.disableCoverage,
            disablePrivate: COMPODOC_DEFAULTS.disablePrivate,
            disableInternal: COMPODOC_DEFAULTS.disableInternal,
            disableProtected: COMPODOC_DEFAULTS.disableProtected,
            disableLifeCycleHooks: COMPODOC_DEFAULTS.disableLifeCycleHooks,
            disableRoutesGraph: COMPODOC_DEFAULTS.disableRoutesGraph,
            disableSearch: false,
            disableDependencies: COMPODOC_DEFAULTS.disableDependencies,
            watch: false,
            mainGraph: '',
            coverageTest: false,
            coverageTestThreshold: COMPODOC_DEFAULTS.defaultCoverageThreshold,
            coverageTestThresholdFail: COMPODOC_DEFAULTS.coverageTestThresholdFail,
            coverageTestPerFile: false,
            coverageMinimumPerFile: COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile,
            unitTestCoverage: '',
            unitTestData: undefined,
            coverageTestShowOnlyFailed: COMPODOC_DEFAULTS.coverageTestShowOnlyFailed,
            routesLength: 0,
            angularVersion: '',
            exportFormat: COMPODOC_DEFAULTS.exportFormat,
            coverageData: {},
            customFavicon: '',
            customLogo: '',
            packageDependencies: [],
            packagePeerDependencies: [],
            gaID: '',
            gaSite: '',
            angularProject: false,
            angularJSProject: false,
            language: COMPODOC_DEFAULTS.language
        };
    }
    Configuration.getInstance = function () {
        if (!Configuration.instance) {
            Configuration.instance = new Configuration();
        }
        return Configuration.instance;
    };
    Configuration.prototype.addPage = function (page) {
        var indexPage = _.findIndex(this._pages, { name: page.name });
        if (indexPage === -1) {
            this._pages.push(page);
        }
    };
    Configuration.prototype.hasPage = function (name) {
        var indexPage = _.findIndex(this._pages, { name: name });
        return indexPage !== -1;
    };
    Configuration.prototype.addAdditionalPage = function (page) {
        this._mainData.additionalPages.push(page);
    };
    Configuration.prototype.getAdditionalPageById = function (id) {
        return this._mainData.additionalPages.find(function (page) { return page.id === id; });
    };
    Configuration.prototype.resetPages = function () {
        this._pages = [];
    };
    Configuration.prototype.resetAdditionalPages = function () {
        this._mainData.additionalPages = [];
    };
    Configuration.prototype.resetRootMarkdownPages = function () {
        var indexPage = _.findIndex(this._pages, { name: 'index' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'changelog' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'contributing' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'license' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'todo' });
        this._pages.splice(indexPage, 1);
        this._mainData.markdowns = [];
    };
    Object.defineProperty(Configuration.prototype, "pages", {
        get: function () {
            return this._pages;
        },
        set: function (pages) {
            this._pages = [];
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "markDownPages", {
        get: function () {
            return this._pages.filter(function (page) { return page.markdown; });
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "mainData", {
        get: function () {
            return this._mainData;
        },
        set: function (data) {
            Object.assign(this._mainData, data);
        },
        enumerable: true,
        configurable: true
    });
    return Configuration;
}());
var Configuration$1 = Configuration.getInstance();

var AngularAPIs = require('../src/data/api-list.json');
var AngularApiUtil = /** @class */ (function () {
    function AngularApiUtil() {
    }
    AngularApiUtil.getInstance = function () {
        if (!AngularApiUtil.instance) {
            AngularApiUtil.instance = new AngularApiUtil();
        }
        return AngularApiUtil.instance;
    };
    AngularApiUtil.prototype.findApi = function (type) {
        var foundedApi;
        _.forEach(AngularAPIs, function (mainApi) {
            _.forEach(mainApi.items, function (api) {
                if (api.title === type) {
                    foundedApi = api;
                }
            });
        });
        return {
            source: 'external',
            data: foundedApi
        };
    };
    return AngularApiUtil;
}());
var AngularApiUtil$1 = AngularApiUtil.getInstance();

function extractLeadingText(string, completeTag) {
    var tagIndex = string.indexOf(completeTag);
    var leadingText = undefined;
    var leadingTextRegExp = /\[(.+?)\]/g;
    var leadingTextInfo = leadingTextRegExp.exec(string);
    // did we find leading text, and if so, does it immediately precede the tag?
    while (leadingTextInfo && leadingTextInfo.length) {
        if (leadingTextInfo.index + leadingTextInfo[0].length === tagIndex) {
            string = string.replace(leadingTextInfo[0], '');
            leadingText = leadingTextInfo[1];
            break;
        }
        leadingTextInfo = leadingTextRegExp.exec(string);
    }
    return {
        leadingText: leadingText,
        string: string
    };
}
function splitLinkText(text) {
    var linkText;
    var target;
    var splitIndex;
    // if a pipe is not present, we split on the first space
    splitIndex = text.indexOf('|');
    if (splitIndex === -1) {
        splitIndex = text.search(/\s/);
    }
    if (splitIndex !== -1) {
        linkText = text.substr(splitIndex + 1);
        // Normalize subsequent newlines to a single space.
        linkText = linkText.replace(/\n+/, ' ');
        target = text.substr(0, splitIndex);
    }
    return {
        linkText: linkText,
        target: target || text
    };
}
var LinkParser = (function () {
    var processTheLink = function (string, tagInfo, leadingText) {
        var leading = extractLeadingText(string, tagInfo.completeTag), linkText, split, target, stringtoReplace;
        linkText = leadingText ? leadingText : leading.leadingText || '';
        split = splitLinkText(tagInfo.text);
        target = split.target;
        if (leading.leadingText !== undefined) {
            stringtoReplace = '[' + leading.leadingText + ']' + tagInfo.completeTag;
        }
        else if (typeof split.linkText !== 'undefined') {
            stringtoReplace = tagInfo.completeTag;
            linkText = split.linkText;
        }
        return string.replace(stringtoReplace, '[' + linkText + '](' + target + ')');
    };
    /**
     * Convert
     * {@link http://www.google.com|Google} or {@link https://github.com GitHub} or [Github]{@link https://github.com} to [Github](https://github.com)
     */
    var replaceLinkTag = function (str) {
        if (typeof str === 'undefined') {
            return {
                newString: ''
            };
        }
        // new RegExp('\\[((?:.|\n)+?)]\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TO DO]{@link Todo} fo') -> "[TO DO]{@link Todo}", "TO DO", "Todo"
        // new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TODO]{@link Todo} fo') -> "{@link Todo}", "Todo"
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExp, matches, previousString;
        tagRegExp = str.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(str, matchedTag, linkText);
            }
            else {
                return replacer(str, matchedTag);
            }
        }
        do {
            matches = tagRegExp.exec(str);
            if (matches) {
                previousString = str;
                if (matches.length === 2) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== str);
        return {
            newString: str
        };
    };
    var _resolveLinks = function (str) {
        return replaceLinkTag(str).newString;
    };
    return {
        resolveLinks: _resolveLinks
    };
})();

var AngularLifecycleHooks;
(function (AngularLifecycleHooks) {
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnChanges"] = 0] = "ngOnChanges";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnInit"] = 1] = "ngOnInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngDoCheck"] = 2] = "ngDoCheck";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentInit"] = 3] = "ngAfterContentInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentChecked"] = 4] = "ngAfterContentChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewInit"] = 5] = "ngAfterViewInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewChecked"] = 6] = "ngAfterViewChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnDestroy"] = 7] = "ngOnDestroy";
})(AngularLifecycleHooks || (AngularLifecycleHooks = {}));

function kindToType(kind) {
    var _type = '';
    switch (kind) {
        case Ast.SyntaxKind.StringKeyword:
        case Ast.SyntaxKind.StringLiteral:
            _type = 'string';
            break;
        case Ast.SyntaxKind.NumberKeyword:
        case Ast.SyntaxKind.NumericLiteral:
            _type = 'number';
            break;
        case Ast.SyntaxKind.ArrayType:
        case Ast.SyntaxKind.ArrayLiteralExpression:
            _type = '[]';
            break;
        case Ast.SyntaxKind.VoidKeyword:
            _type = 'void';
            break;
        case Ast.SyntaxKind.FunctionType:
            _type = 'function';
            break;
        case Ast.SyntaxKind.TypeLiteral:
            _type = 'literal type';
            break;
        case Ast.SyntaxKind.BooleanKeyword:
            _type = 'boolean';
            break;
        case Ast.SyntaxKind.AnyKeyword:
            _type = 'any';
            break;
        case Ast.SyntaxKind.NullKeyword:
            _type = 'null';
            break;
        case Ast.SyntaxKind.SymbolKeyword:
            _type = 'symbol';
            break;
        case Ast.SyntaxKind.NeverKeyword:
            _type = 'never';
            break;
        case Ast.SyntaxKind.UndefinedKeyword:
            _type = 'undefined';
            break;
        case Ast.SyntaxKind.ObjectKeyword:
        case Ast.SyntaxKind.ObjectLiteralExpression:
            _type = 'object';
            break;
    }
    return _type;
}

var getCurrentDirectory = Ast.ts.sys.getCurrentDirectory;
var useCaseSensitiveFileNames = Ast.ts.sys.useCaseSensitiveFileNames;
var newLine = Ast.ts.sys.newLine;
var marked = require('marked');
function getNewLine() {
    return newLine;
}
function cleanNameWithoutSpaceAndToLowerCase(name) {
    return name.toLowerCase().replace(/ /g, '-');
}
function getCanonicalFileName(fileName) {
    return useCaseSensitiveFileNames ? fileName : fileName.toLowerCase();
}
var formatDiagnosticsHost = {
    getCurrentDirectory: getCurrentDirectory,
    getCanonicalFileName: getCanonicalFileName,
    getNewLine: getNewLine
};
function markedtags(tags) {
    var mtags = tags;
    _.forEach(mtags, function (tag) {
        tag.comment = marked(LinkParser.resolveLinks(tag.comment));
    });
    return mtags;
}
function mergeTagsAndArgs(args, jsdoctags) {
    var margs = _.cloneDeep(args);
    _.forEach(margs, function (arg) {
        arg.tagName = {
            text: 'param'
        };
        if (jsdoctags) {
            _.forEach(jsdoctags, function (jsdoctag) {
                if (jsdoctag.name && jsdoctag.name.text === arg.name) {
                    arg.tagName = jsdoctag.tagName;
                    arg.name = jsdoctag.name;
                    arg.comment = jsdoctag.comment;
                    arg.typeExpression = jsdoctag.typeExpression;
                }
            });
        }
    });
    // Add example & returns & private
    if (jsdoctags) {
        _.forEach(jsdoctags, function (jsdoctag) {
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'example' || jsdoctag.tagName.text === 'private')) {
                margs.push({
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                });
            }
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'returns' || jsdoctag.tagName.text === 'return')) {
                var ret = {
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                };
                if (jsdoctag.typeExpression && jsdoctag.typeExpression.type) {
                    ret.returnType = kindToType(jsdoctag.typeExpression.type.kind);
                }
                margs.push(ret);
            }
        });
    }
    return margs;
}
function readConfig(configFile) {
    var result = Ast.ts.readConfigFile(configFile, Ast.ts.sys.readFile);
    if (result.error) {
        var message = Ast.ts.formatDiagnostics([result.error], formatDiagnosticsHost);
        throw new Error(message);
    }
    return result.config;
}
function stripBom(source) {
    if (source.charCodeAt(0) === 0xfeff) {
        return source.slice(1);
    }
    return source;
}
function hasBom(source) {
    return source.charCodeAt(0) === 0xfeff;
}
function handlePath(files, cwd) {
    var _files = files;
    var i = 0;
    var len = files.length;
    for (i; i < len; i++) {
        if (files[i].indexOf(cwd) === -1) {
            files[i] = path.resolve(cwd + path.sep + files[i]);
        }
    }
    return _files;
}
function cleanLifecycleHooksFromMethods(methods) {
    var result = [];
    if (typeof methods !== 'undefined') {
        var i = 0;
        var len = methods.length;
        for (i; i < len; i++) {
            if (!(methods[i].name in AngularLifecycleHooks)) {
                result.push(methods[i]);
            }
        }
    }
    return result;
}
function cleanSourcesForWatch(list) {
    return list.filter(function (element) {
        if (fs.existsSync(process.cwd() + path.sep + element)) {
            return element;
        }
    });
}
function getNamesCompareFn(name) {
    /**
     * Copyright https://github.com/ng-bootstrap/ng-bootstrap
     */
    name = name || 'name';
    var t = function (a, b) {
        if (a[name]) {
            return a[name].localeCompare(b[name]);
        }
        else {
            return 0;
        }
    };
    return t;
}
function isIgnore(member) {
    if (member.jsDoc) {
        for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
            var doc = _a[_i];
            if (doc.tags) {
                for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                    var tag = _c[_b];
                    if (tag.tagName.text.indexOf('ignore') > -1) {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}
// https://tc39.github.io/ecma262/#sec-array.prototype.includes
if (!Array.prototype.includes) {
    Object.defineProperty(Array.prototype, 'includes', {
        value: function (searchElement, fromIndex) {
            if (this == null) {
                throw new TypeError('"this" is null or not defined');
            }
            // 1. Let O be ? ToObject(this value).
            var o = Object(this);
            // 2. Let len be ? ToLength(? Get(O, "length")).
            var len = o.length >>> 0;
            // 3. If len is 0, return false.
            if (len === 0) {
                return false;
            }
            // 4. Let n be ? ToInteger(fromIndex).
            //    (If fromIndex is undefined, this step produces the value 0.)
            var n = fromIndex | 0;
            // 5. If n ≥ 0, then
            //  a. Let k be n.
            // 6. Else n < 0,
            //  a. Let k be len + n.
            //  b. If k < 0, let k be 0.
            var k = Math.max(n >= 0 ? n : len - Math.abs(n), 0);
            function sameValueZero(x, y) {
                return (x === y ||
                    (typeof x === 'number' && typeof y === 'number' && isNaN(x) && isNaN(y)));
            }
            // 7. Repeat, while k < len
            while (k < len) {
                // a. Let elementK be the result of ? Get(O, ! ToString(k)).
                // b. If SameValueZero(searchElement, elementK) is true, return true.
                if (sameValueZero(o[k], searchElement)) {
                    return true;
                }
                // c. Increase k by 1.
                k++;
            }
            // 8. Return false
            return false;
        }
    });
}
function findMainSourceFolder(files) {
    var mainFolder = '';
    var mainFolderCount = 0;
    var rawFolders = files.map(function (filepath) {
        var shortPath = filepath.replace(process.cwd() + path.sep, '');
        return path.dirname(shortPath);
    });
    var folders = {};
    rawFolders = _.uniq(rawFolders);
    for (var i = 0; i < rawFolders.length; i++) {
        var sep = rawFolders[i].split(path.sep);
        sep.forEach(function (folder) {
            if (folders[folder]) {
                folders[folder] += 1;
            }
            else {
                folders[folder] = 1;
            }
        });
    }
    for (var f in folders) {
        if (folders[f] > mainFolderCount) {
            mainFolderCount = folders[f];
            mainFolder = f;
        }
    }
    return mainFolder;
}
// Create a compilerHost object to allow the compiler to read and write files
function compilerHost(transpileOptions) {
    var inputFileName = transpileOptions.fileName || (transpileOptions.jsx ? 'module.tsx' : 'module.ts');
    var toReturn = {
        getSourceFile: function (fileName) {
            if (fileName.lastIndexOf('.ts') !== -1 || fileName.lastIndexOf('.js') !== -1) {
                if (fileName === 'lib.d.ts') {
                    return undefined;
                }
                if (fileName.substr(-5) === '.d.ts') {
                    return undefined;
                }
                if (path.isAbsolute(fileName) === false) {
                    fileName = path.join(transpileOptions.tsconfigDirectory, fileName);
                }
                if (!fs.existsSync(fileName)) {
                    return undefined;
                }
                var libSource = '';
                try {
                    libSource = fs.readFileSync(fileName).toString();
                    if (hasBom(libSource)) {
                        libSource = stripBom(libSource);
                    }
                }
                catch (e) {
                    logger.debug(e, fileName);
                }
                return Ast.ts.createSourceFile(fileName, libSource, transpileOptions.target, false);
            }
            return undefined;
        },
        writeFile: function (name, text) { },
        getDefaultLibFileName: function () { return 'lib.d.ts'; },
        useCaseSensitiveFileNames: function () { return false; },
        getCanonicalFileName: function (fileName) { return fileName; },
        getCurrentDirectory: function () { return ''; },
        getNewLine: function () { return '\n'; },
        fileExists: function (fileName) { return fileName === inputFileName; },
        readFile: function () { return ''; },
        directoryExists: function () { return true; },
        getDirectories: function () { return []; }
    };
    return toReturn;
}
function detectIndent(str, count) {
    var stripIndent = function (stripedString) {
        var match = stripedString.match(/^[ \t]*(?=\S)/gm);
        if (!match) {
            return stripedString;
        }
        // TODO: use spread operator when targeting Node.js 6
        var indent = Math.min.apply(Math, match.map(function (x) { return x.length; })); // eslint-disable-line
        var re = new RegExp("^[ \\t]{" + indent + "}", 'gm');
        return indent > 0 ? stripedString.replace(re, '') : stripedString;
    };
    var repeating = function (n, repeatString) {
        repeatString = repeatString === undefined ? ' ' : repeatString;
        if (typeof repeatString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof repeatString + "`");
        }
        if (n < 0) {
            throw new TypeError("Expected `count` to be a positive finite number, got `" + n + "`");
        }
        var ret = '';
        do {
            if (n & 1) {
                ret += repeatString;
            }
            repeatString += repeatString;
        } while ((n >>= 1));
        return ret;
    };
    var indentString = function (indentedString, indentCount) {
        var indent = ' ';
        indentCount = indentCount === undefined ? 1 : indentCount;
        if (typeof indentedString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof indentedString + "`");
        }
        if (typeof indentCount !== 'number') {
            throw new TypeError("Expected `count` to be a `number`, got `" + typeof indentCount + "`");
        }
        if (typeof indent !== 'string') {
            throw new TypeError("Expected `indent` to be a `string`, got `" + typeof indent + "`");
        }
        if (indentCount === 0) {
            return indentedString;
        }
        indent = indentCount > 1 ? repeating(indentCount, indent) : indent;
        return indentedString.replace(/^(?!\s*$)/gm, indent);
    };
    return indentString(stripIndent(str), count || 0);
}

var traverse = require('traverse');
var DependenciesEngine = /** @class */ (function () {
    function DependenciesEngine() {
        this.miscellaneous = {
            variables: [],
            functions: [],
            typealiases: [],
            enumerations: [],
            groupedVariables: [],
            groupedFunctions: [],
            groupedEnumerations: [],
            groupedTypeAliases: []
        };
    }
    DependenciesEngine.getInstance = function () {
        if (!DependenciesEngine.instance) {
            DependenciesEngine.instance = new DependenciesEngine();
        }
        return DependenciesEngine.instance;
    };
    DependenciesEngine.prototype.updateModulesDeclarationsExportsTypes = function () {
        var _this = this;
        var mergeTypes = function (entry) {
            var directive = _this.findInCompodocDependencies(entry.name, _this.directives, entry.file);
            if (typeof directive.data !== 'undefined') {
                entry.type = 'directive';
                entry.id = directive.data.id;
            }
            var component = _this.findInCompodocDependencies(entry.name, _this.components, entry.file);
            if (typeof component.data !== 'undefined') {
                entry.type = 'component';
                entry.id = component.data.id;
            }
            var pipe = _this.findInCompodocDependencies(entry.name, _this.pipes, entry.file);
            if (typeof pipe.data !== 'undefined') {
                entry.type = 'pipe';
                entry.id = pipe.data.id;
            }
        };
        this.modules.forEach(function (module) {
            module.declarations.forEach(function (declaration) {
                mergeTypes(declaration);
            });
            module.exports.forEach(function (expt) {
                mergeTypes(expt);
            });
            module.entryComponents.forEach(function (ent) {
                mergeTypes(ent);
            });
        });
    };
    DependenciesEngine.prototype.init = function (data) {
        traverse(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
            }
        });
        this.rawData = data;
        this.modules = _.sortBy(this.rawData.modules, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModulesForOverview = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModules = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.components = _.sortBy(this.rawData.components, [function (el) { return el.name.toLowerCase(); }]);
        this.controllers = _.sortBy(this.rawData.controllers, [function (el) { return el.name.toLowerCase(); }]);
        this.directives = _.sortBy(this.rawData.directives, [function (el) { return el.name.toLowerCase(); }]);
        this.injectables = _.sortBy(this.rawData.injectables, [function (el) { return el.name.toLowerCase(); }]);
        this.interceptors = _.sortBy(this.rawData.interceptors, [function (el) { return el.name.toLowerCase(); }]);
        this.guards = _.sortBy(this.rawData.guards, [function (el) { return el.name.toLowerCase(); }]);
        this.interfaces = _.sortBy(this.rawData.interfaces, [function (el) { return el.name.toLowerCase(); }]);
        this.pipes = _.sortBy(this.rawData.pipes, [function (el) { return el.name.toLowerCase(); }]);
        this.classes = _.sortBy(this.rawData.classes, [function (el) { return el.name.toLowerCase(); }]);
        this.miscellaneous = this.rawData.miscellaneous;
        this.prepareMiscellaneous();
        this.updateModulesDeclarationsExportsTypes();
        this.routes = this.rawData.routesTree;
        this.manageDuplicatesName();
        this.cleanRawModulesNames();
    };
    DependenciesEngine.prototype.cleanRawModulesNames = function () {
        this.rawModulesForOverview = this.rawModulesForOverview.map(function (module) {
            module.name = module.name.replace('$', '');
            return module;
        });
    };
    DependenciesEngine.prototype.findInCompodocDependencies = function (name, data, file) {
        var _result = {
            source: 'internal',
            data: undefined
        };
        var nameFoundCounter = 0;
        if (data && data.length > 0) {
            for (var i = 0; i < data.length; i++) {
                if (typeof name !== 'undefined') {
                    if (typeof file !== 'undefined') {
                        if (name.indexOf(data[i].name) !== -1 &&
                            file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                    else {
                        if (name.indexOf(data[i].name) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                }
            }
            // Prevent wrong matching like MultiSelectOptionDirective with SelectOptionDirective, or QueryParamGroupService with QueryParamGroup
            if (nameFoundCounter > 1) {
                var found = false;
                for (var i = 0; i < data.length; i++) {
                    if (typeof name !== 'undefined') {
                        if (typeof file !== 'undefined') {
                            if (name === data[i].name &&
                                file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                        else {
                            if (name === data[i].name) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                    }
                }
                if (!found) {
                    _result = {
                        source: 'internal',
                        data: undefined
                    };
                }
            }
        }
        return _result;
    };
    DependenciesEngine.prototype.manageDuplicatesName = function () {
        var processDuplicates = function (element, index, array) {
            var elementsWithSameName = _.filter(array, { name: element.name });
            if (elementsWithSameName.length > 1) {
                // First element is the reference for duplicates
                for (var i = 1; i < elementsWithSameName.length; i++) {
                    var elementToEdit = elementsWithSameName[i];
                    if (typeof elementToEdit.isDuplicate === 'undefined') {
                        elementToEdit.isDuplicate = true;
                        elementToEdit.duplicateId = i;
                        elementToEdit.duplicateName =
                            elementToEdit.name + '-' + elementToEdit.duplicateId;
                        elementToEdit.id = elementToEdit.id + '-' + elementToEdit.duplicateId;
                    }
                }
            }
            return element;
        };
        this.classes = this.classes.map(processDuplicates);
        this.interfaces = this.interfaces.map(processDuplicates);
        this.injectables = this.injectables.map(processDuplicates);
        this.pipes = this.pipes.map(processDuplicates);
        this.interceptors = this.interceptors.map(processDuplicates);
        this.guards = this.guards.map(processDuplicates);
        this.modules = this.modules.map(processDuplicates);
        this.components = this.components.map(processDuplicates);
        this.controllers = this.controllers.map(processDuplicates);
        this.directives = this.directives.map(processDuplicates);
    };
    DependenciesEngine.prototype.find = function (name) {
        var _this = this;
        var searchFunctions = [
            function () { return _this.findInCompodocDependencies(name, _this.injectables); },
            function () { return _this.findInCompodocDependencies(name, _this.interceptors); },
            function () { return _this.findInCompodocDependencies(name, _this.guards); },
            function () { return _this.findInCompodocDependencies(name, _this.interfaces); },
            function () { return _this.findInCompodocDependencies(name, _this.classes); },
            function () { return _this.findInCompodocDependencies(name, _this.components); },
            function () { return _this.findInCompodocDependencies(name, _this.controllers); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.variables); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.functions); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.typealiases); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.enumerations); },
            function () { return AngularApiUtil$1.findApi(name); }
        ];
        for (var _i = 0, searchFunctions_1 = searchFunctions; _i < searchFunctions_1.length; _i++) {
            var searchFunction = searchFunctions_1[_i];
            var result = searchFunction();
            if (result.data) {
                return result;
            }
        }
        return undefined;
    };
    DependenciesEngine.prototype.update = function (updatedData) {
        var _this = this;
        if (updatedData.modules.length > 0) {
            _.forEach(updatedData.modules, function (module) {
                var _index = _.findIndex(_this.modules, { name: module.name });
                _this.modules[_index] = module;
            });
        }
        if (updatedData.components.length > 0) {
            _.forEach(updatedData.components, function (component) {
                var _index = _.findIndex(_this.components, { name: component.name });
                _this.components[_index] = component;
            });
        }
        if (updatedData.controllers.length > 0) {
            _.forEach(updatedData.controllers, function (controller) {
                var _index = _.findIndex(_this.controllers, { name: controller.name });
                _this.controllers[_index] = controller;
            });
        }
        if (updatedData.directives.length > 0) {
            _.forEach(updatedData.directives, function (directive) {
                var _index = _.findIndex(_this.directives, { name: directive.name });
                _this.directives[_index] = directive;
            });
        }
        if (updatedData.injectables.length > 0) {
            _.forEach(updatedData.injectables, function (injectable) {
                var _index = _.findIndex(_this.injectables, { name: injectable.name });
                _this.injectables[_index] = injectable;
            });
        }
        if (updatedData.interceptors.length > 0) {
            _.forEach(updatedData.interceptors, function (interceptor) {
                var _index = _.findIndex(_this.interceptors, { name: interceptor.name });
                _this.interceptors[_index] = interceptor;
            });
        }
        if (updatedData.guards.length > 0) {
            _.forEach(updatedData.guards, function (guard) {
                var _index = _.findIndex(_this.guards, { name: guard.name });
                _this.guards[_index] = guard;
            });
        }
        if (updatedData.interfaces.length > 0) {
            _.forEach(updatedData.interfaces, function (int) {
                var _index = _.findIndex(_this.interfaces, { name: int.name });
                _this.interfaces[_index] = int;
            });
        }
        if (updatedData.pipes.length > 0) {
            _.forEach(updatedData.pipes, function (pipe) {
                var _index = _.findIndex(_this.pipes, { name: pipe.name });
                _this.pipes[_index] = pipe;
            });
        }
        if (updatedData.classes.length > 0) {
            _.forEach(updatedData.classes, function (classe) {
                var _index = _.findIndex(_this.classes, { name: classe.name });
                _this.classes[_index] = classe;
            });
        }
        /**
         * Miscellaneous update
         */
        if (updatedData.miscellaneous.variables.length > 0) {
            _.forEach(updatedData.miscellaneous.variables, function (variable) {
                var _index = _.findIndex(_this.miscellaneous.variables, {
                    name: variable.name,
                    file: variable.file
                });
                _this.miscellaneous.variables[_index] = variable;
            });
        }
        if (updatedData.miscellaneous.functions.length > 0) {
            _.forEach(updatedData.miscellaneous.functions, function (func) {
                var _index = _.findIndex(_this.miscellaneous.functions, {
                    name: func.name,
                    file: func.file
                });
                _this.miscellaneous.functions[_index] = func;
            });
        }
        if (updatedData.miscellaneous.typealiases.length > 0) {
            _.forEach(updatedData.miscellaneous.typealiases, function (typealias) {
                var _index = _.findIndex(_this.miscellaneous.typealiases, {
                    name: typealias.name,
                    file: typealias.file
                });
                _this.miscellaneous.typealiases[_index] = typealias;
            });
        }
        if (updatedData.miscellaneous.enumerations.length > 0) {
            _.forEach(updatedData.miscellaneous.enumerations, function (enumeration) {
                var _index = _.findIndex(_this.miscellaneous.enumerations, {
                    name: enumeration.name,
                    file: enumeration.file
                });
                _this.miscellaneous.enumerations[_index] = enumeration;
            });
        }
        this.prepareMiscellaneous();
    };
    DependenciesEngine.prototype.findInCompodoc = function (name) {
        var mergedData = _.concat([], this.modules, this.components, this.controllers, this.directives, this.injectables, this.interceptors, this.guards, this.interfaces, this.pipes, this.classes, this.miscellaneous.enumerations, this.miscellaneous.typealiases, this.miscellaneous.variables, this.miscellaneous.functions);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    DependenciesEngine.prototype.prepareMiscellaneous = function () {
        this.miscellaneous.variables.sort(getNamesCompareFn());
        this.miscellaneous.functions.sort(getNamesCompareFn());
        this.miscellaneous.enumerations.sort(getNamesCompareFn());
        this.miscellaneous.typealiases.sort(getNamesCompareFn());
        // group each subgoup by file
        this.miscellaneous.groupedVariables = _.groupBy(this.miscellaneous.variables, 'file');
        this.miscellaneous.groupedFunctions = _.groupBy(this.miscellaneous.functions, 'file');
        this.miscellaneous.groupedEnumerations = _.groupBy(this.miscellaneous.enumerations, 'file');
        this.miscellaneous.groupedTypeAliases = _.groupBy(this.miscellaneous.typealiases, 'file');
    };
    DependenciesEngine.prototype.getModule = function (name) {
        return _.find(this.modules, ['name', name]);
    };
    DependenciesEngine.prototype.getRawModule = function (name) {
        return _.find(this.rawModules, ['name', name]);
    };
    DependenciesEngine.prototype.getModules = function () {
        return this.modules;
    };
    DependenciesEngine.prototype.getComponents = function () {
        return this.components;
    };
    DependenciesEngine.prototype.getControllers = function () {
        return this.controllers;
    };
    DependenciesEngine.prototype.getDirectives = function () {
        return this.directives;
    };
    DependenciesEngine.prototype.getInjectables = function () {
        return this.injectables;
    };
    DependenciesEngine.prototype.getInterceptors = function () {
        return this.interceptors;
    };
    DependenciesEngine.prototype.getGuards = function () {
        return this.guards;
    };
    DependenciesEngine.prototype.getInterfaces = function () {
        return this.interfaces;
    };
    DependenciesEngine.prototype.getRoutes = function () {
        return this.routes;
    };
    DependenciesEngine.prototype.getPipes = function () {
        return this.pipes;
    };
    DependenciesEngine.prototype.getClasses = function () {
        return this.classes;
    };
    DependenciesEngine.prototype.getMiscellaneous = function () {
        return this.miscellaneous;
    };
    return DependenciesEngine;
}());
var DependenciesEngine$1 = DependenciesEngine.getInstance();

var FileEngine = /** @class */ (function () {
    function FileEngine() {
    }
    FileEngine.getInstance = function () {
        if (!FileEngine.instance) {
            FileEngine.instance = new FileEngine();
        }
        return FileEngine.instance;
    };
    FileEngine.prototype.get = function (filepath) {
        return new Promise(function (resolve, reject) {
            fs.readFile(path.resolve(filepath), 'utf8', function (err, data) {
                if (err) {
                    reject('Error during ' + filepath + ' read');
                }
                else {
                    resolve(data);
                }
            });
        });
    };
    FileEngine.prototype.write = function (filepath, contents) {
        return new Promise(function (resolve, reject) {
            fs.outputFile(path.resolve(filepath), contents, function (err) {
                if (err) {
                    reject(err);
                }
                else {
                    resolve();
                }
            });
        });
    };
    FileEngine.prototype.writeSync = function (filepath, contents) {
        fs.outputFileSync(filepath, contents);
    };
    FileEngine.prototype.getSync = function (filepath) {
        return fs.readFileSync(path.resolve(filepath), 'utf8');
    };
    /**
     * @param file The file to check
     */
    FileEngine.prototype.existsSync = function (file) {
        return fs.existsSync(file);
    };
    return FileEngine;
}());
var FileEngine$1 = FileEngine.getInstance();

var traverse$1 = require('traverse');
var ExportJsonEngine = /** @class */ (function () {
    function ExportJsonEngine() {
    }
    ExportJsonEngine.getInstance = function () {
        if (!ExportJsonEngine.instance) {
            ExportJsonEngine.instance = new ExportJsonEngine();
        }
        return ExportJsonEngine.instance;
    };
    ExportJsonEngine.prototype.export = function (outputFolder, data) {
        var exportData = {};
        traverse$1(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (Configuration$1.mainData.disableSourceCode) {
                    delete node.sourceCode;
                }
            }
        });
        exportData.pipes = data.pipes;
        exportData.interfaces = data.interfaces;
        exportData.injectables = data.injectables;
        exportData.classes = data.classes;
        exportData.directives = data.directives;
        exportData.components = data.components;
        exportData.modules = this.processModules();
        exportData.miscellaneous = data.miscellaneous;
        if (!Configuration$1.mainData.disableRoutesGraph) {
            exportData.routes = data.routes;
        }
        if (!Configuration$1.mainData.disableCoverage) {
            exportData.coverage = data.coverageData;
        }
        return FileEngine$1.write(outputFolder + path.sep + '/documentation.json', JSON.stringify(exportData, undefined, 4)).catch(function (err) {
            logger.error('Error during export file generation ', err);
            return Promise.reject(err);
        });
    };
    ExportJsonEngine.prototype.processModules = function () {
        var modules = DependenciesEngine$1.getModules();
        var _resultedModules = [];
        for (var moduleNr = 0; moduleNr < modules.length; moduleNr++) {
            var moduleElement = {
                name: modules[moduleNr].name,
                children: [
                    {
                        type: 'providers',
                        elements: []
                    },
                    {
                        type: 'declarations',
                        elements: []
                    },
                    {
                        type: 'imports',
                        elements: []
                    },
                    {
                        type: 'exports',
                        elements: []
                    },
                    {
                        type: 'bootstrap',
                        elements: []
                    },
                    {
                        type: 'classes',
                        elements: []
                    }
                ]
            };
            for (var k = 0; k < modules[moduleNr].providers.length; k++) {
                var providerElement = {
                    name: modules[moduleNr].providers[k].name
                };
                moduleElement.children[0].elements.push(providerElement);
            }
            for (var k = 0; k < modules[moduleNr].declarations.length; k++) {
                var declarationElement = {
                    name: modules[moduleNr].declarations[k].name
                };
                moduleElement.children[1].elements.push(declarationElement);
            }
            for (var k = 0; k < modules[moduleNr].imports.length; k++) {
                var importElement = {
                    name: modules[moduleNr].imports[k].name
                };
                moduleElement.children[2].elements.push(importElement);
            }
            for (var k = 0; k < modules[moduleNr].exports.length; k++) {
                var exportElement = {
                    name: modules[moduleNr].exports[k].name
                };
                moduleElement.children[3].elements.push(exportElement);
            }
            for (var k = 0; k < modules[moduleNr].bootstrap.length; k++) {
                var bootstrapElement = {
                    name: modules[moduleNr].bootstrap[k].name
                };
                moduleElement.children[4].elements.push(bootstrapElement);
            }
            _resultedModules.push(moduleElement);
        }
        return _resultedModules;
    };
    return ExportJsonEngine;
}());
var ExportJsonEngine$1 = ExportJsonEngine.getInstance();

var MarkdownToPDFEngine = /** @class */ (function () {
    function MarkdownToPDFEngine() {
        // console.log('MarkdownToPDFEngine');
        this.markedInstance = require('marked');
        this.renderer = new this.markedInstance.Renderer();
        this.renderer.strong = function (text) {
            console.log('MarkdownToPDFEngine strong: ', text);
            return { text: text, bold: true };
        };
        this.renderer.paragraph = function (text) {
            console.log('MarkdownToPDFEngine paragraph: ', text);
            return text;
        };
        // TODO Add custom parser... -> https://github.com/markedjs/marked/issues/504
        this.markedInstance.setOptions({
            gfm: false,
            breaks: false
        });
    }
    MarkdownToPDFEngine.getInstance = function () {
        if (!MarkdownToPDFEngine.instance) {
            MarkdownToPDFEngine.instance = new MarkdownToPDFEngine();
        }
        return MarkdownToPDFEngine.instance;
    };
    MarkdownToPDFEngine.prototype.convert = function (data) {
        // console.log('MarkdownToPDFEngine convert');
        /*const tokens = this.markedInstance.lexer('**This is bold text**');
        console.log(tokens);
        const html = this.markedInstance.parser(tokens);
        console.log(html);*/
        return this.markedInstance(data); // '**This is bold text**', { renderer: this.renderer });
    };
    return MarkdownToPDFEngine;
}());
var MarkdownToPdfEngine = MarkdownToPDFEngine.getInstance();

var PdfPrinter = require('pdfmake');
var ExportPdfEngine = /** @class */ (function () {
    function ExportPdfEngine() {
    }
    ExportPdfEngine.getInstance = function () {
        if (!ExportPdfEngine.instance) {
            ExportPdfEngine.instance = new ExportPdfEngine();
        }
        return ExportPdfEngine.instance;
    };
    ExportPdfEngine.prototype.export = function (outputFolder) {
        var fonts = {
            Roboto: {
                normal: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-regular.ttf'),
                bold: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-700.ttf')
            }
        };
        var printer = new PdfPrinter(fonts);
        var docDefinition = {
            info: {
                title: Configuration$1.mainData.documentationMainName
            },
            content: [],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    color: '#008cff',
                    margin: [0, 0, 0, 15]
                },
                subheader: {
                    fontSize: 15,
                    bold: true
                }
            }
        };
        docDefinition.content.push({
            text: Configuration$1.mainData.documentationMainName,
            alignment: 'center',
            bold: true,
            fontSize: 22,
            margin: [10, 350, 10, 300]
        });
        if (!Configuration$1.mainData.hideGenerator) {
            docDefinition.content.push({
                text: 'Documentation generated using',
                alignment: 'center'
            });
            docDefinition.content.push({
                image: "data:image/jpeg;base64,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",
                width: 70,
                alignment: 'center',
                pageBreak: 'after'
            });
        }
        docDefinition.content.push({
            toc: {
                title: {
                    text: 'Table of contents',
                    bold: true,
                    alignment: 'center',
                    fontSize: 18,
                    margin: [10, 10, 10, 50]
                },
                numberStyle: {
                    bold: true
                }
            },
            pageBreak: 'after'
        });
        // Add README page if available
        docDefinition.content.push(this.generateMarkdownContent());
        // Add CHANGELOG page if available
        // Add CONTRIBUTING page if available
        // Add LICENSE page if available
        // Add TODO page if available
        // Add Dependencies page if available
        // Add Additional pages if available
        docDefinition.content.push(this.generateModulesContent());
        docDefinition.content.push(this.generateComponentsContent());
        // Classes
        // Injectables
        // Interceptors
        // Guards
        // Interfaces
        // Pipes
        // Miscellaneous
        // Routes
        // Coverage - docDefinition.content.push(...this.coverageEngine.calculateTable());
        var pdfDoc = printer.createPdfKitDocument(docDefinition);
        return new Promise(function (resolve, reject) {
            fs.ensureFile(outputFolder + path.sep + 'documentation.pdf', function (err) {
                if (err) {
                    reject("Error during pdf generation: " + err);
                }
                else {
                    pdfDoc.pipe(fs.createWriteStream(outputFolder + path.sep + 'documentation.pdf'));
                    pdfDoc.end();
                    resolve();
                }
            });
        });
    };
    ExportPdfEngine.prototype.firstCharacterUpperCase = function (sentence) {
        return sentence.charAt(0).toUpperCase() + sentence.slice(1);
    };
    ExportPdfEngine.prototype.generateMarkdownContent = function () {
        var _this = this;
        var pages = Configuration$1.markDownPages;
        var data = [];
        pages.forEach(function (page) {
            data.push({
                text: "" + _this.firstCharacterUpperCase(page.name),
                tocItem: true,
                style: 'header'
            });
            data.push({
                text: MarkdownToPdfEngine.convert(page.markdown),
                margin: [0, 10]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.insertPageReturn = function (data) {
        data.push({
            text: " ",
            margin: [0, 0, 0, 20],
            pageBreak: 'after'
        });
    };
    ExportPdfEngine.prototype.generateModulesContent = function () {
        var data = [];
        data.push({
            text: 'Modules',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.modules, function (module) {
            data.push({
                text: "" + module.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: module.file
                    }
                ],
                margin: [0, 10]
            });
            if (module.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + module.rawdescription,
                    margin: [0, 5]
                });
            }
            if (module.declarations.length > 0) {
                data.push({
                    text: "Declarations :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_1 = { ul: [] };
                _.forEach(module.declarations, function (declaration) {
                    list_1.ul.push({
                        text: "" + declaration.name
                    });
                });
                data.push(list_1);
            }
            if (module.providers.length > 0) {
                data.push({
                    text: "Providers :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_2 = { ul: [] };
                _.forEach(module.providers, function (provider) {
                    list_2.ul.push({
                        text: "" + provider.name
                    });
                });
                data.push(list_2);
            }
            if (module.imports.length > 0) {
                data.push({
                    text: "Imports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_3 = { ul: [] };
                _.forEach(module.imports, function (importRef) {
                    list_3.ul.push({
                        text: "" + importRef.name
                    });
                });
                data.push(list_3);
            }
            if (module.exports.length > 0) {
                data.push({
                    text: "Exports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_4 = { ul: [] };
                _.forEach(module.exports, function (exportRef) {
                    list_4.ul.push({
                        text: "" + exportRef.name
                    });
                });
                data.push(list_4);
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.generateComponentsContent = function () {
        var data = [];
        data.push({
            text: 'Components',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.components, function (component) {
            data.push({
                text: "" + component.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: component.file
                    }
                ],
                margin: [0, 10]
            });
            if (component.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + component.rawdescription,
                    margin: [0, 5]
                });
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    return ExportPdfEngine;
}());
var ExportPdfEngine$1 = ExportPdfEngine.getInstance();

var ExportEngine = /** @class */ (function () {
    function ExportEngine() {
    }
    ExportEngine.getInstance = function () {
        if (!ExportEngine.instance) {
            ExportEngine.instance = new ExportEngine();
        }
        return ExportEngine.instance;
    };
    ExportEngine.prototype.export = function (outputFolder, data) {
        switch (Configuration$1.mainData.exportFormat) {
            case 'json':
                return ExportJsonEngine$1.export(outputFolder, data);
            case 'pdf':
                return ExportPdfEngine$1.export(outputFolder);
        }
    };
    return ExportEngine;
}());
var ExportEngine$1 = ExportEngine.getInstance();

/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */
/* global Reflect, Promise */

var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return extendStatics(d, b);
};

function __extends(d, b) {
    extendStatics(d, b);
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
}

var __assign = function() {
    __assign = Object.assign || function __assign(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};

var BreakCommaHelper = /** @class */ (function () {
    function BreakCommaHelper(bars) {
        this.bars = bars;
    }
    BreakCommaHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/,/g, ',<br>');
        return new Handlebars.SafeString(text);
    };
    return BreakCommaHelper;
}());

var BreakLinesHelper = /** @class */ (function () {
    function BreakLinesHelper(bars) {
        this.bars = bars;
    }
    BreakLinesHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/(\r\n|\n|\r)/gm, '<br>');
        text = text.replace(/ /gm, '&nbsp;');
        text = text.replace(/	/gm, '&nbsp;&nbsp;&nbsp;&nbsp;');
        return new Handlebars.SafeString(text);
    };
    return BreakLinesHelper;
}());

var CleanParagraphHelper = /** @class */ (function () {
    function CleanParagraphHelper() {
    }
    CleanParagraphHelper.prototype.helperFunc = function (context, text) {
        text = text.replace(/<p>/gm, '');
        text = text.replace(/<\/p>/gm, '');
        return new Handlebars.SafeString(text);
    };
    return CleanParagraphHelper;
}());

var CompareHelper = /** @class */ (function () {
    function CompareHelper() {
    }
    CompareHelper.prototype.helperFunc = function (context, a, operator, b, options) {
        if (arguments.length < 4) {
            throw new Error('handlebars Helper {{compare}} expects 4 arguments');
        }
        var result;
        switch (operator) {
            case 'indexof':
                result = b.indexOf(a) !== -1;
                break;
            case '===':
                result = a === b;
                break;
            case '!==':
                result = a !== b;
                break;
            case '>':
                result = a > b;
                break;
            default: {
                throw new Error('helper {{compare}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return CompareHelper;
}());

var DebugHelper = /** @class */ (function () {
    function DebugHelper() {
    }
    DebugHelper.prototype.helperFunc = function (context, optionalValue) {
        console.log('Current Context');
        console.log('====================');
        console.log(context);
        if (optionalValue) {
            console.log('OptionalValue');
            console.log('====================');
            console.log(optionalValue);
        }
    };
    return DebugHelper;
}());

var ElementAloneHelper = /** @class */ (function () {
    function ElementAloneHelper() {
    }
    ElementAloneHelper.prototype.helperFunc = function (context, elements, elementType, options) {
        var alones = [];
        var modules = DependenciesEngine$1.modules;
        elements.forEach(function (element) {
            var foundInOneModule = false;
            modules.forEach(function (module) {
                module.declarations.forEach(function (declaration) {
                    if (declaration.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (declaration.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.controllers.forEach(function (controller) {
                    if (controller.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (controller.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.providers.forEach(function (provider) {
                    if (provider.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (provider.file === element.file) {
                        foundInOneModule = true;
                    }
                });
            });
            if (!foundInOneModule) {
                alones.push(element);
            }
        });
        if (alones.length > 0) {
            switch (elementType) {
                case 'component':
                    context.components = alones;
                    break;
                case 'directive':
                    context.directives = alones;
                    break;
                case 'controller':
                    context.controllers = alones;
                    break;
                case 'injectable':
                    context.injectables = alones;
                    break;
                case 'pipe':
                    context.pipes = alones;
                    break;
            }
            return options.fn(context);
        }
    };
    return ElementAloneHelper;
}());

var EscapeSimpleQuoteHelper = /** @class */ (function () {
    function EscapeSimpleQuoteHelper() {
    }
    EscapeSimpleQuoteHelper.prototype.helperFunc = function (context, text) {
        if (!text) {
            return;
        }
        text = text.replace(/'/g, "\\'");
        text = text.replace(/(\r\n|\n|\r)/gm, '');
        return text;
    };
    return EscapeSimpleQuoteHelper;
}());

var FilterAngular2ModulesHelper = /** @class */ (function () {
    function FilterAngular2ModulesHelper() {
    }
    FilterAngular2ModulesHelper.prototype.helperFunc = function (context, text, options) {
        var NG2_MODULES = [
            'BrowserModule',
            'FormsModule',
            'HttpModule',
            'RouterModule'
        ];
        var len = NG2_MODULES.length;
        var i = 0;
        var result = false;
        for (i; i < len; i++) {
            if (text.indexOf(NG2_MODULES[i]) > -1) {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return FilterAngular2ModulesHelper;
}());

var AngularVersionUtil = /** @class */ (function () {
    function AngularVersionUtil() {
    }
    AngularVersionUtil.getInstance = function () {
        if (!AngularVersionUtil.instance) {
            AngularVersionUtil.instance = new AngularVersionUtil();
        }
        return AngularVersionUtil.instance;
    };
    AngularVersionUtil.prototype.cleanVersion = function (version) {
        return version
            .replace('~', '')
            .replace('^', '')
            .replace('=', '')
            .replace('<', '')
            .replace('>', '');
    };
    AngularVersionUtil.prototype.getAngularVersionOfProject = function (packageData) {
        var _result = '';
        if (packageData.dependencies) {
            var angularCore = packageData.dependencies[AngularVersionUtil.CorePackage];
            if (angularCore) {
                _result = this.cleanVersion(angularCore);
            }
        }
        return _result;
    };
    AngularVersionUtil.prototype.isAngularVersionArchived = function (version) {
        var result;
        try {
            result = semver.compare(version, '2.4.10') <= 0;
        }
        catch (e) { }
        return result;
    };
    AngularVersionUtil.prototype.prefixOfficialDoc = function (version) {
        return this.isAngularVersionArchived(version) ? 'v2.' : '';
    };
    AngularVersionUtil.prototype.getApiLink = function (api, angularVersion) {
        var angularDocPrefix = this.prefixOfficialDoc(angularVersion);
        return "https://" + angularDocPrefix + "angular.io/" + api.path;
    };
    AngularVersionUtil.CorePackage = '@angular/core';
    return AngularVersionUtil;
}());
var AngularVersionUtil$1 = AngularVersionUtil.getInstance();

var BasicTypes;
(function (BasicTypes) {
    BasicTypes[BasicTypes["number"] = 0] = "number";
    BasicTypes[BasicTypes["boolean"] = 1] = "boolean";
    BasicTypes[BasicTypes["string"] = 2] = "string";
    BasicTypes[BasicTypes["object"] = 3] = "object";
    BasicTypes[BasicTypes["date"] = 4] = "date";
    BasicTypes[BasicTypes["function"] = 5] = "function";
})(BasicTypes || (BasicTypes = {}));
var BasicTypeScriptTypes;
(function (BasicTypeScriptTypes) {
    BasicTypeScriptTypes[BasicTypeScriptTypes["any"] = 0] = "any";
    BasicTypeScriptTypes[BasicTypeScriptTypes["void"] = 1] = "void";
})(BasicTypeScriptTypes || (BasicTypeScriptTypes = {}));
var BasicTypeUtil = /** @class */ (function () {
    function BasicTypeUtil() {
    }
    BasicTypeUtil.getInstance = function () {
        if (!BasicTypeUtil.instance) {
            BasicTypeUtil.instance = new BasicTypeUtil();
        }
        return BasicTypeUtil.instance;
    };
    /**
     * Checks if a given types is a basic javascript type
     * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isJavascriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Checks if a given type is a typescript type (That is not a javascript type)
     * https://www.typescriptlang.org/docs/handbook/basic-types.html
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isTypeScriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypeScriptTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Check if the type is a typescript or javascript type
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isKnownType = function (type) {
        return this.isJavascriptType(type) || this.isTypeScriptType(type);
    };
    /**
     * Returns a official documentation link to either the javascript or typescript type
     * @param type The type to check
     * @returns The documentation link or undefined if type not found
     */
    BasicTypeUtil.prototype.getTypeUrl = function (type) {
        if (this.isJavascriptType(type)) {
            return "https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/" + type;
        }
        if (this.isTypeScriptType(type)) {
            return "https://www.typescriptlang.org/docs/handbook/basic-types.html";
        }
        return undefined;
    };
    return BasicTypeUtil;
}());
var BasicTypeUtil$1 = BasicTypeUtil.getInstance();

var FunctionSignatureHelper = /** @class */ (function () {
    function FunctionSignatureHelper() {
    }
    FunctionSignatureHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg.function.length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg.function.map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path = _result.data.type;
                    if (_result.data.type === 'class') {
                        path = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    FunctionSignatureHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    FunctionSignatureHelper.prototype.helperFunc = function (context, method) {
        var _this = this;
        var args = [];
        if (method.args) {
            args = method.args
                .map(function (arg) {
                var _result = DependenciesEngine$1.find(arg.type);
                if (_result) {
                    if (_result.source === 'internal') {
                        var path = _result.data.type;
                        if (_result.data.type === 'class') {
                            path = 'classe';
                        }
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                    }
                    else {
                        var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                    }
                }
                else if (arg.dotDotDotToken) {
                    return "..." + arg.name + ": " + arg.type;
                }
                else if (arg.function) {
                    return _this.handleFunction(arg);
                }
                else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                    var path = BasicTypeUtil$1.getTypeUrl(arg.type);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                }
                else {
                    if (arg.type) {
                        return "" + arg.name + _this.getOptionalString(arg) + ": " + arg.type;
                    }
                    else {
                        return "" + arg.name + _this.getOptionalString(arg);
                    }
                }
            })
                .join(', ');
        }
        if (method.name) {
            return method.name + "(" + args + ")";
        }
        else {
            return "(" + args + ")";
        }
    };
    return FunctionSignatureHelper;
}());

var HasOwnHelper = /** @class */ (function () {
    function HasOwnHelper() {
    }
    HasOwnHelper.prototype.helperFunc = function (context, entity, key, options) {
        if (Object.hasOwnProperty.call(entity, key)) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return HasOwnHelper;
}());

var TRANSLATION_EN_US = {
    accessors: 'Accessors',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Class',
    classes: 'Classes',
    component: 'Component',
    components: 'Components',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentation coverage',
    declarations: 'Declarations',
    decorators: 'Decorators',
    'default-value': 'Default value',
    'defined-in': 'Defined in',
    dependencies: 'Dependencies',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Example',
    exports: 'Exports',
    extends: 'Extends',
    file: 'File',
    functions: 'Functions',
    'generated-using': 'Documentation generated using',
    'getting-started': 'Getting started',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element with directive',
    identifier: 'Identifier',
    implements: 'Implements',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Inherited from',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'License',
    lines: 'Lines',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Miscellaneous',
    module: 'Module',
    modules: 'Modules',
    name: 'Name',
    no: 'No',
    'no-graph': 'No graph available.',
    'no-iframe': 'Your browser does not support iframes.',
    'no-result-matching': 'No results matching',
    'no-svg': 'Your browser does not support SVG',
    optional: 'Optional',
    outputs: 'Outputs',
    overview: 'Overview',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefix',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'results matching',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Type to search',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Unamed property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Value',
    variables: 'Variables',
    yes: 'Yes',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_ES_ES = {
    accessors: 'Accesorios',
    arguments: 'Argumentos',
    bootstrap: 'Arranque',
    branches: 'Ramas',
    browse: 'Navegar',
    classe: 'Clase',
    classes: 'Clases',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Constructor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura de la documentación',
    declarations: 'Declaraciones',
    decorators: 'Decoradores',
    'default-value': 'Valor por defecto',
    'defined-in': 'Definido en',
    dependencies: 'Dependencias',
    description: 'Descripción',
    directive: 'Directiva',
    directives: 'Directivas',
    entrycomponents: 'Componentes de entrada',
    enumerations: 'Enumeraciones',
    enums: 'Enums',
    example: 'Ejemplo',
    exports: 'Exporta',
    extends: 'Extiende',
    file: 'Fichero',
    functions: 'Funciones',
    'generated-using': 'Documentación generada utilizando',
    'getting-started': 'Comenzando',
    guard: 'Guardia',
    guards: 'Guardias',
    hostbindings: 'Fijaciones de Host',
    hostlisteners: 'Escuchadores de Host',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento Html con directiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Índice',
    indexable: 'Indexable',
    'inherited-from': 'Heredado desde',
    injectable: 'Inyectable',
    injectables: 'Inyectables',
    inputs: 'Entradas',
    interceptors: 'Interceptores',
    interface: 'Interfaz',
    interfaces: 'Interfaces',
    legend: 'Leyenda',
    license: 'Licencia',
    lines: 'Líneas',
    metadata: 'Meta datos',
    methods: 'Métodos',
    miscellaneous: 'Miscelánea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nombre',
    no: 'No',
    'no-graph': 'No hay gráfica disponible.',
    'no-iframe': 'Tu navegador no soporta iframes.',
    'no-result-matching': 'No hay resultados que coincidan',
    'no-svg': 'Tu navegador no soporta SVG',
    optional: 'Opcional',
    outputs: 'Salidas',
    overview: 'Descripción general',
    parameters: 'Parámetros',
    'peer-dependencies': 'Dependencias entre pares',
    pipe: 'Tubería',
    pipes: 'Tuberías',
    prefix: 'Prefijo',
    properties: 'Propiedades',
    providers: 'Proveedores',
    pure: 'Puro',
    readme: 'Léeme',
    reset: 'Restablecer',
    'results-matching': 'comparación de resultados',
    returns: 'Devuelve',
    route: 'Ruta',
    routes: 'Rutas',
    schemas: 'Esquemas',
    'search-placeholder': 'Escribe para buscar',
    selector: 'Selector',
    signature: 'Firma',
    statements: 'Declaraciones',
    type: 'Tipo',
    'type-aliases': 'Alias de tipo',
    'type-parameters': 'Parámetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propiedad sin nombre',
    'unit-test-coverage': 'Cobertura de las pruebas unitarias',
    value: 'Valor',
    variables: 'Variables',
    yes: 'Si',
    zoomin: 'Ampliar',
    zoomout: 'Alejar'
};

var TRANSLATION_FR_FR = {
    accessors: 'Accesseurs',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Parcourir',
    classe: 'Class',
    classes: 'Classes',
    component: 'Composant',
    components: 'Composants',
    constructor: 'Constructeur',
    controllers: 'Contrôleurs',
    controller: 'Contrôleur',
    'coverage-page-title': 'Couverture de documentation',
    declarations: 'Déclarations',
    decorators: 'Décorateurs',
    'default-value': 'Valeur par défaut',
    'defined-in': 'Défini dans',
    dependencies: 'Dépendances',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: "Composants d'entrée",
    enumerations: 'Enumérations',
    enums: 'Enumérations',
    example: 'Example',
    exports: 'Exports',
    extends: 'Etend',
    file: 'Fichier',
    functions: 'Fonctions',
    'generated-using': 'Documentation générée avec',
    'getting-started': 'Démarrage',
    guard: 'Garde',
    guards: 'Gardes',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elément Html',
    'html-element-with-directive': 'Elément Html avec une directive',
    identifier: 'Identifiant',
    implements: 'Implémente',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Hérité de',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Entrées',
    interceptors: 'Intercepteurs',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Légende',
    license: 'License',
    lines: 'Lignes',
    metadata: 'Métadonnées',
    methods: 'Méthodes',
    miscellaneous: 'Divers',
    module: 'Module',
    modules: 'Modules',
    name: 'Nom',
    no: 'Non',
    'no-graph': 'Aucun graphique disponible.',
    'no-iframe': 'Votre navigateur ne supporte pas les iframes.',
    'no-result-matching': 'Aucun résultat matchant',
    'no-svg': 'Votre navigateur ne supporte pas le SVG',
    optional: 'Optionnel',
    outputs: 'Sorties',
    overview: "Vue d'ensemble",
    parameters: 'Paramètres',
    'peer-dependencies': 'Dépendances de pair',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Préfixe',
    properties: 'Propriétés',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Remise à zéro',
    'results-matching': 'résultats matchant',
    returns: 'Renvoie',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schémas',
    'search-placeholder': 'Saisissez un texte',
    selector: 'Sélecteur',
    signature: 'Signature',
    statements: 'Déclarations',
    type: 'Type',
    'type-aliases': 'Alias de type',
    'type-parameters': 'Paramètres de type',
    types: 'Types',
    'unamed-property': 'Propriété non nommée',
    'unit-test-coverage': 'Couverture de test unitaire',
    value: 'Valeur',
    variables: 'Variables',
    yes: 'Oui',
    zoomin: 'Zoom avant',
    zoomout: 'Zoom arrière'
};

var TRANSLATION_HU_HU = {
    accessors: 'Getter/setter metódusok',
    arguments: 'Argumentumok',
    bootstrap: 'Betöltés',
    branches: 'Branchek',
    browse: 'Böngészés',
    classe: 'Osztály',
    classes: 'Osztályok',
    component: 'Komponens',
    components: 'Komponensek',
    constructor: 'Konstruktor',
    controllers: 'Kontrollerek',
    controller: 'Kontroller',
    'coverage-page-title': 'Dokumentáció lefedettség',
    declarations: 'Deklarációk',
    decorators: 'Dekorátorok',
    'default-value': 'Alapértelmezett érték',
    'defined-in': 'Definíció helye:',
    dependencies: 'Függőségek',
    description: 'Leírás',
    directive: 'Direktíva',
    directives: 'Direktívák',
    entrycomponents: 'Entry komponensek',
    enumerations: 'Enumerációk',
    enums: 'Enumok',
    example: 'Példa',
    exports: 'Exportok',
    extends: 'Ősosztály',
    file: 'File',
    functions: 'Függvények',
    'generated-using': 'A dokumentációt generálta:',
    'getting-started': 'Bevezető',
    guard: 'Guard',
    guards: 'Guardok',
    hostbindings: 'HostBindingok',
    hostlisteners: 'HostListenerek',
    'html-element': 'Html elem',
    'html-element-with-directive': 'Html elem direktívával',
    identifier: 'Azonosító',
    implements: 'Implementált interfészek',
    imports: 'Importok',
    index: 'Tartalomjegyzék',
    indexable: 'Indexelhető',
    'inherited-from': 'Örökölve innen:',
    injectable: 'Injektálható',
    injectables: 'Injektálhatók',
    inputs: 'Bemenetek',
    interceptors: 'Interceptorok',
    interface: 'Interfész',
    interfaces: 'Interfészek',
    legend: 'Jelmagyarázat',
    license: 'Licenc',
    lines: 'Sorok',
    metadata: 'Metaadatok',
    methods: 'Metódusok',
    miscellaneous: 'Egyéb',
    module: 'Modul',
    modules: 'Modulok',
    name: 'Név',
    no: 'Nem',
    'no-graph': 'Grafikon nem elérhető.',
    'no-iframe': 'A böngészője nem támogatja az iframe-eket.',
    'no-result-matching': 'Nincs találat',
    'no-svg': 'A böngészője nem támogatja az SVG formátumot.',
    optional: 'Opcionális',
    outputs: 'Kimenetek',
    overview: 'Áttekintés',
    parameters: 'Paraméterek',
    'peer-dependencies': 'Peer függőségek',
    pipe: 'Pipe',
    pipes: 'Pipe-ok',
    prefix: 'Előtag',
    properties: 'Tagváltozók',
    providers: 'Providerek',
    pure: 'Pure',
    readme: 'README',
    reset: 'Visszaállít',
    'results-matching': 'találat',
    returns: 'Visszatérési érték',
    route: 'Útvonal',
    routes: 'Útvonalak',
    schemas: 'Sémák',
    'search-placeholder': 'Keresendő kifejezés',
    selector: 'Szelektor',
    signature: 'Aláírás',
    statements: 'Utasítások',
    type: 'Típus',
    'type-aliases': 'Típus álnév',
    'type-parameters': 'Típus paraméterek',
    types: 'Típusok',
    'unamed-property': 'Névtelen property',
    'unit-test-coverage': 'Unit teszt lefedettség',
    value: 'Érték',
    variables: 'Változók',
    yes: 'Igen',
    zoomin: 'Nagyítás',
    zoomout: 'Kicsinyítés'
};

var TRANSLATION_IT_IT = {
    accessors: 'Accessori',
    arguments: 'Argomenti',
    bootstrap: 'Bootstrap',
    branches: 'Rami',
    browse: 'Cerca',
    classe: 'Classe',
    classes: 'Classi',
    component: 'Componente',
    components: 'Componenti',
    constructor: 'Costruttore',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Copertura codice',
    declarations: 'Dichiarazioni',
    decorators: 'Decorators',
    'default-value': 'Valore predefinito',
    'defined-in': 'Definito in',
    dependencies: 'Dependencies',
    description: 'Descrizione',
    directive: 'Direttiva',
    directives: 'Direttive',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Esempio',
    exports: 'Exports',
    extends: 'Extends',
    file: 'File',
    functions: 'Funzioni',
    'generated-using': 'Documentazione generata usando',
    'getting-started': 'Iniziamo',
    guard: 'Guardia',
    guards: 'Guardie',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento html con direttive',
    identifier: 'Identificatore',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Indice',
    indexable: 'Indicizzabile',
    'inherited-from': 'ereditato da',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Input',
    interceptors: 'Interceptors',
    interface: 'Interfaccia',
    interfaces: 'Interfacce',
    legend: 'Legenda',
    license: 'Licenza',
    lines: 'Linee',
    metadata: 'Metadati',
    methods: 'Metodi',
    miscellaneous: 'Varie',
    module: 'Modulo',
    modules: 'Moduli',
    name: 'Nome',
    no: 'No',
    'no-graph': 'Grafico non disponibile.',
    'no-iframe': 'Il tuo browser non supporta iframe.',
    'no-result-matching': 'Nessun risultato corrispondente',
    'no-svg': 'Il tuo browser non supporta SVG',
    optional: 'Opzionale',
    outputs: 'Output',
    overview: 'Sommario',
    parameters: 'Parametri',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefisso',
    properties: 'Proprietà',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'corrispondenza',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Digita per avviare la ricerca',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Tipi',
    'unamed-property': 'Proprietà senza nome',
    'unit-test-coverage': 'Copertura unit test',
    value: 'Valori',
    variables: 'Variabili',
    yes: 'Si',
    zoomin: 'Ingrandisci',
    zoomout: 'Rimpocciolisci'
};

var TRANSLATION_JA_JP = {
    accessors: 'アクセサ',
    arguments: '引数',
    bootstrap: 'ブートストラップ',
    branches: 'ブランチ',
    browse: 'ブラウズ',
    classe: 'クラス',
    classes: 'クラス',
    component: 'コンポーネント',
    components: 'コンポーネント',
    constructor: 'コンストラクタ',
    controllers: 'コントローラー',
    controller: 'コントローラー',
    'coverage-page-title': 'カバレッジ',
    declarations: '宣言',
    decorators: 'デコレーター',
    'default-value': '初期値',
    'defined-in': 'Defined in',
    dependencies: '依存関係',
    description: '説明',
    directive: 'ディレクティブ',
    directives: 'ディレクティブ',
    entrycomponents: 'エントリーコンポーネント',
    enumerations: '列挙型',
    enums: 'Enums',
    example: '例',
    exports: 'エクスポート',
    extends: '継承',
    file: 'ファイル',
    functions: '関数',
    'generated-using': 'このドキュメントは以下を使用して生成されています',
    'getting-started': 'はじめに',
    guard: 'ガード',
    guards: 'ガード',
    hostbindings: 'ホストバインディング',
    hostlisteners: 'ホストリスナー',
    'html-element': 'Html要素',
    'html-element-with-directive': 'ディレクティブHtml要素',
    identifier: '識別子',
    implements: '実装',
    imports: 'インポート',
    index: '索引',
    indexable: 'インデクサブル',
    'inherited-from': 'Inherited from',
    injectable: 'インジェクタブル',
    injectables: 'インジェクタブル',
    inputs: '入力',
    interceptors: 'インターセプター',
    interface: 'インターフェイス',
    interfaces: 'インターフェイス',
    legend: '凡例',
    license: 'ライセンス',
    lines: '行数',
    metadata: 'メタデータ',
    methods: 'メソッド',
    miscellaneous: 'その他',
    module: 'モジュール',
    modules: 'モジュール',
    name: '名前',
    no: 'いいえ',
    'no-graph': '使用できるグラフがありません',
    'no-iframe': 'ブラウザがiframeを対応していません',
    'no-result-matching': '見つかりませんでした',
    'no-svg': 'ブラウザがSVGに対応してません',
    optional: 'オプション',
    outputs: '出力',
    overview: '概要',
    parameters: 'パラメータ',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'パイプ',
    pipes: 'パイプ',
    prefix: '接頭辞',
    properties: 'プロパティ',
    providers: 'プロバイダー',
    pure: 'Pure',
    readme: 'README',
    reset: 'リセット',
    'results-matching': '件の結果が一致しました',
    returns: '戻り値',
    route: 'ルート',
    routes: 'ルート',
    schemas: 'スキーマ',
    'search-placeholder': '入力して検索',
    selector: 'セレクタ',
    signature: 'シグネチャ',
    statements: '文',
    type: '型',
    'type-aliases': 'タイプエイリアス',
    'type-parameters': '型パラメーター',
    types: '型',
    'unamed-property': '匿名プロパティ',
    'unit-test-coverage': 'ユニットテストカバレッジ',
    value: '値',
    variables: '変数',
    yes: 'はい',
    zoomin: '拡大',
    zoomout: '縮小'
};

var TRANSLATION_PT_BR = {
    accessors: 'Acessores',
    arguments: 'Argumentos',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Navegar',
    classe: 'Classe',
    classes: 'Classes',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Construtor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura da documentação',
    declarations: 'Declarações',
    decorators: 'Decoradores',
    'default-value': 'Valor padrão',
    'defined-in': 'Definido em',
    dependencies: 'Dependências',
    description: 'Descrição',
    directive: 'Diretiva',
    directives: 'Diretivas',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerações',
    enums: 'Enums',
    example: 'Exemplo',
    exports: 'Exports',
    extends: 'Extende',
    file: 'Arquivo',
    functions: 'Funções',
    'generated-using': 'Documentação gerada usando',
    'getting-started': 'Começando',
    guard: 'Guarda',
    guards: 'Guardas',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento HTML',
    'html-element-with-directive': 'Elemento HTML com diretiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexável',
    'inherited-from': 'Herdado de',
    injectable: 'Injetável',
    injectables: 'Injetáveis',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'Licença',
    lines: 'Linhas',
    metadata: 'Metadata',
    methods: 'Métodos',
    miscellaneous: 'Miscelânea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nome',
    no: 'Não',
    'no-graph': 'Sem gráfico disponível.',
    'no-iframe': 'Seu browser não tem suporte a iframes.',
    'no-result-matching': 'Nenhum resultado correspondente',
    'no-svg': 'Seu browser não tem suporte a SVG',
    optional: 'Opcional',
    outputs: 'Outputs',
    overview: 'Visão geral',
    parameters: 'Parâmetros',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefixo',
    properties: 'Propriedades',
    providers: 'Providers',
    pure: 'Puro',
    readme: 'README',
    reset: 'Resetar',
    'results-matching': 'resultados correspondentes',
    returns: 'Retorna',
    route: 'Rota',
    routes: 'Rotas',
    schemas: 'Esquemas',
    'search-placeholder': 'Digite para pesquisar',
    selector: 'Seletor',
    signature: 'Assinatura',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Aliases de tipo',
    'type-parameters': 'Parâmetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propriedade não-nomeada',
    'unit-test-coverage': 'Cobertua de teste unitário',
    value: 'Valor',
    variables: 'Variáveis',
    yes: 'Sim',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_ZH_CN = {
    accessors: '存取器',
    arguments: 'Arguments',
    bootstrap: '根组件',
    branches: '分支',
    browse: '查看',
    classe: '类',
    classes: '类列表',
    component: '组件',
    components: '组件列表',
    constructor: '构造方法',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': '文档概览',
    declarations: '可声明对象列表',
    decorators: '装饰器列表',
    'default-value': '缺省值',
    'defined-in': '被定义在',
    dependencies: '依赖项',
    description: '描述',
    directive: '指令',
    directives: '指令列表',
    entrycomponents: '入口组件列表',
    enumerations: '列举',
    enums: '枚举列表',
    example: '例子',
    exports: '导出',
    extends: '继承',
    file: '文件',
    functions: '函数',
    'generated-using': '文档生成使用',
    'getting-started': '入门指南',
    guard: '路由守卫',
    guards: '路由守卫列表',
    hostbindings: '宿主绑定',
    hostlisteners: '宿主监听',
    'html-element': 'Html 元素',
    'html-element-with-directive': '带指令的Html元素',
    identifier: '标识符',
    implements: '实现',
    imports: '引入',
    index: '索引',
    indexable: 'Indexable',
    'inherited-from': '继承自',
    injectable: '可注入的',
    injectables: '可注入的',
    inputs: '输入属性',
    interceptors: '拦截器',
    interface: '接口',
    interfaces: '接口',
    legend: '图例',
    license: '许可协议',
    lines: 'Lines',
    metadata: '元数据',
    methods: '方法',
    miscellaneous: '其他',
    module: '模块',
    modules: '模块列表',
    name: '名称',
    no: '否',
    'no-graph': '无数据显示',
    'no-iframe': '你的浏览器不支持iframes',
    'no-result-matching': '无匹配的结果',
    'no-svg': '你的浏览器不支持SVG',
    optional: '可选的',
    outputs: '输出属性',
    overview: '概述',
    parameters: '参数列表',
    'peer-dependencies': '同级依赖',
    pipe: '管道',
    pipes: '管道列表',
    prefix: '字首',
    properties: '属性列表',
    providers: '提供商列表',
    pure: 'Pure',
    readme: '手册',
    reset: '重置',
    'results-matching': '匹配的结果',
    returns: '返回',
    route: '路由',
    routes: '路由列表',
    schemas: '模式',
    'search-placeholder': '请输入查询关键字',
    selector: '选择器',
    signature: '签名',
    statements: '注释',
    type: '类型',
    'type-aliases': '类型别名',
    'type-parameters': '类型参数',
    types: '类型',
    'unamed-property': '未命名属性',
    'unit-test-coverage': '单元测试概览',
    value: '值',
    variables: '变量',
    yes: '是',
    zoomin: '放大',
    zoomout: '缩小'
};

var TRANSLATION_NL_NL = {
    accessors: 'Accessors',
    arguments: 'Argumenten',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Klasse',
    classes: 'Klassen',
    component: 'Component',
    components: 'Componenten',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentatie coverage',
    declarations: 'Declaraties',
    decorators: 'Decorators',
    'default-value': 'Default waarde',
    'defined-in': 'Gedefinieerd in',
    dependencies: 'Dependencies',
    description: 'Omschrijving',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Voorbeeld',
    exports: 'Exports',
    extends: 'Extends',
    file: 'Bestand',
    functions: 'Functies',
    'generated-using': 'Documentatie gegenereed met',
    'getting-started': 'Aan de slag',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element met directive',
    identifier: 'Identifier',
    implements: 'Implementeert',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexeerbaar',
    'inherited-from': 'Inherited van',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legenda',
    license: 'Licentie',
    lines: 'Regels',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Diversen',
    module: 'Module',
    modules: 'Modules',
    name: 'Naam',
    no: 'Nee',
    'no-graph': 'Geen diagram beschikbaar.',
    'no-iframe': 'Uw browser ondersteund geen iframes.',
    'no-result-matching': 'Geen overeenkomende resultaten',
    'no-svg': 'Uw browser ondersteund geen SVG',
    optional: 'Optioneel',
    outputs: 'Outputs',
    overview: 'Overzicht',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Voorvoegsel',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Puur',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'overeenkomende resultaten',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: "Schema's",
    'search-placeholder': 'Type om te zoeken',
    selector: 'Selector',
    signature: 'Handtekening',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliassen',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Naamloze property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Waarde',
    variables: 'Variabelen',
    yes: 'Ja',
    zoomin: 'Zoom in',
    zoomout: 'Zoom uit'
};

var I18nEngine = /** @class */ (function () {
    function I18nEngine() {
        this.availablesLanguages = {
            'en-US': 'en-US',
            'es-ES': 'es-ES',
            'fr-FR': 'fr-FR',
            'hu-HU': 'hu-HU',
            'it-IT': 'it-IT',
            'ja-JP': 'ja-JP',
            'nl-NL': 'nl-NL',
            'pt-BR': 'pt-BR',
            'zh-CN': 'zh-CN'
        };
        this.fallbackLanguage = 'en-US';
    }
    I18nEngine.getInstance = function () {
        if (!I18nEngine.instance) {
            I18nEngine.instance = new I18nEngine();
        }
        return I18nEngine.instance;
    };
    I18nEngine.prototype.init = function (language) {
        i18next.init({
            lng: language,
            fallbackLng: this.fallbackLanguage
        });
        i18next.addResources('en-US', 'translation', TRANSLATION_EN_US);
        i18next.addResources('es-ES', 'translation', TRANSLATION_ES_ES);
        i18next.addResources('fr-FR', 'translation', TRANSLATION_FR_FR);
        i18next.addResources('hu-HU', 'translation', TRANSLATION_HU_HU);
        i18next.addResources('it-IT', 'translation', TRANSLATION_IT_IT);
        i18next.addResources('ja-JP', 'translation', TRANSLATION_JA_JP);
        i18next.addResources('nl-NL', 'translation', TRANSLATION_NL_NL);
        i18next.addResources('pt-BR', 'translation', TRANSLATION_PT_BR);
        i18next.addResources('zh-CN', 'translation', TRANSLATION_ZH_CN);
    };
    I18nEngine.prototype.translate = function (key) {
        return i18next.t(key);
    };
    I18nEngine.prototype.supportLanguage = function (language) {
        return typeof this.availablesLanguages[language] !== 'undefined';
    };
    return I18nEngine;
}());
var I18nEngine$1 = I18nEngine.getInstance();

var I18nHelper = /** @class */ (function () {
    function I18nHelper() {
    }
    I18nHelper.prototype.helperFunc = function (context, i18n_key, options) {
        var result = I18nEngine$1.translate(i18n_key);
        return result;
    };
    return I18nHelper;
}());

var IfStringHelper = /** @class */ (function () {
    function IfStringHelper() {
    }
    IfStringHelper.prototype.helperFunc = function (context, a, options) {
        if (typeof a === 'string') {
            return options.fn(context);
        }
        return options.inverse(context);
    };
    return IfStringHelper;
}());

var IndexableSignatureHelper = /** @class */ (function () {
    function IndexableSignatureHelper() {
    }
    IndexableSignatureHelper.prototype.helperFunc = function (context, method) {
        var args = method.args.map(function (arg) { return arg.name + ": " + arg.type; }).join(', ');
        if (method.name) {
            return method.name + "[" + args + "]";
        }
        else {
            return "[" + args + "]";
        }
    };
    return IndexableSignatureHelper;
}());

var IsInitialTabHelper = /** @class */ (function () {
    function IsInitialTabHelper() {
    }
    IsInitialTabHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        return tabs[0].id === tabId ? options.fn(context) : options.inverse(context);
    };
    return IsInitialTabHelper;
}());

var IsNotToggleHelper = /** @class */ (function () {
    function IsNotToggleHelper() {
    }
    IsNotToggleHelper.prototype.helperFunc = function (context, type, options) {
        var result = Configuration$1.mainData.toggleMenuItems.indexOf(type);
        if (Configuration$1.mainData.toggleMenuItems.indexOf('all') !== -1) {
            return options.inverse(context);
        }
        else if (result !== -1) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return IsNotToggleHelper;
}());

var IsTabEnabledHelper = /** @class */ (function () {
    function IsTabEnabledHelper() {
    }
    IsTabEnabledHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        var isTabEnabled = -1 !== _.findIndex(tabs, { id: tabId });
        return isTabEnabled ? options.fn(context) : options.inverse(context);
    };
    return IsTabEnabledHelper;
}());

var JsdocCodeExampleHelper = /** @class */ (function () {
    function JsdocCodeExampleHelper() {
    }
    JsdocCodeExampleHelper.prototype.cleanTag = function (comment) {
        if (comment.charAt(0) === '*') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.charAt(0) === ' ') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.indexOf('<p>') === 0) {
            comment = comment.substring(3, comment.length);
        }
        if (comment.substr(-1) === '\n') {
            comment = comment.substring(0, comment.length - 1);
        }
        if (comment.substr(-4) === '</p>') {
            comment = comment.substring(0, comment.length - 4);
        }
        return comment;
    };
    JsdocCodeExampleHelper.prototype.getHtmlEntities = function (str) {
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;');
    };
    JsdocCodeExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        var type = 'html';
        if (options.hash.type) {
            type = options.hash.type;
        }
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        if (jsdocTags[i].comment.indexOf('<caption>') !== -1) {
                            tag.comment = jsdocTags[i].comment
                                .replace(/<caption>/g, '<b><i>')
                                .replace(/\/caption>/g, '/b></i>');
                        }
                        else {
                            tag.comment =
                                "<pre class=\"line-numbers\"><code class=\"language-" + type + "\">" +
                                    this.getHtmlEntities(this.cleanTag(jsdocTags[i].comment)) +
                                    "</code></pre>";
                        }
                        tags.push(tag);
                    }
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocCodeExampleHelper;
}());

var JsdocDefaultHelper = /** @class */ (function () {
    function JsdocDefaultHelper() {
    }
    JsdocDefaultHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        if (jsdocTags) {
            var i = 0;
            var len = jsdocTags.length;
            var tag = {};
            var defaultValue = false;
            for (i; i < len; i++) {
                if (jsdocTags[i].tagName) {
                    if (jsdocTags[i].tagName.text === 'default') {
                        defaultValue = true;
                        if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                            tag.type = jsdocTags[i].typeExpression.type.name.text;
                        }
                        if (jsdocTags[i].comment) {
                            tag.comment = jsdocTags[i].comment;
                        }
                        if (jsdocTags[i].name) {
                            tag.name = jsdocTags[i].name.text;
                        }
                    }
                }
            }
            if (defaultValue) {
                context.tag = tag;
                return options.fn(context);
            }
        }
    };
    return JsdocDefaultHelper;
}());

var JsdocExampleHelper = /** @class */ (function () {
    function JsdocExampleHelper() {
    }
    JsdocExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment
                            .replace(/<caption>/g, '<b><i>')
                            .replace(/\/caption>/g, '/b></i>');
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocExampleHelper;
}());

var JsdocParamsValidHelper = /** @class */ (function () {
    function JsdocParamsValidHelper() {
    }
    JsdocParamsValidHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var valid = false;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    valid = true;
                }
            }
        }
        if (valid) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return JsdocParamsValidHelper;
}());

var JsdocParamsHelper = /** @class */ (function () {
    function JsdocParamsHelper() {
    }
    JsdocParamsHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    var tag = {};
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.kind) {
                        tag.type = kindToType(jsdocTags[i].typeExpression.type.kind);
                    }
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                        tag.type = jsdocTags[i].typeExpression.type.name.text;
                    }
                    else {
                        tag.type = jsdocTags[i].type;
                    }
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment;
                    }
                    if (jsdocTags[i].defaultValue) {
                        tag.defaultValue = jsdocTags[i].defaultValue;
                    }
                    if (jsdocTags[i].name) {
                        if (jsdocTags[i].name.text) {
                            tag.name = jsdocTags[i].name.text;
                        }
                        else {
                            tag.name = jsdocTags[i].name;
                        }
                    }
                    if (jsdocTags[i].optional) {
                        tag.optional = true;
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length >= 1) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocParamsHelper;
}());

var JsdocReturnsCommentHelper = /** @class */ (function () {
    function JsdocReturnsCommentHelper() {
    }
    JsdocReturnsCommentHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var result;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'returns' ||
                    jsdocTags[i].tagName.text === 'return') {
                    result = jsdocTags[i].comment;
                    break;
                }
            }
        }
        return result;
    };
    return JsdocReturnsCommentHelper;
}());

var LinkTypeHelper = /** @class */ (function () {
    function LinkTypeHelper() {
    }
    LinkTypeHelper.prototype.helperFunc = function (context, name, options) {
        var _result = DependenciesEngine$1.find(name);
        var angularDocPrefix = AngularVersionUtil$1.prefixOfficialDoc(Configuration$1.mainData.angularVersion);
        if (_result) {
            context.type = {
                raw: name
            };
            if (_result.source === 'internal') {
                if (_result.data.type === 'class') {
                    _result.data.type = 'classe';
                }
                context.type.href = '../' + _result.data.type + 's/' + _result.data.name + '.html';
                if (_result.data.type === 'miscellaneous' ||
                    (_result.data.ctype && _result.data.ctype === 'miscellaneous')) {
                    var mainpage = '';
                    switch (_result.data.subtype) {
                        case 'enum':
                            mainpage = 'enumerations';
                            break;
                        case 'function':
                            mainpage = 'functions';
                            break;
                        case 'typealias':
                            mainpage = 'typealiases';
                            break;
                        case 'variable':
                            mainpage = 'variables';
                    }
                    context.type.href =
                        '../' + _result.data.ctype + '/' + mainpage + '.html#' + _result.data.name;
                }
                context.type.target = '_self';
            }
            else {
                context.type.href = "https://" + angularDocPrefix + "angular.io/" + _result.data.path;
                context.type.target = '_blank';
            }
            return options.fn(context);
        }
        else if (BasicTypeUtil$1.isKnownType(name)) {
            context.type = {
                raw: name
            };
            context.type.target = '_blank';
            context.type.href = BasicTypeUtil$1.getTypeUrl(name);
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return LinkTypeHelper;
}());

var ModifIconHelper = /** @class */ (function () {
    function ModifIconHelper() {
    }
    ModifIconHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'lock'; // private
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'lock'; // protected
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'reset'; // static
                break;
            case Ast.SyntaxKind.ExportKeyword:
                _kindText = 'export'; // export
                break;
            default:
                _kindText = 'reset';
                break;
        }
        return _kindText;
    };
    return ModifIconHelper;
}());

var ModifKindHelper = /** @class */ (function () {
    function ModifKindHelper() {
    }
    /**
     * Transform SyntaxKind into string
     * @param  {any}           context Handlebars context
     * @param  {SyntaxKind[]} kind  SyntaxKind concatenated
     * @return {string}                Parsed string
     */
    ModifKindHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'Private';
                break;
            case Ast.SyntaxKind.ReadonlyKeyword:
                _kindText = 'Readonly';
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'Protected';
                break;
            case Ast.SyntaxKind.PublicKeyword:
                _kindText = 'Public';
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'Static';
                break;
            case Ast.SyntaxKind.AsyncKeyword:
                _kindText = 'Async';
                break;
            case Ast.SyntaxKind.AbstractKeyword:
                _kindText = 'Abstract';
                break;
        }
        return new Handlebars.SafeString(_kindText);
    };
    return ModifKindHelper;
}());

var ObjectLengthHelper = /** @class */ (function () {
    function ObjectLengthHelper() {
    }
    ObjectLengthHelper.prototype.helperFunc = function (context, obj, operator, length) {
        var len = arguments.length - 1;
        var options = arguments[len];
        if (typeof obj !== 'object') {
            return options.inverse(context);
        }
        var size = 0, key;
        for (key in obj) {
            if (obj.hasOwnProperty(key)) {
                size++;
            }
        }
        var result;
        switch (operator) {
            case '===':
                result = size === length;
                break;
            case '!==':
                result = size !== length;
                break;
            case '>':
                result = size > length;
                break;
            default: {
                throw new Error('helper {{objectLength}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return ObjectLengthHelper;
}());

var ObjectHelper = /** @class */ (function () {
    function ObjectHelper() {
    }
    ObjectHelper.prototype.helperFunc = function (context, text) {
        text = JSON.stringify(text);
        text = text.replace(/{"/, '{<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/,"/, ',<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/}$/, '<br>}');
        return new Handlebars.SafeString(text);
    };
    return ObjectHelper;
}());

var OneParameterHasHelper = /** @class */ (function () {
    function OneParameterHasHelper() {
    }
    OneParameterHasHelper.prototype.helperFunc = function (context, tags, typeToCheck) {
        var result = false;
        var len = arguments.length - 1;
        var options = arguments[len];
        var i = 0, leng = tags.length;
        for (i; i < leng; i++) {
            if (typeof tags[i][typeToCheck] !== 'undefined' && tags[i][typeToCheck] !== '') {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return OneParameterHasHelper;
}());

var OrLengthHelper = /** @class */ (function () {
    function OrLengthHelper() {
    }
    OrLengthHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (typeof arguments[i] !== 'undefined') {
                if (Object.keys(arguments[i]).length > 0) {
                    return options.fn(context);
                }
            }
        }
        return options.inverse(context);
    };
    return OrLengthHelper;
}());

var OrHelper = /** @class */ (function () {
    function OrHelper() {
    }
    OrHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (arguments[i]) {
                return options.fn(context);
            }
        }
        return options.inverse(context);
    };
    return OrHelper;
}());

var ParseDescriptionHelper = /** @class */ (function () {
    function ParseDescriptionHelper() {
    }
    ParseDescriptionHelper.prototype.helperFunc = function (context, description, depth) {
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExp;
        var matches;
        var previousString;
        tagRegExp = description.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        var processTheLink = function (originalDescription, matchedTag, leadingText) {
            var leading = extractLeadingText(originalDescription, matchedTag.completeTag);
            var split;
            var resultInCompodoc;
            var newLink;
            var rootPath;
            var stringtoReplace;
            var anchor = '';
            var label;
            var pageName;
            split = splitLinkText(matchedTag.text);
            if (typeof split.linkText !== 'undefined') {
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(split.target);
            }
            else {
                var info = matchedTag.text;
                if (matchedTag.text.indexOf('#') !== -1) {
                    anchor = matchedTag.text.substr(matchedTag.text.indexOf('#'), matchedTag.text.length);
                    info = matchedTag.text.substr(0, matchedTag.text.indexOf('#'));
                }
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(info);
            }
            if (resultInCompodoc) {
                label = resultInCompodoc.name;
                pageName = resultInCompodoc.name;
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                if (resultInCompodoc.type === 'class') {
                    resultInCompodoc.type = 'classe';
                }
                else if (resultInCompodoc.type === 'miscellaneous' ||
                    (resultInCompodoc.ctype && resultInCompodoc.ctype === 'miscellaneous')) {
                    resultInCompodoc.type = 'miscellaneou'; // Not a typo, it is for matching other single types : component, module etc
                    label = resultInCompodoc.name;
                    anchor = '#' + resultInCompodoc.name;
                    if (resultInCompodoc.subtype === 'enum') {
                        pageName = 'enumerations';
                    }
                    else if (resultInCompodoc.subtype === 'function') {
                        pageName = 'functions';
                    }
                    else if (resultInCompodoc.subtype === 'typealias') {
                        pageName = 'typealiases';
                    }
                    else if (resultInCompodoc.subtype === 'variable') {
                        pageName = 'variables';
                    }
                }
                rootPath = '';
                switch (depth) {
                    case 0:
                        rootPath = './';
                        break;
                    case 1:
                    case 2:
                    case 3:
                    case 4:
                    case 5:
                        rootPath = '../'.repeat(depth);
                        break;
                }
                if (leading.leadingText !== undefined) {
                    label = leading.leadingText;
                }
                if (typeof split.linkText !== 'undefined') {
                    label = split.linkText;
                }
                newLink = "<a href=\"" + rootPath + resultInCompodoc.type + "s/" + pageName + ".html" + anchor + "\">" + label + "</a>";
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.linkText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && leading && typeof leading.leadingText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + leading.leadingText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText === 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.target + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else {
                return originalDescription;
            }
        };
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(description, matchedTag, linkText);
            }
            else {
                return replacer(description, matchedTag);
            }
        }
        // Clean description for marked a tag parsed too early
        if (description.indexOf('href=') !== -1) {
            var insideMarkedATagResults = description.match(/<a [^>]+>([^<]+)<\/a>/g);
            if (insideMarkedATagResults && insideMarkedATagResults.length > 0) {
                for (var i = 0; i < insideMarkedATagResults.length; i++) {
                    var markedATagRegExp = new RegExp('<a [^>]+>([^<]+)</a>', 'gm');
                    var parsedATag = markedATagRegExp.exec(description);
                    if (parsedATag && parsedATag.length === 2) {
                        var insideMarkedATag = parsedATag[1];
                        description = description.replace("{@link <a href=\"" + encodeURI(insideMarkedATag) + "\">" + insideMarkedATag + "</a>", "{@link " + insideMarkedATag);
                    }
                }
            }
        }
        do {
            matches = tagRegExp.exec(description);
            // Did we have {@link ?
            if (matches) {
                previousString = description;
                if (matches.length === 2) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== description);
        return description;
    };
    return ParseDescriptionHelper;
}());

var RelativeURLHelper = /** @class */ (function () {
    function RelativeURLHelper() {
    }
    RelativeURLHelper.prototype.helperFunc = function (context, currentDepth, options) {
        switch (currentDepth) {
            case 0:
                return './';
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
                return '../'.repeat(currentDepth);
        }
        return '';
    };
    return RelativeURLHelper;
}());

var ShortURLHelper = /** @class */ (function () {
    function ShortURLHelper() {
    }
    ShortURLHelper.prototype.helperFunc = function (context, url, options) {
        var newUrl = url;
        var firstIndexOfSlash = newUrl.indexOf('/');
        var lastIndexOfSlash = newUrl.lastIndexOf('/');
        if (firstIndexOfSlash !== -1 || lastIndexOfSlash !== -1) {
            newUrl =
                newUrl.substr(0, firstIndexOfSlash + 1) +
                    '...' +
                    newUrl.substr(lastIndexOfSlash, newUrl.length);
        }
        return newUrl;
    };
    return ShortURLHelper;
}());

var StripURLHelper = /** @class */ (function () {
    function StripURLHelper() {
    }
    StripURLHelper.prototype.helperFunc = function (context, prefix, url, options) {
        return prefix + url.split("/").pop();
    };
    return StripURLHelper;
}());

var HtmlEngineHelpers = /** @class */ (function () {
    function HtmlEngineHelpers() {
    }
    HtmlEngineHelpers.prototype.registerHelpers = function (bars) {
        this.registerHelper(bars, 'compare', new CompareHelper());
        this.registerHelper(bars, 'or', new OrHelper());
        this.registerHelper(bars, 'functionSignature', new FunctionSignatureHelper());
        this.registerHelper(bars, 'isNotToggle', new IsNotToggleHelper());
        this.registerHelper(bars, 'isInitialTab', new IsInitialTabHelper());
        this.registerHelper(bars, 'isTabEnabled', new IsTabEnabledHelper());
        this.registerHelper(bars, 'ifString', new IfStringHelper());
        this.registerHelper(bars, 'orLength', new OrLengthHelper());
        this.registerHelper(bars, 'filterAngular2Modules', new FilterAngular2ModulesHelper());
        this.registerHelper(bars, 'debug', new DebugHelper());
        this.registerHelper(bars, 'breaklines', new BreakLinesHelper(bars));
        this.registerHelper(bars, 'clean-paragraph', new CleanParagraphHelper());
        this.registerHelper(bars, 'escapeSimpleQuote', new EscapeSimpleQuoteHelper());
        this.registerHelper(bars, 'breakComma', new BreakCommaHelper(bars));
        this.registerHelper(bars, 'modifKind', new ModifKindHelper());
        this.registerHelper(bars, 'modifIcon', new ModifIconHelper());
        this.registerHelper(bars, 'relativeURL', new RelativeURLHelper());
        this.registerHelper(bars, 'jsdoc-returns-comment', new JsdocReturnsCommentHelper());
        this.registerHelper(bars, 'jsdoc-code-example', new JsdocCodeExampleHelper());
        this.registerHelper(bars, 'jsdoc-example', new JsdocExampleHelper());
        this.registerHelper(bars, 'jsdoc-params', new JsdocParamsHelper());
        this.registerHelper(bars, 'jsdoc-params-valid', new JsdocParamsValidHelper());
        this.registerHelper(bars, 'jsdoc-default', new JsdocDefaultHelper());
        this.registerHelper(bars, 'linkType', new LinkTypeHelper());
        this.registerHelper(bars, 'indexableSignature', new IndexableSignatureHelper());
        this.registerHelper(bars, 'object', new ObjectHelper());
        this.registerHelper(bars, 'objectLength', new ObjectLengthHelper());
        this.registerHelper(bars, 'parseDescription', new ParseDescriptionHelper());
        this.registerHelper(bars, 'one-parameter-has', new OneParameterHasHelper());
        this.registerHelper(bars, 'element-alone', new ElementAloneHelper());
        this.registerHelper(bars, 'hasOwn', new HasOwnHelper());
        this.registerHelper(bars, 'short-url', new ShortURLHelper());
        this.registerHelper(bars, 'strip-url', new StripURLHelper());
        this.registerHelper(bars, 't', new I18nHelper());
    };
    HtmlEngineHelpers.prototype.registerHelper = function (bars, key, helper) {
        Handlebars.registerHelper(key, function () {
            // tslint:disable-next-line:no-invalid-this
            return helper.helperFunc.apply(helper, [this].concat(_.slice(arguments)));
        });
    };
    return HtmlEngineHelpers;
}());

var HtmlEngine = /** @class */ (function () {
    function HtmlEngine() {
        this.cache = {};
        var helper = new HtmlEngineHelpers();
        helper.registerHelpers(Handlebars);
    }
    HtmlEngine.getInstance = function () {
        if (!HtmlEngine.instance) {
            HtmlEngine.instance = new HtmlEngine();
        }
        return HtmlEngine.instance;
    };
    HtmlEngine.prototype.init = function (templatePath) {
        var _this = this;
        var partials = [
            'overview',
            'markdown',
            'modules',
            'module',
            'components',
            'component',
            'controller',
            'component-detail',
            'directives',
            'directive',
            'injectables',
            'injectable',
            'interceptor',
            'guard',
            'pipes',
            'pipe',
            'classes',
            'class',
            'interface',
            'routes',
            'index',
            'index-misc',
            'search-results',
            'search-input',
            'link-type',
            'block-method',
            'block-enum',
            'block-property',
            'block-index',
            'block-constructor',
            'block-typealias',
            'block-accessors',
            'block-input',
            'block-output',
            'coverage-report',
            'unit-test-report',
            'miscellaneous-functions',
            'miscellaneous-variables',
            'miscellaneous-typealiases',
            'miscellaneous-enumerations',
            'additional-page',
            'package-dependencies'
        ];
        if (templatePath) {
            if (FileEngine$1.existsSync(path.resolve(process.cwd() + path.sep + templatePath)) ===
                false) {
                logger.warn('Template path specificed but does not exist...using default templates');
            }
        }
        return Promise.all(partials.map(function (partial) {
            var partialPath = _this.determineTemplatePath(templatePath, 'partials/' + partial + '.hbs');
            return FileEngine$1.get(partialPath).then(function (data) {
                return Handlebars.registerPartial(partial, data);
            });
        }))
            .then(function () {
            var pagePath = _this.determineTemplatePath(templatePath, 'page.hbs');
            return FileEngine$1.get(pagePath).then(function (data) {
                _this.cache.page = data;
                _this.compiledPage = Handlebars.compile(_this.cache.page, {
                    preventIndent: true,
                    strict: true
                });
            });
        })
            .then(function () {
            var menuPath = _this.determineTemplatePath(templatePath, 'partials/menu.hbs');
            return FileEngine$1.get(menuPath).then(function (menuTemplate) {
                _this.precompiledMenu = Handlebars.compile(menuTemplate, {
                    preventIndent: true,
                    strict: true
                });
            });
        });
    };
    HtmlEngine.prototype.renderMenu = function (templatePath, data) {
        var menuPath = this.determineTemplatePath(templatePath, 'partials/menu.hbs');
        return FileEngine$1.get(menuPath).then(function (menuTemplate) {
            data.menu = 'normal';
            return Handlebars.compile(menuTemplate, {
                preventIndent: true,
                strict: true
            })(__assign({}, data));
        });
    };
    HtmlEngine.prototype.render = function (mainData, page) {
        var o = mainData;
        Object.assign(o, page);
        // let mem = process.memoryUsage();
        // console.log(`heapTotal: ${mem.heapTotal} | heapUsed: ${mem.heapUsed}`);
        return this.compiledPage({
            data: o
        });
    };
    HtmlEngine.prototype.determineTemplatePath = function (templatePath, filePath) {
        var outPath = path.resolve(__dirname + '/../src/templates/' + filePath);
        if (templatePath) {
            var testPath = path.resolve(process.cwd() + path.sep + templatePath + path.sep + filePath);
            outPath = FileEngine$1.existsSync(testPath) ? testPath : outPath;
        }
        return outPath;
    };
    HtmlEngine.prototype.generateCoverageBadge = function (outputFolder, label, coverageData) {
        return FileEngine$1.get(path.resolve(__dirname + '/../src/templates/partials/coverage-badge.hbs')).then(function (data) {
            var template = Handlebars.compile(data);
            coverageData.label = label;
            var result = template({
                data: coverageData
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/images/coverage-badge-' + label + '.svg', result).catch(function (err) {
                logger.error('Error during coverage badge ' + label + ' file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during coverage badge generation'); });
    };
    return HtmlEngine;
}());
var HtmlEngine$1 = HtmlEngine.getInstance();

var MarkdownEngine = /** @class */ (function () {
    function MarkdownEngine() {
        var _this = this;
        /**
         * List of markdown files without .md extension
         */
        this.markdownFiles = ['README', 'CHANGELOG', 'LICENSE', 'CONTRIBUTING', 'TODO'];
        this.markedInstance = require('marked');
        var renderer = new this.markedInstance.Renderer();
        renderer.code = function (code, language) {
            var highlighted = code;
            if (!language) {
                language = 'none';
            }
            highlighted = _this.escape(code);
            return "<div><pre class=\"line-numbers\"><code class=\"language-" + language + "\">" + highlighted + "</code></pre></div>";
        };
        renderer.table = function (header, body) {
            return ('<table class="table table-bordered compodoc-table">\n' +
                '<thead>\n' +
                header +
                '</thead>\n' +
                '<tbody>\n' +
                body +
                '</tbody>\n' +
                '</table>\n');
        };
        renderer.image = function (href, title, text) {
            var out = '<img src="' + href + '" alt="' + text + '" class="img-responsive"';
            if (title) {
                out += ' title="' + title + '"';
            }
            out += '>';
            return out;
        };
        this.markedInstance.setOptions({
            renderer: renderer,
            gfm: true,
            breaks: false
        });
    }
    MarkdownEngine.getInstance = function () {
        if (!MarkdownEngine.instance) {
            MarkdownEngine.instance = new MarkdownEngine();
        }
        return MarkdownEngine.instance;
    };
    MarkdownEngine.prototype.getTraditionalMarkdown = function (filepath) {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + filepath + '.md')
            .catch(function (err) { return FileEngine$1.get(process.cwd() + path.sep + filepath).then(); })
            .then(function (data) { return _this.markedInstance(data); });
    };
    MarkdownEngine.prototype.getTraditionalMarkdownSync = function (filepath) {
        return this.markedInstance(FileEngine$1.getSync(process.cwd() + path.sep + filepath));
    };
    MarkdownEngine.prototype.getReadmeFile = function () {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + 'README.md').then(function (data) {
            return _this.markedInstance(data);
        });
    };
    MarkdownEngine.prototype.readNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return fs.readFileSync(readmeFile, 'utf8');
    };
    MarkdownEngine.prototype.hasNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return FileEngine$1.existsSync(readmeFile);
    };
    MarkdownEngine.prototype.componentReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + 'README.md';
        var readmeAlternativeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        var finalPath = '';
        if (FileEngine$1.existsSync(readmeFile)) {
            finalPath = readmeFile;
        }
        else {
            finalPath = readmeAlternativeFile;
        }
        return finalPath;
    };
    /**
     * Checks if any of the markdown files is exists with or without endings
     */
    MarkdownEngine.prototype.hasRootMarkdowns = function () {
        return this.addEndings(this.markdownFiles).some(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
    };
    MarkdownEngine.prototype.listRootMarkdowns = function () {
        var foundFiles = this.markdownFiles.filter(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x + '.md') ||
                FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
        return this.addEndings(foundFiles);
    };
    MarkdownEngine.prototype.escape = function (html) {
        return html
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
            .replace(/@/g, '&#64;');
    };
    /**
     * ['README'] => ['README', 'README.md']
     */
    MarkdownEngine.prototype.addEndings = function (files) {
        return _.flatMap(files, function (x) { return [x, x + '.md']; });
    };
    return MarkdownEngine;
}());
var MarkdownEngine$1 = MarkdownEngine.getInstance();

var ngdT = require('@compodoc/ngd-transformer');
var NgdEngine = /** @class */ (function () {
    function NgdEngine() {
    }
    NgdEngine.getInstance = function () {
        if (!NgdEngine.instance) {
            NgdEngine.instance = new NgdEngine();
        }
        return NgdEngine.instance;
    };
    NgdEngine.prototype.init = function (outputpath) {
        this.engine = new ngdT.DotEngine({
            output: outputpath,
            displayLegend: true,
            outputFormats: 'svg',
            silent: true
        });
    };
    NgdEngine.prototype.renderGraph = function (filepath, outputpath, type, name) {
        this.engine.updateOutput(outputpath);
        if (type === 'f') {
            return this.engine.generateGraph([DependenciesEngine$1.getRawModule(name)]);
        }
        else {
            return this.engine.generateGraph(DependenciesEngine$1.rawModulesForOverview);
        }
    };
    NgdEngine.prototype.readGraph = function (filepath, name) {
        return FileEngine$1.get(filepath).catch(function (err) {
            return Promise.reject('Error during graph read ' + name);
        });
    };
    return NgdEngine;
}());
var NgdEngine$1 = NgdEngine.getInstance();

var COMPODOC_CONSTANTS = {
    navTabDefinitions: [
        {
            id: 'info',
            href: '#info',
            'data-link': 'info',
            label: 'Info',
            depTypes: ['all']
        },
        {
            id: 'readme',
            href: '#readme',
            'data-link': 'readme',
            label: 'README',
            depTypes: ['all']
        },
        {
            id: 'source',
            href: '#source',
            'data-link': 'source',
            label: 'Source',
            depTypes: ['all']
        },
        {
            id: 'templateData',
            href: '#templateData',
            'data-link': 'template',
            label: 'Template',
            depTypes: ['component']
        },
        {
            id: 'styleData',
            href: '#styleData',
            'data-link': 'style',
            label: 'Styles',
            depTypes: ['component']
        },
        {
            id: 'tree',
            href: '#tree',
            'data-link': 'dom-tree',
            label: 'DOM Tree',
            depTypes: ['component']
        },
        {
            id: 'example',
            href: '#example',
            'data-link': 'example',
            label: 'Examples',
            depTypes: ['component', 'directive', 'injectable', 'pipe']
        }
    ]
};
/**
 * Max length for the string of a file during Lunr search engine indexing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_SEARCH_INDEX = 50000;
/**
 * Max length for the string of a file during cheerio parsing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_CHEERIO_PARSING = 400000000;

var lunr = require('lunr');
var cheerio = require('cheerio');
var Entities = require('html-entities').AllHtmlEntities;
var Html = new Entities();
var SearchEngine = /** @class */ (function () {
    function SearchEngine() {
        this.searchDocuments = [];
        this.documentsStore = {};
        this.amountOfMemory = 0;
    }
    SearchEngine.getInstance = function () {
        if (!SearchEngine.instance) {
            SearchEngine.instance = new SearchEngine();
        }
        return SearchEngine.instance;
    };
    SearchEngine.prototype.indexPage = function (page) {
        var text;
        this.amountOfMemory += page.rawData.length;
        if (this.amountOfMemory < MAX_SIZE_FILE_CHEERIO_PARSING) {
            var indexStartContent = page.rawData.indexOf('<!-- START CONTENT -->');
            var indexEndContent = page.rawData.indexOf('<!-- END CONTENT -->');
            var $ = cheerio.load(page.rawData.substring(indexStartContent + 1, indexEndContent));
            text = $('.content').html();
            text = Html.decode(text);
            text = text.replace(/(<([^>]+)>)/gi, '');
            page.url = page.url.replace(Configuration$1.mainData.output, '');
            var doc = {
                url: page.url,
                title: page.infos.context + ' - ' + page.infos.name,
                body: text
            };
            if (!this.documentsStore.hasOwnProperty(doc.url) &&
                doc.body.length < MAX_SIZE_FILE_SEARCH_INDEX) {
                this.documentsStore[doc.url] = doc;
                this.searchDocuments.push(doc);
            }
        }
    };
    SearchEngine.prototype.generateSearchIndexJson = function (outputFolder) {
        var _this = this;
        var that = this;
        var searchIndex = lunr(function () {
            /* tslint:disable:no-invalid-this */
            this.ref('url');
            this.field('title');
            this.field('body');
            this.pipeline.remove(lunr.stemmer);
            var i = 0;
            var len = that.searchDocuments.length;
            for (i; i < len; i++) {
                this.add(that.searchDocuments[i]);
            }
        });
        return FileEngine$1.get(__dirname + '/../src/templates/partials/search-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                index: JSON.stringify(searchIndex),
                store: JSON.stringify(_this.documentsStore)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/search/search_index.js', result).catch(function (err) {
                logger.error('Error during search index file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during search index generation'); });
    };
    return SearchEngine;
}());
var SearchEngine$1 = SearchEngine.getInstance();

var $ = require('cheerio');
var ComponentsTreeEngine = /** @class */ (function () {
    function ComponentsTreeEngine() {
        this.components = [];
        this.componentsForTree = [];
    }
    ComponentsTreeEngine.getInstance = function () {
        if (!ComponentsTreeEngine.instance) {
            ComponentsTreeEngine.instance = new ComponentsTreeEngine();
        }
        return ComponentsTreeEngine.instance;
    };
    ComponentsTreeEngine.prototype.addComponent = function (component) {
        this.components.push(component);
    };
    ComponentsTreeEngine.prototype.readTemplates = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = _this.componentsForTree.length;
            var loop = function () {
                if (i <= len - 1) {
                    if (_this.componentsForTree[i].templateUrl) {
                        var filePath = process.cwd() +
                            path.sep +
                            path.dirname(_this.componentsForTree[i].file) +
                            path.sep +
                            _this.componentsForTree[i].templateUrl;
                        FileEngine$1.get(filePath).then(function (templateData) {
                            _this.componentsForTree[i].templateData = templateData;
                            i++;
                            loop();
                        }, function (e) {
                            logger.error(e);
                            reject();
                        });
                    }
                    else {
                        _this.componentsForTree[i].templateData = _this.componentsForTree[i].template;
                        i++;
                        loop();
                    }
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    ComponentsTreeEngine.prototype.findChildrenAndParents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.componentsForTree, function (component) {
                var $component = $(component.templateData);
                _.forEach(_this.componentsForTree, function (componentToFind) {
                    if ($component.find(componentToFind.selector).length > 0) {
                        console.log(componentToFind.name + ' found in ' + component.name);
                        component.children.push(componentToFind.name);
                    }
                });
            });
            resolve();
        });
    };
    ComponentsTreeEngine.prototype.createTreesForComponents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.components, function (component) {
                var _component = {
                    name: component.name,
                    file: component.file,
                    selector: component.selector,
                    children: [],
                    template: '',
                    templateUrl: ''
                };
                if (typeof component.template !== 'undefined') {
                    _component.template = component.template;
                }
                if (component.templateUrl.length > 0) {
                    _component.templateUrl = component.templateUrl[0];
                }
                _this.componentsForTree.push(_component);
            });
            _this.readTemplates().then(function () {
                _this.findChildrenAndParents().then(function () {
                    console.log('this.componentsForTree: ', _this.componentsForTree);
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }, function (e) {
                logger.error(e);
            });
        });
    };
    return ComponentsTreeEngine;
}());
var ComponentsTreeEngine$1 = ComponentsTreeEngine.getInstance();

var JsdocParserUtil = /** @class */ (function () {
    function JsdocParserUtil() {
    }
    JsdocParserUtil.prototype.isVariableLike = function (node) {
        if (node) {
            switch (node.kind) {
                case Ast.SyntaxKind.BindingElement:
                case Ast.SyntaxKind.EnumMember:
                case Ast.SyntaxKind.Parameter:
                case Ast.SyntaxKind.PropertyAssignment:
                case Ast.SyntaxKind.PropertyDeclaration:
                case Ast.SyntaxKind.PropertySignature:
                case Ast.SyntaxKind.ShorthandPropertyAssignment:
                case Ast.SyntaxKind.VariableDeclaration:
                    return true;
            }
        }
        return false;
    };
    JsdocParserUtil.prototype.getMainCommentOfNode = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = node.jsDoc[0].comment;
                }
            }
        }
        return description;
    };
    JsdocParserUtil.prototype.getJSDocTags = function (node, kind) {
        var docs = this.getJSDocs(node);
        if (docs) {
            var result = [];
            for (var _i = 0, docs_1 = docs; _i < docs_1.length; _i++) {
                var doc = docs_1[_i];
                if (Ast.ts.isJSDocParameterTag(doc)) {
                    if (doc.kind === kind) {
                        result.push(doc);
                    }
                }
                else if (Ast.ts.isJSDoc(doc)) {
                    result.push.apply(result, _.filter(doc.tags, function (tag) { return tag.kind === kind; }));
                }
                else {
                    throw new Error('Unexpected type');
                }
            }
            return result;
        }
    };
    JsdocParserUtil.prototype.getJSDocs = function (node) {
        // TODO: jsDocCache is internal, see if there's a way around it
        var cache = node.jsDocCache;
        if (!cache) {
            cache = this.getJSDocsWorker(node, []).filter(function (x) { return x; });
            node.jsDocCache = cache;
        }
        return cache;
    };
    // Try to recognize this pattern when node is initializer
    // of variable declaration and JSDoc comments are on containing variable statement.
    // /**
    //   * @param {number} name
    //   * @returns {number}
    //   */
    // var x = function(name) { return name.length; }
    JsdocParserUtil.prototype.getJSDocsWorker = function (node, cache) {
        var parent = node.parent;
        var isInitializerOfVariableDeclarationInStatement = this.isVariableLike(parent) &&
            parent.initializer === node &&
            Ast.ts.isVariableStatement(parent.parent.parent);
        var isVariableOfVariableDeclarationStatement = this.isVariableLike(node) && Ast.ts.isVariableStatement(parent.parent);
        var variableStatementNode = isInitializerOfVariableDeclarationInStatement
            ? parent.parent.parent
            : isVariableOfVariableDeclarationStatement
                ? parent.parent
                : undefined;
        if (variableStatementNode) {
            cache = this.getJSDocsWorker(variableStatementNode, cache);
        }
        // Also recognize when the node is the RHS of an assignment expression
        var isSourceOfAssignmentExpressionStatement = parent &&
            parent.parent &&
            Ast.ts.isBinaryExpression(parent) &&
            parent.operatorToken.kind === Ast.SyntaxKind.EqualsToken &&
            Ast.ts.isExpressionStatement(parent.parent);
        if (isSourceOfAssignmentExpressionStatement) {
            cache = this.getJSDocsWorker(parent.parent, cache);
        }
        var isModuleDeclaration = Ast.ts.isModuleDeclaration(node) && parent && Ast.ts.isModuleDeclaration(parent);
        var isPropertyAssignmentExpression = parent && Ast.ts.isPropertyAssignment(parent);
        if (isModuleDeclaration || isPropertyAssignmentExpression) {
            cache = this.getJSDocsWorker(parent, cache);
        }
        // Pull parameter comments from declaring function as well
        if (Ast.ts.isParameter(node)) {
            cache = _.concat(cache, this.getJSDocParameterTags(node));
        }
        if (this.isVariableLike(node) && node.initializer) {
            cache = _.concat(cache, node.initializer.jsDoc);
        }
        cache = _.concat(cache, node.jsDoc);
        return cache;
    };
    JsdocParserUtil.prototype.getJSDocParameterTags = function (param) {
        var func = param.parent;
        var tags = this.getJSDocTags(func, Ast.SyntaxKind.JSDocParameterTag);
        if (!param.name) {
            // this is an anonymous jsdoc param from a `function(type1, type2): type3` specification
            var i = func.parameters.indexOf(param);
            var paramTags = _.filter(tags, function (tag) { return Ast.ts.isJSDocParameterTag(tag); });
            if (paramTags && 0 <= i && i < paramTags.length) {
                return [paramTags[i]];
            }
        }
        else if (Ast.ts.isIdentifier(param.name)) {
            var name_1 = param.name.text;
            return _.filter(tags, function (tag) {
                if (Ast.ts && Ast.ts.isJSDocParameterTag(tag)) {
                    var t = tag;
                    if (typeof t.parameterName !== 'undefined') {
                        return t.parameterName.text === name_1;
                    }
                    else if (typeof t.name !== 'undefined') {
                        if (typeof t.name.escapedText !== 'undefined') {
                            return t.name.escapedText === name_1;
                        }
                    }
                }
            });
        }
        else {
            // TODO: it's a destructured parameter, so it should look up an "object type" series of multiple lines
            // But multi-line object types aren't supported yet either
            return undefined;
        }
    };
    return JsdocParserUtil;
}());

var ast = new Ast__default();
var ImportsUtil = /** @class */ (function () {
    function ImportsUtil() {
    }
    ImportsUtil.getInstance = function () {
        if (!ImportsUtil.instance) {
            ImportsUtil.instance = new ImportsUtil();
        }
        return ImportsUtil.instance;
    };
    /**
     * Find for a sourceFile a variable value in a local enum
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInEnums = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getEnum(function (e) {
            if (e.getName() === variableName) {
                e.getMember(function (m) {
                    if (m.getName() === variableValue) {
                        res = m.getValue();
                    }
                });
            }
        });
        return res;
    };
    /**
     * Find for a sourceFile a variable value in a local static class
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInClasses = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getClass(function (c) {
            var staticProperty = c.getStaticProperty(variableValue);
            if (staticProperty) {
                if (staticProperty.getInitializer()) {
                    res = staticProperty.getInitializer().getText();
                }
            }
        });
        return res;
    };
    /**
     * Find a value in a local variable declaration like an object
     * @param variableDeclaration
     * @param variablesAttributes
     */
    ImportsUtil.prototype.findInObjectVariableDeclaration = function (variableDeclaration, variablesAttributes) {
        var variableKind = variableDeclaration.getKind();
        if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
            var initializer = variableDeclaration.getInitializer();
            if (initializer) {
                var initializerKind = initializer.getKind();
                if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                    var compilerNode = initializer.compilerNode, finalValue_1 = '';
                    // Find thestring from AVAR.BVAR.thestring inside properties
                    var depth_1 = 0;
                    var loopProperties_1 = function (properties) {
                        properties.forEach(function (prop) {
                            if (prop.name) {
                                if (variablesAttributes[depth_1 + 1]) {
                                    if (prop.name.getText() === variablesAttributes[depth_1 + 1]) {
                                        if (prop.initializer) {
                                            if (prop.initializer.properties) {
                                                depth_1 += 1;
                                                loopProperties_1(prop.initializer.properties);
                                            }
                                            else {
                                                finalValue_1 = prop.initializer.text;
                                            }
                                        }
                                        else {
                                            finalValue_1 = prop.initializer.text;
                                        }
                                    }
                                }
                            }
                        });
                    };
                    loopProperties_1(compilerNode.properties);
                    return finalValue_1;
                }
            }
        }
    };
    /**
     * Find in imports something like myvar
     * @param  {string} inputVariableName              like myvar
     * @return {[type]}                                myvar value
     */
    ImportsUtil.prototype.findValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var metadataVariableName = inputVariableName, searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFileIfExists(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        function hasFoundValues(variableDeclaration) {
            var variableKind = variableDeclaration.getKind();
            if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                var initializer = variableDeclaration.getInitializer();
                if (initializer) {
                    var initializerKind = initializer.getKind();
                    if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                        var compilerNode = initializer.compilerNode;
                        return compilerNode.properties;
                    }
                }
            }
        }
        if (typeof searchedImport !== 'undefined') {
            var importPathReference = searchedImport.getModuleSpecifierSourceFile();
            var importPath = void 0;
            if (typeof importPathReference !== 'undefined') {
                importPath = importPathReference.compilerNode.fileName;
                var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                    ? ast.getSourceFile(importPath)
                    : ast.addExistingSourceFileIfExists(importPath); // tslint:disable-line
                if (sourceFileImport) {
                    var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                    var variableDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    if (variableDeclaration) {
                        return hasFoundValues(variableDeclaration);
                    }
                    else {
                        // Try with exports
                        var exportDeclarations = sourceFileImport.getExportDeclarations();
                        if (exportDeclarations && exportDeclarations.length > 0) {
                            var i = 0, len = exportDeclarations.length;
                            for (i; i < len; i++) {
                                var exportDeclaration = exportDeclarations[i];
                                var sourceFileExportedReference = exportDeclaration.getModuleSpecifierSourceFile();
                                if (sourceFileExportedReference) {
                                    var sourceFileExportedReferencePath = sourceFileExportedReference.getFilePath();
                                    var sourceFileExported = typeof ast.getSourceFile(sourceFileExportedReferencePath) !== 'undefined'
                                        ? ast.getSourceFile(sourceFileExportedReferencePath)
                                        : ast.addExistingSourceFileIfExists(sourceFileExportedReferencePath);
                                    if (sourceFileExported) {
                                        variableDeclaration = sourceFileExported.getVariableDeclaration(variableName);
                                        if (variableDeclaration) {
                                            return hasFoundValues(variableDeclaration);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        else {
            // Find in local variables of the file
            var variableDeclaration = file.getVariableDeclaration(metadataVariableName);
            if (variableDeclaration) {
                var variableKind = variableDeclaration.getKind();
                if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                    var initializer = variableDeclaration.getInitializer();
                    if (initializer) {
                        var initializerKind = initializer.getKind();
                        if (initializerKind &&
                            initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                            var compilerNode = initializer.compilerNode;
                            return compilerNode.properties;
                        }
                        else if (initializerKind) {
                            return variableDeclaration.compilerNode;
                        }
                    }
                }
            }
        }
        return [];
    };
    ImportsUtil.prototype.getFileNameOfImport = function (variableName, sourceFile) {
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        var searchedImport, finalPath = '';
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === variableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === variableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
            finalPath = importPath.replace(cleaner, '');
        }
        return finalPath;
    };
    /**
     * Find the file path of imported variable
     * @param  {string} inputVariableName  like thestring
     * @return {[type]}                    thestring destination path
     */
    ImportsUtil.prototype.findFilePathOfImportedVariable = function (inputVariableName, sourceFilePath) {
        var searchedImport, finalPath = '';
        var file = typeof ast.getSourceFile(sourceFilePath) !== 'undefined'
            ? ast.getSourceFile(sourceFilePath)
            : ast.addExistingSourceFile(sourceFilePath); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            finalPath = path.resolve(path.dirname(sourceFilePath) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
        }
        return finalPath;
    };
    /**
     * Find in imports something like VAR.AVAR.BVAR.thestring
     * @param  {string} inputVariableName                   like VAR.AVAR.BVAR.thestring
     * @return {[type]}                                thestring value
     */
    ImportsUtil.prototype.findPropertyValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var variablesAttributes = inputVariableName.split('.'), metadataVariableName = variablesAttributes[0], searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        var fileToSearchIn, variableDeclaration;
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                ? ast.getSourceFile(importPath)
                : ast.addExistingSourceFile(importPath); // tslint:disable-line
            if (sourceFileImport) {
                fileToSearchIn = sourceFileImport;
                var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                variableDeclaration = fileToSearchIn.getVariableDeclaration(variableName);
            }
        }
        else {
            fileToSearchIn = file;
            // Find in local variables of the file
            variableDeclaration = fileToSearchIn.getVariableDeclaration(metadataVariableName);
        }
        if (variableDeclaration) {
            return this.findInObjectVariableDeclaration(variableDeclaration, variablesAttributes);
        }
        // Try find it in enums
        if (variablesAttributes.length > 0) {
            if (typeof fileToSearchIn !== 'undefined') {
                var val = this.findInEnums(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
                val = this.findInClasses(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
            }
        }
    };
    return ImportsUtil;
}());
var ImportsUtil$1 = ImportsUtil.getInstance();

var traverse$2 = require('traverse');
var ast$1 = new Ast__default();
var RouterParserUtil = /** @class */ (function () {
    function RouterParserUtil() {
        this.routes = [];
        this.incompleteRoutes = [];
        this.modules = [];
        this.modulesWithRoutes = [];
    }
    RouterParserUtil.getInstance = function () {
        if (!RouterParserUtil.instance) {
            RouterParserUtil.instance = new RouterParserUtil();
        }
        return RouterParserUtil.instance;
    };
    RouterParserUtil.prototype.addRoute = function (route) {
        this.routes.push(route);
        this.routes = _.sortBy(_.uniqWith(this.routes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addIncompleteRoute = function (route) {
        this.incompleteRoutes.push(route);
        this.incompleteRoutes = _.sortBy(_.uniqWith(this.incompleteRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModuleWithRoutes = function (moduleName, moduleImports, filename) {
        this.modulesWithRoutes.push({
            name: moduleName,
            importsNode: moduleImports,
            filename: filename
        });
        this.modulesWithRoutes = _.sortBy(_.uniqWith(this.modulesWithRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModule = function (moduleName, moduleImports) {
        this.modules.push({
            name: moduleName,
            importsNode: moduleImports
        });
        this.modules = _.sortBy(_.uniqWith(this.modules, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.cleanRawRouteParsed = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        return JSON5.parse(routesWithoutSpaces);
    };
    RouterParserUtil.prototype.cleanRawRoute = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        return routesWithoutSpaces;
    };
    RouterParserUtil.prototype.setRootModule = function (module) {
        this.rootModule = module;
    };
    RouterParserUtil.prototype.hasRouterModuleInImports = function (imports) {
        for (var i = 0; i < imports.length; i++) {
            if (imports[i].name.indexOf('RouterModule.forChild') !== -1 ||
                imports[i].name.indexOf('RouterModule.forRoot') !== -1 ||
                imports[i].name.indexOf('RouterModule') !== -1) {
                return true;
            }
        }
        return false;
    };
    RouterParserUtil.prototype.fixIncompleteRoutes = function (miscellaneousVariables) {
        var matchingVariables = [];
        // For each incompleteRoute, scan if one misc variable is in code
        // if ok, try recreating complete route
        for (var i = 0; i < this.incompleteRoutes.length; i++) {
            for (var j = 0; j < miscellaneousVariables.length; j++) {
                if (this.incompleteRoutes[i].data.indexOf(miscellaneousVariables[j].name) !== -1) {
                    console.log('found one misc var inside incompleteRoute');
                    console.log(miscellaneousVariables[j].name);
                    matchingVariables.push(miscellaneousVariables[j]);
                }
            }
            // Clean incompleteRoute
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace('[', '');
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace(']', '');
        }
    };
    RouterParserUtil.prototype.linkModulesAndRoutes = function () {
        var _this = this;
        var i = 0;
        var len = this.modulesWithRoutes.length;
        for (i; i < len; i++) {
            _.forEach(this.modulesWithRoutes[i].importsNode, function (node) {
                var initializer = node.initializer;
                if (initializer) {
                    if (initializer.elements) {
                        _.forEach(initializer.elements, function (element) {
                            // find element with arguments
                            if (element.arguments) {
                                _.forEach(element.arguments, function (argument) {
                                    _.forEach(_this.routes, function (route) {
                                        if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename === _this.modulesWithRoutes[i].filename) {
                                            route.module = _this.modulesWithRoutes[i].name;
                                        }
                                        else if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename !== _this.modulesWithRoutes[i].filename) {
                                            var argumentImportPath = ImportsUtil$1.findFilePathOfImportedVariable(argument.text, _this.modulesWithRoutes[i].filename);
                                            var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
                                            argumentImportPath = argumentImportPath.replace(cleaner, '');
                                            if (argument.text &&
                                                route.name === argument.text &&
                                                route.filename === argumentImportPath) {
                                                route.module = _this.modulesWithRoutes[i].name;
                                            }
                                        }
                                    });
                                });
                            }
                        });
                    }
                }
                /**
                 * direct support of for example
                 * export const HomeRoutingModule: ModuleWithProviders = RouterModule.forChild(HOME_ROUTES);
                 */
                if (Ast.ts.isCallExpression(node)) {
                    if (node.arguments) {
                        _.forEach(node.arguments, function (argument) {
                            _.forEach(_this.routes, function (route) {
                                if (argument.text &&
                                    route.name === argument.text &&
                                    route.filename === _this.modulesWithRoutes[i].filename) {
                                    route.module = _this.modulesWithRoutes[i].name;
                                }
                            });
                        });
                    }
                }
            });
        }
    };
    RouterParserUtil.prototype.foundRouteWithModuleName = function (moduleName) {
        return _.find(this.routes, { module: moduleName });
    };
    RouterParserUtil.prototype.foundLazyModuleWithPath = function (modulePath) {
        // path is like app/customers/customers.module#CustomersModule
        var split = modulePath.split('#');
        var lazyModulePath = split[0];
        var lazyModuleName = split[1];
        return lazyModuleName;
    };
    RouterParserUtil.prototype.constructRoutesTree = function () {
        var _this = this;
        // routes[] contains routes with module link
        // modulesTree contains modules tree
        // make a final routes tree with that
        traverse$2(this.modulesTree).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (node.importsNode) {
                    delete node.importsNode;
                }
            }
        });
        this.cleanModulesTree = _.cloneDeep(this.modulesTree);
        var routesTree = {
            name: '<root>',
            kind: 'module',
            className: this.rootModule,
            children: []
        };
        var loopModulesParser = function (node) {
            if (node.children && node.children.length > 0) {
                // If module has child modules
                for (var i in node.children) {
                    var route = _this.foundRouteWithModuleName(node.children[i].name);
                    if (route && route.data) {
                        try {
                            route.children = JSON5.parse(route.data);
                        }
                        catch (e) {
                            logger.error('Error during generation of routes JSON file, maybe a trailing comma or an external variable inside one route.');
                        }
                        delete route.data;
                        route.kind = 'module';
                        routesTree.children.push(route);
                    }
                    if (node.children[i].children) {
                        loopModulesParser(node.children[i]);
                    }
                }
            }
            else {
                // else routes are directly inside the module
                var rawRoutes = _this.foundRouteWithModuleName(node.name);
                if (rawRoutes) {
                    var routes = JSON5.parse(rawRoutes.data);
                    if (routes) {
                        var i = 0;
                        var len = routes.length;
                        var routeAddedOnce = false;
                        for (i; i < len; i++) {
                            var route = routes[i];
                            if (routes[i].component) {
                                routeAddedOnce = true;
                                routesTree.children.push({
                                    kind: 'component',
                                    component: routes[i].component,
                                    path: routes[i].path
                                });
                            }
                        }
                        if (!routeAddedOnce) {
                            routesTree.children = routesTree.children.concat(routes);
                        }
                    }
                }
            }
        };
        var startModule = _.find(this.cleanModulesTree, { name: this.rootModule });
        if (startModule) {
            loopModulesParser(startModule);
            // Loop twice for routes with lazy loading
            // loopModulesParser(routesTree);
        }
        var cleanedRoutesTree = undefined;
        var cleanRoutesTree = function (route) {
            for (var i in route.children) {
                var routes = route.children[i].routes;
            }
            return route;
        };
        cleanedRoutesTree = cleanRoutesTree(routesTree);
        // Try updating routes with lazy loading
        var loopInsideModule = function (mod, _rawModule) {
            if (mod.children) {
                for (var z in mod.children) {
                    var route = _this.foundRouteWithModuleName(mod.children[z].name);
                    if (typeof route !== 'undefined') {
                        if (route.data) {
                            route.children = JSON5.parse(route.data);
                            delete route.data;
                            route.kind = 'module';
                            _rawModule.children.push(route);
                        }
                    }
                }
            }
            else {
                var route = _this.foundRouteWithModuleName(mod.name);
                if (typeof route !== 'undefined') {
                    if (route.data) {
                        route.children = JSON5.parse(route.data);
                        delete route.data;
                        route.kind = 'module';
                        _rawModule.children.push(route);
                    }
                }
            }
        };
        var loopRoutesParser = function (route) {
            if (route.children) {
                for (var i in route.children) {
                    if (route.children[i].loadChildren) {
                        var child = _this.foundLazyModuleWithPath(route.children[i].loadChildren);
                        var module = _.find(_this.cleanModulesTree, {
                            name: child
                        });
                        if (module) {
                            var _rawModule = {};
                            _rawModule.kind = 'module';
                            _rawModule.children = [];
                            _rawModule.module = module.name;
                            loopInsideModule(module, _rawModule);
                            route.children[i].children = [];
                            route.children[i].children.push(_rawModule);
                        }
                    }
                    loopRoutesParser(route.children[i]);
                }
            }
        };
        loopRoutesParser(cleanedRoutesTree);
        return cleanedRoutesTree;
    };
    RouterParserUtil.prototype.constructModulesTree = function () {
        var _this = this;
        var getNestedChildren = function (arr, parent) {
            var out = [];
            for (var i in arr) {
                if (arr[i].parent === parent) {
                    var children = getNestedChildren(arr, arr[i].name);
                    if (children.length) {
                        arr[i].children = children;
                    }
                    out.push(arr[i]);
                }
            }
            return out;
        };
        // Scan each module and add parent property
        _.forEach(this.modules, function (firstLoopModule) {
            _.forEach(firstLoopModule.importsNode, function (importNode) {
                _.forEach(_this.modules, function (module) {
                    if (module.name === importNode.name) {
                        module.parent = firstLoopModule.name;
                    }
                });
            });
        });
        this.modulesTree = getNestedChildren(this.modules);
    };
    RouterParserUtil.prototype.generateRoutesIndex = function (outputFolder, routes) {
        return FileEngine$1.get(__dirname + '/../src/templates/partials/routes-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                routes: JSON.stringify(routes)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/routes/routes_index.js', result);
        }, function (err) { return Promise.reject('Error during routes index generation'); });
    };
    RouterParserUtil.prototype.routesLength = function () {
        var _n = 0;
        var routesParser = function (route) {
            if (typeof route.path !== 'undefined') {
                _n += 1;
            }
            if (route.children) {
                for (var j in route.children) {
                    routesParser(route.children[j]);
                }
            }
        };
        for (var i in this.routes) {
            routesParser(this.routes[i]);
        }
        return _n;
    };
    RouterParserUtil.prototype.printRoutes = function () {
        console.log('');
        console.log('printRoutes: ');
        console.log(this.routes);
    };
    RouterParserUtil.prototype.printModulesRoutes = function () {
        console.log('');
        console.log('printModulesRoutes: ');
        console.log(this.modulesWithRoutes);
    };
    RouterParserUtil.prototype.isVariableRoutes = function (node) {
        var result = false;
        if (node.declarationList && node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                if (node.declarationList.declarations[i].type) {
                    if (node.declarationList.declarations[i].type.typeName &&
                        node.declarationList.declarations[i].type.typeName.text === 'Routes') {
                        result = true;
                    }
                }
            }
        }
        return result;
    };
    RouterParserUtil.prototype.cleanFileIdentifiers = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var identifiers = file.getDescendantsOfKind(Ast.SyntaxKind.Identifier).filter(function (p) {
            return (Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()) ||
                Ast.TypeGuards.isPropertyAssignment(p.getParentOrThrow()));
        });
        var identifiersInRoutesVariableStatement = [];
        var _loop_1 = function (identifier) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = identifier.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                identifiersInRoutesVariableStatement.push(identifier);
            }
        };
        for (var _i = 0, identifiers_1 = identifiers; _i < identifiers_1.length; _i++) {
            var identifier = identifiers_1[_i];
            _loop_1(identifier);
        }
        // inline the property access expressions
        for (var _a = 0, identifiersInRoutesVariableStatement_1 = identifiersInRoutesVariableStatement; _a < identifiersInRoutesVariableStatement_1.length; _a++) {
            var identifier = identifiersInRoutesVariableStatement_1[_a];
            var identifierDeclaration = identifier
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                Ast.TypeGuards.isVariableDeclaration(identifierDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                    !Ast.TypeGuards.isVariableDeclaration(identifierDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + identifierDeclaration.getKindName());
            }
            if (Ast.TypeGuards.isVariableDeclaration(identifierDeclaration)) {
                identifier.replaceWithText(identifierDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileSpreads = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var spreadElements = file
            .getDescendantsOfKind(Ast.SyntaxKind.SpreadElement)
            .filter(function (p) { return Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()); });
        var spreadElementsInRoutesVariableStatement = [];
        var _loop_2 = function (spreadElement) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = spreadElement.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                spreadElementsInRoutesVariableStatement.push(spreadElement);
            }
        };
        for (var _i = 0, spreadElements_1 = spreadElements; _i < spreadElements_1.length; _i++) {
            var spreadElement = spreadElements_1[_i];
            _loop_2(spreadElement);
        }
        var _loop_3 = function (spreadElement) {
            var spreadElementIdentifier = spreadElement.getExpression().getText(), searchedImport, aliasOriginalName = '', foundWithAliasInImports = false, foundWithAlias = false;
            // Try to find it in imports
            var imports = file.getImportDeclarations();
            imports.forEach(function (i) {
                var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
                if (namedImportsLength > 0) {
                    for (j; j < namedImportsLength; j++) {
                        var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                        if (namedImports[j].getAliasIdentifier()) {
                            importAlias = namedImports[j].getAliasIdentifier().getText();
                        }
                        if (importName === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            searchedImport = i;
                            break;
                        }
                        if (importAlias === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            foundWithAlias = true;
                            aliasOriginalName = importName;
                            searchedImport = i;
                            break;
                        }
                    }
                }
            });
            var referencedDeclaration = void 0;
            if (foundWithAliasInImports) {
                if (typeof searchedImport !== 'undefined') {
                    var importPath = path.resolve(path.dirname(file.getFilePath()) +
                        '/' +
                        searchedImport.getModuleSpecifierValue() +
                        '.ts');
                    var sourceFileImport = typeof ast$1.getSourceFile(importPath) !== 'undefined'
                        ? ast$1.getSourceFile(importPath)
                        : ast$1.addExistingSourceFile(importPath);
                    if (sourceFileImport) {
                        var variableName = foundWithAlias
                            ? aliasOriginalName
                            : spreadElementIdentifier;
                        referencedDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    }
                }
            }
            else {
                // if not, try directly in file
                referencedDeclaration = spreadElement
                    .getExpression()
                    .getSymbolOrThrow()
                    .getValueDeclarationOrThrow();
            }
            if (!Ast.TypeGuards.isVariableDeclaration(referencedDeclaration)) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            var referencedArray = referencedDeclaration.getInitializerIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var spreadElementArray = spreadElement.getParentIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var insertIndex = spreadElementArray.getElements().indexOf(spreadElement);
            spreadElementArray.removeElement(spreadElement);
            spreadElementArray.insertElements(insertIndex, referencedArray.getElements().map(function (e) { return e.getText(); }));
        };
        // inline the ArrayLiteralExpression SpreadElements
        for (var _a = 0, spreadElementsInRoutesVariableStatement_1 = spreadElementsInRoutesVariableStatement; _a < spreadElementsInRoutesVariableStatement_1.length; _a++) {
            var spreadElement = spreadElementsInRoutesVariableStatement_1[_a];
            _loop_3(spreadElement);
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileDynamics = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var propertyAccessExpressions = file
            .getDescendantsOfKind(Ast.SyntaxKind.PropertyAccessExpression)
            .filter(function (p) { return !Ast.TypeGuards.isPropertyAccessExpression(p.getParentOrThrow()); });
        var propertyAccessExpressionsInRoutesVariableStatement = [];
        var _loop_4 = function (propertyAccessExpression) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = propertyAccessExpression.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                propertyAccessExpressionsInRoutesVariableStatement.push(propertyAccessExpression);
            }
        };
        for (var _i = 0, propertyAccessExpressions_1 = propertyAccessExpressions; _i < propertyAccessExpressions_1.length; _i++) {
            var propertyAccessExpression = propertyAccessExpressions_1[_i];
            _loop_4(propertyAccessExpression);
        }
        // inline the property access expressions
        for (var _a = 0, propertyAccessExpressionsInRoutesVariableStatement_1 = propertyAccessExpressionsInRoutesVariableStatement; _a < propertyAccessExpressionsInRoutesVariableStatement_1.length; _a++) {
            var propertyAccessExpression = propertyAccessExpressionsInRoutesVariableStatement_1[_a];
            var referencedDeclaration = propertyAccessExpression
                .getNameNode()
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                Ast.TypeGuards.isEnumMember(referencedDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                    !Ast.TypeGuards.isEnumMember(referencedDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            if (typeof referencedDeclaration.getInitializerOrThrow !== 'undefined') {
                propertyAccessExpression.replaceWithText(referencedDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    /**
     * replace callexpressions with string : utils.doWork() -> 'utils.doWork()' doWork() -> 'doWork()'
     * @param sourceFile ts.SourceFile
     */
    RouterParserUtil.prototype.cleanCallExpressions = function (sourceFile) {
        var file = sourceFile;
        var variableStatements = sourceFile.getVariableDeclaration(function (v) {
            var result = false;
            if (typeof v.compilerNode.type !== 'undefined') {
                result = v.compilerNode.type.typeName.text === 'Routes';
            }
            return result;
        });
        var initializer = variableStatements.getInitializer();
        var _loop_5 = function (callExpr) {
            if (callExpr.wasForgotten()) {
                return "continue";
            }
            callExpr.replaceWithText(function (writer) { return writer.quote(callExpr.getText()); });
        };
        for (var _i = 0, _a = initializer.getDescendantsOfKind(Ast.SyntaxKind.CallExpression); _i < _a.length; _i++) {
            var callExpr = _a[_i];
            _loop_5(callExpr);
        }
        return file;
    };
    /**
     * Clean routes definition with imported data, for example path, children, or dynamic stuff inside data
     *
     * const MY_ROUTES: Routes = [
     *     {
     *         path: 'home',
     *         component: HomeComponent
     *     },
     *     {
     *         path: PATHS.home,
     *         component: HomeComponent
     *     }
     * ];
     *
     * The initializer is an array (ArrayLiteralExpression - 177 ), it has elements, objects (ObjectLiteralExpression - 178)
     * with properties (PropertyAssignment - 261)
     *
     * For each know property (https://angular.io/api/router/Routes#description), we try to see if we have what we want
     *
     * Ex: path and pathMatch want a string, component a component reference.
     *
     * It is an imperative approach, not a generic way, parsing all the tree
     * and find something like this which willl break JSON.stringify : MYIMPORT.path
     *
     * @param  {ts.Node} initializer The node of routes definition
     * @return {ts.Node}             The edited node
     */
    RouterParserUtil.prototype.cleanRoutesDefinitionWithImport = function (initializer, node, sourceFile) {
        initializer.elements.forEach(function (element) {
            element.properties.forEach(function (property) {
                var propertyName = property.name.getText(), propertyInitializer = property.initializer;
                switch (propertyName) {
                    case 'path':
                    case 'redirectTo':
                    case 'outlet':
                    case 'pathMatch':
                        if (propertyInitializer) {
                            if (propertyInitializer.kind !== Ast.SyntaxKind.StringLiteral) {
                                // Identifier(71) won't break parsing, but it will be better to retrive them
                                // PropertyAccessExpression(179) ex: MYIMPORT.path will break it, find it in import
                                if (propertyInitializer.kind === Ast.SyntaxKind.PropertyAccessExpression) {
                                    var lastObjectLiteralAttributeName = propertyInitializer.name.getText(), firstObjectLiteralAttributeName = void 0;
                                    if (propertyInitializer.expression) {
                                        firstObjectLiteralAttributeName = propertyInitializer.expression.getText();
                                        var result = ImportsUtil$1.findPropertyValueInImportOrLocalVariables(firstObjectLiteralAttributeName +
                                            '.' +
                                            lastObjectLiteralAttributeName, sourceFile); // tslint:disable-line
                                        if (result !== '') {
                                            propertyInitializer.kind = 9;
                                            propertyInitializer.text = result;
                                        }
                                    }
                                }
                            }
                        }
                        break;
                }
            });
        });
        return initializer;
    };
    return RouterParserUtil;
}());
var RouterParserUtil$1 = RouterParserUtil.getInstance();

function isModuleWithProviders(node) {
    var result = false;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, declarations = node.declarationList.declarations, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = true;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function getModuleWithProviders(node) {
    var result;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = declaration.initializer;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function StringifyObjectLiteralExpression(ole) {
    var returnedString = '{';
    if (ole.properties && ole.properties.length > 0) {
        ole.properties.forEach(function (property, index) {
            if (property.name) {
                returnedString += property.name.text + ': ';
            }
            if (property.initializer) {
                if (property.initializer.kind === Ast.SyntaxKind.StringLiteral) {
                    returnedString += "'" + property.initializer.text + "'";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.TrueKeyword) {
                    returnedString += "true";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.FalseKeyword) {
                    returnedString += "false";
                }
                else {
                    returnedString += property.initializer.text;
                }
            }
            if (index < ole.properties.length - 1) {
                returnedString += ', ';
            }
        });
    }
    returnedString += '}';
    return returnedString;
}

var crypto = require('crypto');
var marked$1 = require('marked');
var ClassHelper = /** @class */ (function () {
    function ClassHelper(typeChecker) {
        this.typeChecker = typeChecker;
        this.jsdocParserUtil = new JsdocParserUtil();
    }
    /**
     * HELPERS
     */
    ClassHelper.prototype.stringifyDefaultValue = function (node) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (node.getText()) {
            return node.getText();
        }
        else if (node.kind === Ast.SyntaxKind.FalseKeyword) {
            return 'false';
        }
        else if (node.kind === Ast.SyntaxKind.TrueKeyword) {
            return 'true';
        }
    };
    ClassHelper.prototype.getDecoratorOfType = function (node, decoratorType) {
        var decorators = node.decorators || [];
        for (var i = 0; i < decorators.length; i++) {
            if (decorators[i].expression.expression) {
                if (decorators[i].expression.expression.text === decoratorType) {
                    return decorators[i];
                }
            }
        }
        return undefined;
    };
    ClassHelper.prototype.formatDecorators = function (decorators) {
        var _this = this;
        var _decorators = [];
        _.forEach(decorators, function (decorator) {
            if (decorator.expression) {
                if (decorator.expression.text) {
                    _decorators.push({ name: decorator.expression.text });
                }
                if (decorator.expression.expression) {
                    var info = { name: decorator.expression.expression.text };
                    if (decorator.expression.arguments) {
                        info.stringifiedArguments = _this.stringifyArguments(decorator.expression.arguments);
                    }
                    _decorators.push(info);
                }
            }
        });
        return _decorators;
    };
    ClassHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg.function.length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg.function.map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_1 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_1 = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_1 + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path_2 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_2 + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path_3 = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_3 + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    ClassHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    ClassHelper.prototype.stringifyArguments = function (args) {
        var _this = this;
        var stringifyArgs = [];
        stringifyArgs = args
            .map(function (arg) {
            var _result = DependenciesEngine$1.find(arg.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_4 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_4 = 'classe';
                    }
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_4 + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                }
                else {
                    var path_5 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_5 + "\" target=\"_blank\">" + arg.type + "</a>";
                }
            }
            else if (arg.dotDotDotToken) {
                return "..." + arg.name + ": " + arg.type;
            }
            else if (arg.function) {
                return _this.handleFunction(arg);
            }
            else if (arg.expression && arg.name) {
                return arg.expression.text + '.' + arg.name.text;
            }
            else if (arg.expression && arg.kind === Ast.SyntaxKind.NewExpression) {
                return 'new ' + arg.expression.text + '()';
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.StringLiteral) {
                return "'" + arg.text + "'";
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
                return StringifyObjectLiteralExpression(arg);
            }
            else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                var path_6 = BasicTypeUtil$1.getTypeUrl(arg.type);
                return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_6 + "\" target=\"_blank\">" + arg.type + "</a>";
            }
            else {
                if (arg.type) {
                    var finalStringifiedArgument = '';
                    var separator = ':';
                    if (arg.name) {
                        finalStringifiedArgument += arg.name;
                    }
                    if (arg.kind === Ast.SyntaxKind.AsExpression &&
                        arg.expression &&
                        arg.expression.text) {
                        finalStringifiedArgument += arg.expression.text;
                        separator = ' as';
                    }
                    if (arg.optional) {
                        finalStringifiedArgument += _this.getOptionalString(arg);
                    }
                    if (arg.type) {
                        finalStringifiedArgument += separator + ' ' + _this.visitType(arg.type);
                    }
                    return finalStringifiedArgument;
                }
                else if (arg.text) {
                    return "" + arg.text;
                }
                else {
                    return "" + arg.name + _this.getOptionalString(arg);
                }
            }
        })
            .join(', ');
        return stringifyArgs;
    };
    ClassHelper.prototype.getPosition = function (node, sourceFile) {
        var position;
        if (node.name && node.name.end) {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.name.end);
        }
        else {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.pos);
        }
        return position;
    };
    ClassHelper.prototype.addAccessor = function (accessors, nodeAccessor, sourceFile) {
        var nodeName = '';
        if (nodeAccessor.name) {
            nodeName = nodeAccessor.name.text;
            var jsdoctags = this.jsdocParserUtil.getJSDocs(nodeAccessor);
            if (!accessors[nodeName]) {
                accessors[nodeName] = {
                    name: nodeName,
                    setSignature: undefined,
                    getSignature: undefined
                };
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.SetAccessor) {
                var setSignature = {
                    name: nodeName,
                    type: 'void',
                    args: nodeAccessor.parameters.map(function (param) {
                        return {
                            name: param.name.text,
                            type: param.type ? kindToType(param.type.kind) : ''
                        };
                    }),
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : 'void',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        setSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        setSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                if (setSignature.jsdoctags && setSignature.jsdoctags.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args, setSignature.jsdoctags);
                }
                else if (setSignature.args && setSignature.args.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args);
                }
                accessors[nodeName].setSignature = setSignature;
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.GetAccessor) {
                var getSignature = {
                    name: nodeName,
                    type: nodeAccessor.type ? kindToType(nodeAccessor.type.kind) : '',
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : '',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        getSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        getSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                accessors[nodeName].getSignature = getSignature;
            }
        }
    };
    ClassHelper.prototype.isDirectiveDecorator = function (decorator) {
        if (decorator.expression.expression) {
            var decoratorIdentifierText = decorator.expression.expression.text;
            return (decoratorIdentifierText === 'Directive' || decoratorIdentifierText === 'Component');
        }
        else {
            return false;
        }
    };
    ClassHelper.prototype.isServiceDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Injectable'
            : false;
    };
    ClassHelper.prototype.isPrivate = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (member.modifiers) {
            var isPrivate = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PrivateKeyword; });
            if (isPrivate) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isProtected = function (member) {
        if (member.modifiers) {
            var isProtected = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.ProtectedKeyword; });
            if (isProtected) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isInternal = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['internal'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPublic = function (member) {
        if (member.modifiers) {
            var isPublic = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PublicKeyword; });
            if (isPublic) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isHiddenMember = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['hidden'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPipeDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Pipe'
            : false;
    };
    ClassHelper.prototype.isModuleDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'NgModule'
            : false;
    };
    /**
     * VISITERS
     */
    ClassHelper.prototype.visitClassDeclaration = function (fileName, classDeclaration, sourceFile) {
        var symbol = this.typeChecker.getSymbolAtLocation(classDeclaration.name);
        var rawdescription = '';
        var description = '';
        if (symbol) {
            rawdescription = this.jsdocParserUtil.getMainCommentOfNode(classDeclaration);
            description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(classDeclaration));
            if (symbol.valueDeclaration && isIgnore(symbol.valueDeclaration)) {
                return [{ ignore: true }];
            }
            if (symbol.declarations && symbol.declarations.length > 0) {
                if (isIgnore(symbol.declarations[0])) {
                    return [{ ignore: true }];
                }
            }
        }
        var className = classDeclaration.name.text;
        var members;
        var implementsElements = [];
        var extendsElement;
        var jsdoctags = [];
        if (typeof Ast.ts.getClassImplementsHeritageClauseElements !== 'undefined') {
            var implementedTypes = Ast.ts.getClassImplementsHeritageClauseElements(classDeclaration);
            if (implementedTypes) {
                var i = 0;
                var len = implementedTypes.length;
                for (i; i < len; i++) {
                    if (implementedTypes[i].expression) {
                        implementsElements.push(implementedTypes[i].expression.text);
                    }
                }
            }
        }
        if (typeof Ast.ts.getClassExtendsHeritageClauseElement !== 'undefined') {
            var extendsTypes = Ast.ts.getClassExtendsHeritageClauseElement(classDeclaration);
            if (extendsTypes) {
                if (extendsTypes.expression) {
                    extendsElement = extendsTypes.expression.text;
                }
            }
        }
        if (symbol) {
            if (symbol.valueDeclaration) {
                jsdoctags = this.jsdocParserUtil.getJSDocs(symbol.valueDeclaration);
            }
        }
        members = this.visitMembers(classDeclaration.members, sourceFile);
        if (classDeclaration.decorators) {
            for (var i = 0; i < classDeclaration.decorators.length; i++) {
                if (this.isDirectiveDecorator(classDeclaration.decorators[i])) {
                    return {
                        description: description,
                        rawdescription: rawdescription,
                        inputs: members.inputs,
                        outputs: members.outputs,
                        hostBindings: members.hostBindings,
                        hostListeners: members.hostListeners,
                        properties: members.properties,
                        methods: members.methods,
                        indexSignatures: members.indexSignatures,
                        kind: members.kind,
                        constructor: members.constructor,
                        jsdoctags: jsdoctags,
                        extends: extendsElement,
                        implements: implementsElements,
                        accessors: members.accessors
                    };
                }
                else if (this.isServiceDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            extends: extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
                else if (this.isPipeDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            properties: members.properties,
                            methods: members.methods
                        }
                    ];
                }
                else if (this.isModuleDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            methods: members.methods
                        }
                    ];
                }
                else {
                    return [
                        {
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            extends: extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
            }
        }
        else if (description) {
            return [
                {
                    description: description,
                    rawdescription: rawdescription,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    methods: members.methods,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    extends: extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        else {
            return [
                {
                    methods: members.methods,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    extends: extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        return [];
    };
    ClassHelper.prototype.visitMembers = function (members, sourceFile) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var inputs = [];
        var outputs = [];
        var hostBindings = [];
        var hostListeners = [];
        var methods = [];
        var properties = [];
        var indexSignatures = [];
        var kind;
        var inputDecorator;
        var hostBinding;
        var hostListener;
        var constructor;
        var outDecorator;
        var accessors = {};
        var result = {};
        for (var i = 0; i < members.length; i++) {
            // Allows typescript guess type when using ts.is*
            var member = members[i];
            inputDecorator = this.getDecoratorOfType(member, 'Input');
            outDecorator = this.getDecoratorOfType(member, 'Output');
            hostBinding = this.getDecoratorOfType(member, 'HostBinding');
            hostListener = this.getDecoratorOfType(member, 'HostListener');
            kind = member.kind;
            if (isIgnore(member)) {
                continue;
            }
            if (inputDecorator) {
                inputs.push(this.visitInputAndHostBinding(member, inputDecorator, sourceFile));
                if (Ast.ts.isSetAccessorDeclaration(member)) {
                    this.addAccessor(accessors, members[i], sourceFile);
                }
            }
            else if (outDecorator) {
                outputs.push(this.visitOutput(member, outDecorator, sourceFile));
            }
            else if (hostBinding) {
                hostBindings.push(this.visitInputAndHostBinding(member, hostBinding, sourceFile));
            }
            else if (hostListener) {
                hostListeners.push(this.visitHostListener(member, hostListener, sourceFile));
            }
            else if (!this.isHiddenMember(member)) {
                if (!(this.isPrivate(member) && Configuration$1.mainData.disablePrivate)) {
                    if (!(this.isInternal(member) && Configuration$1.mainData.disableInternal)) {
                        if (!(this.isProtected(member) && Configuration$1.mainData.disableProtected)) {
                            if (Ast.ts.isMethodDeclaration(member) || Ast.ts.isMethodSignature(member)) {
                                methods.push(this.visitMethodDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isPropertyDeclaration(member) ||
                                Ast.ts.isPropertySignature(member)) {
                                properties.push(this.visitProperty(member, sourceFile));
                            }
                            else if (Ast.ts.isCallSignatureDeclaration(member)) {
                                properties.push(this.visitCallDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isGetAccessorDeclaration(member) ||
                                Ast.ts.isSetAccessorDeclaration(member)) {
                                this.addAccessor(accessors, members[i], sourceFile);
                            }
                            else if (Ast.ts.isIndexSignatureDeclaration(member)) {
                                indexSignatures.push(this.visitIndexDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isConstructorDeclaration(member)) {
                                var _constructorProperties = this.visitConstructorProperties(member, sourceFile);
                                var j = 0;
                                var len = _constructorProperties.length;
                                for (j; j < len; j++) {
                                    properties.push(_constructorProperties[j]);
                                }
                                constructor = this.visitConstructorDeclaration(member, sourceFile);
                            }
                        }
                    }
                }
            }
        }
        inputs.sort(getNamesCompareFn());
        outputs.sort(getNamesCompareFn());
        hostBindings.sort(getNamesCompareFn());
        hostListeners.sort(getNamesCompareFn());
        properties.sort(getNamesCompareFn());
        methods.sort(getNamesCompareFn());
        indexSignatures.sort(getNamesCompareFn());
        result = {
            inputs: inputs,
            outputs: outputs,
            hostBindings: hostBindings,
            hostListeners: hostListeners,
            methods: methods,
            properties: properties,
            indexSignatures: indexSignatures,
            kind: kind,
            constructor: constructor
        };
        if (Object.keys(accessors).length) {
            result['accessors'] = accessors;
        }
        return result;
    };
    ClassHelper.prototype.visitTypeName = function (typeName) {
        if (typeName.text) {
            return typeName.text;
        }
        return this.visitTypeName(typeName.left) + "." + this.visitTypeName(typeName.right);
    };
    ClassHelper.prototype.visitType = function (node) {
        var _return = 'void';
        if (!node) {
            return _return;
        }
        if (node.typeName) {
            _return = this.visitTypeName(node.typeName);
        }
        else if (node.type) {
            if (node.type.kind) {
                _return = kindToType(node.type.kind);
            }
            if (node.type.typeName) {
                _return = this.visitTypeName(node.type.typeName);
            }
            if (node.type.typeArguments) {
                _return += '<';
                var typeArguments = [];
                for (var _i = 0, _a = node.type.typeArguments; _i < _a.length; _i++) {
                    var argument = _a[_i];
                    typeArguments.push(this.visitType(argument));
                }
                _return += typeArguments.join(' | ');
                _return += '>';
            }
            if (node.type.elementType) {
                var _firstPart = this.visitType(node.type.elementType);
                _return = _firstPart + kindToType(node.type.kind);
                if (node.type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                    _return = '(' + _firstPart + ')' + kindToType(node.type.kind);
                }
            }
            if (node.type.types && Ast.ts.isUnionTypeNode(node.type)) {
                _return = '';
                var i = 0;
                var len = node.type.types.length;
                for (i; i < len; i++) {
                    var type = node.type.types[i];
                    if (type.elementType) {
                        var _firstPart = this.visitType(type.elementType);
                        if (type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                            _return += '(' + _firstPart + ')' + kindToType(type.kind);
                        }
                        else {
                            _return += _firstPart + kindToType(type.kind);
                        }
                    }
                    else {
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (type.typeArguments) {
                            _return += '<';
                            var typeArguments = [];
                            for (var _b = 0, _c = type.typeArguments; _b < _c.length; _b++) {
                                var argument = _c[_b];
                                typeArguments.push(this.visitType(argument));
                            }
                            _return += typeArguments.join(' | ');
                            _return += '>';
                        }
                    }
                    if (i < len - 1) {
                        _return += ' | ';
                    }
                }
            }
            if (node.type.elementTypes) {
                var elementTypes = node.type.elementTypes;
                var i = 0;
                var len = elementTypes.length;
                if (len > 0) {
                    _return = '[';
                    for (i; i < len; i++) {
                        var type = elementTypes[i];
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (i < len - 1) {
                            _return += ', ';
                        }
                    }
                    _return += ']';
                }
            }
        }
        else if (node.elementType) {
            _return = kindToType(node.elementType.kind) + kindToType(node.kind);
            if (node.elementType.typeName) {
                _return = this.visitTypeName(node.elementType.typeName) + kindToType(node.kind);
            }
        }
        else if (node.types && Ast.ts.isUnionTypeNode(node)) {
            _return = '';
            var i = 0;
            var len = node.types.length;
            for (i; i < len; i++) {
                var type = node.types[i];
                _return += kindToType(type.kind);
                if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                    _return += '"' + type.literal.text + '"';
                }
                if (type.typeName) {
                    _return += this.visitTypeName(type.typeName);
                }
                if (i < len - 1) {
                    _return += ' | ';
                }
            }
        }
        else if (node.dotDotDotToken) {
            _return = 'any[]';
        }
        else {
            _return = kindToType(node.kind);
            if (_return === '' &&
                node.initializer &&
                node.initializer.kind &&
                (node.kind === Ast.SyntaxKind.PropertyDeclaration || node.kind === Ast.SyntaxKind.Parameter)) {
                _return = kindToType(node.initializer.kind);
            }
            if (node.kind === Ast.SyntaxKind.TypeParameter) {
                _return = node.name.text;
            }
            if (node.kind === Ast.SyntaxKind.LiteralType) {
                _return = node.literal.text;
            }
        }
        if (node.typeArguments && node.typeArguments.length > 0) {
            _return += '<';
            var i = 0, len = node.typeArguments.length;
            for (i; i < len; i++) {
                var argument = node.typeArguments[i];
                _return += this.visitType(argument);
                if (i >= 0 && i < len - 1) {
                    _return += ', ';
                }
            }
            _return += '>';
        }
        return _return;
    };
    ClassHelper.prototype.visitCallDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'call-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(method)));
        }
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitIndexDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'index-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorDeclaration = function (method, sourceFile) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var result = {
            name: 'constructor',
            description: '',
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitProperty = function (property, sourceFile) {
        var result = {
            name: property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined,
            type: this.visitType(property),
            optional: typeof property.questionToken !== 'undefined',
            description: '',
            line: this.getPosition(property, sourceFile).line + 1
        };
        var jsdoctags;
        if (property.initializer && property.initializer.kind === Ast.SyntaxKind.ArrowFunction) {
            result.defaultValue = '() => {...}';
        }
        if (typeof result.name === 'undefined' && typeof property.name.expression !== 'undefined') {
            result.name = property.name.expression.text;
        }
        if (property.jsDoc) {
            jsdoctags = this.jsdocParserUtil.getJSDocs(property);
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (property.decorators) {
            result.decorators = this.formatDecorators(property.decorators);
        }
        if (property.modifiers) {
            if (property.modifiers.length > 0) {
                var kinds = property.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorProperties = function (constr, sourceFile) {
        if (constr.parameters) {
            var _parameters_1 = [];
            var i = 0;
            var len = constr.parameters.length;
            for (i; i < len; i++) {
                if (this.isPublic(constr.parameters[i])) {
                    _parameters_1.push(this.visitProperty(constr.parameters[i], sourceFile));
                }
            }
            /**
             * Merge JSDoc tags description from constructor with parameters
             */
            if (constr.jsDoc) {
                if (constr.jsDoc.length > 0) {
                    var constrTags = constr.jsDoc[0].tags;
                    if (constrTags && constrTags.length > 0) {
                        constrTags.forEach(function (tag) {
                            _parameters_1.forEach(function (param) {
                                if (tag.tagName &&
                                    tag.tagName.escapedText &&
                                    tag.tagName.escapedText === 'param') {
                                    if (tag.name &&
                                        tag.name.escapedText &&
                                        tag.name.escapedText === param.name) {
                                        param.description = tag.comment;
                                    }
                                }
                            });
                        });
                    }
                }
            }
            return _parameters_1;
        }
        else {
            return [];
        }
    };
    ClassHelper.prototype.visitInputAndHostBinding = function (property, inDecorator, sourceFile) {
        var inArgs = inDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.defaultValue = property.initializer
            ? this.stringifyDefaultValue(property.initializer)
            : undefined;
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        if (property.kind === Ast.SyntaxKind.SetAccessor) {
            // For setter accessor, find type in first parameter
            if (property.parameters && property.parameters.length === 1) {
                if (property.parameters[0].type) {
                    _return.type = kindToType(property.parameters[0].type.kind);
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitMethodDeclaration = function (method, sourceFile) {
        var _this = this;
        var result = {
            name: method.name.text,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            optional: typeof method.questionToken !== 'undefined',
            returnType: this.visitType(method.type),
            typeParameters: [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type === 'undefined') {
            // Try to get inferred type
            if (method.symbol) {
                var symbol = method.symbol;
                if (symbol.valueDeclaration) {
                    var symbolType = this.typeChecker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration);
                    if (symbolType) {
                        try {
                            var signature = this.typeChecker.getSignatureFromDeclaration(method);
                            var returnType = signature.getReturnType();
                            result.returnType = this.typeChecker.typeToString(returnType);
                            // tslint:disable-next-line:no-empty
                        }
                        catch (error) { }
                    }
                }
            }
        }
        if (method.typeParameters && method.typeParameters.length > 0) {
            result.typeParameters = method.typeParameters.map(function (typeParameter) {
                return _this.visitType(typeParameter);
            });
        }
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.decorators) {
            result.decorators = this.formatDecorators(method.decorators);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitOutput = function (property, outDecorator, sourceFile) {
        var inArgs = outDecorator.expression.arguments;
        var _return = {
            name: inArgs.length > 0 ? inArgs[0].text : property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined
        };
        if (property.jsDoc) {
            _return.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(property)));
        }
        if (!_return.description) {
            if (property.jsDoc && property.jsDoc.length > 0) {
                if (typeof property.jsDoc[0].comment !== 'undefined') {
                    _return.description = marked$1(property.jsDoc[0].comment);
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitArgument = function (arg) {
        var _this = this;
        var _result = { name: arg.name.text, type: this.visitType(arg) };
        if (arg.dotDotDotToken) {
            _result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            _result.optional = true;
        }
        if (arg.type) {
            if (arg.type.kind) {
                if (Ast.ts.isFunctionTypeNode(arg.type)) {
                    _result.function = arg.type.parameters
                        ? arg.type.parameters.map(function (prop) { return _this.visitArgument(prop); })
                        : [];
                }
            }
        }
        if (arg.initializer) {
            _result.defaultValue = this.stringifyDefaultValue(arg.initializer);
        }
        return _result;
    };
    ClassHelper.prototype.visitHostListener = function (property, hostListenerDecorator, sourceFile) {
        var _this = this;
        var inArgs = hostListenerDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.args = property.parameters
            ? property.parameters.map(function (prop) { return _this.visitArgument(prop); })
            : [];
        _return.argsDecorator =
            inArgs.length > 1
                ? inArgs[1].elements.map(function (prop) {
                    return prop.text;
                })
                : [];
        if (property.jsDoc) {
            _return.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        return _return;
    };
    return ClassHelper;
}());

var TsPrinterUtil = /** @class */ (function () {
    function TsPrinterUtil() {
        this.printer = Ast.ts.createPrinter({
            newLine: Ast.ts.NewLineKind.LineFeed
        });
    }
    TsPrinterUtil.prototype.print = function (node) {
        return this.printer.printNode(Ast.ts.EmitHint.Unspecified, node, Ast.ts.createSourceFile('', '', Ast.ts.ScriptTarget.Latest));
    };
    return TsPrinterUtil;
}());

var SymbolHelper = /** @class */ (function () {
    function SymbolHelper() {
        this.unknown = '???';
    }
    SymbolHelper.prototype.parseDeepIndentifier = function (name, srcFile) {
        var result = {
            name: '',
            type: ''
        };
        if (typeof name === 'undefined') {
            return result;
        }
        var nsModule = name.split('.');
        var type = this.getType(name);
        if (nsModule.length > 1) {
            result.ns = nsModule[0];
            result.name = name;
            result.type = type;
            return result;
        }
        if (typeof srcFile !== 'undefined') {
            result.file = ImportsUtil$1.getFileNameOfImport(name, srcFile);
        }
        result.name = name;
        result.type = type;
        return result;
    };
    SymbolHelper.prototype.getType = function (name) {
        var type;
        if (name.toLowerCase().indexOf('component') !== -1) {
            type = 'component';
        }
        else if (name.toLowerCase().indexOf('pipe') !== -1) {
            type = 'pipe';
        }
        else if (name.toLowerCase().indexOf('controller') !== -1) {
            type = 'controller';
        }
        else if (name.toLowerCase().indexOf('module') !== -1) {
            type = 'module';
        }
        else if (name.toLowerCase().indexOf('directive') !== -1) {
            type = 'directive';
        }
        return type;
    };
    /**
     * Output
     * RouterModule.forRoot 179
     */
    SymbolHelper.prototype.buildIdentifierName = function (node, name) {
        if (Ast.ts.isIdentifier(node) && !Ast.ts.isPropertyAccessExpression(node)) {
            return node.text + "." + name;
        }
        name = name ? "." + name : '';
        var nodeName = this.unknown;
        if (node.name) {
            nodeName = node.name.text;
        }
        else if (node.text) {
            nodeName = node.text;
        }
        else if (node.expression) {
            if (node.expression.text) {
                nodeName = node.expression.text;
            }
            else if (node.expression.elements) {
                if (Ast.ts.isArrayLiteralExpression(node.expression)) {
                    nodeName = node.expression.elements.map(function (el) { return el.text; }).join(', ');
                    nodeName = "[" + nodeName + "]";
                }
            }
        }
        if (Ast.ts.isSpreadElement(node)) {
            return "..." + nodeName;
        }
        return "" + this.buildIdentifierName(node.expression, nodeName) + name;
    };
    /**
     * parse expressions such as:
     * { provide: APP_BASE_HREF, useValue: '/' }
     * { provide: 'Date', useFactory: (d1, d2) => new Date(), deps: ['d1', 'd2'] }
     */
    SymbolHelper.prototype.parseProviderConfiguration = function (node) {
        if (node.kind && node.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
            // Search for provide: HTTP_INTERCEPTORS
            // and if true, return type: 'interceptor' + name
            var interceptorName_1, hasInterceptor_1;
            if (node.properties) {
                if (node.properties.length > 0) {
                    _.forEach(node.properties, function (property) {
                        if (property.kind && property.kind === Ast.SyntaxKind.PropertyAssignment) {
                            if (property.name.text === 'provide') {
                                if (property.initializer.text === 'HTTP_INTERCEPTORS') {
                                    hasInterceptor_1 = true;
                                }
                            }
                            if (property.name.text === 'useClass' ||
                                property.name.text === 'useExisting') {
                                interceptorName_1 = property.initializer.text;
                            }
                        }
                    });
                }
            }
            if (hasInterceptor_1) {
                return interceptorName_1;
            }
            else {
                return new TsPrinterUtil().print(node);
            }
        }
        else {
            return new TsPrinterUtil().print(node);
        }
    };
    /**
     * Kind
     *  181 CallExpression => "RouterModule.forRoot(args)"
     *   71 Identifier     => "RouterModule" "TodoStore"
     *    9 StringLiteral  => "./app.component.css" "./tab.scss"
     */
    SymbolHelper.prototype.parseSymbolElements = function (node) {
        // parse expressions such as: AngularFireModule.initializeApp(firebaseConfig)
        // if (ts.isCallExpression(node) && ts.isPropertyAccessExpression(node.expression)) {
        if ((Ast.ts.isCallExpression(node) && Ast.ts.isPropertyAccessExpression(node.expression)) ||
            (Ast.ts.isNewExpression(node) && Ast.ts.isElementAccessExpression(node.expression))) {
            var className = this.buildIdentifierName(node.expression);
            // function arguments could be really complex. There are so
            // many use cases that we can't handle. Just print "args" to indicate
            // that we have arguments.
            var functionArgs = node.arguments.length > 0 ? 'args' : '';
            var text = className + "(" + functionArgs + ")";
            return text;
        }
        else if (Ast.ts.isPropertyAccessExpression(node)) {
            // parse expressions such as: Shared.Module
            return this.buildIdentifierName(node);
        }
        else if (Ast.ts.isIdentifier(node)) {
            // parse expressions such as: MyComponent
            if (node.text) {
                return node.text;
            }
            if (node.escapedText) {
                return node.escapedText;
            }
        }
        else if (Ast.ts.isSpreadElement(node)) {
            // parse expressions such as: ...MYARRAY
            // Resolve MYARRAY in imports or local file variables after full scan, just return the name of the variable
            if (node.expression && node.expression.text) {
                return node.expression.text;
            }
        }
        return node.text ? node.text : this.parseProviderConfiguration(node);
    };
    /**
     * Kind
     *  177 ArrayLiteralExpression
     *  122 BooleanKeyword
     *    9 StringLiteral
     */
    SymbolHelper.prototype.parseSymbols = function (node, srcFile) {
        var _this = this;
        var localNode = node;
        if (Ast.ts.isShorthandPropertyAssignment(localNode)) {
            localNode = ImportsUtil$1.findValueInImportOrLocalVariables(node.name.text, srcFile);
        }
        if (Ast.ts.isArrayLiteralExpression(localNode.initializer)) {
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
        else if (Ast.ts.isStringLiteral(localNode.initializer) ||
            Ast.ts.isTemplateLiteral(localNode.initializer) ||
            (Ast.ts.isPropertyAssignment(localNode) && localNode.initializer.text)) {
            return [localNode.initializer.text];
        }
        else if (localNode.initializer.kind &&
            (localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ||
                localNode.initializer.kind === Ast.SyntaxKind.FalseKeyword)) {
            return [localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ? true : false];
        }
        else if (Ast.ts.isPropertyAccessExpression(localNode.initializer)) {
            var identifier = this.parseSymbolElements(localNode.initializer);
            return [identifier];
        }
        else if (localNode.initializer &&
            localNode.initializer.elements &&
            localNode.initializer.elements.length > 0) {
            // Node replaced by ts-simple-ast & kind = 265
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
    };
    SymbolHelper.prototype.getSymbolDeps = function (props, type, srcFile, multiLine) {
        var _this = this;
        if (props.length === 0) {
            return [];
        }
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseSymbols(x, srcFile); }).pop() || [];
    };
    SymbolHelper.prototype.getSymbolDepsRaw = function (props, type, multiLine) {
        return props.filter(function (node) { return node.name.text === type; });
    };
    return SymbolHelper;
}());

var ComponentHelper = /** @class */ (function () {
    function ComponentHelper(classHelper, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.classHelper = classHelper;
        this.symbolHelper = symbolHelper;
    }
    ComponentHelper.prototype.getComponentChangeDetection = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'changeDetection', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentEncapsulation = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'encapsulation', srcFile);
    };
    ComponentHelper.prototype.getComponentPure = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'pure', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentName = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'name', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentExportAs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'exportAs', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentHost = function (props) {
        return this.getSymbolDepsObject(props, 'host');
    };
    ComponentHelper.prototype.getComponentTag = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'tag', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentInputsMetadata = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'inputs', srcFile);
    };
    ComponentHelper.prototype.getComponentTemplate = function (props, srcFile) {
        var t = this.symbolHelper.getSymbolDeps(props, 'template', srcFile, true).pop();
        if (t) {
            t = detectIndent(t, 0);
            t = t.replace(/\n/, '');
            t = t.replace(/ +$/gm, '');
        }
        return t;
    };
    ComponentHelper.prototype.getComponentStyleUrls = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrls', srcFile);
    };
    ComponentHelper.prototype.getComponentStyleUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrl', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentShadow = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'shadow', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentScoped = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'scoped', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDir = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDirs = function (props, srcFile) {
        return this.sanitizeUrls(this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile));
    };
    ComponentHelper.prototype.getComponentStyles = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styles', srcFile);
    };
    ComponentHelper.prototype.getComponentModuleId = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'moduleId', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentOutputs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'outputs', srcFile);
    };
    ComponentHelper.prototype.getComponentProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'entryComponents', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentViewProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'viewProviders', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentTemplateUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'templateUrl', srcFile);
    };
    ComponentHelper.prototype.getComponentExampleUrls = function (text) {
        var exampleUrlsMatches = text.match(/<example-url>(.*?)<\/example-url>/g);
        var exampleUrls = undefined;
        if (exampleUrlsMatches && exampleUrlsMatches.length) {
            exampleUrls = exampleUrlsMatches.map(function (val) {
                return val.replace(/<\/?example-url>/g, '');
            });
        }
        return exampleUrls;
    };
    ComponentHelper.prototype.getComponentPreserveWhitespaces = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'preserveWhitespaces', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentSelector = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'selector', srcFile).pop();
    };
    ComponentHelper.prototype.parseProperties = function (node) {
        var obj = new Map();
        var properties = node.initializer.properties || [];
        properties.forEach(function (prop) {
            obj.set(prop.name.text, prop.initializer.text);
        });
        return obj;
    };
    ComponentHelper.prototype.getSymbolDepsObject = function (props, type, multiLine) {
        var _this = this;
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseProperties(x); }).pop();
    };
    ComponentHelper.prototype.getComponentIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    ComponentHelper.prototype.sanitizeUrls = function (urls) {
        return urls.map(function (url) { return url.replace('./', ''); });
    };
    return ComponentHelper;
}());
var ComponentCache = /** @class */ (function () {
    function ComponentCache() {
        this.cache = new Map();
    }
    ComponentCache.prototype.get = function (key) {
        return this.cache.get(key);
    };
    ComponentCache.prototype.set = function (key, value) {
        this.cache.set(key, value);
    };
    return ComponentCache;
}());

var FrameworkDependencies = /** @class */ (function () {
    function FrameworkDependencies(files, options) {
        this.files = files;
        var transpileOptions = {
            target: Ast.ts.ScriptTarget.ES5,
            module: Ast.ts.ModuleKind.CommonJS,
            tsconfigDirectory: options.tsconfigDirectory,
            allowJs: true
        };
        this.program = Ast.ts.createProgram(this.files, transpileOptions, compilerHost(transpileOptions));
        this.typeChecker = this.program.getTypeChecker();
        this.classHelper = new ClassHelper(this.typeChecker);
        this.componentHelper = new ComponentHelper(this.classHelper);
    }
    return FrameworkDependencies;
}());

var ExtendsMerger = /** @class */ (function () {
    function ExtendsMerger() {
    }
    ExtendsMerger.getInstance = function () {
        if (!ExtendsMerger.instance) {
            ExtendsMerger.instance = new ExtendsMerger();
        }
        return ExtendsMerger.instance;
    };
    ExtendsMerger.prototype.merge = function (deps) {
        var _this = this;
        this.components = deps.components;
        this.classes = deps.classes;
        this.injectables = deps.injectables;
        this.components.forEach(function (component) {
            var ext;
            if (typeof component.extends !== 'undefined') {
                ext = _this.findInDependencies(component.extends);
                if (ext) {
                    var recursiveScanWithInheritance_1 = function (cls) {
                        // From class to component
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        // From component to component
                        if (typeof cls.inputsClass !== 'undefined' && cls.inputsClass.length > 0) {
                            var newInputs = _.cloneDeep(cls.inputsClass);
                            newInputs = _this.markInheritance(newInputs, cls);
                            if (typeof component.inputsClass !== 'undefined') {
                                component.inputsClass = component.inputsClass.concat(newInputs);
                            }
                        }
                        if (typeof cls.outputsClass !== 'undefined' &&
                            cls.outputsClass.length > 0) {
                            var newOutputs = _.cloneDeep(cls.outputsClass);
                            newOutputs = _this.markInheritance(newOutputs, cls);
                            if (typeof component.outputsClass !== 'undefined') {
                                component.outputsClass = component.outputsClass.concat(newOutputs);
                            }
                        }
                        if (typeof cls.methodsClass !== 'undefined' &&
                            cls.methodsClass.length > 0) {
                            var newMethods = _.cloneDeep(cls.methodsClass);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.propertiesClass !== 'undefined' &&
                            cls.propertiesClass.length > 0) {
                            var newProperties = _.cloneDeep(cls.propertiesClass);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        if (typeof cls.hostBindings !== 'undefined' &&
                            cls.hostBindings.length > 0) {
                            var newHostBindings = _.cloneDeep(cls.hostBindings);
                            newHostBindings = _this.markInheritance(newHostBindings, cls);
                            if (typeof component.hostBindings !== 'undefined') {
                                component.hostBindings = component.hostBindings.concat(newHostBindings);
                            }
                        }
                        if (typeof cls.hostListeners !== 'undefined' &&
                            cls.hostListeners.length > 0) {
                            var newHostListeners = _.cloneDeep(cls.hostListeners);
                            newHostListeners = _this.markInheritance(newHostListeners, cls);
                            if (typeof component.hostListeners !== 'undefined') {
                                component.hostListeners = component.hostListeners.concat(newHostListeners);
                            }
                        }
                        if (Configuration$1.mainData.disableLifeCycleHooks) {
                            component.methodsClass = cleanLifecycleHooksFromMethods(component.methodsClass);
                        }
                        if (cls.extends) {
                            recursiveScanWithInheritance_1(_this.findInDependencies(cls.extends));
                        }
                    };
                    // From class to class
                    recursiveScanWithInheritance_1(ext);
                }
            }
        });
        var mergeExtendedClasses = function (el) {
            var ext;
            if (typeof el.extends !== 'undefined') {
                ext = _this.findInDependencies(el.extends);
                if (ext) {
                    var recursiveScanWithInheritance_2 = function (cls) {
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof el.methods !== 'undefined') {
                                el.methods = el.methods.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof el.properties !== 'undefined') {
                                el.properties = el.properties.concat(newProperties);
                            }
                        }
                        if (cls.extends) {
                            recursiveScanWithInheritance_2(_this.findInDependencies(cls.extends));
                        }
                    };
                    // From elss to elss
                    recursiveScanWithInheritance_2(ext);
                }
            }
        };
        this.classes.forEach(mergeExtendedClasses);
        this.injectables.forEach(mergeExtendedClasses);
        return deps;
    };
    ExtendsMerger.prototype.markInheritance = function (data, originalource) {
        return data.map(function (el) {
            var newElement = el;
            newElement.inheritance = {
                file: originalource.name
            };
            return newElement;
        });
    };
    ExtendsMerger.prototype.findInDependencies = function (name) {
        var mergedData = _.concat([], this.components, this.classes, this.injectables);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    return ExtendsMerger;
}());
var ExtendsMerger$1 = ExtendsMerger.getInstance();

var CodeGenerator = /** @class */ (function () {
    function CodeGenerator() {
    }
    CodeGenerator.prototype.generate = function (node) {
        return this.visitAndRecognize(node, []).join('');
    };
    CodeGenerator.prototype.visitAndRecognize = function (node, code, depth) {
        var _this = this;
        if (depth === void 0) { depth = 0; }
        this.recognize(node, code);
        node.getChildren().forEach(function (c) { return _this.visitAndRecognize(c, code, depth + 1); });
        return code;
    };
    CodeGenerator.prototype.recognize = function (node, code) {
        var _this = this;
        var conversion = TsKindConversion.find(function (x) { return x.kinds.some(function (z) { return z === node.kind; }); });
        if (conversion) {
            var result = conversion.output(node);
            result.forEach(function (text) { return _this.gen(text, code); });
        }
    };
    CodeGenerator.prototype.gen = function (token, code) {
        if (!token) {
            return;
        }
        if (token === '\n') {
            code.push('');
        }
        else {
            code.push(token);
        }
    };
    return CodeGenerator;
}());
var TsKindsToText = /** @class */ (function () {
    function TsKindsToText(output, kinds) {
        this.output = output;
        this.kinds = kinds;
    }
    return TsKindsToText;
}());
var TsKindConversion = [
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [
        Ast.SyntaxKind.FirstLiteralToken,
        Ast.SyntaxKind.Identifier
    ]),
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [Ast.SyntaxKind.StringLiteral]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.ArrayLiteralExpression]),
    new TsKindsToText(function (node) { return ['import', ' ']; }, [Ast.SyntaxKind.ImportKeyword]),
    new TsKindsToText(function (node) { return ['from', ' ']; }, [Ast.SyntaxKind.FromKeyword]),
    new TsKindsToText(function (node) { return ['\n', 'export', ' ']; }, [Ast.SyntaxKind.ExportKeyword]),
    new TsKindsToText(function (node) { return ['class', ' ']; }, [Ast.SyntaxKind.ClassKeyword]),
    new TsKindsToText(function (node) { return ['this']; }, [Ast.SyntaxKind.ThisKeyword]),
    new TsKindsToText(function (node) { return ['constructor']; }, [Ast.SyntaxKind.ConstructorKeyword]),
    new TsKindsToText(function (node) { return ['false']; }, [Ast.SyntaxKind.FalseKeyword]),
    new TsKindsToText(function (node) { return ['true']; }, [Ast.SyntaxKind.TrueKeyword]),
    new TsKindsToText(function (node) { return ['null']; }, [Ast.SyntaxKind.NullKeyword]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.AtToken]),
    new TsKindsToText(function (node) { return ['+']; }, [Ast.SyntaxKind.PlusToken]),
    new TsKindsToText(function (node) { return [' => ']; }, [Ast.SyntaxKind.EqualsGreaterThanToken]),
    new TsKindsToText(function (node) { return ['(']; }, [Ast.SyntaxKind.OpenParenToken]),
    new TsKindsToText(function (node) { return ['{', ' ']; }, [
        Ast.SyntaxKind.ImportClause,
        Ast.SyntaxKind.ObjectLiteralExpression
    ]),
    new TsKindsToText(function (node) { return ['{', '\n']; }, [Ast.SyntaxKind.Block]),
    new TsKindsToText(function (node) { return ['}']; }, [Ast.SyntaxKind.CloseBraceToken]),
    new TsKindsToText(function (node) { return [')']; }, [Ast.SyntaxKind.CloseParenToken]),
    new TsKindsToText(function (node) { return ['[']; }, [Ast.SyntaxKind.OpenBracketToken]),
    new TsKindsToText(function (node) { return [']']; }, [Ast.SyntaxKind.CloseBracketToken]),
    new TsKindsToText(function (node) { return [';', '\n']; }, [Ast.SyntaxKind.SemicolonToken]),
    new TsKindsToText(function (node) { return [',', ' ']; }, [Ast.SyntaxKind.CommaToken]),
    new TsKindsToText(function (node) { return [' ', ':', ' ']; }, [Ast.SyntaxKind.ColonToken]),
    new TsKindsToText(function (node) { return ['.']; }, [Ast.SyntaxKind.DotToken]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.DoStatement]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.Decorator]),
    new TsKindsToText(function (node) { return [' = ']; }, [Ast.SyntaxKind.FirstAssignment]),
    new TsKindsToText(function (node) { return [' ']; }, [Ast.SyntaxKind.FirstPunctuation]),
    new TsKindsToText(function (node) { return ['private', ' ']; }, [Ast.SyntaxKind.PrivateKeyword]),
    new TsKindsToText(function (node) { return ['public', ' ']; }, [Ast.SyntaxKind.PublicKeyword])
];

var crypto$1 = require('crypto');
var ComponentDepFactory = /** @class */ (function () {
    function ComponentDepFactory(helper) {
        this.helper = helper;
    }
    ComponentDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        // console.log(util.inspect(props, { showHidden: true, depth: 10 }));
        var sourceCode = srcFile.getText();
        var hash = crypto$1
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var componentDep = {
            name: name,
            id: 'component-' + name + '-' + hash,
            file: file,
            // animations?: string[]; // TODO
            changeDetection: this.helper.getComponentChangeDetection(props, srcFile),
            encapsulation: this.helper.getComponentEncapsulation(props, srcFile),
            entryComponents: this.helper.getComponentEntryComponents(props, srcFile),
            exportAs: this.helper.getComponentExportAs(props, srcFile),
            host: this.helper.getComponentHost(props),
            inputs: this.helper.getComponentInputsMetadata(props, srcFile),
            // interpolation?: string; // TODO waiting doc infos
            moduleId: this.helper.getComponentModuleId(props, srcFile),
            outputs: this.helper.getComponentOutputs(props, srcFile),
            providers: this.helper.getComponentProviders(props, srcFile),
            // queries?: Deps[]; // TODO
            selector: this.helper.getComponentSelector(props, srcFile),
            styleUrls: this.helper.getComponentStyleUrls(props, srcFile),
            styles: this.helper.getComponentStyles(props, srcFile),
            template: this.helper.getComponentTemplate(props, srcFile),
            templateUrl: this.helper.getComponentTemplateUrl(props, srcFile),
            viewProviders: this.helper.getComponentViewProviders(props, srcFile),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            description: IO.description,
            rawdescription: IO.rawdescription,
            type: 'component',
            sourceCode: srcFile.getText(),
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText()),
            tag: this.helper.getComponentTag(props, srcFile),
            styleUrl: this.helper.getComponentStyleUrl(props, srcFile),
            shadow: this.helper.getComponentShadow(props, srcFile),
            scoped: this.helper.getComponentScoped(props, srcFile),
            assetsDir: this.helper.getComponentAssetsDir(props, srcFile),
            assetsDirs: this.helper.getComponentAssetsDirs(props, srcFile),
            styleUrlsData: '',
            stylesData: ''
        };
        if (typeof this.helper.getComponentPreserveWhitespaces(props, srcFile) !== 'undefined') {
            componentDep.preserveWhitespaces = this.helper.getComponentPreserveWhitespaces(props, srcFile);
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            componentDep.methodsClass = cleanLifecycleHooksFromMethods(componentDep.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            componentDep.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.constructor) {
            componentDep.constructorObj = IO.constructor;
        }
        if (IO.extends) {
            componentDep.extends = IO.extends;
        }
        if (IO.implements && IO.implements.length > 0) {
            componentDep.implements = IO.implements;
        }
        if (IO.accessors) {
            componentDep.accessors = IO.accessors;
        }
        return componentDep;
    };
    return ComponentDepFactory;
}());

var crypto$2 = require('crypto');
var ControllerDepFactory = /** @class */ (function () {
    function ControllerDepFactory() {
    }
    ControllerDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$2
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var infos = {
            name: name,
            id: 'controller-' + name + '-' + hash,
            file: file,
            methodsClass: IO.methods,
            type: 'controller',
            description: IO.description,
            sourceCode: srcFile.text
        };
        if (properties && properties.length === 1) {
            if (properties[0].text) {
                infos.prefix = properties[0].text;
            }
        }
        return infos;
    };
    return ControllerDepFactory;
}());

var crypto$3 = require('crypto');
var DirectiveDepFactory = /** @class */ (function () {
    function DirectiveDepFactory(helper) {
        this.helper = helper;
    }
    DirectiveDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$3
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var directiveDeps = {
            name: name,
            id: 'directive-' + name + '-' + hash,
            file: file,
            type: 'directive',
            description: IO.description,
            sourceCode: srcFile.getText(),
            selector: this.helper.getComponentSelector(props),
            providers: this.helper.getComponentProviders(props),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText())
        };
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            directiveDeps.methodsClass = cleanLifecycleHooksFromMethods(directiveDeps.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            directiveDeps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.implements && IO.implements.length > 0) {
            directiveDeps.implements = IO.implements;
        }
        if (IO.constructor) {
            directiveDeps.constructorObj = IO.constructor;
        }
        if (IO.accessors) {
            directiveDeps.accessors = IO.accessors;
        }
        return directiveDeps;
    };
    return DirectiveDepFactory;
}());

var JsDocHelper = /** @class */ (function () {
    function JsDocHelper() {
    }
    JsDocHelper.prototype.hasJSDocInternalTag = function (filename, sourceFile, node) {
        if (typeof sourceFile.statements !== 'undefined') {
            return this.checkStatements(sourceFile.statements, node);
        }
        return false;
    };
    JsDocHelper.prototype.checkStatements = function (statements, node) {
        var _this = this;
        return statements.some(function (x) { return _this.checkStatement(x, node); });
    };
    JsDocHelper.prototype.checkStatement = function (statement, node) {
        if (statement.pos === node.pos && statement.end === node.end) {
            if (node.jsDoc && node.jsDoc.length > 0) {
                return this.checkJsDocs(node.jsDoc);
            }
        }
        return false;
    };
    JsDocHelper.prototype.checkJsDocs = function (jsDocs) {
        var _this = this;
        return jsDocs
            .filter(function (x) { return x.tags && x.tags.length > 0; })
            .some(function (x) { return _this.checkJsDocTags(x.tags); });
    };
    JsDocHelper.prototype.checkJsDocTags = function (tags) {
        return tags.some(function (x) { return x.tagName && x.tagName.text === 'internal'; });
    };
    return JsDocHelper;
}());

var ModuleHelper = /** @class */ (function () {
    function ModuleHelper(cache, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.cache = cache;
        this.symbolHelper = symbolHelper;
    }
    ModuleHelper.prototype.getModuleProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleControllers = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'controllers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleDeclarations = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'declarations', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.getModuleEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'entryComponents', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.cleanImportForRootForChild = function (name) {
        var nsModule = name.split('.');
        if (nsModule.length > 0) {
            name = nsModule[0];
        }
        return name;
    };
    ModuleHelper.prototype.getModuleImports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'imports', srcFile)
            .map(function (name) { return _this.cleanImportForRootForChild(name); })
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ModuleHelper.prototype.getModuleExports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'exports', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    ModuleHelper.prototype.getModuleImportsRaw = function (props, srcFile) {
        return this.symbolHelper.getSymbolDepsRaw(props, 'imports');
    };
    ModuleHelper.prototype.getModuleId = function (props, srcFile) {
        var _id = this.symbolHelper.getSymbolDeps(props, 'id', srcFile), id;
        if (_id.length === 1) {
            id = _id[0];
        }
        return id;
    };
    ModuleHelper.prototype.getModuleSchemas = function (props, srcFile) {
        var schemas = this.symbolHelper.getSymbolDeps(props, 'schemas', srcFile);
        return schemas;
    };
    ModuleHelper.prototype.getModuleBootstrap = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'bootstrap', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    return ModuleHelper;
}());

var crypto$4 = require('crypto');
var ModuleDepFactory = /** @class */ (function () {
    function ModuleDepFactory(moduleHelper) {
        this.moduleHelper = moduleHelper;
    }
    ModuleDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$4
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        return {
            name: name,
            id: 'module-' + name + '-' + hash,
            file: file,
            ngid: this.moduleHelper.getModuleId(properties, srcFile),
            providers: this.moduleHelper.getModuleProviders(properties, srcFile),
            declarations: this.moduleHelper.getModuleDeclarations(properties, srcFile),
            controllers: this.moduleHelper.getModuleControllers(properties, srcFile),
            entryComponents: this.moduleHelper.getModuleEntryComponents(properties, srcFile),
            imports: this.moduleHelper.getModuleImports(properties, srcFile),
            exports: this.moduleHelper.getModuleExports(properties, srcFile),
            schemas: this.moduleHelper.getModuleSchemas(properties, srcFile),
            bootstrap: this.moduleHelper.getModuleBootstrap(properties, srcFile),
            type: 'module',
            rawdescription: IO.rawdescription,
            methods: IO.methods,
            description: IO.description,
            sourceCode: srcFile.text
        };
    };
    return ModuleDepFactory;
}());

var crypto$5 = require('crypto');
var marked$2 = require('marked');
var ast$2 = new Ast__default();
// TypeScript reference : https://github.com/Microsoft/TypeScript/blob/master/lib/typescript.d.ts
var AngularDependencies = /** @class */ (function (_super) {
    __extends(AngularDependencies, _super);
    function AngularDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        _this.jsdocParserUtil = new JsdocParserUtil();
        return _this;
    }
    AngularDependencies.prototype.getDependencies = function () {
        var _this = this;
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            controllers: [],
            injectables: [],
            interceptors: [],
            guards: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        var sourceFiles = this.program.getSourceFiles() || [];
        sourceFiles.map(function (file) {
            var filePath = file.fileName;
            if (path.extname(filePath) === '.ts' || path.extname(filePath) === '.tsx') {
                if (!Configuration$1.mainData.angularJSProject && path.extname(filePath) === '.js') {
                    logger.info('parsing', filePath);
                    _this.getSourceFileDecorators(file, deps);
                }
                else {
                    if (filePath.lastIndexOf('.d.ts') === -1 &&
                        filePath.lastIndexOf('spec.ts') === -1) {
                        logger.info('parsing', filePath);
                        _this.getSourceFileDecorators(file, deps);
                    }
                }
            }
            return deps;
        });
        // End of file scanning
        // Try merging inside the same file declarated variables & modules with imports | exports | declarations | providers
        if (deps.miscellaneous.variables.length > 0) {
            deps.miscellaneous.variables.forEach(function (_variable) {
                var newVar = [];
                (function (_var, _newVar) {
                    // getType pr reconstruire....
                    if (_var.initializer) {
                        if (_var.initializer.elements) {
                            if (_var.initializer.elements.length > 0) {
                                _var.initializer.elements.forEach(function (element) {
                                    if (element.text) {
                                        newVar.push({
                                            name: element.text,
                                            type: _this.symbolHelper.getType(element.text)
                                        });
                                    }
                                });
                            }
                        }
                    }
                })(_variable);
                var onLink = function (mod) {
                    var process = function (initialArray, _var) {
                        var indexToClean = 0;
                        var found = false;
                        var findVariableInArray = function (el, index, theArray) {
                            if (el.name === _var.name) {
                                indexToClean = index;
                                found = true;
                            }
                        };
                        initialArray.forEach(findVariableInArray);
                        // Clean indexes to replace
                        if (found) {
                            initialArray.splice(indexToClean, 1);
                            // Add variable
                            newVar.forEach(function (newEle) {
                                if (typeof _.find(initialArray, { name: newEle.name }) ===
                                    'undefined') {
                                    initialArray.push(newEle);
                                }
                            });
                        }
                    };
                    process(mod.imports, _variable);
                    process(mod.exports, _variable);
                    process(mod.controllers, _variable);
                    process(mod.declarations, _variable);
                    process(mod.providers, _variable);
                };
                deps.modules.forEach(onLink);
                deps.modulesForGraph.forEach(onLink);
            });
        }
        /**
         * If one thing extends another, merge them, only for internal sources
         * - classes
         * - components
         * - injectables
         * for
         * - inputs
         * - outputs
         * - properties
         * - methods
         */
        deps = ExtendsMerger$1.merge(deps);
        // RouterParserUtil.printModulesRoutes();
        // RouterParserUtil.printRoutes();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            RouterParserUtil$1.linkModulesAndRoutes();
            RouterParserUtil$1.constructModulesTree();
            deps.routesTree = RouterParserUtil$1.constructRoutesTree();
        }
        return deps;
    };
    AngularDependencies.prototype.processClass = function (node, file, srcFile, outputSymbols, fileBody) {
        var name = this.getSymboleName(node);
        var IO = this.getClassIO(file, srcFile, node, fileBody);
        var sourceCode = srcFile.getText();
        var hash = crypto$5
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var deps = {
            name: name,
            id: 'class-' + name + '-' + hash,
            file: file,
            type: 'class',
            sourceCode: srcFile.getText()
        };
        var excludeFromClassArray = false;
        if (IO.constructor) {
            deps.constructorObj = IO.constructor;
        }
        if (IO.properties) {
            deps.properties = IO.properties;
        }
        if (IO.description) {
            deps.description = IO.description;
        }
        if (IO.rawdescription) {
            deps.rawdescription = IO.rawdescription;
        }
        if (IO.methods) {
            deps.methods = IO.methods;
        }
        if (IO.indexSignatures) {
            deps.indexSignatures = IO.indexSignatures;
        }
        if (IO.extends) {
            deps.extends = IO.extends;
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            deps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.accessors) {
            deps.accessors = IO.accessors;
        }
        if (IO.inputs) {
            deps.inputsClass = IO.inputs;
        }
        if (IO.outputs) {
            deps.outputsClass = IO.outputs;
        }
        if (IO.hostBindings) {
            deps.hostBindings = IO.hostBindings;
        }
        if (IO.hostListeners) {
            deps.hostListeners = IO.hostListeners;
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            deps.methods = cleanLifecycleHooksFromMethods(deps.methods);
        }
        if (IO.implements && IO.implements.length > 0) {
            deps.implements = IO.implements;
            if (this.isGuard(IO.implements)) {
                // We don't want the Guard to show up in the Classes menu
                excludeFromClassArray = true;
                deps.type = 'guard';
                outputSymbols.guards.push(deps);
            }
        }
        if (typeof IO.ignore === 'undefined') {
            this.debug(deps);
            if (!excludeFromClassArray) {
                outputSymbols.classes.push(deps);
            }
        }
        else {
            this.ignore(deps);
        }
    };
    AngularDependencies.prototype.getSourceFileDecorators = function (initialSrcFile, outputSymbols) {
        var _this = this;
        var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
        var fileName = initialSrcFile.fileName.replace(cleaner, '');
        var scannedFile = initialSrcFile;
        // Search in file for variable statement as routes definitions
        var astFile = typeof ast$2.getSourceFile(initialSrcFile.fileName) !== 'undefined'
            ? ast$2.getSourceFile(initialSrcFile.fileName)
            : ast$2.addExistingSourceFile(initialSrcFile.fileName);
        var variableRoutesStatements = astFile.getVariableStatements();
        var hasRoutesStatements = false;
        if (variableRoutesStatements.length > 0) {
            // Clean file for spread and dynamics inside routes definitions
            variableRoutesStatements.forEach(function (s) {
                var variableDeclarations = s.getDeclarations();
                var len = variableDeclarations.length;
                var i = 0;
                for (i; i < len; i++) {
                    if (variableDeclarations[i].compilerNode.type) {
                        if (variableDeclarations[i].compilerNode.type.typeName &&
                            variableDeclarations[i].compilerNode.type.typeName.text === 'Routes') {
                            hasRoutesStatements = true;
                        }
                    }
                }
            });
        }
        if (hasRoutesStatements && !Configuration$1.mainData.disableRoutesGraph) {
            // Clean file for spread and dynamics inside routes definitions
            logger.info('Analysing routes definitions and clean them if necessary');
            // scannedFile = RouterParserUtil.cleanFileIdentifiers(astFile).compilerNode;
            var firstClean = RouterParserUtil$1.cleanFileSpreads(astFile).compilerNode;
            scannedFile = RouterParserUtil$1.cleanCallExpressions(astFile).compilerNode;
            scannedFile = RouterParserUtil$1.cleanFileDynamics(astFile).compilerNode;
            scannedFile.kind = Ast.SyntaxKind.SourceFile;
        }
        Ast.ts.forEachChild(scannedFile, function (initialNode) {
            if (_this.jsDocHelper.hasJSDocInternalTag(fileName, scannedFile, initialNode) &&
                Configuration$1.mainData.disableInternal) {
                return;
            }
            var parseNode = function (file, srcFile, node, fileBody) {
                var sourceCode = srcFile.getText();
                var hash = crypto$5
                    .createHash('md5')
                    .update(sourceCode)
                    .digest('hex');
                if (node.decorators) {
                    var classWithCustomDecorator_1 = false;
                    var visitDecorator = function (visitedDecorator, index) {
                        var deps;
                        var metadata = node.decorators;
                        var name = _this.getSymboleName(node);
                        var props = _this.findProperties(visitedDecorator, srcFile);
                        var IO = _this.componentHelper.getComponentIO(file, srcFile, node, fileBody);
                        if (_this.isModule(visitedDecorator)) {
                            var moduleDep = new ModuleDepFactory(_this.moduleHelper).create(file, srcFile, name, props, IO);
                            if (RouterParserUtil$1.hasRouterModuleInImports(moduleDep.imports)) {
                                RouterParserUtil$1.addModuleWithRoutes(name, _this.moduleHelper.getModuleImportsRaw(props, srcFile), file);
                            }
                            deps = moduleDep;
                            if (typeof IO.ignore === 'undefined') {
                                RouterParserUtil$1.addModule(name, moduleDep.imports);
                                outputSymbols.modules.push(moduleDep);
                                outputSymbols.modulesForGraph.push(moduleDep);
                            }
                        }
                        else if (_this.isComponent(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var componentDep = new ComponentDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = componentDep;
                            if (typeof IO.ignore === 'undefined') {
                                ComponentsTreeEngine$1.addComponent(componentDep);
                                outputSymbols.components.push(componentDep);
                            }
                        }
                        else if (_this.isController(visitedDecorator)) {
                            var controllerDep = new ControllerDepFactory().create(file, srcFile, name, props, IO);
                            deps = controllerDep;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.controllers.push(controllerDep);
                            }
                        }
                        else if (_this.isInjectable(visitedDecorator)) {
                            var injectableDeps = {
                                name: name,
                                id: 'injectable-' + name + '-' + hash,
                                file: file,
                                properties: IO.properties,
                                methods: IO.methods,
                                description: IO.description,
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.constructor) {
                                injectableDeps.constructorObj = IO.constructor;
                            }
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                injectableDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            if (IO.accessors) {
                                injectableDeps.accessors = IO.accessors;
                            }
                            if (IO.extends) {
                                injectableDeps.extends = IO.extends;
                            }
                            deps = injectableDeps;
                            if (typeof IO.ignore === 'undefined') {
                                if (_.includes(IO.implements, 'HttpInterceptor')) {
                                    injectableDeps.type = 'interceptor';
                                    outputSymbols.interceptors.push(injectableDeps);
                                }
                                else if (_this.isGuard(IO.implements)) {
                                    injectableDeps.type = 'guard';
                                    outputSymbols.guards.push(injectableDeps);
                                }
                                else {
                                    injectableDeps.type = 'injectable';
                                    _this.addNewEntityInStore(injectableDeps, outputSymbols.injectables);
                                }
                            }
                        }
                        else if (_this.isPipe(visitedDecorator)) {
                            var pipeDeps = {
                                name: name,
                                id: 'pipe-' + name + '-' + hash,
                                file: file,
                                type: 'pipe',
                                description: IO.description,
                                properties: IO.properties,
                                methods: IO.methods,
                                pure: _this.componentHelper.getComponentPure(props, srcFile),
                                ngname: _this.componentHelper.getComponentName(props, srcFile),
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                pipeDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            deps = pipeDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.pipes.push(pipeDeps);
                            }
                        }
                        else if (_this.isDirective(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var directiveDeps = new DirectiveDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = directiveDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.directives.push(directiveDeps);
                            }
                        }
                        else {
                            var hasMultipleDecoratorsWithInternalOne = _this.hasInternalDecorator(node.decorators);
                            // Just a class
                            if (!classWithCustomDecorator_1 &&
                                !hasMultipleDecoratorsWithInternalOne) {
                                classWithCustomDecorator_1 = true;
                                _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                            }
                        }
                        _this.cache.set(name, deps);
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(deps);
                        }
                        else {
                            _this.ignore(deps);
                        }
                    };
                    var filterByDecorators = function (filteredNode) {
                        if (filteredNode.expression && filteredNode.expression.expression) {
                            var _test = /(NgModule|Component|Injectable|Pipe|Directive)/.test(filteredNode.expression.expression.text);
                            if (!_test && Ast.ts.isClassDeclaration(node)) {
                                _test = true;
                            }
                            return _test;
                        }
                        if (Ast.ts.isClassDeclaration(node)) {
                            return true;
                        }
                        return false;
                    };
                    node.decorators.filter(filterByDecorators).forEach(visitDecorator);
                }
                else if (node.symbol) {
                    if (node.symbol.flags === Ast.ts.SymbolFlags.Class) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    else if (node.symbol.flags === Ast.ts.SymbolFlags.Interface) {
                        var name = _this.getSymboleName(node);
                        var IO = _this.getInterfaceIO(file, srcFile, node, fileBody);
                        var interfaceDeps = {
                            name: name,
                            id: 'interface-' + name + '-' + hash,
                            file: file,
                            type: 'interface',
                            sourceCode: srcFile.getText()
                        };
                        if (IO.properties) {
                            interfaceDeps.properties = IO.properties;
                        }
                        if (IO.indexSignatures) {
                            interfaceDeps.indexSignatures = IO.indexSignatures;
                        }
                        if (IO.kind) {
                            interfaceDeps.kind = IO.kind;
                        }
                        if (IO.description) {
                            interfaceDeps.description = IO.description;
                        }
                        if (IO.methods) {
                            interfaceDeps.methods = IO.methods;
                        }
                        if (IO.extends) {
                            interfaceDeps.extends = IO.extends;
                        }
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(interfaceDeps);
                            outputSymbols.interfaces.push(interfaceDeps);
                        }
                        else {
                            _this.ignore(interfaceDeps);
                        }
                    }
                    else if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        // let tags = this.visitFunctionDeclarationJSDocTags(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            file: file,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    else if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                    else if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var typeAliasDeps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            typeAliasDeps.kind = node.type.kind;
                            if (typeAliasDeps.rawtype === '') {
                                typeAliasDeps.rawtype = kindToType(node.type.kind);
                            }
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(typeAliasDeps);
                        }
                    }
                    else if (Ast.ts.isModuleDeclaration(node)) {
                        if (node.body) {
                            if (node.body.statements && node.body.statements.length > 0) {
                                node.body.statements.forEach(function (statement) {
                                    return parseNode(file, srcFile, statement, node.body);
                                });
                            }
                        }
                    }
                }
                else {
                    var IO = _this.getRouteIO(file, srcFile, node);
                    if (IO.routes) {
                        var newRoutes = void 0;
                        try {
                            newRoutes = RouterParserUtil$1.cleanRawRouteParsed(IO.routes);
                        }
                        catch (e) {
                            // tslint:disable-next-line:max-line-length
                            logger.error('Routes parsing error, maybe a trailing comma or an external variable, trying to fix that later after sources scanning.');
                            newRoutes = IO.routes.replace(/ /gm, '');
                            RouterParserUtil$1.addIncompleteRoute({
                                data: newRoutes,
                                file: file
                            });
                            return true;
                        }
                        outputSymbols.routes = outputSymbols.routes.concat(newRoutes);
                    }
                    if (Ast.ts.isClassDeclaration(node)) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    if (Ast.ts.isExpressionStatement(node) || Ast.ts.isIfStatement(node)) {
                        var bootstrapModuleReference = 'bootstrapModule';
                        // Find the root module with bootstrapModule call
                        // 1. find a simple call : platformBrowserDynamic().bootstrapModule(AppModule);
                        // 2. or inside a call :
                        // () => {
                        //     platformBrowserDynamic().bootstrapModule(AppModule);
                        // });
                        // 3. with a catch : platformBrowserDynamic().bootstrapModule(AppModule).catch(error => console.error(error));
                        // 4. with parameters : platformBrowserDynamic().bootstrapModule(AppModule, {}).catch(error => console.error(error));
                        // Find recusively in expression nodes one with name 'bootstrapModule'
                        var rootModule_1;
                        var resultNode = void 0;
                        if (srcFile.text.indexOf(bootstrapModuleReference) !== -1) {
                            if (node.expression) {
                                resultNode = _this.findExpressionByNameInExpressions(node.expression, 'bootstrapModule');
                            }
                            if (typeof node.thenStatement !== 'undefined') {
                                if (node.thenStatement.statements &&
                                    node.thenStatement.statements.length > 0) {
                                    var firstStatement = node.thenStatement.statements[0];
                                    resultNode = _this.findExpressionByNameInExpressions(firstStatement.expression, 'bootstrapModule');
                                }
                            }
                            if (!resultNode) {
                                if (node.expression &&
                                    node.expression.arguments &&
                                    node.expression.arguments.length > 0) {
                                    resultNode = _this.findExpressionByNameInExpressionArguments(node.expression.arguments, 'bootstrapModule');
                                }
                            }
                            if (resultNode) {
                                if (resultNode.arguments.length > 0) {
                                    _.forEach(resultNode.arguments, function (argument) {
                                        if (argument.text) {
                                            rootModule_1 = argument.text;
                                        }
                                    });
                                }
                                if (rootModule_1) {
                                    RouterParserUtil$1.setRootModule(rootModule_1);
                                }
                            }
                        }
                    }
                    if (Ast.ts.isVariableStatement(node) && !RouterParserUtil$1.isVariableRoutes(node)) {
                        var infos = _this.visitVariableDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'variable',
                            file: file
                        };
                        deps.type = infos.type ? infos.type : '';
                        if (infos.defaultValue) {
                            deps.defaultValue = infos.defaultValue;
                        }
                        if (infos.initializer) {
                            deps.initializer = infos.initializer;
                        }
                        if (node.jsDoc && node.jsDoc.length > 0 && node.jsDoc[0].comment) {
                            deps.description = marked$2(node.jsDoc[0].comment);
                        }
                        if (isModuleWithProviders(node)) {
                            var routingInitializer = getModuleWithProviders(node);
                            RouterParserUtil$1.addModuleWithRoutes(name, [routingInitializer], file);
                            RouterParserUtil$1.addModule(name, [routingInitializer]);
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.variables.push(deps);
                        }
                    }
                    if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            deps.kind = node.type.kind;
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(deps);
                        }
                    }
                    if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                }
            };
            parseNode(fileName, scannedFile, initialNode);
        });
    };
    /**
     * Function to in a specific store an entity, and check before is there is not the same one
     * in that store : same name, id and file
     * @param entity Entity to store
     * @param store Store
     */
    AngularDependencies.prototype.addNewEntityInStore = function (entity, store) {
        var findSameEntityInStore = _.filter(store, {
            name: entity.name,
            id: entity.id,
            file: entity.file
        });
        if (findSameEntityInStore.length === 0) {
            store.push(entity);
        }
    };
    AngularDependencies.prototype.debug = function (deps) {
        if (deps) {
            logger.debug('found', "" + deps.name);
        }
        else {
            return;
        }
        ['imports', 'exports', 'declarations', 'providers', 'bootstrap'].forEach(function (symbols) {
            if (deps[symbols] && deps[symbols].length > 0) {
                logger.debug('', "- " + symbols + ":");
                deps[symbols]
                    .map(function (i) { return i.name; })
                    .forEach(function (d) {
                    logger.debug('', "\t- " + d);
                });
            }
        });
    };
    AngularDependencies.prototype.ignore = function (deps) {
        if (deps) {
            logger.warn('ignore', "" + deps.name);
        }
        else {
            return;
        }
    };
    AngularDependencies.prototype.findExpressionByNameInExpressions = function (entryNode, name) {
        var result;
        var loop = function (node, z) {
            if (node) {
                if (node.expression && !node.expression.name) {
                    loop(node.expression, z);
                }
                if (node.expression && node.expression.name) {
                    if (node.expression.name.text === z) {
                        result = node;
                    }
                    else {
                        loop(node.expression, z);
                    }
                }
            }
        };
        loop(entryNode, name);
        return result;
    };
    AngularDependencies.prototype.findExpressionByNameInExpressionArguments = function (arg, name) {
        var result;
        var that = this;
        var i = 0;
        var len = arg.length;
        var loop = function (node, z) {
            if (node.body) {
                if (node.body.statements && node.body.statements.length > 0) {
                    var j = 0;
                    var leng = node.body.statements.length;
                    for (j; j < leng; j++) {
                        result = that.findExpressionByNameInExpressions(node.body.statements[j], z);
                    }
                }
            }
        };
        for (i; i < len; i++) {
            loop(arg[i], name);
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorators = function (decorators, type) {
        var result = false;
        if (decorators.length > 1) {
            _.forEach(decorators, function (decorator) {
                if (decorator.expression.expression) {
                    if (decorator.expression.expression.text === type) {
                        result = true;
                    }
                }
            });
        }
        else {
            if (decorators[0].expression.expression) {
                if (decorators[0].expression.expression.text === type) {
                    result = true;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorator = function (decorator, type) {
        var result = false;
        if (decorator.expression.expression) {
            if (decorator.expression.expression.text === type) {
                result = true;
            }
        }
        return result;
    };
    AngularDependencies.prototype.isController = function (metadata) {
        return this.parseDecorator(metadata, 'Controller');
    };
    AngularDependencies.prototype.isComponent = function (metadata) {
        return this.parseDecorator(metadata, 'Component');
    };
    AngularDependencies.prototype.isPipe = function (metadata) {
        return this.parseDecorator(metadata, 'Pipe');
    };
    AngularDependencies.prototype.isDirective = function (metadata) {
        return this.parseDecorator(metadata, 'Directive');
    };
    AngularDependencies.prototype.isInjectable = function (metadata) {
        return this.parseDecorator(metadata, 'Injectable');
    };
    AngularDependencies.prototype.isModule = function (metadata) {
        return this.parseDecorator(metadata, 'NgModule') || this.parseDecorator(metadata, 'Module');
    };
    AngularDependencies.prototype.hasInternalDecorator = function (metadatas) {
        return (this.parseDecorators(metadatas, 'Controller') ||
            this.parseDecorators(metadatas, 'Component') ||
            this.parseDecorators(metadatas, 'Pipe') ||
            this.parseDecorators(metadatas, 'Directive') ||
            this.parseDecorators(metadatas, 'Injectable') ||
            this.parseDecorators(metadatas, 'NgModule') ||
            this.parseDecorators(metadatas, 'Module'));
    };
    AngularDependencies.prototype.isGuard = function (ioImplements) {
        return (_.includes(ioImplements, 'CanActivate') ||
            _.includes(ioImplements, 'CanActivateChild') ||
            _.includes(ioImplements, 'CanDeactivate') ||
            _.includes(ioImplements, 'Resolve') ||
            _.includes(ioImplements, 'CanLoad'));
    };
    AngularDependencies.prototype.getSymboleName = function (node) {
        return node.name.text;
    };
    AngularDependencies.prototype.findProperties = function (visitedNode, sourceFile) {
        if (visitedNode.expression &&
            visitedNode.expression.arguments &&
            visitedNode.expression.arguments.length > 0) {
            var pop = visitedNode.expression.arguments[0];
            if (pop && pop.properties && pop.properties.length >= 0) {
                return pop.properties;
            }
            else if (pop && pop.kind && pop.kind === Ast.SyntaxKind.StringLiteral) {
                return [pop];
            }
            else {
                logger.warn('Empty metadatas, trying to found it with imports.');
                return ImportsUtil$1.findValueInImportOrLocalVariables(pop.text, sourceFile);
            }
        }
        return [];
    };
    AngularDependencies.prototype.isAngularLifecycleHook = function (methodName) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var ANGULAR_LIFECYCLE_METHODS = [
            'ngOnInit',
            'ngOnChanges',
            'ngDoCheck',
            'ngOnDestroy',
            'ngAfterContentInit',
            'ngAfterContentChecked',
            'ngAfterViewInit',
            'ngAfterViewChecked',
            'writeValue',
            'registerOnChange',
            'registerOnTouched',
            'setDisabledState'
        ];
        return ANGULAR_LIFECYCLE_METHODS.indexOf(methodName) >= 0;
    };
    AngularDependencies.prototype.visitTypeDeclaration = function (node) {
        var result = {
            name: node.name.text,
            kind: node.kind
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitArgument = function (arg) {
        var result = {
            name: arg.name.text,
            type: this.classHelper.visitType(arg)
        };
        if (arg.dotDotDotToken) {
            result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            result.optional = true;
        }
        if (arg.type) {
            result.type = this.mapType(arg.type.kind);
            if (arg.type.kind === 157) {
                // try replace TypeReference with typeName
                if (arg.type.typeName) {
                    result.type = arg.type.typeName.text;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.mapType = function (type) {
        switch (type) {
            case 95:
                return 'null';
            case 119:
                return 'any';
            case 122:
                return 'boolean';
            case 130:
                return 'never';
            case 133:
                return 'number';
            case 136:
                return 'string';
            case 139:
                return 'undefined';
            case 159:
                return 'typeReference';
        }
    };
    AngularDependencies.prototype.hasPrivateJSDocTag = function (tags) {
        var result = false;
        if (tags) {
            tags.forEach(function (tag) {
                if (tag.tagName && tag.tagName.text && tag.tagName.text === 'private') {
                    result = true;
                }
            });
        }
        return result;
    };
    AngularDependencies.prototype.visitFunctionDeclaration = function (method) {
        var _this = this;
        var methodName = method.name ? method.name.text : 'Unnamed function';
        var result = {
            name: methodName,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : []
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type !== 'undefined') {
            result.returnType = this.classHelper.visitType(method.type);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers
                    .map(function (modifier) {
                    return modifier.kind;
                })
                    .reverse();
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
                _.forEach(jsdoctags[0].tags, function (tag) {
                    if (tag.tagName) {
                        if (tag.tagName.text) {
                            if (tag.tagName.text.indexOf('ignore') > -1) {
                                result.ignore = true;
                            }
                        }
                    }
                });
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    AngularDependencies.prototype.visitVariableDeclaration = function (node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var result = {
                    name: node.declarationList.declarations[i].name.text,
                    defaultValue: node.declarationList.declarations[i].initializer
                        ? this.classHelper.stringifyDefaultValue(node.declarationList.declarations[i].initializer)
                        : undefined
                };
                if (node.declarationList.declarations[i].initializer) {
                    result.initializer = node.declarationList.declarations[i].initializer;
                }
                if (node.declarationList.declarations[i].type) {
                    result.type = this.classHelper.visitType(node.declarationList.declarations[i].type);
                }
                if (typeof result.type === 'undefined' && result.initializer) {
                    result.type = kindToType(result.initializer.kind);
                }
                return result;
            }
        }
    };
    AngularDependencies.prototype.visitFunctionDeclarationJSDocTags = function (node) {
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        var result;
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumTypeAliasFunctionDeclarationDescription = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = marked$2(node.jsDoc[0].comment);
                }
            }
        }
        return description;
    };
    AngularDependencies.prototype.visitEnumDeclaration = function (node) {
        var result = [];
        if (node.members) {
            var i = 0;
            var len = node.members.length;
            for (i; i < len; i++) {
                var member = {
                    name: node.members[i].name.text
                };
                if (node.members[i].initializer) {
                    member.value = node.members[i].initializer.text;
                }
                result.push(member);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumDeclarationForRoutes = function (fileName, node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var routesInitializer = node.declarationList.declarations[i].initializer;
                var data = new CodeGenerator().generate(routesInitializer);
                RouterParserUtil$1.addRoute({
                    name: node.declarationList.declarations[i].name.text,
                    data: RouterParserUtil$1.cleanRawRoute(data),
                    filename: fileName
                });
                return [
                    {
                        routes: data
                    }
                ];
            }
        }
        return [];
    };
    AngularDependencies.prototype.getRouteIO = function (filename, sourceFile, node) {
        var _this = this;
        var res;
        if (sourceFile.statements) {
            res = sourceFile.statements.reduce(function (directive, statement) {
                if (RouterParserUtil$1.isVariableRoutes(statement)) {
                    if (statement.pos === node.pos && statement.end === node.end) {
                        return directive.concat(_this.visitEnumDeclarationForRoutes(filename, statement));
                    }
                }
                return directive;
            }, []);
            return res[0] || {};
        }
        else {
            return {};
        }
    };
    AngularDependencies.prototype.getClassIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    AngularDependencies.prototype.getInterfaceIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isInterfaceDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    return AngularDependencies;
}(FrameworkDependencies));

var AngularJSDependencies = /** @class */ (function (_super) {
    __extends(AngularJSDependencies, _super);
    function AngularJSDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        return _this;
    }
    AngularJSDependencies.prototype.getDependencies = function () {
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            injectables: [],
            interceptors: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        return deps;
    };
    return AngularJSDependencies;
}(FrameworkDependencies));

function promiseSequential(promises) {
    if (!Array.isArray(promises)) {
        throw new Error('First argument need to be an array of Promises');
    }
    return new Promise(function (resolve, reject) {
        var count = 0;
        var results = [];
        var iterateeFunc = function (previousPromise, currentPromise) {
            return previousPromise
                .then(function (result) {
                if (count++ !== 0) {
                    results = results.concat(result);
                }
                return currentPromise(result, results, count);
            })
                .catch(function (err) {
                return reject(err);
            });
        };
        promises = promises.concat(function () { return Promise.resolve(); });
        promises.reduce(iterateeFunc, Promise.resolve(false)).then(function (res) {
            resolve(results);
        });
    });
}

var chokidar = require('chokidar');
var marked$3 = require('marked');
var traverse$3 = require('traverse');
var crypto$6 = require('crypto');
var cwd = process.cwd();
var startTime = new Date();
var generationPromiseResolve;
var generationPromiseReject;
var generationPromise = new Promise(function (resolve, reject) {
    generationPromiseResolve = resolve;
    generationPromiseReject = reject;
});
var Application = /** @class */ (function () {
    /**
     * Create a new compodoc application instance.
     *
     * @param options An object containing the options that should be used.
     */
    function Application(options) {
        var _this = this;
        /**
         * Files changed during watch scanning
         */
        this.watchChangedFiles = [];
        /**
         * Boolean for watching status
         * @type {boolean}
         */
        this.isWatching = false;
        /**
         * Store package.json data
         */
        this.packageJsonData = {};
        this.preparePipes = function (somePipes) {
            logger.info('Prepare pipes');
            Configuration$1.mainData.pipes = somePipes ? somePipes : DependenciesEngine$1.getPipes();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.pipes.length;
                var loop = function () {
                    if (i < len) {
                        var pipe = Configuration$1.mainData.pipes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(pipe.file)) {
                            logger.info(" " + pipe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(pipe.file);
                            pipe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'pipes',
                            name: pipe.name,
                            id: pipe.id,
                            navTabs: _this.getNavTabs(pipe),
                            context: 'pipe',
                            pipe: pipe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (pipe.isDuplicate) {
                            page.name += '-' + pipe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        this.prepareClasses = function (someClasses) {
            logger.info('Prepare classes');
            Configuration$1.mainData.classes = someClasses
                ? someClasses
                : DependenciesEngine$1.getClasses();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.classes.length;
                var loop = function () {
                    if (i < len) {
                        var classe = Configuration$1.mainData.classes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(classe.file)) {
                            logger.info(" " + classe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(classe.file);
                            classe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'classes',
                            name: classe.name,
                            id: classe.id,
                            navTabs: _this.getNavTabs(classe),
                            context: 'class',
                            class: classe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (classe.isDuplicate) {
                            page.name += '-' + classe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        for (var option in options) {
            if (typeof Configuration$1.mainData[option] !== 'undefined') {
                Configuration$1.mainData[option] = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'name') {
                Configuration$1.mainData.documentationMainName = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'silent') {
                logger.silent = false;
            }
        }
    }
    /**
     * Start compodoc process
     */
    Application.prototype.generate = function () {
        var _this = this;
        process.on('unhandledRejection', this.unhandledRejectionListener);
        process.on('uncaughtException', this.uncaughtExceptionListener);
        I18nEngine$1.init(Configuration$1.mainData.language);
        if (Configuration$1.mainData.output.charAt(Configuration$1.mainData.output.length - 1) !== '/') {
            Configuration$1.mainData.output += '/';
        }
        if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
            this.processPackageJson();
        }
        else {
            HtmlEngine$1.init(Configuration$1.mainData.templates).then(function () { return _this.processPackageJson(); });
        }
        return generationPromise;
    };
    Application.prototype.endCallback = function () {
        process.removeListener('unhandledRejection', this.unhandledRejectionListener);
        process.removeListener('uncaughtException', this.uncaughtExceptionListener);
    };
    Application.prototype.unhandledRejectionListener = function (err, p) {
        console.log('Unhandled Rejection at:', p, 'reason:', err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    Application.prototype.uncaughtExceptionListener = function (err) {
        logger.error(err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    /**
     * Start compodoc documentation coverage
     */
    Application.prototype.testCoverage = function () {
        this.getDependenciesData();
    };
    /**
     * Store files for initial processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setFiles = function (files) {
        this.files = files;
    };
    /**
     * Store files for watch processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setUpdatedFiles = function (files) {
        this.updatedFiles = files;
    };
    /**
     * Return a boolean indicating presence of one TypeScript file in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesTSFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.ts') {
                result = true;
            }
        });
        return result;
    };
    /**
     * Return a boolean indicating presence of one root markdown files in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesRootMarkdownFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.md' && path.dirname(file) === cwd) {
                result = true;
            }
        });
        return result;
    };
    /**
     * Clear files for watch processing
     */
    Application.prototype.clearUpdatedFiles = function () {
        this.updatedFiles = [];
        this.watchChangedFiles = [];
    };
    Application.prototype.processPackageJson = function () {
        var _this = this;
        logger.info('Searching package.json file');
        FileEngine$1.get(cwd + path.sep + 'package.json').then(function (packageData) {
            var parsedData = JSON.parse(packageData);
            _this.packageJsonData = parsedData;
            if (typeof parsedData.name !== 'undefined' &&
                Configuration$1.mainData.documentationMainName === COMPODOC_DEFAULTS.title) {
                Configuration$1.mainData.documentationMainName =
                    parsedData.name + ' documentation';
            }
            if (typeof parsedData.description !== 'undefined') {
                Configuration$1.mainData.documentationMainDescription = parsedData.description;
            }
            Configuration$1.mainData.angularVersion = AngularVersionUtil$1.getAngularVersionOfProject(parsedData);
            logger.info('package.json file found');
            if (!Configuration$1.mainData.disableDependencies) {
                if (typeof parsedData.dependencies !== 'undefined') {
                    _this.processPackageDependencies(parsedData.dependencies);
                }
                if (typeof parsedData.peerDependencies !== 'undefined') {
                    _this.processPackagePeerDependencies(parsedData.peerDependencies);
                }
            }
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage) {
                logger.error(errorMessage);
            });
        }, function (errorMessage) {
            logger.error(errorMessage);
            logger.error('Continuing without package.json file');
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage1) {
                logger.error(errorMessage1);
            });
        });
    };
    Application.prototype.processPackagePeerDependencies = function (dependencies) {
        logger.info('Processing package.json peerDependencies');
        Configuration$1.mainData.packagePeerDependencies = dependencies;
        if (!Configuration$1.hasPage('dependencies')) {
            Configuration$1.addPage({
                name: 'dependencies',
                id: 'packageDependencies',
                context: 'package-dependencies',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
        }
    };
    Application.prototype.processPackageDependencies = function (dependencies) {
        logger.info('Processing package.json dependencies');
        Configuration$1.mainData.packageDependencies = dependencies;
        Configuration$1.addPage({
            name: 'dependencies',
            id: 'packageDependencies',
            context: 'package-dependencies',
            depth: 0,
            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
        });
    };
    Application.prototype.processMarkdowns = function () {
        logger.info('Searching README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md files');
        return new Promise(function (resolve, reject) {
            var i = 0;
            var markdowns = ['readme', 'changelog', 'contributing', 'license', 'todo'];
            var numberOfMarkdowns = 5;
            var loop = function () {
                if (i < numberOfMarkdowns) {
                    MarkdownEngine$1.getTraditionalMarkdown(markdowns[i].toUpperCase()).then(function (readmeData) {
                        Configuration$1.addPage({
                            name: markdowns[i] === 'readme' ? 'index' : markdowns[i],
                            context: 'getting-started',
                            id: 'getting-started',
                            markdown: readmeData,
                            depth: 0,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                        });
                        if (markdowns[i] === 'readme') {
                            Configuration$1.mainData.readme = true;
                            Configuration$1.addPage({
                                name: 'overview',
                                id: 'overview',
                                context: 'overview',
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        else {
                            Configuration$1.mainData.markdowns.push({
                                name: markdowns[i],
                                uppername: markdowns[i].toUpperCase(),
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        logger.info(markdowns[i].toUpperCase() + ".md file found");
                        i++;
                        loop();
                    }, function (errorMessage) {
                        logger.warn(errorMessage);
                        logger.warn("Continuing without " + markdowns[i].toUpperCase() + ".md file");
                        if (markdowns[i] === 'readme') {
                            Configuration$1.addPage({
                                name: 'index',
                                id: 'index',
                                context: 'overview',
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        i++;
                        loop();
                    });
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.rebuildRootMarkdowns = function () {
        var _this = this;
        logger.info('Regenerating README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md pages');
        var actions = [];
        Configuration$1.resetRootMarkdownPages();
        actions.push(function () {
            return _this.processMarkdowns();
        });
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    /**
     * Get dependency data for small group of updated files during watch process
     */
    Application.prototype.getMicroDependenciesData = function () {
        logger.info('Get diff dependencies data');
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.updatedFiles, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.update(dependenciesData);
        this.prepareJustAFewThings(dependenciesData);
    };
    /**
     * Rebuild external documentation during watch process
     */
    Application.prototype.rebuildExternalDocumentation = function () {
        var _this = this;
        logger.info('Rebuild external documentation');
        var actions = [];
        Configuration$1.resetAdditionalPages();
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.detectAngularJSProjects = function () {
        if (typeof this.packageJsonData.dependencies !== 'undefined') {
            if (typeof this.packageJsonData.dependencies.angular !== 'undefined') ;
            else {
                var countJSFiles_1 = 0;
                this.files.forEach(function (file) {
                    if (path.extname(file) === '.js') {
                        countJSFiles_1 += 1;
                    }
                });
                var percentOfJSFiles = (countJSFiles_1 * 100) / this.files.length;
            }
        }
        return false;
    };
    Application.prototype.getDependenciesData = function () {
        logger.info('Get dependencies data');
        /**
         * AngularJS detection strategy :
         * - if in package.json
         * - if 75% of scanned files are *.js files
         */
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.files, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.init(dependenciesData);
        Configuration$1.mainData.routesLength = RouterParserUtil$1.routesLength();
        this.printStatistics();
        this.prepareEverything();
    };
    Application.prototype.prepareJustAFewThings = function (diffCrawledData) {
        var _this = this;
        var actions = [];
        Configuration$1.resetPages();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () { return _this.prepareRoutes(); });
        }
        if (diffCrawledData.components.length > 0) {
            actions.push(function () { return _this.prepareComponents(); });
        }
        if (diffCrawledData.controllers.length > 0) {
            actions.push(function () { return _this.prepareControllers(); });
        }
        if (diffCrawledData.modules.length > 0) {
            actions.push(function () { return _this.prepareModules(); });
        }
        if (diffCrawledData.directives.length > 0) {
            actions.push(function () { return _this.prepareDirectives(); });
        }
        if (diffCrawledData.injectables.length > 0) {
            actions.push(function () { return _this.prepareInjectables(); });
        }
        if (diffCrawledData.interceptors.length > 0) {
            actions.push(function () { return _this.prepareInterceptors(); });
        }
        if (diffCrawledData.guards.length > 0) {
            actions.push(function () { return _this.prepareGuards(); });
        }
        if (diffCrawledData.pipes.length > 0) {
            actions.push(function () { return _this.preparePipes(); });
        }
        if (diffCrawledData.classes.length > 0) {
            actions.push(function () { return _this.prepareClasses(); });
        }
        if (diffCrawledData.interfaces.length > 0) {
            actions.push(function () { return _this.prepareInterfaces(); });
        }
        if (diffCrawledData.miscellaneous.variables.length > 0 ||
            diffCrawledData.miscellaneous.functions.length > 0 ||
            diffCrawledData.miscellaneous.typealiases.length > 0 ||
            diffCrawledData.miscellaneous.enumerations.length > 0) {
            actions.push(function () { return _this.prepareMiscellaneous(); });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () { return _this.prepareCoverage(); });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processGraphs();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.printStatistics = function () {
        logger.info('-------------------');
        logger.info('Project statistics ');
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- files      : " + this.files.length);
        }
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- module     : " + DependenciesEngine$1.modules.length);
        }
        if (DependenciesEngine$1.components.length > 0) {
            logger.info("- component  : " + DependenciesEngine$1.components.length);
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            logger.info("- controller : " + DependenciesEngine$1.controllers.length);
        }
        if (DependenciesEngine$1.directives.length > 0) {
            logger.info("- directive  : " + DependenciesEngine$1.directives.length);
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            logger.info("- injectable : " + DependenciesEngine$1.injectables.length);
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            logger.info("- injector   : " + DependenciesEngine$1.interceptors.length);
        }
        if (DependenciesEngine$1.guards.length > 0) {
            logger.info("- guard      : " + DependenciesEngine$1.guards.length);
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            logger.info("- pipe       : " + DependenciesEngine$1.pipes.length);
        }
        if (DependenciesEngine$1.classes.length > 0) {
            logger.info("- class      : " + DependenciesEngine$1.classes.length);
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            logger.info("- interface  : " + DependenciesEngine$1.interfaces.length);
        }
        if (Configuration$1.mainData.routesLength > 0) {
            logger.info("- route      : " + Configuration$1.mainData.routesLength);
        }
        logger.info('-------------------');
    };
    Application.prototype.prepareEverything = function () {
        var _this = this;
        var actions = [];
        actions.push(function () {
            return _this.prepareComponents();
        });
        actions.push(function () {
            return _this.prepareModules();
        });
        if (DependenciesEngine$1.directives.length > 0) {
            actions.push(function () {
                return _this.prepareDirectives();
            });
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            actions.push(function () {
                return _this.prepareControllers();
            });
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            actions.push(function () {
                return _this.prepareInjectables();
            });
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            actions.push(function () {
                return _this.prepareInterceptors();
            });
        }
        if (DependenciesEngine$1.guards.length > 0) {
            actions.push(function () {
                return _this.prepareGuards();
            });
        }
        if (DependenciesEngine$1.routes &&
            DependenciesEngine$1.routes.children.length > 0 &&
            !Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () {
                return _this.prepareRoutes();
            });
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            actions.push(function () {
                return _this.preparePipes();
            });
        }
        if (DependenciesEngine$1.classes.length > 0) {
            actions.push(function () {
                return _this.prepareClasses();
            });
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            actions.push(function () {
                return _this.prepareInterfaces();
            });
        }
        if (DependenciesEngine$1.miscellaneous.variables.length > 0 ||
            DependenciesEngine$1.miscellaneous.functions.length > 0 ||
            DependenciesEngine$1.miscellaneous.typealiases.length > 0 ||
            DependenciesEngine$1.miscellaneous.enumerations.length > 0) {
            actions.push(function () {
                return _this.prepareMiscellaneous();
            });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () {
                return _this.prepareCoverage();
            });
        }
        if (Configuration$1.mainData.unitTestCoverage !== '') {
            actions.push(function () {
                return _this.prepareUnitTestCoverage();
            });
        }
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
                if (COMPODOC_DEFAULTS.exportFormatsSupported.indexOf(Configuration$1.mainData.exportFormat) > -1) {
                    logger.info("Generating documentation in export format " + Configuration$1.mainData.exportFormat);
                    ExportEngine$1.export(Configuration$1.mainData.output, Configuration$1.mainData).then(function () {
                        generationPromiseResolve();
                        _this.endCallback();
                        logger.info('Documentation generated in ' +
                            Configuration$1.mainData.output +
                            ' in ' +
                            _this.getElapsedTime() +
                            ' seconds');
                    });
                }
                else {
                    logger.warn("Exported format not supported");
                }
            }
            else {
                _this.processGraphs();
            }
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.getIncludedPathForFile = function (file) {
        return path.join(Configuration$1.mainData.includes, file);
    };
    Application.prototype.prepareExternalIncludes = function () {
        var _this = this;
        logger.info('Adding external markdown files');
        // Scan include folder for files detailed in summary.json
        // For each file, add to Configuration.mainData.additionalPages
        // Each file will be converted to html page, inside COMPODOC_DEFAULTS.additionalEntryPath
        return new Promise(function (resolve, reject) {
            FileEngine$1.get(_this.getIncludedPathForFile('summary.json')).then(function (summaryData) {
                logger.info('Additional documentation: summary.json file found');
                var parsedSummaryData = JSON.parse(summaryData);
                var that = _this;
                var lastLevelOnePage = undefined;
                traverse$3(parsedSummaryData).forEach(function () {
                    // tslint:disable-next-line:no-invalid-this
                    if (this.notRoot && typeof this.node === 'object') {
                        // tslint:disable-next-line:no-invalid-this
                        var rawPath = this.path;
                        // tslint:disable-next-line:no-invalid-this
                        var additionalNode = this.node;
                        var file = additionalNode.file;
                        var title = additionalNode.title;
                        var finalPath_1 = Configuration$1.mainData.includesFolder;
                        var finalDepth = rawPath.filter(function (el) {
                            return !isNaN(parseInt(el, 10));
                        });
                        if (typeof file !== 'undefined' && typeof title !== 'undefined') {
                            var url = cleanNameWithoutSpaceAndToLowerCase(title);
                            /**
                             * Id created with title + file path hash, seems to be hypothetically unique here
                             */
                            var id = crypto$6
                                .createHash('md5')
                                .update(title + file)
                                .digest('hex');
                            // tslint:disable-next-line:no-invalid-this
                            this.node.id = id;
                            var lastElementRootTree_1 = undefined;
                            finalDepth.forEach(function (el) {
                                var elementTree = typeof lastElementRootTree_1 === 'undefined'
                                    ? parsedSummaryData
                                    : lastElementRootTree_1;
                                if (typeof elementTree.children !== 'undefined') {
                                    elementTree = elementTree.children[el];
                                }
                                else {
                                    elementTree = elementTree[el];
                                }
                                finalPath_1 +=
                                    '/' +
                                        cleanNameWithoutSpaceAndToLowerCase(elementTree.title);
                                lastElementRootTree_1 = elementTree;
                            });
                            finalPath_1 = finalPath_1.replace('/' + url, '');
                            var markdownFile = MarkdownEngine$1.getTraditionalMarkdownSync(that.getIncludedPathForFile(file));
                            if (finalDepth.length > 5) {
                                logger.error('Only 5 levels of depth are supported');
                            }
                            else {
                                var _page = {
                                    name: title,
                                    id: id,
                                    filename: url,
                                    context: 'additional-page',
                                    path: finalPath_1,
                                    additionalPage: markdownFile,
                                    depth: finalDepth.length,
                                    childrenLength: additionalNode.children
                                        ? additionalNode.children.length
                                        : 0,
                                    children: [],
                                    lastChild: false,
                                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                                };
                                if (finalDepth.length === 1) {
                                    lastLevelOnePage = _page;
                                }
                                if (finalDepth.length > 1) {
                                    // store all child pages of the last root level 1 page inside it
                                    lastLevelOnePage.children.push(_page);
                                }
                                else {
                                    Configuration$1.addAdditionalPage(_page);
                                }
                            }
                        }
                    }
                });
                resolve();
            }, function (errorMessage) {
                logger.error(errorMessage);
                reject('Error during Additional documentation generation');
            });
        });
    };
    Application.prototype.prepareModules = function (someModules) {
        var _this = this;
        logger.info('Prepare modules');
        var i = 0;
        var _modules = someModules ? someModules : DependenciesEngine$1.getModules();
        return new Promise(function (resolve, reject) {
            Configuration$1.mainData.modules = _modules.map(function (ngModule) {
                ngModule.compodocLinks = {
                    components: [],
                    controllers: [],
                    directives: [],
                    injectables: [],
                    pipes: []
                };
                ['declarations', 'bootstrap', 'imports', 'exports', 'controllers'].forEach(function (metadataType) {
                    ngModule[metadataType] = ngModule[metadataType].filter(function (metaDataItem) {
                        switch (metaDataItem.type) {
                            case 'directive':
                                return DependenciesEngine$1.getDirectives().some(function (directive) {
                                    var selectedDirective;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedDirective =
                                            directive.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedDirective =
                                            directive.name === metaDataItem.name;
                                    }
                                    if (selectedDirective &&
                                        !ngModule.compodocLinks.directives.includes(directive)) {
                                        ngModule.compodocLinks.directives.push(directive);
                                    }
                                    return selectedDirective;
                                });
                            case 'component':
                                return DependenciesEngine$1.getComponents().some(function (component) {
                                    var selectedComponent;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedComponent =
                                            component.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedComponent =
                                            component.name === metaDataItem.name;
                                    }
                                    if (selectedComponent &&
                                        !ngModule.compodocLinks.components.includes(component)) {
                                        ngModule.compodocLinks.components.push(component);
                                    }
                                    return selectedComponent;
                                });
                            case 'controller':
                                return DependenciesEngine$1.getControllers().some(function (controller) {
                                    var selectedController;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedController =
                                            controller.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedController =
                                            controller.name === metaDataItem.name;
                                    }
                                    if (selectedController &&
                                        !ngModule.compodocLinks.controllers.includes(controller)) {
                                        ngModule.compodocLinks.controllers.push(controller);
                                    }
                                    return selectedController;
                                });
                            case 'module':
                                return DependenciesEngine$1.getModules().some(function (module) { return module.name === metaDataItem.name; });
                            case 'pipe':
                                return DependenciesEngine$1.getPipes().some(function (pipe) {
                                    var selectedPipe;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedPipe = pipe.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedPipe = pipe.name === metaDataItem.name;
                                    }
                                    if (selectedPipe &&
                                        !ngModule.compodocLinks.pipes.includes(pipe)) {
                                        ngModule.compodocLinks.pipes.push(pipe);
                                    }
                                    return selectedPipe;
                                });
                            default:
                                return true;
                        }
                    });
                });
                ngModule.providers = ngModule.providers.filter(function (provider) {
                    return (DependenciesEngine$1.getInjectables().some(function (injectable) {
                        var selectedInjectable = injectable.name === provider.name;
                        if (selectedInjectable &&
                            !ngModule.compodocLinks.injectables.includes(injectable)) {
                            ngModule.compodocLinks.injectables.push(injectable);
                        }
                        return selectedInjectable;
                    }) ||
                        DependenciesEngine$1.getInterceptors().some(function (interceptor) { return interceptor.name === provider.name; }));
                });
                // Try fixing type undefined for each providers
                _.forEach(ngModule.providers, function (provider) {
                    if (DependenciesEngine$1.getInjectables().find(function (injectable) { return injectable.name === provider.name; })) {
                        provider.type = 'injectable';
                    }
                    if (DependenciesEngine$1.getInterceptors().find(function (interceptor) { return interceptor.name === provider.name; })) {
                        provider.type = 'interceptor';
                    }
                });
                // Order things
                ngModule.compodocLinks.components = _.sortBy(ngModule.compodocLinks.components, [
                    'name'
                ]);
                ngModule.compodocLinks.controllers = _.sortBy(ngModule.compodocLinks.controllers, [
                    'name'
                ]);
                ngModule.compodocLinks.directives = _.sortBy(ngModule.compodocLinks.directives, [
                    'name'
                ]);
                ngModule.compodocLinks.injectables = _.sortBy(ngModule.compodocLinks.injectables, [
                    'name'
                ]);
                ngModule.compodocLinks.pipes = _.sortBy(ngModule.compodocLinks.pipes, ['name']);
                ngModule.declarations = _.sortBy(ngModule.declarations, ['name']);
                ngModule.entryComponents = _.sortBy(ngModule.entryComponents, ['name']);
                ngModule.providers = _.sortBy(ngModule.providers, ['name']);
                ngModule.imports = _.sortBy(ngModule.imports, ['name']);
                ngModule.exports = _.sortBy(ngModule.exports, ['name']);
                return ngModule;
            });
            Configuration$1.addPage({
                name: 'modules',
                id: 'modules',
                context: 'modules',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            var len = Configuration$1.mainData.modules.length;
            var loop = function () {
                if (i < len) {
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(Configuration$1.mainData.modules[i].file)) {
                        logger.info(" " + Configuration$1.mainData.modules[i].name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(Configuration$1.mainData.modules[i].file);
                        Configuration$1.mainData.modules[i].readme = marked$3(readme);
                    }
                    Configuration$1.addPage({
                        path: 'modules',
                        name: Configuration$1.mainData.modules[i].name,
                        id: Configuration$1.mainData.modules[i].id,
                        navTabs: _this.getNavTabs(Configuration$1.mainData.modules[i]),
                        context: 'module',
                        module: Configuration$1.mainData.modules[i],
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    });
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterfaces = function (someInterfaces) {
        var _this = this;
        logger.info('Prepare interfaces');
        Configuration$1.mainData.interfaces = someInterfaces
            ? someInterfaces
            : DependenciesEngine$1.getInterfaces();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interfaces.length;
            var loop = function () {
                if (i < len) {
                    var interf = Configuration$1.mainData.interfaces[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interf.file)) {
                        logger.info(" " + interf.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interf.file);
                        interf.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interfaces',
                        name: interf.name,
                        id: interf.id,
                        navTabs: _this.getNavTabs(interf),
                        context: 'interface',
                        interface: interf,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interf.isDuplicate) {
                        page.name += '-' + interf.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareMiscellaneous = function (someMisc) {
        logger.info('Prepare miscellaneous');
        Configuration$1.mainData.miscellaneous = someMisc
            ? someMisc
            : DependenciesEngine$1.getMiscellaneous();
        return new Promise(function (resolve, reject) {
            if (Configuration$1.mainData.miscellaneous.functions.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'functions',
                    id: 'miscellaneous-functions',
                    context: 'miscellaneous-functions',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.variables.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'variables',
                    id: 'miscellaneous-variables',
                    context: 'miscellaneous-variables',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.typealiases.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'typealiases',
                    id: 'miscellaneous-typealiases',
                    context: 'miscellaneous-typealiases',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.enumerations.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'enumerations',
                    id: 'miscellaneous-enumerations',
                    context: 'miscellaneous-enumerations',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            resolve();
        });
    };
    Application.prototype.handleTemplateurl = function (component) {
        var dirname = path.dirname(component.file);
        var templatePath = path.resolve(dirname + path.sep + component.templateUrl);
        if (!FileEngine$1.existsSync(templatePath)) {
            var err = "Cannot read template for " + component.name;
            logger.error(err);
            return new Promise(function (resolve, reject) { });
        }
        return FileEngine$1.get(templatePath).then(function (data) { return (component.templateData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.handleStyles = function (component) {
        var styles = component.styles;
        component.stylesData = '';
        return new Promise(function (resolveStyles, rejectStyles) {
            styles.forEach(function (style) {
                component.stylesData = component.stylesData + style + '\n';
            });
            resolveStyles();
        });
    };
    Application.prototype.handleStyleurls = function (component) {
        var dirname = path.dirname(component.file);
        var styleDataPromise = component.styleUrls.map(function (styleUrl) {
            var stylePath = path.resolve(dirname + path.sep + styleUrl);
            if (!FileEngine$1.existsSync(stylePath)) {
                var err = "Cannot read style url " + stylePath + " for " + component.name;
                logger.error(err);
                return new Promise(function (resolve, reject) { });
            }
            return new Promise(function (resolve, reject) {
                FileEngine$1.get(stylePath).then(function (data) {
                    resolve({
                        data: data,
                        styleUrl: styleUrl
                    });
                });
            });
        });
        return Promise.all(styleDataPromise).then(function (data) { return (component.styleUrlsData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.getNavTabs = function (dependency) {
        var navTabConfig = Configuration$1.mainData.navTabConfig;
        navTabConfig =
            navTabConfig.length === 0
                ? _.cloneDeep(COMPODOC_CONSTANTS.navTabDefinitions)
                : navTabConfig;
        var matchDepType = function (depType) {
            return depType === 'all' || depType === dependency.type;
        };
        var navTabs = [];
        _.forEach(navTabConfig, function (customTab) {
            var navTab = _.find(COMPODOC_CONSTANTS.navTabDefinitions, { id: customTab.id });
            if (!navTab) {
                throw new Error("Invalid tab ID '" + customTab.id + "' specified in tab configuration");
            }
            navTab.label = customTab.label;
            // is tab applicable to target dependency?
            if (-1 === _.findIndex(navTab.depTypes, matchDepType)) {
                return;
            }
            // global config
            if (customTab.id === 'tree' && Configuration$1.mainData.disableDomTree) {
                return;
            }
            if (customTab.id === 'source' && Configuration$1.mainData.disableSourceCode) {
                return;
            }
            if (customTab.id === 'templateData' && Configuration$1.mainData.disableTemplateTab) {
                return;
            }
            if (customTab.id === 'styleData' && Configuration$1.mainData.disableStyleTab) {
                return;
            }
            // per dependency config
            if (customTab.id === 'readme' && !dependency.readme) {
                return;
            }
            if (customTab.id === 'example' && !dependency.exampleUrls) {
                return;
            }
            if (customTab.id === 'templateData' &&
                (!dependency.templateUrl || dependency.templateUrl.length === 0)) {
                return;
            }
            if (customTab.id === 'styleData' &&
                ((!dependency.styleUrls || dependency.styleUrls.length === 0) &&
                    (!dependency.styles || dependency.styles.length === 0))) {
                return;
            }
            navTabs.push(navTab);
        });
        if (navTabs.length === 0) {
            throw new Error("No valid navigation tabs have been defined for dependency type '" + dependency.type + "'. Specify at least one config for the 'info' or 'source' tab in --navTabConfig.");
        }
        return navTabs;
    };
    Application.prototype.prepareControllers = function (someControllers) {
        var _this = this;
        logger.info('Prepare controllers');
        Configuration$1.mainData.controllers = someControllers
            ? someControllers
            : DependenciesEngine$1.getControllers();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.controllers.length;
            var loop = function () {
                if (i < len) {
                    var controller = Configuration$1.mainData.controllers[i];
                    var page = {
                        path: 'controllers',
                        name: controller.name,
                        id: controller.id,
                        navTabs: _this.getNavTabs(controller),
                        context: 'controller',
                        controller: controller,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (controller.isDuplicate) {
                        page.name += '-' + controller.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareComponents = function (someComponents) {
        var _this = this;
        logger.info('Prepare components');
        Configuration$1.mainData.components = someComponents
            ? someComponents
            : DependenciesEngine$1.getComponents();
        return new Promise(function (mainPrepareComponentResolve, mainPrepareComponentReject) {
            var i = 0;
            var len = Configuration$1.mainData.components.length;
            var loop = function () {
                if (i <= len - 1) {
                    var component_1 = Configuration$1.mainData.components[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(component_1.file)) {
                        logger.info(" " + component_1.name + " has a README file, include it");
                        var readmeFile = MarkdownEngine$1.readNeighbourReadmeFile(component_1.file);
                        component_1.readme = marked$3(readmeFile);
                    }
                    var page = {
                        path: 'components',
                        name: component_1.name,
                        id: component_1.id,
                        navTabs: _this.getNavTabs(component_1),
                        context: 'component',
                        component: component_1,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (component_1.isDuplicate) {
                        page.name += '-' + component_1.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    var componentTemplateUrlPromise = new Promise(function (componentTemplateUrlResolve, componentTemplateUrlReject) {
                        if (component_1.templateUrl.length > 0) {
                            logger.info(" " + component_1.name + " has a templateUrl, include it");
                            _this.handleTemplateurl(component_1).then(function () {
                                componentTemplateUrlResolve();
                            }, function (e) {
                                logger.error(e);
                                componentTemplateUrlReject();
                            });
                        }
                        else {
                            componentTemplateUrlResolve();
                        }
                    });
                    var componentStyleUrlsPromise = new Promise(function (componentStyleUrlsResolve, componentStyleUrlsReject) {
                        if (component_1.styleUrls.length > 0) {
                            logger.info(" " + component_1.name + " has styleUrls, include them");
                            _this.handleStyleurls(component_1).then(function () {
                                componentStyleUrlsResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStyleUrlsReject();
                            });
                        }
                        else {
                            componentStyleUrlsResolve();
                        }
                    });
                    var componentStylesPromise = new Promise(function (componentStylesResolve, componentStylesReject) {
                        if (component_1.styles.length > 0) {
                            logger.info(" " + component_1.name + " has styles, include them");
                            _this.handleStyles(component_1).then(function () {
                                componentStylesResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStylesReject();
                            });
                        }
                        else {
                            componentStylesResolve();
                        }
                    });
                    Promise.all([
                        componentTemplateUrlPromise,
                        componentStyleUrlsPromise,
                        componentStylesPromise
                    ]).then(function () {
                        i++;
                        loop();
                    });
                }
                else {
                    mainPrepareComponentResolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareDirectives = function (someDirectives) {
        var _this = this;
        logger.info('Prepare directives');
        Configuration$1.mainData.directives = someDirectives
            ? someDirectives
            : DependenciesEngine$1.getDirectives();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.directives.length;
            var loop = function () {
                if (i < len) {
                    var directive = Configuration$1.mainData.directives[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(directive.file)) {
                        logger.info(" " + directive.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(directive.file);
                        directive.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'directives',
                        name: directive.name,
                        id: directive.id,
                        navTabs: _this.getNavTabs(directive),
                        context: 'directive',
                        directive: directive,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (directive.isDuplicate) {
                        page.name += '-' + directive.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInjectables = function (someInjectables) {
        var _this = this;
        logger.info('Prepare injectables');
        Configuration$1.mainData.injectables = someInjectables
            ? someInjectables
            : DependenciesEngine$1.getInjectables();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.injectables.length;
            var loop = function () {
                if (i < len) {
                    var injec = Configuration$1.mainData.injectables[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(injec.file)) {
                        logger.info(" " + injec.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(injec.file);
                        injec.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'injectables',
                        name: injec.name,
                        id: injec.id,
                        navTabs: _this.getNavTabs(injec),
                        context: 'injectable',
                        injectable: injec,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (injec.isDuplicate) {
                        page.name += '-' + injec.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterceptors = function (someInterceptors) {
        var _this = this;
        logger.info('Prepare interceptors');
        Configuration$1.mainData.interceptors = someInterceptors
            ? someInterceptors
            : DependenciesEngine$1.getInterceptors();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interceptors.length;
            var loop = function () {
                if (i < len) {
                    var interceptor = Configuration$1.mainData.interceptors[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interceptor.file)) {
                        logger.info(" " + interceptor.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interceptor.file);
                        interceptor.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interceptors',
                        name: interceptor.name,
                        id: interceptor.id,
                        navTabs: _this.getNavTabs(interceptor),
                        context: 'interceptor',
                        injectable: interceptor,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interceptor.isDuplicate) {
                        page.name += '-' + interceptor.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareGuards = function (someGuards) {
        var _this = this;
        logger.info('Prepare guards');
        Configuration$1.mainData.guards = someGuards ? someGuards : DependenciesEngine$1.getGuards();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.guards.length;
            var loop = function () {
                if (i < len) {
                    var guard = Configuration$1.mainData.guards[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(guard.file)) {
                        logger.info(" " + guard.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(guard.file);
                        guard.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'guards',
                        name: guard.name,
                        id: guard.id,
                        navTabs: _this.getNavTabs(guard),
                        context: 'guard',
                        injectable: guard,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (guard.isDuplicate) {
                        page.name += '-' + guard.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareRoutes = function () {
        logger.info('Process routes');
        Configuration$1.mainData.routes = DependenciesEngine$1.getRoutes();
        return new Promise(function (resolve, reject) {
            Configuration$1.addPage({
                name: 'routes',
                id: 'routes',
                context: 'routes',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                RouterParserUtil$1.generateRoutesIndex(Configuration$1.mainData.output, Configuration$1.mainData.routes).then(function () {
                    logger.info(' Routes index generated');
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareCoverage = function () {
        logger.info('Process documentation coverage report');
        return new Promise(function (resolve, reject) {
            /*
             * loop with components, directives, controllers, classes, injectables, interfaces, pipes, guards, misc functions variables
             */
            var files = [];
            var totalProjectStatementDocumented = 0;
            var getStatus = function (percent) {
                var status;
                if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var processComponentsAndDirectivesAndControllers = function (list) {
                _.forEach(list, function (el) {
                    var element = Object.assign({}, el);
                    if (!element.propertiesClass) {
                        element.propertiesClass = [];
                    }
                    if (!element.methodsClass) {
                        element.methodsClass = [];
                    }
                    if (!element.hostBindings) {
                        element.hostBindings = [];
                    }
                    if (!element.hostListeners) {
                        element.hostListeners = [];
                    }
                    if (!element.inputsClass) {
                        element.inputsClass = [];
                    }
                    if (!element.outputsClass) {
                        element.outputsClass = [];
                    }
                    var cl = {
                        filePath: element.file,
                        type: element.type,
                        linktype: element.type,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.propertiesClass.length +
                        element.methodsClass.length +
                        element.inputsClass.length +
                        element.hostBindings.length +
                        element.hostListeners.length +
                        element.outputsClass.length +
                        1; // +1 for element decorator comment
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.propertiesClass, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methodsClass, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostBindings, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostListeners, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.inputsClass, function (input) {
                        if (input.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (input.description &&
                            input.description !== '' &&
                            input.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.outputsClass, function (output) {
                        if (output.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (output.description &&
                            output.description !== '' &&
                            output.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cl.coveragePercent = 0;
                    }
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processCoveragePerFile = function () {
                logger.info('Process documentation coverage per file');
                logger.info('-------------------');
                var overFiles = files.filter(function (f) {
                    var overTest = f.coveragePercent >= Configuration$1.mainData.coverageMinimumPerFile;
                    if (overTest && !Configuration$1.mainData.coverageTestShowOnlyFailed) {
                        logger.info(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - over minimum per file");
                    }
                    return overTest;
                });
                var underFiles = files.filter(function (f) {
                    var underTest = f.coveragePercent < Configuration$1.mainData.coverageMinimumPerFile;
                    if (underTest) {
                        logger.error(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - under minimum per file");
                    }
                    return underTest;
                });
                logger.info('-------------------');
                return {
                    overFiles: overFiles,
                    underFiles: underFiles
                };
            };
            var processFunctionsAndVariables = function (id, type) {
                _.forEach(id, function (el) {
                    var cl = {
                        filePath: el.file,
                        type: type,
                        linktype: el.type,
                        linksubtype: el.subtype,
                        name: el.name
                    };
                    if (type === 'variable') {
                        cl.linktype = 'miscellaneous';
                    }
                    var totalStatementDocumented = 0;
                    var totalStatements = 1;
                    if (el.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                        // Doesn't handle private for coverage
                        totalStatements -= 1;
                    }
                    if (el.description &&
                        el.description !== '' &&
                        el.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                        totalStatementDocumented += 1;
                    }
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processClasses = function (list, type, linktype) {
                _.forEach(list, function (cl) {
                    var element = Object.assign({}, cl);
                    if (!element.properties) {
                        element.properties = [];
                    }
                    if (!element.methods) {
                        element.methods = [];
                    }
                    var cla = {
                        filePath: element.file,
                        type: type,
                        linktype: linktype,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.properties.length + element.methods.length + 1; // +1 for element itself
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.properties, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methods, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cla.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cla.coveragePercent = 0;
                    }
                    cla.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cla.status = getStatus(cla.coveragePercent);
                    totalProjectStatementDocumented += cla.coveragePercent;
                    files.push(cla);
                });
            };
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.components);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.directives);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.controllers);
            processClasses(Configuration$1.mainData.classes, 'class', 'classe');
            processClasses(Configuration$1.mainData.injectables, 'injectable', 'injectable');
            processClasses(Configuration$1.mainData.interfaces, 'interface', 'interface');
            processClasses(Configuration$1.mainData.guards, 'guard', 'guard');
            processClasses(Configuration$1.mainData.interceptors, 'interceptor', 'interceptor');
            _.forEach(Configuration$1.mainData.pipes, function (pipe) {
                var cl = {
                    filePath: pipe.file,
                    type: pipe.type,
                    linktype: pipe.type,
                    name: pipe.name
                };
                var totalStatementDocumented = 0;
                var totalStatements = 1;
                if (pipe.description && pipe.description !== '') {
                    totalStatementDocumented += 1;
                }
                cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                cl.status = getStatus(cl.coveragePercent);
                totalProjectStatementDocumented += cl.coveragePercent;
                files.push(cl);
            });
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.functions, 'function');
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.variables, 'variable');
            files = _.sortBy(files, ['filePath']);
            var coverageData = {
                count: files.length > 0
                    ? Math.floor(totalProjectStatementDocumented / files.length)
                    : 0,
                status: '',
                files: files
            };
            coverageData.status = getStatus(coverageData.count);
            Configuration$1.addPage({
                name: 'coverage',
                id: 'coverage',
                context: 'coverage',
                files: files,
                data: coverageData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            coverageData.files = files;
            Configuration$1.mainData.coverageData = coverageData;
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, 'documentation', coverageData);
            }
            files = _.sortBy(files, ['coveragePercent']);
            var coverageTestPerFileResults;
            if (Configuration$1.mainData.coverageTest &&
                !Configuration$1.mainData.coverageTestPerFile) {
                // Global coverage test and not per file
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
            }
            else if (!Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                coverageTestPerFileResults = processCoveragePerFile();
                // Per file coverage test and not global
                if (coverageTestPerFileResults.underFiles.length > 0) {
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else {
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
            }
            else if (Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                // Per file coverage test and global
                coverageTestPerFileResults = processCoveragePerFile();
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length === 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else if (coverageData.count < Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    var messageGlobal = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(messageGlobal);
                        logger.error(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(messageGlobal);
                        logger.warn(messagePerFile);
                        process.exit(0);
                    }
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        logger.info(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        logger.info(messagePerFile);
                        process.exit(0);
                    }
                }
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareUnitTestCoverage = function () {
        logger.info('Process unit test coverage report');
        return new Promise(function (resolve, reject) {
            var covDat, covFileNames;
            var coverageData = Configuration$1.mainData.coverageData;
            if (!coverageData.files) {
                logger.warn('Missing documentation coverage data');
            }
            else {
                covDat = {};
                covFileNames = _.map(coverageData.files, function (el) {
                    var fileName = el.filePath;
                    covDat[fileName] = {
                        type: el.type,
                        linktype: el.linktype,
                        linksubtype: el.linksubtype,
                        name: el.name
                    };
                    return fileName;
                });
            }
            // read coverage summary file and data
            var unitTestSummary = {};
            var fileDat = FileEngine$1.getSync(Configuration$1.mainData.unitTestCoverage);
            if (fileDat) {
                unitTestSummary = JSON.parse(fileDat);
            }
            else {
                return Promise.reject('Error reading unit test coverage file');
            }
            var getCovStatus = function (percent, totalLines) {
                var status;
                if (totalLines === 0) {
                    status = 'uncovered';
                }
                else if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var getCoverageData = function (data, fileName) {
                var out = {};
                if (fileName !== 'total') {
                    if (covDat === undefined) {
                        // need a name to include in output but this isn't visible
                        out = { name: fileName, filePath: fileName };
                    }
                    else {
                        var findMatch = _.filter(covFileNames, function (el) {
                            return el.includes(fileName) || fileName.includes(el);
                        });
                        if (findMatch.length > 0) {
                            out = _.clone(covDat[findMatch[0]]);
                            out['filePath'] = fileName;
                        }
                    }
                }
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (data[key]) {
                        var t = data[key];
                        out[key] = {
                            coveragePercent: Math.round(t.pct),
                            coverageCount: '' + t.covered + '/' + t.total,
                            status: getCovStatus(t.pct, t.total)
                        };
                    }
                });
                return out;
            };
            var unitTestData = {};
            var files = [];
            for (var file in unitTestSummary) {
                var dat = getCoverageData(unitTestSummary[file], file);
                if (file === 'total') {
                    unitTestData['total'] = dat;
                }
                else {
                    files.push(dat);
                }
            }
            unitTestData['files'] = files;
            unitTestData['idColumn'] = covDat !== undefined; // should we include the id column
            Configuration$1.mainData.unitTestData = unitTestData;
            Configuration$1.addPage({
                name: 'unit-test',
                id: 'unit-test',
                context: 'unit-test',
                files: files,
                data: unitTestData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (unitTestData['total'][key]) {
                        HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, key, {
                            count: unitTestData['total'][key]['coveragePercent'],
                            status: unitTestData['total'][key]['status']
                        });
                    }
                });
            }
            resolve();
        });
    };
    Application.prototype.processPage = function (page) {
        logger.info('Process page', page.name);
        var htmlData = HtmlEngine$1.render(Configuration$1.mainData, page);
        var finalPath = Configuration$1.mainData.output;
        if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
            finalPath += '/';
        }
        if (page.path) {
            finalPath += page.path + '/';
        }
        if (page.filename) {
            finalPath += page.filename + '.html';
        }
        else {
            finalPath += page.name + '.html';
        }
        if (!Configuration$1.mainData.disableSearch) {
            SearchEngine$1.indexPage({
                infos: page,
                rawData: htmlData,
                url: finalPath
            });
        }
        FileEngine$1.writeSync(finalPath, htmlData);
        return Promise.resolve();
    };
    Application.prototype.processPages = function () {
        var _this = this;
        var pages = _.sortBy(Configuration$1.pages, ['name']);
        logger.info('Process pages');
        Promise.all(pages.map(function (page) { return _this.processPage(page); }))
            .then(function () {
            var callbacksAfterGenerateSearchIndexJson = function () {
                if (Configuration$1.mainData.additionalPages.length > 0) {
                    _this.processAdditionalPages();
                }
                else {
                    if (Configuration$1.mainData.assetsFolder !== '') {
                        _this.processAssetsFolder();
                    }
                    _this.processResources();
                }
            };
            if (!Configuration$1.mainData.disableSearch) {
                SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                    callbacksAfterGenerateSearchIndexJson();
                }, function (e) {
                    logger.error(e);
                });
            }
            else {
                callbacksAfterGenerateSearchIndexJson();
            }
        })
            .then(function () {
            return _this.processMenu(Configuration$1.mainData);
        })
            .catch(function (e) {
            logger.error(e);
        });
    };
    Application.prototype.processMenu = function (mainData) {
        logger.info('Process menu...');
        return HtmlEngine$1.renderMenu(Configuration$1.mainData.templates, mainData).then(function (htmlData) {
            var finalPath = mainData.output + "/js/menu-wc.js";
            return FileEngine$1.write(finalPath, htmlData).catch(function (err) {
                logger.error('Error during ' + finalPath + ' page generation');
                return Promise.reject('');
            });
        });
    };
    Application.prototype.processAdditionalPages = function () {
        var _this = this;
        logger.info('Process additional pages');
        var pages = Configuration$1.mainData.additionalPages;
        Promise.all(pages.map(function (page) {
            if (page.children.length > 0) {
                return Promise.all([
                    _this.processPage(page)
                ].concat(page.children.map(function (childPage) { return _this.processPage(childPage); })));
            }
            else {
                return _this.processPage(page);
            }
        }))
            .then(function () {
            SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                if (Configuration$1.mainData.assetsFolder !== '') {
                    _this.processAssetsFolder();
                }
                _this.processResources();
            });
        })
            .catch(function (e) {
            logger.error(e);
            return Promise.reject(e);
        });
    };
    Application.prototype.processAssetsFolder = function () {
        logger.info('Copy assets folder');
        if (!FileEngine$1.existsSync(Configuration$1.mainData.assetsFolder)) {
            logger.error("Provided assets folder " + Configuration$1.mainData.assetsFolder + " did not exist");
        }
        else {
            var finalOutput = Configuration$1.mainData.output;
            var testOutputDir = Configuration$1.mainData.output.match(cwd);
            if (testOutputDir && testOutputDir.length > 0) {
                finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
            }
            var destination = path.join(finalOutput, path.basename(Configuration$1.mainData.assetsFolder));
            fs.copy(path.resolve(Configuration$1.mainData.assetsFolder), path.resolve(destination), function (err) {
                if (err) {
                    logger.error('Error during resources copy ', err);
                }
            });
        }
    };
    Application.prototype.processResources = function () {
        var _this = this;
        logger.info('Copy main resources');
        var onComplete = function () {
            logger.info('Documentation generated in ' +
                Configuration$1.mainData.output +
                ' in ' +
                _this.getElapsedTime() +
                ' seconds using ' +
                Configuration$1.mainData.theme +
                ' theme');
            if (Configuration$1.mainData.serve) {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + Configuration$1.mainData.port);
                _this.runWebServer(Configuration$1.mainData.output);
            }
            else {
                generationPromiseResolve();
                _this.endCallback();
            }
        };
        var finalOutput = Configuration$1.mainData.output;
        var testOutputDir = Configuration$1.mainData.output.match(cwd);
        if (testOutputDir && testOutputDir.length > 0) {
            finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
        }
        fs.copy(path.resolve(__dirname + '/../src/resources/'), path.resolve(finalOutput), function (errorCopy) {
            if (errorCopy) {
                logger.error('Error during resources copy ', errorCopy);
            }
            else {
                var extThemePromise = new Promise(function (extThemeResolve, extThemeReject) {
                    if (Configuration$1.mainData.extTheme) {
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.extTheme), path.resolve(finalOutput + '/styles/'), function (errorCopyTheme) {
                            if (errorCopyTheme) {
                                logger.error('Error during external styling theme copy ', errorCopyTheme);
                                extThemeReject();
                            }
                            else {
                                logger.info('External styling theme copy succeeded');
                                extThemeResolve();
                            }
                        });
                    }
                    else {
                        extThemeResolve();
                    }
                });
                var customFaviconPromise = new Promise(function (customFaviconResolve, customFaviconReject) {
                    if (Configuration$1.mainData.customFavicon !== '') {
                        logger.info("Custom favicon supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customFavicon), path.resolve(finalOutput + '/images/favicon.ico'), function (errorCopyFavicon) {
                            // tslint:disable-line
                            if (errorCopyFavicon) {
                                logger.error('Error during resources copy of favicon', errorCopyFavicon);
                                customFaviconReject();
                            }
                            else {
                                logger.info('External custom favicon copy succeeded');
                                customFaviconResolve();
                            }
                        });
                    }
                    else {
                        customFaviconResolve();
                    }
                });
                var customLogoPromise = new Promise(function (customLogoResolve, customLogoReject) {
                    if (Configuration$1.mainData.customLogo !== '') {
                        logger.info("Custom logo supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customLogo), path.resolve(finalOutput +
                            '/images/' +
                            Configuration$1.mainData.customLogo.split('/').pop()), function (errorCopyLogo) {
                            // tslint:disable-line
                            if (errorCopyLogo) {
                                logger.error('Error during resources copy of logo', errorCopyLogo);
                                customLogoReject();
                            }
                            else {
                                logger.info('External custom logo copy succeeded');
                                customLogoResolve();
                            }
                        });
                    }
                    else {
                        customLogoResolve();
                    }
                });
                Promise.all([extThemePromise, customFaviconPromise, customLogoPromise]).then(function () {
                    onComplete();
                });
            }
        });
    };
    /**
     * Calculates the elapsed time since the program was started.
     *
     * @returns {number}
     */
    Application.prototype.getElapsedTime = function () {
        return (new Date().valueOf() - startTime.valueOf()) / 1000;
    };
    Application.prototype.processGraphs = function () {
        var _this = this;
        if (Configuration$1.mainData.disableGraph) {
            logger.info('Graph generation disabled');
            this.processPages();
        }
        else {
            logger.info('Process main graph');
            var modules_1 = Configuration$1.mainData.modules;
            var i_1 = 0;
            var len_1 = modules_1.length;
            var loop_1 = function () {
                if (i_1 <= len_1 - 1) {
                    logger.info('Process module graph ', modules_1[i_1].name);
                    var finalPath_2 = Configuration$1.mainData.output;
                    if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
                        finalPath_2 += '/';
                    }
                    finalPath_2 += 'modules/' + modules_1[i_1].name;
                    var _rawModule = DependenciesEngine$1.getRawModule(modules_1[i_1].name);
                    if (_rawModule.declarations.length > 0 ||
                        _rawModule.bootstrap.length > 0 ||
                        _rawModule.imports.length > 0 ||
                        _rawModule.exports.length > 0 ||
                        _rawModule.providers.length > 0) {
                        NgdEngine$1.renderGraph(modules_1[i_1].file, finalPath_2, 'f', modules_1[i_1].name).then(function () {
                            NgdEngine$1.readGraph(path.resolve(finalPath_2 + path.sep + 'dependencies.svg'), modules_1[i_1].name).then(function (data) {
                                modules_1[i_1].graph = data;
                                i_1++;
                                loop_1();
                            }, function (err) {
                                logger.error('Error during graph read: ', err);
                            });
                        }, function (errorMessage) {
                            logger.error(errorMessage);
                        });
                    }
                    else {
                        i_1++;
                        loop_1();
                    }
                }
                else {
                    _this.processPages();
                }
            };
            var finalMainGraphPath_1 = Configuration$1.mainData.output;
            if (finalMainGraphPath_1.lastIndexOf('/') === -1) {
                finalMainGraphPath_1 += '/';
            }
            finalMainGraphPath_1 += 'graph';
            NgdEngine$1.init(path.resolve(finalMainGraphPath_1));
            NgdEngine$1.renderGraph(Configuration$1.mainData.tsconfig, path.resolve(finalMainGraphPath_1), 'p').then(function () {
                NgdEngine$1.readGraph(path.resolve(finalMainGraphPath_1 + path.sep + 'dependencies.svg'), 'Main graph').then(function (data) {
                    Configuration$1.mainData.mainGraph = data;
                    loop_1();
                }, function (err) {
                    logger.error('Error during main graph reading : ', err);
                    Configuration$1.mainData.disableMainGraph = true;
                    loop_1();
                });
            }, function (err) {
                logger.error('Ooops error during main graph generation, moving on next part with main graph disabled : ', err);
                Configuration$1.mainData.disableMainGraph = true;
                loop_1();
            });
        }
    };
    Application.prototype.runWebServer = function (folder) {
        if (!this.isWatching) {
            var liveServerConfiguration = {
                root: folder,
                open: Configuration$1.mainData.open,
                quiet: true,
                logLevel: 0,
                wait: 1000,
                port: Configuration$1.mainData.port
            };
            if (Configuration$1.mainData.host !== '') {
                liveServerConfiguration.host = Configuration$1.mainData.host;
            }
            LiveServer.start(liveServerConfiguration);
        }
        if (Configuration$1.mainData.watch && !this.isWatching) {
            if (typeof this.files === 'undefined') {
                logger.error('No sources files available, please use -p flag');
                generationPromiseReject();
                process.exit(1);
            }
            else {
                this.runWatch();
            }
        }
        else if (Configuration$1.mainData.watch && this.isWatching) {
            var srcFolder = findMainSourceFolder(this.files);
            logger.info("Already watching sources in " + srcFolder + " folder");
        }
    };
    Application.prototype.runWatch = function () {
        var _this = this;
        var sources = [findMainSourceFolder(this.files)];
        var watcherReady = false;
        this.isWatching = true;
        logger.info("Watching sources in " + findMainSourceFolder(this.files) + " folder");
        if (MarkdownEngine$1.hasRootMarkdowns()) {
            sources = sources.concat(MarkdownEngine$1.listRootMarkdowns());
        }
        if (Configuration$1.mainData.includes !== '') {
            sources = sources.concat(Configuration$1.mainData.includes);
        }
        // Check all elements of sources list exist
        sources = cleanSourcesForWatch(sources);
        var watcher = chokidar.watch(sources, {
            awaitWriteFinish: true,
            ignoreInitial: true,
            ignored: /(spec|\.d)\.ts/
        });
        var timerAddAndRemoveRef;
        var timerChangeRef;
        var runnerAddAndRemove = function () {
            startTime = new Date();
            _this.generate();
        };
        var waiterAddAndRemove = function () {
            clearTimeout(timerAddAndRemoveRef);
            timerAddAndRemoveRef = setTimeout(runnerAddAndRemove, 1000);
        };
        var runnerChange = function () {
            startTime = new Date();
            _this.setUpdatedFiles(_this.watchChangedFiles);
            if (_this.hasWatchedFilesTSFiles()) {
                _this.getMicroDependenciesData();
            }
            else if (_this.hasWatchedFilesRootMarkdownFiles()) {
                _this.rebuildRootMarkdowns();
            }
            else {
                _this.rebuildExternalDocumentation();
            }
        };
        var waiterChange = function () {
            clearTimeout(timerChangeRef);
            timerChangeRef = setTimeout(runnerChange, 1000);
        };
        watcher.on('ready', function () {
            if (!watcherReady) {
                watcherReady = true;
                watcher
                    .on('add', function (file) {
                    logger.debug("File " + file + " has been added");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                })
                    .on('change', function (file) {
                    logger.debug("File " + file + " has been changed");
                    // Test extension, if ts
                    // rescan only file
                    if (path.extname(file) === '.ts' ||
                        path.extname(file) === '.md' ||
                        path.extname(file) === '.json') {
                        _this.watchChangedFiles.push(path.join(cwd + path.sep + file));
                        waiterChange();
                    }
                })
                    .on('unlink', function (file) {
                    logger.debug("File " + file + " has been removed");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                });
            }
        });
    };
    Object.defineProperty(Application.prototype, "application", {
        /**
         * Return the application / root component instance.
         */
        get: function () {
            return this;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Application.prototype, "isCLI", {
        get: function () {
            return false;
        },
        enumerable: true,
        configurable: true
    });
    return Application;
}());

var glob = require('glob');
var ParserUtil = /** @class */ (function () {
    function ParserUtil() {
        this._globFiles = [];
    }
    ParserUtil.prototype.init = function (exclude, cwd) {
        this._files = exclude;
        this._cwd = cwd;
        var i = 0;
        var len = exclude.length;
        for (i; i < len; i++) {
            this._globFiles = this._globFiles.concat(glob.sync(exclude[i], { cwd: this._cwd }));
        }
    };
    ParserUtil.prototype.testFilesWithCwdDepth = function () {
        var i = 0;
        var len = this._files.length;
        var result = {
            status: true,
            level: 0
        };
        for (i; i < len; i++) {
            var elementPath = path.resolve(this._cwd + path.sep, this._files[i]);
            if (elementPath.indexOf(this._cwd) === -1) {
                result.status = false;
                var level = this._files[i].match(/\..\//g).length;
                if (level > result.level) {
                    result.level = level;
                }
            }
        }
        return result;
    };
    ParserUtil.prototype.updateCwd = function (cwd, level) {
        var _cwd = cwd, _rewind = '';
        for (var i = 0; i < level; i++) {
            _rewind += '../';
        }
        _cwd = path.resolve(_cwd, _rewind);
        return _cwd;
    };
    ParserUtil.prototype.testFile = function (file) {
        var _this = this;
        var i = 0;
        var len = this._files.length;
        var fileBasename = path.basename(file);
        var fileNameInCwd = file.replace(this._cwd + path.sep, '');
        var result = false;
        if (path.sep === '\\') {
            fileNameInCwd = fileNameInCwd.replace(new RegExp('\\' + path.sep, 'g'), '/');
        }
        for (i; i < len; i++) {
            if (glob.hasMagic(this._files[i]) && this._globFiles.length > 0) {
                var resultGlobSearch = this._globFiles.findIndex(function (element) {
                    var elementPath = path.resolve(_this._cwd + path.sep, element);
                    var elementPathInCwd = elementPath.replace(_this._cwd + path.sep, '');
                    elementPathInCwd = elementPathInCwd.replace(new RegExp('\\' + path.sep, 'g'), '/');
                    return elementPathInCwd === fileNameInCwd;
                });
                result = resultGlobSearch !== -1;
            }
            else {
                result = fileNameInCwd === this._files[i];
            }
            if (result) {
                break;
            }
        }
        return result;
    };
    return ParserUtil;
}());

var cosmiconfig = require('cosmiconfig');
var os = require('os');
var osName = require('os-name');
var pkg$1 = require('../package.json');
var program = require('commander');
var cosmiconfigModuleName = 'compodoc';
var scannedFiles = [];
var excludeFiles;
var includeFiles;
var cwd$1 = process.cwd();
process.setMaxListeners(0);
var CliApplication = /** @class */ (function (_super) {
    __extends(CliApplication, _super);
    function CliApplication() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Run compodoc from the command line.
     */
    CliApplication.prototype.generate = function () {
        var _this = this;
        function list(val) {
            return val.split(',');
        }
        program
            .version(pkg$1.version)
            .usage('<src> [options]')
            .option('-c, --config [config]', 'A configuration file : .compodocrc, .compodocrc.json, .compodocrc.yaml or compodoc property in package.json')
            .option('-p, --tsconfig [config]', 'A tsconfig.json file')
            .option('-d, --output [folder]', 'Where to store the generated documentation', COMPODOC_DEFAULTS.folder)
            .option('-y, --extTheme [file]', 'External styling theme file')
            .option('-n, --name [name]', 'Title documentation', COMPODOC_DEFAULTS.title)
            .option('-a, --assetsFolder [folder]', 'External assets folder to copy in generated documentation folder')
            .option('-o, --open [value]', 'Open the generated documentation')
            .option('-t, --silent', "In silent mode, log messages aren't logged in the console", false)
            .option('-s, --serve', 'Serve generated documentation (default http://localhost:8080/)', false)
            .option('--host [host]', 'Change default host address')
            .option('-r, --port [port]', 'Change default serving port', COMPODOC_DEFAULTS.port)
            .option('-w, --watch', 'Watch source files after serve and force documentation rebuild', false)
            .option('-e, --exportFormat [format]', 'Export in specified format (json, html)', COMPODOC_DEFAULTS.exportFormat)
            .option('--files [files]', 'Files provided by external tool, used for coverage test')
            .option('--language [language]', 'Language used for the generated documentation (en-US, es-ES, fr-FR, hu-HU, it-IT, ja-JP, nl-NL, pt-BR, zh-CN)', COMPODOC_DEFAULTS.language)
            .option('--theme [theme]', "Choose one of available themes, default is 'gitbook' (laravel, original, material, postmark, readthedocs, stripe, vagrant)")
            .option('--hideGenerator', 'Do not print the Compodoc link at the bottom of the page', false)
            .option('--toggleMenuItems <items>', "Close by default items in the menu values : ['all'] or one of these ['modules','components','directives','controllers','classes','injectables','guards','interfaces','interceptors','pipes','miscellaneous','additionalPages']", list, COMPODOC_DEFAULTS.toggleMenuItems)
            .option('--navTabConfig <tab configs>', "List navigation tab objects in the desired order with two string properties (\"id\" and \"label\"). Double-quotes must be escaped with '\\'. Available tab IDs are \"info\", \"readme\", \"source\", \"templateData\", \"styleData\", \"tree\", and \"example\". Note: Certain tabs will only be shown if applicable to a given dependency", list, JSON.stringify(COMPODOC_DEFAULTS.navTabConfig))
            .option('--templates [folder]', 'Path to directory of Handlebars templates to override built-in templates')
            .option('--includes [path]', 'Path of external markdown files to include')
            .option('--includesName [name]', 'Name of item menu of externals markdown files', COMPODOC_DEFAULTS.additionalEntryName)
            .option('--coverageTest [threshold]', 'Test command of documentation coverage with a threshold (default 70)')
            .option('--coverageMinimumPerFile [minimum]', 'Test command of documentation coverage per file with a minimum (default 0)')
            .option('--coverageTestThresholdFail [true|false]', 'Test command of documentation coverage (global or per file) will fail with error or just warn user (true: error, false: warn)', COMPODOC_DEFAULTS.coverageTestThresholdFail)
            .option('--coverageTestShowOnlyFailed', 'Display only failed files for a coverage test')
            .option('--unitTestCoverage [json-summary]', 'To include unit test coverage, specify istanbul JSON coverage summary file')
            .option('--disableSourceCode', 'Do not add source code tab and links to source code', false)
            .option('--disableDomTree', 'Do not add dom tree tab', false)
            .option('--disableTemplateTab', 'Do not add template tab', false)
            .option('--disableStyleTab', 'Do not add style tab', false)
            .option('--disableGraph', 'Do not add the dependency graph', false)
            .option('--disableCoverage', 'Do not add the documentation coverage report', false)
            .option('--disablePrivate', 'Do not show private in generated documentation', false)
            .option('--disableProtected', 'Do not show protected in generated documentation', false)
            .option('--disableInternal', 'Do not show @internal in generated documentation', false)
            .option('--disableLifeCycleHooks', 'Do not show Angular lifecycle hooks in generated documentation', false)
            .option('--disableRoutesGraph', 'Do not add the routes graph', COMPODOC_DEFAULTS.disableRoutesGraph)
            .option('--disableSearch', 'Do not add the search input', false)
            .option('--disableDependencies', 'Do not add the dependencies list', COMPODOC_DEFAULTS.disableDependencies)
            .option('--minimal', 'Minimal mode with only documentation. No search, no graph, no coverage.', false)
            .option('--customFavicon [path]', 'Use a custom favicon')
            .option('--customLogo [path]', 'Use a custom logo')
            .option('--gaID [id]', 'Google Analytics tracking ID')
            .option('--gaSite [site]', 'Google Analytics site name', COMPODOC_DEFAULTS.gaSite)
            .parse(process.argv);
        var outputHelp = function () {
            program.outputHelp();
            process.exit(1);
        };
        var configExplorer = cosmiconfig(cosmiconfigModuleName);
        var configExplorerResult;
        var configFile = {};
        if (program.config) {
            var configFilePath = program.config;
            var testConfigFilePath = configFilePath.match(process.cwd());
            if (testConfigFilePath && testConfigFilePath.length > 0) {
                configFilePath = configFilePath.replace(process.cwd() + path.sep, '');
            }
            configExplorerResult = configExplorer.loadSync(path.resolve(configFilePath));
        }
        else {
            configExplorerResult = configExplorer.searchSync();
        }
        if (configExplorerResult) {
            if (typeof configExplorerResult.config !== 'undefined') {
                configFile = configExplorerResult.config;
            }
        }
        if (configFile.output) {
            Configuration$1.mainData.output = configFile.output;
        }
        if (program.output && program.output !== COMPODOC_DEFAULTS.folder) {
            Configuration$1.mainData.output = program.output;
        }
        if (configFile.extTheme) {
            Configuration$1.mainData.extTheme = configFile.extTheme;
        }
        if (program.extTheme) {
            Configuration$1.mainData.extTheme = program.extTheme;
        }
        if (configFile.language) {
            Configuration$1.mainData.language = configFile.language;
        }
        if (program.language) {
            Configuration$1.mainData.language = program.language;
        }
        if (configFile.theme) {
            Configuration$1.mainData.theme = configFile.theme;
        }
        if (program.theme) {
            Configuration$1.mainData.theme = program.theme;
        }
        if (configFile.name) {
            Configuration$1.mainData.documentationMainName = configFile.name;
        }
        if (program.name && program.name !== COMPODOC_DEFAULTS.title) {
            Configuration$1.mainData.documentationMainName = program.name;
        }
        if (configFile.assetsFolder) {
            Configuration$1.mainData.assetsFolder = configFile.assetsFolder;
        }
        if (program.assetsFolder) {
            Configuration$1.mainData.assetsFolder = program.assetsFolder;
        }
        if (configFile.open) {
            Configuration$1.mainData.open = configFile.open;
        }
        if (program.open) {
            Configuration$1.mainData.open = program.open;
        }
        if (configFile.toggleMenuItems) {
            Configuration$1.mainData.toggleMenuItems = configFile.toggleMenuItems;
        }
        if (program.toggleMenuItems &&
            program.toggleMenuItems !== COMPODOC_DEFAULTS.toggleMenuItems) {
            Configuration$1.mainData.toggleMenuItems = program.toggleMenuItems;
        }
        if (configFile.templates) {
            Configuration$1.mainData.templates = configFile.templates;
        }
        if (program.templates) {
            Configuration$1.mainData.templates = program.templates;
        }
        if (configFile.navTabConfig) {
            Configuration$1.mainData.navTabConfig = configFile.navTabConfig;
        }
        if (program.navTabConfig &&
            JSON.parse(program.navTabConfig).length !== COMPODOC_DEFAULTS.navTabConfig.length) {
            Configuration$1.mainData.navTabConfig = JSON.parse(program.navTabConfig);
        }
        if (configFile.includes) {
            Configuration$1.mainData.includes = configFile.includes;
        }
        if (program.includes) {
            Configuration$1.mainData.includes = program.includes;
        }
        if (configFile.includesName) {
            Configuration$1.mainData.includesName = configFile.includesName;
        }
        if (program.includesName &&
            program.includesName !== COMPODOC_DEFAULTS.additionalEntryName) {
            Configuration$1.mainData.includesName = program.includesName;
        }
        if (configFile.silent) {
            logger.silent = false;
        }
        if (program.silent) {
            logger.silent = false;
        }
        if (configFile.serve) {
            Configuration$1.mainData.serve = configFile.serve;
        }
        if (program.serve) {
            Configuration$1.mainData.serve = program.serve;
        }
        if (configFile.host) {
            Configuration$1.mainData.host = configFile.host;
            Configuration$1.mainData.hostname = configFile.host;
        }
        if (program.host) {
            Configuration$1.mainData.host = program.host;
            Configuration$1.mainData.hostname = program.host;
        }
        if (configFile.port) {
            Configuration$1.mainData.port = configFile.port;
        }
        if (program.port && program.port !== COMPODOC_DEFAULTS.port) {
            Configuration$1.mainData.port = program.port;
        }
        if (configFile.watch) {
            Configuration$1.mainData.watch = configFile.watch;
        }
        if (program.watch) {
            Configuration$1.mainData.watch = program.watch;
        }
        if (configFile.exportFormat) {
            Configuration$1.mainData.exportFormat = configFile.exportFormat;
        }
        if (program.exportFormat && program.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
            Configuration$1.mainData.exportFormat = program.exportFormat;
        }
        if (configFile.hideGenerator) {
            Configuration$1.mainData.hideGenerator = configFile.hideGenerator;
        }
        if (program.hideGenerator) {
            Configuration$1.mainData.hideGenerator = program.hideGenerator;
        }
        if (configFile.coverageTest) {
            Configuration$1.mainData.coverageTest = true;
            Configuration$1.mainData.coverageTestThreshold =
                typeof configFile.coverageTest === 'string'
                    ? parseInt(configFile.coverageTest, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageThreshold;
        }
        if (program.coverageTest) {
            Configuration$1.mainData.coverageTest = true;
            Configuration$1.mainData.coverageTestThreshold =
                typeof program.coverageTest === 'string'
                    ? parseInt(program.coverageTest, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageThreshold;
        }
        if (configFile.coverageMinimumPerFile) {
            Configuration$1.mainData.coverageTestPerFile = true;
            Configuration$1.mainData.coverageMinimumPerFile =
                typeof configFile.coverageMinimumPerFile === 'string'
                    ? parseInt(configFile.coverageMinimumPerFile, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile;
        }
        if (program.coverageMinimumPerFile) {
            Configuration$1.mainData.coverageTestPerFile = true;
            Configuration$1.mainData.coverageMinimumPerFile =
                typeof program.coverageMinimumPerFile === 'string'
                    ? parseInt(program.coverageMinimumPerFile, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile;
        }
        if (configFile.coverageTestThresholdFail) {
            Configuration$1.mainData.coverageTestThresholdFail =
                configFile.coverageTestThresholdFail === 'false' ? false : true;
        }
        if (program.coverageTestThresholdFail) {
            Configuration$1.mainData.coverageTestThresholdFail =
                program.coverageTestThresholdFail === 'false' ? false : true;
        }
        if (configFile.coverageTestShowOnlyFailed) {
            Configuration$1.mainData.coverageTestShowOnlyFailed =
                configFile.coverageTestShowOnlyFailed;
        }
        if (program.coverageTestShowOnlyFailed) {
            Configuration$1.mainData.coverageTestShowOnlyFailed = program.coverageTestShowOnlyFailed;
        }
        if (configFile.unitTestCoverage) {
            Configuration$1.mainData.unitTestCoverage = configFile.unitTestCoverage;
        }
        if (program.unitTestCoverage) {
            Configuration$1.mainData.unitTestCoverage = program.unitTestCoverage;
        }
        if (configFile.disableSourceCode) {
            Configuration$1.mainData.disableSourceCode = configFile.disableSourceCode;
        }
        if (program.disableSourceCode) {
            Configuration$1.mainData.disableSourceCode = program.disableSourceCode;
        }
        if (configFile.disableDomTree) {
            Configuration$1.mainData.disableDomTree = configFile.disableDomTree;
        }
        if (program.disableDomTree) {
            Configuration$1.mainData.disableDomTree = program.disableDomTree;
        }
        if (configFile.disableTemplateTab) {
            Configuration$1.mainData.disableTemplateTab = configFile.disableTemplateTab;
        }
        if (program.disableTemplateTab) {
            Configuration$1.mainData.disableTemplateTab = program.disableTemplateTab;
        }
        if (configFile.disableStyleTab) {
            Configuration$1.mainData.disableStyleTab = configFile.disableStyleTab;
        }
        if (program.disableStyleTab) {
            Configuration$1.mainData.disableStyleTab = program.disableStyleTab;
        }
        if (configFile.disableGraph) {
            Configuration$1.mainData.disableGraph = configFile.disableGraph;
        }
        if (program.disableGraph) {
            Configuration$1.mainData.disableGraph = program.disableGraph;
        }
        if (configFile.disableCoverage) {
            Configuration$1.mainData.disableCoverage = configFile.disableCoverage;
        }
        if (program.disableCoverage) {
            Configuration$1.mainData.disableCoverage = program.disableCoverage;
        }
        if (configFile.disablePrivate) {
            Configuration$1.mainData.disablePrivate = configFile.disablePrivate;
        }
        if (program.disablePrivate) {
            Configuration$1.mainData.disablePrivate = program.disablePrivate;
        }
        if (configFile.disableProtected) {
            Configuration$1.mainData.disableProtected = configFile.disableProtected;
        }
        if (program.disableProtected) {
            Configuration$1.mainData.disableProtected = program.disableProtected;
        }
        if (configFile.disableInternal) {
            Configuration$1.mainData.disableInternal = configFile.disableInternal;
        }
        if (program.disableInternal) {
            Configuration$1.mainData.disableInternal = program.disableInternal;
        }
        if (configFile.disableLifeCycleHooks) {
            Configuration$1.mainData.disableLifeCycleHooks = configFile.disableLifeCycleHooks;
        }
        if (program.disableLifeCycleHooks) {
            Configuration$1.mainData.disableLifeCycleHooks = program.disableLifeCycleHooks;
        }
        if (configFile.disableRoutesGraph) {
            Configuration$1.mainData.disableRoutesGraph = configFile.disableRoutesGraph;
        }
        if (program.disableRoutesGraph) {
            Configuration$1.mainData.disableRoutesGraph = program.disableRoutesGraph;
        }
        if (configFile.disableSearch) {
            Configuration$1.mainData.disableSearch = configFile.disableSearch;
        }
        if (program.disableSearch) {
            Configuration$1.mainData.disableSearch = program.disableSearch;
        }
        if (configFile.disableDependencies) {
            Configuration$1.mainData.disableDependencies = configFile.disableDependencies;
        }
        if (program.disableDependencies) {
            Configuration$1.mainData.disableDependencies = program.disableDependencies;
        }
        if (configFile.minimal) {
            Configuration$1.mainData.disableSearch = true;
            Configuration$1.mainData.disableRoutesGraph = true;
            Configuration$1.mainData.disableGraph = true;
            Configuration$1.mainData.disableCoverage = true;
        }
        if (program.minimal) {
            Configuration$1.mainData.disableSearch = true;
            Configuration$1.mainData.disableRoutesGraph = true;
            Configuration$1.mainData.disableGraph = true;
            Configuration$1.mainData.disableCoverage = true;
        }
        if (configFile.customFavicon) {
            Configuration$1.mainData.customFavicon = configFile.customFavicon;
        }
        if (program.customFavicon) {
            Configuration$1.mainData.customFavicon = program.customFavicon;
        }
        if (configFile.customLogo) {
            Configuration$1.mainData.customLogo = configFile.customLogo;
        }
        if (program.customLogo) {
            Configuration$1.mainData.customLogo = program.customLogo;
        }
        if (configFile.gaID) {
            Configuration$1.mainData.gaID = configFile.gaID;
        }
        if (program.gaID) {
            Configuration$1.mainData.gaID = program.gaID;
        }
        if (configFile.gaSite) {
            Configuration$1.mainData.gaSite = configFile.gaSite;
        }
        if (program.gaSite && program.gaSite !== COMPODOC_DEFAULTS.gaSite) {
            Configuration$1.mainData.gaSite = program.gaSite;
        }
        if (!this.isWatching) {
            if (!logger.silent) {
                console.log("Compodoc v" + pkg$1.version);
            }
            else {
                console.log(fs.readFileSync(path.join(__dirname, '../src/banner')).toString());
                console.log(pkg$1.version);
                console.log('');
                console.log("TypeScript version used by Compodoc : " + Ast.ts.version);
                console.log('');
                if (FileEngine$1.existsSync(cwd$1 + path.sep + 'package.json')) {
                    var packageData = FileEngine$1.getSync(cwd$1 + path.sep + 'package.json');
                    if (packageData) {
                        var parsedData = JSON.parse(packageData);
                        var projectDevDependencies = parsedData.devDependencies;
                        if (projectDevDependencies && projectDevDependencies.typescript) {
                            var tsProjectVersion = AngularVersionUtil$1.cleanVersion(projectDevDependencies.typescript);
                            console.log("TypeScript version of current project : " + tsProjectVersion);
                            console.log('');
                        }
                    }
                }
                console.log("Node.js version : " + process.version);
                console.log('');
                console.log("Operating system : " + osName(os.platform(), os.release()));
                console.log('');
            }
        }
        if (configExplorerResult) {
            if (typeof configExplorerResult.config !== 'undefined') {
                logger.info("Using configuration file : " + configExplorerResult.filepath);
            }
        }
        if (!configExplorerResult) {
            logger.warn("No configuration file found, switching to CLI flags.");
        }
        if (program.language && !I18nEngine$1.supportLanguage(program.language)) {
            logger.warn("The language " + program.language + " is not available, falling back to " + I18nEngine$1.fallbackLanguage);
        }
        if (program.tsconfig && typeof program.tsconfig === 'boolean') {
            logger.error("Please provide a tsconfig file.");
            process.exit(1);
        }
        if (configFile.tsconfig) {
            Configuration$1.mainData.tsconfig = configFile.tsconfig;
        }
        if (program.tsconfig) {
            Configuration$1.mainData.tsconfig = program.tsconfig;
        }
        if (configFile.files) {
            scannedFiles = configFile.files;
        }
        if (configFile.exclude) {
            excludeFiles = configFile.exclude;
        }
        if (configFile.include) {
            includeFiles = configFile.include;
        }
        /**
         * Check --files argument call
         */
        var argv = require('minimist')(process.argv.slice(2));
        if (argv && argv.files) {
            Configuration$1.mainData.hasFilesToCoverage = true;
            if (typeof argv.files === 'string') {
                _super.prototype.setFiles.call(this, [argv.files]);
            }
            else {
                _super.prototype.setFiles.call(this, argv.files);
            }
        }
        if (program.serve && !Configuration$1.mainData.tsconfig && program.output) {
            // if -s & -d, serve it
            if (!FileEngine$1.existsSync(Configuration$1.mainData.output)) {
                logger.error(Configuration$1.mainData.output + " folder doesn't exist");
                process.exit(1);
            }
            else {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + program.port);
                _super.prototype.runWebServer.call(this, Configuration$1.mainData.output);
            }
        }
        else if (program.serve && !Configuration$1.mainData.tsconfig && !program.output) {
            // if only -s find ./documentation, if ok serve, else error provide -d
            if (!FileEngine$1.existsSync(Configuration$1.mainData.output)) {
                logger.error('Provide output generated folder with -d flag');
                process.exit(1);
            }
            else {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + program.port);
                _super.prototype.runWebServer.call(this, Configuration$1.mainData.output);
            }
        }
        else if (Configuration$1.mainData.hasFilesToCoverage) {
            if (program.coverageMinimumPerFile) {
                logger.info('Run documentation coverage test for files');
                _super.prototype.testCoverage.call(this);
            }
            else {
                logger.error('Missing coverage configuration');
            }
        }
        else {
            if (program.hideGenerator) {
                Configuration$1.mainData.hideGenerator = true;
            }
            if (Configuration$1.mainData.tsconfig && program.args.length === 0) {
                /**
                 * tsconfig file provided only
                 */
                var testTsConfigPath = Configuration$1.mainData.tsconfig.indexOf(process.cwd());
                if (testTsConfigPath !== -1) {
                    Configuration$1.mainData.tsconfig = Configuration$1.mainData.tsconfig.replace(process.cwd() + path.sep, '');
                }
                if (!FileEngine$1.existsSync(Configuration$1.mainData.tsconfig)) {
                    logger.error("\"" + Configuration$1.mainData.tsconfig + "\" file was not found in the current directory");
                    process.exit(1);
                }
                else {
                    var _file = path.join(path.join(process.cwd(), path.dirname(Configuration$1.mainData.tsconfig)), path.basename(Configuration$1.mainData.tsconfig));
                    // use the current directory of tsconfig.json as a working directory
                    cwd$1 = _file
                        .split(path.sep)
                        .slice(0, -1)
                        .join(path.sep);
                    logger.info('Using tsconfig file ', _file);
                    var tsConfigFile = readConfig(_file);
                    scannedFiles = tsConfigFile.files;
                    if (scannedFiles) {
                        scannedFiles = handlePath(scannedFiles, cwd$1);
                    }
                    if (typeof scannedFiles === 'undefined') {
                        excludeFiles = tsConfigFile.exclude || [];
                        includeFiles = tsConfigFile.include || [];
                        scannedFiles = [];
                        var excludeParser_1 = new ParserUtil(), includeParser_1 = new ParserUtil();
                        excludeParser_1.init(excludeFiles, cwd$1);
                        includeParser_1.init(includeFiles, cwd$1);
                        var startCwd = cwd$1;
                        var excludeParserTestFilesWithCwdDepth = excludeParser_1.testFilesWithCwdDepth();
                        if (!excludeParserTestFilesWithCwdDepth.status) {
                            startCwd = excludeParser_1.updateCwd(cwd$1, excludeParserTestFilesWithCwdDepth.level);
                        }
                        var includeParserTestFilesWithCwdDepth = includeParser_1.testFilesWithCwdDepth();
                        if (!includeParser_1.testFilesWithCwdDepth().status) {
                            startCwd = includeParser_1.updateCwd(cwd$1, includeParserTestFilesWithCwdDepth.level);
                        }
                        var finder = require('findit2')(startCwd || '.');
                        finder.on('directory', function (dir, stat, stop) {
                            var base = path.basename(dir);
                            if (base === '.git' || base === 'node_modules') {
                                stop();
                            }
                        });
                        finder.on('file', function (file, stat) {
                            if (/(spec|\.d)\.ts/.test(file)) {
                                logger.warn('Ignoring', file);
                            }
                            else if (excludeParser_1.testFile(file) &&
                                path.extname(file) === '.ts') {
                                logger.warn('Excluding', file);
                            }
                            else if (includeFiles.length > 0) {
                                /**
                                 * If include provided in tsconfig, use only this source,
                                 * and not files found with global findit scan in working directory
                                 */
                                if (path.extname(file) === '.ts' && includeParser_1.testFile(file)) {
                                    logger.debug('Including', file);
                                    scannedFiles.push(file);
                                }
                                else {
                                    if (path.extname(file) === '.ts') {
                                        logger.warn('Excluding', file);
                                    }
                                }
                            }
                            else {
                                logger.debug('Including', file);
                                scannedFiles.push(file);
                            }
                        });
                        finder.on('end', function () {
                            _super.prototype.setFiles.call(_this, scannedFiles);
                            if (program.coverageTest || program.coverageTestPerFile) {
                                logger.info('Run documentation coverage test');
                                _super.prototype.testCoverage.call(_this);
                            }
                            else {
                                _super.prototype.generate.call(_this);
                            }
                        });
                    }
                    else {
                        _super.prototype.setFiles.call(this, scannedFiles);
                        if (program.coverageTest || program.coverageTestPerFile) {
                            logger.info('Run documentation coverage test');
                            _super.prototype.testCoverage.call(this);
                        }
                        else {
                            _super.prototype.generate.call(this);
                        }
                    }
                }
            }
            else if (Configuration$1.mainData.tsconfig && program.args.length > 0) {
                /**
                 * tsconfig file provided with source folder in arg
                 */
                var testTsConfigPath = Configuration$1.mainData.tsconfig.indexOf(process.cwd());
                if (testTsConfigPath !== -1) {
                    Configuration$1.mainData.tsconfig = Configuration$1.mainData.tsconfig.replace(process.cwd() + path.sep, '');
                }
                var sourceFolder = program.args[0];
                if (!FileEngine$1.existsSync(sourceFolder)) {
                    logger.error("Provided source folder " + sourceFolder + " was not found in the current directory");
                    process.exit(1);
                }
                else {
                    logger.info('Using provided source folder');
                    if (!FileEngine$1.existsSync(Configuration$1.mainData.tsconfig)) {
                        logger.error("\"" + Configuration$1.mainData.tsconfig + "\" file was not found in the current directory");
                        process.exit(1);
                    }
                    else {
                        var _file = path.join(path.join(process.cwd(), path.dirname(Configuration$1.mainData.tsconfig)), path.basename(Configuration$1.mainData.tsconfig));
                        // use the current directory of tsconfig.json as a working directory
                        cwd$1 = _file
                            .split(path.sep)
                            .slice(0, -1)
                            .join(path.sep);
                        logger.info('Using tsconfig file ', _file);
                        var tsConfigFile = readConfig(_file);
                        scannedFiles = tsConfigFile.files;
                        if (scannedFiles) {
                            scannedFiles = handlePath(scannedFiles, cwd$1);
                        }
                        if (typeof scannedFiles === 'undefined') {
                            excludeFiles = tsConfigFile.exclude || [];
                            includeFiles = tsConfigFile.include || [];
                            scannedFiles = [];
                            var excludeParser_2 = new ParserUtil(), includeParser_2 = new ParserUtil();
                            excludeParser_2.init(excludeFiles, cwd$1);
                            includeParser_2.init(includeFiles, cwd$1);
                            var startCwd = sourceFolder;
                            var excludeParserTestFilesWithCwdDepth = excludeParser_2.testFilesWithCwdDepth();
                            if (!excludeParserTestFilesWithCwdDepth.status) {
                                startCwd = excludeParser_2.updateCwd(cwd$1, excludeParserTestFilesWithCwdDepth.level);
                            }
                            var includeParserTestFilesWithCwdDepth = includeParser_2.testFilesWithCwdDepth();
                            if (!includeParser_2.testFilesWithCwdDepth().status) {
                                startCwd = includeParser_2.updateCwd(cwd$1, includeParserTestFilesWithCwdDepth.level);
                            }
                            var finder = require('findit2')(path.resolve(startCwd));
                            finder.on('directory', function (dir, stat, stop) {
                                var base = path.basename(dir);
                                if (base === '.git' || base === 'node_modules') {
                                    stop();
                                }
                            });
                            finder.on('file', function (file, stat) {
                                if (/(spec|\.d)\.ts/.test(file)) {
                                    logger.warn('Ignoring', file);
                                }
                                else if (excludeParser_2.testFile(file)) {
                                    logger.warn('Excluding', file);
                                }
                                else if (includeFiles.length > 0) {
                                    /**
                                     * If include provided in tsconfig, use only this source,
                                     * and not files found with global findit scan in working directory
                                     */
                                    if (path.extname(file) === '.ts' &&
                                        includeParser_2.testFile(file)) {
                                        logger.debug('Including', file);
                                        scannedFiles.push(file);
                                    }
                                    else {
                                        if (path.extname(file) === '.ts') {
                                            logger.warn('Excluding', file);
                                        }
                                    }
                                }
                                else {
                                    logger.debug('Including', file);
                                    scannedFiles.push(file);
                                }
                            });
                            finder.on('end', function () {
                                _super.prototype.setFiles.call(_this, scannedFiles);
                                if (program.coverageTest || program.coverageTestPerFile) {
                                    logger.info('Run documentation coverage test');
                                    _super.prototype.testCoverage.call(_this);
                                }
                                else {
                                    _super.prototype.generate.call(_this);
                                }
                            });
                        }
                        else {
                            _super.prototype.setFiles.call(this, scannedFiles);
                            if (program.coverageTest || program.coverageTestPerFile) {
                                logger.info('Run documentation coverage test');
                                _super.prototype.testCoverage.call(this);
                            }
                            else {
                                _super.prototype.generate.call(this);
                            }
                        }
                    }
                }
            }
            else {
                logger.error('tsconfig.json file was not found, please use -p flag');
                outputHelp();
            }
        }
    };
    return CliApplication;
}(Application));

exports.Application = Application;
exports.CliApplication = CliApplication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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