'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var fs = require('fs-extra');
var path = require('path');
var Ast = require('ts-simple-ast');
var Ast__default = _interopDefault(Ast);
var LiveServer = require('live-server');
var _ = require('lodash');
var Handlebars = require('handlebars');
var semver = require('semver');
var i18next = _interopDefault(require('i18next'));
var JSON5 = require('json5');

/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */
/* global Reflect, Promise */

var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return extendStatics(d, b);
};

function __extends(d, b) {
    extendStatics(d, b);
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
}

var __assign = function() {
    __assign = Object.assign || function __assign(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};

var log = require('fancy-log');
var c = require('chalk');
var pkg = require('../package.json');
var LEVEL;
(function (LEVEL) {
    LEVEL[LEVEL["INFO"] = 0] = "INFO";
    LEVEL[LEVEL["DEBUG"] = 1] = "DEBUG";
    LEVEL[LEVEL["ERROR"] = 2] = "ERROR";
    LEVEL[LEVEL["WARN"] = 3] = "WARN";
})(LEVEL || (LEVEL = {}));
var Logger = /** @class */ (function () {
    function Logger() {
        this.name = pkg.name;
        this.version = pkg.version;
        this.logger = log;
        this.silent = true;
    }
    Logger.prototype.info = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.INFO].concat(args)));
    };
    Logger.prototype.error = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.ERROR].concat(args)));
    };
    Logger.prototype.warn = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.WARN].concat(args)));
    };
    Logger.prototype.debug = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.DEBUG].concat(args)));
    };
    Logger.prototype.format = function (level) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var pad = function (s, l, z) {
            if (z === void 0) { z = ''; }
            return s + Array(Math.max(0, l - s.length + 1)).join(z);
        };
        var msg = args.join(' ');
        if (args.length > 1) {
            msg = pad(args.shift(), 15, ' ') + ": " + args.join(' ');
        }
        switch (level) {
            case LEVEL.INFO:
                msg = c.green(msg);
                break;
            case LEVEL.DEBUG:
                msg = c.cyan(msg);
                break;
            case LEVEL.WARN:
                msg = c.yellow(msg);
                break;
            case LEVEL.ERROR:
                msg = c.red(msg);
                break;
        }
        return [msg].join('');
    };
    return Logger;
}());
var logger = new Logger();

var COMPODOC_DEFAULTS = {
    title: 'Application documentation',
    additionalEntryName: 'Additional documentation',
    additionalEntryPath: 'additional-documentation',
    folder: './documentation/',
    hostname: '127.0.0.1',
    port: 8080,
    theme: 'gitbook',
    exportFormat: 'html',
    exportFormatsSupported: ['html', 'json', 'pdf'],
    base: '/',
    defaultCoverageThreshold: 70,
    defaultCoverageMinimumPerFile: 0,
    coverageTestThresholdFail: true,
    toggleMenuItems: ['all'],
    navTabConfig: [],
    disableSourceCode: false,
    disableDomTree: false,
    disableTemplateTab: false,
    disableStyleTab: false,
    disableGraph: false,
    disableMainGraph: false,
    disableCoverage: false,
    disablePrivate: false,
    disableProtected: false,
    disableInternal: false,
    disableLifeCycleHooks: false,
    disableRoutesGraph: false,
    disableDependencies: false,
    PAGE_TYPES: {
        ROOT: 'root',
        INTERNAL: 'internal'
    },
    gaSite: 'auto',
    coverageTestShowOnlyFailed: false,
    language: 'en-US'
};

var Configuration = /** @class */ (function () {
    function Configuration() {
        this._pages = [];
        this._mainData = {
            output: COMPODOC_DEFAULTS.folder,
            theme: COMPODOC_DEFAULTS.theme,
            extTheme: '',
            serve: false,
            hostname: COMPODOC_DEFAULTS.hostname,
            host: '',
            port: COMPODOC_DEFAULTS.port,
            open: false,
            assetsFolder: '',
            documentationMainName: COMPODOC_DEFAULTS.title,
            documentationMainDescription: '',
            base: COMPODOC_DEFAULTS.base,
            hideGenerator: false,
            hasFilesToCoverage: false,
            modules: [],
            readme: false,
            changelog: '',
            contributing: '',
            license: '',
            todo: '',
            markdowns: [],
            additionalPages: [],
            pipes: [],
            classes: [],
            interfaces: [],
            components: [],
            controllers: [],
            directives: [],
            injectables: [],
            interceptors: [],
            guards: [],
            miscellaneous: [],
            routes: [],
            tsconfig: '',
            toggleMenuItems: COMPODOC_DEFAULTS.toggleMenuItems,
            navTabConfig: [],
            templates: '',
            includes: '',
            includesName: COMPODOC_DEFAULTS.additionalEntryName,
            includesFolder: COMPODOC_DEFAULTS.additionalEntryPath,
            disableSourceCode: COMPODOC_DEFAULTS.disableSourceCode,
            disableDomTree: COMPODOC_DEFAULTS.disableDomTree,
            disableTemplateTab: COMPODOC_DEFAULTS.disableTemplateTab,
            disableStyleTab: COMPODOC_DEFAULTS.disableStyleTab,
            disableGraph: COMPODOC_DEFAULTS.disableGraph,
            disableMainGraph: COMPODOC_DEFAULTS.disableMainGraph,
            disableCoverage: COMPODOC_DEFAULTS.disableCoverage,
            disablePrivate: COMPODOC_DEFAULTS.disablePrivate,
            disableInternal: COMPODOC_DEFAULTS.disableInternal,
            disableProtected: COMPODOC_DEFAULTS.disableProtected,
            disableLifeCycleHooks: COMPODOC_DEFAULTS.disableLifeCycleHooks,
            disableRoutesGraph: COMPODOC_DEFAULTS.disableRoutesGraph,
            disableSearch: false,
            disableDependencies: COMPODOC_DEFAULTS.disableDependencies,
            watch: false,
            mainGraph: '',
            coverageTest: false,
            coverageTestThreshold: COMPODOC_DEFAULTS.defaultCoverageThreshold,
            coverageTestThresholdFail: COMPODOC_DEFAULTS.coverageTestThresholdFail,
            coverageTestPerFile: false,
            coverageMinimumPerFile: COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile,
            unitTestCoverage: '',
            unitTestData: undefined,
            coverageTestShowOnlyFailed: COMPODOC_DEFAULTS.coverageTestShowOnlyFailed,
            routesLength: 0,
            angularVersion: '',
            exportFormat: COMPODOC_DEFAULTS.exportFormat,
            coverageData: {},
            customFavicon: '',
            customLogo: '',
            packageDependencies: [],
            packagePeerDependencies: [],
            gaID: '',
            gaSite: '',
            angularProject: false,
            angularJSProject: false,
            language: COMPODOC_DEFAULTS.language
        };
    }
    Configuration.getInstance = function () {
        if (!Configuration.instance) {
            Configuration.instance = new Configuration();
        }
        return Configuration.instance;
    };
    Configuration.prototype.addPage = function (page) {
        var indexPage = _.findIndex(this._pages, { name: page.name });
        if (indexPage === -1) {
            this._pages.push(page);
        }
    };
    Configuration.prototype.hasPage = function (name) {
        var indexPage = _.findIndex(this._pages, { name: name });
        return indexPage !== -1;
    };
    Configuration.prototype.addAdditionalPage = function (page) {
        this._mainData.additionalPages.push(page);
    };
    Configuration.prototype.getAdditionalPageById = function (id) {
        return this._mainData.additionalPages.find(function (page) { return page.id === id; });
    };
    Configuration.prototype.resetPages = function () {
        this._pages = [];
    };
    Configuration.prototype.resetAdditionalPages = function () {
        this._mainData.additionalPages = [];
    };
    Configuration.prototype.resetRootMarkdownPages = function () {
        var indexPage = _.findIndex(this._pages, { name: 'index' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'changelog' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'contributing' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'license' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'todo' });
        this._pages.splice(indexPage, 1);
        this._mainData.markdowns = [];
    };
    Object.defineProperty(Configuration.prototype, "pages", {
        get: function () {
            return this._pages;
        },
        set: function (pages) {
            this._pages = [];
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "markDownPages", {
        get: function () {
            return this._pages.filter(function (page) { return page.markdown; });
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "mainData", {
        get: function () {
            return this._mainData;
        },
        set: function (data) {
            Object.assign(this._mainData, data);
        },
        enumerable: true,
        configurable: true
    });
    return Configuration;
}());
var Configuration$1 = Configuration.getInstance();

var AngularAPIs = require('../src/data/api-list.json');
var AngularApiUtil = /** @class */ (function () {
    function AngularApiUtil() {
    }
    AngularApiUtil.getInstance = function () {
        if (!AngularApiUtil.instance) {
            AngularApiUtil.instance = new AngularApiUtil();
        }
        return AngularApiUtil.instance;
    };
    AngularApiUtil.prototype.findApi = function (type) {
        var foundedApi;
        _.forEach(AngularAPIs, function (mainApi) {
            _.forEach(mainApi.items, function (api) {
                if (api.title === type) {
                    foundedApi = api;
                }
            });
        });
        return {
            source: 'external',
            data: foundedApi
        };
    };
    return AngularApiUtil;
}());
var AngularApiUtil$1 = AngularApiUtil.getInstance();

function extractLeadingText(string, completeTag) {
    var tagIndex = string.indexOf(completeTag);
    var leadingText = undefined;
    var leadingTextRegExp = /\[(.+?)\]/g;
    var leadingTextInfo = leadingTextRegExp.exec(string);
    // did we find leading text, and if so, does it immediately precede the tag?
    while (leadingTextInfo && leadingTextInfo.length) {
        if (leadingTextInfo.index + leadingTextInfo[0].length === tagIndex) {
            string = string.replace(leadingTextInfo[0], '');
            leadingText = leadingTextInfo[1];
            break;
        }
        leadingTextInfo = leadingTextRegExp.exec(string);
    }
    return {
        leadingText: leadingText,
        string: string
    };
}
function splitLinkText(text) {
    var linkText;
    var target;
    var splitIndex;
    // if a pipe is not present, we split on the first space
    splitIndex = text.indexOf('|');
    if (splitIndex === -1) {
        splitIndex = text.search(/\s/);
    }
    if (splitIndex !== -1) {
        linkText = text.substr(splitIndex + 1);
        // Normalize subsequent newlines to a single space.
        linkText = linkText.replace(/\n+/, ' ');
        target = text.substr(0, splitIndex);
    }
    return {
        linkText: linkText,
        target: target || text
    };
}
var LinkParser = (function () {
    var processTheLink = function (string, tagInfo, leadingText) {
        var leading = extractLeadingText(string, tagInfo.completeTag), linkText, split, target, stringtoReplace;
        linkText = leadingText ? leadingText : leading.leadingText || '';
        split = splitLinkText(tagInfo.text);
        target = split.target;
        if (leading.leadingText !== undefined) {
            stringtoReplace = '[' + leading.leadingText + ']' + tagInfo.completeTag;
        }
        else if (typeof split.linkText !== 'undefined') {
            stringtoReplace = tagInfo.completeTag;
            linkText = split.linkText;
        }
        return string.replace(stringtoReplace, '[' + linkText + '](' + target + ')');
    };
    /**
     * Convert
     * {@link http://www.google.com|Google} or {@link https://github.com GitHub} or [Github]{@link https://github.com} to [Github](https://github.com)
     */
    var replaceLinkTag = function (str) {
        if (typeof str === 'undefined') {
            return {
                newString: ''
            };
        }
        // new RegExp('\\[((?:.|\n)+?)]\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TO DO]{@link Todo} fo') -> "[TO DO]{@link Todo}", "TO DO", "Todo"
        // new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TODO]{@link Todo} fo') -> "{@link Todo}", "Todo"
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExp, matches, previousString;
        tagRegExp = str.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(str, matchedTag, linkText);
            }
            else {
                return replacer(str, matchedTag);
            }
        }
        do {
            matches = tagRegExp.exec(str);
            if (matches) {
                previousString = str;
                if (matches.length === 2) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== str);
        return {
            newString: str
        };
    };
    var _resolveLinks = function (str) {
        return replaceLinkTag(str).newString;
    };
    return {
        resolveLinks: _resolveLinks
    };
})();

var AngularLifecycleHooks;
(function (AngularLifecycleHooks) {
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnChanges"] = 0] = "ngOnChanges";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnInit"] = 1] = "ngOnInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngDoCheck"] = 2] = "ngDoCheck";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentInit"] = 3] = "ngAfterContentInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentChecked"] = 4] = "ngAfterContentChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewInit"] = 5] = "ngAfterViewInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewChecked"] = 6] = "ngAfterViewChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnDestroy"] = 7] = "ngOnDestroy";
})(AngularLifecycleHooks || (AngularLifecycleHooks = {}));

function kindToType(kind) {
    var _type = '';
    switch (kind) {
        case Ast.SyntaxKind.StringKeyword:
        case Ast.SyntaxKind.StringLiteral:
            _type = 'string';
            break;
        case Ast.SyntaxKind.NumberKeyword:
        case Ast.SyntaxKind.NumericLiteral:
            _type = 'number';
            break;
        case Ast.SyntaxKind.ArrayType:
        case Ast.SyntaxKind.ArrayLiteralExpression:
            _type = '[]';
            break;
        case Ast.SyntaxKind.VoidKeyword:
            _type = 'void';
            break;
        case Ast.SyntaxKind.FunctionType:
            _type = 'function';
            break;
        case Ast.SyntaxKind.TypeLiteral:
            _type = 'literal type';
            break;
        case Ast.SyntaxKind.BooleanKeyword:
            _type = 'boolean';
            break;
        case Ast.SyntaxKind.AnyKeyword:
            _type = 'any';
            break;
        case Ast.SyntaxKind.NullKeyword:
            _type = 'null';
            break;
        case Ast.SyntaxKind.SymbolKeyword:
            _type = 'symbol';
            break;
        case Ast.SyntaxKind.NeverKeyword:
            _type = 'never';
            break;
        case Ast.SyntaxKind.UndefinedKeyword:
            _type = 'undefined';
            break;
        case Ast.SyntaxKind.ObjectKeyword:
        case Ast.SyntaxKind.ObjectLiteralExpression:
            _type = 'object';
            break;
    }
    return _type;
}

var getCurrentDirectory = Ast.ts.sys.getCurrentDirectory;
var useCaseSensitiveFileNames = Ast.ts.sys.useCaseSensitiveFileNames;
var newLine = Ast.ts.sys.newLine;
var marked = require('marked');
function getNewLine() {
    return newLine;
}
function cleanNameWithoutSpaceAndToLowerCase(name) {
    return name.toLowerCase().replace(/ /g, '-');
}
function getCanonicalFileName(fileName) {
    return useCaseSensitiveFileNames ? fileName : fileName.toLowerCase();
}
var formatDiagnosticsHost = {
    getCurrentDirectory: getCurrentDirectory,
    getCanonicalFileName: getCanonicalFileName,
    getNewLine: getNewLine
};
function markedtags(tags) {
    var mtags = tags;
    _.forEach(mtags, function (tag) {
        tag.comment = marked(LinkParser.resolveLinks(tag.comment));
    });
    return mtags;
}
function mergeTagsAndArgs(args, jsdoctags) {
    var margs = _.cloneDeep(args);
    _.forEach(margs, function (arg) {
        arg.tagName = {
            text: 'param'
        };
        if (jsdoctags) {
            _.forEach(jsdoctags, function (jsdoctag) {
                if (jsdoctag.name && jsdoctag.name.text === arg.name) {
                    arg.tagName = jsdoctag.tagName;
                    arg.name = jsdoctag.name;
                    arg.comment = jsdoctag.comment;
                    arg.typeExpression = jsdoctag.typeExpression;
                }
            });
        }
    });
    // Add example & returns & private
    if (jsdoctags) {
        _.forEach(jsdoctags, function (jsdoctag) {
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'example' || jsdoctag.tagName.text === 'private')) {
                margs.push({
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                });
            }
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'returns' || jsdoctag.tagName.text === 'return')) {
                var ret = {
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                };
                if (jsdoctag.typeExpression && jsdoctag.typeExpression.type) {
                    ret.returnType = kindToType(jsdoctag.typeExpression.type.kind);
                }
                margs.push(ret);
            }
        });
    }
    return margs;
}
function readConfig(configFile) {
    var result = Ast.ts.readConfigFile(configFile, Ast.ts.sys.readFile);
    if (result.error) {
        var message = Ast.ts.formatDiagnostics([result.error], formatDiagnosticsHost);
        throw new Error(message);
    }
    return result.config;
}
function stripBom(source) {
    if (source.charCodeAt(0) === 0xfeff) {
        return source.slice(1);
    }
    return source;
}
function hasBom(source) {
    return source.charCodeAt(0) === 0xfeff;
}
function handlePath(files, cwd) {
    var _files = files;
    var i = 0;
    var len = files.length;
    for (i; i < len; i++) {
        if (files[i].indexOf(cwd) === -1) {
            files[i] = path.resolve(cwd + path.sep + files[i]);
        }
    }
    return _files;
}
function cleanLifecycleHooksFromMethods(methods) {
    var result = [];
    if (typeof methods !== 'undefined') {
        var i = 0;
        var len = methods.length;
        for (i; i < len; i++) {
            if (!(methods[i].name in AngularLifecycleHooks)) {
                result.push(methods[i]);
            }
        }
    }
    return result;
}
function cleanSourcesForWatch(list) {
    return list.filter(function (element) {
        if (fs.existsSync(process.cwd() + path.sep + element)) {
            return element;
        }
    });
}
function getNamesCompareFn(name) {
    /**
     * Copyright https://github.com/ng-bootstrap/ng-bootstrap
     */
    name = name || 'name';
    var t = function (a, b) {
        if (a[name]) {
            return a[name].localeCompare(b[name]);
        }
        else {
            return 0;
        }
    };
    return t;
}
function isIgnore(member) {
    if (member.jsDoc) {
        for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
            var doc = _a[_i];
            if (doc.tags) {
                for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                    var tag = _c[_b];
                    if (tag.tagName.text.indexOf('ignore') > -1) {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}
// https://tc39.github.io/ecma262/#sec-array.prototype.includes
if (!Array.prototype.includes) {
    Object.defineProperty(Array.prototype, 'includes', {
        value: function (searchElement, fromIndex) {
            if (this == null) {
                throw new TypeError('"this" is null or not defined');
            }
            // 1. Let O be ? ToObject(this value).
            var o = Object(this);
            // 2. Let len be ? ToLength(? Get(O, "length")).
            var len = o.length >>> 0;
            // 3. If len is 0, return false.
            if (len === 0) {
                return false;
            }
            // 4. Let n be ? ToInteger(fromIndex).
            //    (If fromIndex is undefined, this step produces the value 0.)
            var n = fromIndex | 0;
            // 5. If n ≥ 0, then
            //  a. Let k be n.
            // 6. Else n < 0,
            //  a. Let k be len + n.
            //  b. If k < 0, let k be 0.
            var k = Math.max(n >= 0 ? n : len - Math.abs(n), 0);
            function sameValueZero(x, y) {
                return (x === y ||
                    (typeof x === 'number' && typeof y === 'number' && isNaN(x) && isNaN(y)));
            }
            // 7. Repeat, while k < len
            while (k < len) {
                // a. Let elementK be the result of ? Get(O, ! ToString(k)).
                // b. If SameValueZero(searchElement, elementK) is true, return true.
                if (sameValueZero(o[k], searchElement)) {
                    return true;
                }
                // c. Increase k by 1.
                k++;
            }
            // 8. Return false
            return false;
        }
    });
}
function findMainSourceFolder(files) {
    var mainFolder = '';
    var mainFolderCount = 0;
    var rawFolders = files.map(function (filepath) {
        var shortPath = filepath.replace(process.cwd() + path.sep, '');
        return path.dirname(shortPath);
    });
    var folders = {};
    rawFolders = _.uniq(rawFolders);
    for (var i = 0; i < rawFolders.length; i++) {
        var sep = rawFolders[i].split(path.sep);
        sep.forEach(function (folder) {
            if (folders[folder]) {
                folders[folder] += 1;
            }
            else {
                folders[folder] = 1;
            }
        });
    }
    for (var f in folders) {
        if (folders[f] > mainFolderCount) {
            mainFolderCount = folders[f];
            mainFolder = f;
        }
    }
    return mainFolder;
}
// Create a compilerHost object to allow the compiler to read and write files
function compilerHost(transpileOptions) {
    var inputFileName = transpileOptions.fileName || (transpileOptions.jsx ? 'module.tsx' : 'module.ts');
    var toReturn = {
        getSourceFile: function (fileName) {
            if (fileName.lastIndexOf('.ts') !== -1 || fileName.lastIndexOf('.js') !== -1) {
                if (fileName === 'lib.d.ts') {
                    return undefined;
                }
                if (fileName.substr(-5) === '.d.ts') {
                    return undefined;
                }
                if (path.isAbsolute(fileName) === false) {
                    fileName = path.join(transpileOptions.tsconfigDirectory, fileName);
                }
                if (!fs.existsSync(fileName)) {
                    return undefined;
                }
                var libSource = '';
                try {
                    libSource = fs.readFileSync(fileName).toString();
                    if (hasBom(libSource)) {
                        libSource = stripBom(libSource);
                    }
                }
                catch (e) {
                    logger.debug(e, fileName);
                }
                return Ast.ts.createSourceFile(fileName, libSource, transpileOptions.target, false);
            }
            return undefined;
        },
        writeFile: function (name, text) { },
        getDefaultLibFileName: function () { return 'lib.d.ts'; },
        useCaseSensitiveFileNames: function () { return false; },
        getCanonicalFileName: function (fileName) { return fileName; },
        getCurrentDirectory: function () { return ''; },
        getNewLine: function () { return '\n'; },
        fileExists: function (fileName) { return fileName === inputFileName; },
        readFile: function () { return ''; },
        directoryExists: function () { return true; },
        getDirectories: function () { return []; }
    };
    return toReturn;
}
function detectIndent(str, count) {
    var stripIndent = function (stripedString) {
        var match = stripedString.match(/^[ \t]*(?=\S)/gm);
        if (!match) {
            return stripedString;
        }
        // TODO: use spread operator when targeting Node.js 6
        var indent = Math.min.apply(Math, match.map(function (x) { return x.length; })); // eslint-disable-line
        var re = new RegExp("^[ \\t]{" + indent + "}", 'gm');
        return indent > 0 ? stripedString.replace(re, '') : stripedString;
    };
    var repeating = function (n, repeatString) {
        repeatString = repeatString === undefined ? ' ' : repeatString;
        if (typeof repeatString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof repeatString + "`");
        }
        if (n < 0) {
            throw new TypeError("Expected `count` to be a positive finite number, got `" + n + "`");
        }
        var ret = '';
        do {
            if (n & 1) {
                ret += repeatString;
            }
            repeatString += repeatString;
        } while ((n >>= 1));
        return ret;
    };
    var indentString = function (indentedString, indentCount) {
        var indent = ' ';
        indentCount = indentCount === undefined ? 1 : indentCount;
        if (typeof indentedString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof indentedString + "`");
        }
        if (typeof indentCount !== 'number') {
            throw new TypeError("Expected `count` to be a `number`, got `" + typeof indentCount + "`");
        }
        if (typeof indent !== 'string') {
            throw new TypeError("Expected `indent` to be a `string`, got `" + typeof indent + "`");
        }
        if (indentCount === 0) {
            return indentedString;
        }
        indent = indentCount > 1 ? repeating(indentCount, indent) : indent;
        return indentedString.replace(/^(?!\s*$)/gm, indent);
    };
    return indentString(stripIndent(str), count || 0);
}

var traverse = require('traverse');
var DependenciesEngine = /** @class */ (function () {
    function DependenciesEngine() {
        this.miscellaneous = {
            variables: [],
            functions: [],
            typealiases: [],
            enumerations: [],
            groupedVariables: [],
            groupedFunctions: [],
            groupedEnumerations: [],
            groupedTypeAliases: []
        };
    }
    DependenciesEngine.getInstance = function () {
        if (!DependenciesEngine.instance) {
            DependenciesEngine.instance = new DependenciesEngine();
        }
        return DependenciesEngine.instance;
    };
    DependenciesEngine.prototype.updateModulesDeclarationsExportsTypes = function () {
        var _this = this;
        var mergeTypes = function (entry) {
            var directive = _this.findInCompodocDependencies(entry.name, _this.directives, entry.file);
            if (typeof directive.data !== 'undefined') {
                entry.type = 'directive';
                entry.id = directive.data.id;
            }
            var component = _this.findInCompodocDependencies(entry.name, _this.components, entry.file);
            if (typeof component.data !== 'undefined') {
                entry.type = 'component';
                entry.id = component.data.id;
            }
            var pipe = _this.findInCompodocDependencies(entry.name, _this.pipes, entry.file);
            if (typeof pipe.data !== 'undefined') {
                entry.type = 'pipe';
                entry.id = pipe.data.id;
            }
        };
        this.modules.forEach(function (module) {
            module.declarations.forEach(function (declaration) {
                mergeTypes(declaration);
            });
            module.exports.forEach(function (expt) {
                mergeTypes(expt);
            });
            module.entryComponents.forEach(function (ent) {
                mergeTypes(ent);
            });
        });
    };
    DependenciesEngine.prototype.init = function (data) {
        traverse(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
            }
        });
        this.rawData = data;
        this.modules = _.sortBy(this.rawData.modules, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModulesForOverview = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModules = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.components = _.sortBy(this.rawData.components, [function (el) { return el.name.toLowerCase(); }]);
        this.controllers = _.sortBy(this.rawData.controllers, [function (el) { return el.name.toLowerCase(); }]);
        this.directives = _.sortBy(this.rawData.directives, [function (el) { return el.name.toLowerCase(); }]);
        this.injectables = _.sortBy(this.rawData.injectables, [function (el) { return el.name.toLowerCase(); }]);
        this.interceptors = _.sortBy(this.rawData.interceptors, [function (el) { return el.name.toLowerCase(); }]);
        this.guards = _.sortBy(this.rawData.guards, [function (el) { return el.name.toLowerCase(); }]);
        this.interfaces = _.sortBy(this.rawData.interfaces, [function (el) { return el.name.toLowerCase(); }]);
        this.pipes = _.sortBy(this.rawData.pipes, [function (el) { return el.name.toLowerCase(); }]);
        this.classes = _.sortBy(this.rawData.classes, [function (el) { return el.name.toLowerCase(); }]);
        this.miscellaneous = this.rawData.miscellaneous;
        this.prepareMiscellaneous();
        this.updateModulesDeclarationsExportsTypes();
        this.routes = this.rawData.routesTree;
        this.manageDuplicatesName();
        this.cleanRawModulesNames();
    };
    DependenciesEngine.prototype.cleanRawModulesNames = function () {
        this.rawModulesForOverview = this.rawModulesForOverview.map(function (module) {
            module.name = module.name.replace('$', '');
            return module;
        });
    };
    DependenciesEngine.prototype.findInCompodocDependencies = function (name, data, file) {
        var _result = {
            source: 'internal',
            data: undefined
        };
        var nameFoundCounter = 0;
        if (data && data.length > 0) {
            for (var i = 0; i < data.length; i++) {
                if (typeof name !== 'undefined') {
                    if (typeof file !== 'undefined') {
                        if (name.indexOf(data[i].name) !== -1 &&
                            file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                    else {
                        if (name.indexOf(data[i].name) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                }
            }
            // Prevent wrong matching like MultiSelectOptionDirective with SelectOptionDirective, or QueryParamGroupService with QueryParamGroup
            if (nameFoundCounter > 1) {
                var found = false;
                for (var i = 0; i < data.length; i++) {
                    if (typeof name !== 'undefined') {
                        if (typeof file !== 'undefined') {
                            if (name === data[i].name &&
                                file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                        else {
                            if (name === data[i].name) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                    }
                }
                if (!found) {
                    _result = {
                        source: 'internal',
                        data: undefined
                    };
                }
            }
        }
        return _result;
    };
    DependenciesEngine.prototype.manageDuplicatesName = function () {
        var processDuplicates = function (element, index, array) {
            var elementsWithSameName = _.filter(array, { name: element.name });
            if (elementsWithSameName.length > 1) {
                // First element is the reference for duplicates
                for (var i = 1; i < elementsWithSameName.length; i++) {
                    var elementToEdit = elementsWithSameName[i];
                    if (typeof elementToEdit.isDuplicate === 'undefined') {
                        elementToEdit.isDuplicate = true;
                        elementToEdit.duplicateId = i;
                        elementToEdit.duplicateName =
                            elementToEdit.name + '-' + elementToEdit.duplicateId;
                        elementToEdit.id = elementToEdit.id + '-' + elementToEdit.duplicateId;
                    }
                }
            }
            return element;
        };
        this.classes = this.classes.map(processDuplicates);
        this.interfaces = this.interfaces.map(processDuplicates);
        this.injectables = this.injectables.map(processDuplicates);
        this.pipes = this.pipes.map(processDuplicates);
        this.interceptors = this.interceptors.map(processDuplicates);
        this.guards = this.guards.map(processDuplicates);
        this.modules = this.modules.map(processDuplicates);
        this.components = this.components.map(processDuplicates);
        this.controllers = this.controllers.map(processDuplicates);
        this.directives = this.directives.map(processDuplicates);
    };
    DependenciesEngine.prototype.find = function (name) {
        var _this = this;
        var searchFunctions = [
            function () { return _this.findInCompodocDependencies(name, _this.injectables); },
            function () { return _this.findInCompodocDependencies(name, _this.interceptors); },
            function () { return _this.findInCompodocDependencies(name, _this.guards); },
            function () { return _this.findInCompodocDependencies(name, _this.interfaces); },
            function () { return _this.findInCompodocDependencies(name, _this.classes); },
            function () { return _this.findInCompodocDependencies(name, _this.components); },
            function () { return _this.findInCompodocDependencies(name, _this.controllers); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.variables); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.functions); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.typealiases); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.enumerations); },
            function () { return AngularApiUtil$1.findApi(name); }
        ];
        for (var _i = 0, searchFunctions_1 = searchFunctions; _i < searchFunctions_1.length; _i++) {
            var searchFunction = searchFunctions_1[_i];
            var result = searchFunction();
            if (result.data) {
                return result;
            }
        }
        return undefined;
    };
    DependenciesEngine.prototype.update = function (updatedData) {
        var _this = this;
        if (updatedData.modules.length > 0) {
            _.forEach(updatedData.modules, function (module) {
                var _index = _.findIndex(_this.modules, { name: module.name });
                _this.modules[_index] = module;
            });
        }
        if (updatedData.components.length > 0) {
            _.forEach(updatedData.components, function (component) {
                var _index = _.findIndex(_this.components, { name: component.name });
                _this.components[_index] = component;
            });
        }
        if (updatedData.controllers.length > 0) {
            _.forEach(updatedData.controllers, function (controller) {
                var _index = _.findIndex(_this.controllers, { name: controller.name });
                _this.controllers[_index] = controller;
            });
        }
        if (updatedData.directives.length > 0) {
            _.forEach(updatedData.directives, function (directive) {
                var _index = _.findIndex(_this.directives, { name: directive.name });
                _this.directives[_index] = directive;
            });
        }
        if (updatedData.injectables.length > 0) {
            _.forEach(updatedData.injectables, function (injectable) {
                var _index = _.findIndex(_this.injectables, { name: injectable.name });
                _this.injectables[_index] = injectable;
            });
        }
        if (updatedData.interceptors.length > 0) {
            _.forEach(updatedData.interceptors, function (interceptor) {
                var _index = _.findIndex(_this.interceptors, { name: interceptor.name });
                _this.interceptors[_index] = interceptor;
            });
        }
        if (updatedData.guards.length > 0) {
            _.forEach(updatedData.guards, function (guard) {
                var _index = _.findIndex(_this.guards, { name: guard.name });
                _this.guards[_index] = guard;
            });
        }
        if (updatedData.interfaces.length > 0) {
            _.forEach(updatedData.interfaces, function (int) {
                var _index = _.findIndex(_this.interfaces, { name: int.name });
                _this.interfaces[_index] = int;
            });
        }
        if (updatedData.pipes.length > 0) {
            _.forEach(updatedData.pipes, function (pipe) {
                var _index = _.findIndex(_this.pipes, { name: pipe.name });
                _this.pipes[_index] = pipe;
            });
        }
        if (updatedData.classes.length > 0) {
            _.forEach(updatedData.classes, function (classe) {
                var _index = _.findIndex(_this.classes, { name: classe.name });
                _this.classes[_index] = classe;
            });
        }
        /**
         * Miscellaneous update
         */
        if (updatedData.miscellaneous.variables.length > 0) {
            _.forEach(updatedData.miscellaneous.variables, function (variable) {
                var _index = _.findIndex(_this.miscellaneous.variables, {
                    name: variable.name,
                    file: variable.file
                });
                _this.miscellaneous.variables[_index] = variable;
            });
        }
        if (updatedData.miscellaneous.functions.length > 0) {
            _.forEach(updatedData.miscellaneous.functions, function (func) {
                var _index = _.findIndex(_this.miscellaneous.functions, {
                    name: func.name,
                    file: func.file
                });
                _this.miscellaneous.functions[_index] = func;
            });
        }
        if (updatedData.miscellaneous.typealiases.length > 0) {
            _.forEach(updatedData.miscellaneous.typealiases, function (typealias) {
                var _index = _.findIndex(_this.miscellaneous.typealiases, {
                    name: typealias.name,
                    file: typealias.file
                });
                _this.miscellaneous.typealiases[_index] = typealias;
            });
        }
        if (updatedData.miscellaneous.enumerations.length > 0) {
            _.forEach(updatedData.miscellaneous.enumerations, function (enumeration) {
                var _index = _.findIndex(_this.miscellaneous.enumerations, {
                    name: enumeration.name,
                    file: enumeration.file
                });
                _this.miscellaneous.enumerations[_index] = enumeration;
            });
        }
        this.prepareMiscellaneous();
    };
    DependenciesEngine.prototype.findInCompodoc = function (name) {
        var mergedData = _.concat([], this.modules, this.components, this.controllers, this.directives, this.injectables, this.interceptors, this.guards, this.interfaces, this.pipes, this.classes, this.miscellaneous.enumerations, this.miscellaneous.typealiases, this.miscellaneous.variables, this.miscellaneous.functions);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    DependenciesEngine.prototype.prepareMiscellaneous = function () {
        this.miscellaneous.variables.sort(getNamesCompareFn());
        this.miscellaneous.functions.sort(getNamesCompareFn());
        this.miscellaneous.enumerations.sort(getNamesCompareFn());
        this.miscellaneous.typealiases.sort(getNamesCompareFn());
        // group each subgoup by file
        this.miscellaneous.groupedVariables = _.groupBy(this.miscellaneous.variables, 'file');
        this.miscellaneous.groupedFunctions = _.groupBy(this.miscellaneous.functions, 'file');
        this.miscellaneous.groupedEnumerations = _.groupBy(this.miscellaneous.enumerations, 'file');
        this.miscellaneous.groupedTypeAliases = _.groupBy(this.miscellaneous.typealiases, 'file');
    };
    DependenciesEngine.prototype.getModule = function (name) {
        return _.find(this.modules, ['name', name]);
    };
    DependenciesEngine.prototype.getRawModule = function (name) {
        return _.find(this.rawModules, ['name', name]);
    };
    DependenciesEngine.prototype.getModules = function () {
        return this.modules;
    };
    DependenciesEngine.prototype.getComponents = function () {
        return this.components;
    };
    DependenciesEngine.prototype.getControllers = function () {
        return this.controllers;
    };
    DependenciesEngine.prototype.getDirectives = function () {
        return this.directives;
    };
    DependenciesEngine.prototype.getInjectables = function () {
        return this.injectables;
    };
    DependenciesEngine.prototype.getInterceptors = function () {
        return this.interceptors;
    };
    DependenciesEngine.prototype.getGuards = function () {
        return this.guards;
    };
    DependenciesEngine.prototype.getInterfaces = function () {
        return this.interfaces;
    };
    DependenciesEngine.prototype.getRoutes = function () {
        return this.routes;
    };
    DependenciesEngine.prototype.getPipes = function () {
        return this.pipes;
    };
    DependenciesEngine.prototype.getClasses = function () {
        return this.classes;
    };
    DependenciesEngine.prototype.getMiscellaneous = function () {
        return this.miscellaneous;
    };
    return DependenciesEngine;
}());
var DependenciesEngine$1 = DependenciesEngine.getInstance();

var FileEngine = /** @class */ (function () {
    function FileEngine() {
    }
    FileEngine.getInstance = function () {
        if (!FileEngine.instance) {
            FileEngine.instance = new FileEngine();
        }
        return FileEngine.instance;
    };
    FileEngine.prototype.get = function (filepath) {
        return new Promise(function (resolve, reject) {
            fs.readFile(path.resolve(filepath), 'utf8', function (err, data) {
                if (err) {
                    reject('Error during ' + filepath + ' read');
                }
                else {
                    resolve(data);
                }
            });
        });
    };
    FileEngine.prototype.write = function (filepath, contents) {
        return new Promise(function (resolve, reject) {
            fs.outputFile(path.resolve(filepath), contents, function (err) {
                if (err) {
                    reject(err);
                }
                else {
                    resolve();
                }
            });
        });
    };
    FileEngine.prototype.writeSync = function (filepath, contents) {
        fs.outputFileSync(filepath, contents);
    };
    FileEngine.prototype.getSync = function (filepath) {
        return fs.readFileSync(path.resolve(filepath), 'utf8');
    };
    /**
     * @param file The file to check
     */
    FileEngine.prototype.existsSync = function (file) {
        return fs.existsSync(file);
    };
    return FileEngine;
}());
var FileEngine$1 = FileEngine.getInstance();

var traverse$1 = require('traverse');
var ExportJsonEngine = /** @class */ (function () {
    function ExportJsonEngine() {
    }
    ExportJsonEngine.getInstance = function () {
        if (!ExportJsonEngine.instance) {
            ExportJsonEngine.instance = new ExportJsonEngine();
        }
        return ExportJsonEngine.instance;
    };
    ExportJsonEngine.prototype.export = function (outputFolder, data) {
        var exportData = {};
        traverse$1(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (Configuration$1.mainData.disableSourceCode) {
                    delete node.sourceCode;
                }
            }
        });
        exportData.pipes = data.pipes;
        exportData.interfaces = data.interfaces;
        exportData.injectables = data.injectables;
        exportData.classes = data.classes;
        exportData.directives = data.directives;
        exportData.components = data.components;
        exportData.modules = this.processModules();
        exportData.miscellaneous = data.miscellaneous;
        if (!Configuration$1.mainData.disableRoutesGraph) {
            exportData.routes = data.routes;
        }
        if (!Configuration$1.mainData.disableCoverage) {
            exportData.coverage = data.coverageData;
        }
        return FileEngine$1.write(outputFolder + path.sep + '/documentation.json', JSON.stringify(exportData, undefined, 4)).catch(function (err) {
            logger.error('Error during export file generation ', err);
            return Promise.reject(err);
        });
    };
    ExportJsonEngine.prototype.processModules = function () {
        var modules = DependenciesEngine$1.getModules();
        var _resultedModules = [];
        for (var moduleNr = 0; moduleNr < modules.length; moduleNr++) {
            var moduleElement = {
                name: modules[moduleNr].name,
                children: [
                    {
                        type: 'providers',
                        elements: []
                    },
                    {
                        type: 'declarations',
                        elements: []
                    },
                    {
                        type: 'imports',
                        elements: []
                    },
                    {
                        type: 'exports',
                        elements: []
                    },
                    {
                        type: 'bootstrap',
                        elements: []
                    },
                    {
                        type: 'classes',
                        elements: []
                    }
                ]
            };
            for (var k = 0; k < modules[moduleNr].providers.length; k++) {
                var providerElement = {
                    name: modules[moduleNr].providers[k].name
                };
                moduleElement.children[0].elements.push(providerElement);
            }
            for (var k = 0; k < modules[moduleNr].declarations.length; k++) {
                var declarationElement = {
                    name: modules[moduleNr].declarations[k].name
                };
                moduleElement.children[1].elements.push(declarationElement);
            }
            for (var k = 0; k < modules[moduleNr].imports.length; k++) {
                var importElement = {
                    name: modules[moduleNr].imports[k].name
                };
                moduleElement.children[2].elements.push(importElement);
            }
            for (var k = 0; k < modules[moduleNr].exports.length; k++) {
                var exportElement = {
                    name: modules[moduleNr].exports[k].name
                };
                moduleElement.children[3].elements.push(exportElement);
            }
            for (var k = 0; k < modules[moduleNr].bootstrap.length; k++) {
                var bootstrapElement = {
                    name: modules[moduleNr].bootstrap[k].name
                };
                moduleElement.children[4].elements.push(bootstrapElement);
            }
            _resultedModules.push(moduleElement);
        }
        return _resultedModules;
    };
    return ExportJsonEngine;
}());
var ExportJsonEngine$1 = ExportJsonEngine.getInstance();

var MarkdownToPDFEngine = /** @class */ (function () {
    function MarkdownToPDFEngine() {
        // console.log('MarkdownToPDFEngine');
        this.markedInstance = require('marked');
        this.renderer = new this.markedInstance.Renderer();
        this.renderer.strong = function (text) {
            console.log('MarkdownToPDFEngine strong: ', text);
            return { text: text, bold: true };
        };
        this.renderer.paragraph = function (text) {
            console.log('MarkdownToPDFEngine paragraph: ', text);
            return text;
        };
        // TODO Add custom parser... -> https://github.com/markedjs/marked/issues/504
        this.markedInstance.setOptions({
            gfm: false,
            breaks: false
        });
    }
    MarkdownToPDFEngine.getInstance = function () {
        if (!MarkdownToPDFEngine.instance) {
            MarkdownToPDFEngine.instance = new MarkdownToPDFEngine();
        }
        return MarkdownToPDFEngine.instance;
    };
    MarkdownToPDFEngine.prototype.convert = function (data) {
        // console.log('MarkdownToPDFEngine convert');
        /*const tokens = this.markedInstance.lexer('**This is bold text**');
        console.log(tokens);
        const html = this.markedInstance.parser(tokens);
        console.log(html);*/
        return this.markedInstance(data); // '**This is bold text**', { renderer: this.renderer });
    };
    return MarkdownToPDFEngine;
}());
var MarkdownToPdfEngine = MarkdownToPDFEngine.getInstance();

var PdfPrinter = require('pdfmake');
var ExportPdfEngine = /** @class */ (function () {
    function ExportPdfEngine() {
    }
    ExportPdfEngine.getInstance = function () {
        if (!ExportPdfEngine.instance) {
            ExportPdfEngine.instance = new ExportPdfEngine();
        }
        return ExportPdfEngine.instance;
    };
    ExportPdfEngine.prototype.export = function (outputFolder) {
        var fonts = {
            Roboto: {
                normal: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-regular.ttf'),
                bold: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-700.ttf')
            }
        };
        var printer = new PdfPrinter(fonts);
        var docDefinition = {
            info: {
                title: Configuration$1.mainData.documentationMainName
            },
            content: [],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    color: '#008cff',
                    margin: [0, 0, 0, 15]
                },
                subheader: {
                    fontSize: 15,
                    bold: true
                }
            }
        };
        docDefinition.content.push({
            text: Configuration$1.mainData.documentationMainName,
            alignment: 'center',
            bold: true,
            fontSize: 22,
            margin: [10, 350, 10, 300]
        });
        if (!Configuration$1.mainData.hideGenerator) {
            docDefinition.content.push({
                text: 'Documentation generated using',
                alignment: 'center'
            });
            docDefinition.content.push({
                image: "data:image/jpeg;base64,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",
                width: 70,
                alignment: 'center',
                pageBreak: 'after'
            });
        }
        docDefinition.content.push({
            toc: {
                title: {
                    text: 'Table of contents',
                    bold: true,
                    alignment: 'center',
                    fontSize: 18,
                    margin: [10, 10, 10, 50]
                },
                numberStyle: {
                    bold: true
                }
            },
            pageBreak: 'after'
        });
        // Add README page if available
        docDefinition.content.push(this.generateMarkdownContent());
        // Add CHANGELOG page if available
        // Add CONTRIBUTING page if available
        // Add LICENSE page if available
        // Add TODO page if available
        // Add Dependencies page if available
        // Add Additional pages if available
        docDefinition.content.push(this.generateModulesContent());
        docDefinition.content.push(this.generateComponentsContent());
        // Classes
        // Injectables
        // Interceptors
        // Guards
        // Interfaces
        // Pipes
        // Miscellaneous
        // Routes
        // Coverage - docDefinition.content.push(...this.coverageEngine.calculateTable());
        var pdfDoc = printer.createPdfKitDocument(docDefinition);
        return new Promise(function (resolve, reject) {
            fs.ensureFile(outputFolder + path.sep + 'documentation.pdf', function (err) {
                if (err) {
                    reject("Error during pdf generation: " + err);
                }
                else {
                    pdfDoc.pipe(fs.createWriteStream(outputFolder + path.sep + 'documentation.pdf'));
                    pdfDoc.end();
                    resolve();
                }
            });
        });
    };
    ExportPdfEngine.prototype.firstCharacterUpperCase = function (sentence) {
        return sentence.charAt(0).toUpperCase() + sentence.slice(1);
    };
    ExportPdfEngine.prototype.generateMarkdownContent = function () {
        var _this = this;
        var pages = Configuration$1.markDownPages;
        var data = [];
        pages.forEach(function (page) {
            data.push({
                text: "" + _this.firstCharacterUpperCase(page.name),
                tocItem: true,
                style: 'header'
            });
            data.push({
                text: MarkdownToPdfEngine.convert(page.markdown),
                margin: [0, 10]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.insertPageReturn = function (data) {
        data.push({
            text: " ",
            margin: [0, 0, 0, 20],
            pageBreak: 'after'
        });
    };
    ExportPdfEngine.prototype.generateModulesContent = function () {
        var data = [];
        data.push({
            text: 'Modules',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.modules, function (module) {
            data.push({
                text: "" + module.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: module.file
                    }
                ],
                margin: [0, 10]
            });
            if (module.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + module.rawdescription,
                    margin: [0, 5]
                });
            }
            if (module.declarations.length > 0) {
                data.push({
                    text: "Declarations :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_1 = { ul: [] };
                _.forEach(module.declarations, function (declaration) {
                    list_1.ul.push({
                        text: "" + declaration.name
                    });
                });
                data.push(list_1);
            }
            if (module.providers.length > 0) {
                data.push({
                    text: "Providers :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_2 = { ul: [] };
                _.forEach(module.providers, function (provider) {
                    list_2.ul.push({
                        text: "" + provider.name
                    });
                });
                data.push(list_2);
            }
            if (module.imports.length > 0) {
                data.push({
                    text: "Imports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_3 = { ul: [] };
                _.forEach(module.imports, function (importRef) {
                    list_3.ul.push({
                        text: "" + importRef.name
                    });
                });
                data.push(list_3);
            }
            if (module.exports.length > 0) {
                data.push({
                    text: "Exports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_4 = { ul: [] };
                _.forEach(module.exports, function (exportRef) {
                    list_4.ul.push({
                        text: "" + exportRef.name
                    });
                });
                data.push(list_4);
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.generateComponentsContent = function () {
        var data = [];
        data.push({
            text: 'Components',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.components, function (component) {
            data.push({
                text: "" + component.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: component.file
                    }
                ],
                margin: [0, 10]
            });
            if (component.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + component.rawdescription,
                    margin: [0, 5]
                });
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    return ExportPdfEngine;
}());
var ExportPdfEngine$1 = ExportPdfEngine.getInstance();

var ExportEngine = /** @class */ (function () {
    function ExportEngine() {
    }
    ExportEngine.getInstance = function () {
        if (!ExportEngine.instance) {
            ExportEngine.instance = new ExportEngine();
        }
        return ExportEngine.instance;
    };
    ExportEngine.prototype.export = function (outputFolder, data) {
        switch (Configuration$1.mainData.exportFormat) {
            case 'json':
                return ExportJsonEngine$1.export(outputFolder, data);
            case 'pdf':
                return ExportPdfEngine$1.export(outputFolder);
        }
    };
    return ExportEngine;
}());
var ExportEngine$1 = ExportEngine.getInstance();

var BreakCommaHelper = /** @class */ (function () {
    function BreakCommaHelper(bars) {
        this.bars = bars;
    }
    BreakCommaHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/,/g, ',<br>');
        return new Handlebars.SafeString(text);
    };
    return BreakCommaHelper;
}());

var BreakLinesHelper = /** @class */ (function () {
    function BreakLinesHelper(bars) {
        this.bars = bars;
    }
    BreakLinesHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/(\r\n|\n|\r)/gm, '<br>');
        text = text.replace(/ /gm, '&nbsp;');
        text = text.replace(/	/gm, '&nbsp;&nbsp;&nbsp;&nbsp;');
        return new Handlebars.SafeString(text);
    };
    return BreakLinesHelper;
}());

var CleanParagraphHelper = /** @class */ (function () {
    function CleanParagraphHelper() {
    }
    CleanParagraphHelper.prototype.helperFunc = function (context, text) {
        text = text.replace(/<p>/gm, '');
        text = text.replace(/<\/p>/gm, '');
        return new Handlebars.SafeString(text);
    };
    return CleanParagraphHelper;
}());

var CompareHelper = /** @class */ (function () {
    function CompareHelper() {
    }
    CompareHelper.prototype.helperFunc = function (context, a, operator, b, options) {
        if (arguments.length < 4) {
            throw new Error('handlebars Helper {{compare}} expects 4 arguments');
        }
        var result;
        switch (operator) {
            case 'indexof':
                result = b.indexOf(a) !== -1;
                break;
            case '===':
                result = a === b;
                break;
            case '!==':
                result = a !== b;
                break;
            case '>':
                result = a > b;
                break;
            default: {
                throw new Error('helper {{compare}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return CompareHelper;
}());

var DebugHelper = /** @class */ (function () {
    function DebugHelper() {
    }
    DebugHelper.prototype.helperFunc = function (context, optionalValue) {
        console.log('Current Context');
        console.log('====================');
        console.log(context);
        if (optionalValue) {
            console.log('OptionalValue');
            console.log('====================');
            console.log(optionalValue);
        }
    };
    return DebugHelper;
}());

var ElementAloneHelper = /** @class */ (function () {
    function ElementAloneHelper() {
    }
    ElementAloneHelper.prototype.helperFunc = function (context, elements, elementType, options) {
        var alones = [];
        var modules = DependenciesEngine$1.modules;
        elements.forEach(function (element) {
            var foundInOneModule = false;
            modules.forEach(function (module) {
                module.declarations.forEach(function (declaration) {
                    if (declaration.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (declaration.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.controllers.forEach(function (controller) {
                    if (controller.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (controller.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.providers.forEach(function (provider) {
                    if (provider.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (provider.file === element.file) {
                        foundInOneModule = true;
                    }
                });
            });
            if (!foundInOneModule) {
                alones.push(element);
            }
        });
        if (alones.length > 0) {
            switch (elementType) {
                case 'component':
                    context.components = alones;
                    break;
                case 'directive':
                    context.directives = alones;
                    break;
                case 'controller':
                    context.controllers = alones;
                    break;
                case 'injectable':
                    context.injectables = alones;
                    break;
                case 'pipe':
                    context.pipes = alones;
                    break;
            }
            return options.fn(context);
        }
    };
    return ElementAloneHelper;
}());

var EscapeSimpleQuoteHelper = /** @class */ (function () {
    function EscapeSimpleQuoteHelper() {
    }
    EscapeSimpleQuoteHelper.prototype.helperFunc = function (context, text) {
        if (!text) {
            return;
        }
        text = text.replace(/'/g, "\\'");
        text = text.replace(/(\r\n|\n|\r)/gm, '');
        return text;
    };
    return EscapeSimpleQuoteHelper;
}());

var FilterAngular2ModulesHelper = /** @class */ (function () {
    function FilterAngular2ModulesHelper() {
    }
    FilterAngular2ModulesHelper.prototype.helperFunc = function (context, text, options) {
        var NG2_MODULES = [
            'BrowserModule',
            'FormsModule',
            'HttpModule',
            'RouterModule'
        ];
        var len = NG2_MODULES.length;
        var i = 0;
        var result = false;
        for (i; i < len; i++) {
            if (text.indexOf(NG2_MODULES[i]) > -1) {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return FilterAngular2ModulesHelper;
}());

var AngularVersionUtil = /** @class */ (function () {
    function AngularVersionUtil() {
    }
    AngularVersionUtil.getInstance = function () {
        if (!AngularVersionUtil.instance) {
            AngularVersionUtil.instance = new AngularVersionUtil();
        }
        return AngularVersionUtil.instance;
    };
    AngularVersionUtil.prototype.cleanVersion = function (version) {
        return version
            .replace('~', '')
            .replace('^', '')
            .replace('=', '')
            .replace('<', '')
            .replace('>', '');
    };
    AngularVersionUtil.prototype.getAngularVersionOfProject = function (packageData) {
        var _result = '';
        if (packageData.dependencies) {
            var angularCore = packageData.dependencies[AngularVersionUtil.CorePackage];
            if (angularCore) {
                _result = this.cleanVersion(angularCore);
            }
        }
        return _result;
    };
    AngularVersionUtil.prototype.isAngularVersionArchived = function (version) {
        var result;
        try {
            result = semver.compare(version, '2.4.10') <= 0;
        }
        catch (e) { }
        return result;
    };
    AngularVersionUtil.prototype.prefixOfficialDoc = function (version) {
        return this.isAngularVersionArchived(version) ? 'v2.' : '';
    };
    AngularVersionUtil.prototype.getApiLink = function (api, angularVersion) {
        var angularDocPrefix = this.prefixOfficialDoc(angularVersion);
        return "https://" + angularDocPrefix + "angular.io/" + api.path;
    };
    AngularVersionUtil.CorePackage = '@angular/core';
    return AngularVersionUtil;
}());
var AngularVersionUtil$1 = AngularVersionUtil.getInstance();

var BasicTypes;
(function (BasicTypes) {
    BasicTypes[BasicTypes["number"] = 0] = "number";
    BasicTypes[BasicTypes["boolean"] = 1] = "boolean";
    BasicTypes[BasicTypes["string"] = 2] = "string";
    BasicTypes[BasicTypes["object"] = 3] = "object";
    BasicTypes[BasicTypes["date"] = 4] = "date";
    BasicTypes[BasicTypes["function"] = 5] = "function";
})(BasicTypes || (BasicTypes = {}));
var BasicTypeScriptTypes;
(function (BasicTypeScriptTypes) {
    BasicTypeScriptTypes[BasicTypeScriptTypes["any"] = 0] = "any";
    BasicTypeScriptTypes[BasicTypeScriptTypes["void"] = 1] = "void";
})(BasicTypeScriptTypes || (BasicTypeScriptTypes = {}));
var BasicTypeUtil = /** @class */ (function () {
    function BasicTypeUtil() {
    }
    BasicTypeUtil.getInstance = function () {
        if (!BasicTypeUtil.instance) {
            BasicTypeUtil.instance = new BasicTypeUtil();
        }
        return BasicTypeUtil.instance;
    };
    /**
     * Checks if a given types is a basic javascript type
     * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isJavascriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Checks if a given type is a typescript type (That is not a javascript type)
     * https://www.typescriptlang.org/docs/handbook/basic-types.html
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isTypeScriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypeScriptTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Check if the type is a typescript or javascript type
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isKnownType = function (type) {
        return this.isJavascriptType(type) || this.isTypeScriptType(type);
    };
    /**
     * Returns a official documentation link to either the javascript or typescript type
     * @param type The type to check
     * @returns The documentation link or undefined if type not found
     */
    BasicTypeUtil.prototype.getTypeUrl = function (type) {
        if (this.isJavascriptType(type)) {
            return "https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/" + type;
        }
        if (this.isTypeScriptType(type)) {
            return "https://www.typescriptlang.org/docs/handbook/basic-types.html";
        }
        return undefined;
    };
    return BasicTypeUtil;
}());
var BasicTypeUtil$1 = BasicTypeUtil.getInstance();

var FunctionSignatureHelper = /** @class */ (function () {
    function FunctionSignatureHelper() {
    }
    FunctionSignatureHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg.function.length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg.function.map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path = _result.data.type;
                    if (_result.data.type === 'class') {
                        path = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    FunctionSignatureHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    FunctionSignatureHelper.prototype.helperFunc = function (context, method) {
        var _this = this;
        var args = [];
        if (method.args) {
            args = method.args
                .map(function (arg) {
                var _result = DependenciesEngine$1.find(arg.type);
                if (_result) {
                    if (_result.source === 'internal') {
                        var path = _result.data.type;
                        if (_result.data.type === 'class') {
                            path = 'classe';
                        }
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                    }
                    else {
                        var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                    }
                }
                else if (arg.dotDotDotToken) {
                    return "..." + arg.name + ": " + arg.type;
                }
                else if (arg.function) {
                    return _this.handleFunction(arg);
                }
                else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                    var path = BasicTypeUtil$1.getTypeUrl(arg.type);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                }
                else {
                    if (arg.type) {
                        return "" + arg.name + _this.getOptionalString(arg) + ": " + arg.type;
                    }
                    else {
                        return "" + arg.name + _this.getOptionalString(arg);
                    }
                }
            })
                .join(', ');
        }
        if (method.name) {
            return method.name + "(" + args + ")";
        }
        else {
            return "(" + args + ")";
        }
    };
    return FunctionSignatureHelper;
}());

var HasOwnHelper = /** @class */ (function () {
    function HasOwnHelper() {
    }
    HasOwnHelper.prototype.helperFunc = function (context, entity, key, options) {
        if (Object.hasOwnProperty.call(entity, key)) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return HasOwnHelper;
}());

var TRANSLATION_EN_US = {
    accessors: 'Accessors',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Class',
    classes: 'Classes',
    component: 'Component',
    components: 'Components',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentation coverage',
    declarations: 'Declarations',
    decorators: 'Decorators',
    'default-value': 'Default value',
    'defined-in': 'Defined in',
    dependencies: 'Dependencies',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Example',
    exports: 'Exports',
    extends: 'Extends',
    file: 'File',
    functions: 'Functions',
    'generated-using': 'Documentation generated using',
    'getting-started': 'Getting started',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element with directive',
    identifier: 'Identifier',
    implements: 'Implements',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Inherited from',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'License',
    lines: 'Lines',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Miscellaneous',
    module: 'Module',
    modules: 'Modules',
    name: 'Name',
    no: 'No',
    'no-graph': 'No graph available.',
    'no-iframe': 'Your browser does not support iframes.',
    'no-result-matching': 'No results matching',
    'no-svg': 'Your browser does not support SVG',
    optional: 'Optional',
    outputs: 'Outputs',
    overview: 'Overview',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefix',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'results matching',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Type to search',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Unamed property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Value',
    variables: 'Variables',
    yes: 'Yes',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_ES_ES = {
    accessors: 'Accesorios',
    arguments: 'Argumentos',
    bootstrap: 'Arranque',
    branches: 'Ramas',
    browse: 'Navegar',
    classe: 'Clase',
    classes: 'Clases',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Constructor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura de la documentación',
    declarations: 'Declaraciones',
    decorators: 'Decoradores',
    'default-value': 'Valor por defecto',
    'defined-in': 'Definido en',
    dependencies: 'Dependencias',
    description: 'Descripción',
    directive: 'Directiva',
    directives: 'Directivas',
    entrycomponents: 'Componentes de entrada',
    enumerations: 'Enumeraciones',
    enums: 'Enums',
    example: 'Ejemplo',
    exports: 'Exporta',
    extends: 'Extiende',
    file: 'Fichero',
    functions: 'Funciones',
    'generated-using': 'Documentación generada utilizando',
    'getting-started': 'Comenzando',
    guard: 'Guardia',
    guards: 'Guardias',
    hostbindings: 'Fijaciones de Host',
    hostlisteners: 'Escuchadores de Host',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento Html con directiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Índice',
    indexable: 'Indexable',
    'inherited-from': 'Heredado desde',
    injectable: 'Inyectable',
    injectables: 'Inyectables',
    inputs: 'Entradas',
    interceptors: 'Interceptores',
    interface: 'Interfaz',
    interfaces: 'Interfaces',
    legend: 'Leyenda',
    license: 'Licencia',
    lines: 'Líneas',
    metadata: 'Meta datos',
    methods: 'Métodos',
    miscellaneous: 'Miscelánea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nombre',
    no: 'No',
    'no-graph': 'No hay gráfica disponible.',
    'no-iframe': 'Tu navegador no soporta iframes.',
    'no-result-matching': 'No hay resultados que coincidan',
    'no-svg': 'Tu navegador no soporta SVG',
    optional: 'Opcional',
    outputs: 'Salidas',
    overview: 'Descripción general',
    parameters: 'Parámetros',
    'peer-dependencies': 'Dependencias entre pares',
    pipe: 'Tubería',
    pipes: 'Tuberías',
    prefix: 'Prefijo',
    properties: 'Propiedades',
    providers: 'Proveedores',
    pure: 'Puro',
    readme: 'Léeme',
    reset: 'Restablecer',
    'results-matching': 'comparación de resultados',
    returns: 'Devuelve',
    route: 'Ruta',
    routes: 'Rutas',
    schemas: 'Esquemas',
    'search-placeholder': 'Escribe para buscar',
    selector: 'Selector',
    signature: 'Firma',
    statements: 'Declaraciones',
    type: 'Tipo',
    'type-aliases': 'Alias de tipo',
    'type-parameters': 'Parámetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propiedad sin nombre',
    'unit-test-coverage': 'Cobertura de las pruebas unitarias',
    value: 'Valor',
    variables: 'Variables',
    yes: 'Si',
    zoomin: 'Ampliar',
    zoomout: 'Alejar'
};

var TRANSLATION_FR_FR = {
    accessors: 'Accesseurs',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Parcourir',
    classe: 'Class',
    classes: 'Classes',
    component: 'Composant',
    components: 'Composants',
    constructor: 'Constructeur',
    controllers: 'Contrôleurs',
    controller: 'Contrôleur',
    'coverage-page-title': 'Couverture de documentation',
    declarations: 'Déclarations',
    decorators: 'Décorateurs',
    'default-value': 'Valeur par défaut',
    'defined-in': 'Défini dans',
    dependencies: 'Dépendances',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: "Composants d'entrée",
    enumerations: 'Enumérations',
    enums: 'Enumérations',
    example: 'Example',
    exports: 'Exports',
    extends: 'Etend',
    file: 'Fichier',
    functions: 'Fonctions',
    'generated-using': 'Documentation générée avec',
    'getting-started': 'Démarrage',
    guard: 'Garde',
    guards: 'Gardes',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elément Html',
    'html-element-with-directive': 'Elément Html avec une directive',
    identifier: 'Identifiant',
    implements: 'Implémente',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Hérité de',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Entrées',
    interceptors: 'Intercepteurs',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Légende',
    license: 'License',
    lines: 'Lignes',
    metadata: 'Métadonnées',
    methods: 'Méthodes',
    miscellaneous: 'Divers',
    module: 'Module',
    modules: 'Modules',
    name: 'Nom',
    no: 'Non',
    'no-graph': 'Aucun graphique disponible.',
    'no-iframe': 'Votre navigateur ne supporte pas les iframes.',
    'no-result-matching': 'Aucun résultat matchant',
    'no-svg': 'Votre navigateur ne supporte pas le SVG',
    optional: 'Optionnel',
    outputs: 'Sorties',
    overview: "Vue d'ensemble",
    parameters: 'Paramètres',
    'peer-dependencies': 'Dépendances de pair',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Préfixe',
    properties: 'Propriétés',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Remise à zéro',
    'results-matching': 'résultats matchant',
    returns: 'Renvoie',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schémas',
    'search-placeholder': 'Saisissez un texte',
    selector: 'Sélecteur',
    signature: 'Signature',
    statements: 'Déclarations',
    type: 'Type',
    'type-aliases': 'Alias de type',
    'type-parameters': 'Paramètres de type',
    types: 'Types',
    'unamed-property': 'Propriété non nommée',
    'unit-test-coverage': 'Couverture de test unitaire',
    value: 'Valeur',
    variables: 'Variables',
    yes: 'Oui',
    zoomin: 'Zoom avant',
    zoomout: 'Zoom arrière'
};

var TRANSLATION_HU_HU = {
    accessors: 'Getter/setter metódusok',
    arguments: 'Argumentumok',
    bootstrap: 'Betöltés',
    branches: 'Branchek',
    browse: 'Böngészés',
    classe: 'Osztály',
    classes: 'Osztályok',
    component: 'Komponens',
    components: 'Komponensek',
    constructor: 'Konstruktor',
    controllers: 'Kontrollerek',
    controller: 'Kontroller',
    'coverage-page-title': 'Dokumentáció lefedettség',
    declarations: 'Deklarációk',
    decorators: 'Dekorátorok',
    'default-value': 'Alapértelmezett érték',
    'defined-in': 'Definíció helye:',
    dependencies: 'Függőségek',
    description: 'Leírás',
    directive: 'Direktíva',
    directives: 'Direktívák',
    entrycomponents: 'Entry komponensek',
    enumerations: 'Enumerációk',
    enums: 'Enumok',
    example: 'Példa',
    exports: 'Exportok',
    extends: 'Ősosztály',
    file: 'File',
    functions: 'Függvények',
    'generated-using': 'A dokumentációt generálta:',
    'getting-started': 'Bevezető',
    guard: 'Guard',
    guards: 'Guardok',
    hostbindings: 'HostBindingok',
    hostlisteners: 'HostListenerek',
    'html-element': 'Html elem',
    'html-element-with-directive': 'Html elem direktívával',
    identifier: 'Azonosító',
    implements: 'Implementált interfészek',
    imports: 'Importok',
    index: 'Tartalomjegyzék',
    indexable: 'Indexelhető',
    'inherited-from': 'Örökölve innen:',
    injectable: 'Injektálható',
    injectables: 'Injektálhatók',
    inputs: 'Bemenetek',
    interceptors: 'Interceptorok',
    interface: 'Interfész',
    interfaces: 'Interfészek',
    legend: 'Jelmagyarázat',
    license: 'Licenc',
    lines: 'Sorok',
    metadata: 'Metaadatok',
    methods: 'Metódusok',
    miscellaneous: 'Egyéb',
    module: 'Modul',
    modules: 'Modulok',
    name: 'Név',
    no: 'Nem',
    'no-graph': 'Grafikon nem elérhető.',
    'no-iframe': 'A böngészője nem támogatja az iframe-eket.',
    'no-result-matching': 'Nincs találat',
    'no-svg': 'A böngészője nem támogatja az SVG formátumot.',
    optional: 'Opcionális',
    outputs: 'Kimenetek',
    overview: 'Áttekintés',
    parameters: 'Paraméterek',
    'peer-dependencies': 'Peer függőségek',
    pipe: 'Pipe',
    pipes: 'Pipe-ok',
    prefix: 'Előtag',
    properties: 'Tagváltozók',
    providers: 'Providerek',
    pure: 'Pure',
    readme: 'README',
    reset: 'Visszaállít',
    'results-matching': 'találat',
    returns: 'Visszatérési érték',
    route: 'Útvonal',
    routes: 'Útvonalak',
    schemas: 'Sémák',
    'search-placeholder': 'Keresendő kifejezés',
    selector: 'Szelektor',
    signature: 'Aláírás',
    statements: 'Utasítások',
    type: 'Típus',
    'type-aliases': 'Típus álnév',
    'type-parameters': 'Típus paraméterek',
    types: 'Típusok',
    'unamed-property': 'Névtelen property',
    'unit-test-coverage': 'Unit teszt lefedettség',
    value: 'Érték',
    variables: 'Változók',
    yes: 'Igen',
    zoomin: 'Nagyítás',
    zoomout: 'Kicsinyítés'
};

var TRANSLATION_IT_IT = {
    accessors: 'Accessori',
    arguments: 'Argomenti',
    bootstrap: 'Bootstrap',
    branches: 'Rami',
    browse: 'Cerca',
    classe: 'Classe',
    classes: 'Classi',
    component: 'Componente',
    components: 'Componenti',
    constructor: 'Costruttore',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Copertura codice',
    declarations: 'Dichiarazioni',
    decorators: 'Decorators',
    'default-value': 'Valore predefinito',
    'defined-in': 'Definito in',
    dependencies: 'Dependencies',
    description: 'Descrizione',
    directive: 'Direttiva',
    directives: 'Direttive',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Esempio',
    exports: 'Exports',
    extends: 'Extends',
    file: 'File',
    functions: 'Funzioni',
    'generated-using': 'Documentazione generata usando',
    'getting-started': 'Iniziamo',
    guard: 'Guardia',
    guards: 'Guardie',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento html con direttive',
    identifier: 'Identificatore',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Indice',
    indexable: 'Indicizzabile',
    'inherited-from': 'ereditato da',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Input',
    interceptors: 'Interceptors',
    interface: 'Interfaccia',
    interfaces: 'Interfacce',
    legend: 'Legenda',
    license: 'Licenza',
    lines: 'Linee',
    metadata: 'Metadati',
    methods: 'Metodi',
    miscellaneous: 'Varie',
    module: 'Modulo',
    modules: 'Moduli',
    name: 'Nome',
    no: 'No',
    'no-graph': 'Grafico non disponibile.',
    'no-iframe': 'Il tuo browser non supporta iframe.',
    'no-result-matching': 'Nessun risultato corrispondente',
    'no-svg': 'Il tuo browser non supporta SVG',
    optional: 'Opzionale',
    outputs: 'Output',
    overview: 'Sommario',
    parameters: 'Parametri',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefisso',
    properties: 'Proprietà',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'corrispondenza',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Digita per avviare la ricerca',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Tipi',
    'unamed-property': 'Proprietà senza nome',
    'unit-test-coverage': 'Copertura unit test',
    value: 'Valori',
    variables: 'Variabili',
    yes: 'Si',
    zoomin: 'Ingrandisci',
    zoomout: 'Rimpocciolisci'
};

var TRANSLATION_JA_JP = {
    accessors: 'アクセサ',
    arguments: '引数',
    bootstrap: 'ブートストラップ',
    branches: 'ブランチ',
    browse: 'ブラウズ',
    classe: 'クラス',
    classes: 'クラス',
    component: 'コンポーネント',
    components: 'コンポーネント',
    constructor: 'コンストラクタ',
    controllers: 'コントローラー',
    controller: 'コントローラー',
    'coverage-page-title': 'カバレッジ',
    declarations: '宣言',
    decorators: 'デコレーター',
    'default-value': '初期値',
    'defined-in': 'Defined in',
    dependencies: '依存関係',
    description: '説明',
    directive: 'ディレクティブ',
    directives: 'ディレクティブ',
    entrycomponents: 'エントリーコンポーネント',
    enumerations: '列挙型',
    enums: 'Enums',
    example: '例',
    exports: 'エクスポート',
    extends: '継承',
    file: 'ファイル',
    functions: '関数',
    'generated-using': 'このドキュメントは以下を使用して生成されています',
    'getting-started': 'はじめに',
    guard: 'ガード',
    guards: 'ガード',
    hostbindings: 'ホストバインディング',
    hostlisteners: 'ホストリスナー',
    'html-element': 'Html要素',
    'html-element-with-directive': 'ディレクティブHtml要素',
    identifier: '識別子',
    implements: '実装',
    imports: 'インポート',
    index: '索引',
    indexable: 'インデクサブル',
    'inherited-from': 'Inherited from',
    injectable: 'インジェクタブル',
    injectables: 'インジェクタブル',
    inputs: '入力',
    interceptors: 'インターセプター',
    interface: 'インターフェイス',
    interfaces: 'インターフェイス',
    legend: '凡例',
    license: 'ライセンス',
    lines: '行数',
    metadata: 'メタデータ',
    methods: 'メソッド',
    miscellaneous: 'その他',
    module: 'モジュール',
    modules: 'モジュール',
    name: '名前',
    no: 'いいえ',
    'no-graph': '使用できるグラフがありません',
    'no-iframe': 'ブラウザがiframeを対応していません',
    'no-result-matching': '見つかりませんでした',
    'no-svg': 'ブラウザがSVGに対応してません',
    optional: 'オプション',
    outputs: '出力',
    overview: '概要',
    parameters: 'パラメータ',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'パイプ',
    pipes: 'パイプ',
    prefix: '接頭辞',
    properties: 'プロパティ',
    providers: 'プロバイダー',
    pure: 'Pure',
    readme: 'README',
    reset: 'リセット',
    'results-matching': '件の結果が一致しました',
    returns: '戻り値',
    route: 'ルート',
    routes: 'ルート',
    schemas: 'スキーマ',
    'search-placeholder': '入力して検索',
    selector: 'セレクタ',
    signature: 'シグネチャ',
    statements: '文',
    type: '型',
    'type-aliases': 'タイプエイリアス',
    'type-parameters': '型パラメーター',
    types: '型',
    'unamed-property': '匿名プロパティ',
    'unit-test-coverage': 'ユニットテストカバレッジ',
    value: '値',
    variables: '変数',
    yes: 'はい',
    zoomin: '拡大',
    zoomout: '縮小'
};

var TRANSLATION_PT_BR = {
    accessors: 'Acessores',
    arguments: 'Argumentos',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Navegar',
    classe: 'Classe',
    classes: 'Classes',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Construtor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura da documentação',
    declarations: 'Declarações',
    decorators: 'Decoradores',
    'default-value': 'Valor padrão',
    'defined-in': 'Definido em',
    dependencies: 'Dependências',
    description: 'Descrição',
    directive: 'Diretiva',
    directives: 'Diretivas',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerações',
    enums: 'Enums',
    example: 'Exemplo',
    exports: 'Exports',
    extends: 'Extende',
    file: 'Arquivo',
    functions: 'Funções',
    'generated-using': 'Documentação gerada usando',
    'getting-started': 'Começando',
    guard: 'Guarda',
    guards: 'Guardas',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento HTML',
    'html-element-with-directive': 'Elemento HTML com diretiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexável',
    'inherited-from': 'Herdado de',
    injectable: 'Injetável',
    injectables: 'Injetáveis',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'Licença',
    lines: 'Linhas',
    metadata: 'Metadata',
    methods: 'Métodos',
    miscellaneous: 'Miscelânea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nome',
    no: 'Não',
    'no-graph': 'Sem gráfico disponível.',
    'no-iframe': 'Seu browser não tem suporte a iframes.',
    'no-result-matching': 'Nenhum resultado correspondente',
    'no-svg': 'Seu browser não tem suporte a SVG',
    optional: 'Opcional',
    outputs: 'Outputs',
    overview: 'Visão geral',
    parameters: 'Parâmetros',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefixo',
    properties: 'Propriedades',
    providers: 'Providers',
    pure: 'Puro',
    readme: 'README',
    reset: 'Resetar',
    'results-matching': 'resultados correspondentes',
    returns: 'Retorna',
    route: 'Rota',
    routes: 'Rotas',
    schemas: 'Esquemas',
    'search-placeholder': 'Digite para pesquisar',
    selector: 'Seletor',
    signature: 'Assinatura',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Aliases de tipo',
    'type-parameters': 'Parâmetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propriedade não-nomeada',
    'unit-test-coverage': 'Cobertua de teste unitário',
    value: 'Valor',
    variables: 'Variáveis',
    yes: 'Sim',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_ZH_CN = {
    accessors: '存取器',
    arguments: 'Arguments',
    bootstrap: '根组件',
    branches: '分支',
    browse: '查看',
    classe: '类',
    classes: '类列表',
    component: '组件',
    components: '组件列表',
    constructor: '构造方法',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': '文档概览',
    declarations: '可声明对象列表',
    decorators: '装饰器列表',
    'default-value': '缺省值',
    'defined-in': '被定义在',
    dependencies: '依赖项',
    description: '描述',
    directive: '指令',
    directives: '指令列表',
    entrycomponents: '入口组件列表',
    enumerations: '列举',
    enums: '枚举列表',
    example: '例子',
    exports: '导出',
    extends: '继承',
    file: '文件',
    functions: '函数',
    'generated-using': '文档生成使用',
    'getting-started': '入门指南',
    guard: '路由守卫',
    guards: '路由守卫列表',
    hostbindings: '宿主绑定',
    hostlisteners: '宿主监听',
    'html-element': 'Html 元素',
    'html-element-with-directive': '带指令的Html元素',
    identifier: '标识符',
    implements: '实现',
    imports: '引入',
    index: '索引',
    indexable: 'Indexable',
    'inherited-from': '继承自',
    injectable: '可注入的',
    injectables: '可注入的',
    inputs: '输入属性',
    interceptors: '拦截器',
    interface: '接口',
    interfaces: '接口',
    legend: '图例',
    license: '许可协议',
    lines: 'Lines',
    metadata: '元数据',
    methods: '方法',
    miscellaneous: '其他',
    module: '模块',
    modules: '模块列表',
    name: '名称',
    no: '否',
    'no-graph': '无数据显示',
    'no-iframe': '你的浏览器不支持iframes',
    'no-result-matching': '无匹配的结果',
    'no-svg': '你的浏览器不支持SVG',
    optional: '可选的',
    outputs: '输出属性',
    overview: '概述',
    parameters: '参数列表',
    'peer-dependencies': '同级依赖',
    pipe: '管道',
    pipes: '管道列表',
    prefix: '字首',
    properties: '属性列表',
    providers: '提供商列表',
    pure: 'Pure',
    readme: '手册',
    reset: '重置',
    'results-matching': '匹配的结果',
    returns: '返回',
    route: '路由',
    routes: '路由列表',
    schemas: '模式',
    'search-placeholder': '请输入查询关键字',
    selector: '选择器',
    signature: '签名',
    statements: '注释',
    type: '类型',
    'type-aliases': '类型别名',
    'type-parameters': '类型参数',
    types: '类型',
    'unamed-property': '未命名属性',
    'unit-test-coverage': '单元测试概览',
    value: '值',
    variables: '变量',
    yes: '是',
    zoomin: '放大',
    zoomout: '缩小'
};

var TRANSLATION_NL_NL = {
    accessors: 'Accessors',
    arguments: 'Argumenten',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Klasse',
    classes: 'Klassen',
    component: 'Component',
    components: 'Componenten',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentatie coverage',
    declarations: 'Declaraties',
    decorators: 'Decorators',
    'default-value': 'Default waarde',
    'defined-in': 'Gedefinieerd in',
    dependencies: 'Dependencies',
    description: 'Omschrijving',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Voorbeeld',
    exports: 'Exports',
    extends: 'Extends',
    file: 'Bestand',
    functions: 'Functies',
    'generated-using': 'Documentatie gegenereed met',
    'getting-started': 'Aan de slag',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element met directive',
    identifier: 'Identifier',
    implements: 'Implementeert',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexeerbaar',
    'inherited-from': 'Inherited van',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legenda',
    license: 'Licentie',
    lines: 'Regels',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Diversen',
    module: 'Module',
    modules: 'Modules',
    name: 'Naam',
    no: 'Nee',
    'no-graph': 'Geen diagram beschikbaar.',
    'no-iframe': 'Uw browser ondersteund geen iframes.',
    'no-result-matching': 'Geen overeenkomende resultaten',
    'no-svg': 'Uw browser ondersteund geen SVG',
    optional: 'Optioneel',
    outputs: 'Outputs',
    overview: 'Overzicht',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Voorvoegsel',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Puur',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'overeenkomende resultaten',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: "Schema's",
    'search-placeholder': 'Type om te zoeken',
    selector: 'Selector',
    signature: 'Handtekening',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliassen',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Naamloze property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Waarde',
    variables: 'Variabelen',
    yes: 'Ja',
    zoomin: 'Zoom in',
    zoomout: 'Zoom uit'
};

var I18nEngine = /** @class */ (function () {
    function I18nEngine() {
        this.availablesLanguages = {
            'en-US': 'en-US',
            'es-ES': 'es-ES',
            'fr-FR': 'fr-FR',
            'hu-HU': 'hu-HU',
            'it-IT': 'it-IT',
            'ja-JP': 'ja-JP',
            'nl-NL': 'nl-NL',
            'pt-BR': 'pt-BR',
            'zh-CN': 'zh-CN'
        };
        this.fallbackLanguage = 'en-US';
    }
    I18nEngine.getInstance = function () {
        if (!I18nEngine.instance) {
            I18nEngine.instance = new I18nEngine();
        }
        return I18nEngine.instance;
    };
    I18nEngine.prototype.init = function (language) {
        i18next.init({
            lng: language,
            fallbackLng: this.fallbackLanguage
        });
        i18next.addResources('en-US', 'translation', TRANSLATION_EN_US);
        i18next.addResources('es-ES', 'translation', TRANSLATION_ES_ES);
        i18next.addResources('fr-FR', 'translation', TRANSLATION_FR_FR);
        i18next.addResources('hu-HU', 'translation', TRANSLATION_HU_HU);
        i18next.addResources('it-IT', 'translation', TRANSLATION_IT_IT);
        i18next.addResources('ja-JP', 'translation', TRANSLATION_JA_JP);
        i18next.addResources('nl-NL', 'translation', TRANSLATION_NL_NL);
        i18next.addResources('pt-BR', 'translation', TRANSLATION_PT_BR);
        i18next.addResources('zh-CN', 'translation', TRANSLATION_ZH_CN);
    };
    I18nEngine.prototype.translate = function (key) {
        return i18next.t(key);
    };
    I18nEngine.prototype.supportLanguage = function (language) {
        return typeof this.availablesLanguages[language] !== 'undefined';
    };
    return I18nEngine;
}());
var I18nEngine$1 = I18nEngine.getInstance();

var I18nHelper = /** @class */ (function () {
    function I18nHelper() {
    }
    I18nHelper.prototype.helperFunc = function (context, i18n_key, options) {
        var result = I18nEngine$1.translate(i18n_key);
        return result;
    };
    return I18nHelper;
}());

var IfStringHelper = /** @class */ (function () {
    function IfStringHelper() {
    }
    IfStringHelper.prototype.helperFunc = function (context, a, options) {
        if (typeof a === 'string') {
            return options.fn(context);
        }
        return options.inverse(context);
    };
    return IfStringHelper;
}());

var IndexableSignatureHelper = /** @class */ (function () {
    function IndexableSignatureHelper() {
    }
    IndexableSignatureHelper.prototype.helperFunc = function (context, method) {
        var args = method.args.map(function (arg) { return arg.name + ": " + arg.type; }).join(', ');
        if (method.name) {
            return method.name + "[" + args + "]";
        }
        else {
            return "[" + args + "]";
        }
    };
    return IndexableSignatureHelper;
}());

var IsInitialTabHelper = /** @class */ (function () {
    function IsInitialTabHelper() {
    }
    IsInitialTabHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        return tabs[0].id === tabId ? options.fn(context) : options.inverse(context);
    };
    return IsInitialTabHelper;
}());

var IsNotToggleHelper = /** @class */ (function () {
    function IsNotToggleHelper() {
    }
    IsNotToggleHelper.prototype.helperFunc = function (context, type, options) {
        var result = Configuration$1.mainData.toggleMenuItems.indexOf(type);
        if (Configuration$1.mainData.toggleMenuItems.indexOf('all') !== -1) {
            return options.inverse(context);
        }
        else if (result !== -1) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return IsNotToggleHelper;
}());

var IsTabEnabledHelper = /** @class */ (function () {
    function IsTabEnabledHelper() {
    }
    IsTabEnabledHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        var isTabEnabled = -1 !== _.findIndex(tabs, { id: tabId });
        return isTabEnabled ? options.fn(context) : options.inverse(context);
    };
    return IsTabEnabledHelper;
}());

var JsdocCodeExampleHelper = /** @class */ (function () {
    function JsdocCodeExampleHelper() {
    }
    JsdocCodeExampleHelper.prototype.cleanTag = function (comment) {
        if (comment.charAt(0) === '*') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.charAt(0) === ' ') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.indexOf('<p>') === 0) {
            comment = comment.substring(3, comment.length);
        }
        if (comment.substr(-1) === '\n') {
            comment = comment.substring(0, comment.length - 1);
        }
        if (comment.substr(-4) === '</p>') {
            comment = comment.substring(0, comment.length - 4);
        }
        return comment;
    };
    JsdocCodeExampleHelper.prototype.getHtmlEntities = function (str) {
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;');
    };
    JsdocCodeExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        var type = 'html';
        if (options.hash.type) {
            type = options.hash.type;
        }
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        if (jsdocTags[i].comment.indexOf('<caption>') !== -1) {
                            tag.comment = jsdocTags[i].comment
                                .replace(/<caption>/g, '<b><i>')
                                .replace(/\/caption>/g, '/b></i>');
                        }
                        else {
                            tag.comment =
                                "<pre class=\"line-numbers\"><code class=\"language-" + type + "\">" +
                                    this.getHtmlEntities(this.cleanTag(jsdocTags[i].comment)) +
                                    "</code></pre>";
                        }
                        tags.push(tag);
                    }
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocCodeExampleHelper;
}());

var JsdocDefaultHelper = /** @class */ (function () {
    function JsdocDefaultHelper() {
    }
    JsdocDefaultHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        if (jsdocTags) {
            var i = 0;
            var len = jsdocTags.length;
            var tag = {};
            var defaultValue = false;
            for (i; i < len; i++) {
                if (jsdocTags[i].tagName) {
                    if (jsdocTags[i].tagName.text === 'default') {
                        defaultValue = true;
                        if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                            tag.type = jsdocTags[i].typeExpression.type.name.text;
                        }
                        if (jsdocTags[i].comment) {
                            tag.comment = jsdocTags[i].comment;
                        }
                        if (jsdocTags[i].name) {
                            tag.name = jsdocTags[i].name.text;
                        }
                    }
                }
            }
            if (defaultValue) {
                context.tag = tag;
                return options.fn(context);
            }
        }
    };
    return JsdocDefaultHelper;
}());

var JsdocExampleHelper = /** @class */ (function () {
    function JsdocExampleHelper() {
    }
    JsdocExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment
                            .replace(/<caption>/g, '<b><i>')
                            .replace(/\/caption>/g, '/b></i>');
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocExampleHelper;
}());

var JsdocParamsValidHelper = /** @class */ (function () {
    function JsdocParamsValidHelper() {
    }
    JsdocParamsValidHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var valid = false;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    valid = true;
                }
            }
        }
        if (valid) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return JsdocParamsValidHelper;
}());

var JsdocParamsHelper = /** @class */ (function () {
    function JsdocParamsHelper() {
    }
    JsdocParamsHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    var tag = {};
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.kind) {
                        tag.type = kindToType(jsdocTags[i].typeExpression.type.kind);
                    }
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                        tag.type = jsdocTags[i].typeExpression.type.name.text;
                    }
                    else {
                        tag.type = jsdocTags[i].type;
                    }
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment;
                    }
                    if (jsdocTags[i].defaultValue) {
                        tag.defaultValue = jsdocTags[i].defaultValue;
                    }
                    if (jsdocTags[i].name) {
                        if (jsdocTags[i].name.text) {
                            tag.name = jsdocTags[i].name.text;
                        }
                        else {
                            tag.name = jsdocTags[i].name;
                        }
                    }
                    if (jsdocTags[i].optional) {
                        tag.optional = true;
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length >= 1) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocParamsHelper;
}());

var JsdocReturnsCommentHelper = /** @class */ (function () {
    function JsdocReturnsCommentHelper() {
    }
    JsdocReturnsCommentHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var result;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'returns' ||
                    jsdocTags[i].tagName.text === 'return') {
                    result = jsdocTags[i].comment;
                    break;
                }
            }
        }
        return result;
    };
    return JsdocReturnsCommentHelper;
}());

var LinkTypeHelper = /** @class */ (function () {
    function LinkTypeHelper() {
    }
    LinkTypeHelper.prototype.helperFunc = function (context, name, options) {
        var _result = DependenciesEngine$1.find(name);
        var angularDocPrefix = AngularVersionUtil$1.prefixOfficialDoc(Configuration$1.mainData.angularVersion);
        if (_result) {
            context.type = {
                raw: name
            };
            if (_result.source === 'internal') {
                if (_result.data.type === 'class') {
                    _result.data.type = 'classe';
                }
                context.type.href = '../' + _result.data.type + 's/' + _result.data.name + '.html';
                if (_result.data.type === 'miscellaneous' ||
                    (_result.data.ctype && _result.data.ctype === 'miscellaneous')) {
                    var mainpage = '';
                    switch (_result.data.subtype) {
                        case 'enum':
                            mainpage = 'enumerations';
                            break;
                        case 'function':
                            mainpage = 'functions';
                            break;
                        case 'typealias':
                            mainpage = 'typealiases';
                            break;
                        case 'variable':
                            mainpage = 'variables';
                    }
                    context.type.href =
                        '../' + _result.data.ctype + '/' + mainpage + '.html#' + _result.data.name;
                }
                context.type.target = '_self';
            }
            else {
                context.type.href = "https://" + angularDocPrefix + "angular.io/" + _result.data.path;
                context.type.target = '_blank';
            }
            return options.fn(context);
        }
        else if (BasicTypeUtil$1.isKnownType(name)) {
            context.type = {
                raw: name
            };
            context.type.target = '_blank';
            context.type.href = BasicTypeUtil$1.getTypeUrl(name);
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return LinkTypeHelper;
}());

var ModifIconHelper = /** @class */ (function () {
    function ModifIconHelper() {
    }
    ModifIconHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'lock'; // private
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'lock'; // protected
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'reset'; // static
                break;
            case Ast.SyntaxKind.ExportKeyword:
                _kindText = 'export'; // export
                break;
            default:
                _kindText = 'reset';
                break;
        }
        return _kindText;
    };
    return ModifIconHelper;
}());

var ModifKindHelper = /** @class */ (function () {
    function ModifKindHelper() {
    }
    /**
     * Transform SyntaxKind into string
     * @param  {any}           context Handlebars context
     * @param  {SyntaxKind[]} kind  SyntaxKind concatenated
     * @return {string}                Parsed string
     */
    ModifKindHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'Private';
                break;
            case Ast.SyntaxKind.ReadonlyKeyword:
                _kindText = 'Readonly';
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'Protected';
                break;
            case Ast.SyntaxKind.PublicKeyword:
                _kindText = 'Public';
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'Static';
                break;
            case Ast.SyntaxKind.AsyncKeyword:
                _kindText = 'Async';
                break;
            case Ast.SyntaxKind.AbstractKeyword:
                _kindText = 'Abstract';
                break;
        }
        return new Handlebars.SafeString(_kindText);
    };
    return ModifKindHelper;
}());

var ObjectLengthHelper = /** @class */ (function () {
    function ObjectLengthHelper() {
    }
    ObjectLengthHelper.prototype.helperFunc = function (context, obj, operator, length) {
        var len = arguments.length - 1;
        var options = arguments[len];
        if (typeof obj !== 'object') {
            return options.inverse(context);
        }
        var size = 0, key;
        for (key in obj) {
            if (obj.hasOwnProperty(key)) {
                size++;
            }
        }
        var result;
        switch (operator) {
            case '===':
                result = size === length;
                break;
            case '!==':
                result = size !== length;
                break;
            case '>':
                result = size > length;
                break;
            default: {
                throw new Error('helper {{objectLength}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return ObjectLengthHelper;
}());

var ObjectHelper = /** @class */ (function () {
    function ObjectHelper() {
    }
    ObjectHelper.prototype.helperFunc = function (context, text) {
        text = JSON.stringify(text);
        text = text.replace(/{"/, '{<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/,"/, ',<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/}$/, '<br>}');
        return new Handlebars.SafeString(text);
    };
    return ObjectHelper;
}());

var OneParameterHasHelper = /** @class */ (function () {
    function OneParameterHasHelper() {
    }
    OneParameterHasHelper.prototype.helperFunc = function (context, tags, typeToCheck) {
        var result = false;
        var len = arguments.length - 1;
        var options = arguments[len];
        var i = 0, leng = tags.length;
        for (i; i < leng; i++) {
            if (typeof tags[i][typeToCheck] !== 'undefined' && tags[i][typeToCheck] !== '') {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return OneParameterHasHelper;
}());

var OrLengthHelper = /** @class */ (function () {
    function OrLengthHelper() {
    }
    OrLengthHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (typeof arguments[i] !== 'undefined') {
                if (Object.keys(arguments[i]).length > 0) {
                    return options.fn(context);
                }
            }
        }
        return options.inverse(context);
    };
    return OrLengthHelper;
}());

var OrHelper = /** @class */ (function () {
    function OrHelper() {
    }
    OrHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (arguments[i]) {
                return options.fn(context);
            }
        }
        return options.inverse(context);
    };
    return OrHelper;
}());

var ParseDescriptionHelper = /** @class */ (function () {
    function ParseDescriptionHelper() {
    }
    ParseDescriptionHelper.prototype.helperFunc = function (context, description, depth) {
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExp;
        var matches;
        var previousString;
        tagRegExp = description.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        var processTheLink = function (originalDescription, matchedTag, leadingText) {
            var leading = extractLeadingText(originalDescription, matchedTag.completeTag);
            var split;
            var resultInCompodoc;
            var newLink;
            var rootPath;
            var stringtoReplace;
            var anchor = '';
            var label;
            var pageName;
            split = splitLinkText(matchedTag.text);
            if (typeof split.linkText !== 'undefined') {
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(split.target);
            }
            else {
                var info = matchedTag.text;
                if (matchedTag.text.indexOf('#') !== -1) {
                    anchor = matchedTag.text.substr(matchedTag.text.indexOf('#'), matchedTag.text.length);
                    info = matchedTag.text.substr(0, matchedTag.text.indexOf('#'));
                }
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(info);
            }
            if (resultInCompodoc) {
                label = resultInCompodoc.name;
                pageName = resultInCompodoc.name;
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                if (resultInCompodoc.type === 'class') {
                    resultInCompodoc.type = 'classe';
                }
                else if (resultInCompodoc.type === 'miscellaneous' ||
                    (resultInCompodoc.ctype && resultInCompodoc.ctype === 'miscellaneous')) {
                    resultInCompodoc.type = 'miscellaneou'; // Not a typo, it is for matching other single types : component, module etc
                    label = resultInCompodoc.name;
                    anchor = '#' + resultInCompodoc.name;
                    if (resultInCompodoc.subtype === 'enum') {
                        pageName = 'enumerations';
                    }
                    else if (resultInCompodoc.subtype === 'function') {
                        pageName = 'functions';
                    }
                    else if (resultInCompodoc.subtype === 'typealias') {
                        pageName = 'typealiases';
                    }
                    else if (resultInCompodoc.subtype === 'variable') {
                        pageName = 'variables';
                    }
                }
                rootPath = '';
                switch (depth) {
                    case 0:
                        rootPath = './';
                        break;
                    case 1:
                    case 2:
                    case 3:
                    case 4:
                    case 5:
                        rootPath = '../'.repeat(depth);
                        break;
                }
                if (leading.leadingText !== undefined) {
                    label = leading.leadingText;
                }
                if (typeof split.linkText !== 'undefined') {
                    label = split.linkText;
                }
                newLink = "<a href=\"" + rootPath + resultInCompodoc.type + "s/" + pageName + ".html" + anchor + "\">" + label + "</a>";
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.linkText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && leading && typeof leading.leadingText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + leading.leadingText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText === 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.target + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else {
                return originalDescription;
            }
        };
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(description, matchedTag, linkText);
            }
            else {
                return replacer(description, matchedTag);
            }
        }
        // Clean description for marked a tag parsed too early
        if (description.indexOf('href=') !== -1) {
            var insideMarkedATagResults = description.match(/<a [^>]+>([^<]+)<\/a>/g);
            if (insideMarkedATagResults && insideMarkedATagResults.length > 0) {
                for (var i = 0; i < insideMarkedATagResults.length; i++) {
                    var markedATagRegExp = new RegExp('<a [^>]+>([^<]+)</a>', 'gm');
                    var parsedATag = markedATagRegExp.exec(description);
                    if (parsedATag && parsedATag.length === 2) {
                        var insideMarkedATag = parsedATag[1];
                        description = description.replace("{@link <a href=\"" + encodeURI(insideMarkedATag) + "\">" + insideMarkedATag + "</a>", "{@link " + insideMarkedATag);
                    }
                }
            }
        }
        do {
            matches = tagRegExp.exec(description);
            // Did we have {@link ?
            if (matches) {
                previousString = description;
                if (matches.length === 2) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== description);
        return description;
    };
    return ParseDescriptionHelper;
}());

var RelativeURLHelper = /** @class */ (function () {
    function RelativeURLHelper() {
    }
    RelativeURLHelper.prototype.helperFunc = function (context, currentDepth, options) {
        switch (currentDepth) {
            case 0:
                return './';
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
                return '../'.repeat(currentDepth);
        }
        return '';
    };
    return RelativeURLHelper;
}());

var ShortURLHelper = /** @class */ (function () {
    function ShortURLHelper() {
    }
    ShortURLHelper.prototype.helperFunc = function (context, url, options) {
        var newUrl = url;
        var firstIndexOfSlash = newUrl.indexOf('/');
        var lastIndexOfSlash = newUrl.lastIndexOf('/');
        if (firstIndexOfSlash !== -1 || lastIndexOfSlash !== -1) {
            newUrl =
                newUrl.substr(0, firstIndexOfSlash + 1) +
                    '...' +
                    newUrl.substr(lastIndexOfSlash, newUrl.length);
        }
        return newUrl;
    };
    return ShortURLHelper;
}());

var StripURLHelper = /** @class */ (function () {
    function StripURLHelper() {
    }
    StripURLHelper.prototype.helperFunc = function (context, prefix, url, options) {
        return prefix + url.split("/").pop();
    };
    return StripURLHelper;
}());

var HtmlEngineHelpers = /** @class */ (function () {
    function HtmlEngineHelpers() {
    }
    HtmlEngineHelpers.prototype.registerHelpers = function (bars) {
        this.registerHelper(bars, 'compare', new CompareHelper());
        this.registerHelper(bars, 'or', new OrHelper());
        this.registerHelper(bars, 'functionSignature', new FunctionSignatureHelper());
        this.registerHelper(bars, 'isNotToggle', new IsNotToggleHelper());
        this.registerHelper(bars, 'isInitialTab', new IsInitialTabHelper());
        this.registerHelper(bars, 'isTabEnabled', new IsTabEnabledHelper());
        this.registerHelper(bars, 'ifString', new IfStringHelper());
        this.registerHelper(bars, 'orLength', new OrLengthHelper());
        this.registerHelper(bars, 'filterAngular2Modules', new FilterAngular2ModulesHelper());
        this.registerHelper(bars, 'debug', new DebugHelper());
        this.registerHelper(bars, 'breaklines', new BreakLinesHelper(bars));
        this.registerHelper(bars, 'clean-paragraph', new CleanParagraphHelper());
        this.registerHelper(bars, 'escapeSimpleQuote', new EscapeSimpleQuoteHelper());
        this.registerHelper(bars, 'breakComma', new BreakCommaHelper(bars));
        this.registerHelper(bars, 'modifKind', new ModifKindHelper());
        this.registerHelper(bars, 'modifIcon', new ModifIconHelper());
        this.registerHelper(bars, 'relativeURL', new RelativeURLHelper());
        this.registerHelper(bars, 'jsdoc-returns-comment', new JsdocReturnsCommentHelper());
        this.registerHelper(bars, 'jsdoc-code-example', new JsdocCodeExampleHelper());
        this.registerHelper(bars, 'jsdoc-example', new JsdocExampleHelper());
        this.registerHelper(bars, 'jsdoc-params', new JsdocParamsHelper());
        this.registerHelper(bars, 'jsdoc-params-valid', new JsdocParamsValidHelper());
        this.registerHelper(bars, 'jsdoc-default', new JsdocDefaultHelper());
        this.registerHelper(bars, 'linkType', new LinkTypeHelper());
        this.registerHelper(bars, 'indexableSignature', new IndexableSignatureHelper());
        this.registerHelper(bars, 'object', new ObjectHelper());
        this.registerHelper(bars, 'objectLength', new ObjectLengthHelper());
        this.registerHelper(bars, 'parseDescription', new ParseDescriptionHelper());
        this.registerHelper(bars, 'one-parameter-has', new OneParameterHasHelper());
        this.registerHelper(bars, 'element-alone', new ElementAloneHelper());
        this.registerHelper(bars, 'hasOwn', new HasOwnHelper());
        this.registerHelper(bars, 'short-url', new ShortURLHelper());
        this.registerHelper(bars, 'strip-url', new StripURLHelper());
        this.registerHelper(bars, 't', new I18nHelper());
    };
    HtmlEngineHelpers.prototype.registerHelper = function (bars, key, helper) {
        Handlebars.registerHelper(key, function () {
            // tslint:disable-next-line:no-invalid-this
            return helper.helperFunc.apply(helper, [this].concat(_.slice(arguments)));
        });
    };
    return HtmlEngineHelpers;
}());

var HtmlEngine = /** @class */ (function () {
    function HtmlEngine() {
        this.cache = {};
        var helper = new HtmlEngineHelpers();
        helper.registerHelpers(Handlebars);
    }
    HtmlEngine.getInstance = function () {
        if (!HtmlEngine.instance) {
            HtmlEngine.instance = new HtmlEngine();
        }
        return HtmlEngine.instance;
    };
    HtmlEngine.prototype.init = function (templatePath) {
        var _this = this;
        var partials = [
            'overview',
            'markdown',
            'modules',
            'module',
            'components',
            'component',
            'controller',
            'component-detail',
            'directives',
            'directive',
            'injectables',
            'injectable',
            'interceptor',
            'guard',
            'pipes',
            'pipe',
            'classes',
            'class',
            'interface',
            'routes',
            'index',
            'index-misc',
            'search-results',
            'search-input',
            'link-type',
            'block-method',
            'block-enum',
            'block-property',
            'block-index',
            'block-constructor',
            'block-typealias',
            'block-accessors',
            'block-input',
            'block-output',
            'coverage-report',
            'unit-test-report',
            'miscellaneous-functions',
            'miscellaneous-variables',
            'miscellaneous-typealiases',
            'miscellaneous-enumerations',
            'additional-page',
            'package-dependencies'
        ];
        if (templatePath) {
            if (FileEngine$1.existsSync(path.resolve(process.cwd() + path.sep + templatePath)) ===
                false) {
                logger.warn('Template path specificed but does not exist...using default templates');
            }
        }
        return Promise.all(partials.map(function (partial) {
            var partialPath = _this.determineTemplatePath(templatePath, 'partials/' + partial + '.hbs');
            return FileEngine$1.get(partialPath).then(function (data) {
                return Handlebars.registerPartial(partial, data);
            });
        }))
            .then(function () {
            var pagePath = _this.determineTemplatePath(templatePath, 'page.hbs');
            return FileEngine$1.get(pagePath).then(function (data) {
                _this.cache.page = data;
                _this.compiledPage = Handlebars.compile(_this.cache.page, {
                    preventIndent: true,
                    strict: true
                });
            });
        })
            .then(function () {
            var menuPath = _this.determineTemplatePath(templatePath, 'partials/menu.hbs');
            return FileEngine$1.get(menuPath).then(function (menuTemplate) {
                _this.precompiledMenu = Handlebars.compile(menuTemplate, {
                    preventIndent: true,
                    strict: true
                });
            });
        });
    };
    HtmlEngine.prototype.renderMenu = function (templatePath, data) {
        var menuPath = this.determineTemplatePath(templatePath, 'partials/menu.hbs');
        return FileEngine$1.get(menuPath).then(function (menuTemplate) {
            data.menu = 'normal';
            return Handlebars.compile(menuTemplate, {
                preventIndent: true,
                strict: true
            })(__assign({}, data));
        });
    };
    HtmlEngine.prototype.render = function (mainData, page) {
        var o = mainData;
        Object.assign(o, page);
        // let mem = process.memoryUsage();
        // console.log(`heapTotal: ${mem.heapTotal} | heapUsed: ${mem.heapUsed}`);
        return this.compiledPage({
            data: o
        });
    };
    HtmlEngine.prototype.determineTemplatePath = function (templatePath, filePath) {
        var outPath = path.resolve(__dirname + '/../src/templates/' + filePath);
        if (templatePath) {
            var testPath = path.resolve(process.cwd() + path.sep + templatePath + path.sep + filePath);
            outPath = FileEngine$1.existsSync(testPath) ? testPath : outPath;
        }
        return outPath;
    };
    HtmlEngine.prototype.generateCoverageBadge = function (outputFolder, label, coverageData) {
        return FileEngine$1.get(path.resolve(__dirname + '/../src/templates/partials/coverage-badge.hbs')).then(function (data) {
            var template = Handlebars.compile(data);
            coverageData.label = label;
            var result = template({
                data: coverageData
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/images/coverage-badge-' + label + '.svg', result).catch(function (err) {
                logger.error('Error during coverage badge ' + label + ' file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during coverage badge generation'); });
    };
    return HtmlEngine;
}());
var HtmlEngine$1 = HtmlEngine.getInstance();

var MarkdownEngine = /** @class */ (function () {
    function MarkdownEngine() {
        var _this = this;
        /**
         * List of markdown files without .md extension
         */
        this.markdownFiles = ['README', 'CHANGELOG', 'LICENSE', 'CONTRIBUTING', 'TODO'];
        this.markedInstance = require('marked');
        var renderer = new this.markedInstance.Renderer();
        renderer.code = function (code, language) {
            var highlighted = code;
            if (!language) {
                language = 'none';
            }
            highlighted = _this.escape(code);
            return "<div><pre class=\"line-numbers\"><code class=\"language-" + language + "\">" + highlighted + "</code></pre></div>";
        };
        renderer.table = function (header, body) {
            return ('<table class="table table-bordered compodoc-table">\n' +
                '<thead>\n' +
                header +
                '</thead>\n' +
                '<tbody>\n' +
                body +
                '</tbody>\n' +
                '</table>\n');
        };
        renderer.image = function (href, title, text) {
            var out = '<img src="' + href + '" alt="' + text + '" class="img-responsive"';
            if (title) {
                out += ' title="' + title + '"';
            }
            out += '>';
            return out;
        };
        this.markedInstance.setOptions({
            renderer: renderer,
            gfm: true,
            breaks: false
        });
    }
    MarkdownEngine.getInstance = function () {
        if (!MarkdownEngine.instance) {
            MarkdownEngine.instance = new MarkdownEngine();
        }
        return MarkdownEngine.instance;
    };
    MarkdownEngine.prototype.getTraditionalMarkdown = function (filepath) {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + filepath + '.md')
            .catch(function (err) { return FileEngine$1.get(process.cwd() + path.sep + filepath).then(); })
            .then(function (data) { return _this.markedInstance(data); });
    };
    MarkdownEngine.prototype.getTraditionalMarkdownSync = function (filepath) {
        return this.markedInstance(FileEngine$1.getSync(process.cwd() + path.sep + filepath));
    };
    MarkdownEngine.prototype.getReadmeFile = function () {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + 'README.md').then(function (data) {
            return _this.markedInstance(data);
        });
    };
    MarkdownEngine.prototype.readNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return fs.readFileSync(readmeFile, 'utf8');
    };
    MarkdownEngine.prototype.hasNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return FileEngine$1.existsSync(readmeFile);
    };
    MarkdownEngine.prototype.componentReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + 'README.md';
        var readmeAlternativeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        var finalPath = '';
        if (FileEngine$1.existsSync(readmeFile)) {
            finalPath = readmeFile;
        }
        else {
            finalPath = readmeAlternativeFile;
        }
        return finalPath;
    };
    /**
     * Checks if any of the markdown files is exists with or without endings
     */
    MarkdownEngine.prototype.hasRootMarkdowns = function () {
        return this.addEndings(this.markdownFiles).some(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
    };
    MarkdownEngine.prototype.listRootMarkdowns = function () {
        var foundFiles = this.markdownFiles.filter(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x + '.md') ||
                FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
        return this.addEndings(foundFiles);
    };
    MarkdownEngine.prototype.escape = function (html) {
        return html
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
            .replace(/@/g, '&#64;');
    };
    /**
     * ['README'] => ['README', 'README.md']
     */
    MarkdownEngine.prototype.addEndings = function (files) {
        return _.flatMap(files, function (x) { return [x, x + '.md']; });
    };
    return MarkdownEngine;
}());
var MarkdownEngine$1 = MarkdownEngine.getInstance();

var ngdT = require('@compodoc/ngd-transformer');
var NgdEngine = /** @class */ (function () {
    function NgdEngine() {
    }
    NgdEngine.getInstance = function () {
        if (!NgdEngine.instance) {
            NgdEngine.instance = new NgdEngine();
        }
        return NgdEngine.instance;
    };
    NgdEngine.prototype.init = function (outputpath) {
        this.engine = new ngdT.DotEngine({
            output: outputpath,
            displayLegend: true,
            outputFormats: 'svg',
            silent: true
        });
    };
    NgdEngine.prototype.renderGraph = function (filepath, outputpath, type, name) {
        this.engine.updateOutput(outputpath);
        if (type === 'f') {
            return this.engine.generateGraph([DependenciesEngine$1.getRawModule(name)]);
        }
        else {
            return this.engine.generateGraph(DependenciesEngine$1.rawModulesForOverview);
        }
    };
    NgdEngine.prototype.readGraph = function (filepath, name) {
        return FileEngine$1.get(filepath).catch(function (err) {
            return Promise.reject('Error during graph read ' + name);
        });
    };
    return NgdEngine;
}());
var NgdEngine$1 = NgdEngine.getInstance();

var COMPODOC_CONSTANTS = {
    navTabDefinitions: [
        {
            id: 'info',
            href: '#info',
            'data-link': 'info',
            label: 'Info',
            depTypes: ['all']
        },
        {
            id: 'readme',
            href: '#readme',
            'data-link': 'readme',
            label: 'README',
            depTypes: ['all']
        },
        {
            id: 'source',
            href: '#source',
            'data-link': 'source',
            label: 'Source',
            depTypes: ['all']
        },
        {
            id: 'templateData',
            href: '#templateData',
            'data-link': 'template',
            label: 'Template',
            depTypes: ['component']
        },
        {
            id: 'styleData',
            href: '#styleData',
            'data-link': 'style',
            label: 'Styles',
            depTypes: ['component']
        },
        {
            id: 'tree',
            href: '#tree',
            'data-link': 'dom-tree',
            label: 'DOM Tree',
            depTypes: ['component']
        },
        {
            id: 'example',
            href: '#example',
            'data-link': 'example',
            label: 'Examples',
            depTypes: ['component', 'directive', 'injectable', 'pipe']
        }
    ]
};
/**
 * Max length for the string of a file during Lunr search engine indexing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_SEARCH_INDEX = 50000;
/**
 * Max length for the string of a file during cheerio parsing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_CHEERIO_PARSING = 400000000;

var lunr = require('lunr');
var cheerio = require('cheerio');
var Entities = require('html-entities').AllHtmlEntities;
var Html = new Entities();
var SearchEngine = /** @class */ (function () {
    function SearchEngine() {
        this.searchDocuments = [];
        this.documentsStore = {};
        this.amountOfMemory = 0;
    }
    SearchEngine.getInstance = function () {
        if (!SearchEngine.instance) {
            SearchEngine.instance = new SearchEngine();
        }
        return SearchEngine.instance;
    };
    SearchEngine.prototype.indexPage = function (page) {
        var text;
        this.amountOfMemory += page.rawData.length;
        if (this.amountOfMemory < MAX_SIZE_FILE_CHEERIO_PARSING) {
            var indexStartContent = page.rawData.indexOf('<!-- START CONTENT -->');
            var indexEndContent = page.rawData.indexOf('<!-- END CONTENT -->');
            var $ = cheerio.load(page.rawData.substring(indexStartContent + 1, indexEndContent));
            text = $('.content').html();
            text = Html.decode(text);
            text = text.replace(/(<([^>]+)>)/gi, '');
            page.url = page.url.replace(Configuration$1.mainData.output, '');
            var doc = {
                url: page.url,
                title: page.infos.context + ' - ' + page.infos.name,
                body: text
            };
            if (!this.documentsStore.hasOwnProperty(doc.url) &&
                doc.body.length < MAX_SIZE_FILE_SEARCH_INDEX) {
                this.documentsStore[doc.url] = doc;
                this.searchDocuments.push(doc);
            }
        }
    };
    SearchEngine.prototype.generateSearchIndexJson = function (outputFolder) {
        var _this = this;
        var that = this;
        var searchIndex = lunr(function () {
            /* tslint:disable:no-invalid-this */
            this.ref('url');
            this.field('title');
            this.field('body');
            this.pipeline.remove(lunr.stemmer);
            var i = 0;
            var len = that.searchDocuments.length;
            for (i; i < len; i++) {
                this.add(that.searchDocuments[i]);
            }
        });
        return FileEngine$1.get(__dirname + '/../src/templates/partials/search-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                index: JSON.stringify(searchIndex),
                store: JSON.stringify(_this.documentsStore)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/search/search_index.js', result).catch(function (err) {
                logger.error('Error during search index file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during search index generation'); });
    };
    return SearchEngine;
}());
var SearchEngine$1 = SearchEngine.getInstance();

var $ = require('cheerio');
var ComponentsTreeEngine = /** @class */ (function () {
    function ComponentsTreeEngine() {
        this.components = [];
        this.componentsForTree = [];
    }
    ComponentsTreeEngine.getInstance = function () {
        if (!ComponentsTreeEngine.instance) {
            ComponentsTreeEngine.instance = new ComponentsTreeEngine();
        }
        return ComponentsTreeEngine.instance;
    };
    ComponentsTreeEngine.prototype.addComponent = function (component) {
        this.components.push(component);
    };
    ComponentsTreeEngine.prototype.readTemplates = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = _this.componentsForTree.length;
            var loop = function () {
                if (i <= len - 1) {
                    if (_this.componentsForTree[i].templateUrl) {
                        var filePath = process.cwd() +
                            path.sep +
                            path.dirname(_this.componentsForTree[i].file) +
                            path.sep +
                            _this.componentsForTree[i].templateUrl;
                        FileEngine$1.get(filePath).then(function (templateData) {
                            _this.componentsForTree[i].templateData = templateData;
                            i++;
                            loop();
                        }, function (e) {
                            logger.error(e);
                            reject();
                        });
                    }
                    else {
                        _this.componentsForTree[i].templateData = _this.componentsForTree[i].template;
                        i++;
                        loop();
                    }
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    ComponentsTreeEngine.prototype.findChildrenAndParents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.componentsForTree, function (component) {
                var $component = $(component.templateData);
                _.forEach(_this.componentsForTree, function (componentToFind) {
                    if ($component.find(componentToFind.selector).length > 0) {
                        console.log(componentToFind.name + ' found in ' + component.name);
                        component.children.push(componentToFind.name);
                    }
                });
            });
            resolve();
        });
    };
    ComponentsTreeEngine.prototype.createTreesForComponents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.components, function (component) {
                var _component = {
                    name: component.name,
                    file: component.file,
                    selector: component.selector,
                    children: [],
                    template: '',
                    templateUrl: ''
                };
                if (typeof component.template !== 'undefined') {
                    _component.template = component.template;
                }
                if (component.templateUrl.length > 0) {
                    _component.templateUrl = component.templateUrl[0];
                }
                _this.componentsForTree.push(_component);
            });
            _this.readTemplates().then(function () {
                _this.findChildrenAndParents().then(function () {
                    console.log('this.componentsForTree: ', _this.componentsForTree);
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }, function (e) {
                logger.error(e);
            });
        });
    };
    return ComponentsTreeEngine;
}());
var ComponentsTreeEngine$1 = ComponentsTreeEngine.getInstance();

var JsdocParserUtil = /** @class */ (function () {
    function JsdocParserUtil() {
    }
    JsdocParserUtil.prototype.isVariableLike = function (node) {
        if (node) {
            switch (node.kind) {
                case Ast.SyntaxKind.BindingElement:
                case Ast.SyntaxKind.EnumMember:
                case Ast.SyntaxKind.Parameter:
                case Ast.SyntaxKind.PropertyAssignment:
                case Ast.SyntaxKind.PropertyDeclaration:
                case Ast.SyntaxKind.PropertySignature:
                case Ast.SyntaxKind.ShorthandPropertyAssignment:
                case Ast.SyntaxKind.VariableDeclaration:
                    return true;
            }
        }
        return false;
    };
    JsdocParserUtil.prototype.getMainCommentOfNode = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = node.jsDoc[0].comment;
                }
            }
        }
        return description;
    };
    JsdocParserUtil.prototype.getJSDocTags = function (node, kind) {
        var docs = this.getJSDocs(node);
        if (docs) {
            var result = [];
            for (var _i = 0, docs_1 = docs; _i < docs_1.length; _i++) {
                var doc = docs_1[_i];
                if (Ast.ts.isJSDocParameterTag(doc)) {
                    if (doc.kind === kind) {
                        result.push(doc);
                    }
                }
                else if (Ast.ts.isJSDoc(doc)) {
                    result.push.apply(result, _.filter(doc.tags, function (tag) { return tag.kind === kind; }));
                }
                else {
                    throw new Error('Unexpected type');
                }
            }
            return result;
        }
    };
    JsdocParserUtil.prototype.getJSDocs = function (node) {
        // TODO: jsDocCache is internal, see if there's a way around it
        var cache = node.jsDocCache;
        if (!cache) {
            cache = this.getJSDocsWorker(node, []).filter(function (x) { return x; });
            node.jsDocCache = cache;
        }
        return cache;
    };
    // Try to recognize this pattern when node is initializer
    // of variable declaration and JSDoc comments are on containing variable statement.
    // /**
    //   * @param {number} name
    //   * @returns {number}
    //   */
    // var x = function(name) { return name.length; }
    JsdocParserUtil.prototype.getJSDocsWorker = function (node, cache) {
        var parent = node.parent;
        var isInitializerOfVariableDeclarationInStatement = this.isVariableLike(parent) &&
            parent.initializer === node &&
            Ast.ts.isVariableStatement(parent.parent.parent);
        var isVariableOfVariableDeclarationStatement = this.isVariableLike(node) && Ast.ts.isVariableStatement(parent.parent);
        var variableStatementNode = isInitializerOfVariableDeclarationInStatement
            ? parent.parent.parent
            : isVariableOfVariableDeclarationStatement
                ? parent.parent
                : undefined;
        if (variableStatementNode) {
            cache = this.getJSDocsWorker(variableStatementNode, cache);
        }
        // Also recognize when the node is the RHS of an assignment expression
        var isSourceOfAssignmentExpressionStatement = parent &&
            parent.parent &&
            Ast.ts.isBinaryExpression(parent) &&
            parent.operatorToken.kind === Ast.SyntaxKind.EqualsToken &&
            Ast.ts.isExpressionStatement(parent.parent);
        if (isSourceOfAssignmentExpressionStatement) {
            cache = this.getJSDocsWorker(parent.parent, cache);
        }
        var isModuleDeclaration = Ast.ts.isModuleDeclaration(node) && parent && Ast.ts.isModuleDeclaration(parent);
        var isPropertyAssignmentExpression = parent && Ast.ts.isPropertyAssignment(parent);
        if (isModuleDeclaration || isPropertyAssignmentExpression) {
            cache = this.getJSDocsWorker(parent, cache);
        }
        // Pull parameter comments from declaring function as well
        if (Ast.ts.isParameter(node)) {
            cache = _.concat(cache, this.getJSDocParameterTags(node));
        }
        if (this.isVariableLike(node) && node.initializer) {
            cache = _.concat(cache, node.initializer.jsDoc);
        }
        cache = _.concat(cache, node.jsDoc);
        return cache;
    };
    JsdocParserUtil.prototype.getJSDocParameterTags = function (param) {
        var func = param.parent;
        var tags = this.getJSDocTags(func, Ast.SyntaxKind.JSDocParameterTag);
        if (!param.name) {
            // this is an anonymous jsdoc param from a `function(type1, type2): type3` specification
            var i = func.parameters.indexOf(param);
            var paramTags = _.filter(tags, function (tag) { return Ast.ts.isJSDocParameterTag(tag); });
            if (paramTags && 0 <= i && i < paramTags.length) {
                return [paramTags[i]];
            }
        }
        else if (Ast.ts.isIdentifier(param.name)) {
            var name_1 = param.name.text;
            return _.filter(tags, function (tag) {
                if (Ast.ts && Ast.ts.isJSDocParameterTag(tag)) {
                    var t = tag;
                    if (typeof t.parameterName !== 'undefined') {
                        return t.parameterName.text === name_1;
                    }
                    else if (typeof t.name !== 'undefined') {
                        if (typeof t.name.escapedText !== 'undefined') {
                            return t.name.escapedText === name_1;
                        }
                    }
                }
            });
        }
        else {
            // TODO: it's a destructured parameter, so it should look up an "object type" series of multiple lines
            // But multi-line object types aren't supported yet either
            return undefined;
        }
    };
    return JsdocParserUtil;
}());

var ast = new Ast__default();
var ImportsUtil = /** @class */ (function () {
    function ImportsUtil() {
    }
    ImportsUtil.getInstance = function () {
        if (!ImportsUtil.instance) {
            ImportsUtil.instance = new ImportsUtil();
        }
        return ImportsUtil.instance;
    };
    /**
     * Find for a sourceFile a variable value in a local enum
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInEnums = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getEnum(function (e) {
            if (e.getName() === variableName) {
                e.getMember(function (m) {
                    if (m.getName() === variableValue) {
                        res = m.getValue();
                    }
                });
            }
        });
        return res;
    };
    /**
     * Find for a sourceFile a variable value in a local static class
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInClasses = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getClass(function (c) {
            var staticProperty = c.getStaticProperty(variableValue);
            if (staticProperty) {
                if (staticProperty.getInitializer()) {
                    res = staticProperty.getInitializer().getText();
                }
            }
        });
        return res;
    };
    /**
     * Find a value in a local variable declaration like an object
     * @param variableDeclaration
     * @param variablesAttributes
     */
    ImportsUtil.prototype.findInObjectVariableDeclaration = function (variableDeclaration, variablesAttributes) {
        var variableKind = variableDeclaration.getKind();
        if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
            var initializer = variableDeclaration.getInitializer();
            if (initializer) {
                var initializerKind = initializer.getKind();
                if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                    var compilerNode = initializer.compilerNode, finalValue_1 = '';
                    // Find thestring from AVAR.BVAR.thestring inside properties
                    var depth_1 = 0;
                    var loopProperties_1 = function (properties) {
                        properties.forEach(function (prop) {
                            if (prop.name) {
                                if (variablesAttributes[depth_1 + 1]) {
                                    if (prop.name.getText() === variablesAttributes[depth_1 + 1]) {
                                        if (prop.initializer) {
                                            if (prop.initializer.properties) {
                                                depth_1 += 1;
                                                loopProperties_1(prop.initializer.properties);
                                            }
                                            else {
                                                finalValue_1 = prop.initializer.text;
                                            }
                                        }
                                        else {
                                            finalValue_1 = prop.initializer.text;
                                        }
                                    }
                                }
                            }
                        });
                    };
                    loopProperties_1(compilerNode.properties);
                    return finalValue_1;
                }
            }
        }
    };
    /**
     * Find in imports something like myvar
     * @param  {string} inputVariableName              like myvar
     * @return {[type]}                                myvar value
     */
    ImportsUtil.prototype.findValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var metadataVariableName = inputVariableName, searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFileIfExists(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        function hasFoundValues(variableDeclaration) {
            var variableKind = variableDeclaration.getKind();
            if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                var initializer = variableDeclaration.getInitializer();
                if (initializer) {
                    var initializerKind = initializer.getKind();
                    if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                        var compilerNode = initializer.compilerNode;
                        return compilerNode.properties;
                    }
                }
            }
        }
        if (typeof searchedImport !== 'undefined') {
            var importPathReference = searchedImport.getModuleSpecifierSourceFile();
            var importPath = void 0;
            if (typeof importPathReference !== 'undefined') {
                importPath = importPathReference.compilerNode.fileName;
                var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                    ? ast.getSourceFile(importPath)
                    : ast.addExistingSourceFileIfExists(importPath); // tslint:disable-line
                if (sourceFileImport) {
                    var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                    var variableDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    if (variableDeclaration) {
                        return hasFoundValues(variableDeclaration);
                    }
                    else {
                        // Try with exports
                        var exportDeclarations = sourceFileImport.getExportDeclarations();
                        if (exportDeclarations && exportDeclarations.length > 0) {
                            var i = 0, len = exportDeclarations.length;
                            for (i; i < len; i++) {
                                var exportDeclaration = exportDeclarations[i];
                                var sourceFileExportedReference = exportDeclaration.getModuleSpecifierSourceFile();
                                if (sourceFileExportedReference) {
                                    var sourceFileExportedReferencePath = sourceFileExportedReference.getFilePath();
                                    var sourceFileExported = typeof ast.getSourceFile(sourceFileExportedReferencePath) !== 'undefined'
                                        ? ast.getSourceFile(sourceFileExportedReferencePath)
                                        : ast.addExistingSourceFileIfExists(sourceFileExportedReferencePath);
                                    if (sourceFileExported) {
                                        variableDeclaration = sourceFileExported.getVariableDeclaration(variableName);
                                        if (variableDeclaration) {
                                            return hasFoundValues(variableDeclaration);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        else {
            // Find in local variables of the file
            var variableDeclaration = file.getVariableDeclaration(metadataVariableName);
            if (variableDeclaration) {
                var variableKind = variableDeclaration.getKind();
                if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                    var initializer = variableDeclaration.getInitializer();
                    if (initializer) {
                        var initializerKind = initializer.getKind();
                        if (initializerKind &&
                            initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                            var compilerNode = initializer.compilerNode;
                            return compilerNode.properties;
                        }
                        else if (initializerKind) {
                            return variableDeclaration.compilerNode;
                        }
                    }
                }
            }
        }
        return [];
    };
    ImportsUtil.prototype.getFileNameOfImport = function (variableName, sourceFile) {
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        var searchedImport, finalPath = '';
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === variableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === variableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
            finalPath = importPath.replace(cleaner, '');
        }
        return finalPath;
    };
    /**
     * Find the file path of imported variable
     * @param  {string} inputVariableName  like thestring
     * @return {[type]}                    thestring destination path
     */
    ImportsUtil.prototype.findFilePathOfImportedVariable = function (inputVariableName, sourceFilePath) {
        var searchedImport, finalPath = '';
        var file = typeof ast.getSourceFile(sourceFilePath) !== 'undefined'
            ? ast.getSourceFile(sourceFilePath)
            : ast.addExistingSourceFile(sourceFilePath); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            finalPath = path.resolve(path.dirname(sourceFilePath) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
        }
        return finalPath;
    };
    /**
     * Find in imports something like VAR.AVAR.BVAR.thestring
     * @param  {string} inputVariableName                   like VAR.AVAR.BVAR.thestring
     * @return {[type]}                                thestring value
     */
    ImportsUtil.prototype.findPropertyValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var variablesAttributes = inputVariableName.split('.'), metadataVariableName = variablesAttributes[0], searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        var fileToSearchIn, variableDeclaration;
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                ? ast.getSourceFile(importPath)
                : ast.addExistingSourceFile(importPath); // tslint:disable-line
            if (sourceFileImport) {
                fileToSearchIn = sourceFileImport;
                var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                variableDeclaration = fileToSearchIn.getVariableDeclaration(variableName);
            }
        }
        else {
            fileToSearchIn = file;
            // Find in local variables of the file
            variableDeclaration = fileToSearchIn.getVariableDeclaration(metadataVariableName);
        }
        if (variableDeclaration) {
            return this.findInObjectVariableDeclaration(variableDeclaration, variablesAttributes);
        }
        // Try find it in enums
        if (variablesAttributes.length > 0) {
            if (typeof fileToSearchIn !== 'undefined') {
                var val = this.findInEnums(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
                val = this.findInClasses(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
            }
        }
    };
    return ImportsUtil;
}());
var ImportsUtil$1 = ImportsUtil.getInstance();

var traverse$2 = require('traverse');
var ast$1 = new Ast__default();
var RouterParserUtil = /** @class */ (function () {
    function RouterParserUtil() {
        this.routes = [];
        this.incompleteRoutes = [];
        this.modules = [];
        this.modulesWithRoutes = [];
    }
    RouterParserUtil.getInstance = function () {
        if (!RouterParserUtil.instance) {
            RouterParserUtil.instance = new RouterParserUtil();
        }
        return RouterParserUtil.instance;
    };
    RouterParserUtil.prototype.addRoute = function (route) {
        this.routes.push(route);
        this.routes = _.sortBy(_.uniqWith(this.routes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addIncompleteRoute = function (route) {
        this.incompleteRoutes.push(route);
        this.incompleteRoutes = _.sortBy(_.uniqWith(this.incompleteRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModuleWithRoutes = function (moduleName, moduleImports, filename) {
        this.modulesWithRoutes.push({
            name: moduleName,
            importsNode: moduleImports,
            filename: filename
        });
        this.modulesWithRoutes = _.sortBy(_.uniqWith(this.modulesWithRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModule = function (moduleName, moduleImports) {
        this.modules.push({
            name: moduleName,
            importsNode: moduleImports
        });
        this.modules = _.sortBy(_.uniqWith(this.modules, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.cleanRawRouteParsed = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        return JSON5.parse(routesWithoutSpaces);
    };
    RouterParserUtil.prototype.cleanRawRoute = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        return routesWithoutSpaces;
    };
    RouterParserUtil.prototype.setRootModule = function (module) {
        this.rootModule = module;
    };
    RouterParserUtil.prototype.hasRouterModuleInImports = function (imports) {
        for (var i = 0; i < imports.length; i++) {
            if (imports[i].name.indexOf('RouterModule.forChild') !== -1 ||
                imports[i].name.indexOf('RouterModule.forRoot') !== -1 ||
                imports[i].name.indexOf('RouterModule') !== -1) {
                return true;
            }
        }
        return false;
    };
    RouterParserUtil.prototype.fixIncompleteRoutes = function (miscellaneousVariables) {
        var matchingVariables = [];
        // For each incompleteRoute, scan if one misc variable is in code
        // if ok, try recreating complete route
        for (var i = 0; i < this.incompleteRoutes.length; i++) {
            for (var j = 0; j < miscellaneousVariables.length; j++) {
                if (this.incompleteRoutes[i].data.indexOf(miscellaneousVariables[j].name) !== -1) {
                    console.log('found one misc var inside incompleteRoute');
                    console.log(miscellaneousVariables[j].name);
                    matchingVariables.push(miscellaneousVariables[j]);
                }
            }
            // Clean incompleteRoute
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace('[', '');
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace(']', '');
        }
    };
    RouterParserUtil.prototype.linkModulesAndRoutes = function () {
        var _this = this;
        var i = 0;
        var len = this.modulesWithRoutes.length;
        for (i; i < len; i++) {
            _.forEach(this.modulesWithRoutes[i].importsNode, function (node) {
                var initializer = node.initializer;
                if (initializer) {
                    if (initializer.elements) {
                        _.forEach(initializer.elements, function (element) {
                            // find element with arguments
                            if (element.arguments) {
                                _.forEach(element.arguments, function (argument) {
                                    _.forEach(_this.routes, function (route) {
                                        if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename === _this.modulesWithRoutes[i].filename) {
                                            route.module = _this.modulesWithRoutes[i].name;
                                        }
                                        else if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename !== _this.modulesWithRoutes[i].filename) {
                                            var argumentImportPath = ImportsUtil$1.findFilePathOfImportedVariable(argument.text, _this.modulesWithRoutes[i].filename);
                                            var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
                                            argumentImportPath = argumentImportPath.replace(cleaner, '');
                                            if (argument.text &&
                                                route.name === argument.text &&
                                                route.filename === argumentImportPath) {
                                                route.module = _this.modulesWithRoutes[i].name;
                                            }
                                        }
                                    });
                                });
                            }
                        });
                    }
                }
                /**
                 * direct support of for example
                 * export const HomeRoutingModule: ModuleWithProviders = RouterModule.forChild(HOME_ROUTES);
                 */
                if (Ast.ts.isCallExpression(node)) {
                    if (node.arguments) {
                        _.forEach(node.arguments, function (argument) {
                            _.forEach(_this.routes, function (route) {
                                if (argument.text &&
                                    route.name === argument.text &&
                                    route.filename === _this.modulesWithRoutes[i].filename) {
                                    route.module = _this.modulesWithRoutes[i].name;
                                }
                            });
                        });
                    }
                }
            });
        }
    };
    RouterParserUtil.prototype.foundRouteWithModuleName = function (moduleName) {
        return _.find(this.routes, { module: moduleName });
    };
    RouterParserUtil.prototype.foundLazyModuleWithPath = function (modulePath) {
        // path is like app/customers/customers.module#CustomersModule
        var split = modulePath.split('#');
        var lazyModulePath = split[0];
        var lazyModuleName = split[1];
        return lazyModuleName;
    };
    RouterParserUtil.prototype.constructRoutesTree = function () {
        var _this = this;
        // routes[] contains routes with module link
        // modulesTree contains modules tree
        // make a final routes tree with that
        traverse$2(this.modulesTree).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (node.importsNode) {
                    delete node.importsNode;
                }
            }
        });
        this.cleanModulesTree = _.cloneDeep(this.modulesTree);
        var routesTree = {
            name: '<root>',
            kind: 'module',
            className: this.rootModule,
            children: []
        };
        var loopModulesParser = function (node) {
            if (node.children && node.children.length > 0) {
                // If module has child modules
                for (var i in node.children) {
                    var route = _this.foundRouteWithModuleName(node.children[i].name);
                    if (route && route.data) {
                        try {
                            route.children = JSON5.parse(route.data);
                        }
                        catch (e) {
                            logger.error('Error during generation of routes JSON file, maybe a trailing comma or an external variable inside one route.');
                        }
                        delete route.data;
                        route.kind = 'module';
                        routesTree.children.push(route);
                    }
                    if (node.children[i].children) {
                        loopModulesParser(node.children[i]);
                    }
                }
            }
            else {
                // else routes are directly inside the module
                var rawRoutes = _this.foundRouteWithModuleName(node.name);
                if (rawRoutes) {
                    var routes = JSON5.parse(rawRoutes.data);
                    if (routes) {
                        var i = 0;
                        var len = routes.length;
                        var routeAddedOnce = false;
                        for (i; i < len; i++) {
                            var route = routes[i];
                            if (routes[i].component) {
                                routeAddedOnce = true;
                                routesTree.children.push({
                                    kind: 'component',
                                    component: routes[i].component,
                                    path: routes[i].path
                                });
                            }
                        }
                        if (!routeAddedOnce) {
                            routesTree.children = routesTree.children.concat(routes);
                        }
                    }
                }
            }
        };
        var startModule = _.find(this.cleanModulesTree, { name: this.rootModule });
        if (startModule) {
            loopModulesParser(startModule);
            // Loop twice for routes with lazy loading
            // loopModulesParser(routesTree);
        }
        var cleanedRoutesTree = undefined;
        var cleanRoutesTree = function (route) {
            for (var i in route.children) {
                var routes = route.children[i].routes;
            }
            return route;
        };
        cleanedRoutesTree = cleanRoutesTree(routesTree);
        // Try updating routes with lazy loading
        var loopInsideModule = function (mod, _rawModule) {
            if (mod.children) {
                for (var z in mod.children) {
                    var route = _this.foundRouteWithModuleName(mod.children[z].name);
                    if (typeof route !== 'undefined') {
                        if (route.data) {
                            route.children = JSON5.parse(route.data);
                            delete route.data;
                            route.kind = 'module';
                            _rawModule.children.push(route);
                        }
                    }
                }
            }
            else {
                var route = _this.foundRouteWithModuleName(mod.name);
                if (typeof route !== 'undefined') {
                    if (route.data) {
                        route.children = JSON5.parse(route.data);
                        delete route.data;
                        route.kind = 'module';
                        _rawModule.children.push(route);
                    }
                }
            }
        };
        var loopRoutesParser = function (route) {
            if (route.children) {
                for (var i in route.children) {
                    if (route.children[i].loadChildren) {
                        var child = _this.foundLazyModuleWithPath(route.children[i].loadChildren);
                        var module = _.find(_this.cleanModulesTree, {
                            name: child
                        });
                        if (module) {
                            var _rawModule = {};
                            _rawModule.kind = 'module';
                            _rawModule.children = [];
                            _rawModule.module = module.name;
                            loopInsideModule(module, _rawModule);
                            route.children[i].children = [];
                            route.children[i].children.push(_rawModule);
                        }
                    }
                    loopRoutesParser(route.children[i]);
                }
            }
        };
        loopRoutesParser(cleanedRoutesTree);
        return cleanedRoutesTree;
    };
    RouterParserUtil.prototype.constructModulesTree = function () {
        var _this = this;
        var getNestedChildren = function (arr, parent) {
            var out = [];
            for (var i in arr) {
                if (arr[i].parent === parent) {
                    var children = getNestedChildren(arr, arr[i].name);
                    if (children.length) {
                        arr[i].children = children;
                    }
                    out.push(arr[i]);
                }
            }
            return out;
        };
        // Scan each module and add parent property
        _.forEach(this.modules, function (firstLoopModule) {
            _.forEach(firstLoopModule.importsNode, function (importNode) {
                _.forEach(_this.modules, function (module) {
                    if (module.name === importNode.name) {
                        module.parent = firstLoopModule.name;
                    }
                });
            });
        });
        this.modulesTree = getNestedChildren(this.modules);
    };
    RouterParserUtil.prototype.generateRoutesIndex = function (outputFolder, routes) {
        return FileEngine$1.get(__dirname + '/../src/templates/partials/routes-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                routes: JSON.stringify(routes)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/routes/routes_index.js', result);
        }, function (err) { return Promise.reject('Error during routes index generation'); });
    };
    RouterParserUtil.prototype.routesLength = function () {
        var _n = 0;
        var routesParser = function (route) {
            if (typeof route.path !== 'undefined') {
                _n += 1;
            }
            if (route.children) {
                for (var j in route.children) {
                    routesParser(route.children[j]);
                }
            }
        };
        for (var i in this.routes) {
            routesParser(this.routes[i]);
        }
        return _n;
    };
    RouterParserUtil.prototype.printRoutes = function () {
        console.log('');
        console.log('printRoutes: ');
        console.log(this.routes);
    };
    RouterParserUtil.prototype.printModulesRoutes = function () {
        console.log('');
        console.log('printModulesRoutes: ');
        console.log(this.modulesWithRoutes);
    };
    RouterParserUtil.prototype.isVariableRoutes = function (node) {
        var result = false;
        if (node.declarationList && node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                if (node.declarationList.declarations[i].type) {
                    if (node.declarationList.declarations[i].type.typeName &&
                        node.declarationList.declarations[i].type.typeName.text === 'Routes') {
                        result = true;
                    }
                }
            }
        }
        return result;
    };
    RouterParserUtil.prototype.cleanFileIdentifiers = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var identifiers = file.getDescendantsOfKind(Ast.SyntaxKind.Identifier).filter(function (p) {
            return (Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()) ||
                Ast.TypeGuards.isPropertyAssignment(p.getParentOrThrow()));
        });
        var identifiersInRoutesVariableStatement = [];
        var _loop_1 = function (identifier) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = identifier.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                identifiersInRoutesVariableStatement.push(identifier);
            }
        };
        for (var _i = 0, identifiers_1 = identifiers; _i < identifiers_1.length; _i++) {
            var identifier = identifiers_1[_i];
            _loop_1(identifier);
        }
        // inline the property access expressions
        for (var _a = 0, identifiersInRoutesVariableStatement_1 = identifiersInRoutesVariableStatement; _a < identifiersInRoutesVariableStatement_1.length; _a++) {
            var identifier = identifiersInRoutesVariableStatement_1[_a];
            var identifierDeclaration = identifier
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                Ast.TypeGuards.isVariableDeclaration(identifierDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                    !Ast.TypeGuards.isVariableDeclaration(identifierDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + identifierDeclaration.getKindName());
            }
            if (Ast.TypeGuards.isVariableDeclaration(identifierDeclaration)) {
                identifier.replaceWithText(identifierDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileSpreads = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var spreadElements = file
            .getDescendantsOfKind(Ast.SyntaxKind.SpreadElement)
            .filter(function (p) { return Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()); });
        var spreadElementsInRoutesVariableStatement = [];
        var _loop_2 = function (spreadElement) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = spreadElement.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                spreadElementsInRoutesVariableStatement.push(spreadElement);
            }
        };
        for (var _i = 0, spreadElements_1 = spreadElements; _i < spreadElements_1.length; _i++) {
            var spreadElement = spreadElements_1[_i];
            _loop_2(spreadElement);
        }
        var _loop_3 = function (spreadElement) {
            var spreadElementIdentifier = spreadElement.getExpression().getText(), searchedImport, aliasOriginalName = '', foundWithAliasInImports = false, foundWithAlias = false;
            // Try to find it in imports
            var imports = file.getImportDeclarations();
            imports.forEach(function (i) {
                var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
                if (namedImportsLength > 0) {
                    for (j; j < namedImportsLength; j++) {
                        var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                        if (namedImports[j].getAliasIdentifier()) {
                            importAlias = namedImports[j].getAliasIdentifier().getText();
                        }
                        if (importName === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            searchedImport = i;
                            break;
                        }
                        if (importAlias === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            foundWithAlias = true;
                            aliasOriginalName = importName;
                            searchedImport = i;
                            break;
                        }
                    }
                }
            });
            var referencedDeclaration = void 0;
            if (foundWithAliasInImports) {
                if (typeof searchedImport !== 'undefined') {
                    var importPath = path.resolve(path.dirname(file.getFilePath()) +
                        '/' +
                        searchedImport.getModuleSpecifierValue() +
                        '.ts');
                    var sourceFileImport = typeof ast$1.getSourceFile(importPath) !== 'undefined'
                        ? ast$1.getSourceFile(importPath)
                        : ast$1.addExistingSourceFile(importPath);
                    if (sourceFileImport) {
                        var variableName = foundWithAlias
                            ? aliasOriginalName
                            : spreadElementIdentifier;
                        referencedDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    }
                }
            }
            else {
                // if not, try directly in file
                referencedDeclaration = spreadElement
                    .getExpression()
                    .getSymbolOrThrow()
                    .getValueDeclarationOrThrow();
            }
            if (!Ast.TypeGuards.isVariableDeclaration(referencedDeclaration)) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            var referencedArray = referencedDeclaration.getInitializerIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var spreadElementArray = spreadElement.getParentIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var insertIndex = spreadElementArray.getElements().indexOf(spreadElement);
            spreadElementArray.removeElement(spreadElement);
            spreadElementArray.insertElements(insertIndex, referencedArray.getElements().map(function (e) { return e.getText(); }));
        };
        // inline the ArrayLiteralExpression SpreadElements
        for (var _a = 0, spreadElementsInRoutesVariableStatement_1 = spreadElementsInRoutesVariableStatement; _a < spreadElementsInRoutesVariableStatement_1.length; _a++) {
            var spreadElement = spreadElementsInRoutesVariableStatement_1[_a];
            _loop_3(spreadElement);
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileDynamics = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var propertyAccessExpressions = file
            .getDescendantsOfKind(Ast.SyntaxKind.PropertyAccessExpression)
            .filter(function (p) { return !Ast.TypeGuards.isPropertyAccessExpression(p.getParentOrThrow()); });
        var propertyAccessExpressionsInRoutesVariableStatement = [];
        var _loop_4 = function (propertyAccessExpression) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = propertyAccessExpression.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                propertyAccessExpressionsInRoutesVariableStatement.push(propertyAccessExpression);
            }
        };
        for (var _i = 0, propertyAccessExpressions_1 = propertyAccessExpressions; _i < propertyAccessExpressions_1.length; _i++) {
            var propertyAccessExpression = propertyAccessExpressions_1[_i];
            _loop_4(propertyAccessExpression);
        }
        // inline the property access expressions
        for (var _a = 0, propertyAccessExpressionsInRoutesVariableStatement_1 = propertyAccessExpressionsInRoutesVariableStatement; _a < propertyAccessExpressionsInRoutesVariableStatement_1.length; _a++) {
            var propertyAccessExpression = propertyAccessExpressionsInRoutesVariableStatement_1[_a];
            var referencedDeclaration = propertyAccessExpression
                .getNameNode()
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                Ast.TypeGuards.isEnumMember(referencedDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                    !Ast.TypeGuards.isEnumMember(referencedDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            if (typeof referencedDeclaration.getInitializerOrThrow !== 'undefined') {
                propertyAccessExpression.replaceWithText(referencedDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    /**
     * replace callexpressions with string : utils.doWork() -> 'utils.doWork()' doWork() -> 'doWork()'
     * @param sourceFile ts.SourceFile
     */
    RouterParserUtil.prototype.cleanCallExpressions = function (sourceFile) {
        var file = sourceFile;
        var variableStatements = sourceFile.getVariableDeclaration(function (v) {
            var result = false;
            if (typeof v.compilerNode.type !== 'undefined') {
                result = v.compilerNode.type.typeName.text === 'Routes';
            }
            return result;
        });
        var initializer = variableStatements.getInitializer();
        var _loop_5 = function (callExpr) {
            if (callExpr.wasForgotten()) {
                return "continue";
            }
            callExpr.replaceWithText(function (writer) { return writer.quote(callExpr.getText()); });
        };
        for (var _i = 0, _a = initializer.getDescendantsOfKind(Ast.SyntaxKind.CallExpression); _i < _a.length; _i++) {
            var callExpr = _a[_i];
            _loop_5(callExpr);
        }
        return file;
    };
    /**
     * Clean routes definition with imported data, for example path, children, or dynamic stuff inside data
     *
     * const MY_ROUTES: Routes = [
     *     {
     *         path: 'home',
     *         component: HomeComponent
     *     },
     *     {
     *         path: PATHS.home,
     *         component: HomeComponent
     *     }
     * ];
     *
     * The initializer is an array (ArrayLiteralExpression - 177 ), it has elements, objects (ObjectLiteralExpression - 178)
     * with properties (PropertyAssignment - 261)
     *
     * For each know property (https://angular.io/api/router/Routes#description), we try to see if we have what we want
     *
     * Ex: path and pathMatch want a string, component a component reference.
     *
     * It is an imperative approach, not a generic way, parsing all the tree
     * and find something like this which willl break JSON.stringify : MYIMPORT.path
     *
     * @param  {ts.Node} initializer The node of routes definition
     * @return {ts.Node}             The edited node
     */
    RouterParserUtil.prototype.cleanRoutesDefinitionWithImport = function (initializer, node, sourceFile) {
        initializer.elements.forEach(function (element) {
            element.properties.forEach(function (property) {
                var propertyName = property.name.getText(), propertyInitializer = property.initializer;
                switch (propertyName) {
                    case 'path':
                    case 'redirectTo':
                    case 'outlet':
                    case 'pathMatch':
                        if (propertyInitializer) {
                            if (propertyInitializer.kind !== Ast.SyntaxKind.StringLiteral) {
                                // Identifier(71) won't break parsing, but it will be better to retrive them
                                // PropertyAccessExpression(179) ex: MYIMPORT.path will break it, find it in import
                                if (propertyInitializer.kind === Ast.SyntaxKind.PropertyAccessExpression) {
                                    var lastObjectLiteralAttributeName = propertyInitializer.name.getText(), firstObjectLiteralAttributeName = void 0;
                                    if (propertyInitializer.expression) {
                                        firstObjectLiteralAttributeName = propertyInitializer.expression.getText();
                                        var result = ImportsUtil$1.findPropertyValueInImportOrLocalVariables(firstObjectLiteralAttributeName +
                                            '.' +
                                            lastObjectLiteralAttributeName, sourceFile); // tslint:disable-line
                                        if (result !== '') {
                                            propertyInitializer.kind = 9;
                                            propertyInitializer.text = result;
                                        }
                                    }
                                }
                            }
                        }
                        break;
                }
            });
        });
        return initializer;
    };
    return RouterParserUtil;
}());
var RouterParserUtil$1 = RouterParserUtil.getInstance();

function isModuleWithProviders(node) {
    var result = false;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, declarations = node.declarationList.declarations, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = true;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function getModuleWithProviders(node) {
    var result;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = declaration.initializer;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function StringifyObjectLiteralExpression(ole) {
    var returnedString = '{';
    if (ole.properties && ole.properties.length > 0) {
        ole.properties.forEach(function (property, index) {
            if (property.name) {
                returnedString += property.name.text + ': ';
            }
            if (property.initializer) {
                if (property.initializer.kind === Ast.SyntaxKind.StringLiteral) {
                    returnedString += "'" + property.initializer.text + "'";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.TrueKeyword) {
                    returnedString += "true";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.FalseKeyword) {
                    returnedString += "false";
                }
                else {
                    returnedString += property.initializer.text;
                }
            }
            if (index < ole.properties.length - 1) {
                returnedString += ', ';
            }
        });
    }
    returnedString += '}';
    return returnedString;
}

var crypto = require('crypto');
var marked$1 = require('marked');
var ClassHelper = /** @class */ (function () {
    function ClassHelper(typeChecker) {
        this.typeChecker = typeChecker;
        this.jsdocParserUtil = new JsdocParserUtil();
    }
    /**
     * HELPERS
     */
    ClassHelper.prototype.stringifyDefaultValue = function (node) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (node.getText()) {
            return node.getText();
        }
        else if (node.kind === Ast.SyntaxKind.FalseKeyword) {
            return 'false';
        }
        else if (node.kind === Ast.SyntaxKind.TrueKeyword) {
            return 'true';
        }
    };
    ClassHelper.prototype.getDecoratorOfType = function (node, decoratorType) {
        var decorators = node.decorators || [];
        for (var i = 0; i < decorators.length; i++) {
            if (decorators[i].expression.expression) {
                if (decorators[i].expression.expression.text === decoratorType) {
                    return decorators[i];
                }
            }
        }
        return undefined;
    };
    ClassHelper.prototype.formatDecorators = function (decorators) {
        var _this = this;
        var _decorators = [];
        _.forEach(decorators, function (decorator) {
            if (decorator.expression) {
                if (decorator.expression.text) {
                    _decorators.push({ name: decorator.expression.text });
                }
                if (decorator.expression.expression) {
                    var info = { name: decorator.expression.expression.text };
                    if (decorator.expression.arguments) {
                        info.stringifiedArguments = _this.stringifyArguments(decorator.expression.arguments);
                    }
                    _decorators.push(info);
                }
            }
        });
        return _decorators;
    };
    ClassHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg.function.length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg.function.map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_1 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_1 = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_1 + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path_2 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_2 + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path_3 = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_3 + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    ClassHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    ClassHelper.prototype.stringifyArguments = function (args) {
        var _this = this;
        var stringifyArgs = [];
        stringifyArgs = args
            .map(function (arg) {
            var _result = DependenciesEngine$1.find(arg.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_4 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_4 = 'classe';
                    }
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_4 + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                }
                else {
                    var path_5 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_5 + "\" target=\"_blank\">" + arg.type + "</a>";
                }
            }
            else if (arg.dotDotDotToken) {
                return "..." + arg.name + ": " + arg.type;
            }
            else if (arg.function) {
                return _this.handleFunction(arg);
            }
            else if (arg.expression && arg.name) {
                return arg.expression.text + '.' + arg.name.text;
            }
            else if (arg.expression && arg.kind === Ast.SyntaxKind.NewExpression) {
                return 'new ' + arg.expression.text + '()';
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.StringLiteral) {
                return "'" + arg.text + "'";
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
                return StringifyObjectLiteralExpression(arg);
            }
            else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                var path_6 = BasicTypeUtil$1.getTypeUrl(arg.type);
                return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_6 + "\" target=\"_blank\">" + arg.type + "</a>";
            }
            else {
                if (arg.type) {
                    var finalStringifiedArgument = '';
                    var separator = ':';
                    if (arg.name) {
                        finalStringifiedArgument += arg.name;
                    }
                    if (arg.kind === Ast.SyntaxKind.AsExpression &&
                        arg.expression &&
                        arg.expression.text) {
                        finalStringifiedArgument += arg.expression.text;
                        separator = ' as';
                    }
                    if (arg.optional) {
                        finalStringifiedArgument += _this.getOptionalString(arg);
                    }
                    if (arg.type) {
                        finalStringifiedArgument += separator + ' ' + _this.visitType(arg.type);
                    }
                    return finalStringifiedArgument;
                }
                else if (arg.text) {
                    return "" + arg.text;
                }
                else {
                    return "" + arg.name + _this.getOptionalString(arg);
                }
            }
        })
            .join(', ');
        return stringifyArgs;
    };
    ClassHelper.prototype.getPosition = function (node, sourceFile) {
        var position;
        if (node.name && node.name.end) {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.name.end);
        }
        else {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.pos);
        }
        return position;
    };
    ClassHelper.prototype.addAccessor = function (accessors, nodeAccessor, sourceFile) {
        var nodeName = '';
        if (nodeAccessor.name) {
            nodeName = nodeAccessor.name.text;
            var jsdoctags = this.jsdocParserUtil.getJSDocs(nodeAccessor);
            if (!accessors[nodeName]) {
                accessors[nodeName] = {
                    name: nodeName,
                    setSignature: undefined,
                    getSignature: undefined
                };
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.SetAccessor) {
                var setSignature = {
                    name: nodeName,
                    type: 'void',
                    args: nodeAccessor.parameters.map(function (param) {
                        return {
                            name: param.name.text,
                            type: param.type ? kindToType(param.type.kind) : ''
                        };
                    }),
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : 'void',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        setSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        setSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                if (setSignature.jsdoctags && setSignature.jsdoctags.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args, setSignature.jsdoctags);
                }
                else if (setSignature.args && setSignature.args.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args);
                }
                accessors[nodeName].setSignature = setSignature;
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.GetAccessor) {
                var getSignature = {
                    name: nodeName,
                    type: nodeAccessor.type ? kindToType(nodeAccessor.type.kind) : '',
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : '',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        getSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        getSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                accessors[nodeName].getSignature = getSignature;
            }
        }
    };
    ClassHelper.prototype.isDirectiveDecorator = function (decorator) {
        if (decorator.expression.expression) {
            var decoratorIdentifierText = decorator.expression.expression.text;
            return (decoratorIdentifierText === 'Directive' || decoratorIdentifierText === 'Component');
        }
        else {
            return false;
        }
    };
    ClassHelper.prototype.isServiceDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Injectable'
            : false;
    };
    ClassHelper.prototype.isPrivate = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (member.modifiers) {
            var isPrivate = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PrivateKeyword; });
            if (isPrivate) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isProtected = function (member) {
        if (member.modifiers) {
            var isProtected = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.ProtectedKeyword; });
            if (isProtected) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isInternal = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['internal'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPublic = function (member) {
        if (member.modifiers) {
            var isPublic = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PublicKeyword; });
            if (isPublic) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isHiddenMember = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['hidden'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPipeDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Pipe'
            : false;
    };
    ClassHelper.prototype.isModuleDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'NgModule'
            : false;
    };
    /**
     * VISITERS
     */
    ClassHelper.prototype.visitClassDeclaration = function (fileName, classDeclaration, sourceFile) {
        var symbol = this.typeChecker.getSymbolAtLocation(classDeclaration.name);
        var rawdescription = '';
        var description = '';
        if (symbol) {
            rawdescription = this.jsdocParserUtil.getMainCommentOfNode(classDeclaration);
            description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(classDeclaration));
            if (symbol.valueDeclaration && isIgnore(symbol.valueDeclaration)) {
                return [{ ignore: true }];
            }
            if (symbol.declarations && symbol.declarations.length > 0) {
                if (isIgnore(symbol.declarations[0])) {
                    return [{ ignore: true }];
                }
            }
        }
        var className = classDeclaration.name.text;
        var members;
        var implementsElements = [];
        var extendsElement;
        var jsdoctags = [];
        if (typeof Ast.ts.getClassImplementsHeritageClauseElements !== 'undefined') {
            var implementedTypes = Ast.ts.getClassImplementsHeritageClauseElements(classDeclaration);
            if (implementedTypes) {
                var i = 0;
                var len = implementedTypes.length;
                for (i; i < len; i++) {
                    if (implementedTypes[i].expression) {
                        implementsElements.push(implementedTypes[i].expression.text);
                    }
                }
            }
        }
        if (typeof Ast.ts.getClassExtendsHeritageClauseElement !== 'undefined') {
            var extendsTypes = Ast.ts.getClassExtendsHeritageClauseElement(classDeclaration);
            if (extendsTypes) {
                if (extendsTypes.expression) {
                    extendsElement = extendsTypes.expression.text;
                }
            }
        }
        if (symbol) {
            if (symbol.valueDeclaration) {
                jsdoctags = this.jsdocParserUtil.getJSDocs(symbol.valueDeclaration);
            }
        }
        members = this.visitMembers(classDeclaration.members, sourceFile);
        if (classDeclaration.decorators) {
            for (var i = 0; i < classDeclaration.decorators.length; i++) {
                if (this.isDirectiveDecorator(classDeclaration.decorators[i])) {
                    return {
                        description: description,
                        rawdescription: rawdescription,
                        inputs: members.inputs,
                        outputs: members.outputs,
                        hostBindings: members.hostBindings,
                        hostListeners: members.hostListeners,
                        properties: members.properties,
                        methods: members.methods,
                        indexSignatures: members.indexSignatures,
                        kind: members.kind,
                        constructor: members.constructor,
                        jsdoctags: jsdoctags,
                        extends: extendsElement,
                        implements: implementsElements,
                        accessors: members.accessors
                    };
                }
                else if (this.isServiceDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            extends: extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
                else if (this.isPipeDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            properties: members.properties,
                            methods: members.methods
                        }
                    ];
                }
                else if (this.isModuleDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            methods: members.methods
                        }
                    ];
                }
                else {
                    return [
                        {
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            extends: extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
            }
        }
        else if (description) {
            return [
                {
                    description: description,
                    rawdescription: rawdescription,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    methods: members.methods,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    extends: extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        else {
            return [
                {
                    methods: members.methods,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    extends: extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        return [];
    };
    ClassHelper.prototype.visitMembers = function (members, sourceFile) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var inputs = [];
        var outputs = [];
        var hostBindings = [];
        var hostListeners = [];
        var methods = [];
        var properties = [];
        var indexSignatures = [];
        var kind;
        var inputDecorator;
        var hostBinding;
        var hostListener;
        var constructor;
        var outDecorator;
        var accessors = {};
        var result = {};
        for (var i = 0; i < members.length; i++) {
            // Allows typescript guess type when using ts.is*
            var member = members[i];
            inputDecorator = this.getDecoratorOfType(member, 'Input');
            outDecorator = this.getDecoratorOfType(member, 'Output');
            hostBinding = this.getDecoratorOfType(member, 'HostBinding');
            hostListener = this.getDecoratorOfType(member, 'HostListener');
            kind = member.kind;
            if (isIgnore(member)) {
                continue;
            }
            if (inputDecorator) {
                inputs.push(this.visitInputAndHostBinding(member, inputDecorator, sourceFile));
                if (Ast.ts.isSetAccessorDeclaration(member)) {
                    this.addAccessor(accessors, members[i], sourceFile);
                }
            }
            else if (outDecorator) {
                outputs.push(this.visitOutput(member, outDecorator, sourceFile));
            }
            else if (hostBinding) {
                hostBindings.push(this.visitInputAndHostBinding(member, hostBinding, sourceFile));
            }
            else if (hostListener) {
                hostListeners.push(this.visitHostListener(member, hostListener, sourceFile));
            }
            else if (!this.isHiddenMember(member)) {
                if (!(this.isPrivate(member) && Configuration$1.mainData.disablePrivate)) {
                    if (!(this.isInternal(member) && Configuration$1.mainData.disableInternal)) {
                        if (!(this.isProtected(member) && Configuration$1.mainData.disableProtected)) {
                            if (Ast.ts.isMethodDeclaration(member) || Ast.ts.isMethodSignature(member)) {
                                methods.push(this.visitMethodDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isPropertyDeclaration(member) ||
                                Ast.ts.isPropertySignature(member)) {
                                properties.push(this.visitProperty(member, sourceFile));
                            }
                            else if (Ast.ts.isCallSignatureDeclaration(member)) {
                                properties.push(this.visitCallDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isGetAccessorDeclaration(member) ||
                                Ast.ts.isSetAccessorDeclaration(member)) {
                                this.addAccessor(accessors, members[i], sourceFile);
                            }
                            else if (Ast.ts.isIndexSignatureDeclaration(member)) {
                                indexSignatures.push(this.visitIndexDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isConstructorDeclaration(member)) {
                                var _constructorProperties = this.visitConstructorProperties(member, sourceFile);
                                var j = 0;
                                var len = _constructorProperties.length;
                                for (j; j < len; j++) {
                                    properties.push(_constructorProperties[j]);
                                }
                                constructor = this.visitConstructorDeclaration(member, sourceFile);
                            }
                        }
                    }
                }
            }
        }
        inputs.sort(getNamesCompareFn());
        outputs.sort(getNamesCompareFn());
        hostBindings.sort(getNamesCompareFn());
        hostListeners.sort(getNamesCompareFn());
        properties.sort(getNamesCompareFn());
        methods.sort(getNamesCompareFn());
        indexSignatures.sort(getNamesCompareFn());
        result = {
            inputs: inputs,
            outputs: outputs,
            hostBindings: hostBindings,
            hostListeners: hostListeners,
            methods: methods,
            properties: properties,
            indexSignatures: indexSignatures,
            kind: kind,
            constructor: constructor
        };
        if (Object.keys(accessors).length) {
            result['accessors'] = accessors;
        }
        return result;
    };
    ClassHelper.prototype.visitTypeName = function (typeName) {
        if (typeName.text) {
            return typeName.text;
        }
        return this.visitTypeName(typeName.left) + "." + this.visitTypeName(typeName.right);
    };
    ClassHelper.prototype.visitType = function (node) {
        var _return = 'void';
        if (!node) {
            return _return;
        }
        if (node.typeName) {
            _return = this.visitTypeName(node.typeName);
        }
        else if (node.type) {
            if (node.type.kind) {
                _return = kindToType(node.type.kind);
            }
            if (node.type.typeName) {
                _return = this.visitTypeName(node.type.typeName);
            }
            if (node.type.typeArguments) {
                _return += '<';
                var typeArguments = [];
                for (var _i = 0, _a = node.type.typeArguments; _i < _a.length; _i++) {
                    var argument = _a[_i];
                    typeArguments.push(this.visitType(argument));
                }
                _return += typeArguments.join(' | ');
                _return += '>';
            }
            if (node.type.elementType) {
                var _firstPart = this.visitType(node.type.elementType);
                _return = _firstPart + kindToType(node.type.kind);
                if (node.type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                    _return = '(' + _firstPart + ')' + kindToType(node.type.kind);
                }
            }
            if (node.type.types && Ast.ts.isUnionTypeNode(node.type)) {
                _return = '';
                var i = 0;
                var len = node.type.types.length;
                for (i; i < len; i++) {
                    var type = node.type.types[i];
                    if (type.elementType) {
                        var _firstPart = this.visitType(type.elementType);
                        if (type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                            _return += '(' + _firstPart + ')' + kindToType(type.kind);
                        }
                        else {
                            _return += _firstPart + kindToType(type.kind);
                        }
                    }
                    else {
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (type.typeArguments) {
                            _return += '<';
                            var typeArguments = [];
                            for (var _b = 0, _c = type.typeArguments; _b < _c.length; _b++) {
                                var argument = _c[_b];
                                typeArguments.push(this.visitType(argument));
                            }
                            _return += typeArguments.join(' | ');
                            _return += '>';
                        }
                    }
                    if (i < len - 1) {
                        _return += ' | ';
                    }
                }
            }
            if (node.type.elementTypes) {
                var elementTypes = node.type.elementTypes;
                var i = 0;
                var len = elementTypes.length;
                if (len > 0) {
                    _return = '[';
                    for (i; i < len; i++) {
                        var type = elementTypes[i];
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (i < len - 1) {
                            _return += ', ';
                        }
                    }
                    _return += ']';
                }
            }
        }
        else if (node.elementType) {
            _return = kindToType(node.elementType.kind) + kindToType(node.kind);
            if (node.elementType.typeName) {
                _return = this.visitTypeName(node.elementType.typeName) + kindToType(node.kind);
            }
        }
        else if (node.types && Ast.ts.isUnionTypeNode(node)) {
            _return = '';
            var i = 0;
            var len = node.types.length;
            for (i; i < len; i++) {
                var type = node.types[i];
                _return += kindToType(type.kind);
                if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                    _return += '"' + type.literal.text + '"';
                }
                if (type.typeName) {
                    _return += this.visitTypeName(type.typeName);
                }
                if (i < len - 1) {
                    _return += ' | ';
                }
            }
        }
        else if (node.dotDotDotToken) {
            _return = 'any[]';
        }
        else {
            _return = kindToType(node.kind);
            if (_return === '' &&
                node.initializer &&
                node.initializer.kind &&
                (node.kind === Ast.SyntaxKind.PropertyDeclaration || node.kind === Ast.SyntaxKind.Parameter)) {
                _return = kindToType(node.initializer.kind);
            }
            if (node.kind === Ast.SyntaxKind.TypeParameter) {
                _return = node.name.text;
            }
            if (node.kind === Ast.SyntaxKind.LiteralType) {
                _return = node.literal.text;
            }
        }
        if (node.typeArguments && node.typeArguments.length > 0) {
            _return += '<';
            var i = 0, len = node.typeArguments.length;
            for (i; i < len; i++) {
                var argument = node.typeArguments[i];
                _return += this.visitType(argument);
                if (i >= 0 && i < len - 1) {
                    _return += ', ';
                }
            }
            _return += '>';
        }
        return _return;
    };
    ClassHelper.prototype.visitCallDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'call-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(method)));
        }
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitIndexDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'index-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorDeclaration = function (method, sourceFile) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var result = {
            name: 'constructor',
            description: '',
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitProperty = function (property, sourceFile) {
        var result = {
            name: property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined,
            type: this.visitType(property),
            optional: typeof property.questionToken !== 'undefined',
            description: '',
            line: this.getPosition(property, sourceFile).line + 1
        };
        var jsdoctags;
        if (property.initializer && property.initializer.kind === Ast.SyntaxKind.ArrowFunction) {
            result.defaultValue = '() => {...}';
        }
        if (typeof result.name === 'undefined' && typeof property.name.expression !== 'undefined') {
            result.name = property.name.expression.text;
        }
        if (property.jsDoc) {
            jsdoctags = this.jsdocParserUtil.getJSDocs(property);
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (property.decorators) {
            result.decorators = this.formatDecorators(property.decorators);
        }
        if (property.modifiers) {
            if (property.modifiers.length > 0) {
                var kinds = property.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorProperties = function (constr, sourceFile) {
        if (constr.parameters) {
            var _parameters_1 = [];
            var i = 0;
            var len = constr.parameters.length;
            for (i; i < len; i++) {
                if (this.isPublic(constr.parameters[i])) {
                    _parameters_1.push(this.visitProperty(constr.parameters[i], sourceFile));
                }
            }
            /**
             * Merge JSDoc tags description from constructor with parameters
             */
            if (constr.jsDoc) {
                if (constr.jsDoc.length > 0) {
                    var constrTags = constr.jsDoc[0].tags;
                    if (constrTags && constrTags.length > 0) {
                        constrTags.forEach(function (tag) {
                            _parameters_1.forEach(function (param) {
                                if (tag.tagName &&
                                    tag.tagName.escapedText &&
                                    tag.tagName.escapedText === 'param') {
                                    if (tag.name &&
                                        tag.name.escapedText &&
                                        tag.name.escapedText === param.name) {
                                        param.description = tag.comment;
                                    }
                                }
                            });
                        });
                    }
                }
            }
            return _parameters_1;
        }
        else {
            return [];
        }
    };
    ClassHelper.prototype.visitInputAndHostBinding = function (property, inDecorator, sourceFile) {
        var inArgs = inDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.defaultValue = property.initializer
            ? this.stringifyDefaultValue(property.initializer)
            : undefined;
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        if (property.kind === Ast.SyntaxKind.SetAccessor) {
            // For setter accessor, find type in first parameter
            if (property.parameters && property.parameters.length === 1) {
                if (property.parameters[0].type) {
                    _return.type = kindToType(property.parameters[0].type.kind);
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitMethodDeclaration = function (method, sourceFile) {
        var _this = this;
        var result = {
            name: method.name.text,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            optional: typeof method.questionToken !== 'undefined',
            returnType: this.visitType(method.type),
            typeParameters: [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type === 'undefined') {
            // Try to get inferred type
            if (method.symbol) {
                var symbol = method.symbol;
                if (symbol.valueDeclaration) {
                    var symbolType = this.typeChecker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration);
                    if (symbolType) {
                        try {
                            var signature = this.typeChecker.getSignatureFromDeclaration(method);
                            var returnType = signature.getReturnType();
                            result.returnType = this.typeChecker.typeToString(returnType);
                            // tslint:disable-next-line:no-empty
                        }
                        catch (error) { }
                    }
                }
            }
        }
        if (method.typeParameters && method.typeParameters.length > 0) {
            result.typeParameters = method.typeParameters.map(function (typeParameter) {
                return _this.visitType(typeParameter);
            });
        }
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.decorators) {
            result.decorators = this.formatDecorators(method.decorators);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitOutput = function (property, outDecorator, sourceFile) {
        var inArgs = outDecorator.expression.arguments;
        var _return = {
            name: inArgs.length > 0 ? inArgs[0].text : property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined
        };
        if (property.jsDoc) {
            _return.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(property)));
        }
        if (!_return.description) {
            if (property.jsDoc && property.jsDoc.length > 0) {
                if (typeof property.jsDoc[0].comment !== 'undefined') {
                    _return.description = marked$1(property.jsDoc[0].comment);
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitArgument = function (arg) {
        var _this = this;
        var _result = { name: arg.name.text, type: this.visitType(arg) };
        if (arg.dotDotDotToken) {
            _result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            _result.optional = true;
        }
        if (arg.type) {
            if (arg.type.kind) {
                if (Ast.ts.isFunctionTypeNode(arg.type)) {
                    _result.function = arg.type.parameters
                        ? arg.type.parameters.map(function (prop) { return _this.visitArgument(prop); })
                        : [];
                }
            }
        }
        if (arg.initializer) {
            _result.defaultValue = this.stringifyDefaultValue(arg.initializer);
        }
        return _result;
    };
    ClassHelper.prototype.visitHostListener = function (property, hostListenerDecorator, sourceFile) {
        var _this = this;
        var inArgs = hostListenerDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.args = property.parameters
            ? property.parameters.map(function (prop) { return _this.visitArgument(prop); })
            : [];
        _return.argsDecorator =
            inArgs.length > 1
                ? inArgs[1].elements.map(function (prop) {
                    return prop.text;
                })
                : [];
        if (property.jsDoc) {
            _return.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        return _return;
    };
    return ClassHelper;
}());

var TsPrinterUtil = /** @class */ (function () {
    function TsPrinterUtil() {
        this.printer = Ast.ts.createPrinter({
            newLine: Ast.ts.NewLineKind.LineFeed
        });
    }
    TsPrinterUtil.prototype.print = function (node) {
        return this.printer.printNode(Ast.ts.EmitHint.Unspecified, node, Ast.ts.createSourceFile('', '', Ast.ts.ScriptTarget.Latest));
    };
    return TsPrinterUtil;
}());

var SymbolHelper = /** @class */ (function () {
    function SymbolHelper() {
        this.unknown = '???';
    }
    SymbolHelper.prototype.parseDeepIndentifier = function (name, srcFile) {
        var result = {
            name: '',
            type: ''
        };
        if (typeof name === 'undefined') {
            return result;
        }
        var nsModule = name.split('.');
        var type = this.getType(name);
        if (nsModule.length > 1) {
            result.ns = nsModule[0];
            result.name = name;
            result.type = type;
            return result;
        }
        if (typeof srcFile !== 'undefined') {
            result.file = ImportsUtil$1.getFileNameOfImport(name, srcFile);
        }
        result.name = name;
        result.type = type;
        return result;
    };
    SymbolHelper.prototype.getType = function (name) {
        var type;
        if (name.toLowerCase().indexOf('component') !== -1) {
            type = 'component';
        }
        else if (name.toLowerCase().indexOf('pipe') !== -1) {
            type = 'pipe';
        }
        else if (name.toLowerCase().indexOf('controller') !== -1) {
            type = 'controller';
        }
        else if (name.toLowerCase().indexOf('module') !== -1) {
            type = 'module';
        }
        else if (name.toLowerCase().indexOf('directive') !== -1) {
            type = 'directive';
        }
        return type;
    };
    /**
     * Output
     * RouterModule.forRoot 179
     */
    SymbolHelper.prototype.buildIdentifierName = function (node, name) {
        if (Ast.ts.isIdentifier(node) && !Ast.ts.isPropertyAccessExpression(node)) {
            return node.text + "." + name;
        }
        name = name ? "." + name : '';
        var nodeName = this.unknown;
        if (node.name) {
            nodeName = node.name.text;
        }
        else if (node.text) {
            nodeName = node.text;
        }
        else if (node.expression) {
            if (node.expression.text) {
                nodeName = node.expression.text;
            }
            else if (node.expression.elements) {
                if (Ast.ts.isArrayLiteralExpression(node.expression)) {
                    nodeName = node.expression.elements.map(function (el) { return el.text; }).join(', ');
                    nodeName = "[" + nodeName + "]";
                }
            }
        }
        if (Ast.ts.isSpreadElement(node)) {
            return "..." + nodeName;
        }
        return "" + this.buildIdentifierName(node.expression, nodeName) + name;
    };
    /**
     * parse expressions such as:
     * { provide: APP_BASE_HREF, useValue: '/' }
     * { provide: 'Date', useFactory: (d1, d2) => new Date(), deps: ['d1', 'd2'] }
     */
    SymbolHelper.prototype.parseProviderConfiguration = function (node) {
        if (node.kind && node.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
            // Search for provide: HTTP_INTERCEPTORS
            // and if true, return type: 'interceptor' + name
            var interceptorName_1, hasInterceptor_1;
            if (node.properties) {
                if (node.properties.length > 0) {
                    _.forEach(node.properties, function (property) {
                        if (property.kind && property.kind === Ast.SyntaxKind.PropertyAssignment) {
                            if (property.name.text === 'provide') {
                                if (property.initializer.text === 'HTTP_INTERCEPTORS') {
                                    hasInterceptor_1 = true;
                                }
                            }
                            if (property.name.text === 'useClass' ||
                                property.name.text === 'useExisting') {
                                interceptorName_1 = property.initializer.text;
                            }
                        }
                    });
                }
            }
            if (hasInterceptor_1) {
                return interceptorName_1;
            }
            else {
                return new TsPrinterUtil().print(node);
            }
        }
        else {
            return new TsPrinterUtil().print(node);
        }
    };
    /**
     * Kind
     *  181 CallExpression => "RouterModule.forRoot(args)"
     *   71 Identifier     => "RouterModule" "TodoStore"
     *    9 StringLiteral  => "./app.component.css" "./tab.scss"
     */
    SymbolHelper.prototype.parseSymbolElements = function (node) {
        // parse expressions such as: AngularFireModule.initializeApp(firebaseConfig)
        // if (ts.isCallExpression(node) && ts.isPropertyAccessExpression(node.expression)) {
        if ((Ast.ts.isCallExpression(node) && Ast.ts.isPropertyAccessExpression(node.expression)) ||
            (Ast.ts.isNewExpression(node) && Ast.ts.isElementAccessExpression(node.expression))) {
            var className = this.buildIdentifierName(node.expression);
            // function arguments could be really complex. There are so
            // many use cases that we can't handle. Just print "args" to indicate
            // that we have arguments.
            var functionArgs = node.arguments.length > 0 ? 'args' : '';
            var text = className + "(" + functionArgs + ")";
            return text;
        }
        else if (Ast.ts.isPropertyAccessExpression(node)) {
            // parse expressions such as: Shared.Module
            return this.buildIdentifierName(node);
        }
        else if (Ast.ts.isIdentifier(node)) {
            // parse expressions such as: MyComponent
            if (node.text) {
                return node.text;
            }
            if (node.escapedText) {
                return node.escapedText;
            }
        }
        else if (Ast.ts.isSpreadElement(node)) {
            // parse expressions such as: ...MYARRAY
            // Resolve MYARRAY in imports or local file variables after full scan, just return the name of the variable
            if (node.expression && node.expression.text) {
                return node.expression.text;
            }
        }
        return node.text ? node.text : this.parseProviderConfiguration(node);
    };
    /**
     * Kind
     *  177 ArrayLiteralExpression
     *  122 BooleanKeyword
     *    9 StringLiteral
     */
    SymbolHelper.prototype.parseSymbols = function (node, srcFile) {
        var _this = this;
        var localNode = node;
        if (Ast.ts.isShorthandPropertyAssignment(localNode)) {
            localNode = ImportsUtil$1.findValueInImportOrLocalVariables(node.name.text, srcFile);
        }
        if (Ast.ts.isArrayLiteralExpression(localNode.initializer)) {
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
        else if (Ast.ts.isStringLiteral(localNode.initializer) ||
            Ast.ts.isTemplateLiteral(localNode.initializer) ||
            (Ast.ts.isPropertyAssignment(localNode) && localNode.initializer.text)) {
            return [localNode.initializer.text];
        }
        else if (localNode.initializer.kind &&
            (localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ||
                localNode.initializer.kind === Ast.SyntaxKind.FalseKeyword)) {
            return [localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ? true : false];
        }
        else if (Ast.ts.isPropertyAccessExpression(localNode.initializer)) {
            var identifier = this.parseSymbolElements(localNode.initializer);
            return [identifier];
        }
        else if (localNode.initializer &&
            localNode.initializer.elements &&
            localNode.initializer.elements.length > 0) {
            // Node replaced by ts-simple-ast & kind = 265
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
    };
    SymbolHelper.prototype.getSymbolDeps = function (props, type, srcFile, multiLine) {
        var _this = this;
        if (props.length === 0) {
            return [];
        }
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseSymbols(x, srcFile); }).pop() || [];
    };
    SymbolHelper.prototype.getSymbolDepsRaw = function (props, type, multiLine) {
        return props.filter(function (node) { return node.name.text === type; });
    };
    return SymbolHelper;
}());

var ComponentHelper = /** @class */ (function () {
    function ComponentHelper(classHelper, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.classHelper = classHelper;
        this.symbolHelper = symbolHelper;
    }
    ComponentHelper.prototype.getComponentChangeDetection = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'changeDetection', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentEncapsulation = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'encapsulation', srcFile);
    };
    ComponentHelper.prototype.getComponentPure = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'pure', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentName = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'name', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentExportAs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'exportAs', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentHost = function (props) {
        return this.getSymbolDepsObject(props, 'host');
    };
    ComponentHelper.prototype.getComponentTag = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'tag', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentInputsMetadata = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'inputs', srcFile);
    };
    ComponentHelper.prototype.getComponentTemplate = function (props, srcFile) {
        var t = this.symbolHelper.getSymbolDeps(props, 'template', srcFile, true).pop();
        if (t) {
            t = detectIndent(t, 0);
            t = t.replace(/\n/, '');
            t = t.replace(/ +$/gm, '');
        }
        return t;
    };
    ComponentHelper.prototype.getComponentStyleUrls = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrls', srcFile);
    };
    ComponentHelper.prototype.getComponentStyleUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrl', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentShadow = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'shadow', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentScoped = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'scoped', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDir = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDirs = function (props, srcFile) {
        return this.sanitizeUrls(this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile));
    };
    ComponentHelper.prototype.getComponentStyles = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styles', srcFile);
    };
    ComponentHelper.prototype.getComponentModuleId = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'moduleId', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentOutputs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'outputs', srcFile);
    };
    ComponentHelper.prototype.getComponentProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'entryComponents', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentViewProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'viewProviders', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentTemplateUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'templateUrl', srcFile);
    };
    ComponentHelper.prototype.getComponentExampleUrls = function (text) {
        var exampleUrlsMatches = text.match(/<example-url>(.*?)<\/example-url>/g);
        var exampleUrls = undefined;
        if (exampleUrlsMatches && exampleUrlsMatches.length) {
            exampleUrls = exampleUrlsMatches.map(function (val) {
                return val.replace(/<\/?example-url>/g, '');
            });
        }
        return exampleUrls;
    };
    ComponentHelper.prototype.getComponentPreserveWhitespaces = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'preserveWhitespaces', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentSelector = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'selector', srcFile).pop();
    };
    ComponentHelper.prototype.parseProperties = function (node) {
        var obj = new Map();
        var properties = node.initializer.properties || [];
        properties.forEach(function (prop) {
            obj.set(prop.name.text, prop.initializer.text);
        });
        return obj;
    };
    ComponentHelper.prototype.getSymbolDepsObject = function (props, type, multiLine) {
        var _this = this;
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseProperties(x); }).pop();
    };
    ComponentHelper.prototype.getComponentIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    ComponentHelper.prototype.sanitizeUrls = function (urls) {
        return urls.map(function (url) { return url.replace('./', ''); });
    };
    return ComponentHelper;
}());
var ComponentCache = /** @class */ (function () {
    function ComponentCache() {
        this.cache = new Map();
    }
    ComponentCache.prototype.get = function (key) {
        return this.cache.get(key);
    };
    ComponentCache.prototype.set = function (key, value) {
        this.cache.set(key, value);
    };
    return ComponentCache;
}());

var FrameworkDependencies = /** @class */ (function () {
    function FrameworkDependencies(files, options) {
        this.files = files;
        var transpileOptions = {
            target: Ast.ts.ScriptTarget.ES5,
            module: Ast.ts.ModuleKind.CommonJS,
            tsconfigDirectory: options.tsconfigDirectory,
            allowJs: true
        };
        this.program = Ast.ts.createProgram(this.files, transpileOptions, compilerHost(transpileOptions));
        this.typeChecker = this.program.getTypeChecker();
        this.classHelper = new ClassHelper(this.typeChecker);
        this.componentHelper = new ComponentHelper(this.classHelper);
    }
    return FrameworkDependencies;
}());

var ExtendsMerger = /** @class */ (function () {
    function ExtendsMerger() {
    }
    ExtendsMerger.getInstance = function () {
        if (!ExtendsMerger.instance) {
            ExtendsMerger.instance = new ExtendsMerger();
        }
        return ExtendsMerger.instance;
    };
    ExtendsMerger.prototype.merge = function (deps) {
        var _this = this;
        this.components = deps.components;
        this.classes = deps.classes;
        this.injectables = deps.injectables;
        this.components.forEach(function (component) {
            var ext;
            if (typeof component.extends !== 'undefined') {
                ext = _this.findInDependencies(component.extends);
                if (ext) {
                    var recursiveScanWithInheritance_1 = function (cls) {
                        // From class to component
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        // From component to component
                        if (typeof cls.inputsClass !== 'undefined' && cls.inputsClass.length > 0) {
                            var newInputs = _.cloneDeep(cls.inputsClass);
                            newInputs = _this.markInheritance(newInputs, cls);
                            if (typeof component.inputsClass !== 'undefined') {
                                component.inputsClass = component.inputsClass.concat(newInputs);
                            }
                        }
                        if (typeof cls.outputsClass !== 'undefined' &&
                            cls.outputsClass.length > 0) {
                            var newOutputs = _.cloneDeep(cls.outputsClass);
                            newOutputs = _this.markInheritance(newOutputs, cls);
                            if (typeof component.outputsClass !== 'undefined') {
                                component.outputsClass = component.outputsClass.concat(newOutputs);
                            }
                        }
                        if (typeof cls.methodsClass !== 'undefined' &&
                            cls.methodsClass.length > 0) {
                            var newMethods = _.cloneDeep(cls.methodsClass);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.propertiesClass !== 'undefined' &&
                            cls.propertiesClass.length > 0) {
                            var newProperties = _.cloneDeep(cls.propertiesClass);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        if (typeof cls.hostBindings !== 'undefined' &&
                            cls.hostBindings.length > 0) {
                            var newHostBindings = _.cloneDeep(cls.hostBindings);
                            newHostBindings = _this.markInheritance(newHostBindings, cls);
                            if (typeof component.hostBindings !== 'undefined') {
                                component.hostBindings = component.hostBindings.concat(newHostBindings);
                            }
                        }
                        if (typeof cls.hostListeners !== 'undefined' &&
                            cls.hostListeners.length > 0) {
                            var newHostListeners = _.cloneDeep(cls.hostListeners);
                            newHostListeners = _this.markInheritance(newHostListeners, cls);
                            if (typeof component.hostListeners !== 'undefined') {
                                component.hostListeners = component.hostListeners.concat(newHostListeners);
                            }
                        }
                        if (Configuration$1.mainData.disableLifeCycleHooks) {
                            component.methodsClass = cleanLifecycleHooksFromMethods(component.methodsClass);
                        }
                        if (cls.extends) {
                            recursiveScanWithInheritance_1(_this.findInDependencies(cls.extends));
                        }
                    };
                    // From class to class
                    recursiveScanWithInheritance_1(ext);
                }
            }
        });
        var mergeExtendedClasses = function (el) {
            var ext;
            if (typeof el.extends !== 'undefined') {
                ext = _this.findInDependencies(el.extends);
                if (ext) {
                    var recursiveScanWithInheritance_2 = function (cls) {
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof el.methods !== 'undefined') {
                                el.methods = el.methods.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof el.properties !== 'undefined') {
                                el.properties = el.properties.concat(newProperties);
                            }
                        }
                        if (cls.extends) {
                            recursiveScanWithInheritance_2(_this.findInDependencies(cls.extends));
                        }
                    };
                    // From elss to elss
                    recursiveScanWithInheritance_2(ext);
                }
            }
        };
        this.classes.forEach(mergeExtendedClasses);
        this.injectables.forEach(mergeExtendedClasses);
        return deps;
    };
    ExtendsMerger.prototype.markInheritance = function (data, originalource) {
        return data.map(function (el) {
            var newElement = el;
            newElement.inheritance = {
                file: originalource.name
            };
            return newElement;
        });
    };
    ExtendsMerger.prototype.findInDependencies = function (name) {
        var mergedData = _.concat([], this.components, this.classes, this.injectables);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    return ExtendsMerger;
}());
var ExtendsMerger$1 = ExtendsMerger.getInstance();

var CodeGenerator = /** @class */ (function () {
    function CodeGenerator() {
    }
    CodeGenerator.prototype.generate = function (node) {
        return this.visitAndRecognize(node, []).join('');
    };
    CodeGenerator.prototype.visitAndRecognize = function (node, code, depth) {
        var _this = this;
        if (depth === void 0) { depth = 0; }
        this.recognize(node, code);
        node.getChildren().forEach(function (c) { return _this.visitAndRecognize(c, code, depth + 1); });
        return code;
    };
    CodeGenerator.prototype.recognize = function (node, code) {
        var _this = this;
        var conversion = TsKindConversion.find(function (x) { return x.kinds.some(function (z) { return z === node.kind; }); });
        if (conversion) {
            var result = conversion.output(node);
            result.forEach(function (text) { return _this.gen(text, code); });
        }
    };
    CodeGenerator.prototype.gen = function (token, code) {
        if (!token) {
            return;
        }
        if (token === '\n') {
            code.push('');
        }
        else {
            code.push(token);
        }
    };
    return CodeGenerator;
}());
var TsKindsToText = /** @class */ (function () {
    function TsKindsToText(output, kinds) {
        this.output = output;
        this.kinds = kinds;
    }
    return TsKindsToText;
}());
var TsKindConversion = [
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [
        Ast.SyntaxKind.FirstLiteralToken,
        Ast.SyntaxKind.Identifier
    ]),
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [Ast.SyntaxKind.StringLiteral]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.ArrayLiteralExpression]),
    new TsKindsToText(function (node) { return ['import', ' ']; }, [Ast.SyntaxKind.ImportKeyword]),
    new TsKindsToText(function (node) { return ['from', ' ']; }, [Ast.SyntaxKind.FromKeyword]),
    new TsKindsToText(function (node) { return ['\n', 'export', ' ']; }, [Ast.SyntaxKind.ExportKeyword]),
    new TsKindsToText(function (node) { return ['class', ' ']; }, [Ast.SyntaxKind.ClassKeyword]),
    new TsKindsToText(function (node) { return ['this']; }, [Ast.SyntaxKind.ThisKeyword]),
    new TsKindsToText(function (node) { return ['constructor']; }, [Ast.SyntaxKind.ConstructorKeyword]),
    new TsKindsToText(function (node) { return ['false']; }, [Ast.SyntaxKind.FalseKeyword]),
    new TsKindsToText(function (node) { return ['true']; }, [Ast.SyntaxKind.TrueKeyword]),
    new TsKindsToText(function (node) { return ['null']; }, [Ast.SyntaxKind.NullKeyword]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.AtToken]),
    new TsKindsToText(function (node) { return ['+']; }, [Ast.SyntaxKind.PlusToken]),
    new TsKindsToText(function (node) { return [' => ']; }, [Ast.SyntaxKind.EqualsGreaterThanToken]),
    new TsKindsToText(function (node) { return ['(']; }, [Ast.SyntaxKind.OpenParenToken]),
    new TsKindsToText(function (node) { return ['{', ' ']; }, [
        Ast.SyntaxKind.ImportClause,
        Ast.SyntaxKind.ObjectLiteralExpression
    ]),
    new TsKindsToText(function (node) { return ['{', '\n']; }, [Ast.SyntaxKind.Block]),
    new TsKindsToText(function (node) { return ['}']; }, [Ast.SyntaxKind.CloseBraceToken]),
    new TsKindsToText(function (node) { return [')']; }, [Ast.SyntaxKind.CloseParenToken]),
    new TsKindsToText(function (node) { return ['[']; }, [Ast.SyntaxKind.OpenBracketToken]),
    new TsKindsToText(function (node) { return [']']; }, [Ast.SyntaxKind.CloseBracketToken]),
    new TsKindsToText(function (node) { return [';', '\n']; }, [Ast.SyntaxKind.SemicolonToken]),
    new TsKindsToText(function (node) { return [',', ' ']; }, [Ast.SyntaxKind.CommaToken]),
    new TsKindsToText(function (node) { return [' ', ':', ' ']; }, [Ast.SyntaxKind.ColonToken]),
    new TsKindsToText(function (node) { return ['.']; }, [Ast.SyntaxKind.DotToken]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.DoStatement]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.Decorator]),
    new TsKindsToText(function (node) { return [' = ']; }, [Ast.SyntaxKind.FirstAssignment]),
    new TsKindsToText(function (node) { return [' ']; }, [Ast.SyntaxKind.FirstPunctuation]),
    new TsKindsToText(function (node) { return ['private', ' ']; }, [Ast.SyntaxKind.PrivateKeyword]),
    new TsKindsToText(function (node) { return ['public', ' ']; }, [Ast.SyntaxKind.PublicKeyword])
];

var crypto$1 = require('crypto');
var ComponentDepFactory = /** @class */ (function () {
    function ComponentDepFactory(helper) {
        this.helper = helper;
    }
    ComponentDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        // console.log(util.inspect(props, { showHidden: true, depth: 10 }));
        var sourceCode = srcFile.getText();
        var hash = crypto$1
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var componentDep = {
            name: name,
            id: 'component-' + name + '-' + hash,
            file: file,
            // animations?: string[]; // TODO
            changeDetection: this.helper.getComponentChangeDetection(props, srcFile),
            encapsulation: this.helper.getComponentEncapsulation(props, srcFile),
            entryComponents: this.helper.getComponentEntryComponents(props, srcFile),
            exportAs: this.helper.getComponentExportAs(props, srcFile),
            host: this.helper.getComponentHost(props),
            inputs: this.helper.getComponentInputsMetadata(props, srcFile),
            // interpolation?: string; // TODO waiting doc infos
            moduleId: this.helper.getComponentModuleId(props, srcFile),
            outputs: this.helper.getComponentOutputs(props, srcFile),
            providers: this.helper.getComponentProviders(props, srcFile),
            // queries?: Deps[]; // TODO
            selector: this.helper.getComponentSelector(props, srcFile),
            styleUrls: this.helper.getComponentStyleUrls(props, srcFile),
            styles: this.helper.getComponentStyles(props, srcFile),
            template: this.helper.getComponentTemplate(props, srcFile),
            templateUrl: this.helper.getComponentTemplateUrl(props, srcFile),
            viewProviders: this.helper.getComponentViewProviders(props, srcFile),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            description: IO.description,
            rawdescription: IO.rawdescription,
            type: 'component',
            sourceCode: srcFile.getText(),
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText()),
            tag: this.helper.getComponentTag(props, srcFile),
            styleUrl: this.helper.getComponentStyleUrl(props, srcFile),
            shadow: this.helper.getComponentShadow(props, srcFile),
            scoped: this.helper.getComponentScoped(props, srcFile),
            assetsDir: this.helper.getComponentAssetsDir(props, srcFile),
            assetsDirs: this.helper.getComponentAssetsDirs(props, srcFile),
            styleUrlsData: '',
            stylesData: ''
        };
        if (typeof this.helper.getComponentPreserveWhitespaces(props, srcFile) !== 'undefined') {
            componentDep.preserveWhitespaces = this.helper.getComponentPreserveWhitespaces(props, srcFile);
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            componentDep.methodsClass = cleanLifecycleHooksFromMethods(componentDep.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            componentDep.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.constructor) {
            componentDep.constructorObj = IO.constructor;
        }
        if (IO.extends) {
            componentDep.extends = IO.extends;
        }
        if (IO.implements && IO.implements.length > 0) {
            componentDep.implements = IO.implements;
        }
        if (IO.accessors) {
            componentDep.accessors = IO.accessors;
        }
        return componentDep;
    };
    return ComponentDepFactory;
}());

var crypto$2 = require('crypto');
var ControllerDepFactory = /** @class */ (function () {
    function ControllerDepFactory() {
    }
    ControllerDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$2
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var infos = {
            name: name,
            id: 'controller-' + name + '-' + hash,
            file: file,
            methodsClass: IO.methods,
            type: 'controller',
            description: IO.description,
            sourceCode: srcFile.text
        };
        if (properties && properties.length === 1) {
            if (properties[0].text) {
                infos.prefix = properties[0].text;
            }
        }
        return infos;
    };
    return ControllerDepFactory;
}());

var crypto$3 = require('crypto');
var DirectiveDepFactory = /** @class */ (function () {
    function DirectiveDepFactory(helper) {
        this.helper = helper;
    }
    DirectiveDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$3
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var directiveDeps = {
            name: name,
            id: 'directive-' + name + '-' + hash,
            file: file,
            type: 'directive',
            description: IO.description,
            sourceCode: srcFile.getText(),
            selector: this.helper.getComponentSelector(props),
            providers: this.helper.getComponentProviders(props),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText())
        };
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            directiveDeps.methodsClass = cleanLifecycleHooksFromMethods(directiveDeps.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            directiveDeps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.implements && IO.implements.length > 0) {
            directiveDeps.implements = IO.implements;
        }
        if (IO.constructor) {
            directiveDeps.constructorObj = IO.constructor;
        }
        if (IO.accessors) {
            directiveDeps.accessors = IO.accessors;
        }
        return directiveDeps;
    };
    return DirectiveDepFactory;
}());

var JsDocHelper = /** @class */ (function () {
    function JsDocHelper() {
    }
    JsDocHelper.prototype.hasJSDocInternalTag = function (filename, sourceFile, node) {
        if (typeof sourceFile.statements !== 'undefined') {
            return this.checkStatements(sourceFile.statements, node);
        }
        return false;
    };
    JsDocHelper.prototype.checkStatements = function (statements, node) {
        var _this = this;
        return statements.some(function (x) { return _this.checkStatement(x, node); });
    };
    JsDocHelper.prototype.checkStatement = function (statement, node) {
        if (statement.pos === node.pos && statement.end === node.end) {
            if (node.jsDoc && node.jsDoc.length > 0) {
                return this.checkJsDocs(node.jsDoc);
            }
        }
        return false;
    };
    JsDocHelper.prototype.checkJsDocs = function (jsDocs) {
        var _this = this;
        return jsDocs
            .filter(function (x) { return x.tags && x.tags.length > 0; })
            .some(function (x) { return _this.checkJsDocTags(x.tags); });
    };
    JsDocHelper.prototype.checkJsDocTags = function (tags) {
        return tags.some(function (x) { return x.tagName && x.tagName.text === 'internal'; });
    };
    return JsDocHelper;
}());

var ModuleHelper = /** @class */ (function () {
    function ModuleHelper(cache, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.cache = cache;
        this.symbolHelper = symbolHelper;
    }
    ModuleHelper.prototype.getModuleProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleControllers = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'controllers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleDeclarations = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'declarations', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.getModuleEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'entryComponents', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.cleanImportForRootForChild = function (name) {
        var nsModule = name.split('.');
        if (nsModule.length > 0) {
            name = nsModule[0];
        }
        return name;
    };
    ModuleHelper.prototype.getModuleImports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'imports', srcFile)
            .map(function (name) { return _this.cleanImportForRootForChild(name); })
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ModuleHelper.prototype.getModuleExports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'exports', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    ModuleHelper.prototype.getModuleImportsRaw = function (props, srcFile) {
        return this.symbolHelper.getSymbolDepsRaw(props, 'imports');
    };
    ModuleHelper.prototype.getModuleId = function (props, srcFile) {
        var _id = this.symbolHelper.getSymbolDeps(props, 'id', srcFile), id;
        if (_id.length === 1) {
            id = _id[0];
        }
        return id;
    };
    ModuleHelper.prototype.getModuleSchemas = function (props, srcFile) {
        var schemas = this.symbolHelper.getSymbolDeps(props, 'schemas', srcFile);
        return schemas;
    };
    ModuleHelper.prototype.getModuleBootstrap = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'bootstrap', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    return ModuleHelper;
}());

var crypto$4 = require('crypto');
var ModuleDepFactory = /** @class */ (function () {
    function ModuleDepFactory(moduleHelper) {
        this.moduleHelper = moduleHelper;
    }
    ModuleDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$4
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        return {
            name: name,
            id: 'module-' + name + '-' + hash,
            file: file,
            ngid: this.moduleHelper.getModuleId(properties, srcFile),
            providers: this.moduleHelper.getModuleProviders(properties, srcFile),
            declarations: this.moduleHelper.getModuleDeclarations(properties, srcFile),
            controllers: this.moduleHelper.getModuleControllers(properties, srcFile),
            entryComponents: this.moduleHelper.getModuleEntryComponents(properties, srcFile),
            imports: this.moduleHelper.getModuleImports(properties, srcFile),
            exports: this.moduleHelper.getModuleExports(properties, srcFile),
            schemas: this.moduleHelper.getModuleSchemas(properties, srcFile),
            bootstrap: this.moduleHelper.getModuleBootstrap(properties, srcFile),
            type: 'module',
            rawdescription: IO.rawdescription,
            methods: IO.methods,
            description: IO.description,
            sourceCode: srcFile.text
        };
    };
    return ModuleDepFactory;
}());

var crypto$5 = require('crypto');
var marked$2 = require('marked');
var ast$2 = new Ast__default();
// TypeScript reference : https://github.com/Microsoft/TypeScript/blob/master/lib/typescript.d.ts
var AngularDependencies = /** @class */ (function (_super) {
    __extends(AngularDependencies, _super);
    function AngularDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        _this.jsdocParserUtil = new JsdocParserUtil();
        return _this;
    }
    AngularDependencies.prototype.getDependencies = function () {
        var _this = this;
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            controllers: [],
            injectables: [],
            interceptors: [],
            guards: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        var sourceFiles = this.program.getSourceFiles() || [];
        sourceFiles.map(function (file) {
            var filePath = file.fileName;
            if (path.extname(filePath) === '.ts' || path.extname(filePath) === '.tsx') {
                if (!Configuration$1.mainData.angularJSProject && path.extname(filePath) === '.js') {
                    logger.info('parsing', filePath);
                    _this.getSourceFileDecorators(file, deps);
                }
                else {
                    if (filePath.lastIndexOf('.d.ts') === -1 &&
                        filePath.lastIndexOf('spec.ts') === -1) {
                        logger.info('parsing', filePath);
                        _this.getSourceFileDecorators(file, deps);
                    }
                }
            }
            return deps;
        });
        // End of file scanning
        // Try merging inside the same file declarated variables & modules with imports | exports | declarations | providers
        if (deps.miscellaneous.variables.length > 0) {
            deps.miscellaneous.variables.forEach(function (_variable) {
                var newVar = [];
                (function (_var, _newVar) {
                    // getType pr reconstruire....
                    if (_var.initializer) {
                        if (_var.initializer.elements) {
                            if (_var.initializer.elements.length > 0) {
                                _var.initializer.elements.forEach(function (element) {
                                    if (element.text) {
                                        newVar.push({
                                            name: element.text,
                                            type: _this.symbolHelper.getType(element.text)
                                        });
                                    }
                                });
                            }
                        }
                    }
                })(_variable);
                var onLink = function (mod) {
                    var process = function (initialArray, _var) {
                        var indexToClean = 0;
                        var found = false;
                        var findVariableInArray = function (el, index, theArray) {
                            if (el.name === _var.name) {
                                indexToClean = index;
                                found = true;
                            }
                        };
                        initialArray.forEach(findVariableInArray);
                        // Clean indexes to replace
                        if (found) {
                            initialArray.splice(indexToClean, 1);
                            // Add variable
                            newVar.forEach(function (newEle) {
                                if (typeof _.find(initialArray, { name: newEle.name }) ===
                                    'undefined') {
                                    initialArray.push(newEle);
                                }
                            });
                        }
                    };
                    process(mod.imports, _variable);
                    process(mod.exports, _variable);
                    process(mod.controllers, _variable);
                    process(mod.declarations, _variable);
                    process(mod.providers, _variable);
                };
                deps.modules.forEach(onLink);
                deps.modulesForGraph.forEach(onLink);
            });
        }
        /**
         * If one thing extends another, merge them, only for internal sources
         * - classes
         * - components
         * - injectables
         * for
         * - inputs
         * - outputs
         * - properties
         * - methods
         */
        deps = ExtendsMerger$1.merge(deps);
        // RouterParserUtil.printModulesRoutes();
        // RouterParserUtil.printRoutes();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            RouterParserUtil$1.linkModulesAndRoutes();
            RouterParserUtil$1.constructModulesTree();
            deps.routesTree = RouterParserUtil$1.constructRoutesTree();
        }
        return deps;
    };
    AngularDependencies.prototype.processClass = function (node, file, srcFile, outputSymbols, fileBody) {
        var name = this.getSymboleName(node);
        var IO = this.getClassIO(file, srcFile, node, fileBody);
        var sourceCode = srcFile.getText();
        var hash = crypto$5
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var deps = {
            name: name,
            id: 'class-' + name + '-' + hash,
            file: file,
            type: 'class',
            sourceCode: srcFile.getText()
        };
        var excludeFromClassArray = false;
        if (IO.constructor) {
            deps.constructorObj = IO.constructor;
        }
        if (IO.properties) {
            deps.properties = IO.properties;
        }
        if (IO.description) {
            deps.description = IO.description;
        }
        if (IO.rawdescription) {
            deps.rawdescription = IO.rawdescription;
        }
        if (IO.methods) {
            deps.methods = IO.methods;
        }
        if (IO.indexSignatures) {
            deps.indexSignatures = IO.indexSignatures;
        }
        if (IO.extends) {
            deps.extends = IO.extends;
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            deps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.accessors) {
            deps.accessors = IO.accessors;
        }
        if (IO.inputs) {
            deps.inputsClass = IO.inputs;
        }
        if (IO.outputs) {
            deps.outputsClass = IO.outputs;
        }
        if (IO.hostBindings) {
            deps.hostBindings = IO.hostBindings;
        }
        if (IO.hostListeners) {
            deps.hostListeners = IO.hostListeners;
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            deps.methods = cleanLifecycleHooksFromMethods(deps.methods);
        }
        if (IO.implements && IO.implements.length > 0) {
            deps.implements = IO.implements;
            if (this.isGuard(IO.implements)) {
                // We don't want the Guard to show up in the Classes menu
                excludeFromClassArray = true;
                deps.type = 'guard';
                outputSymbols.guards.push(deps);
            }
        }
        if (typeof IO.ignore === 'undefined') {
            this.debug(deps);
            if (!excludeFromClassArray) {
                outputSymbols.classes.push(deps);
            }
        }
        else {
            this.ignore(deps);
        }
    };
    AngularDependencies.prototype.getSourceFileDecorators = function (initialSrcFile, outputSymbols) {
        var _this = this;
        var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
        var fileName = initialSrcFile.fileName.replace(cleaner, '');
        var scannedFile = initialSrcFile;
        // Search in file for variable statement as routes definitions
        var astFile = typeof ast$2.getSourceFile(initialSrcFile.fileName) !== 'undefined'
            ? ast$2.getSourceFile(initialSrcFile.fileName)
            : ast$2.addExistingSourceFile(initialSrcFile.fileName);
        var variableRoutesStatements = astFile.getVariableStatements();
        var hasRoutesStatements = false;
        if (variableRoutesStatements.length > 0) {
            // Clean file for spread and dynamics inside routes definitions
            variableRoutesStatements.forEach(function (s) {
                var variableDeclarations = s.getDeclarations();
                var len = variableDeclarations.length;
                var i = 0;
                for (i; i < len; i++) {
                    if (variableDeclarations[i].compilerNode.type) {
                        if (variableDeclarations[i].compilerNode.type.typeName &&
                            variableDeclarations[i].compilerNode.type.typeName.text === 'Routes') {
                            hasRoutesStatements = true;
                        }
                    }
                }
            });
        }
        if (hasRoutesStatements && !Configuration$1.mainData.disableRoutesGraph) {
            // Clean file for spread and dynamics inside routes definitions
            logger.info('Analysing routes definitions and clean them if necessary');
            // scannedFile = RouterParserUtil.cleanFileIdentifiers(astFile).compilerNode;
            var firstClean = RouterParserUtil$1.cleanFileSpreads(astFile).compilerNode;
            scannedFile = RouterParserUtil$1.cleanCallExpressions(astFile).compilerNode;
            scannedFile = RouterParserUtil$1.cleanFileDynamics(astFile).compilerNode;
            scannedFile.kind = Ast.SyntaxKind.SourceFile;
        }
        Ast.ts.forEachChild(scannedFile, function (initialNode) {
            if (_this.jsDocHelper.hasJSDocInternalTag(fileName, scannedFile, initialNode) &&
                Configuration$1.mainData.disableInternal) {
                return;
            }
            var parseNode = function (file, srcFile, node, fileBody) {
                var sourceCode = srcFile.getText();
                var hash = crypto$5
                    .createHash('md5')
                    .update(sourceCode)
                    .digest('hex');
                if (node.decorators) {
                    var classWithCustomDecorator_1 = false;
                    var visitDecorator = function (visitedDecorator, index) {
                        var deps;
                        var metadata = node.decorators;
                        var name = _this.getSymboleName(node);
                        var props = _this.findProperties(visitedDecorator, srcFile);
                        var IO = _this.componentHelper.getComponentIO(file, srcFile, node, fileBody);
                        if (_this.isModule(visitedDecorator)) {
                            var moduleDep = new ModuleDepFactory(_this.moduleHelper).create(file, srcFile, name, props, IO);
                            if (RouterParserUtil$1.hasRouterModuleInImports(moduleDep.imports)) {
                                RouterParserUtil$1.addModuleWithRoutes(name, _this.moduleHelper.getModuleImportsRaw(props, srcFile), file);
                            }
                            deps = moduleDep;
                            if (typeof IO.ignore === 'undefined') {
                                RouterParserUtil$1.addModule(name, moduleDep.imports);
                                outputSymbols.modules.push(moduleDep);
                                outputSymbols.modulesForGraph.push(moduleDep);
                            }
                        }
                        else if (_this.isComponent(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var componentDep = new ComponentDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = componentDep;
                            if (typeof IO.ignore === 'undefined') {
                                ComponentsTreeEngine$1.addComponent(componentDep);
                                outputSymbols.components.push(componentDep);
                            }
                        }
                        else if (_this.isController(visitedDecorator)) {
                            var controllerDep = new ControllerDepFactory().create(file, srcFile, name, props, IO);
                            deps = controllerDep;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.controllers.push(controllerDep);
                            }
                        }
                        else if (_this.isInjectable(visitedDecorator)) {
                            var injectableDeps = {
                                name: name,
                                id: 'injectable-' + name + '-' + hash,
                                file: file,
                                properties: IO.properties,
                                methods: IO.methods,
                                description: IO.description,
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.constructor) {
                                injectableDeps.constructorObj = IO.constructor;
                            }
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                injectableDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            if (IO.accessors) {
                                injectableDeps.accessors = IO.accessors;
                            }
                            if (IO.extends) {
                                injectableDeps.extends = IO.extends;
                            }
                            deps = injectableDeps;
                            if (typeof IO.ignore === 'undefined') {
                                if (_.includes(IO.implements, 'HttpInterceptor')) {
                                    injectableDeps.type = 'interceptor';
                                    outputSymbols.interceptors.push(injectableDeps);
                                }
                                else if (_this.isGuard(IO.implements)) {
                                    injectableDeps.type = 'guard';
                                    outputSymbols.guards.push(injectableDeps);
                                }
                                else {
                                    injectableDeps.type = 'injectable';
                                    _this.addNewEntityInStore(injectableDeps, outputSymbols.injectables);
                                }
                            }
                        }
                        else if (_this.isPipe(visitedDecorator)) {
                            var pipeDeps = {
                                name: name,
                                id: 'pipe-' + name + '-' + hash,
                                file: file,
                                type: 'pipe',
                                description: IO.description,
                                properties: IO.properties,
                                methods: IO.methods,
                                pure: _this.componentHelper.getComponentPure(props, srcFile),
                                ngname: _this.componentHelper.getComponentName(props, srcFile),
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                pipeDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            deps = pipeDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.pipes.push(pipeDeps);
                            }
                        }
                        else if (_this.isDirective(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var directiveDeps = new DirectiveDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = directiveDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.directives.push(directiveDeps);
                            }
                        }
                        else {
                            var hasMultipleDecoratorsWithInternalOne = _this.hasInternalDecorator(node.decorators);
                            // Just a class
                            if (!classWithCustomDecorator_1 &&
                                !hasMultipleDecoratorsWithInternalOne) {
                                classWithCustomDecorator_1 = true;
                                _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                            }
                        }
                        _this.cache.set(name, deps);
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(deps);
                        }
                        else {
                            _this.ignore(deps);
                        }
                    };
                    var filterByDecorators = function (filteredNode) {
                        if (filteredNode.expression && filteredNode.expression.expression) {
                            var _test = /(NgModule|Component|Injectable|Pipe|Directive)/.test(filteredNode.expression.expression.text);
                            if (!_test && Ast.ts.isClassDeclaration(node)) {
                                _test = true;
                            }
                            return _test;
                        }
                        if (Ast.ts.isClassDeclaration(node)) {
                            return true;
                        }
                        return false;
                    };
                    node.decorators.filter(filterByDecorators).forEach(visitDecorator);
                }
                else if (node.symbol) {
                    if (node.symbol.flags === Ast.ts.SymbolFlags.Class) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    else if (node.symbol.flags === Ast.ts.SymbolFlags.Interface) {
                        var name = _this.getSymboleName(node);
                        var IO = _this.getInterfaceIO(file, srcFile, node, fileBody);
                        var interfaceDeps = {
                            name: name,
                            id: 'interface-' + name + '-' + hash,
                            file: file,
                            type: 'interface',
                            sourceCode: srcFile.getText()
                        };
                        if (IO.properties) {
                            interfaceDeps.properties = IO.properties;
                        }
                        if (IO.indexSignatures) {
                            interfaceDeps.indexSignatures = IO.indexSignatures;
                        }
                        if (IO.kind) {
                            interfaceDeps.kind = IO.kind;
                        }
                        if (IO.description) {
                            interfaceDeps.description = IO.description;
                        }
                        if (IO.methods) {
                            interfaceDeps.methods = IO.methods;
                        }
                        if (IO.extends) {
                            interfaceDeps.extends = IO.extends;
                        }
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(interfaceDeps);
                            outputSymbols.interfaces.push(interfaceDeps);
                        }
                        else {
                            _this.ignore(interfaceDeps);
                        }
                    }
                    else if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        // let tags = this.visitFunctionDeclarationJSDocTags(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            file: file,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    else if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                    else if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var typeAliasDeps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            typeAliasDeps.kind = node.type.kind;
                            if (typeAliasDeps.rawtype === '') {
                                typeAliasDeps.rawtype = kindToType(node.type.kind);
                            }
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(typeAliasDeps);
                        }
                    }
                    else if (Ast.ts.isModuleDeclaration(node)) {
                        if (node.body) {
                            if (node.body.statements && node.body.statements.length > 0) {
                                node.body.statements.forEach(function (statement) {
                                    return parseNode(file, srcFile, statement, node.body);
                                });
                            }
                        }
                    }
                }
                else {
                    var IO = _this.getRouteIO(file, srcFile, node);
                    if (IO.routes) {
                        var newRoutes = void 0;
                        try {
                            newRoutes = RouterParserUtil$1.cleanRawRouteParsed(IO.routes);
                        }
                        catch (e) {
                            // tslint:disable-next-line:max-line-length
                            logger.error('Routes parsing error, maybe a trailing comma or an external variable, trying to fix that later after sources scanning.');
                            newRoutes = IO.routes.replace(/ /gm, '');
                            RouterParserUtil$1.addIncompleteRoute({
                                data: newRoutes,
                                file: file
                            });
                            return true;
                        }
                        outputSymbols.routes = outputSymbols.routes.concat(newRoutes);
                    }
                    if (Ast.ts.isClassDeclaration(node)) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    if (Ast.ts.isExpressionStatement(node) || Ast.ts.isIfStatement(node)) {
                        var bootstrapModuleReference = 'bootstrapModule';
                        // Find the root module with bootstrapModule call
                        // 1. find a simple call : platformBrowserDynamic().bootstrapModule(AppModule);
                        // 2. or inside a call :
                        // () => {
                        //     platformBrowserDynamic().bootstrapModule(AppModule);
                        // });
                        // 3. with a catch : platformBrowserDynamic().bootstrapModule(AppModule).catch(error => console.error(error));
                        // 4. with parameters : platformBrowserDynamic().bootstrapModule(AppModule, {}).catch(error => console.error(error));
                        // Find recusively in expression nodes one with name 'bootstrapModule'
                        var rootModule_1;
                        var resultNode = void 0;
                        if (srcFile.text.indexOf(bootstrapModuleReference) !== -1) {
                            if (node.expression) {
                                resultNode = _this.findExpressionByNameInExpressions(node.expression, 'bootstrapModule');
                            }
                            if (typeof node.thenStatement !== 'undefined') {
                                if (node.thenStatement.statements &&
                                    node.thenStatement.statements.length > 0) {
                                    var firstStatement = node.thenStatement.statements[0];
                                    resultNode = _this.findExpressionByNameInExpressions(firstStatement.expression, 'bootstrapModule');
                                }
                            }
                            if (!resultNode) {
                                if (node.expression &&
                                    node.expression.arguments &&
                                    node.expression.arguments.length > 0) {
                                    resultNode = _this.findExpressionByNameInExpressionArguments(node.expression.arguments, 'bootstrapModule');
                                }
                            }
                            if (resultNode) {
                                if (resultNode.arguments.length > 0) {
                                    _.forEach(resultNode.arguments, function (argument) {
                                        if (argument.text) {
                                            rootModule_1 = argument.text;
                                        }
                                    });
                                }
                                if (rootModule_1) {
                                    RouterParserUtil$1.setRootModule(rootModule_1);
                                }
                            }
                        }
                    }
                    if (Ast.ts.isVariableStatement(node) && !RouterParserUtil$1.isVariableRoutes(node)) {
                        var infos = _this.visitVariableDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'variable',
                            file: file
                        };
                        deps.type = infos.type ? infos.type : '';
                        if (infos.defaultValue) {
                            deps.defaultValue = infos.defaultValue;
                        }
                        if (infos.initializer) {
                            deps.initializer = infos.initializer;
                        }
                        if (node.jsDoc && node.jsDoc.length > 0 && node.jsDoc[0].comment) {
                            deps.description = marked$2(node.jsDoc[0].comment);
                        }
                        if (isModuleWithProviders(node)) {
                            var routingInitializer = getModuleWithProviders(node);
                            RouterParserUtil$1.addModuleWithRoutes(name, [routingInitializer], file);
                            RouterParserUtil$1.addModule(name, [routingInitializer]);
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.variables.push(deps);
                        }
                    }
                    if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            deps.kind = node.type.kind;
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(deps);
                        }
                    }
                    if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                }
            };
            parseNode(fileName, scannedFile, initialNode);
        });
    };
    /**
     * Function to in a specific store an entity, and check before is there is not the same one
     * in that store : same name, id and file
     * @param entity Entity to store
     * @param store Store
     */
    AngularDependencies.prototype.addNewEntityInStore = function (entity, store) {
        var findSameEntityInStore = _.filter(store, {
            name: entity.name,
            id: entity.id,
            file: entity.file
        });
        if (findSameEntityInStore.length === 0) {
            store.push(entity);
        }
    };
    AngularDependencies.prototype.debug = function (deps) {
        if (deps) {
            logger.debug('found', "" + deps.name);
        }
        else {
            return;
        }
        ['imports', 'exports', 'declarations', 'providers', 'bootstrap'].forEach(function (symbols) {
            if (deps[symbols] && deps[symbols].length > 0) {
                logger.debug('', "- " + symbols + ":");
                deps[symbols]
                    .map(function (i) { return i.name; })
                    .forEach(function (d) {
                    logger.debug('', "\t- " + d);
                });
            }
        });
    };
    AngularDependencies.prototype.ignore = function (deps) {
        if (deps) {
            logger.warn('ignore', "" + deps.name);
        }
        else {
            return;
        }
    };
    AngularDependencies.prototype.findExpressionByNameInExpressions = function (entryNode, name) {
        var result;
        var loop = function (node, z) {
            if (node) {
                if (node.expression && !node.expression.name) {
                    loop(node.expression, z);
                }
                if (node.expression && node.expression.name) {
                    if (node.expression.name.text === z) {
                        result = node;
                    }
                    else {
                        loop(node.expression, z);
                    }
                }
            }
        };
        loop(entryNode, name);
        return result;
    };
    AngularDependencies.prototype.findExpressionByNameInExpressionArguments = function (arg, name) {
        var result;
        var that = this;
        var i = 0;
        var len = arg.length;
        var loop = function (node, z) {
            if (node.body) {
                if (node.body.statements && node.body.statements.length > 0) {
                    var j = 0;
                    var leng = node.body.statements.length;
                    for (j; j < leng; j++) {
                        result = that.findExpressionByNameInExpressions(node.body.statements[j], z);
                    }
                }
            }
        };
        for (i; i < len; i++) {
            loop(arg[i], name);
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorators = function (decorators, type) {
        var result = false;
        if (decorators.length > 1) {
            _.forEach(decorators, function (decorator) {
                if (decorator.expression.expression) {
                    if (decorator.expression.expression.text === type) {
                        result = true;
                    }
                }
            });
        }
        else {
            if (decorators[0].expression.expression) {
                if (decorators[0].expression.expression.text === type) {
                    result = true;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorator = function (decorator, type) {
        var result = false;
        if (decorator.expression.expression) {
            if (decorator.expression.expression.text === type) {
                result = true;
            }
        }
        return result;
    };
    AngularDependencies.prototype.isController = function (metadata) {
        return this.parseDecorator(metadata, 'Controller');
    };
    AngularDependencies.prototype.isComponent = function (metadata) {
        return this.parseDecorator(metadata, 'Component');
    };
    AngularDependencies.prototype.isPipe = function (metadata) {
        return this.parseDecorator(metadata, 'Pipe');
    };
    AngularDependencies.prototype.isDirective = function (metadata) {
        return this.parseDecorator(metadata, 'Directive');
    };
    AngularDependencies.prototype.isInjectable = function (metadata) {
        return this.parseDecorator(metadata, 'Injectable');
    };
    AngularDependencies.prototype.isModule = function (metadata) {
        return this.parseDecorator(metadata, 'NgModule') || this.parseDecorator(metadata, 'Module');
    };
    AngularDependencies.prototype.hasInternalDecorator = function (metadatas) {
        return (this.parseDecorators(metadatas, 'Controller') ||
            this.parseDecorators(metadatas, 'Component') ||
            this.parseDecorators(metadatas, 'Pipe') ||
            this.parseDecorators(metadatas, 'Directive') ||
            this.parseDecorators(metadatas, 'Injectable') ||
            this.parseDecorators(metadatas, 'NgModule') ||
            this.parseDecorators(metadatas, 'Module'));
    };
    AngularDependencies.prototype.isGuard = function (ioImplements) {
        return (_.includes(ioImplements, 'CanActivate') ||
            _.includes(ioImplements, 'CanActivateChild') ||
            _.includes(ioImplements, 'CanDeactivate') ||
            _.includes(ioImplements, 'Resolve') ||
            _.includes(ioImplements, 'CanLoad'));
    };
    AngularDependencies.prototype.getSymboleName = function (node) {
        return node.name.text;
    };
    AngularDependencies.prototype.findProperties = function (visitedNode, sourceFile) {
        if (visitedNode.expression &&
            visitedNode.expression.arguments &&
            visitedNode.expression.arguments.length > 0) {
            var pop = visitedNode.expression.arguments[0];
            if (pop && pop.properties && pop.properties.length >= 0) {
                return pop.properties;
            }
            else if (pop && pop.kind && pop.kind === Ast.SyntaxKind.StringLiteral) {
                return [pop];
            }
            else {
                logger.warn('Empty metadatas, trying to found it with imports.');
                return ImportsUtil$1.findValueInImportOrLocalVariables(pop.text, sourceFile);
            }
        }
        return [];
    };
    AngularDependencies.prototype.isAngularLifecycleHook = function (methodName) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var ANGULAR_LIFECYCLE_METHODS = [
            'ngOnInit',
            'ngOnChanges',
            'ngDoCheck',
            'ngOnDestroy',
            'ngAfterContentInit',
            'ngAfterContentChecked',
            'ngAfterViewInit',
            'ngAfterViewChecked',
            'writeValue',
            'registerOnChange',
            'registerOnTouched',
            'setDisabledState'
        ];
        return ANGULAR_LIFECYCLE_METHODS.indexOf(methodName) >= 0;
    };
    AngularDependencies.prototype.visitTypeDeclaration = function (node) {
        var result = {
            name: node.name.text,
            kind: node.kind
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitArgument = function (arg) {
        var result = {
            name: arg.name.text,
            type: this.classHelper.visitType(arg)
        };
        if (arg.dotDotDotToken) {
            result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            result.optional = true;
        }
        if (arg.type) {
            result.type = this.mapType(arg.type.kind);
            if (arg.type.kind === 157) {
                // try replace TypeReference with typeName
                if (arg.type.typeName) {
                    result.type = arg.type.typeName.text;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.mapType = function (type) {
        switch (type) {
            case 95:
                return 'null';
            case 119:
                return 'any';
            case 122:
                return 'boolean';
            case 130:
                return 'never';
            case 133:
                return 'number';
            case 136:
                return 'string';
            case 139:
                return 'undefined';
            case 159:
                return 'typeReference';
        }
    };
    AngularDependencies.prototype.hasPrivateJSDocTag = function (tags) {
        var result = false;
        if (tags) {
            tags.forEach(function (tag) {
                if (tag.tagName && tag.tagName.text && tag.tagName.text === 'private') {
                    result = true;
                }
            });
        }
        return result;
    };
    AngularDependencies.prototype.visitFunctionDeclaration = function (method) {
        var _this = this;
        var methodName = method.name ? method.name.text : 'Unnamed function';
        var result = {
            name: methodName,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : []
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type !== 'undefined') {
            result.returnType = this.classHelper.visitType(method.type);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers
                    .map(function (modifier) {
                    return modifier.kind;
                })
                    .reverse();
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
                _.forEach(jsdoctags[0].tags, function (tag) {
                    if (tag.tagName) {
                        if (tag.tagName.text) {
                            if (tag.tagName.text.indexOf('ignore') > -1) {
                                result.ignore = true;
                            }
                        }
                    }
                });
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    AngularDependencies.prototype.visitVariableDeclaration = function (node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var result = {
                    name: node.declarationList.declarations[i].name.text,
                    defaultValue: node.declarationList.declarations[i].initializer
                        ? this.classHelper.stringifyDefaultValue(node.declarationList.declarations[i].initializer)
                        : undefined
                };
                if (node.declarationList.declarations[i].initializer) {
                    result.initializer = node.declarationList.declarations[i].initializer;
                }
                if (node.declarationList.declarations[i].type) {
                    result.type = this.classHelper.visitType(node.declarationList.declarations[i].type);
                }
                if (typeof result.type === 'undefined' && result.initializer) {
                    result.type = kindToType(result.initializer.kind);
                }
                return result;
            }
        }
    };
    AngularDependencies.prototype.visitFunctionDeclarationJSDocTags = function (node) {
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        var result;
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumTypeAliasFunctionDeclarationDescription = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = marked$2(node.jsDoc[0].comment);
                }
            }
        }
        return description;
    };
    AngularDependencies.prototype.visitEnumDeclaration = function (node) {
        var result = [];
        if (node.members) {
            var i = 0;
            var len = node.members.length;
            for (i; i < len; i++) {
                var member = {
                    name: node.members[i].name.text
                };
                if (node.members[i].initializer) {
                    member.value = node.members[i].initializer.text;
                }
                result.push(member);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumDeclarationForRoutes = function (fileName, node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var routesInitializer = node.declarationList.declarations[i].initializer;
                var data = new CodeGenerator().generate(routesInitializer);
                RouterParserUtil$1.addRoute({
                    name: node.declarationList.declarations[i].name.text,
                    data: RouterParserUtil$1.cleanRawRoute(data),
                    filename: fileName
                });
                return [
                    {
                        routes: data
                    }
                ];
            }
        }
        return [];
    };
    AngularDependencies.prototype.getRouteIO = function (filename, sourceFile, node) {
        var _this = this;
        var res;
        if (sourceFile.statements) {
            res = sourceFile.statements.reduce(function (directive, statement) {
                if (RouterParserUtil$1.isVariableRoutes(statement)) {
                    if (statement.pos === node.pos && statement.end === node.end) {
                        return directive.concat(_this.visitEnumDeclarationForRoutes(filename, statement));
                    }
                }
                return directive;
            }, []);
            return res[0] || {};
        }
        else {
            return {};
        }
    };
    AngularDependencies.prototype.getClassIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    AngularDependencies.prototype.getInterfaceIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isInterfaceDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    return AngularDependencies;
}(FrameworkDependencies));

var AngularJSDependencies = /** @class */ (function (_super) {
    __extends(AngularJSDependencies, _super);
    function AngularJSDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        return _this;
    }
    AngularJSDependencies.prototype.getDependencies = function () {
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            injectables: [],
            interceptors: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        return deps;
    };
    return AngularJSDependencies;
}(FrameworkDependencies));

function promiseSequential(promises) {
    if (!Array.isArray(promises)) {
        throw new Error('First argument need to be an array of Promises');
    }
    return new Promise(function (resolve, reject) {
        var count = 0;
        var results = [];
        var iterateeFunc = function (previousPromise, currentPromise) {
            return previousPromise
                .then(function (result) {
                if (count++ !== 0) {
                    results = results.concat(result);
                }
                return currentPromise(result, results, count);
            })
                .catch(function (err) {
                return reject(err);
            });
        };
        promises = promises.concat(function () { return Promise.resolve(); });
        promises.reduce(iterateeFunc, Promise.resolve(false)).then(function (res) {
            resolve(results);
        });
    });
}

var chokidar = require('chokidar');
var marked$3 = require('marked');
var traverse$3 = require('traverse');
var crypto$6 = require('crypto');
var cwd = process.cwd();
var startTime = new Date();
var generationPromiseResolve;
var generationPromiseReject;
var generationPromise = new Promise(function (resolve, reject) {
    generationPromiseResolve = resolve;
    generationPromiseReject = reject;
});
var Application = /** @class */ (function () {
    /**
     * Create a new compodoc application instance.
     *
     * @param options An object containing the options that should be used.
     */
    function Application(options) {
        var _this = this;
        /**
         * Files changed during watch scanning
         */
        this.watchChangedFiles = [];
        /**
         * Boolean for watching status
         * @type {boolean}
         */
        this.isWatching = false;
        /**
         * Store package.json data
         */
        this.packageJsonData = {};
        this.preparePipes = function (somePipes) {
            logger.info('Prepare pipes');
            Configuration$1.mainData.pipes = somePipes ? somePipes : DependenciesEngine$1.getPipes();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.pipes.length;
                var loop = function () {
                    if (i < len) {
                        var pipe = Configuration$1.mainData.pipes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(pipe.file)) {
                            logger.info(" " + pipe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(pipe.file);
                            pipe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'pipes',
                            name: pipe.name,
                            id: pipe.id,
                            navTabs: _this.getNavTabs(pipe),
                            context: 'pipe',
                            pipe: pipe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (pipe.isDuplicate) {
                            page.name += '-' + pipe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        this.prepareClasses = function (someClasses) {
            logger.info('Prepare classes');
            Configuration$1.mainData.classes = someClasses
                ? someClasses
                : DependenciesEngine$1.getClasses();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.classes.length;
                var loop = function () {
                    if (i < len) {
                        var classe = Configuration$1.mainData.classes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(classe.file)) {
                            logger.info(" " + classe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(classe.file);
                            classe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'classes',
                            name: classe.name,
                            id: classe.id,
                            navTabs: _this.getNavTabs(classe),
                            context: 'class',
                            class: classe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (classe.isDuplicate) {
                            page.name += '-' + classe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        for (var option in options) {
            if (typeof Configuration$1.mainData[option] !== 'undefined') {
                Configuration$1.mainData[option] = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'name') {
                Configuration$1.mainData.documentationMainName = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'silent') {
                logger.silent = false;
            }
        }
    }
    /**
     * Start compodoc process
     */
    Application.prototype.generate = function () {
        var _this = this;
        process.on('unhandledRejection', this.unhandledRejectionListener);
        process.on('uncaughtException', this.uncaughtExceptionListener);
        I18nEngine$1.init(Configuration$1.mainData.language);
        if (Configuration$1.mainData.output.charAt(Configuration$1.mainData.output.length - 1) !== '/') {
            Configuration$1.mainData.output += '/';
        }
        if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
            this.processPackageJson();
        }
        else {
            HtmlEngine$1.init(Configuration$1.mainData.templates).then(function () { return _this.processPackageJson(); });
        }
        return generationPromise;
    };
    Application.prototype.endCallback = function () {
        process.removeListener('unhandledRejection', this.unhandledRejectionListener);
        process.removeListener('uncaughtException', this.uncaughtExceptionListener);
    };
    Application.prototype.unhandledRejectionListener = function (err, p) {
        console.log('Unhandled Rejection at:', p, 'reason:', err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    Application.prototype.uncaughtExceptionListener = function (err) {
        logger.error(err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    /**
     * Start compodoc documentation coverage
     */
    Application.prototype.testCoverage = function () {
        this.getDependenciesData();
    };
    /**
     * Store files for initial processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setFiles = function (files) {
        this.files = files;
    };
    /**
     * Store files for watch processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setUpdatedFiles = function (files) {
        this.updatedFiles = files;
    };
    /**
     * Return a boolean indicating presence of one TypeScript file in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesTSFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.ts') {
                result = true;
            }
        });
        return result;
    };
    /**
     * Return a boolean indicating presence of one root markdown files in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesRootMarkdownFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.md' && path.dirname(file) === cwd) {
                result = true;
            }
        });
        return result;
    };
    /**
     * Clear files for watch processing
     */
    Application.prototype.clearUpdatedFiles = function () {
        this.updatedFiles = [];
        this.watchChangedFiles = [];
    };
    Application.prototype.processPackageJson = function () {
        var _this = this;
        logger.info('Searching package.json file');
        FileEngine$1.get(cwd + path.sep + 'package.json').then(function (packageData) {
            var parsedData = JSON.parse(packageData);
            _this.packageJsonData = parsedData;
            if (typeof parsedData.name !== 'undefined' &&
                Configuration$1.mainData.documentationMainName === COMPODOC_DEFAULTS.title) {
                Configuration$1.mainData.documentationMainName =
                    parsedData.name + ' documentation';
            }
            if (typeof parsedData.description !== 'undefined') {
                Configuration$1.mainData.documentationMainDescription = parsedData.description;
            }
            Configuration$1.mainData.angularVersion = AngularVersionUtil$1.getAngularVersionOfProject(parsedData);
            logger.info('package.json file found');
            if (!Configuration$1.mainData.disableDependencies) {
                if (typeof parsedData.dependencies !== 'undefined') {
                    _this.processPackageDependencies(parsedData.dependencies);
                }
                if (typeof parsedData.peerDependencies !== 'undefined') {
                    _this.processPackagePeerDependencies(parsedData.peerDependencies);
                }
            }
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage) {
                logger.error(errorMessage);
            });
        }, function (errorMessage) {
            logger.error(errorMessage);
            logger.error('Continuing without package.json file');
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage1) {
                logger.error(errorMessage1);
            });
        });
    };
    Application.prototype.processPackagePeerDependencies = function (dependencies) {
        logger.info('Processing package.json peerDependencies');
        Configuration$1.mainData.packagePeerDependencies = dependencies;
        if (!Configuration$1.hasPage('dependencies')) {
            Configuration$1.addPage({
                name: 'dependencies',
                id: 'packageDependencies',
                context: 'package-dependencies',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
        }
    };
    Application.prototype.processPackageDependencies = function (dependencies) {
        logger.info('Processing package.json dependencies');
        Configuration$1.mainData.packageDependencies = dependencies;
        Configuration$1.addPage({
            name: 'dependencies',
            id: 'packageDependencies',
            context: 'package-dependencies',
            depth: 0,
            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
        });
    };
    Application.prototype.processMarkdowns = function () {
        logger.info('Searching README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md files');
        return new Promise(function (resolve, reject) {
            var i = 0;
            var markdowns = ['readme', 'changelog', 'contributing', 'license', 'todo'];
            var numberOfMarkdowns = 5;
            var loop = function () {
                if (i < numberOfMarkdowns) {
                    MarkdownEngine$1.getTraditionalMarkdown(markdowns[i].toUpperCase()).then(function (readmeData) {
                        Configuration$1.addPage({
                            name: markdowns[i] === 'readme' ? 'index' : markdowns[i],
                            context: 'getting-started',
                            id: 'getting-started',
                            markdown: readmeData,
                            depth: 0,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                        });
                        if (markdowns[i] === 'readme') {
                            Configuration$1.mainData.readme = true;
                            Configuration$1.addPage({
                                name: 'overview',
                                id: 'overview',
                                context: 'overview',
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        else {
                            Configuration$1.mainData.markdowns.push({
                                name: markdowns[i],
                                uppername: markdowns[i].toUpperCase(),
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        logger.info(markdowns[i].toUpperCase() + ".md file found");
                        i++;
                        loop();
                    }, function (errorMessage) {
                        logger.warn(errorMessage);
                        logger.warn("Continuing without " + markdowns[i].toUpperCase() + ".md file");
                        if (markdowns[i] === 'readme') {
                            Configuration$1.addPage({
                                name: 'index',
                                id: 'index',
                                context: 'overview',
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        i++;
                        loop();
                    });
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.rebuildRootMarkdowns = function () {
        var _this = this;
        logger.info('Regenerating README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md pages');
        var actions = [];
        Configuration$1.resetRootMarkdownPages();
        actions.push(function () {
            return _this.processMarkdowns();
        });
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    /**
     * Get dependency data for small group of updated files during watch process
     */
    Application.prototype.getMicroDependenciesData = function () {
        logger.info('Get diff dependencies data');
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.updatedFiles, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.update(dependenciesData);
        this.prepareJustAFewThings(dependenciesData);
    };
    /**
     * Rebuild external documentation during watch process
     */
    Application.prototype.rebuildExternalDocumentation = function () {
        var _this = this;
        logger.info('Rebuild external documentation');
        var actions = [];
        Configuration$1.resetAdditionalPages();
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.detectAngularJSProjects = function () {
        if (typeof this.packageJsonData.dependencies !== 'undefined') {
            if (typeof this.packageJsonData.dependencies.angular !== 'undefined') ;
            else {
                var countJSFiles_1 = 0;
                this.files.forEach(function (file) {
                    if (path.extname(file) === '.js') {
                        countJSFiles_1 += 1;
                    }
                });
                var percentOfJSFiles = (countJSFiles_1 * 100) / this.files.length;
            }
        }
        return false;
    };
    Application.prototype.getDependenciesData = function () {
        logger.info('Get dependencies data');
        /**
         * AngularJS detection strategy :
         * - if in package.json
         * - if 75% of scanned files are *.js files
         */
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.files, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.init(dependenciesData);
        Configuration$1.mainData.routesLength = RouterParserUtil$1.routesLength();
        this.printStatistics();
        this.prepareEverything();
    };
    Application.prototype.prepareJustAFewThings = function (diffCrawledData) {
        var _this = this;
        var actions = [];
        Configuration$1.resetPages();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () { return _this.prepareRoutes(); });
        }
        if (diffCrawledData.components.length > 0) {
            actions.push(function () { return _this.prepareComponents(); });
        }
        if (diffCrawledData.controllers.length > 0) {
            actions.push(function () { return _this.prepareControllers(); });
        }
        if (diffCrawledData.modules.length > 0) {
            actions.push(function () { return _this.prepareModules(); });
        }
        if (diffCrawledData.directives.length > 0) {
            actions.push(function () { return _this.prepareDirectives(); });
        }
        if (diffCrawledData.injectables.length > 0) {
            actions.push(function () { return _this.prepareInjectables(); });
        }
        if (diffCrawledData.interceptors.length > 0) {
            actions.push(function () { return _this.prepareInterceptors(); });
        }
        if (diffCrawledData.guards.length > 0) {
            actions.push(function () { return _this.prepareGuards(); });
        }
        if (diffCrawledData.pipes.length > 0) {
            actions.push(function () { return _this.preparePipes(); });
        }
        if (diffCrawledData.classes.length > 0) {
            actions.push(function () { return _this.prepareClasses(); });
        }
        if (diffCrawledData.interfaces.length > 0) {
            actions.push(function () { return _this.prepareInterfaces(); });
        }
        if (diffCrawledData.miscellaneous.variables.length > 0 ||
            diffCrawledData.miscellaneous.functions.length > 0 ||
            diffCrawledData.miscellaneous.typealiases.length > 0 ||
            diffCrawledData.miscellaneous.enumerations.length > 0) {
            actions.push(function () { return _this.prepareMiscellaneous(); });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () { return _this.prepareCoverage(); });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processGraphs();
            _this.clearUpdatedFiles();
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.printStatistics = function () {
        logger.info('-------------------');
        logger.info('Project statistics ');
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- files      : " + this.files.length);
        }
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- module     : " + DependenciesEngine$1.modules.length);
        }
        if (DependenciesEngine$1.components.length > 0) {
            logger.info("- component  : " + DependenciesEngine$1.components.length);
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            logger.info("- controller : " + DependenciesEngine$1.controllers.length);
        }
        if (DependenciesEngine$1.directives.length > 0) {
            logger.info("- directive  : " + DependenciesEngine$1.directives.length);
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            logger.info("- injectable : " + DependenciesEngine$1.injectables.length);
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            logger.info("- injector   : " + DependenciesEngine$1.interceptors.length);
        }
        if (DependenciesEngine$1.guards.length > 0) {
            logger.info("- guard      : " + DependenciesEngine$1.guards.length);
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            logger.info("- pipe       : " + DependenciesEngine$1.pipes.length);
        }
        if (DependenciesEngine$1.classes.length > 0) {
            logger.info("- class      : " + DependenciesEngine$1.classes.length);
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            logger.info("- interface  : " + DependenciesEngine$1.interfaces.length);
        }
        if (Configuration$1.mainData.routesLength > 0) {
            logger.info("- route      : " + Configuration$1.mainData.routesLength);
        }
        logger.info('-------------------');
    };
    Application.prototype.prepareEverything = function () {
        var _this = this;
        var actions = [];
        actions.push(function () {
            return _this.prepareComponents();
        });
        actions.push(function () {
            return _this.prepareModules();
        });
        if (DependenciesEngine$1.directives.length > 0) {
            actions.push(function () {
                return _this.prepareDirectives();
            });
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            actions.push(function () {
                return _this.prepareControllers();
            });
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            actions.push(function () {
                return _this.prepareInjectables();
            });
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            actions.push(function () {
                return _this.prepareInterceptors();
            });
        }
        if (DependenciesEngine$1.guards.length > 0) {
            actions.push(function () {
                return _this.prepareGuards();
            });
        }
        if (DependenciesEngine$1.routes &&
            DependenciesEngine$1.routes.children.length > 0 &&
            !Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () {
                return _this.prepareRoutes();
            });
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            actions.push(function () {
                return _this.preparePipes();
            });
        }
        if (DependenciesEngine$1.classes.length > 0) {
            actions.push(function () {
                return _this.prepareClasses();
            });
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            actions.push(function () {
                return _this.prepareInterfaces();
            });
        }
        if (DependenciesEngine$1.miscellaneous.variables.length > 0 ||
            DependenciesEngine$1.miscellaneous.functions.length > 0 ||
            DependenciesEngine$1.miscellaneous.typealiases.length > 0 ||
            DependenciesEngine$1.miscellaneous.enumerations.length > 0) {
            actions.push(function () {
                return _this.prepareMiscellaneous();
            });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () {
                return _this.prepareCoverage();
            });
        }
        if (Configuration$1.mainData.unitTestCoverage !== '') {
            actions.push(function () {
                return _this.prepareUnitTestCoverage();
            });
        }
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
                if (COMPODOC_DEFAULTS.exportFormatsSupported.indexOf(Configuration$1.mainData.exportFormat) > -1) {
                    logger.info("Generating documentation in export format " + Configuration$1.mainData.exportFormat);
                    ExportEngine$1.export(Configuration$1.mainData.output, Configuration$1.mainData).then(function () {
                        generationPromiseResolve();
                        _this.endCallback();
                        logger.info('Documentation generated in ' +
                            Configuration$1.mainData.output +
                            ' in ' +
                            _this.getElapsedTime() +
                            ' seconds');
                    });
                }
                else {
                    logger.warn("Exported format not supported");
                }
            }
            else {
                _this.processGraphs();
            }
        })
            .catch(function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.getIncludedPathForFile = function (file) {
        return path.join(Configuration$1.mainData.includes, file);
    };
    Application.prototype.prepareExternalIncludes = function () {
        var _this = this;
        logger.info('Adding external markdown files');
        // Scan include folder for files detailed in summary.json
        // For each file, add to Configuration.mainData.additionalPages
        // Each file will be converted to html page, inside COMPODOC_DEFAULTS.additionalEntryPath
        return new Promise(function (resolve, reject) {
            FileEngine$1.get(_this.getIncludedPathForFile('summary.json')).then(function (summaryData) {
                logger.info('Additional documentation: summary.json file found');
                var parsedSummaryData = JSON.parse(summaryData);
                var that = _this;
                var lastLevelOnePage = undefined;
                traverse$3(parsedSummaryData).forEach(function () {
                    // tslint:disable-next-line:no-invalid-this
                    if (this.notRoot && typeof this.node === 'object') {
                        // tslint:disable-next-line:no-invalid-this
                        var rawPath = this.path;
                        // tslint:disable-next-line:no-invalid-this
                        var additionalNode = this.node;
                        var file = additionalNode.file;
                        var title = additionalNode.title;
                        var finalPath_1 = Configuration$1.mainData.includesFolder;
                        var finalDepth = rawPath.filter(function (el) {
                            return !isNaN(parseInt(el, 10));
                        });
                        if (typeof file !== 'undefined' && typeof title !== 'undefined') {
                            var url = cleanNameWithoutSpaceAndToLowerCase(title);
                            /**
                             * Id created with title + file path hash, seems to be hypothetically unique here
                             */
                            var id = crypto$6
                                .createHash('md5')
                                .update(title + file)
                                .digest('hex');
                            // tslint:disable-next-line:no-invalid-this
                            this.node.id = id;
                            var lastElementRootTree_1 = undefined;
                            finalDepth.forEach(function (el) {
                                var elementTree = typeof lastElementRootTree_1 === 'undefined'
                                    ? parsedSummaryData
                                    : lastElementRootTree_1;
                                if (typeof elementTree.children !== 'undefined') {
                                    elementTree = elementTree.children[el];
                                }
                                else {
                                    elementTree = elementTree[el];
                                }
                                finalPath_1 +=
                                    '/' +
                                        cleanNameWithoutSpaceAndToLowerCase(elementTree.title);
                                lastElementRootTree_1 = elementTree;
                            });
                            finalPath_1 = finalPath_1.replace('/' + url, '');
                            var markdownFile = MarkdownEngine$1.getTraditionalMarkdownSync(that.getIncludedPathForFile(file));
                            if (finalDepth.length > 5) {
                                logger.error('Only 5 levels of depth are supported');
                            }
                            else {
                                var _page = {
                                    name: title,
                                    id: id,
                                    filename: url,
                                    context: 'additional-page',
                                    path: finalPath_1,
                                    additionalPage: markdownFile,
                                    depth: finalDepth.length,
                                    childrenLength: additionalNode.children
                                        ? additionalNode.children.length
                                        : 0,
                                    children: [],
                                    lastChild: false,
                                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                                };
                                if (finalDepth.length === 1) {
                                    lastLevelOnePage = _page;
                                }
                                if (finalDepth.length > 1) {
                                    // store all child pages of the last root level 1 page inside it
                                    lastLevelOnePage.children.push(_page);
                                }
                                else {
                                    Configuration$1.addAdditionalPage(_page);
                                }
                            }
                        }
                    }
                });
                resolve();
            }, function (errorMessage) {
                logger.error(errorMessage);
                reject('Error during Additional documentation generation');
            });
        });
    };
    Application.prototype.prepareModules = function (someModules) {
        var _this = this;
        logger.info('Prepare modules');
        var i = 0;
        var _modules = someModules ? someModules : DependenciesEngine$1.getModules();
        return new Promise(function (resolve, reject) {
            Configuration$1.mainData.modules = _modules.map(function (ngModule) {
                ngModule.compodocLinks = {
                    components: [],
                    controllers: [],
                    directives: [],
                    injectables: [],
                    pipes: []
                };
                ['declarations', 'bootstrap', 'imports', 'exports', 'controllers'].forEach(function (metadataType) {
                    ngModule[metadataType] = ngModule[metadataType].filter(function (metaDataItem) {
                        switch (metaDataItem.type) {
                            case 'directive':
                                return DependenciesEngine$1.getDirectives().some(function (directive) {
                                    var selectedDirective;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedDirective =
                                            directive.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedDirective =
                                            directive.name === metaDataItem.name;
                                    }
                                    if (selectedDirective &&
                                        !ngModule.compodocLinks.directives.includes(directive)) {
                                        ngModule.compodocLinks.directives.push(directive);
                                    }
                                    return selectedDirective;
                                });
                            case 'component':
                                return DependenciesEngine$1.getComponents().some(function (component) {
                                    var selectedComponent;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedComponent =
                                            component.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedComponent =
                                            component.name === metaDataItem.name;
                                    }
                                    if (selectedComponent &&
                                        !ngModule.compodocLinks.components.includes(component)) {
                                        ngModule.compodocLinks.components.push(component);
                                    }
                                    return selectedComponent;
                                });
                            case 'controller':
                                return DependenciesEngine$1.getControllers().some(function (controller) {
                                    var selectedController;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedController =
                                            controller.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedController =
                                            controller.name === metaDataItem.name;
                                    }
                                    if (selectedController &&
                                        !ngModule.compodocLinks.controllers.includes(controller)) {
                                        ngModule.compodocLinks.controllers.push(controller);
                                    }
                                    return selectedController;
                                });
                            case 'module':
                                return DependenciesEngine$1.getModules().some(function (module) { return module.name === metaDataItem.name; });
                            case 'pipe':
                                return DependenciesEngine$1.getPipes().some(function (pipe) {
                                    var selectedPipe;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedPipe = pipe.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedPipe = pipe.name === metaDataItem.name;
                                    }
                                    if (selectedPipe &&
                                        !ngModule.compodocLinks.pipes.includes(pipe)) {
                                        ngModule.compodocLinks.pipes.push(pipe);
                                    }
                                    return selectedPipe;
                                });
                            default:
                                return true;
                        }
                    });
                });
                ngModule.providers = ngModule.providers.filter(function (provider) {
                    return (DependenciesEngine$1.getInjectables().some(function (injectable) {
                        var selectedInjectable = injectable.name === provider.name;
                        if (selectedInjectable &&
                            !ngModule.compodocLinks.injectables.includes(injectable)) {
                            ngModule.compodocLinks.injectables.push(injectable);
                        }
                        return selectedInjectable;
                    }) ||
                        DependenciesEngine$1.getInterceptors().some(function (interceptor) { return interceptor.name === provider.name; }));
                });
                // Try fixing type undefined for each providers
                _.forEach(ngModule.providers, function (provider) {
                    if (DependenciesEngine$1.getInjectables().find(function (injectable) { return injectable.name === provider.name; })) {
                        provider.type = 'injectable';
                    }
                    if (DependenciesEngine$1.getInterceptors().find(function (interceptor) { return interceptor.name === provider.name; })) {
                        provider.type = 'interceptor';
                    }
                });
                // Order things
                ngModule.compodocLinks.components = _.sortBy(ngModule.compodocLinks.components, [
                    'name'
                ]);
                ngModule.compodocLinks.controllers = _.sortBy(ngModule.compodocLinks.controllers, [
                    'name'
                ]);
                ngModule.compodocLinks.directives = _.sortBy(ngModule.compodocLinks.directives, [
                    'name'
                ]);
                ngModule.compodocLinks.injectables = _.sortBy(ngModule.compodocLinks.injectables, [
                    'name'
                ]);
                ngModule.compodocLinks.pipes = _.sortBy(ngModule.compodocLinks.pipes, ['name']);
                ngModule.declarations = _.sortBy(ngModule.declarations, ['name']);
                ngModule.entryComponents = _.sortBy(ngModule.entryComponents, ['name']);
                ngModule.providers = _.sortBy(ngModule.providers, ['name']);
                ngModule.imports = _.sortBy(ngModule.imports, ['name']);
                ngModule.exports = _.sortBy(ngModule.exports, ['name']);
                return ngModule;
            });
            Configuration$1.addPage({
                name: 'modules',
                id: 'modules',
                context: 'modules',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            var len = Configuration$1.mainData.modules.length;
            var loop = function () {
                if (i < len) {
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(Configuration$1.mainData.modules[i].file)) {
                        logger.info(" " + Configuration$1.mainData.modules[i].name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(Configuration$1.mainData.modules[i].file);
                        Configuration$1.mainData.modules[i].readme = marked$3(readme);
                    }
                    Configuration$1.addPage({
                        path: 'modules',
                        name: Configuration$1.mainData.modules[i].name,
                        id: Configuration$1.mainData.modules[i].id,
                        navTabs: _this.getNavTabs(Configuration$1.mainData.modules[i]),
                        context: 'module',
                        module: Configuration$1.mainData.modules[i],
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    });
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterfaces = function (someInterfaces) {
        var _this = this;
        logger.info('Prepare interfaces');
        Configuration$1.mainData.interfaces = someInterfaces
            ? someInterfaces
            : DependenciesEngine$1.getInterfaces();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interfaces.length;
            var loop = function () {
                if (i < len) {
                    var interf = Configuration$1.mainData.interfaces[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interf.file)) {
                        logger.info(" " + interf.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interf.file);
                        interf.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interfaces',
                        name: interf.name,
                        id: interf.id,
                        navTabs: _this.getNavTabs(interf),
                        context: 'interface',
                        interface: interf,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interf.isDuplicate) {
                        page.name += '-' + interf.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareMiscellaneous = function (someMisc) {
        logger.info('Prepare miscellaneous');
        Configuration$1.mainData.miscellaneous = someMisc
            ? someMisc
            : DependenciesEngine$1.getMiscellaneous();
        return new Promise(function (resolve, reject) {
            if (Configuration$1.mainData.miscellaneous.functions.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'functions',
                    id: 'miscellaneous-functions',
                    context: 'miscellaneous-functions',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.variables.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'variables',
                    id: 'miscellaneous-variables',
                    context: 'miscellaneous-variables',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.typealiases.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'typealiases',
                    id: 'miscellaneous-typealiases',
                    context: 'miscellaneous-typealiases',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.enumerations.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'enumerations',
                    id: 'miscellaneous-enumerations',
                    context: 'miscellaneous-enumerations',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            resolve();
        });
    };
    Application.prototype.handleTemplateurl = function (component) {
        var dirname = path.dirname(component.file);
        var templatePath = path.resolve(dirname + path.sep + component.templateUrl);
        if (!FileEngine$1.existsSync(templatePath)) {
            var err = "Cannot read template for " + component.name;
            logger.error(err);
            return new Promise(function (resolve, reject) { });
        }
        return FileEngine$1.get(templatePath).then(function (data) { return (component.templateData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.handleStyles = function (component) {
        var styles = component.styles;
        component.stylesData = '';
        return new Promise(function (resolveStyles, rejectStyles) {
            styles.forEach(function (style) {
                component.stylesData = component.stylesData + style + '\n';
            });
            resolveStyles();
        });
    };
    Application.prototype.handleStyleurls = function (component) {
        var dirname = path.dirname(component.file);
        var styleDataPromise = component.styleUrls.map(function (styleUrl) {
            var stylePath = path.resolve(dirname + path.sep + styleUrl);
            if (!FileEngine$1.existsSync(stylePath)) {
                var err = "Cannot read style url " + stylePath + " for " + component.name;
                logger.error(err);
                return new Promise(function (resolve, reject) { });
            }
            return new Promise(function (resolve, reject) {
                FileEngine$1.get(stylePath).then(function (data) {
                    resolve({
                        data: data,
                        styleUrl: styleUrl
                    });
                });
            });
        });
        return Promise.all(styleDataPromise).then(function (data) { return (component.styleUrlsData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.getNavTabs = function (dependency) {
        var navTabConfig = Configuration$1.mainData.navTabConfig;
        navTabConfig =
            navTabConfig.length === 0
                ? _.cloneDeep(COMPODOC_CONSTANTS.navTabDefinitions)
                : navTabConfig;
        var matchDepType = function (depType) {
            return depType === 'all' || depType === dependency.type;
        };
        var navTabs = [];
        _.forEach(navTabConfig, function (customTab) {
            var navTab = _.find(COMPODOC_CONSTANTS.navTabDefinitions, { id: customTab.id });
            if (!navTab) {
                throw new Error("Invalid tab ID '" + customTab.id + "' specified in tab configuration");
            }
            navTab.label = customTab.label;
            // is tab applicable to target dependency?
            if (-1 === _.findIndex(navTab.depTypes, matchDepType)) {
                return;
            }
            // global config
            if (customTab.id === 'tree' && Configuration$1.mainData.disableDomTree) {
                return;
            }
            if (customTab.id === 'source' && Configuration$1.mainData.disableSourceCode) {
                return;
            }
            if (customTab.id === 'templateData' && Configuration$1.mainData.disableTemplateTab) {
                return;
            }
            if (customTab.id === 'styleData' && Configuration$1.mainData.disableStyleTab) {
                return;
            }
            // per dependency config
            if (customTab.id === 'readme' && !dependency.readme) {
                return;
            }
            if (customTab.id === 'example' && !dependency.exampleUrls) {
                return;
            }
            if (customTab.id === 'templateData' &&
                (!dependency.templateUrl || dependency.templateUrl.length === 0)) {
                return;
            }
            if (customTab.id === 'styleData' &&
                ((!dependency.styleUrls || dependency.styleUrls.length === 0) &&
                    (!dependency.styles || dependency.styles.length === 0))) {
                return;
            }
            navTabs.push(navTab);
        });
        if (navTabs.length === 0) {
            throw new Error("No valid navigation tabs have been defined for dependency type '" + dependency.type + "'. Specify at least one config for the 'info' or 'source' tab in --navTabConfig.");
        }
        return navTabs;
    };
    Application.prototype.prepareControllers = function (someControllers) {
        var _this = this;
        logger.info('Prepare controllers');
        Configuration$1.mainData.controllers = someControllers
            ? someControllers
            : DependenciesEngine$1.getControllers();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.controllers.length;
            var loop = function () {
                if (i < len) {
                    var controller = Configuration$1.mainData.controllers[i];
                    var page = {
                        path: 'controllers',
                        name: controller.name,
                        id: controller.id,
                        navTabs: _this.getNavTabs(controller),
                        context: 'controller',
                        controller: controller,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (controller.isDuplicate) {
                        page.name += '-' + controller.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareComponents = function (someComponents) {
        var _this = this;
        logger.info('Prepare components');
        Configuration$1.mainData.components = someComponents
            ? someComponents
            : DependenciesEngine$1.getComponents();
        return new Promise(function (mainPrepareComponentResolve, mainPrepareComponentReject) {
            var i = 0;
            var len = Configuration$1.mainData.components.length;
            var loop = function () {
                if (i <= len - 1) {
                    var component_1 = Configuration$1.mainData.components[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(component_1.file)) {
                        logger.info(" " + component_1.name + " has a README file, include it");
                        var readmeFile = MarkdownEngine$1.readNeighbourReadmeFile(component_1.file);
                        component_1.readme = marked$3(readmeFile);
                    }
                    var page = {
                        path: 'components',
                        name: component_1.name,
                        id: component_1.id,
                        navTabs: _this.getNavTabs(component_1),
                        context: 'component',
                        component: component_1,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (component_1.isDuplicate) {
                        page.name += '-' + component_1.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    var componentTemplateUrlPromise = new Promise(function (componentTemplateUrlResolve, componentTemplateUrlReject) {
                        if (component_1.templateUrl.length > 0) {
                            logger.info(" " + component_1.name + " has a templateUrl, include it");
                            _this.handleTemplateurl(component_1).then(function () {
                                componentTemplateUrlResolve();
                            }, function (e) {
                                logger.error(e);
                                componentTemplateUrlReject();
                            });
                        }
                        else {
                            componentTemplateUrlResolve();
                        }
                    });
                    var componentStyleUrlsPromise = new Promise(function (componentStyleUrlsResolve, componentStyleUrlsReject) {
                        if (component_1.styleUrls.length > 0) {
                            logger.info(" " + component_1.name + " has styleUrls, include them");
                            _this.handleStyleurls(component_1).then(function () {
                                componentStyleUrlsResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStyleUrlsReject();
                            });
                        }
                        else {
                            componentStyleUrlsResolve();
                        }
                    });
                    var componentStylesPromise = new Promise(function (componentStylesResolve, componentStylesReject) {
                        if (component_1.styles.length > 0) {
                            logger.info(" " + component_1.name + " has styles, include them");
                            _this.handleStyles(component_1).then(function () {
                                componentStylesResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStylesReject();
                            });
                        }
                        else {
                            componentStylesResolve();
                        }
                    });
                    Promise.all([
                        componentTemplateUrlPromise,
                        componentStyleUrlsPromise,
                        componentStylesPromise
                    ]).then(function () {
                        i++;
                        loop();
                    });
                }
                else {
                    mainPrepareComponentResolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareDirectives = function (someDirectives) {
        var _this = this;
        logger.info('Prepare directives');
        Configuration$1.mainData.directives = someDirectives
            ? someDirectives
            : DependenciesEngine$1.getDirectives();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.directives.length;
            var loop = function () {
                if (i < len) {
                    var directive = Configuration$1.mainData.directives[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(directive.file)) {
                        logger.info(" " + directive.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(directive.file);
                        directive.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'directives',
                        name: directive.name,
                        id: directive.id,
                        navTabs: _this.getNavTabs(directive),
                        context: 'directive',
                        directive: directive,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (directive.isDuplicate) {
                        page.name += '-' + directive.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInjectables = function (someInjectables) {
        var _this = this;
        logger.info('Prepare injectables');
        Configuration$1.mainData.injectables = someInjectables
            ? someInjectables
            : DependenciesEngine$1.getInjectables();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.injectables.length;
            var loop = function () {
                if (i < len) {
                    var injec = Configuration$1.mainData.injectables[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(injec.file)) {
                        logger.info(" " + injec.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(injec.file);
                        injec.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'injectables',
                        name: injec.name,
                        id: injec.id,
                        navTabs: _this.getNavTabs(injec),
                        context: 'injectable',
                        injectable: injec,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (injec.isDuplicate) {
                        page.name += '-' + injec.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterceptors = function (someInterceptors) {
        var _this = this;
        logger.info('Prepare interceptors');
        Configuration$1.mainData.interceptors = someInterceptors
            ? someInterceptors
            : DependenciesEngine$1.getInterceptors();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interceptors.length;
            var loop = function () {
                if (i < len) {
                    var interceptor = Configuration$1.mainData.interceptors[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interceptor.file)) {
                        logger.info(" " + interceptor.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interceptor.file);
                        interceptor.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interceptors',
                        name: interceptor.name,
                        id: interceptor.id,
                        navTabs: _this.getNavTabs(interceptor),
                        context: 'interceptor',
                        injectable: interceptor,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interceptor.isDuplicate) {
                        page.name += '-' + interceptor.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareGuards = function (someGuards) {
        var _this = this;
        logger.info('Prepare guards');
        Configuration$1.mainData.guards = someGuards ? someGuards : DependenciesEngine$1.getGuards();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.guards.length;
            var loop = function () {
                if (i < len) {
                    var guard = Configuration$1.mainData.guards[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(guard.file)) {
                        logger.info(" " + guard.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(guard.file);
                        guard.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'guards',
                        name: guard.name,
                        id: guard.id,
                        navTabs: _this.getNavTabs(guard),
                        context: 'guard',
                        injectable: guard,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (guard.isDuplicate) {
                        page.name += '-' + guard.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareRoutes = function () {
        logger.info('Process routes');
        Configuration$1.mainData.routes = DependenciesEngine$1.getRoutes();
        return new Promise(function (resolve, reject) {
            Configuration$1.addPage({
                name: 'routes',
                id: 'routes',
                context: 'routes',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                RouterParserUtil$1.generateRoutesIndex(Configuration$1.mainData.output, Configuration$1.mainData.routes).then(function () {
                    logger.info(' Routes index generated');
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareCoverage = function () {
        logger.info('Process documentation coverage report');
        return new Promise(function (resolve, reject) {
            /*
             * loop with components, directives, controllers, classes, injectables, interfaces, pipes, guards, misc functions variables
             */
            var files = [];
            var totalProjectStatementDocumented = 0;
            var getStatus = function (percent) {
                var status;
                if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var processComponentsAndDirectivesAndControllers = function (list) {
                _.forEach(list, function (el) {
                    var element = Object.assign({}, el);
                    if (!element.propertiesClass) {
                        element.propertiesClass = [];
                    }
                    if (!element.methodsClass) {
                        element.methodsClass = [];
                    }
                    if (!element.hostBindings) {
                        element.hostBindings = [];
                    }
                    if (!element.hostListeners) {
                        element.hostListeners = [];
                    }
                    if (!element.inputsClass) {
                        element.inputsClass = [];
                    }
                    if (!element.outputsClass) {
                        element.outputsClass = [];
                    }
                    var cl = {
                        filePath: element.file,
                        type: element.type,
                        linktype: element.type,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.propertiesClass.length +
                        element.methodsClass.length +
                        element.inputsClass.length +
                        element.hostBindings.length +
                        element.hostListeners.length +
                        element.outputsClass.length +
                        1; // +1 for element decorator comment
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.propertiesClass, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methodsClass, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostBindings, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostListeners, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.inputsClass, function (input) {
                        if (input.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (input.description &&
                            input.description !== '' &&
                            input.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.outputsClass, function (output) {
                        if (output.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (output.description &&
                            output.description !== '' &&
                            output.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cl.coveragePercent = 0;
                    }
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processCoveragePerFile = function () {
                logger.info('Process documentation coverage per file');
                logger.info('-------------------');
                var overFiles = files.filter(function (f) {
                    var overTest = f.coveragePercent >= Configuration$1.mainData.coverageMinimumPerFile;
                    if (overTest && !Configuration$1.mainData.coverageTestShowOnlyFailed) {
                        logger.info(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - over minimum per file");
                    }
                    return overTest;
                });
                var underFiles = files.filter(function (f) {
                    var underTest = f.coveragePercent < Configuration$1.mainData.coverageMinimumPerFile;
                    if (underTest) {
                        logger.error(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - under minimum per file");
                    }
                    return underTest;
                });
                logger.info('-------------------');
                return {
                    overFiles: overFiles,
                    underFiles: underFiles
                };
            };
            var processFunctionsAndVariables = function (id, type) {
                _.forEach(id, function (el) {
                    var cl = {
                        filePath: el.file,
                        type: type,
                        linktype: el.type,
                        linksubtype: el.subtype,
                        name: el.name
                    };
                    if (type === 'variable') {
                        cl.linktype = 'miscellaneous';
                    }
                    var totalStatementDocumented = 0;
                    var totalStatements = 1;
                    if (el.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                        // Doesn't handle private for coverage
                        totalStatements -= 1;
                    }
                    if (el.description &&
                        el.description !== '' &&
                        el.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                        totalStatementDocumented += 1;
                    }
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processClasses = function (list, type, linktype) {
                _.forEach(list, function (cl) {
                    var element = Object.assign({}, cl);
                    if (!element.properties) {
                        element.properties = [];
                    }
                    if (!element.methods) {
                        element.methods = [];
                    }
                    var cla = {
                        filePath: element.file,
                        type: type,
                        linktype: linktype,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.properties.length + element.methods.length + 1; // +1 for element itself
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.properties, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methods, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cla.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cla.coveragePercent = 0;
                    }
                    cla.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cla.status = getStatus(cla.coveragePercent);
                    totalProjectStatementDocumented += cla.coveragePercent;
                    files.push(cla);
                });
            };
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.components);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.directives);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.controllers);
            processClasses(Configuration$1.mainData.classes, 'class', 'classe');
            processClasses(Configuration$1.mainData.injectables, 'injectable', 'injectable');
            processClasses(Configuration$1.mainData.interfaces, 'interface', 'interface');
            processClasses(Configuration$1.mainData.guards, 'guard', 'guard');
            processClasses(Configuration$1.mainData.interceptors, 'interceptor', 'interceptor');
            _.forEach(Configuration$1.mainData.pipes, function (pipe) {
                var cl = {
                    filePath: pipe.file,
                    type: pipe.type,
                    linktype: pipe.type,
                    name: pipe.name
                };
                var totalStatementDocumented = 0;
                var totalStatements = 1;
                if (pipe.description && pipe.description !== '') {
                    totalStatementDocumented += 1;
                }
                cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                cl.status = getStatus(cl.coveragePercent);
                totalProjectStatementDocumented += cl.coveragePercent;
                files.push(cl);
            });
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.functions, 'function');
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.variables, 'variable');
            files = _.sortBy(files, ['filePath']);
            var coverageData = {
                count: files.length > 0
                    ? Math.floor(totalProjectStatementDocumented / files.length)
                    : 0,
                status: '',
                files: files
            };
            coverageData.status = getStatus(coverageData.count);
            Configuration$1.addPage({
                name: 'coverage',
                id: 'coverage',
                context: 'coverage',
                files: files,
                data: coverageData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            coverageData.files = files;
            Configuration$1.mainData.coverageData = coverageData;
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, 'documentation', coverageData);
            }
            files = _.sortBy(files, ['coveragePercent']);
            var coverageTestPerFileResults;
            if (Configuration$1.mainData.coverageTest &&
                !Configuration$1.mainData.coverageTestPerFile) {
                // Global coverage test and not per file
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
            }
            else if (!Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                coverageTestPerFileResults = processCoveragePerFile();
                // Per file coverage test and not global
                if (coverageTestPerFileResults.underFiles.length > 0) {
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else {
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
            }
            else if (Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                // Per file coverage test and global
                coverageTestPerFileResults = processCoveragePerFile();
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length === 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else if (coverageData.count < Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    var messageGlobal = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(messageGlobal);
                        logger.error(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(messageGlobal);
                        logger.warn(messagePerFile);
                        process.exit(0);
                    }
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        logger.info(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        logger.info(messagePerFile);
                        process.exit(0);
                    }
                }
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareUnitTestCoverage = function () {
        logger.info('Process unit test coverage report');
        return new Promise(function (resolve, reject) {
            var covDat, covFileNames;
            var coverageData = Configuration$1.mainData.coverageData;
            if (!coverageData.files) {
                logger.warn('Missing documentation coverage data');
            }
            else {
                covDat = {};
                covFileNames = _.map(coverageData.files, function (el) {
                    var fileName = el.filePath;
                    covDat[fileName] = {
                        type: el.type,
                        linktype: el.linktype,
                        linksubtype: el.linksubtype,
                        name: el.name
                    };
                    return fileName;
                });
            }
            // read coverage summary file and data
            var unitTestSummary = {};
            var fileDat = FileEngine$1.getSync(Configuration$1.mainData.unitTestCoverage);
            if (fileDat) {
                unitTestSummary = JSON.parse(fileDat);
            }
            else {
                return Promise.reject('Error reading unit test coverage file');
            }
            var getCovStatus = function (percent, totalLines) {
                var status;
                if (totalLines === 0) {
                    status = 'uncovered';
                }
                else if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var getCoverageData = function (data, fileName) {
                var out = {};
                if (fileName !== 'total') {
                    if (covDat === undefined) {
                        // need a name to include in output but this isn't visible
                        out = { name: fileName, filePath: fileName };
                    }
                    else {
                        var findMatch = _.filter(covFileNames, function (el) {
                            return el.includes(fileName) || fileName.includes(el);
                        });
                        if (findMatch.length > 0) {
                            out = _.clone(covDat[findMatch[0]]);
                            out['filePath'] = fileName;
                        }
                    }
                }
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (data[key]) {
                        var t = data[key];
                        out[key] = {
                            coveragePercent: Math.round(t.pct),
                            coverageCount: '' + t.covered + '/' + t.total,
                            status: getCovStatus(t.pct, t.total)
                        };
                    }
                });
                return out;
            };
            var unitTestData = {};
            var files = [];
            for (var file in unitTestSummary) {
                var dat = getCoverageData(unitTestSummary[file], file);
                if (file === 'total') {
                    unitTestData['total'] = dat;
                }
                else {
                    files.push(dat);
                }
            }
            unitTestData['files'] = files;
            unitTestData['idColumn'] = covDat !== undefined; // should we include the id column
            Configuration$1.mainData.unitTestData = unitTestData;
            Configuration$1.addPage({
                name: 'unit-test',
                id: 'unit-test',
                context: 'unit-test',
                files: files,
                data: unitTestData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (unitTestData['total'][key]) {
                        HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, key, {
                            count: unitTestData['total'][key]['coveragePercent'],
                            status: unitTestData['total'][key]['status']
                        });
                    }
                });
            }
            resolve();
        });
    };
    Application.prototype.processPage = function (page) {
        logger.info('Process page', page.name);
        var htmlData = HtmlEngine$1.render(Configuration$1.mainData, page);
        var finalPath = Configuration$1.mainData.output;
        if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
            finalPath += '/';
        }
        if (page.path) {
            finalPath += page.path + '/';
        }
        if (page.filename) {
            finalPath += page.filename + '.html';
        }
        else {
            finalPath += page.name + '.html';
        }
        if (!Configuration$1.mainData.disableSearch) {
            SearchEngine$1.indexPage({
                infos: page,
                rawData: htmlData,
                url: finalPath
            });
        }
        FileEngine$1.writeSync(finalPath, htmlData);
        return Promise.resolve();
    };
    Application.prototype.processPages = function () {
        var _this = this;
        var pages = _.sortBy(Configuration$1.pages, ['name']);
        logger.info('Process pages');
        Promise.all(pages.map(function (page) { return _this.processPage(page); }))
            .then(function () {
            var callbacksAfterGenerateSearchIndexJson = function () {
                if (Configuration$1.mainData.additionalPages.length > 0) {
                    _this.processAdditionalPages();
                }
                else {
                    if (Configuration$1.mainData.assetsFolder !== '') {
                        _this.processAssetsFolder();
                    }
                    _this.processResources();
                }
            };
            if (!Configuration$1.mainData.disableSearch) {
                SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                    callbacksAfterGenerateSearchIndexJson();
                }, function (e) {
                    logger.error(e);
                });
            }
            else {
                callbacksAfterGenerateSearchIndexJson();
            }
        })
            .then(function () {
            return _this.processMenu(Configuration$1.mainData);
        })
            .catch(function (e) {
            logger.error(e);
        });
    };
    Application.prototype.processMenu = function (mainData) {
        logger.info('Process menu...');
        return HtmlEngine$1.renderMenu(Configuration$1.mainData.templates, mainData).then(function (htmlData) {
            var finalPath = mainData.output + "/js/menu-wc.js";
            return FileEngine$1.write(finalPath, htmlData).catch(function (err) {
                logger.error('Error during ' + finalPath + ' page generation');
                return Promise.reject('');
            });
        });
    };
    Application.prototype.processAdditionalPages = function () {
        var _this = this;
        logger.info('Process additional pages');
        var pages = Configuration$1.mainData.additionalPages;
        Promise.all(pages.map(function (page) {
            if (page.children.length > 0) {
                return Promise.all([
                    _this.processPage(page)
                ].concat(page.children.map(function (childPage) { return _this.processPage(childPage); })));
            }
            else {
                return _this.processPage(page);
            }
        }))
            .then(function () {
            SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                if (Configuration$1.mainData.assetsFolder !== '') {
                    _this.processAssetsFolder();
                }
                _this.processResources();
            });
        })
            .catch(function (e) {
            logger.error(e);
            return Promise.reject(e);
        });
    };
    Application.prototype.processAssetsFolder = function () {
        logger.info('Copy assets folder');
        if (!FileEngine$1.existsSync(Configuration$1.mainData.assetsFolder)) {
            logger.error("Provided assets folder " + Configuration$1.mainData.assetsFolder + " did not exist");
        }
        else {
            var finalOutput = Configuration$1.mainData.output;
            var testOutputDir = Configuration$1.mainData.output.match(cwd);
            if (testOutputDir && testOutputDir.length > 0) {
                finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
            }
            var destination = path.join(finalOutput, path.basename(Configuration$1.mainData.assetsFolder));
            fs.copy(path.resolve(Configuration$1.mainData.assetsFolder), path.resolve(destination), function (err) {
                if (err) {
                    logger.error('Error during resources copy ', err);
                }
            });
        }
    };
    Application.prototype.processResources = function () {
        var _this = this;
        logger.info('Copy main resources');
        var onComplete = function () {
            logger.info('Documentation generated in ' +
                Configuration$1.mainData.output +
                ' in ' +
                _this.getElapsedTime() +
                ' seconds using ' +
                Configuration$1.mainData.theme +
                ' theme');
            if (Configuration$1.mainData.serve) {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + Configuration$1.mainData.port);
                _this.runWebServer(Configuration$1.mainData.output);
            }
            else {
                generationPromiseResolve();
                _this.endCallback();
            }
        };
        var finalOutput = Configuration$1.mainData.output;
        var testOutputDir = Configuration$1.mainData.output.match(cwd);
        if (testOutputDir && testOutputDir.length > 0) {
            finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
        }
        fs.copy(path.resolve(__dirname + '/../src/resources/'), path.resolve(finalOutput), function (errorCopy) {
            if (errorCopy) {
                logger.error('Error during resources copy ', errorCopy);
            }
            else {
                var extThemePromise = new Promise(function (extThemeResolve, extThemeReject) {
                    if (Configuration$1.mainData.extTheme) {
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.extTheme), path.resolve(finalOutput + '/styles/'), function (errorCopyTheme) {
                            if (errorCopyTheme) {
                                logger.error('Error during external styling theme copy ', errorCopyTheme);
                                extThemeReject();
                            }
                            else {
                                logger.info('External styling theme copy succeeded');
                                extThemeResolve();
                            }
                        });
                    }
                    else {
                        extThemeResolve();
                    }
                });
                var customFaviconPromise = new Promise(function (customFaviconResolve, customFaviconReject) {
                    if (Configuration$1.mainData.customFavicon !== '') {
                        logger.info("Custom favicon supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customFavicon), path.resolve(finalOutput + '/images/favicon.ico'), function (errorCopyFavicon) {
                            // tslint:disable-line
                            if (errorCopyFavicon) {
                                logger.error('Error during resources copy of favicon', errorCopyFavicon);
                                customFaviconReject();
                            }
                            else {
                                logger.info('External custom favicon copy succeeded');
                                customFaviconResolve();
                            }
                        });
                    }
                    else {
                        customFaviconResolve();
                    }
                });
                var customLogoPromise = new Promise(function (customLogoResolve, customLogoReject) {
                    if (Configuration$1.mainData.customLogo !== '') {
                        logger.info("Custom logo supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customLogo), path.resolve(finalOutput +
                            '/images/' +
                            Configuration$1.mainData.customLogo.split('/').pop()), function (errorCopyLogo) {
                            // tslint:disable-line
                            if (errorCopyLogo) {
                                logger.error('Error during resources copy of logo', errorCopyLogo);
                                customLogoReject();
                            }
                            else {
                                logger.info('External custom logo copy succeeded');
                                customLogoResolve();
                            }
                        });
                    }
                    else {
                        customLogoResolve();
                    }
                });
                Promise.all([extThemePromise, customFaviconPromise, customLogoPromise]).then(function () {
                    onComplete();
                });
            }
        });
    };
    /**
     * Calculates the elapsed time since the program was started.
     *
     * @returns {number}
     */
    Application.prototype.getElapsedTime = function () {
        return (new Date().valueOf() - startTime.valueOf()) / 1000;
    };
    Application.prototype.processGraphs = function () {
        var _this = this;
        if (Configuration$1.mainData.disableGraph) {
            logger.info('Graph generation disabled');
            this.processPages();
        }
        else {
            logger.info('Process main graph');
            var modules_1 = Configuration$1.mainData.modules;
            var i_1 = 0;
            var len_1 = modules_1.length;
            var loop_1 = function () {
                if (i_1 <= len_1 - 1) {
                    logger.info('Process module graph ', modules_1[i_1].name);
                    var finalPath_2 = Configuration$1.mainData.output;
                    if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
                        finalPath_2 += '/';
                    }
                    finalPath_2 += 'modules/' + modules_1[i_1].name;
                    var _rawModule = DependenciesEngine$1.getRawModule(modules_1[i_1].name);
                    if (_rawModule.declarations.length > 0 ||
                        _rawModule.bootstrap.length > 0 ||
                        _rawModule.imports.length > 0 ||
                        _rawModule.exports.length > 0 ||
                        _rawModule.providers.length > 0) {
                        NgdEngine$1.renderGraph(modules_1[i_1].file, finalPath_2, 'f', modules_1[i_1].name).then(function () {
                            NgdEngine$1.readGraph(path.resolve(finalPath_2 + path.sep + 'dependencies.svg'), modules_1[i_1].name).then(function (data) {
                                modules_1[i_1].graph = data;
                                i_1++;
                                loop_1();
                            }, function (err) {
                                logger.error('Error during graph read: ', err);
                            });
                        }, function (errorMessage) {
                            logger.error(errorMessage);
                        });
                    }
                    else {
                        i_1++;
                        loop_1();
                    }
                }
                else {
                    _this.processPages();
                }
            };
            var finalMainGraphPath_1 = Configuration$1.mainData.output;
            if (finalMainGraphPath_1.lastIndexOf('/') === -1) {
                finalMainGraphPath_1 += '/';
            }
            finalMainGraphPath_1 += 'graph';
            NgdEngine$1.init(path.resolve(finalMainGraphPath_1));
            NgdEngine$1.renderGraph(Configuration$1.mainData.tsconfig, path.resolve(finalMainGraphPath_1), 'p').then(function () {
                NgdEngine$1.readGraph(path.resolve(finalMainGraphPath_1 + path.sep + 'dependencies.svg'), 'Main graph').then(function (data) {
                    Configuration$1.mainData.mainGraph = data;
                    loop_1();
                }, function (err) {
                    logger.error('Error during main graph reading : ', err);
                    Configuration$1.mainData.disableMainGraph = true;
                    loop_1();
                });
            }, function (err) {
                logger.error('Ooops error during main graph generation, moving on next part with main graph disabled : ', err);
                Configuration$1.mainData.disableMainGraph = true;
                loop_1();
            });
        }
    };
    Application.prototype.runWebServer = function (folder) {
        if (!this.isWatching) {
            var liveServerConfiguration = {
                root: folder,
                open: Configuration$1.mainData.open,
                quiet: true,
                logLevel: 0,
                wait: 1000,
                port: Configuration$1.mainData.port
            };
            if (Configuration$1.mainData.host !== '') {
                liveServerConfiguration.host = Configuration$1.mainData.host;
            }
            LiveServer.start(liveServerConfiguration);
        }
        if (Configuration$1.mainData.watch && !this.isWatching) {
            if (typeof this.files === 'undefined') {
                logger.error('No sources files available, please use -p flag');
                generationPromiseReject();
                process.exit(1);
            }
            else {
                this.runWatch();
            }
        }
        else if (Configuration$1.mainData.watch && this.isWatching) {
            var srcFolder = findMainSourceFolder(this.files);
            logger.info("Already watching sources in " + srcFolder + " folder");
        }
    };
    Application.prototype.runWatch = function () {
        var _this = this;
        var sources = [findMainSourceFolder(this.files)];
        var watcherReady = false;
        this.isWatching = true;
        logger.info("Watching sources in " + findMainSourceFolder(this.files) + " folder");
        if (MarkdownEngine$1.hasRootMarkdowns()) {
            sources = sources.concat(MarkdownEngine$1.listRootMarkdowns());
        }
        if (Configuration$1.mainData.includes !== '') {
            sources = sources.concat(Configuration$1.mainData.includes);
        }
        // Check all elements of sources list exist
        sources = cleanSourcesForWatch(sources);
        var watcher = chokidar.watch(sources, {
            awaitWriteFinish: true,
            ignoreInitial: true,
            ignored: /(spec|\.d)\.ts/
        });
        var timerAddAndRemoveRef;
        var timerChangeRef;
        var runnerAddAndRemove = function () {
            startTime = new Date();
            _this.generate();
        };
        var waiterAddAndRemove = function () {
            clearTimeout(timerAddAndRemoveRef);
            timerAddAndRemoveRef = setTimeout(runnerAddAndRemove, 1000);
        };
        var runnerChange = function () {
            startTime = new Date();
            _this.setUpdatedFiles(_this.watchChangedFiles);
            if (_this.hasWatchedFilesTSFiles()) {
                _this.getMicroDependenciesData();
            }
            else if (_this.hasWatchedFilesRootMarkdownFiles()) {
                _this.rebuildRootMarkdowns();
            }
            else {
                _this.rebuildExternalDocumentation();
            }
        };
        var waiterChange = function () {
            clearTimeout(timerChangeRef);
            timerChangeRef = setTimeout(runnerChange, 1000);
        };
        watcher.on('ready', function () {
            if (!watcherReady) {
                watcherReady = true;
                watcher
                    .on('add', function (file) {
                    logger.debug("File " + file + " has been added");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                })
                    .on('change', function (file) {
                    logger.debug("File " + file + " has been changed");
                    // Test extension, if ts
                    // rescan only file
                    if (path.extname(file) === '.ts' ||
                        path.extname(file) === '.md' ||
                        path.extname(file) === '.json') {
                        _this.watchChangedFiles.push(path.join(cwd + path.sep + file));
                        waiterChange();
                    }
                })
                    .on('unlink', function (file) {
                    logger.debug("File " + file + " has been removed");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                });
            }
        });
    };
    Object.defineProperty(Application.prototype, "application", {
        /**
         * Return the application / root component instance.
         */
        get: function () {
            return this;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Application.prototype, "isCLI", {
        get: function () {
            return false;
        },
        enumerable: true,
        configurable: true
    });
    return Application;
}());

var glob = require('glob');
var ParserUtil = /** @class */ (function () {
    function ParserUtil() {
        this._globFiles = [];
    }
    ParserUtil.prototype.init = function (exclude, cwd) {
        this._files = exclude;
        this._cwd = cwd;
        var i = 0;
        var len = exclude.length;
        for (i; i < len; i++) {
            this._globFiles = this._globFiles.concat(glob.sync(exclude[i], { cwd: this._cwd }));
        }
    };
    ParserUtil.prototype.testFilesWithCwdDepth = function () {
        var i = 0;
        var len = this._files.length;
        var result = {
            status: true,
            level: 0
        };
        for (i; i < len; i++) {
            var elementPath = path.resolve(this._cwd + path.sep, this._files[i]);
            if (elementPath.indexOf(this._cwd) === -1) {
                result.status = false;
                var level = this._files[i].match(/\..\//g).length;
                if (level > result.level) {
                    result.level = level;
                }
            }
        }
        return result;
    };
    ParserUtil.prototype.updateCwd = function (cwd, level) {
        var _cwd = cwd, _rewind = '';
        for (var i = 0; i < level; i++) {
            _rewind += '../';
        }
        _cwd = path.resolve(_cwd, _rewind);
        return _cwd;
    };
    ParserUtil.prototype.testFile = function (file) {
        var _this = this;
        var i = 0;
        var len = this._files.length;
        var fileBasename = path.basename(file);
        var fileNameInCwd = file.replace(this._cwd + path.sep, '');
        var result = false;
        if (path.sep === '\\') {
            fileNameInCwd = fileNameInCwd.replace(new RegExp('\\' + path.sep, 'g'), '/');
        }
        for (i; i < len; i++) {
            if (glob.hasMagic(this._files[i]) && this._globFiles.length > 0) {
                var resultGlobSearch = this._globFiles.findIndex(function (element) {
                    var elementPath = path.resolve(_this._cwd + path.sep, element);
                    var elementPathInCwd = elementPath.replace(_this._cwd + path.sep, '');
                    elementPathInCwd = elementPathInCwd.replace(new RegExp('\\' + path.sep, 'g'), '/');
                    return elementPathInCwd === fileNameInCwd;
                });
                result = resultGlobSearch !== -1;
            }
            else {
                result = fileNameInCwd === this._files[i];
            }
            if (result) {
                break;
            }
        }
        return result;
    };
    return ParserUtil;
}());

var cosmiconfig = require('cosmiconfig');
var os = require('os');
var osName = require('os-name');
var pkg$1 = require('../package.json');
var program = require('commander');
var cosmiconfigModuleName = 'compodoc';
var scannedFiles = [];
var excludeFiles;
var includeFiles;
var cwd$1 = process.cwd();
process.setMaxListeners(0);
var CliApplication = /** @class */ (function (_super) {
    __extends(CliApplication, _super);
    function CliApplication() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Run compodoc from the command line.
     */
    CliApplication.prototype.generate = function () {
        var _this = this;
        function list(val) {
            return val.split(',');
        }
        program
            .version(pkg$1.version)
            .usage('<src> [options]')
            .option('-c, --config [config]', 'A configuration file : .compodocrc, .compodocrc.json, .compodocrc.yaml or compodoc property in package.json')
            .option('-p, --tsconfig [config]', 'A tsconfig.json file')
            .option('-d, --output [folder]', 'Where to store the generated documentation', COMPODOC_DEFAULTS.folder)
            .option('-y, --extTheme [file]', 'External styling theme file')
            .option('-n, --name [name]', 'Title documentation', COMPODOC_DEFAULTS.title)
            .option('-a, --assetsFolder [folder]', 'External assets folder to copy in generated documentation folder')
            .option('-o, --open [value]', 'Open the generated documentation')
            .option('-t, --silent', "In silent mode, log messages aren't logged in the console", false)
            .option('-s, --serve', 'Serve generated documentation (default http://localhost:8080/)', false)
            .option('--host [host]', 'Change default host address')
            .option('-r, --port [port]', 'Change default serving port', COMPODOC_DEFAULTS.port)
            .option('-w, --watch', 'Watch source files after serve and force documentation rebuild', false)
            .option('-e, --exportFormat [format]', 'Export in specified format (json, html)', COMPODOC_DEFAULTS.exportFormat)
            .option('--files [files]', 'Files provided by external tool, used for coverage test')
            .option('--language [language]', 'Language used for the generated documentation (en-US, es-ES, fr-FR, hu-HU, it-IT, ja-JP, nl-NL, pt-BR, zh-CN)', COMPODOC_DEFAULTS.language)
            .option('--theme [theme]', "Choose one of available themes, default is 'gitbook' (laravel, original, material, postmark, readthedocs, stripe, vagrant)")
            .option('--hideGenerator', 'Do not print the Compodoc link at the bottom of the page', false)
            .option('--toggleMenuItems <items>', "Close by default items in the menu values : ['all'] or one of these ['modules','components','directives','controllers','classes','injectables','guards','interfaces','interceptors','pipes','miscellaneous','additionalPages']", list, COMPODOC_DEFAULTS.toggleMenuItems)
            .option('--navTabConfig <tab configs>', "List navigation tab objects in the desired order with two string properties (\"id\" and \"label\"). Double-quotes must be escaped with '\\'. Available tab IDs are \"info\", \"readme\", \"source\", \"templateData\", \"styleData\", \"tree\", and \"example\". Note: Certain tabs will only be shown if applicable to a given dependency", list, JSON.stringify(COMPODOC_DEFAULTS.navTabConfig))
            .option('--templates [folder]', 'Path to directory of Handlebars templates to override built-in templates')
            .option('--includes [path]', 'Path of external markdown files to include')
            .option('--includesName [name]', 'Name of item menu of externals markdown files', COMPODOC_DEFAULTS.additionalEntryName)
            .option('--coverageTest [threshold]', 'Test command of documentation coverage with a threshold (default 70)')
            .option('--coverageMinimumPerFile [minimum]', 'Test command of documentation coverage per file with a minimum (default 0)')
            .option('--coverageTestThresholdFail [true|false]', 'Test command of documentation coverage (global or per file) will fail with error or just warn user (true: error, false: warn)', COMPODOC_DEFAULTS.coverageTestThresholdFail)
            .option('--coverageTestShowOnlyFailed', 'Display only failed files for a coverage test')
            .option('--unitTestCoverage [json-summary]', 'To include unit test coverage, specify istanbul JSON coverage summary file')
            .option('--disableSourceCode', 'Do not add source code tab and links to source code', false)
            .option('--disableDomTree', 'Do not add dom tree tab', false)
            .option('--disableTemplateTab', 'Do not add template tab', false)
            .option('--disableStyleTab', 'Do not add style tab', false)
            .option('--disableGraph', 'Do not add the dependency graph', false)
            .option('--disableCoverage', 'Do not add the documentation coverage report', false)
            .option('--disablePrivate', 'Do not show private in generated documentation', false)
            .option('--disableProtected', 'Do not show protected in generated documentation', false)
            .option('--disableInternal', 'Do not show @internal in generated documentation', false)
            .option('--disableLifeCycleHooks', 'Do not show Angular lifecycle hooks in generated documentation', false)
            .option('--disableRoutesGraph', 'Do not add the routes graph', COMPODOC_DEFAULTS.disableRoutesGraph)
            .option('--disableSearch', 'Do not add the search input', false)
            .option('--disableDependencies', 'Do not add the dependencies list', COMPODOC_DEFAULTS.disableDependencies)
            .option('--minimal', 'Minimal mode with only documentation. No search, no graph, no coverage.', false)
            .option('--customFavicon [path]', 'Use a custom favicon')
            .option('--customLogo [path]', 'Use a custom logo')
            .option('--gaID [id]', 'Google Analytics tracking ID')
            .option('--gaSite [site]', 'Google Analytics site name', COMPODOC_DEFAULTS.gaSite)
            .parse(process.argv);
        var outputHelp = function () {
            program.outputHelp();
            process.exit(1);
        };
        var configExplorer = cosmiconfig(cosmiconfigModuleName);
        var configExplorerResult;
        var configFile = {};
        if (program.config) {
            var configFilePath = program.config;
            var testConfigFilePath = configFilePath.match(process.cwd());
            if (testConfigFilePath && testConfigFilePath.length > 0) {
                configFilePath = configFilePath.replace(process.cwd() + path.sep, '');
            }
            configExplorerResult = configExplorer.loadSync(path.resolve(configFilePath));
        }
        else {
            configExplorerResult = configExplorer.searchSync();
        }
        if (configExplorerResult) {
            if (typeof configExplorerResult.config !== 'undefined') {
                configFile = configExplorerResult.config;
            }
        }
        if (configFile.output) {
            Configuration$1.mainData.output = configFile.output;
        }
        if (program.output && program.output !== COMPODOC_DEFAULTS.folder) {
            Configuration$1.mainData.output = program.output;
        }
        if (configFile.extTheme) {
            Configuration$1.mainData.extTheme = configFile.extTheme;
        }
        if (program.extTheme) {
            Configuration$1.mainData.extTheme = program.extTheme;
        }
        if (configFile.language) {
            Configuration$1.mainData.language = configFile.language;
        }
        if (program.language) {
            Configuration$1.mainData.language = program.language;
        }
        if (configFile.theme) {
            Configuration$1.mainData.theme = configFile.theme;
        }
        if (program.theme) {
            Configuration$1.mainData.theme = program.theme;
        }
        if (configFile.name) {
            Configuration$1.mainData.documentationMainName = configFile.name;
        }
        if (program.name && program.name !== COMPODOC_DEFAULTS.title) {
            Configuration$1.mainData.documentationMainName = program.name;
        }
        if (configFile.assetsFolder) {
            Configuration$1.mainData.assetsFolder = configFile.assetsFolder;
        }
        if (program.assetsFolder) {
            Configuration$1.mainData.assetsFolder = program.assetsFolder;
        }
        if (configFile.open) {
            Configuration$1.mainData.open = configFile.open;
        }
        if (program.open) {
            Configuration$1.mainData.open = program.open;
        }
        if (configFile.toggleMenuItems) {
            Configuration$1.mainData.toggleMenuItems = configFile.toggleMenuItems;
        }
        if (program.toggleMenuItems &&
            program.toggleMenuItems !== COMPODOC_DEFAULTS.toggleMenuItems) {
            Configuration$1.mainData.toggleMenuItems = program.toggleMenuItems;
        }
        if (configFile.templates) {
            Configuration$1.mainData.templates = configFile.templates;
        }
        if (program.templates) {
            Configuration$1.mainData.templates = program.templates;
        }
        if (configFile.navTabConfig) {
            Configuration$1.mainData.navTabConfig = configFile.navTabConfig;
        }
        if (program.navTabConfig &&
            JSON.parse(program.navTabConfig).length !== COMPODOC_DEFAULTS.navTabConfig.length) {
            Configuration$1.mainData.navTabConfig = JSON.parse(program.navTabConfig);
        }
        if (configFile.includes) {
            Configuration$1.mainData.includes = configFile.includes;
        }
        if (program.includes) {
            Configuration$1.mainData.includes = program.includes;
        }
        if (configFile.includesName) {
            Configuration$1.mainData.includesName = configFile.includesName;
        }
        if (program.includesName &&
            program.includesName !== COMPODOC_DEFAULTS.additionalEntryName) {
            Configuration$1.mainData.includesName = program.includesName;
        }
        if (configFile.silent) {
            logger.silent = false;
        }
        if (program.silent) {
            logger.silent = false;
        }
        if (configFile.serve) {
            Configuration$1.mainData.serve = configFile.serve;
        }
        if (program.serve) {
            Configuration$1.mainData.serve = program.serve;
        }
        if (configFile.host) {
            Configuration$1.mainData.host = configFile.host;
            Configuration$1.mainData.hostname = configFile.host;
        }
        if (program.host) {
            Configuration$1.mainData.host = program.host;
            Configuration$1.mainData.hostname = program.host;
        }
        if (configFile.port) {
            Configuration$1.mainData.port = configFile.port;
        }
        if (program.port && program.port !== COMPODOC_DEFAULTS.port) {
            Configuration$1.mainData.port = program.port;
        }
        if (configFile.watch) {
            Configuration$1.mainData.watch = configFile.watch;
        }
        if (program.watch) {
            Configuration$1.mainData.watch = program.watch;
        }
        if (configFile.exportFormat) {
            Configuration$1.mainData.exportFormat = configFile.exportFormat;
        }
        if (program.exportFormat && program.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
            Configuration$1.mainData.exportFormat = program.exportFormat;
        }
        if (configFile.hideGenerator) {
            Configuration$1.mainData.hideGenerator = configFile.hideGenerator;
        }
        if (program.hideGenerator) {
            Configuration$1.mainData.hideGenerator = program.hideGenerator;
        }
        if (configFile.coverageTest) {
            Configuration$1.mainData.coverageTest = true;
            Configuration$1.mainData.coverageTestThreshold =
                typeof configFile.coverageTest === 'string'
                    ? parseInt(configFile.coverageTest, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageThreshold;
        }
        if (program.coverageTest) {
            Configuration$1.mainData.coverageTest = true;
            Configuration$1.mainData.coverageTestThreshold =
                typeof program.coverageTest === 'string'
                    ? parseInt(program.coverageTest, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageThreshold;
        }
        if (configFile.coverageMinimumPerFile) {
            Configuration$1.mainData.coverageTestPerFile = true;
            Configuration$1.mainData.coverageMinimumPerFile =
                typeof configFile.coverageMinimumPerFile === 'string'
                    ? parseInt(configFile.coverageMinimumPerFile, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile;
        }
        if (program.coverageMinimumPerFile) {
            Configuration$1.mainData.coverageTestPerFile = true;
            Configuration$1.mainData.coverageMinimumPerFile =
                typeof program.coverageMinimumPerFile === 'string'
                    ? parseInt(program.coverageMinimumPerFile, 10)
                    : COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile;
        }
        if (configFile.coverageTestThresholdFail) {
            Configuration$1.mainData.coverageTestThresholdFail =
                configFile.coverageTestThresholdFail === 'false' ? false : true;
        }
        if (program.coverageTestThresholdFail) {
            Configuration$1.mainData.coverageTestThresholdFail =
                program.coverageTestThresholdFail === 'false' ? false : true;
        }
        if (configFile.coverageTestShowOnlyFailed) {
            Configuration$1.mainData.coverageTestShowOnlyFailed =
                configFile.coverageTestShowOnlyFailed;
        }
        if (program.coverageTestShowOnlyFailed) {
            Configuration$1.mainData.coverageTestShowOnlyFailed = program.coverageTestShowOnlyFailed;
        }
        if (configFile.unitTestCoverage) {
            Configuration$1.mainData.unitTestCoverage = configFile.unitTestCoverage;
        }
        if (program.unitTestCoverage) {
            Configuration$1.mainData.unitTestCoverage = program.unitTestCoverage;
        }
        if (configFile.disableSourceCode) {
            Configuration$1.mainData.disableSourceCode = configFile.disableSourceCode;
        }
        if (program.disableSourceCode) {
            Configuration$1.mainData.disableSourceCode = program.disableSourceCode;
        }
        if (configFile.disableDomTree) {
            Configuration$1.mainData.disableDomTree = configFile.disableDomTree;
        }
        if (program.disableDomTree) {
            Configuration$1.mainData.disableDomTree = program.disableDomTree;
        }
        if (configFile.disableTemplateTab) {
            Configuration$1.mainData.disableTemplateTab = configFile.disableTemplateTab;
        }
        if (program.disableTemplateTab) {
            Configuration$1.mainData.disableTemplateTab = program.disableTemplateTab;
        }
        if (configFile.disableStyleTab) {
            Configuration$1.mainData.disableStyleTab = configFile.disableStyleTab;
        }
        if (program.disableStyleTab) {
            Configuration$1.mainData.disableStyleTab = program.disableStyleTab;
        }
        if (configFile.disableGraph) {
            Configuration$1.mainData.disableGraph = configFile.disableGraph;
        }
        if (program.disableGraph) {
            Configuration$1.mainData.disableGraph = program.disableGraph;
        }
        if (configFile.disableCoverage) {
            Configuration$1.mainData.disableCoverage = configFile.disableCoverage;
        }
        if (program.disableCoverage) {
            Configuration$1.mainData.disableCoverage = program.disableCoverage;
        }
        if (configFile.disablePrivate) {
            Configuration$1.mainData.disablePrivate = configFile.disablePrivate;
        }
        if (program.disablePrivate) {
            Configuration$1.mainData.disablePrivate = program.disablePrivate;
        }
        if (configFile.disableProtected) {
            Configuration$1.mainData.disableProtected = configFile.disableProtected;
        }
        if (program.disableProtected) {
            Configuration$1.mainData.disableProtected = program.disableProtected;
        }
        if (configFile.disableInternal) {
            Configuration$1.mainData.disableInternal = configFile.disableInternal;
        }
        if (program.disableInternal) {
            Configuration$1.mainData.disableInternal = program.disableInternal;
        }
        if (configFile.disableLifeCycleHooks) {
            Configuration$1.mainData.disableLifeCycleHooks = configFile.disableLifeCycleHooks;
        }
        if (program.disableLifeCycleHooks) {
            Configuration$1.mainData.disableLifeCycleHooks = program.disableLifeCycleHooks;
        }
        if (configFile.disableRoutesGraph) {
            Configuration$1.mainData.disableRoutesGraph = configFile.disableRoutesGraph;
        }
        if (program.disableRoutesGraph) {
            Configuration$1.mainData.disableRoutesGraph = program.disableRoutesGraph;
        }
        if (configFile.disableSearch) {
            Configuration$1.mainData.disableSearch = configFile.disableSearch;
        }
        if (program.disableSearch) {
            Configuration$1.mainData.disableSearch = program.disableSearch;
        }
        if (configFile.disableDependencies) {
            Configuration$1.mainData.disableDependencies = configFile.disableDependencies;
        }
        if (program.disableDependencies) {
            Configuration$1.mainData.disableDependencies = program.disableDependencies;
        }
        if (configFile.minimal) {
            Configuration$1.mainData.disableSearch = true;
            Configuration$1.mainData.disableRoutesGraph = true;
            Configuration$1.mainData.disableGraph = true;
            Configuration$1.mainData.disableCoverage = true;
        }
        if (program.minimal) {
            Configuration$1.mainData.disableSearch = true;
            Configuration$1.mainData.disableRoutesGraph = true;
            Configuration$1.mainData.disableGraph = true;
            Configuration$1.mainData.disableCoverage = true;
        }
        if (configFile.customFavicon) {
            Configuration$1.mainData.customFavicon = configFile.customFavicon;
        }
        if (program.customFavicon) {
            Configuration$1.mainData.customFavicon = program.customFavicon;
        }
        if (configFile.customLogo) {
            Configuration$1.mainData.customLogo = configFile.customLogo;
        }
        if (program.customLogo) {
            Configuration$1.mainData.customLogo = program.customLogo;
        }
        if (configFile.gaID) {
            Configuration$1.mainData.gaID = configFile.gaID;
        }
        if (program.gaID) {
            Configuration$1.mainData.gaID = program.gaID;
        }
        if (configFile.gaSite) {
            Configuration$1.mainData.gaSite = configFile.gaSite;
        }
        if (program.gaSite && program.gaSite !== COMPODOC_DEFAULTS.gaSite) {
            Configuration$1.mainData.gaSite = program.gaSite;
        }
        if (!this.isWatching) {
            if (!logger.silent) {
                console.log("Compodoc v" + pkg$1.version);
            }
            else {
                console.log(fs.readFileSync(path.join(__dirname, '../src/banner')).toString());
                console.log(pkg$1.version);
                console.log('');
                console.log("TypeScript version used by Compodoc : " + Ast.ts.version);
                console.log('');
                if (FileEngine$1.existsSync(cwd$1 + path.sep + 'package.json')) {
                    var packageData = FileEngine$1.getSync(cwd$1 + path.sep + 'package.json');
                    if (packageData) {
                        var parsedData = JSON.parse(packageData);
                        var projectDevDependencies = parsedData.devDependencies;
                        if (projectDevDependencies && projectDevDependencies.typescript) {
                            var tsProjectVersion = AngularVersionUtil$1.cleanVersion(projectDevDependencies.typescript);
                            console.log("TypeScript version of current project : " + tsProjectVersion);
                            console.log('');
                        }
                    }
                }
                console.log("Node.js version : " + process.version);
                console.log('');
                console.log("Operating system : " + osName(os.platform(), os.release()));
                console.log('');
            }
        }
        if (configExplorerResult) {
            if (typeof configExplorerResult.config !== 'undefined') {
                logger.info("Using configuration file : " + configExplorerResult.filepath);
            }
        }
        if (!configExplorerResult) {
            logger.warn("No configuration file found, switching to CLI flags.");
        }
        if (program.language && !I18nEngine$1.supportLanguage(program.language)) {
            logger.warn("The language " + program.language + " is not available, falling back to " + I18nEngine$1.fallbackLanguage);
        }
        if (program.tsconfig && typeof program.tsconfig === 'boolean') {
            logger.error("Please provide a tsconfig file.");
            process.exit(1);
        }
        if (configFile.tsconfig) {
            Configuration$1.mainData.tsconfig = configFile.tsconfig;
        }
        if (program.tsconfig) {
            Configuration$1.mainData.tsconfig = program.tsconfig;
        }
        if (configFile.files) {
            scannedFiles = configFile.files;
        }
        if (configFile.exclude) {
            excludeFiles = configFile.exclude;
        }
        if (configFile.include) {
            includeFiles = configFile.include;
        }
        /**
         * Check --files argument call
         */
        var argv = require('minimist')(process.argv.slice(2));
        if (argv && argv.files) {
            Configuration$1.mainData.hasFilesToCoverage = true;
            if (typeof argv.files === 'string') {
                _super.prototype.setFiles.call(this, [argv.files]);
            }
            else {
                _super.prototype.setFiles.call(this, argv.files);
            }
        }
        if (program.serve && !Configuration$1.mainData.tsconfig && program.output) {
            // if -s & -d, serve it
            if (!FileEngine$1.existsSync(Configuration$1.mainData.output)) {
                logger.error(Configuration$1.mainData.output + " folder doesn't exist");
                process.exit(1);
            }
            else {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + program.port);
                _super.prototype.runWebServer.call(this, Configuration$1.mainData.output);
            }
        }
        else if (program.serve && !Configuration$1.mainData.tsconfig && !program.output) {
            // if only -s find ./documentation, if ok serve, else error provide -d
            if (!FileEngine$1.existsSync(Configuration$1.mainData.output)) {
                logger.error('Provide output generated folder with -d flag');
                process.exit(1);
            }
            else {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + program.port);
                _super.prototype.runWebServer.call(this, Configuration$1.mainData.output);
            }
        }
        else if (Configuration$1.mainData.hasFilesToCoverage) {
            if (program.coverageMinimumPerFile) {
                logger.info('Run documentation coverage test for files');
                _super.prototype.testCoverage.call(this);
            }
            else {
                logger.error('Missing coverage configuration');
            }
        }
        else {
            if (program.hideGenerator) {
                Configuration$1.mainData.hideGenerator = true;
            }
            if (Configuration$1.mainData.tsconfig && program.args.length === 0) {
                /**
                 * tsconfig file provided only
                 */
                var testTsConfigPath = Configuration$1.mainData.tsconfig.indexOf(process.cwd());
                if (testTsConfigPath !== -1) {
                    Configuration$1.mainData.tsconfig = Configuration$1.mainData.tsconfig.replace(process.cwd() + path.sep, '');
                }
                if (!FileEngine$1.existsSync(Configuration$1.mainData.tsconfig)) {
                    logger.error("\"" + Configuration$1.mainData.tsconfig + "\" file was not found in the current directory");
                    process.exit(1);
                }
                else {
                    var _file = path.join(path.join(process.cwd(), path.dirname(Configuration$1.mainData.tsconfig)), path.basename(Configuration$1.mainData.tsconfig));
                    // use the current directory of tsconfig.json as a working directory
                    cwd$1 = _file
                        .split(path.sep)
                        .slice(0, -1)
                        .join(path.sep);
                    logger.info('Using tsconfig file ', _file);
                    var tsConfigFile = readConfig(_file);
                    scannedFiles = tsConfigFile.files;
                    if (scannedFiles) {
                        scannedFiles = handlePath(scannedFiles, cwd$1);
                    }
                    if (typeof scannedFiles === 'undefined') {
                        excludeFiles = tsConfigFile.exclude || [];
                        includeFiles = tsConfigFile.include || [];
                        scannedFiles = [];
                        var excludeParser_1 = new ParserUtil(), includeParser_1 = new ParserUtil();
                        excludeParser_1.init(excludeFiles, cwd$1);
                        includeParser_1.init(includeFiles, cwd$1);
                        var startCwd = cwd$1;
                        var excludeParserTestFilesWithCwdDepth = excludeParser_1.testFilesWithCwdDepth();
                        if (!excludeParserTestFilesWithCwdDepth.status) {
                            startCwd = excludeParser_1.updateCwd(cwd$1, excludeParserTestFilesWithCwdDepth.level);
                        }
                        var includeParserTestFilesWithCwdDepth = includeParser_1.testFilesWithCwdDepth();
                        if (!includeParser_1.testFilesWithCwdDepth().status) {
                            startCwd = includeParser_1.updateCwd(cwd$1, includeParserTestFilesWithCwdDepth.level);
                        }
                        var finder = require('findit2')(startCwd || '.');
                        finder.on('directory', function (dir, stat, stop) {
                            var base = path.basename(dir);
                            if (base === '.git' || base === 'node_modules') {
                                stop();
                            }
                        });
                        finder.on('file', function (file, stat) {
                            if (/(spec|\.d)\.ts/.test(file)) {
                                logger.warn('Ignoring', file);
                            }
                            else if (excludeParser_1.testFile(file) &&
                                path.extname(file) === '.ts') {
                                logger.warn('Excluding', file);
                            }
                            else if (includeFiles.length > 0) {
                                /**
                                 * If include provided in tsconfig, use only this source,
                                 * and not files found with global findit scan in working directory
                                 */
                                if (path.extname(file) === '.ts' && includeParser_1.testFile(file)) {
                                    logger.debug('Including', file);
                                    scannedFiles.push(file);
                                }
                                else {
                                    if (path.extname(file) === '.ts') {
                                        logger.warn('Excluding', file);
                                    }
                                }
                            }
                            else {
                                logger.debug('Including', file);
                                scannedFiles.push(file);
                            }
                        });
                        finder.on('end', function () {
                            _super.prototype.setFiles.call(_this, scannedFiles);
                            if (program.coverageTest || program.coverageTestPerFile) {
                                logger.info('Run documentation coverage test');
                                _super.prototype.testCoverage.call(_this);
                            }
                            else {
                                _super.prototype.generate.call(_this);
                            }
                        });
                    }
                    else {
                        _super.prototype.setFiles.call(this, scannedFiles);
                        if (program.coverageTest || program.coverageTestPerFile) {
                            logger.info('Run documentation coverage test');
                            _super.prototype.testCoverage.call(this);
                        }
                        else {
                            _super.prototype.generate.call(this);
                        }
                    }
                }
            }
            else if (Configuration$1.mainData.tsconfig && program.args.length > 0) {
                /**
                 * tsconfig file provided with source folder in arg
                 */
                var testTsConfigPath = Configuration$1.mainData.tsconfig.indexOf(process.cwd());
                if (testTsConfigPath !== -1) {
                    Configuration$1.mainData.tsconfig = Configuration$1.mainData.tsconfig.replace(process.cwd() + path.sep, '');
                }
                var sourceFolder = program.args[0];
                if (!FileEngine$1.existsSync(sourceFolder)) {
                    logger.error("Provided source folder " + sourceFolder + " was not found in the current directory");
                    process.exit(1);
                }
                else {
                    logger.info('Using provided source folder');
                    if (!FileEngine$1.existsSync(Configuration$1.mainData.tsconfig)) {
                        logger.error("\"" + Configuration$1.mainData.tsconfig + "\" file was not found in the current directory");
                        process.exit(1);
                    }
                    else {
                        var _file = path.join(path.join(process.cwd(), path.dirname(Configuration$1.mainData.tsconfig)), path.basename(Configuration$1.mainData.tsconfig));
                        // use the current directory of tsconfig.json as a working directory
                        cwd$1 = _file
                            .split(path.sep)
                            .slice(0, -1)
                            .join(path.sep);
                        logger.info('Using tsconfig file ', _file);
                        var tsConfigFile = readConfig(_file);
                        scannedFiles = tsConfigFile.files;
                        if (scannedFiles) {
                            scannedFiles = handlePath(scannedFiles, cwd$1);
                        }
                        if (typeof scannedFiles === 'undefined') {
                            excludeFiles = tsConfigFile.exclude || [];
                            includeFiles = tsConfigFile.include || [];
                            scannedFiles = [];
                            var excludeParser_2 = new ParserUtil(), includeParser_2 = new ParserUtil();
                            excludeParser_2.init(excludeFiles, cwd$1);
                            includeParser_2.init(includeFiles, cwd$1);
                            var startCwd = sourceFolder;
                            var excludeParserTestFilesWithCwdDepth = excludeParser_2.testFilesWithCwdDepth();
                            if (!excludeParserTestFilesWithCwdDepth.status) {
                                startCwd = excludeParser_2.updateCwd(cwd$1, excludeParserTestFilesWithCwdDepth.level);
                            }
                            var includeParserTestFilesWithCwdDepth = includeParser_2.testFilesWithCwdDepth();
                            if (!includeParser_2.testFilesWithCwdDepth().status) {
                                startCwd = includeParser_2.updateCwd(cwd$1, includeParserTestFilesWithCwdDepth.level);
                            }
                            var finder = require('findit2')(path.resolve(startCwd));
                            finder.on('directory', function (dir, stat, stop) {
                                var base = path.basename(dir);
                                if (base === '.git' || base === 'node_modules') {
                                    stop();
                                }
                            });
                            finder.on('file', function (file, stat) {
                                if (/(spec|\.d)\.ts/.test(file)) {
                                    logger.warn('Ignoring', file);
                                }
                                else if (excludeParser_2.testFile(file)) {
                                    logger.warn('Excluding', file);
                                }
                                else if (includeFiles.length > 0) {
                                    /**
                                     * If include provided in tsconfig, use only this source,
                                     * and not files found with global findit scan in working directory
                                     */
                                    if (path.extname(file) === '.ts' &&
                                        includeParser_2.testFile(file)) {
                                        logger.debug('Including', file);
                                        scannedFiles.push(file);
                                    }
                                    else {
                                        if (path.extname(file) === '.ts') {
                                            logger.warn('Excluding', file);
                                        }
                                    }
                                }
                                else {
                                    logger.debug('Including', file);
                                    scannedFiles.push(file);
                                }
                            });
                            finder.on('end', function () {
                                _super.prototype.setFiles.call(_this, scannedFiles);
                                if (program.coverageTest || program.coverageTestPerFile) {
                                    logger.info('Run documentation coverage test');
                                    _super.prototype.testCoverage.call(_this);
                                }
                                else {
                                    _super.prototype.generate.call(_this);
                                }
                            });
                        }
                        else {
                            _super.prototype.setFiles.call(this, scannedFiles);
                            if (program.coverageTest || program.coverageTestPerFile) {
                                logger.info('Run documentation coverage test');
                                _super.prototype.testCoverage.call(this);
                            }
                            else {
                                _super.prototype.generate.call(this);
                            }
                        }
                    }
                }
            }
            else {
                logger.error('tsconfig.json file was not found, please use -p flag');
                outputHelp();
            }
        }
    };
    return CliApplication;
}(Application));

exports.CliApplication = CliApplication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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