(function() {
  var CSON, ContextMenuItemSet, ContextMenuManager, Disposable, MenuHelpers, _, calculateSpecificity, fs, path, platformContextMenu, ref, ref1, ref2, remote, sortMenuItems, validateSelector;

  path = require('path');

  CSON = require('season');

  fs = require('fs-plus');

  ref = require('clear-cut'), calculateSpecificity = ref.calculateSpecificity, validateSelector = ref.validateSelector;

  Disposable = require('event-kit').Disposable;

  remote = require('electron').remote;

  MenuHelpers = require('./menu-helpers');

  sortMenuItems = require('./menu-sort-helpers').sortMenuItems;

  _ = require('underscore-plus');

  platformContextMenu = (ref1 = require('../package.json')) != null ? (ref2 = ref1._atomMenu) != null ? ref2['context-menu'] : void 0 : void 0;

  module.exports = ContextMenuManager = (function() {
    function ContextMenuManager(arg) {
      this.keymapManager = arg.keymapManager;
      this.definitions = {
        '.overlayer': []
      };
      this.clear();
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
    }

    ContextMenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath, this.devMode = arg.devMode;
    };

    ContextMenuManager.prototype.loadPlatformItems = function() {
      var map, menusDirPath, platformMenuPath, ref3;
      if (platformContextMenu != null) {
        return this.add(platformContextMenu, (ref3 = this.devMode) != null ? ref3 : false);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        map = CSON.readFileSync(platformMenuPath);
        return this.add(map['context-menu']);
      }
    };

    ContextMenuManager.prototype.add = function(itemsBySelector, throwOnInvalidSelector) {
      var addedItemSets, itemSet, items, selector;
      if (throwOnInvalidSelector == null) {
        throwOnInvalidSelector = true;
      }
      addedItemSets = [];
      for (selector in itemsBySelector) {
        items = itemsBySelector[selector];
        if (throwOnInvalidSelector) {
          validateSelector(selector);
        }
        itemSet = new ContextMenuItemSet(selector, items);
        addedItemSets.push(itemSet);
        this.itemSets.push(itemSet);
      }
      return new Disposable((function(_this) {
        return function() {
          var i, len;
          for (i = 0, len = addedItemSets.length; i < len; i++) {
            itemSet = addedItemSets[i];
            _this.itemSets.splice(_this.itemSets.indexOf(itemSet), 1);
          }
        };
      })(this));
    };

    ContextMenuManager.prototype.templateForElement = function(target) {
      return this.templateForEvent({
        target: target
      });
    };

    ContextMenuManager.prototype.templateForEvent = function(event) {
      var currentTarget, currentTargetItems, i, item, itemForEvent, itemSet, j, k, len, len1, len2, matchingItemSets, ref3, template;
      template = [];
      currentTarget = event.target;
      while (currentTarget != null) {
        currentTargetItems = [];
        matchingItemSets = this.itemSets.filter(function(itemSet) {
          return currentTarget.webkitMatchesSelector(itemSet.selector);
        });
        for (i = 0, len = matchingItemSets.length; i < len; i++) {
          itemSet = matchingItemSets[i];
          ref3 = itemSet.items;
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            item = ref3[j];
            itemForEvent = this.cloneItemForEvent(item, event);
            if (itemForEvent) {
              MenuHelpers.merge(currentTargetItems, itemForEvent, itemSet.specificity);
            }
          }
        }
        for (k = 0, len2 = currentTargetItems.length; k < len2; k++) {
          item = currentTargetItems[k];
          MenuHelpers.merge(template, item, false);
        }
        currentTarget = currentTarget.parentElement;
      }
      this.pruneRedundantSeparators(template);
      this.addAccelerators(template);
      return this.sortTemplate(template);
    };

    ContextMenuManager.prototype.addAccelerators = function(template) {
      var id, item, keymaps, keystrokes, ref3, results;
      results = [];
      for (id in template) {
        item = template[id];
        if (item.command) {
          keymaps = this.keymapManager.findKeyBindings({
            command: item.command,
            target: document.activeElement
          });
          keystrokes = keymaps != null ? (ref3 = keymaps[0]) != null ? ref3.keystrokes : void 0 : void 0;
          if (keystrokes) {
            if (keystrokes.includes(' ')) {
              item.label += " [" + (_.humanizeKeystroke(keystrokes)) + "]";
            } else {
              item.accelerator = MenuHelpers.acceleratorForKeystroke(keystrokes);
            }
          }
        }
        if (Array.isArray(item.submenu)) {
          results.push(this.addAccelerators(item.submenu));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.pruneRedundantSeparators = function(menu) {
      var index, keepNextItemIfSeparator, results;
      keepNextItemIfSeparator = false;
      index = 0;
      results = [];
      while (index < menu.length) {
        if (menu[index].type === 'separator') {
          if (!keepNextItemIfSeparator || index === menu.length - 1) {
            results.push(menu.splice(index, 1));
          } else {
            results.push(index++);
          }
        } else {
          keepNextItemIfSeparator = true;
          results.push(index++);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.sortTemplate = function(template) {
      var id, item;
      template = sortMenuItems(template);
      for (id in template) {
        item = template[id];
        if (Array.isArray(item.submenu)) {
          item.submenu = this.sortTemplate(item.submenu);
        }
      }
      return template;
    };

    ContextMenuManager.prototype.cloneItemForEvent = function(item, event) {
      if (item.devMode && !this.devMode) {
        return null;
      }
      item = Object.create(item);
      if (typeof item.shouldDisplay === 'function') {
        if (!item.shouldDisplay(event)) {
          return null;
        }
      }
      if (typeof item.created === "function") {
        item.created(event);
      }
      if (Array.isArray(item.submenu)) {
        item.submenu = item.submenu.map((function(_this) {
          return function(submenuItem) {
            return _this.cloneItemForEvent(submenuItem, event);
          };
        })(this)).filter(function(submenuItem) {
          return submenuItem !== null;
        });
      }
      return item;
    };

    ContextMenuManager.prototype.showForEvent = function(event) {
      var menuTemplate;
      this.activeElement = event.target;
      menuTemplate = this.templateForEvent(event);
      if (!((menuTemplate != null ? menuTemplate.length : void 0) > 0)) {
        return;
      }
      remote.getCurrentWindow().emit('context-menu', menuTemplate);
    };

    ContextMenuManager.prototype.clear = function() {
      var inspectElement;
      this.activeElement = null;
      this.itemSets = [];
      inspectElement = {
        'atom-workspace': [
          {
            label: 'Inspect Element',
            command: 'application:inspect',
            devMode: true,
            created: function(event) {
              var pageX, pageY;
              pageX = event.pageX, pageY = event.pageY;
              return this.commandDetail = {
                x: pageX,
                y: pageY
              };
            }
          }
        ]
      };
      return this.add(inspectElement, false);
    };

    return ContextMenuManager;

  })();

  ContextMenuItemSet = (function() {
    function ContextMenuItemSet(selector1, items1) {
      this.selector = selector1;
      this.items = items1;
      this.specificity = calculateSpecificity(this.selector);
    }

    return ContextMenuItemSet;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
