(function() {
  var AddDialog, BufferedProcess, CompositeDisposable, CopyDialog, Directory, DirectoryView, Emitter, IgnoredNames, MoveDialog, RootDragAndDrop, TREE_VIEW_URI, TreeView, _, fs, getFullExtension, getStyleObject, nextId, path, ref, ref1, repoForPath, shell, toggleConfig,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  shell = require('electron').shell;

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  ref1 = require("./helpers"), repoForPath = ref1.repoForPath, getStyleObject = ref1.getStyleObject, getFullExtension = ref1.getFullExtension;

  fs = require('fs-plus');

  AddDialog = require('./add-dialog');

  MoveDialog = require('./move-dialog');

  CopyDialog = require('./copy-dialog');

  IgnoredNames = null;

  Directory = require('./directory');

  DirectoryView = require('./directory-view');

  RootDragAndDrop = require('./root-drag-and-drop');

  TREE_VIEW_URI = 'atom://tree-view';

  toggleConfig = function(keyPath) {
    return atom.config.set(keyPath, !atom.config.get(keyPath));
  };

  nextId = 1;

  module.exports = TreeView = (function() {
    function TreeView(state) {
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragEnter = bind(this.onDragEnter, this);
      this.onStylesheetsChanged = bind(this.onStylesheetsChanged, this);
      this.moveConflictingEntry = bind(this.moveConflictingEntry, this);
      var j, len, observer, ref2, ref3, selectedPath;
      this.id = nextId++;
      this.element = document.createElement('div');
      this.element.classList.add('tool-panel', 'tree-view');
      this.element.tabIndex = -1;
      this.list = document.createElement('ol');
      this.list.classList.add('tree-view-root', 'full-menu', 'list-tree', 'has-collapsable-children', 'focusable-panel');
      this.element.appendChild(this.list);
      this.disposables = new CompositeDisposable;
      this.emitter = new Emitter;
      this.roots = [];
      this.selectedPath = null;
      this.selectOnMouseUp = null;
      this.lastFocusedEntry = null;
      this.ignoredPatterns = [];
      this.useSyncFS = false;
      this.currentlyOpening = new Map;
      this.editorsToMove = [];
      this.editorsToDestroy = [];
      this.dragEventCounts = new WeakMap;
      this.rootDragAndDrop = new RootDragAndDrop(this);
      this.handleEvents();
      process.nextTick((function(_this) {
        return function() {
          var onStylesheetsChanged;
          _this.onStylesheetsChanged();
          onStylesheetsChanged = _.debounce(_this.onStylesheetsChanged, 100);
          _this.disposables.add(atom.styles.onDidAddStyleElement(onStylesheetsChanged));
          _this.disposables.add(atom.styles.onDidRemoveStyleElement(onStylesheetsChanged));
          return _this.disposables.add(atom.styles.onDidUpdateStyleElement(onStylesheetsChanged));
        };
      })(this));
      this.updateRoots(state.directoryExpansionStates);
      if (((ref2 = state.selectedPaths) != null ? ref2.length : void 0) > 0) {
        ref3 = state.selectedPaths;
        for (j = 0, len = ref3.length; j < len; j++) {
          selectedPath = ref3[j];
          this.selectMultipleEntries(this.entryForPath(selectedPath));
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      if ((state.scrollTop != null) || (state.scrollLeft != null)) {
        observer = new IntersectionObserver((function(_this) {
          return function() {
            if (_this.isVisible()) {
              _this.element.scrollTop = state.scrollTop;
              _this.element.scrollLeft = state.scrollLeft;
              return observer.disconnect();
            }
          };
        })(this));
        observer.observe(this.element);
      }
      if (state.width > 0) {
        this.element.style.width = state.width + "px";
      }
      this.disposables.add(this.onWillMoveEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, initialPath, k, l, len1, len2, newPath, results, results1;
          initialPath = arg.initialPath, newPath = arg.newPath;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(initialPath)) {
            initialPath += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if (filePath != null ? filePath.startsWith(initialPath) : void 0) {
                results.push(_this.editorsToMove.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === initialPath) {
                results1.push(_this.editorsToMove.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryMoved((function(_this) {
        return function(arg) {
          var editor, filePath, index, initialPath, k, len1, newPath, ref4, results;
          initialPath = arg.initialPath, newPath = arg.newPath;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            filePath = editor.getPath();
            index = _this.editorsToMove.indexOf(filePath);
            if (index !== -1) {
              editor.getBuffer().setPath(filePath.replace(initialPath, newPath));
              results.push(_this.editorsToMove.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onMoveEntryFailed((function(_this) {
        return function(arg) {
          var index, initialPath, newPath;
          initialPath = arg.initialPath, newPath = arg.newPath;
          index = _this.editorsToMove.indexOf(initialPath);
          if (index !== -1) {
            return _this.editorsToMove.splice(index, 1);
          }
        };
      })(this)));
      this.disposables.add(this.onWillDeleteEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, k, l, len1, len2, pathToDelete, results, results1;
          pathToDelete = arg.pathToDelete;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(pathToDelete)) {
            pathToDelete += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if ((filePath != null ? filePath.startsWith(pathToDelete) : void 0) && !editor.isModified()) {
                results.push(_this.editorsToDestroy.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === pathToDelete && !editor.isModified()) {
                results1.push(_this.editorsToDestroy.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryDeleted((function(_this) {
        return function(arg) {
          var editor, index, k, len1, pathToDelete, ref4, results;
          pathToDelete = arg.pathToDelete;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            index = _this.editorsToDestroy.indexOf(editor.getPath());
            if (index !== -1) {
              editor.destroy();
              results.push(_this.editorsToDestroy.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onDeleteEntryFailed((function(_this) {
        return function(arg) {
          var index, pathToDelete;
          pathToDelete = arg.pathToDelete;
          index = _this.editorsToDestroy.indexOf(pathToDelete);
          if (index !== -1) {
            return _this.editorsToDestroy.splice(index, 1);
          }
        };
      })(this)));
    }

    TreeView.prototype.serialize = function() {
      return {
        directoryExpansionStates: new (function(roots) {
          var j, len, root;
          for (j = 0, len = roots.length; j < len; j++) {
            root = roots[j];
            this[root.directory.path] = root.directory.serializeExpansionState();
          }
          return this;
        })(this.roots),
        deserializer: 'TreeView',
        selectedPaths: Array.from(this.getSelectedEntries(), function(entry) {
          return entry.getPath();
        }),
        scrollLeft: this.element.scrollLeft,
        scrollTop: this.element.scrollTop,
        width: parseInt(this.element.style.width || 0)
      };
    };

    TreeView.prototype.destroy = function() {
      var j, len, ref2, root;
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        root.directory.destroy();
      }
      this.disposables.dispose();
      this.rootDragAndDrop.dispose();
      return this.emitter.emit('did-destroy');
    };

    TreeView.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TreeView.prototype.getTitle = function() {
      return "Project";
    };

    TreeView.prototype.getURI = function() {
      return TREE_VIEW_URI;
    };

    TreeView.prototype.getPreferredLocation = function() {
      if (atom.config.get('tree-view.showOnRightSide')) {
        return 'right';
      } else {
        return 'left';
      }
    };

    TreeView.prototype.getAllowedLocations = function() {
      return ["left", "right"];
    };

    TreeView.prototype.isPermanentDockItem = function() {
      return true;
    };

    TreeView.prototype.getPreferredWidth = function() {
      var result;
      this.list.style.width = 'min-content';
      result = this.list.offsetWidth;
      this.list.style.width = '';
      return result;
    };

    TreeView.prototype.onDirectoryCreated = function(callback) {
      return this.emitter.on('directory-created', callback);
    };

    TreeView.prototype.onEntryCopied = function(callback) {
      return this.emitter.on('entry-copied', callback);
    };

    TreeView.prototype.onWillDeleteEntry = function(callback) {
      return this.emitter.on('will-delete-entry', callback);
    };

    TreeView.prototype.onEntryDeleted = function(callback) {
      return this.emitter.on('entry-deleted', callback);
    };

    TreeView.prototype.onDeleteEntryFailed = function(callback) {
      return this.emitter.on('delete-entry-failed', callback);
    };

    TreeView.prototype.onWillMoveEntry = function(callback) {
      return this.emitter.on('will-move-entry', callback);
    };

    TreeView.prototype.onEntryMoved = function(callback) {
      return this.emitter.on('entry-moved', callback);
    };

    TreeView.prototype.onMoveEntryFailed = function(callback) {
      return this.emitter.on('move-entry-failed', callback);
    };

    TreeView.prototype.onFileCreated = function(callback) {
      return this.emitter.on('file-created', callback);
    };

    TreeView.prototype.handleEvents = function() {
      this.element.addEventListener('click', (function(_this) {
        return function(e) {
          if (e.target.classList.contains('entries')) {
            return;
          }
          if (!(e.shiftKey || e.metaKey || e.ctrlKey)) {
            return _this.entryClicked(e);
          }
        };
      })(this));
      this.element.addEventListener('mousedown', (function(_this) {
        return function(e) {
          return _this.onMouseDown(e);
        };
      })(this));
      this.element.addEventListener('mouseup', (function(_this) {
        return function(e) {
          return _this.onMouseUp(e);
        };
      })(this));
      this.element.addEventListener('dragstart', (function(_this) {
        return function(e) {
          return _this.onDragStart(e);
        };
      })(this));
      this.element.addEventListener('dragenter', (function(_this) {
        return function(e) {
          return _this.onDragEnter(e);
        };
      })(this));
      this.element.addEventListener('dragleave', (function(_this) {
        return function(e) {
          return _this.onDragLeave(e);
        };
      })(this));
      this.element.addEventListener('dragover', (function(_this) {
        return function(e) {
          return _this.onDragOver(e);
        };
      })(this));
      this.element.addEventListener('drop', (function(_this) {
        return function(e) {
          return _this.onDrop(e);
        };
      })(this));
      atom.commands.add(this.element, {
        'core:move-up': (function(_this) {
          return function(e) {
            return _this.moveUp(e);
          };
        })(this),
        'core:move-down': (function(_this) {
          return function(e) {
            return _this.moveDown(e);
          };
        })(this),
        'core:page-up': (function(_this) {
          return function() {
            return _this.pageUp();
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            return _this.pageDown();
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            return _this.scrollToTop();
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            return _this.scrollToBottom();
          };
        })(this),
        'tree-view:expand-item': (function(_this) {
          return function() {
            return _this.openSelectedEntry({
              pending: true
            }, true);
          };
        })(this),
        'tree-view:recursive-expand-directory': (function(_this) {
          return function() {
            return _this.expandDirectory(true);
          };
        })(this),
        'tree-view:collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory();
          };
        })(this),
        'tree-view:recursive-collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory(true);
          };
        })(this),
        'tree-view:collapse-all': (function(_this) {
          return function() {
            return _this.collapseDirectory(true, true);
          };
        })(this),
        'tree-view:open-selected-entry': (function(_this) {
          return function() {
            return _this.openSelectedEntry();
          };
        })(this),
        'tree-view:open-selected-entry-right': (function(_this) {
          return function() {
            return _this.openSelectedEntryRight();
          };
        })(this),
        'tree-view:open-selected-entry-left': (function(_this) {
          return function() {
            return _this.openSelectedEntryLeft();
          };
        })(this),
        'tree-view:open-selected-entry-up': (function(_this) {
          return function() {
            return _this.openSelectedEntryUp();
          };
        })(this),
        'tree-view:open-selected-entry-down': (function(_this) {
          return function() {
            return _this.openSelectedEntryDown();
          };
        })(this),
        'tree-view:move': (function(_this) {
          return function() {
            return _this.moveSelectedEntry();
          };
        })(this),
        'tree-view:copy': (function(_this) {
          return function() {
            return _this.copySelectedEntries();
          };
        })(this),
        'tree-view:cut': (function(_this) {
          return function() {
            return _this.cutSelectedEntries();
          };
        })(this),
        'tree-view:paste': (function(_this) {
          return function() {
            return _this.pasteEntries();
          };
        })(this),
        'tree-view:copy-full-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(false);
          };
        })(this),
        'tree-view:show-in-file-manager': (function(_this) {
          return function() {
            return _this.showSelectedEntryInFileManager();
          };
        })(this),
        'tree-view:open-in-new-window': (function(_this) {
          return function() {
            return _this.openSelectedEntryInNewWindow();
          };
        })(this),
        'tree-view:copy-project-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(true);
          };
        })(this),
        'tree-view:unfocus': (function(_this) {
          return function() {
            return _this.unfocus();
          };
        })(this),
        'tree-view:toggle-vcs-ignored-files': function() {
          return toggleConfig('tree-view.hideVcsIgnoredFiles');
        },
        'tree-view:toggle-ignored-names': function() {
          return toggleConfig('tree-view.hideIgnoredNames');
        },
        'tree-view:remove-project-folder': (function(_this) {
          return function(e) {
            return _this.removeProjectFolder(e);
          };
        })(this)
      });
      [0, 1, 2, 3, 4, 5, 6, 7, 8].forEach((function(_this) {
        return function(index) {
          return atom.commands.add(_this.element, "tree-view:open-selected-entry-in-pane-" + (index + 1), function() {
            return _this.openSelectedEntryInPane(index);
          });
        };
      })(this));
      this.disposables.add(atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          _this.selectActiveFile();
          if (atom.config.get('tree-view.autoReveal')) {
            return _this.revealActiveFile({
              show: false,
              focus: false
            });
          }
        };
      })(this)));
      this.disposables.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideVcsIgnoredFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideIgnoredNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('core.ignoredNames', (function(_this) {
        return function() {
          if (atom.config.get('tree-view.hideIgnoredNames')) {
            return _this.updateRoots();
          }
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.sortFoldersBeforeFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      return this.disposables.add(atom.config.onDidChange('tree-view.squashDirectoryNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
    };

    TreeView.prototype.toggle = function() {
      return atom.workspace.toggle(this);
    };

    TreeView.prototype.show = function(focus) {
      return atom.workspace.open(this, {
        searchAllPanes: true,
        activatePane: false,
        activateItem: false
      }).then((function(_this) {
        return function() {
          atom.workspace.paneContainerForURI(_this.getURI()).show();
          if (focus) {
            return _this.focus();
          }
        };
      })(this));
    };

    TreeView.prototype.hide = function() {
      return atom.workspace.hide(this);
    };

    TreeView.prototype.focus = function() {
      return this.element.focus();
    };

    TreeView.prototype.unfocus = function() {
      return atom.workspace.getCenter().activate();
    };

    TreeView.prototype.hasFocus = function() {
      return document.activeElement === this.element;
    };

    TreeView.prototype.toggleFocus = function() {
      if (this.hasFocus()) {
        return this.unfocus();
      } else {
        return this.show(true);
      }
    };

    TreeView.prototype.entryClicked = function(e) {
      var entry, isRecursive;
      if (entry = e.target.closest('.entry')) {
        isRecursive = e.altKey || false;
        this.selectEntry(entry);
        if (entry.classList.contains('directory')) {
          return entry.toggleExpansion(isRecursive);
        } else if (entry.classList.contains('file')) {
          return this.fileViewEntryClicked(e);
        }
      }
    };

    TreeView.prototype.fileViewEntryClicked = function(e) {
      var alwaysOpenExisting, detail, filePath, openPromise, ref2;
      filePath = e.target.closest('.entry').getPath();
      detail = (ref2 = e.detail) != null ? ref2 : 1;
      alwaysOpenExisting = atom.config.get('tree-view.alwaysOpenExisting');
      if (detail === 1) {
        if (atom.config.get('core.allowPendingPaneItems')) {
          openPromise = atom.workspace.open(filePath, {
            pending: true,
            activatePane: false,
            searchAllPanes: alwaysOpenExisting
          });
          this.currentlyOpening.set(filePath, openPromise);
          return openPromise.then((function(_this) {
            return function() {
              return _this.currentlyOpening["delete"](filePath);
            };
          })(this));
        }
      } else if (detail === 2) {
        return this.openAfterPromise(filePath, {
          searchAllPanes: alwaysOpenExisting
        });
      }
    };

    TreeView.prototype.openAfterPromise = function(uri, options) {
      var promise;
      if (promise = this.currentlyOpening.get(uri)) {
        return promise.then(function() {
          return atom.workspace.open(uri, options);
        });
      } else {
        return atom.workspace.open(uri, options);
      }
    };

    TreeView.prototype.updateRoots = function(expansionStates) {
      var directory, j, k, key, len, len1, oldExpansionStates, projectPath, ref2, results, root, selectedPath, selectedPaths, stats;
      if (expansionStates == null) {
        expansionStates = {};
      }
      selectedPaths = this.selectedPaths();
      oldExpansionStates = {};
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        oldExpansionStates[root.directory.path] = root.directory.serializeExpansionState();
        root.directory.destroy();
        root.remove();
      }
      if (IgnoredNames == null) {
        IgnoredNames = require('./ignored-names');
      }
      this.roots = (function() {
        var k, l, len1, len2, ref3, ref4, ref5, ref6, results;
        ref3 = atom.project.getPaths();
        results = [];
        for (k = 0, len1 = ref3.length; k < len1; k++) {
          projectPath = ref3[k];
          stats = fs.lstatSyncNoException(projectPath);
          if (!stats) {
            continue;
          }
          stats = _.pick.apply(_, [stats].concat(slice.call(_.keys(stats))));
          ref4 = ["atime", "birthtime", "ctime", "mtime"];
          for (l = 0, len2 = ref4.length; l < len2; l++) {
            key = ref4[l];
            stats[key] = stats[key].getTime();
          }
          directory = new Directory({
            name: path.basename(projectPath),
            fullPath: projectPath,
            symlink: false,
            isRoot: true,
            expansionState: (ref5 = (ref6 = expansionStates[projectPath]) != null ? ref6 : oldExpansionStates[projectPath]) != null ? ref5 : {
              isExpanded: true
            },
            ignoredNames: new IgnoredNames(),
            useSyncFS: this.useSyncFS,
            stats: stats
          });
          root = new DirectoryView(directory).element;
          this.list.appendChild(root);
          results.push(root);
        }
        return results;
      }).call(this);
      results = [];
      for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
        selectedPath = selectedPaths[k];
        results.push(this.selectMultipleEntries(this.entryForPath(selectedPath)));
      }
      return results;
    };

    TreeView.prototype.getActivePath = function() {
      var ref2;
      return (ref2 = atom.workspace.getCenter().getActivePaneItem()) != null ? typeof ref2.getPath === "function" ? ref2.getPath() : void 0 : void 0;
    };

    TreeView.prototype.selectActiveFile = function() {
      var activeFilePath;
      activeFilePath = this.getActivePath();
      if (this.entryForPath(activeFilePath)) {
        return this.selectEntryForPath(activeFilePath);
      } else {
        return this.deselect();
      }
    };

    TreeView.prototype.revealActiveFile = function(options) {
      var focus, promise, show;
      if (options == null) {
        options = {};
      }
      if (!atom.project.getPaths().length) {
        return Promise.resolve();
      }
      show = options.show, focus = options.focus;
      if (focus == null) {
        focus = atom.config.get('tree-view.focusOnReveal');
      }
      promise = show || focus ? this.show(focus) : Promise.resolve();
      return promise.then((function(_this) {
        return function() {
          var activeFilePath, activePathComponents, currentPath, entry, j, len, pathComponent, ref2, relativePath, results, rootPath;
          if (!(activeFilePath = _this.getActivePath())) {
            return;
          }
          ref2 = atom.project.relativizePath(activeFilePath), rootPath = ref2[0], relativePath = ref2[1];
          if (rootPath == null) {
            return;
          }
          activePathComponents = relativePath.split(path.sep);
          activePathComponents.unshift(rootPath.substr(rootPath.lastIndexOf(path.sep) + 1));
          currentPath = rootPath.substr(0, rootPath.lastIndexOf(path.sep));
          results = [];
          for (j = 0, len = activePathComponents.length; j < len; j++) {
            pathComponent = activePathComponents[j];
            currentPath += path.sep + pathComponent;
            entry = _this.entryForPath(currentPath);
            if (entry.classList.contains('directory')) {
              results.push(entry.expand());
            } else {
              _this.selectEntry(entry);
              results.push(_this.scrollToEntry(entry));
            }
          }
          return results;
        };
      })(this));
    };

    TreeView.prototype.copySelectedEntryPath = function(relativePath) {
      var pathToCopy;
      if (relativePath == null) {
        relativePath = false;
      }
      if (pathToCopy = this.selectedPath) {
        if (relativePath) {
          pathToCopy = atom.project.relativize(pathToCopy);
        }
        return atom.clipboard.write(pathToCopy);
      }
    };

    TreeView.prototype.entryForPath = function(entryPath) {
      var bestMatchEntry, bestMatchLength, entry, entryLength, j, len, ref2, ref3;
      bestMatchEntry = null;
      bestMatchLength = 0;
      ref2 = this.list.querySelectorAll('.entry');
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        if (entry.isPathEqual(entryPath)) {
          return entry;
        }
        entryLength = entry.getPath().length;
        if (((ref3 = entry.directory) != null ? ref3.contains(entryPath) : void 0) && entryLength > bestMatchLength) {
          bestMatchEntry = entry;
          bestMatchLength = entryLength;
        }
      }
      return bestMatchEntry;
    };

    TreeView.prototype.selectEntryForPath = function(entryPath) {
      return this.selectEntry(this.entryForPath(entryPath));
    };

    TreeView.prototype.moveDown = function(event) {
      var nextEntry, selectedEntry;
      if (event != null) {
        event.stopImmediatePropagation();
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (selectedEntry.classList.contains('directory')) {
          if (this.selectEntry(selectedEntry.entries.children[0])) {
            this.scrollToEntry(this.selectedEntry(), false);
            return;
          }
        }
        if (nextEntry = this.nextEntry(selectedEntry)) {
          this.selectEntry(nextEntry);
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.moveUp = function(event) {
      var entries, previousEntry, selectedEntry;
      event.stopImmediatePropagation();
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (previousEntry = this.previousEntry(selectedEntry)) {
          this.selectEntry(previousEntry);
        } else {
          this.selectEntry(selectedEntry.parentElement.closest('.directory'));
        }
      } else {
        entries = this.list.querySelectorAll('.entry');
        this.selectEntry(entries[entries.length - 1]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.nextEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        if (currentEntry.nextSibling != null) {
          currentEntry = currentEntry.nextSibling;
          if (currentEntry.matches('.entry')) {
            return currentEntry;
          }
        } else {
          currentEntry = currentEntry.parentElement.closest('.directory');
        }
      }
      return null;
    };

    TreeView.prototype.previousEntry = function(entry) {
      var entries, previousEntry;
      previousEntry = entry.previousSibling;
      while ((previousEntry != null) && !previousEntry.matches('.entry')) {
        previousEntry = previousEntry.previousSibling;
      }
      if (previousEntry == null) {
        return null;
      }
      if (previousEntry.matches('.directory.expanded')) {
        entries = previousEntry.querySelectorAll('.entry');
        if (entries.length > 0) {
          return entries[entries.length - 1];
        }
      }
      return previousEntry;
    };

    TreeView.prototype.expandDirectory = function(isRecursive) {
      var directory, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      directory = selectedEntry.closest('.directory');
      if (isRecursive === false && directory.isExpanded) {
        if (directory.directory.getEntries().length > 0) {
          return this.moveDown();
        }
      } else {
        return directory.expand(isRecursive);
      }
    };

    TreeView.prototype.collapseDirectory = function(isRecursive, allDirectories) {
      var directory, j, len, ref2, root, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (allDirectories == null) {
        allDirectories = false;
      }
      if (allDirectories) {
        ref2 = this.roots;
        for (j = 0, len = ref2.length; j < len; j++) {
          root = ref2[j];
          root.collapse(true);
        }
        return;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (directory = selectedEntry.closest('.expanded.directory')) {
        directory.collapse(isRecursive);
        return this.selectEntry(directory);
      }
    };

    TreeView.prototype.openSelectedEntry = function(options, expandDirectory) {
      var selectedEntry;
      if (options == null) {
        options = {};
      }
      if (expandDirectory == null) {
        expandDirectory = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (selectedEntry.classList.contains('directory')) {
        if (expandDirectory) {
          return this.expandDirectory(false);
        } else {
          return selectedEntry.toggleExpansion();
        }
      } else if (selectedEntry.classList.contains('file')) {
        if (atom.config.get('tree-view.alwaysOpenExisting')) {
          options = Object.assign({
            searchAllPanes: true
          }, options);
        }
        return this.openAfterPromise(selectedEntry.getPath(), options);
      }
    };

    TreeView.prototype.openSelectedEntrySplit = function(orientation, side) {
      var pane, selectedEntry, split;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getActivePane();
      if (pane && selectedEntry.classList.contains('file')) {
        if (atom.workspace.getCenter().getActivePaneItem()) {
          split = pane.split(orientation, side);
          return atom.workspace.openURIInPane(selectedEntry.getPath(), split);
        } else {
          return this.openSelectedEntry(true);
        }
      }
    };

    TreeView.prototype.openSelectedEntryRight = function() {
      return this.openSelectedEntrySplit('horizontal', 'after');
    };

    TreeView.prototype.openSelectedEntryLeft = function() {
      return this.openSelectedEntrySplit('horizontal', 'before');
    };

    TreeView.prototype.openSelectedEntryUp = function() {
      return this.openSelectedEntrySplit('vertical', 'before');
    };

    TreeView.prototype.openSelectedEntryDown = function() {
      return this.openSelectedEntrySplit('vertical', 'after');
    };

    TreeView.prototype.openSelectedEntryInPane = function(index) {
      var pane, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getPanes()[index];
      if (pane && selectedEntry.classList.contains('file')) {
        return atom.workspace.openURIInPane(selectedEntry.getPath(), pane);
      }
    };

    TreeView.prototype.moveSelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if ((entry == null) || indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry.getPath();
      } else {
        oldPath = this.getActivePath();
      }
      if (oldPath) {
        dialog = new MoveDialog(oldPath, {
          willMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('will-move-entry', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMoveFailed: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('move-entry-failed', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this)
        });
        return dialog.attach();
      }
    };

    TreeView.prototype.showSelectedEntryInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in " + (this.getFileManagerName()));
      }
    };

    TreeView.prototype.showCurrentFileInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = atom.workspace.getCenter().getActiveTextEditor()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in " + (this.getFileManagerName()));
      }
    };

    TreeView.prototype.getFileManagerName = function() {
      switch (process.platform) {
        case 'darwin':
          return 'Finder';
        case 'win32':
          return 'Explorer';
        default:
          return 'File Manager';
      }
    };

    TreeView.prototype.openSelectedEntryInNewWindow = function() {
      var pathToOpen, ref2;
      if (pathToOpen = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0) {
        return atom.open({
          pathsToOpen: [pathToOpen],
          newWindow: true
        });
      }
    };

    TreeView.prototype.copySelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if (indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry != null ? entry.getPath() : void 0;
      } else {
        oldPath = this.getActivePath();
      }
      if (!oldPath) {
        return;
      }
      dialog = new CopyDialog(oldPath, {
        onCopy: (function(_this) {
          return function(arg) {
            var initialPath, newPath;
            initialPath = arg.initialPath, newPath = arg.newPath;
            return _this.emitter.emit('entry-copied', {
              initialPath: initialPath,
              newPath: newPath
            });
          };
        })(this)
      });
      return dialog.attach();
    };

    TreeView.prototype.removeSelectedEntries = function() {
      var activePath, j, len, ref2, ref3, root, selectedEntries, selectedPaths;
      if (this.hasFocus()) {
        selectedPaths = this.selectedPaths();
        selectedEntries = this.getSelectedEntries();
      } else if (activePath = this.getActivePath()) {
        selectedPaths = [activePath];
        selectedEntries = [this.entryForPath(activePath)];
      }
      if (!((selectedPaths != null ? selectedPaths.length : void 0) > 0)) {
        return;
      }
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        if (ref3 = root.getPath(), indexOf.call(selectedPaths, ref3) >= 0) {
          atom.confirm({
            message: "The root directory '" + root.directory.name + "' can't be removed.",
            buttons: ['OK']
          }, function() {});
          return;
        }
      }
      return atom.confirm({
        message: "Are you sure you want to delete the selected " + (selectedPaths.length > 1 ? 'items' : 'item') + "?",
        detailedMessage: "You are deleting:\n" + (selectedPaths.join('\n')),
        buttons: ['Move to Trash', 'Cancel']
      }, (function(_this) {
        return function(response) {
          var failedDeletions, firstSelectedEntry, k, len1, repo, selectedPath;
          if (response === 0) {
            failedDeletions = [];
            for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
              selectedPath = selectedPaths[k];
              if (!fs.existsSync(selectedPath)) {
                continue;
              }
              _this.emitter.emit('will-delete-entry', {
                pathToDelete: selectedPath
              });
              if (shell.moveItemToTrash(selectedPath)) {
                _this.emitter.emit('entry-deleted', {
                  pathToDelete: selectedPath
                });
              } else {
                _this.emitter.emit('delete-entry-failed', {
                  pathToDelete: selectedPath
                });
                failedDeletions.push(selectedPath);
              }
              if (repo = repoForPath(selectedPath)) {
                repo.getPathStatus(selectedPath);
              }
            }
            if (failedDeletions.length > 0) {
              atom.notifications.addError(_this.formatTrashFailureMessage(failedDeletions), {
                description: _this.formatTrashEnabledMessage(),
                detail: "" + (failedDeletions.join('\n')),
                dismissable: true
              });
            }
            if (firstSelectedEntry = selectedEntries[0]) {
              _this.selectEntry(firstSelectedEntry.closest('.directory:not(.selected)'));
            }
            if (atom.config.get('tree-view.squashDirectoryNames')) {
              return _this.updateRoots();
            }
          }
        };
      })(this));
    };

    TreeView.prototype.formatTrashFailureMessage = function(failedDeletions) {
      var fileText;
      fileText = failedDeletions.length > 1 ? 'files' : 'file';
      return "The following " + fileText + " couldn't be moved to the trash.";
    };

    TreeView.prototype.formatTrashEnabledMessage = function() {
      switch (process.platform) {
        case 'linux':
          return 'Is `gvfs-trash` installed?';
        case 'darwin':
          return 'Is Trash enabled on the volume where the files are stored?';
        case 'win32':
          return 'Is there a Recycle Bin on the drive where the files are stored?';
      }
    };

    TreeView.prototype.copySelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:cutPath');
      return window.localStorage['tree-view:copyPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.cutSelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:copyPath');
      return window.localStorage['tree-view:cutPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.pasteEntries = function() {
      var copiedPaths, cutPaths, initialPath, initialPaths, j, len, newDirectoryPath, results, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (!selectedEntry) {
        return;
      }
      cutPaths = window.localStorage['tree-view:cutPath'] ? JSON.parse(window.localStorage['tree-view:cutPath']) : null;
      copiedPaths = window.localStorage['tree-view:copyPath'] ? JSON.parse(window.localStorage['tree-view:copyPath']) : null;
      initialPaths = copiedPaths || cutPaths;
      if (!(initialPaths != null ? initialPaths.length : void 0)) {
        return;
      }
      newDirectoryPath = selectedEntry.getPath();
      if (selectedEntry.classList.contains('file')) {
        newDirectoryPath = path.dirname(newDirectoryPath);
      }
      results = [];
      for (j = 0, len = initialPaths.length; j < len; j++) {
        initialPath = initialPaths[j];
        if (fs.existsSync(initialPath)) {
          if (copiedPaths) {
            results.push(this.copyEntry(initialPath, newDirectoryPath));
          } else if (cutPaths) {
            if (!this.moveEntry(initialPath, newDirectoryPath)) {
              break;
            } else {
              results.push(void 0);
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TreeView.prototype.add = function(isCreatingFile) {
      var dialog, ref2, ref3, selectedEntry, selectedPath;
      selectedEntry = (ref2 = this.selectedEntry()) != null ? ref2 : this.roots[0];
      selectedPath = (ref3 = selectedEntry != null ? selectedEntry.getPath() : void 0) != null ? ref3 : '';
      dialog = new AddDialog(selectedPath, isCreatingFile);
      dialog.onDidCreateDirectory((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          _this.selectEntryForPath(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('directory-created', {
            path: createdPath
          });
        };
      })(this));
      dialog.onDidCreateFile((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          atom.workspace.open(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('file-created', {
            path: createdPath
          });
        };
      })(this));
      return dialog.attach();
    };

    TreeView.prototype.removeProjectFolder = function(e) {
      var pathToRemove, ref2, ref3, ref4, ref5, ref6, ref7, ref8, ref9;
      pathToRemove = (ref2 = e.target.closest(".project-root > .header")) != null ? (ref3 = ref2.querySelector(".name")) != null ? ref3.dataset.path : void 0 : void 0;
      if (pathToRemove == null) {
        pathToRemove = (ref4 = this.selectedEntry()) != null ? (ref5 = ref4.closest(".project-root")) != null ? (ref6 = ref5.querySelector(".header")) != null ? (ref7 = ref6.querySelector(".name")) != null ? ref7.dataset.path : void 0 : void 0 : void 0 : void 0;
      }
      if (this.roots.length === 1) {
        if (pathToRemove == null) {
          pathToRemove = (ref8 = this.roots[0].querySelector(".header")) != null ? (ref9 = ref8.querySelector(".name")) != null ? ref9.dataset.path : void 0 : void 0;
        }
      }
      if (pathToRemove != null) {
        return atom.project.removePath(pathToRemove);
      }
    };

    TreeView.prototype.selectedEntry = function() {
      return this.list.querySelector('.selected');
    };

    TreeView.prototype.selectEntry = function(entry) {
      var selectedEntries;
      if (entry == null) {
        return;
      }
      this.selectedPath = entry.getPath();
      this.lastFocusedEntry = entry;
      selectedEntries = this.getSelectedEntries();
      if (selectedEntries.length > 1 || selectedEntries[0] !== entry) {
        this.deselect(selectedEntries);
        entry.classList.add('selected');
      }
      return entry;
    };

    TreeView.prototype.getSelectedEntries = function() {
      return this.list.querySelectorAll('.selected');
    };

    TreeView.prototype.deselect = function(elementsToDeselect) {
      var j, len, selected;
      if (elementsToDeselect == null) {
        elementsToDeselect = this.getSelectedEntries();
      }
      for (j = 0, len = elementsToDeselect.length; j < len; j++) {
        selected = elementsToDeselect[j];
        selected.classList.remove('selected');
      }
      return void 0;
    };

    TreeView.prototype.scrollTop = function(top) {
      if (top != null) {
        return this.element.scrollTop = top;
      } else {
        return this.element.scrollTop;
      }
    };

    TreeView.prototype.scrollBottom = function(bottom) {
      if (bottom != null) {
        return this.element.scrollTop = bottom - this.element.offsetHeight;
      } else {
        return this.element.scrollTop + this.element.offsetHeight;
      }
    };

    TreeView.prototype.scrollToEntry = function(entry, center) {
      var element;
      if (center == null) {
        center = true;
      }
      element = (entry != null ? entry.classList.contains('directory') : void 0) ? entry.header : entry;
      return element != null ? element.scrollIntoViewIfNeeded(center) : void 0;
    };

    TreeView.prototype.scrollToBottom = function() {
      var lastEntry;
      if (lastEntry = _.last(this.list.querySelectorAll('.entry'))) {
        this.selectEntry(lastEntry);
        return this.scrollToEntry(lastEntry);
      }
    };

    TreeView.prototype.scrollToTop = function() {
      if (this.roots[0] != null) {
        this.selectEntry(this.roots[0]);
      }
      return this.element.scrollTop = 0;
    };

    TreeView.prototype.pageUp = function() {
      return this.element.scrollTop -= this.element.offsetHeight;
    };

    TreeView.prototype.pageDown = function() {
      return this.element.scrollTop += this.element.offsetHeight;
    };

    TreeView.prototype.copyEntry = function(initialPath, newDirectoryPath) {
      var error, extension, fileCounter, filePath, initialPathIsDirectory, newPath, originalNewPath, realInitialPath, realNewDirectoryPath, repo;
      initialPathIsDirectory = fs.isDirectorySync(initialPath);
      realNewDirectoryPath = fs.realpathSync(newDirectoryPath) + path.sep;
      realInitialPath = fs.realpathSync(initialPath) + path.sep;
      if (initialPathIsDirectory && realNewDirectoryPath.startsWith(realInitialPath)) {
        if (!fs.isSymbolicLinkSync(initialPath)) {
          atom.notifications.addWarning('Cannot copy a folder into itself');
          return;
        }
      }
      newPath = path.join(newDirectoryPath, path.basename(initialPath));
      fileCounter = 0;
      originalNewPath = newPath;
      while (fs.existsSync(newPath)) {
        if (initialPathIsDirectory) {
          newPath = "" + originalNewPath + fileCounter;
        } else {
          extension = getFullExtension(originalNewPath);
          filePath = path.join(path.dirname(originalNewPath), path.basename(originalNewPath, extension));
          newPath = "" + filePath + fileCounter + extension;
        }
        fileCounter += 1;
      }
      try {
        this.emitter.emit('will-copy-entry', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (initialPathIsDirectory) {
          fs.copySync(initialPath, newPath);
        } else {
          fs.writeFileSync(newPath, fs.readFileSync(initialPath));
        }
        this.emitter.emit('entry-copied', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          return repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        this.emitter.emit('copy-entry-failed', {
          initialPath: initialPath,
          newPath: newPath
        });
        return atom.notifications.addWarning("Failed to copy entry " + initialPath + " to " + newDirectoryPath, {
          detail: error.message
        });
      }
    };

    TreeView.prototype.moveEntry = function(initialPath, newDirectoryPath) {
      var error, newPath, realInitialPath, realNewDirectoryPath, repo;
      realNewDirectoryPath = fs.realpathSync(newDirectoryPath) + path.sep;
      realInitialPath = fs.realpathSync(initialPath) + path.sep;
      if (fs.isDirectorySync(initialPath) && realNewDirectoryPath.startsWith(realInitialPath)) {
        if (!fs.isSymbolicLinkSync(initialPath)) {
          atom.notifications.addWarning('Cannot move a folder into itself');
          return;
        }
      }
      newPath = path.join(newDirectoryPath, path.basename(initialPath));
      try {
        this.emitter.emit('will-move-entry', {
          initialPath: initialPath,
          newPath: newPath
        });
        fs.moveSync(initialPath, newPath);
        this.emitter.emit('entry-moved', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        if (error.code === 'EEXIST') {
          return this.moveConflictingEntry(initialPath, newPath, newDirectoryPath);
        } else {
          this.emitter.emit('move-entry-failed', {
            initialPath: initialPath,
            newPath: newPath
          });
          atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newDirectoryPath, {
            detail: error.message
          });
        }
      }
      return true;
    };

    TreeView.prototype.moveConflictingEntry = function(initialPath, newPath, newDirectoryPath) {
      var chosen, entries, entry, error, j, len, repo;
      try {
        if (!fs.isDirectorySync(initialPath)) {
          chosen = atom.confirm({
            message: "'" + (path.relative(newDirectoryPath, newPath)) + "' already exists",
            detailedMessage: 'Do you want to replace it?',
            buttons: ['Replace file', 'Skip', 'Cancel']
          });
          switch (chosen) {
            case 0:
              fs.renameSync(initialPath, newPath);
              this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
              if (repo = repoForPath(newPath)) {
                repo.getPathStatus(initialPath);
                repo.getPathStatus(newPath);
              }
              break;
            case 2:
              return false;
          }
        } else {
          entries = fs.readdirSync(initialPath);
          for (j = 0, len = entries.length; j < len; j++) {
            entry = entries[j];
            if (fs.existsSync(path.join(newPath, entry))) {
              if (!this.moveConflictingEntry(path.join(initialPath, entry), path.join(newPath, entry), newDirectoryPath)) {
                return false;
              }
            } else {
              this.moveEntry(path.join(initialPath, entry), newPath);
            }
          }
          if (!fs.readdirSync(initialPath).length) {
            fs.rmdirSync(initialPath);
          }
        }
      } catch (error1) {
        error = error1;
        this.emitter.emit('move-entry-failed', {
          initialPath: initialPath,
          newPath: newPath
        });
        atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newPath, {
          detail: error.message
        });
      }
      return true;
    };

    TreeView.prototype.onStylesheetsChanged = function() {
      if (!this.isVisible()) {
        return;
      }
      this.element.style.display = 'none';
      this.element.offsetWidth;
      return this.element.style.display = '';
    };

    TreeView.prototype.onMouseDown = function(e) {
      var cmdKey, entryToSelect, shiftKey;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      cmdKey = e.metaKey || (e.ctrlKey && process.platform !== 'darwin');
      if (entryToSelect.classList.contains('selected')) {
        if (e.button === 2 || (e.ctrlKey && process.platform === 'darwin')) {
          return;
        } else {
          shiftKey = e.shiftKey;
          this.selectOnMouseUp = {
            shiftKey: shiftKey,
            cmdKey: cmdKey
          };
          return;
        }
      }
      if (e.shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (e.shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.selectMultipleEntries(entryToSelect);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.onMouseUp = function(e) {
      var cmdKey, entryToSelect, ref2, shiftKey;
      if (this.selectOnMouseUp == null) {
        return;
      }
      ref2 = this.selectOnMouseUp, shiftKey = ref2.shiftKey, cmdKey = ref2.cmdKey;
      this.selectOnMouseUp = null;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      if (shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.deselect([entryToSelect]);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.selectedPaths = function() {
      var entry, j, len, ref2, results;
      ref2 = this.getSelectedEntries();
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        results.push(entry.getPath());
      }
      return results;
    };

    TreeView.prototype.selectContinuousEntries = function(entry, deselectOthers) {
      var currentSelectedEntry, element, elements, entries, entryIndex, i, j, len, parentContainer, ref2, selectedIndex;
      if (deselectOthers == null) {
        deselectOthers = true;
      }
      currentSelectedEntry = (ref2 = this.lastFocusedEntry) != null ? ref2 : this.selectedEntry();
      parentContainer = entry.parentElement;
      elements = [];
      if (parentContainer === currentSelectedEntry.parentElement) {
        entries = Array.from(parentContainer.querySelectorAll('.entry'));
        entryIndex = entries.indexOf(entry);
        selectedIndex = entries.indexOf(currentSelectedEntry);
        elements = (function() {
          var j, ref3, ref4, results;
          results = [];
          for (i = j = ref3 = entryIndex, ref4 = selectedIndex; ref3 <= ref4 ? j <= ref4 : j >= ref4; i = ref3 <= ref4 ? ++j : --j) {
            results.push(entries[i]);
          }
          return results;
        })();
        if (deselectOthers) {
          this.deselect();
        }
        for (j = 0, len = elements.length; j < len; j++) {
          element = elements[j];
          element.classList.add('selected');
        }
      }
      return elements;
    };

    TreeView.prototype.selectMultipleEntries = function(entry) {
      if (entry != null) {
        entry.classList.toggle('selected');
      }
      return entry;
    };

    TreeView.prototype.showFullMenu = function() {
      this.list.classList.remove('multi-select');
      return this.list.classList.add('full-menu');
    };

    TreeView.prototype.showMultiSelectMenu = function() {
      this.list.classList.remove('full-menu');
      return this.list.classList.add('multi-select');
    };

    TreeView.prototype.showMultiSelectMenuIfNecessary = function() {
      if (this.getSelectedEntries().length > 1) {
        return this.showMultiSelectMenu();
      } else {
        return this.showFullMenu();
      }
    };

    TreeView.prototype.multiSelectEnabled = function() {
      return this.list.classList.contains('multi-select');
    };

    TreeView.prototype.onDragEnter = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        if (!this.isAtomTreeViewEvent(e)) {
          return;
        }
        e.stopPropagation();
        if (!this.dragEventCounts.get(entry)) {
          this.dragEventCounts.set(entry, 0);
        }
        if (!(this.dragEventCounts.get(entry) !== 0 || entry.classList.contains('selected'))) {
          entry.classList.add('drag-over', 'selected');
        }
        return this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) + 1);
      }
    };

    TreeView.prototype.onDragLeave = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        if (!this.isAtomTreeViewEvent(e)) {
          return;
        }
        e.stopPropagation();
        this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) - 1);
        if (this.dragEventCounts.get(entry) === 0 && entry.classList.contains('drag-over')) {
          return entry.classList.remove('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDragStart = function(e) {
      var dragImage, entry, entryPath, initialPaths, j, key, len, newElement, parentSelected, ref2, ref3, target, value;
      this.dragEventCounts = new WeakMap;
      this.selectOnMouseUp = null;
      if (entry = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.rootDragAndDrop.canDragStart(e)) {
          return this.rootDragAndDrop.onDragStart(e);
        }
        dragImage = document.createElement("ol");
        dragImage.classList.add("entries", "list-tree");
        dragImage.style.position = "absolute";
        dragImage.style.top = 0;
        dragImage.style.left = 0;
        dragImage.style.willChange = "transform";
        initialPaths = [];
        ref2 = this.getSelectedEntries();
        for (j = 0, len = ref2.length; j < len; j++) {
          target = ref2[j];
          entryPath = target.querySelector(".name").dataset.path;
          parentSelected = target.parentNode.closest(".entry.selected");
          if (!parentSelected) {
            initialPaths.push(entryPath);
            newElement = target.cloneNode(true);
            if (newElement.classList.contains("directory")) {
              newElement.querySelector(".entries").remove();
            }
            ref3 = getStyleObject(target);
            for (key in ref3) {
              value = ref3[key];
              newElement.style[key] = value;
            }
            newElement.style.paddingLeft = "1em";
            newElement.style.paddingRight = "1em";
            dragImage.append(newElement);
          }
        }
        document.body.appendChild(dragImage);
        e.dataTransfer.effectAllowed = "move";
        e.dataTransfer.setDragImage(dragImage, 0, 0);
        e.dataTransfer.setData("initialPaths", initialPaths);
        e.dataTransfer.setData("atom-tree-view-event", "true");
        return window.requestAnimationFrame(function() {
          return dragImage.remove();
        });
      }
    };

    TreeView.prototype.onDragOver = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        if (!this.isAtomTreeViewEvent(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.dragEventCounts.get(entry) > 0 && !entry.classList.contains('selected')) {
          return entry.classList.add('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDrop = function(e) {
      var entry, file, initialPath, initialPaths, j, k, l, len, len1, newDirectoryPath, ref2, ref3, ref4, ref5, results, results1, results2;
      this.dragEventCounts = new WeakMap;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        if (!this.isAtomTreeViewEvent(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        newDirectoryPath = (ref2 = entry.querySelector('.name')) != null ? ref2.dataset.path : void 0;
        if (!newDirectoryPath) {
          return false;
        }
        initialPaths = e.dataTransfer.getData('initialPaths');
        if (initialPaths) {
          initialPaths = initialPaths.split(',');
          if (initialPaths.includes(newDirectoryPath)) {
            return;
          }
          entry.classList.remove('drag-over', 'selected');
          results = [];
          for (j = initialPaths.length - 1; j >= 0; j += -1) {
            initialPath = initialPaths[j];
            if ((ref3 = this.entryForPath(initialPath)) != null) {
              if (typeof ref3.collapse === "function") {
                ref3.collapse();
              }
            }
            if ((process.platform === 'darwin' && e.metaKey) || e.ctrlKey) {
              results.push(this.copyEntry(initialPath, newDirectoryPath));
            } else {
              if (!this.moveEntry(initialPath, newDirectoryPath)) {
                break;
              } else {
                results.push(void 0);
              }
            }
          }
          return results;
        } else {
          entry.classList.remove('selected');
          ref4 = e.dataTransfer.files;
          results1 = [];
          for (k = 0, len = ref4.length; k < len; k++) {
            file = ref4[k];
            if ((process.platform === 'darwin' && e.metaKey) || e.ctrlKey) {
              results1.push(this.copyEntry(file.path, newDirectoryPath));
            } else {
              if (!this.moveEntry(file.path, newDirectoryPath)) {
                break;
              } else {
                results1.push(void 0);
              }
            }
          }
          return results1;
        }
      } else if (e.dataTransfer.files.length) {
        ref5 = e.dataTransfer.files;
        results2 = [];
        for (l = 0, len1 = ref5.length; l < len1; l++) {
          entry = ref5[l];
          results2.push(atom.project.addPath(entry.path));
        }
        return results2;
      }
    };

    TreeView.prototype.isAtomTreeViewEvent = function(e) {
      var item, j, len, ref2;
      ref2 = e.dataTransfer.items;
      for (j = 0, len = ref2.length; j < len; j++) {
        item = ref2[j];
        if (item.type === 'atom-tree-view-event' || item.kind === 'file') {
          return true;
        }
      }
      return false;
    };

    TreeView.prototype.isVisible = function() {
      return this.element.offsetWidth !== 0 || this.element.offsetHeight !== 0;
    };

    return TreeView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
