"use strict";

const qs = require('querystring');

const {
  remote,
  ipcRenderer: ipc
} = require('electron');

const {
  GitProcess
} = require('dugite');

class AverageTracker {
  constructor({
    limit
  } = {
    limit: 10
  }) {
    // for now this serves a dual purpose - # of values tracked AND # discarded prior to starting avg calculation
    this.limit = limit;
    this.sum = 0;
    this.values = [];
  }

  addValue(value) {
    if (this.values.length >= this.limit) {
      const discardedValue = this.values.shift();
      this.sum -= discardedValue;
    }

    this.values.push(value);
    this.sum += value;
  }

  getAverage() {
    if (this.enoughData()) {
      return this.sum / this.limit;
    } else {
      return null;
    }
  }

  getLimit() {
    return this.limit;
  }

  enoughData() {
    return this.values.length === this.limit;
  }

}

const query = qs.parse(window.location.search.substr(1));
const sourceWebContentsId = remote.getCurrentWindow().webContents.id;
const operationCountLimit = parseInt(query.operationCountLimit, 10);
const averageTracker = new AverageTracker({
  limit: operationCountLimit
});
const childPidsById = new Map();

const destroyRenderer = () => {
  if (!managerWebContents.isDestroyed()) {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  }

  const win = remote.BrowserWindow.fromWebContents(remote.getCurrentWebContents());

  if (win && !win.isDestroyed()) {
    win.destroy();
  }
};

const managerWebContentsId = parseInt(query.managerWebContentsId, 10);
const managerWebContents = remote.webContents.fromId(managerWebContentsId);

if (managerWebContents && !managerWebContents.isDestroyed()) {
  managerWebContents.on('crashed', destroyRenderer);
  managerWebContents.on('destroyed', destroyRenderer);

  window.onbeforeunload = () => {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  };
}

const channelName = query.channelName;
ipc.on(channelName, (event, {
  type,
  data
}) => {
  if (type === 'git-exec') {
    const {
      args,
      workingDir,
      options,
      id
    } = data;

    if (args) {
      document.getElementById('command').textContent = `git ${args.join(' ')}`;
    }

    options.processCallback = child => {
      childPidsById.set(id, child.pid);
      child.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-spawn-error',
          data: {
            id,
            err
          }
        });
      });
      child.stdin.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-stdin-error',
          data: {
            id,
            stdin: options.stdin,
            err
          }
        });
      });
    };

    const spawnStart = performance.now();
    GitProcess.exec(args, workingDir, options).then(({
      stdout,
      stderr,
      exitCode
    }) => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById["delete"](id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout,
            stderr,
            exitCode,
            timing
          }
        }
      });
    }, err => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById["delete"](id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout: err.stdout,
            stderr: err.stderr,
            exitCode: err.code,
            signal: err.signal,
            timing
          }
        }
      });
    });
    const spawnEnd = performance.now();
    averageTracker.addValue(spawnEnd - spawnStart); // TODO: consider using this to avoid duplicate write operations upon crashing.
    // For now we won't do this to avoid clogging up ipc channel
    // event.sender.sendTo(managerWebContentsId, channelName, {sourceWebContentsId, type: 'exec-started', data: {id}});

    if (averageTracker.enoughData() && averageTracker.getAverage() > 20) {
      event.sender.sendTo(managerWebContentsId, channelName, {
        type: 'slow-spawns'
      });
    }
  } else if (type === 'git-cancel') {
    const {
      id
    } = data;
    const childPid = childPidsById.get(id);

    if (childPid !== undefined) {
      require('tree-kill')(childPid, 'SIGINT', () => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-cancelled',
          data: {
            id,
            childPid
          }
        });
      });

      childPidsById["delete"](id);
    }
  } else {
    throw new Error(`Could not identify type ${type}`);
  }
});
ipc.sendTo(managerWebContentsId, channelName, {
  sourceWebContentsId,
  type: 'renderer-ready',
  data: {
    pid: process.pid
  }
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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