"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _electron = require("electron");

var _atom = require("atom");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _path = _interopRequireDefault(require("path"));

var _propTypes2 = require("../prop-types");

var _filePatchListItemView = _interopRequireDefault(require("./file-patch-list-item-view"));

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _mergeConflictListItemView = _interopRequireDefault(require("./merge-conflict-list-item-view"));

var _compositeListSelection = _interopRequireDefault(require("../models/composite-list-selection"));

var _resolutionProgress = _interopRequireDefault(require("../models/conflicts/resolution-progress"));

var _commitView = _interopRequireDefault(require("./commit-view"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  Menu,
  MenuItem
} = _electron.remote;

const debounce = (fn, wait) => {
  let timeout;
  return (...args) => {
    return new Promise(resolve => {
      clearTimeout(timeout);
      timeout = setTimeout(() => {
        resolve(fn(...args));
      }, wait);
    });
  };
};

function calculateTruncatedLists(lists) {
  return Object.keys(lists).reduce((acc, key) => {
    const list = lists[key];
    acc.source[key] = list;

    if (list.length <= MAXIMUM_LISTED_ENTRIES) {
      acc[key] = list;
    } else {
      acc[key] = list.slice(0, MAXIMUM_LISTED_ENTRIES);
    }

    return acc;
  }, {
    source: {}
  });
}

const noop = () => {};

const MAXIMUM_LISTED_ENTRIES = 1000;

class StagingView extends _react["default"].Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "undoLastDiscardFromCoreUndo", () => {
      this.undoLastDiscard({
        eventSource: {
          command: 'core:undo'
        }
      });
    });

    _defineProperty(this, "undoLastDiscardFromCommand", () => {
      this.undoLastDiscard({
        eventSource: {
          command: 'github:undo-last-discard-in-git-tab'
        }
      });
    });

    _defineProperty(this, "undoLastDiscardFromButton", () => {
      this.undoLastDiscard({
        eventSource: 'button'
      });
    });

    _defineProperty(this, "undoLastDiscardFromHeaderMenu", () => {
      this.undoLastDiscard({
        eventSource: 'header-menu'
      });
    });

    _defineProperty(this, "discardChangesFromCommand", () => {
      this.discardChanges({
        eventSource: {
          command: 'github:discard-changes-in-selected-files'
        }
      });
    });

    _defineProperty(this, "discardAllFromCommand", () => {
      this.discardAll({
        eventSource: {
          command: 'github:discard-all-changes'
        }
      });
    });

    _defineProperty(this, "confirmSelectedItems", async () => {
      const itemPaths = this.getSelectedItemFilePaths();
      await this.props.attemptFileStageOperation(itemPaths, this.state.selection.getActiveListKey());
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.coalesce()
        }), resolve);
      });
    });

    (0, _helpers.autobind)(this, 'dblclickOnItem', 'contextMenuOnItem', 'mousedownOnItem', 'mousemoveOnItem', 'mouseup', 'registerItemElement', 'renderBody', 'openFile', 'discardChanges', 'activateNextList', 'activatePreviousList', 'activateLastList', 'stageAll', 'unstageAll', 'stageAllMergeConflicts', 'discardAll', 'confirmSelectedItems', 'selectAll', 'selectFirst', 'selectLast', 'diveIntoSelection', 'showDiffView', 'showBulkResolveMenu', 'showActionsMenu', 'resolveCurrentAsOurs', 'resolveCurrentAsTheirs', 'quietlySelectItem', 'didChangeSelectedItems');
    this.subs = new _eventKit.CompositeDisposable(atom.config.observe('github.keyboardNavigationDelay', value => {
      if (value === 0) {
        this.debouncedDidChangeSelectedItem = this.didChangeSelectedItems;
      } else {
        this.debouncedDidChangeSelectedItem = debounce(this.didChangeSelectedItems, value);
      }
    }));
    this.state = _objectSpread2({}, calculateTruncatedLists({
      unstagedChanges: this.props.unstagedChanges,
      stagedChanges: this.props.stagedChanges,
      mergeConflicts: this.props.mergeConflicts
    }), {
      selection: new _compositeListSelection["default"]({
        listsByKey: [['unstaged', this.props.unstagedChanges], ['conflicts', this.props.mergeConflicts], ['staged', this.props.stagedChanges]],
        idForItem: item => item.filePath
      })
    });
    this.mouseSelectionInProgress = false;
    this.listElementsByItem = new WeakMap();
    this.refRoot = new _refHolder["default"]();
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    let nextState = {};

    if (['unstagedChanges', 'stagedChanges', 'mergeConflicts'].some(key => prevState.source[key] !== nextProps[key])) {
      const nextLists = calculateTruncatedLists({
        unstagedChanges: nextProps.unstagedChanges,
        stagedChanges: nextProps.stagedChanges,
        mergeConflicts: nextProps.mergeConflicts
      });
      nextState = _objectSpread2({}, nextLists, {
        selection: prevState.selection.updateLists([['unstaged', nextLists.unstagedChanges], ['conflicts', nextLists.mergeConflicts], ['staged', nextLists.stagedChanges]])
      });
    }

    return nextState;
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.mouseup);
    this.subs.add(new _eventKit.Disposable(() => window.removeEventListener('mouseup', this.mouseup)), this.props.workspace.onDidChangeActivePaneItem(() => {
      this.syncWithWorkspace();
    }));

    if (this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  componentDidUpdate(prevProps, prevState) {
    const isRepoSame = prevProps.workingDirectoryPath === this.props.workingDirectoryPath;
    const hasSelectionsPresent = prevState.selection.getSelectedItems().size > 0 && this.state.selection.getSelectedItems().size > 0;
    const selectionChanged = this.state.selection !== prevState.selection;

    if (isRepoSame && hasSelectionsPresent && selectionChanged) {
      this.debouncedDidChangeSelectedItem();
    }

    const headItem = this.state.selection.getHeadItem();

    if (headItem) {
      const element = this.listElementsByItem.get(headItem);

      if (element) {
        element.scrollIntoViewIfNeeded();
      }
    }

    if (!this.isPopulated(prevProps) && this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  render() {
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.resolutionProgress,
      fetchData: noop
    }, this.renderBody);
  }

  renderBody() {
    const selectedItems = this.state.selection.getSelectedItems();
    return _react["default"].createElement("div", {
      ref: this.refRoot.setter,
      className: `github-StagingView ${this.state.selection.getActiveListKey()}-changes-focused`,
      tabIndex: "-1"
    }, this.renderCommands(), _react["default"].createElement("div", {
      className: `github-StagingView-group github-UnstagedChanges ${this.getFocusClass('unstaged')}`
    }, _react["default"].createElement("header", {
      className: "github-StagingView-header"
    }, _react["default"].createElement("span", {
      className: "icon icon-list-unordered"
    }), _react["default"].createElement("span", {
      className: "github-StagingView-title"
    }, "Unstaged Changes"), this.renderActionsMenu(), _react["default"].createElement("button", {
      className: "github-StagingView-headerButton icon icon-move-down",
      disabled: this.props.unstagedChanges.length === 0,
      onClick: this.stageAll
    }, "Stage All")), _react["default"].createElement("div", {
      className: "github-StagingView-list github-FilePatchListView github-StagingView-unstaged"
    }, this.state.unstagedChanges.map(filePatch => _react["default"].createElement(_filePatchListItemView["default"], {
      key: filePatch.filePath,
      registerItemElement: this.registerItemElement,
      filePatch: filePatch,
      onDoubleClick: event => this.dblclickOnItem(event, filePatch),
      onContextMenu: event => this.contextMenuOnItem(event, filePatch),
      onMouseDown: event => this.mousedownOnItem(event, filePatch),
      onMouseMove: event => this.mousemoveOnItem(event, filePatch),
      selected: selectedItems.has(filePatch)
    }))), this.renderTruncatedMessage(this.props.unstagedChanges)), this.renderMergeConflicts(), _react["default"].createElement("div", {
      className: `github-StagingView-group github-StagedChanges ${this.getFocusClass('staged')}`
    }, _react["default"].createElement("header", {
      className: "github-StagingView-header"
    }, _react["default"].createElement("span", {
      className: "icon icon-tasklist"
    }), _react["default"].createElement("span", {
      className: "github-StagingView-title"
    }, "Staged Changes"), _react["default"].createElement("button", {
      className: "github-StagingView-headerButton icon icon-move-up",
      disabled: this.props.stagedChanges.length === 0,
      onClick: this.unstageAll
    }, "Unstage All")), _react["default"].createElement("div", {
      className: "github-StagingView-list github-FilePatchListView github-StagingView-staged"
    }, this.state.stagedChanges.map(filePatch => _react["default"].createElement(_filePatchListItemView["default"], {
      key: filePatch.filePath,
      filePatch: filePatch,
      registerItemElement: this.registerItemElement,
      onDoubleClick: event => this.dblclickOnItem(event, filePatch),
      onContextMenu: event => this.contextMenuOnItem(event, filePatch),
      onMouseDown: event => this.mousedownOnItem(event, filePatch),
      onMouseMove: event => this.mousemoveOnItem(event, filePatch),
      selected: selectedItems.has(filePatch)
    }))), this.renderTruncatedMessage(this.props.stagedChanges)));
  }

  renderCommands() {
    return _react["default"].createElement(_react.Fragment, null, _react["default"].createElement(_commands["default"], {
      registry: this.props.commands,
      target: ".github-StagingView"
    }, _react["default"].createElement(_commands.Command, {
      command: "core:move-up",
      callback: () => this.selectPrevious()
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-down",
      callback: () => this.selectNext()
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-left",
      callback: this.diveIntoSelection
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-diff-view",
      callback: this.showDiffView
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-up",
      callback: () => this.selectPrevious(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-down",
      callback: () => this.selectNext(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-all",
      callback: this.selectAll
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-to-top",
      callback: this.selectFirst
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-to-bottom",
      callback: this.selectLast
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-to-top",
      callback: () => this.selectFirst(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-to-bottom",
      callback: () => this.selectLast(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:confirm",
      callback: this.confirmSelectedItems
    }), _react["default"].createElement(_commands.Command, {
      command: "github:activate-next-list",
      callback: this.activateNextList
    }), _react["default"].createElement(_commands.Command, {
      command: "github:activate-previous-list",
      callback: this.activatePreviousList
    }), _react["default"].createElement(_commands.Command, {
      command: "github:jump-to-file",
      callback: this.openFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-file-as-ours",
      callback: this.resolveCurrentAsOurs
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-file-as-theirs",
      callback: this.resolveCurrentAsTheirs
    }), _react["default"].createElement(_commands.Command, {
      command: "github:discard-changes-in-selected-files",
      callback: this.discardChangesFromCommand
    }), _react["default"].createElement(_commands.Command, {
      command: "core:undo",
      callback: this.undoLastDiscardFromCoreUndo
    })), _react["default"].createElement(_commands["default"], {
      registry: this.props.commands,
      target: "atom-workspace"
    }, _react["default"].createElement(_commands.Command, {
      command: "github:stage-all-changes",
      callback: this.stageAll
    }), _react["default"].createElement(_commands.Command, {
      command: "github:unstage-all-changes",
      callback: this.unstageAll
    }), _react["default"].createElement(_commands.Command, {
      command: "github:discard-all-changes",
      callback: this.discardAllFromCommand
    }), _react["default"].createElement(_commands.Command, {
      command: "github:undo-last-discard-in-git-tab",
      callback: this.undoLastDiscardFromCommand
    })));
  }

  renderActionsMenu() {
    if (this.props.unstagedChanges.length || this.props.hasUndoHistory) {
      return _react["default"].createElement("button", {
        className: "github-StagingView-headerButton github-StagingView-headerButton--iconOnly icon icon-ellipses",
        onClick: this.showActionsMenu
      });
    } else {
      return null;
    }
  }

  renderUndoButton() {
    return _react["default"].createElement("button", {
      className: "github-StagingView-headerButton github-StagingView-headerButton--fullWidth icon icon-history",
      onClick: this.undoLastDiscardFromButton
    }, "Undo Discard");
  }

  renderTruncatedMessage(list) {
    if (list.length > MAXIMUM_LISTED_ENTRIES) {
      return _react["default"].createElement("div", {
        className: "github-StagingView-group-truncatedMsg"
      }, "List truncated to the first ", MAXIMUM_LISTED_ENTRIES, " items");
    } else {
      return null;
    }
  }

  renderMergeConflicts() {
    const mergeConflicts = this.state.mergeConflicts;

    if (mergeConflicts && mergeConflicts.length > 0) {
      const selectedItems = this.state.selection.getSelectedItems();
      const resolutionProgress = this.props.resolutionProgress;
      const anyUnresolved = mergeConflicts.map(conflict => _path["default"].join(this.props.workingDirectoryPath, conflict.filePath)).some(conflictPath => resolutionProgress.getRemaining(conflictPath) !== 0);
      const bulkResolveDropdown = anyUnresolved ? _react["default"].createElement("span", {
        className: "inline-block icon icon-ellipses",
        onClick: this.showBulkResolveMenu
      }) : null;
      return _react["default"].createElement("div", {
        className: `github-StagingView-group github-MergeConflictPaths ${this.getFocusClass('conflicts')}`
      }, _react["default"].createElement("header", {
        className: "github-StagingView-header"
      }, _react["default"].createElement("span", {
        className: 'github-FilePatchListView-icon icon icon-alert status-modified'
      }), _react["default"].createElement("span", {
        className: "github-StagingView-title"
      }, "Merge Conflicts"), bulkResolveDropdown, _react["default"].createElement("button", {
        className: "github-StagingView-headerButton icon icon-move-down",
        disabled: anyUnresolved,
        onClick: this.stageAllMergeConflicts
      }, "Stage All")), _react["default"].createElement("div", {
        className: "github-StagingView-list github-FilePatchListView github-StagingView-merge"
      }, mergeConflicts.map(mergeConflict => {
        const fullPath = _path["default"].join(this.props.workingDirectoryPath, mergeConflict.filePath);

        return _react["default"].createElement(_mergeConflictListItemView["default"], {
          key: fullPath,
          mergeConflict: mergeConflict,
          remainingConflicts: resolutionProgress.getRemaining(fullPath),
          registerItemElement: this.registerItemElement,
          onDoubleClick: event => this.dblclickOnItem(event, mergeConflict),
          onContextMenu: event => this.contextMenuOnItem(event, mergeConflict),
          onMouseDown: event => this.mousedownOnItem(event, mergeConflict),
          onMouseMove: event => this.mousemoveOnItem(event, mergeConflict),
          selected: selectedItems.has(mergeConflict)
        });
      })), this.renderTruncatedMessage(mergeConflicts));
    } else {
      return _react["default"].createElement("noscript", null);
    }
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  getSelectedItemFilePaths() {
    return Array.from(this.state.selection.getSelectedItems(), item => item.filePath);
  }

  getSelectedConflictPaths() {
    if (this.state.selection.getActiveListKey() !== 'conflicts') {
      return [];
    }

    return this.getSelectedItemFilePaths();
  }

  openFile() {
    const filePaths = this.getSelectedItemFilePaths();
    return this.props.openFiles(filePaths);
  }

  discardChanges({
    eventSource
  } = {}) {
    const filePaths = this.getSelectedItemFilePaths();
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      "package": 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'selected',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  activateNextList() {
    return new Promise(resolve => {
      let advanced = false;
      this.setState(prevState => {
        const next = prevState.selection.activateNextSelection();

        if (prevState.selection === next) {
          return {};
        }

        advanced = true;
        return {
          selection: next.coalesce()
        };
      }, () => resolve(advanced));
    });
  }

  activatePreviousList() {
    return new Promise(resolve => {
      let retreated = false;
      this.setState(prevState => {
        const next = prevState.selection.activatePreviousSelection();

        if (prevState.selection === next) {
          return {};
        }

        retreated = true;
        return {
          selection: next.coalesce()
        };
      }, () => resolve(retreated));
    });
  }

  activateLastList() {
    return new Promise(resolve => {
      let emptySelection = false;
      this.setState(prevState => {
        const next = prevState.selection.activateLastSelection();
        emptySelection = next.getSelectedItems().size > 0;

        if (prevState.selection === next) {
          return {};
        }

        return {
          selection: next.coalesce()
        };
      }, () => resolve(emptySelection));
    });
  }

  stageAll() {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }

    return this.props.attemptStageAllOperation('unstaged');
  }

  unstageAll() {
    if (this.props.stagedChanges.length === 0) {
      return null;
    }

    return this.props.attemptStageAllOperation('staged');
  }

  stageAllMergeConflicts() {
    if (this.props.mergeConflicts.length === 0) {
      return null;
    }

    const filePaths = this.props.mergeConflicts.map(conflict => conflict.filePath);
    return this.props.attemptFileStageOperation(filePaths, 'unstaged');
  }

  discardAll({
    eventSource
  } = {}) {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }

    const filePaths = this.props.unstagedChanges.map(filePatch => filePatch.filePath);
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      "package": 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'all',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  getNextListUpdatePromise() {
    return this.state.selection.getNextUpdatePromise();
  }

  selectPrevious(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectPreviousItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectNext(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectNextItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectAll() {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectAllItems().coalesce()
      }), resolve);
    });
  }

  selectFirst(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectFirstItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectLast(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectLastItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  async diveIntoSelection() {
    const selectedItems = this.state.selection.getSelectedItems();

    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath, {
        activate: true
      });
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
        activate: true
      });
    }
  }

  async syncWithWorkspace() {
    const item = this.props.workspace.getActivePaneItem();

    if (!item) {
      return;
    }

    const realItemPromise = item.getRealItemPromise && item.getRealItemPromise();
    const realItem = await realItemPromise;

    if (!realItem) {
      return;
    }

    const isFilePatchItem = realItem.isFilePatchItem && realItem.isFilePatchItem();
    const isMatch = realItem.getWorkingDirectory && realItem.getWorkingDirectory() === this.props.workingDirectoryPath;

    if (isFilePatchItem && isMatch) {
      this.quietlySelectItem(realItem.getFilePath(), realItem.getStagingStatus());
    }
  }

  async showDiffView() {
    const selectedItems = this.state.selection.getSelectedItems();

    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath);
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey());
    }
  }

  showBulkResolveMenu(event) {
    const conflictPaths = this.props.mergeConflicts.map(c => c.filePath);
    event.preventDefault();
    const menu = new Menu();
    menu.append(new MenuItem({
      label: 'Resolve All as Ours',
      click: () => this.props.resolveAsOurs(conflictPaths)
    }));
    menu.append(new MenuItem({
      label: 'Resolve All as Theirs',
      click: () => this.props.resolveAsTheirs(conflictPaths)
    }));
    menu.popup(_electron.remote.getCurrentWindow());
  }

  showActionsMenu(event) {
    event.preventDefault();
    const menu = new Menu();
    const selectedItemCount = this.state.selection.getSelectedItems().size;
    const pluralization = selectedItemCount > 1 ? 's' : '';
    menu.append(new MenuItem({
      label: 'Discard All Changes',
      click: () => this.discardAll({
        eventSource: 'header-menu'
      }),
      enabled: this.props.unstagedChanges.length > 0
    }));
    menu.append(new MenuItem({
      label: 'Discard Changes in Selected File' + pluralization,
      click: () => this.discardChanges({
        eventSource: 'header-menu'
      }),
      enabled: !!(this.props.unstagedChanges.length && selectedItemCount)
    }));
    menu.append(new MenuItem({
      label: 'Undo Last Discard',
      click: () => this.undoLastDiscard({
        eventSource: 'header-menu'
      }),
      enabled: this.props.hasUndoHistory
    }));
    menu.popup(_electron.remote.getCurrentWindow());
  }

  resolveCurrentAsOurs() {
    this.props.resolveAsOurs(this.getSelectedConflictPaths());
  }

  resolveCurrentAsTheirs() {
    this.props.resolveAsTheirs(this.getSelectedConflictPaths());
  } // Directly modify the selection to include only the item identified by the file path and stagingStatus tuple.
  // Re-render the component, but don't notify didSelectSingleItem() or other callback functions. This is useful to
  // avoid circular callback loops for actions originating in FilePatchView or TextEditors with merge conflicts.


  quietlySelectItem(filePath, stagingStatus) {
    return new Promise(resolve => {
      this.setState(prevState => {
        const item = prevState.selection.findItem((each, key) => each.filePath === filePath && key === stagingStatus);

        if (!item) {
          // FIXME: make staging view display no selected item
          // eslint-disable-next-line no-console
          console.log(`Unable to find item at path ${filePath} with staging status ${stagingStatus}`);
          return null;
        }

        return {
          selection: prevState.selection.selectItem(item)
        };
      }, resolve);
    });
  }

  getSelectedItems() {
    const stagingStatus = this.state.selection.getActiveListKey();
    return Array.from(this.state.selection.getSelectedItems(), item => {
      return {
        filePath: item.filePath,
        stagingStatus
      };
    });
  }

  didChangeSelectedItems(openNew) {
    const selectedItems = Array.from(this.state.selection.getSelectedItems());

    if (selectedItems.length === 1) {
      this.didSelectSingleItem(selectedItems[0], openNew);
    }
  }

  async didSelectSingleItem(selectedItem, openNew = false) {
    if (!this.hasFocus()) {
      return;
    }

    if (this.state.selection.getActiveListKey() === 'conflicts') {
      if (openNew) {
        await this.showMergeConflictFileForPath(selectedItem.filePath, {
          activate: true
        });
      }
    } else {
      if (openNew) {
        // User explicitly asked to view diff, such as via click
        await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
          activate: false
        });
      } else {
        const panesWithStaleItemsToUpdate = this.getPanesWithStalePendingFilePatchItem();

        if (panesWithStaleItemsToUpdate.length > 0) {
          // Update stale items to reflect new selection
          await Promise.all(panesWithStaleItemsToUpdate.map(async pane => {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane
            });
          }));
        } else {
          // Selection was changed via keyboard navigation, update pending item in active pane
          const activePane = this.props.workspace.getCenter().getActivePane();
          const activePendingItem = activePane.getPendingItem();

          const activePaneHasPendingFilePatchItem = activePendingItem && activePendingItem.getRealItem && activePendingItem.getRealItem() instanceof _changedFileItem["default"];

          if (activePaneHasPendingFilePatchItem) {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane: activePane
            });
          }
        }
      }
    }
  }

  getPanesWithStalePendingFilePatchItem() {
    // "stale" meaning there is no longer a changed file associated with item
    // due to changes being fully staged/unstaged/stashed/deleted/etc
    return this.props.workspace.getPanes().filter(pane => {
      const pendingItem = pane.getPendingItem();

      if (!pendingItem || !pendingItem.getRealItem) {
        return false;
      }

      const realItem = pendingItem.getRealItem();

      if (!(realItem instanceof _changedFileItem["default"])) {
        return false;
      } // We only want to update pending diff views for currently active repo


      const isInActiveRepo = realItem.getWorkingDirectory() === this.props.workingDirectoryPath;
      const isStale = !this.changedFileExists(realItem.getFilePath(), realItem.getStagingStatus());
      return isInActiveRepo && isStale;
    });
  }

  changedFileExists(filePath, stagingStatus) {
    return this.state.selection.findItem((item, key) => {
      return key === stagingStatus && item.filePath === filePath;
    });
  }

  async showFilePatchItem(filePath, stagingStatus, {
    activate,
    pane
  } = {
    activate: false
  }) {
    const uri = _changedFileItem["default"].buildURI(filePath, this.props.workingDirectoryPath, stagingStatus);

    const changedFileItem = await this.props.workspace.open(uri, {
      pending: true,
      activatePane: activate,
      activateItem: activate,
      pane
    });

    if (activate) {
      const itemRoot = changedFileItem.getElement();
      const focusRoot = itemRoot.querySelector('[tabIndex]');

      if (focusRoot) {
        focusRoot.focus();
      }
    } else {
      // simply make item visible
      this.props.workspace.paneForItem(changedFileItem).activateItem(changedFileItem);
    }
  }

  async showMergeConflictFileForPath(relativeFilePath, {
    activate
  } = {
    activate: false
  }) {
    const absolutePath = _path["default"].join(this.props.workingDirectoryPath, relativeFilePath);

    if (await this.fileExists(absolutePath)) {
      return this.props.workspace.open(absolutePath, {
        activatePane: activate,
        activateItem: activate,
        pending: true
      });
    } else {
      this.props.notificationManager.addInfo('File has been deleted.');
      return null;
    }
  }

  fileExists(absolutePath) {
    return new _atom.File(absolutePath).exists();
  }

  dblclickOnItem(event, item) {
    return this.props.attemptFileStageOperation([item.filePath], this.state.selection.listKeyForItem(item));
  }

  async contextMenuOnItem(event, item) {
    if (!this.state.selection.getSelectedItems().has(item)) {
      event.stopPropagation();
      event.persist();
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, event.shiftKey)
        }), resolve);
      });
      const newEvent = new MouseEvent(event.type, event);
      requestAnimationFrame(() => {
        if (!event.target.parentNode) {
          return;
        }

        event.target.parentNode.dispatchEvent(newEvent);
      });
    }
  }

  async mousedownOnItem(event, item) {
    const windows = process.platform === 'win32';

    if (event.ctrlKey && !windows) {
      return;
    } // simply open context menu


    if (event.button === 0) {
      this.mouseSelectionInProgress = true;
      event.persist();
      await new Promise(resolve => {
        if (event.metaKey || event.ctrlKey && windows) {
          this.setState(prevState => ({
            selection: prevState.selection.addOrSubtractSelection(item)
          }), resolve);
        } else {
          this.setState(prevState => ({
            selection: prevState.selection.selectItem(item, event.shiftKey)
          }), resolve);
        }
      });
    }
  }

  async mousemoveOnItem(event, item) {
    if (this.mouseSelectionInProgress) {
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, true)
        }), resolve);
      });
    }
  }

  async mouseup() {
    const hadSelectionInProgress = this.mouseSelectionInProgress;
    this.mouseSelectionInProgress = false;
    await new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.coalesce()
      }), resolve);
    });

    if (hadSelectionInProgress) {
      this.didChangeSelectedItems(true);
    }
  }

  undoLastDiscard({
    eventSource
  } = {}) {
    if (!this.props.hasUndoHistory) {
      return;
    }

    (0, _reporterProxy.addEvent)('undo-last-discard', {
      "package": 'github',
      component: 'StagingView',
      eventSource
    });
    this.props.undoLastDiscard();
  }

  getFocusClass(listKey) {
    return this.state.selection.getActiveListKey() === listKey ? 'is-focused' : '';
  }

  registerItemElement(item, element) {
    this.listElementsByItem.set(item, element);
  }

  getFocus(element) {
    return this.refRoot.map(root => root.contains(element)).getOr(false) ? StagingView.focus.STAGING : null;
  }

  setFocus(focus) {
    if (focus === this.constructor.focus.STAGING) {
      this.refRoot.map(root => root.focus());
      return true;
    }

    return false;
  }

  async advanceFocusFrom(focus) {
    if (focus === this.constructor.focus.STAGING) {
      if (await this.activateNextList()) {
        // There was a next list to activate.
        return this.constructor.focus.STAGING;
      } // We were already on the last list.


      return _commitView["default"].firstFocus;
    }

    return null;
  }

  async retreatFocusFrom(focus) {
    if (focus === _commitView["default"].firstFocus) {
      await this.activateLastList();
      return this.constructor.focus.STAGING;
    }

    if (focus === this.constructor.focus.STAGING) {
      await this.activatePreviousList();
      return this.constructor.focus.STAGING;
    }

    return false;
  }

  hasFocus() {
    return this.refRoot.map(root => root.contains(document.activeElement)).getOr(false);
  }

  isPopulated(props) {
    return props.workingDirectoryPath != null && (props.unstagedChanges.length > 0 || props.mergeConflicts.length > 0 || props.stagedChanges.length > 0);
  }

}

exports["default"] = StagingView;

_defineProperty(StagingView, "propTypes", {
  unstagedChanges: _propTypes["default"].arrayOf(_propTypes2.FilePatchItemPropType).isRequired,
  stagedChanges: _propTypes["default"].arrayOf(_propTypes2.FilePatchItemPropType).isRequired,
  mergeConflicts: _propTypes["default"].arrayOf(_propTypes2.MergeConflictItemPropType),
  workingDirectoryPath: _propTypes["default"].string,
  resolutionProgress: _propTypes["default"].object,
  hasUndoHistory: _propTypes["default"].bool.isRequired,
  commands: _propTypes["default"].object.isRequired,
  notificationManager: _propTypes["default"].object.isRequired,
  workspace: _propTypes["default"].object.isRequired,
  openFiles: _propTypes["default"].func.isRequired,
  attemptFileStageOperation: _propTypes["default"].func.isRequired,
  discardWorkDirChangesForPaths: _propTypes["default"].func.isRequired,
  undoLastDiscard: _propTypes["default"].func.isRequired,
  attemptStageAllOperation: _propTypes["default"].func.isRequired,
  resolveAsOurs: _propTypes["default"].func.isRequired,
  resolveAsTheirs: _propTypes["default"].func.isRequired
});

_defineProperty(StagingView, "defaultProps", {
  mergeConflicts: [],
  resolutionProgress: new _resolutionProgress["default"]()
});

_defineProperty(StagingView, "focus", {
  STAGING: Symbol('staging')
});

_defineProperty(StagingView, "firstFocus", StagingView.focus.STAGING);

_defineProperty(StagingView, "lastFocus", StagingView.focus.STAGING);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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