"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = IssueishLink;
exports.handleClickEvent = handleClickEvent;
exports.openIssueishLinkInNewTab = openIssueishLinkInNewTab;
exports.openLinkInBrowser = openLinkInBrowser;
exports.getDataFromGithubUrl = getDataFromGithubUrl;

var _url = _interopRequireDefault(require("url"));

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

// eslint-disable-next-line no-shadow
function IssueishLink(_ref) {
  let {
    url,
    children
  } = _ref,
      others = _objectWithoutProperties(_ref, ["url", "children"]);

  function clickHandler(event) {
    handleClickEvent(event, url);
  }

  return _react["default"].createElement("a", _extends({}, others, {
    onClick: clickHandler
  }), children);
}

IssueishLink.propTypes = {
  url: _propTypes["default"].string.isRequired,
  children: _propTypes["default"].node
}; // eslint-disable-next-line no-shadow

function handleClickEvent(event, url) {
  event.preventDefault();
  event.stopPropagation();

  if (!event.shiftKey) {
    return openIssueishLinkInNewTab(url, {
      activate: !(event.metaKey || event.ctrlKey)
    });
  } else {
    // Open in browser if shift key held
    return openLinkInBrowser(url);
  }
} // eslint-disable-next-line no-shadow


function openIssueishLinkInNewTab(url, options = {}) {
  const uri = getAtomUriForGithubUrl(url);

  if (uri) {
    return openInNewTab(uri, options);
  } else {
    return null;
  }
}

function openLinkInBrowser(uri) {
  return new Promise((resolve, reject) => {
    _electron.shell.openExternal(uri, {}, err => {
      if (err) {
        reject(err);
      } else {
        (0, _reporterProxy.addEvent)('open-issueish-in-browser', {
          "package": 'github',
          from: 'issueish-link'
        });
        resolve();
      }
    });
  });
}

function getAtomUriForGithubUrl(githubUrl) {
  return getUriForData(getDataFromGithubUrl(githubUrl));
}

function getDataFromGithubUrl(githubUrl) {
  const {
    hostname,
    pathname
  } = _url["default"].parse(githubUrl);

  const [repoOwner, repoName, type, issueishNumber] = pathname.split('/').filter(s => s);
  return {
    hostname,
    repoOwner,
    repoName,
    type,
    issueishNumber: parseInt(issueishNumber, 10)
  };
}

function getUriForData({
  hostname,
  repoOwner,
  repoName,
  type,
  issueishNumber
}) {
  if (hostname !== 'github.com' || !['pull', 'issues'].includes(type) || !issueishNumber || isNaN(issueishNumber)) {
    return null;
  } else {
    return _issueishDetailItem["default"].buildURI({
      host: 'github.com',
      owner: repoOwner,
      repo: repoName,
      number: issueishNumber
    });
  }
}

function openInNewTab(uri, {
  activate
} = {
  activate: true
}) {
  return atom.workspace.open(uri, {
    activateItem: activate
  }).then(() => {
    (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
      "package": 'github',
      from: 'issueish-link',
      target: 'new-tab'
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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