"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _atom = require("atom");

var _eventKit = require("event-kit");

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _lodash = _interopRequireDefault(require("lodash.memoize"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _octicon = _interopRequireDefault(require("../atom/octicon"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  dialog
} = _electron.remote;
const genArray = (0, _lodash["default"])(function genArray(interval, count) {
  const arr = [];

  for (let i = 1; i <= count; i++) {
    arr.push(interval * i);
  }

  return arr;
}, (interval, count) => `${interval}:${count}`);

class Marker {
  static deserialize(data) {
    const marker = new Marker(data.label, () => {});
    marker.end = data.end;
    marker.markers = data.markers;
    return marker;
  }

  constructor(label, didUpdate) {
    this.label = label;
    this.didUpdate = didUpdate;
    this.end = null;
    this.markers = [];
  }

  getStart() {
    return this.markers.length ? this.markers[0].start : null;
  }

  getEnd() {
    return this.end;
  }

  mark(sectionName, start) {
    this.markers.push({
      name: sectionName,
      start: start || performance.now()
    });
  }

  finalize() {
    this.end = performance.now();
    this.didUpdate();
  }

  getTimings() {
    return this.markers.map((timing, idx, ary) => {
      const next = ary[idx + 1];
      const end = next ? next.start : this.getEnd();
      return _objectSpread2({}, timing, {
        end
      });
    });
  }

  serialize() {
    return {
      label: this.label,
      end: this.end,
      markers: this.markers.slice()
    };
  }

}

class MarkerTooltip extends _react["default"].Component {
  render() {
    const {
      marker
    } = this.props;
    const timings = marker.getTimings();
    return _react["default"].createElement("div", {
      style: {
        textAlign: 'left',
        maxWidth: 300,
        whiteSpace: 'initial'
      }
    }, _react["default"].createElement("strong", null, _react["default"].createElement("tt", null, marker.label)), _react["default"].createElement("ul", {
      style: {
        paddingLeft: 20,
        marginTop: 10
      }
    }, timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return _react["default"].createElement("li", {
        key: name
      }, name, ": ", Math.floor(duration * 100) / 100, "ms");
    })));
  }

}

_defineProperty(MarkerTooltip, "propTypes", {
  marker: _propTypes["default"].instanceOf(Marker).isRequired
});

const COLORS = {
  queued: 'red',
  prepare: 'cyan',
  nexttick: 'yellow',
  execute: 'green',
  ipc: 'pink'
};

class MarkerSpan extends _react["default"].Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleMouseOver', 'handleMouseOut');
  }

  render() {
    const _this$props = this.props,
          {
      marker
    } = _this$props,
          others = _objectWithoutProperties(_this$props, ["marker"]);

    const timings = marker.getTimings();
    const totalTime = marker.getEnd() - marker.getStart();
    const percentages = timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return {
        color: COLORS[name],
        percent: duration / totalTime * 100
      };
    });
    return _react["default"].createElement("span", _extends({}, others, {
      ref: c => {
        this.element = c;
      },
      onMouseOver: this.handleMouseOver,
      onMouseOut: this.handleMouseOut
    }), percentages.map(({
      color,
      percent
    }, i) => {
      const style = {
        width: `${percent}%`,
        background: color
      };
      return _react["default"].createElement("span", {
        className: "waterfall-marker-section",
        key: i,
        style: style
      });
    }));
  }

  handleMouseOver(e) {
    const elem = document.createElement('div');

    _reactDom["default"].render(_react["default"].createElement(MarkerTooltip, {
      marker: this.props.marker
    }), elem);

    this.tooltipDisposable = atom.tooltips.add(this.element, {
      item: elem,
      placement: 'auto bottom',
      trigger: 'manual'
    });
  }

  closeTooltip() {
    this.tooltipDisposable && this.tooltipDisposable.dispose();
    this.tooltipDisposable = null;
  }

  handleMouseOut(e) {
    this.closeTooltip();
  }

  componentWillUnmount() {
    this.closeTooltip();
  }

}

_defineProperty(MarkerSpan, "propTypes", {
  marker: _propTypes["default"].instanceOf(Marker).isRequired
});

class Waterfall extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'renderMarker');
    this.state = this.getNextState(props);
  }

  componentWillReceiveProps(nextProps) {
    this.setState(this.getNextState(nextProps));
  }

  getNextState(props) {
    const {
      markers
    } = props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const totalDuration = endTime - startTime;
    let timelineMarkInterval = null;

    if (props.zoomFactor <= 0.15) {
      timelineMarkInterval = 1000;
    } else if (props.zoomFactor <= 0.3) {
      timelineMarkInterval = 500;
    } else if (props.zoomFactor <= 0.6) {
      timelineMarkInterval = 250;
    } else {
      timelineMarkInterval = 100;
    }

    const timelineMarks = genArray(timelineMarkInterval, Math.ceil(totalDuration / timelineMarkInterval));
    return {
      firstMarker,
      lastMarker,
      startTime,
      endTime,
      totalDuration,
      timelineMarks
    };
  }

  render() {
    return _react["default"].createElement("div", {
      className: "waterfall-scroller"
    }, _react["default"].createElement("div", {
      className: "waterfall-container"
    }, this.renderTimeMarkers(), this.renderTimeline(), this.props.markers.map(this.renderMarker)));
  }

  renderTimeline() {
    return _react["default"].createElement("div", {
      className: "waterfall-timeline"
    }, "\xA0", this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react["default"].createElement("span", {
        className: "waterfall-timeline-label",
        style: style,
        key: `tl:${time}`
      }, time, "ms");
    }));
  }

  renderTimeMarkers() {
    return _react["default"].createElement("div", {
      className: "waterfall-time-markers"
    }, this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react["default"].createElement("span", {
        className: "waterfall-time-marker",
        style: style,
        key: `tm:${time}`
      });
    }));
  }

  renderMarker(marker, i) {
    if (marker.getStart() === null || marker.getEnd() === null) {
      return _react["default"].createElement("div", {
        key: i
      });
    }

    const startOffset = marker.getStart() - this.state.startTime;
    const duration = marker.getEnd() - marker.getStart();
    const markerStyle = {
      left: startOffset * this.props.zoomFactor,
      width: duration * this.props.zoomFactor
    };
    return _react["default"].createElement("div", {
      className: "waterfall-row",
      key: i
    }, _react["default"].createElement("span", {
      className: "waterfall-row-label",
      style: {
        paddingLeft: markerStyle.left + markerStyle.width
      }
    }, marker.label), _react["default"].createElement(MarkerSpan, {
      className: "waterfall-marker",
      style: markerStyle,
      marker: marker
    }));
  }

}

_defineProperty(Waterfall, "propTypes", {
  markers: _propTypes["default"].arrayOf(_propTypes["default"].instanceOf(Marker)).isRequired,
  zoomFactor: _propTypes["default"].number.isRequired
});

class WaterfallWidget extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'handleZoomFactorChange', 'handleCollapseClick', 'handleExportClick');
    this.state = {
      zoomFactor: 0.3,
      collapsed: false
    };
  }

  render() {
    const {
      markers
    } = this.props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const duration = endTime - startTime;
    return _react["default"].createElement("div", {
      className: "waterfall-widget inset-pannel"
    }, _react["default"].createElement("div", {
      className: "waterfall-header"
    }, _react["default"].createElement("div", {
      className: "waterfall-header-text"
    }, _react["default"].createElement("span", {
      onClick: this.handleCollapseClick,
      className: "collapse-toggle"
    }, this.state.collapsed ? '\u25b6' : '\u25bc'), this.props.markers.length, " event(s) over ", Math.floor(duration), "ms"), _react["default"].createElement("div", {
      className: "waterfall-header-controls"
    }, _react["default"].createElement("button", {
      className: "waterfall-export-button btn btn-sm",
      onClick: this.handleExportClick
    }, "Export"), _react["default"].createElement(_octicon["default"], {
      icon: "search"
    }), _react["default"].createElement("input", {
      type: "range",
      className: "input-range",
      min: 0.1,
      max: 1,
      step: 0.01,
      value: this.state.zoomFactor,
      onChange: this.handleZoomFactorChange
    }))), this.state.collapsed ? null : _react["default"].createElement(Waterfall, {
      markers: this.props.markers,
      zoomFactor: this.state.zoomFactor
    }));
  }

  handleZoomFactorChange(e) {
    this.setState({
      zoomFactor: parseFloat(e.target.value)
    });
  }

  handleCollapseClick(e) {
    this.setState(s => ({
      collapsed: !s.collapsed
    }));
  }

  handleExportClick(e) {
    e.preventDefault();
    const json = JSON.stringify(this.props.markers.map(m => m.serialize()), null, '  ');
    const buffer = new _atom.TextBuffer({
      text: json
    });
    dialog.showSaveDialog({
      defaultPath: 'git-timings.json'
    }, filename => {
      if (!filename) {
        return;
      }

      buffer.saveAs(filename);
    });
  }

}

_defineProperty(WaterfallWidget, "propTypes", {
  markers: _propTypes["default"].arrayOf(_propTypes["default"].instanceOf(Marker)).isRequired
});

let markers = null;
let groupId = 0;
const groups = [];
let lastMarkerTime = null;
let updateTimer = null;

class GitTimingsView extends _react["default"].Component {
  static buildURI() {
    return this.uriPattern;
  }

  static generateMarker(label) {
    const marker = new Marker(label, () => {
      GitTimingsView.scheduleUpdate();
    });
    const now = performance.now();

    if (!markers || lastMarkerTime && Math.abs(now - lastMarkerTime) >= 5000) {
      groupId++;
      markers = [];
      groups.unshift({
        id: groupId,
        markers
      });

      if (groups.length > 100) {
        groups.pop();
      }
    }

    lastMarkerTime = now;
    markers.push(marker);
    GitTimingsView.scheduleUpdate();
    return marker;
  }

  static restoreGroup(group) {
    groupId++;
    groups.unshift({
      id: groupId,
      markers: group
    });
    GitTimingsView.scheduleUpdate(true);
  }

  static scheduleUpdate(immediate = false) {
    if (updateTimer) {
      clearTimeout(updateTimer);
    }

    updateTimer = setTimeout(() => {
      GitTimingsView.emitter.emit('did-update');
    }, immediate ? 0 : 1000);
  }

  static onDidUpdate(callback) {
    return GitTimingsView.emitter.on('did-update', callback);
  }

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleImportClick');
  }

  componentDidMount() {
    this.subscriptions = new _eventKit.CompositeDisposable(GitTimingsView.onDidUpdate(() => this.forceUpdate()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  render() {
    return _react["default"].createElement("div", {
      className: "github-GitTimingsView"
    }, _react["default"].createElement("div", {
      className: "github-GitTimingsView-header"
    }, _react["default"].createElement("button", {
      className: "import-button btn",
      onClick: this.handleImportClick
    }, "Import")), groups.map((group, idx) => _react["default"].createElement(WaterfallWidget, {
      key: group.id,
      markers: group.markers
    })));
  }

  handleImportClick(e) {
    e.preventDefault();
    dialog.showOpenDialog({
      properties: ['openFile']
    }, async filenames => {
      if (!filenames) {
        return;
      }

      const filename = filenames[0];

      try {
        const contents = await _fsExtra["default"].readFile(filename, {
          encoding: 'utf8'
        });
        const data = JSON.parse(contents);
        const restoredMarkers = data.map(item => Marker.deserialize(item));
        GitTimingsView.restoreGroup(restoredMarkers);
      } catch (_err) {
        atom.notifications.addError(`Could not import timings from ${filename}`);
      }
    });
  }

  serialize() {
    return {
      deserializer: 'GitTimingsView'
    };
  }

  getURI() {
    return this.constructor.buildURI();
  }

  getTitle() {
    return 'GitHub Package Timings View';
  }

}

exports["default"] = GitTimingsView;

_defineProperty(GitTimingsView, "uriPattern", 'atom-github://debug/timings');

_defineProperty(GitTimingsView, "emitter", new _eventKit.Emitter());
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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