"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _helpers = require("./helpers");

const refreshMapPerUniqueId = new WeakMap();

class PeriodicRefresher {
  static getRefreshMap(uniqueId) {
    let refreshMap = refreshMapPerUniqueId.get(uniqueId);

    if (!refreshMap) {
      refreshMap = new Map();
      refreshMapPerUniqueId.set(uniqueId, refreshMap);
    }

    return refreshMap;
  }

  constructor(uniqueId, options) {
    (0, _helpers.autobind)(this, 'refreshNow');
    this.options = options;
    this._refreshesPerId = PeriodicRefresher.getRefreshMap(uniqueId);
  }

  start() {
    if (!this._timer) {
      this._scheduleNext();
    }
  }

  _clearTimer() {
    if (this._timer) {
      clearTimeout(this._timer);
      delete this._timer;
    }
  }

  _scheduleNext() {
    this._timer = setTimeout(this.refreshNow, this.options.interval());
  }

  refreshNow(force = false) {
    const currentId = this.options.getCurrentId();
    const lastRefreshForId = this._refreshesPerId.get(currentId) || 0;
    const delta = performance.now() - lastRefreshForId;

    if (force || delta > this.options.minimumIntervalPerId) {
      this._refreshesPerId.set(currentId, performance.now());

      this.options.refresh();
    } // Schedule another refresh if one is already scheduled


    if (this._timer) {
      this._clearTimer();

      this._scheduleNext();
    }
  }

  destroy() {
    this._clearTimer();
  }

}

exports["default"] = PeriodicRefresher;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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