"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.FOCUS = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _atom = require("atom");

var _eventLogger = _interopRequireDefault(require("./event-logger"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const FOCUS = Symbol('focus');
exports.FOCUS = FOCUS;

class WorkspaceChangeObserver {
  constructor(window, workspace, repository) {
    (0, _helpers.autobind)(this, 'observeTextEditor');
    this.window = window;
    this.repository = repository;
    this.workspace = workspace;
    this.observedBuffers = new WeakSet();
    this.emitter = new _eventKit.Emitter();
    this.disposables = new _eventKit.CompositeDisposable();
    this.logger = new _eventLogger["default"]('workspace watcher');
    this.started = false;
  }

  async start() {
    const focusHandler = event => {
      if (this.repository) {
        this.logger.showFocusEvent();
        this.didChange([{
          special: FOCUS
        }]);
      }
    };

    this.window.addEventListener('focus', focusHandler);
    this.disposables.add(this.workspace.observeTextEditors(this.observeTextEditor), new _eventKit.Disposable(() => this.window.removeEventListener('focus', focusHandler)));
    await this.watchActiveRepositoryGitDirectory();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.observedBuffers = new WeakSet();
    this.emitter.dispose();
    this.disposables.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchActiveRepositoryGitDirectory() {
    const repository = this.getRepository();
    const gitDirectoryPath = repository.getGitDirectoryPath();
    const basenamesOfInterest = ['config', 'index', 'HEAD', 'MERGE_HEAD'];
    const workdirOrHeadBasenames = ['config', 'index'];

    const eventPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const acceptEvent = event => {
      return eventPaths(event).some(eventPath => {
        return basenamesOfInterest.includes(_path["default"].basename(eventPath)) || _path["default"].dirname(eventPath).includes(_path["default"].join('.git', 'refs'));
      });
    };

    const isWorkdirOrHeadEvent = event => {
      return eventPaths(event).some(eventPath => workdirOrHeadBasenames.includes(_path["default"].basename(eventPath)));
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(gitDirectoryPath, {}, events => {
      const filteredEvents = events.filter(acceptEvent);

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);

        if (filteredEvents.some(isWorkdirOrHeadEvent)) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    });
    this.currentFileWatcher.onDidError(error => {
      const workingDirectory = repository.getWorkingDirectoryPath(); // eslint-disable-next-line no-console

      console.warn(`Error in WorkspaceChangeObserver in ${workingDirectory}:`, error);
      this.stopCurrentFileWatcher();
    });
    this.logger.showStarted(gitDirectoryPath, 'workspace emulated');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.currentFileWatcher = null;
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

  activeRepositoryContainsPath(filePath) {
    const repository = this.getRepository();

    if (filePath && repository) {
      return filePath.indexOf(repository.getWorkingDirectoryPath()) !== -1;
    } else {
      return false;
    }
  }

  observeTextEditor(editor) {
    const buffer = editor.getBuffer();

    if (!this.observedBuffers.has(buffer)) {
      let lastPath = buffer.getPath();

      const didChange = () => {
        const currentPath = buffer.getPath();
        const events = currentPath === lastPath ? [{
          action: 'modified',
          path: currentPath
        }] : [{
          action: 'renamed',
          path: currentPath,
          oldPath: lastPath
        }];
        lastPath = currentPath;
        this.logger.showEvents(events);
        this.didChange(events);
      };

      this.observedBuffers.add(buffer);
      const disposables = new _eventKit.CompositeDisposable(buffer.onDidSave(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidReload(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }
      }), buffer.onDidDestroy(() => {
        if (this.activeRepositoryContainsPath(buffer.getPath())) {
          didChange();
        }

        disposables.dispose();
      }));
      this.disposables.add(disposables);
    }
  }

}

exports["default"] = WorkspaceChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndvcmtzcGFjZS1jaGFuZ2Utb2JzZXJ2ZXIuanMiXSwibmFtZXMiOlsiRk9DVVMiLCJTeW1ib2wiLCJXb3Jrc3BhY2VDaGFuZ2VPYnNlcnZlciIsImNvbnN0cnVjdG9yIiwid2luZG93Iiwid29ya3NwYWNlIiwicmVwb3NpdG9yeSIsIm9ic2VydmVkQnVmZmVycyIsIldlYWtTZXQiLCJlbWl0dGVyIiwiRW1pdHRlciIsImRpc3Bvc2FibGVzIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsImxvZ2dlciIsIkV2ZW50TG9nZ2VyIiwic3RhcnRlZCIsInN0YXJ0IiwiZm9jdXNIYW5kbGVyIiwiZXZlbnQiLCJzaG93Rm9jdXNFdmVudCIsImRpZENoYW5nZSIsInNwZWNpYWwiLCJhZGRFdmVudExpc3RlbmVyIiwiYWRkIiwib2JzZXJ2ZVRleHRFZGl0b3JzIiwib2JzZXJ2ZVRleHRFZGl0b3IiLCJEaXNwb3NhYmxlIiwicmVtb3ZlRXZlbnRMaXN0ZW5lciIsIndhdGNoQWN0aXZlUmVwb3NpdG9yeUdpdERpcmVjdG9yeSIsImRlc3Ryb3kiLCJkaXNwb3NlIiwic3RvcEN1cnJlbnRGaWxlV2F0Y2hlciIsImlzU3RhcnRlZCIsInBheWxvYWQiLCJlbWl0IiwiZGlkQ2hhbmdlV29ya2Rpck9ySGVhZCIsIm9uRGlkQ2hhbmdlIiwiY2FsbGJhY2siLCJvbiIsIm9uRGlkQ2hhbmdlV29ya2Rpck9ySGVhZCIsImdldFJlcG9zaXRvcnkiLCJnaXREaXJlY3RvcnlQYXRoIiwiZ2V0R2l0RGlyZWN0b3J5UGF0aCIsImJhc2VuYW1lc09mSW50ZXJlc3QiLCJ3b3JrZGlyT3JIZWFkQmFzZW5hbWVzIiwiZXZlbnRQYXRocyIsInBzIiwicGF0aCIsIm9sZFBhdGgiLCJwdXNoIiwiYWNjZXB0RXZlbnQiLCJzb21lIiwiZXZlbnRQYXRoIiwiaW5jbHVkZXMiLCJiYXNlbmFtZSIsImRpcm5hbWUiLCJqb2luIiwiaXNXb3JrZGlyT3JIZWFkRXZlbnQiLCJjdXJyZW50RmlsZVdhdGNoZXIiLCJldmVudHMiLCJmaWx0ZXJlZEV2ZW50cyIsImZpbHRlciIsImxlbmd0aCIsInNob3dFdmVudHMiLCJzaG93V29ya2Rpck9ySGVhZEV2ZW50cyIsIm9uRGlkRXJyb3IiLCJlcnJvciIsIndvcmtpbmdEaXJlY3RvcnkiLCJnZXRXb3JraW5nRGlyZWN0b3J5UGF0aCIsImNvbnNvbGUiLCJ3YXJuIiwic2hvd1N0YXJ0ZWQiLCJzaG93U3RvcHBlZCIsIlByb21pc2UiLCJyZXNvbHZlIiwiYWN0aXZlUmVwb3NpdG9yeUNvbnRhaW5zUGF0aCIsImZpbGVQYXRoIiwiaW5kZXhPZiIsImVkaXRvciIsImJ1ZmZlciIsImdldEJ1ZmZlciIsImhhcyIsImxhc3RQYXRoIiwiZ2V0UGF0aCIsImN1cnJlbnRQYXRoIiwiYWN0aW9uIiwib25EaWRTYXZlIiwib25EaWRSZWxvYWQiLCJvbkRpZERlc3Ryb3kiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFBQTs7QUFDQTs7QUFDQTs7QUFFQTs7QUFDQTs7OztBQUVPLE1BQU1BLEtBQUssR0FBR0MsTUFBTSxDQUFDLE9BQUQsQ0FBcEI7OztBQUVRLE1BQU1DLHVCQUFOLENBQThCO0FBQzNDQyxFQUFBQSxXQUFXLENBQUNDLE1BQUQsRUFBU0MsU0FBVCxFQUFvQkMsVUFBcEIsRUFBZ0M7QUFDekMsMkJBQVMsSUFBVCxFQUFlLG1CQUFmO0FBRUEsU0FBS0YsTUFBTCxHQUFjQSxNQUFkO0FBQ0EsU0FBS0UsVUFBTCxHQUFrQkEsVUFBbEI7QUFDQSxTQUFLRCxTQUFMLEdBQWlCQSxTQUFqQjtBQUNBLFNBQUtFLGVBQUwsR0FBdUIsSUFBSUMsT0FBSixFQUF2QjtBQUNBLFNBQUtDLE9BQUwsR0FBZSxJQUFJQyxpQkFBSixFQUFmO0FBQ0EsU0FBS0MsV0FBTCxHQUFtQixJQUFJQyw2QkFBSixFQUFuQjtBQUNBLFNBQUtDLE1BQUwsR0FBYyxJQUFJQyx1QkFBSixDQUFnQixtQkFBaEIsQ0FBZDtBQUNBLFNBQUtDLE9BQUwsR0FBZSxLQUFmO0FBQ0Q7O0FBRUQsUUFBTUMsS0FBTixHQUFjO0FBQ1osVUFBTUMsWUFBWSxHQUFHQyxLQUFLLElBQUk7QUFDNUIsVUFBSSxLQUFLWixVQUFULEVBQXFCO0FBQ25CLGFBQUtPLE1BQUwsQ0FBWU0sY0FBWjtBQUNBLGFBQUtDLFNBQUwsQ0FBZSxDQUFDO0FBQUNDLFVBQUFBLE9BQU8sRUFBRXJCO0FBQVYsU0FBRCxDQUFmO0FBQ0Q7QUFDRixLQUxEOztBQU1BLFNBQUtJLE1BQUwsQ0FBWWtCLGdCQUFaLENBQTZCLE9BQTdCLEVBQXNDTCxZQUF0QztBQUNBLFNBQUtOLFdBQUwsQ0FBaUJZLEdBQWpCLENBQ0UsS0FBS2xCLFNBQUwsQ0FBZW1CLGtCQUFmLENBQWtDLEtBQUtDLGlCQUF2QyxDQURGLEVBRUUsSUFBSUMsb0JBQUosQ0FBZSxNQUFNLEtBQUt0QixNQUFMLENBQVl1QixtQkFBWixDQUFnQyxPQUFoQyxFQUF5Q1YsWUFBekMsQ0FBckIsQ0FGRjtBQUlBLFVBQU0sS0FBS1csaUNBQUwsRUFBTjtBQUNBLFNBQUtiLE9BQUwsR0FBZSxJQUFmO0FBQ0EsV0FBTyxJQUFQO0FBQ0Q7O0FBRUQsUUFBTWMsT0FBTixHQUFnQjtBQUNkLFNBQUtkLE9BQUwsR0FBZSxLQUFmO0FBQ0EsU0FBS1IsZUFBTCxHQUF1QixJQUFJQyxPQUFKLEVBQXZCO0FBQ0EsU0FBS0MsT0FBTCxDQUFhcUIsT0FBYjtBQUNBLFNBQUtuQixXQUFMLENBQWlCbUIsT0FBakI7QUFDQSxVQUFNLEtBQUtDLHNCQUFMLEVBQU47QUFDRDs7QUFFREMsRUFBQUEsU0FBUyxHQUFHO0FBQ1YsV0FBTyxLQUFLakIsT0FBWjtBQUNEOztBQUVESyxFQUFBQSxTQUFTLENBQUNhLE9BQUQsRUFBVTtBQUNqQixTQUFLeEIsT0FBTCxDQUFheUIsSUFBYixDQUFrQixZQUFsQixFQUFnQ0QsT0FBaEM7QUFDRDs7QUFFREUsRUFBQUEsc0JBQXNCLEdBQUc7QUFDdkIsU0FBSzFCLE9BQUwsQ0FBYXlCLElBQWIsQ0FBa0IsNEJBQWxCO0FBQ0Q7O0FBRURFLEVBQUFBLFdBQVcsQ0FBQ0MsUUFBRCxFQUFXO0FBQ3BCLFdBQU8sS0FBSzVCLE9BQUwsQ0FBYTZCLEVBQWIsQ0FBZ0IsWUFBaEIsRUFBOEJELFFBQTlCLENBQVA7QUFDRDs7QUFFREUsRUFBQUEsd0JBQXdCLENBQUNGLFFBQUQsRUFBVztBQUNqQyxXQUFPLEtBQUs1QixPQUFMLENBQWE2QixFQUFiLENBQWdCLDRCQUFoQixFQUE4Q0QsUUFBOUMsQ0FBUDtBQUNEOztBQUVERyxFQUFBQSxhQUFhLEdBQUc7QUFDZCxXQUFPLEtBQUtsQyxVQUFaO0FBQ0Q7O0FBRUQsUUFBTXNCLGlDQUFOLEdBQTBDO0FBQ3hDLFVBQU10QixVQUFVLEdBQUcsS0FBS2tDLGFBQUwsRUFBbkI7QUFDQSxVQUFNQyxnQkFBZ0IsR0FBR25DLFVBQVUsQ0FBQ29DLG1CQUFYLEVBQXpCO0FBRUEsVUFBTUMsbUJBQW1CLEdBQUcsQ0FBQyxRQUFELEVBQVcsT0FBWCxFQUFvQixNQUFwQixFQUE0QixZQUE1QixDQUE1QjtBQUNBLFVBQU1DLHNCQUFzQixHQUFHLENBQUMsUUFBRCxFQUFXLE9BQVgsQ0FBL0I7O0FBRUEsVUFBTUMsVUFBVSxHQUFHM0IsS0FBSyxJQUFJO0FBQzFCLFlBQU00QixFQUFFLEdBQUcsQ0FBQzVCLEtBQUssQ0FBQzZCLElBQVAsQ0FBWDs7QUFDQSxVQUFJN0IsS0FBSyxDQUFDOEIsT0FBVixFQUFtQjtBQUFFRixRQUFBQSxFQUFFLENBQUNHLElBQUgsQ0FBUS9CLEtBQUssQ0FBQzhCLE9BQWQ7QUFBeUI7O0FBQzlDLGFBQU9GLEVBQVA7QUFDRCxLQUpEOztBQU1BLFVBQU1JLFdBQVcsR0FBR2hDLEtBQUssSUFBSTtBQUMzQixhQUFPMkIsVUFBVSxDQUFDM0IsS0FBRCxDQUFWLENBQWtCaUMsSUFBbEIsQ0FBdUJDLFNBQVMsSUFBSTtBQUN6QyxlQUFPVCxtQkFBbUIsQ0FBQ1UsUUFBcEIsQ0FBNkJOLGlCQUFLTyxRQUFMLENBQWNGLFNBQWQsQ0FBN0IsS0FDTEwsaUJBQUtRLE9BQUwsQ0FBYUgsU0FBYixFQUF3QkMsUUFBeEIsQ0FBaUNOLGlCQUFLUyxJQUFMLENBQVUsTUFBVixFQUFrQixNQUFsQixDQUFqQyxDQURGO0FBRUQsT0FITSxDQUFQO0FBSUQsS0FMRDs7QUFPQSxVQUFNQyxvQkFBb0IsR0FBR3ZDLEtBQUssSUFBSTtBQUNwQyxhQUFPMkIsVUFBVSxDQUFDM0IsS0FBRCxDQUFWLENBQWtCaUMsSUFBbEIsQ0FBdUJDLFNBQVMsSUFBSVIsc0JBQXNCLENBQUNTLFFBQXZCLENBQWdDTixpQkFBS08sUUFBTCxDQUFjRixTQUFkLENBQWhDLENBQXBDLENBQVA7QUFDRCxLQUZEOztBQUlBLFNBQUtNLGtCQUFMLEdBQTBCLE1BQU0scUJBQVVqQixnQkFBVixFQUE0QixFQUE1QixFQUFnQ2tCLE1BQU0sSUFBSTtBQUN4RSxZQUFNQyxjQUFjLEdBQUdELE1BQU0sQ0FBQ0UsTUFBUCxDQUFjWCxXQUFkLENBQXZCOztBQUVBLFVBQUlVLGNBQWMsQ0FBQ0UsTUFBbkIsRUFBMkI7QUFDekIsYUFBS2pELE1BQUwsQ0FBWWtELFVBQVosQ0FBdUJILGNBQXZCO0FBQ0EsYUFBS3hDLFNBQUwsQ0FBZXdDLGNBQWY7O0FBQ0EsWUFBSUEsY0FBYyxDQUFDVCxJQUFmLENBQW9CTSxvQkFBcEIsQ0FBSixFQUErQztBQUM3QyxlQUFLNUMsTUFBTCxDQUFZbUQsdUJBQVo7QUFDQSxlQUFLN0Isc0JBQUw7QUFDRDtBQUNGO0FBQ0YsS0FYK0IsQ0FBaEM7QUFhQSxTQUFLdUIsa0JBQUwsQ0FBd0JPLFVBQXhCLENBQW1DQyxLQUFLLElBQUk7QUFDMUMsWUFBTUMsZ0JBQWdCLEdBQUc3RCxVQUFVLENBQUM4RCx1QkFBWCxFQUF6QixDQUQwQyxDQUUxQzs7QUFDQUMsTUFBQUEsT0FBTyxDQUFDQyxJQUFSLENBQWMsdUNBQXNDSCxnQkFBaUIsR0FBckUsRUFBeUVELEtBQXpFO0FBQ0EsV0FBS25DLHNCQUFMO0FBQ0QsS0FMRDtBQU9BLFNBQUtsQixNQUFMLENBQVkwRCxXQUFaLENBQXdCOUIsZ0JBQXhCLEVBQTBDLG9CQUExQztBQUNEOztBQUVEVixFQUFBQSxzQkFBc0IsR0FBRztBQUN2QixRQUFJLEtBQUsyQixrQkFBVCxFQUE2QjtBQUMzQixXQUFLQSxrQkFBTCxDQUF3QjVCLE9BQXhCO0FBQ0EsV0FBSzRCLGtCQUFMLEdBQTBCLElBQTFCO0FBQ0EsV0FBSzdDLE1BQUwsQ0FBWTJELFdBQVo7QUFDRDs7QUFDRCxXQUFPQyxPQUFPLENBQUNDLE9BQVIsRUFBUDtBQUNEOztBQUVEQyxFQUFBQSw0QkFBNEIsQ0FBQ0MsUUFBRCxFQUFXO0FBQ3JDLFVBQU10RSxVQUFVLEdBQUcsS0FBS2tDLGFBQUwsRUFBbkI7O0FBQ0EsUUFBSW9DLFFBQVEsSUFBSXRFLFVBQWhCLEVBQTRCO0FBQzFCLGFBQU9zRSxRQUFRLENBQUNDLE9BQVQsQ0FBaUJ2RSxVQUFVLENBQUM4RCx1QkFBWCxFQUFqQixNQUEyRCxDQUFDLENBQW5FO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsYUFBTyxLQUFQO0FBQ0Q7QUFDRjs7QUFFRDNDLEVBQUFBLGlCQUFpQixDQUFDcUQsTUFBRCxFQUFTO0FBQ3hCLFVBQU1DLE1BQU0sR0FBR0QsTUFBTSxDQUFDRSxTQUFQLEVBQWY7O0FBQ0EsUUFBSSxDQUFDLEtBQUt6RSxlQUFMLENBQXFCMEUsR0FBckIsQ0FBeUJGLE1BQXpCLENBQUwsRUFBdUM7QUFDckMsVUFBSUcsUUFBUSxHQUFHSCxNQUFNLENBQUNJLE9BQVAsRUFBZjs7QUFDQSxZQUFNL0QsU0FBUyxHQUFHLE1BQU07QUFDdEIsY0FBTWdFLFdBQVcsR0FBR0wsTUFBTSxDQUFDSSxPQUFQLEVBQXBCO0FBQ0EsY0FBTXhCLE1BQU0sR0FBR3lCLFdBQVcsS0FBS0YsUUFBaEIsR0FDYixDQUFDO0FBQUNHLFVBQUFBLE1BQU0sRUFBRSxVQUFUO0FBQXFCdEMsVUFBQUEsSUFBSSxFQUFFcUM7QUFBM0IsU0FBRCxDQURhLEdBRWIsQ0FBQztBQUFDQyxVQUFBQSxNQUFNLEVBQUUsU0FBVDtBQUFvQnRDLFVBQUFBLElBQUksRUFBRXFDLFdBQTFCO0FBQXVDcEMsVUFBQUEsT0FBTyxFQUFFa0M7QUFBaEQsU0FBRCxDQUZGO0FBR0FBLFFBQUFBLFFBQVEsR0FBR0UsV0FBWDtBQUNBLGFBQUt2RSxNQUFMLENBQVlrRCxVQUFaLENBQXVCSixNQUF2QjtBQUNBLGFBQUt2QyxTQUFMLENBQWV1QyxNQUFmO0FBQ0QsT0FSRDs7QUFVQSxXQUFLcEQsZUFBTCxDQUFxQmdCLEdBQXJCLENBQXlCd0QsTUFBekI7QUFDQSxZQUFNcEUsV0FBVyxHQUFHLElBQUlDLDZCQUFKLENBQ2xCbUUsTUFBTSxDQUFDTyxTQUFQLENBQWlCLE1BQU07QUFDckIsWUFBSSxLQUFLWCw0QkFBTCxDQUFrQ0ksTUFBTSxDQUFDSSxPQUFQLEVBQWxDLENBQUosRUFBeUQ7QUFDdkQvRCxVQUFBQSxTQUFTO0FBQ1Y7QUFDRixPQUpELENBRGtCLEVBTWxCMkQsTUFBTSxDQUFDUSxXQUFQLENBQW1CLE1BQU07QUFDdkIsWUFBSSxLQUFLWiw0QkFBTCxDQUFrQ0ksTUFBTSxDQUFDSSxPQUFQLEVBQWxDLENBQUosRUFBeUQ7QUFDdkQvRCxVQUFBQSxTQUFTO0FBQ1Y7QUFDRixPQUpELENBTmtCLEVBV2xCMkQsTUFBTSxDQUFDUyxZQUFQLENBQW9CLE1BQU07QUFDeEIsWUFBSSxLQUFLYiw0QkFBTCxDQUFrQ0ksTUFBTSxDQUFDSSxPQUFQLEVBQWxDLENBQUosRUFBeUQ7QUFDdkQvRCxVQUFBQSxTQUFTO0FBQ1Y7O0FBQ0RULFFBQUFBLFdBQVcsQ0FBQ21CLE9BQVo7QUFDRCxPQUxELENBWGtCLENBQXBCO0FBa0JBLFdBQUtuQixXQUFMLENBQWlCWSxHQUFqQixDQUFxQlosV0FBckI7QUFDRDtBQUNGOztBQW5LMEMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20vb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQge0NvbXBvc2l0ZURpc3Bvc2FibGUsIERpc3Bvc2FibGUsIEVtaXR0ZXJ9IGZyb20gJ2V2ZW50LWtpdCc7XG5pbXBvcnQge3dhdGNoUGF0aH0gZnJvbSAnYXRvbSc7XG5cbmltcG9ydCBFdmVudExvZ2dlciBmcm9tICcuL2V2ZW50LWxvZ2dlcic7XG5pbXBvcnQge2F1dG9iaW5kfSBmcm9tICcuLi9oZWxwZXJzJztcblxuZXhwb3J0IGNvbnN0IEZPQ1VTID0gU3ltYm9sKCdmb2N1cycpO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBXb3Jrc3BhY2VDaGFuZ2VPYnNlcnZlciB7XG4gIGNvbnN0cnVjdG9yKHdpbmRvdywgd29ya3NwYWNlLCByZXBvc2l0b3J5KSB7XG4gICAgYXV0b2JpbmQodGhpcywgJ29ic2VydmVUZXh0RWRpdG9yJyk7XG5cbiAgICB0aGlzLndpbmRvdyA9IHdpbmRvdztcbiAgICB0aGlzLnJlcG9zaXRvcnkgPSByZXBvc2l0b3J5O1xuICAgIHRoaXMud29ya3NwYWNlID0gd29ya3NwYWNlO1xuICAgIHRoaXMub2JzZXJ2ZWRCdWZmZXJzID0gbmV3IFdlYWtTZXQoKTtcbiAgICB0aGlzLmVtaXR0ZXIgPSBuZXcgRW1pdHRlcigpO1xuICAgIHRoaXMuZGlzcG9zYWJsZXMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuICAgIHRoaXMubG9nZ2VyID0gbmV3IEV2ZW50TG9nZ2VyKCd3b3Jrc3BhY2Ugd2F0Y2hlcicpO1xuICAgIHRoaXMuc3RhcnRlZCA9IGZhbHNlO1xuICB9XG5cbiAgYXN5bmMgc3RhcnQoKSB7XG4gICAgY29uc3QgZm9jdXNIYW5kbGVyID0gZXZlbnQgPT4ge1xuICAgICAgaWYgKHRoaXMucmVwb3NpdG9yeSkge1xuICAgICAgICB0aGlzLmxvZ2dlci5zaG93Rm9jdXNFdmVudCgpO1xuICAgICAgICB0aGlzLmRpZENoYW5nZShbe3NwZWNpYWw6IEZPQ1VTfV0pO1xuICAgICAgfVxuICAgIH07XG4gICAgdGhpcy53aW5kb3cuYWRkRXZlbnRMaXN0ZW5lcignZm9jdXMnLCBmb2N1c0hhbmRsZXIpO1xuICAgIHRoaXMuZGlzcG9zYWJsZXMuYWRkKFxuICAgICAgdGhpcy53b3Jrc3BhY2Uub2JzZXJ2ZVRleHRFZGl0b3JzKHRoaXMub2JzZXJ2ZVRleHRFZGl0b3IpLFxuICAgICAgbmV3IERpc3Bvc2FibGUoKCkgPT4gdGhpcy53aW5kb3cucmVtb3ZlRXZlbnRMaXN0ZW5lcignZm9jdXMnLCBmb2N1c0hhbmRsZXIpKSxcbiAgICApO1xuICAgIGF3YWl0IHRoaXMud2F0Y2hBY3RpdmVSZXBvc2l0b3J5R2l0RGlyZWN0b3J5KCk7XG4gICAgdGhpcy5zdGFydGVkID0gdHJ1ZTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIGFzeW5jIGRlc3Ryb3koKSB7XG4gICAgdGhpcy5zdGFydGVkID0gZmFsc2U7XG4gICAgdGhpcy5vYnNlcnZlZEJ1ZmZlcnMgPSBuZXcgV2Vha1NldCgpO1xuICAgIHRoaXMuZW1pdHRlci5kaXNwb3NlKCk7XG4gICAgdGhpcy5kaXNwb3NhYmxlcy5kaXNwb3NlKCk7XG4gICAgYXdhaXQgdGhpcy5zdG9wQ3VycmVudEZpbGVXYXRjaGVyKCk7XG4gIH1cblxuICBpc1N0YXJ0ZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuc3RhcnRlZDtcbiAgfVxuXG4gIGRpZENoYW5nZShwYXlsb2FkKSB7XG4gICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC1jaGFuZ2UnLCBwYXlsb2FkKTtcbiAgfVxuXG4gIGRpZENoYW5nZVdvcmtkaXJPckhlYWQoKSB7XG4gICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC1jaGFuZ2Utd29ya2Rpci1vci1oZWFkJyk7XG4gIH1cblxuICBvbkRpZENoYW5nZShjYWxsYmFjaykge1xuICAgIHJldHVybiB0aGlzLmVtaXR0ZXIub24oJ2RpZC1jaGFuZ2UnLCBjYWxsYmFjayk7XG4gIH1cblxuICBvbkRpZENoYW5nZVdvcmtkaXJPckhlYWQoY2FsbGJhY2spIHtcbiAgICByZXR1cm4gdGhpcy5lbWl0dGVyLm9uKCdkaWQtY2hhbmdlLXdvcmtkaXItb3ItaGVhZCcsIGNhbGxiYWNrKTtcbiAgfVxuXG4gIGdldFJlcG9zaXRvcnkoKSB7XG4gICAgcmV0dXJuIHRoaXMucmVwb3NpdG9yeTtcbiAgfVxuXG4gIGFzeW5jIHdhdGNoQWN0aXZlUmVwb3NpdG9yeUdpdERpcmVjdG9yeSgpIHtcbiAgICBjb25zdCByZXBvc2l0b3J5ID0gdGhpcy5nZXRSZXBvc2l0b3J5KCk7XG4gICAgY29uc3QgZ2l0RGlyZWN0b3J5UGF0aCA9IHJlcG9zaXRvcnkuZ2V0R2l0RGlyZWN0b3J5UGF0aCgpO1xuXG4gICAgY29uc3QgYmFzZW5hbWVzT2ZJbnRlcmVzdCA9IFsnY29uZmlnJywgJ2luZGV4JywgJ0hFQUQnLCAnTUVSR0VfSEVBRCddO1xuICAgIGNvbnN0IHdvcmtkaXJPckhlYWRCYXNlbmFtZXMgPSBbJ2NvbmZpZycsICdpbmRleCddO1xuXG4gICAgY29uc3QgZXZlbnRQYXRocyA9IGV2ZW50ID0+IHtcbiAgICAgIGNvbnN0IHBzID0gW2V2ZW50LnBhdGhdO1xuICAgICAgaWYgKGV2ZW50Lm9sZFBhdGgpIHsgcHMucHVzaChldmVudC5vbGRQYXRoKTsgfVxuICAgICAgcmV0dXJuIHBzO1xuICAgIH07XG5cbiAgICBjb25zdCBhY2NlcHRFdmVudCA9IGV2ZW50ID0+IHtcbiAgICAgIHJldHVybiBldmVudFBhdGhzKGV2ZW50KS5zb21lKGV2ZW50UGF0aCA9PiB7XG4gICAgICAgIHJldHVybiBiYXNlbmFtZXNPZkludGVyZXN0LmluY2x1ZGVzKHBhdGguYmFzZW5hbWUoZXZlbnRQYXRoKSkgfHxcbiAgICAgICAgICBwYXRoLmRpcm5hbWUoZXZlbnRQYXRoKS5pbmNsdWRlcyhwYXRoLmpvaW4oJy5naXQnLCAncmVmcycpKTtcbiAgICAgIH0pO1xuICAgIH07XG5cbiAgICBjb25zdCBpc1dvcmtkaXJPckhlYWRFdmVudCA9IGV2ZW50ID0+IHtcbiAgICAgIHJldHVybiBldmVudFBhdGhzKGV2ZW50KS5zb21lKGV2ZW50UGF0aCA9PiB3b3JrZGlyT3JIZWFkQmFzZW5hbWVzLmluY2x1ZGVzKHBhdGguYmFzZW5hbWUoZXZlbnRQYXRoKSkpO1xuICAgIH07XG5cbiAgICB0aGlzLmN1cnJlbnRGaWxlV2F0Y2hlciA9IGF3YWl0IHdhdGNoUGF0aChnaXREaXJlY3RvcnlQYXRoLCB7fSwgZXZlbnRzID0+IHtcbiAgICAgIGNvbnN0IGZpbHRlcmVkRXZlbnRzID0gZXZlbnRzLmZpbHRlcihhY2NlcHRFdmVudCk7XG5cbiAgICAgIGlmIChmaWx0ZXJlZEV2ZW50cy5sZW5ndGgpIHtcbiAgICAgICAgdGhpcy5sb2dnZXIuc2hvd0V2ZW50cyhmaWx0ZXJlZEV2ZW50cyk7XG4gICAgICAgIHRoaXMuZGlkQ2hhbmdlKGZpbHRlcmVkRXZlbnRzKTtcbiAgICAgICAgaWYgKGZpbHRlcmVkRXZlbnRzLnNvbWUoaXNXb3JrZGlyT3JIZWFkRXZlbnQpKSB7XG4gICAgICAgICAgdGhpcy5sb2dnZXIuc2hvd1dvcmtkaXJPckhlYWRFdmVudHMoKTtcbiAgICAgICAgICB0aGlzLmRpZENoYW5nZVdvcmtkaXJPckhlYWQoKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGhpcy5jdXJyZW50RmlsZVdhdGNoZXIub25EaWRFcnJvcihlcnJvciA9PiB7XG4gICAgICBjb25zdCB3b3JraW5nRGlyZWN0b3J5ID0gcmVwb3NpdG9yeS5nZXRXb3JraW5nRGlyZWN0b3J5UGF0aCgpO1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIGNvbnNvbGUud2FybihgRXJyb3IgaW4gV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIgaW4gJHt3b3JraW5nRGlyZWN0b3J5fTpgLCBlcnJvcik7XG4gICAgICB0aGlzLnN0b3BDdXJyZW50RmlsZVdhdGNoZXIoKTtcbiAgICB9KTtcblxuICAgIHRoaXMubG9nZ2VyLnNob3dTdGFydGVkKGdpdERpcmVjdG9yeVBhdGgsICd3b3Jrc3BhY2UgZW11bGF0ZWQnKTtcbiAgfVxuXG4gIHN0b3BDdXJyZW50RmlsZVdhdGNoZXIoKSB7XG4gICAgaWYgKHRoaXMuY3VycmVudEZpbGVXYXRjaGVyKSB7XG4gICAgICB0aGlzLmN1cnJlbnRGaWxlV2F0Y2hlci5kaXNwb3NlKCk7XG4gICAgICB0aGlzLmN1cnJlbnRGaWxlV2F0Y2hlciA9IG51bGw7XG4gICAgICB0aGlzLmxvZ2dlci5zaG93U3RvcHBlZCgpO1xuICAgIH1cbiAgICByZXR1cm4gUHJvbWlzZS5yZXNvbHZlKCk7XG4gIH1cblxuICBhY3RpdmVSZXBvc2l0b3J5Q29udGFpbnNQYXRoKGZpbGVQYXRoKSB7XG4gICAgY29uc3QgcmVwb3NpdG9yeSA9IHRoaXMuZ2V0UmVwb3NpdG9yeSgpO1xuICAgIGlmIChmaWxlUGF0aCAmJiByZXBvc2l0b3J5KSB7XG4gICAgICByZXR1cm4gZmlsZVBhdGguaW5kZXhPZihyZXBvc2l0b3J5LmdldFdvcmtpbmdEaXJlY3RvcnlQYXRoKCkpICE9PSAtMTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cbiAgfVxuXG4gIG9ic2VydmVUZXh0RWRpdG9yKGVkaXRvcikge1xuICAgIGNvbnN0IGJ1ZmZlciA9IGVkaXRvci5nZXRCdWZmZXIoKTtcbiAgICBpZiAoIXRoaXMub2JzZXJ2ZWRCdWZmZXJzLmhhcyhidWZmZXIpKSB7XG4gICAgICBsZXQgbGFzdFBhdGggPSBidWZmZXIuZ2V0UGF0aCgpO1xuICAgICAgY29uc3QgZGlkQ2hhbmdlID0gKCkgPT4ge1xuICAgICAgICBjb25zdCBjdXJyZW50UGF0aCA9IGJ1ZmZlci5nZXRQYXRoKCk7XG4gICAgICAgIGNvbnN0IGV2ZW50cyA9IGN1cnJlbnRQYXRoID09PSBsYXN0UGF0aCA/XG4gICAgICAgICAgW3thY3Rpb246ICdtb2RpZmllZCcsIHBhdGg6IGN1cnJlbnRQYXRofV0gOlxuICAgICAgICAgIFt7YWN0aW9uOiAncmVuYW1lZCcsIHBhdGg6IGN1cnJlbnRQYXRoLCBvbGRQYXRoOiBsYXN0UGF0aH1dO1xuICAgICAgICBsYXN0UGF0aCA9IGN1cnJlbnRQYXRoO1xuICAgICAgICB0aGlzLmxvZ2dlci5zaG93RXZlbnRzKGV2ZW50cyk7XG4gICAgICAgIHRoaXMuZGlkQ2hhbmdlKGV2ZW50cyk7XG4gICAgICB9O1xuXG4gICAgICB0aGlzLm9ic2VydmVkQnVmZmVycy5hZGQoYnVmZmVyKTtcbiAgICAgIGNvbnN0IGRpc3Bvc2FibGVzID0gbmV3IENvbXBvc2l0ZURpc3Bvc2FibGUoXG4gICAgICAgIGJ1ZmZlci5vbkRpZFNhdmUoKCkgPT4ge1xuICAgICAgICAgIGlmICh0aGlzLmFjdGl2ZVJlcG9zaXRvcnlDb250YWluc1BhdGgoYnVmZmVyLmdldFBhdGgoKSkpIHtcbiAgICAgICAgICAgIGRpZENoYW5nZSgpO1xuICAgICAgICAgIH1cbiAgICAgICAgfSksXG4gICAgICAgIGJ1ZmZlci5vbkRpZFJlbG9hZCgoKSA9PiB7XG4gICAgICAgICAgaWYgKHRoaXMuYWN0aXZlUmVwb3NpdG9yeUNvbnRhaW5zUGF0aChidWZmZXIuZ2V0UGF0aCgpKSkge1xuICAgICAgICAgICAgZGlkQ2hhbmdlKCk7XG4gICAgICAgICAgfVxuICAgICAgICB9KSxcbiAgICAgICAgYnVmZmVyLm9uRGlkRGVzdHJveSgoKSA9PiB7XG4gICAgICAgICAgaWYgKHRoaXMuYWN0aXZlUmVwb3NpdG9yeUNvbnRhaW5zUGF0aChidWZmZXIuZ2V0UGF0aCgpKSkge1xuICAgICAgICAgICAgZGlkQ2hhbmdlKCk7XG4gICAgICAgICAgfVxuICAgICAgICAgIGRpc3Bvc2FibGVzLmRpc3Bvc2UoKTtcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgdGhpcy5kaXNwb3NhYmxlcy5hZGQoZGlzcG9zYWJsZXMpO1xuICAgIH1cbiAgfVxufVxuIl19