"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _state = _interopRequireDefault(require("./state"));

var _gitShellOutStrategy = require("../../git-shell-out-strategy");

var _workspaceChangeObserver = require("../workspace-change-observer");

var _patch = require("../patch");

var _discardHistory = _interopRequireDefault(require("../discard-history"));

var _branch = _interopRequireWildcard(require("../branch"));

var _author = _interopRequireDefault(require("../author"));

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remote = _interopRequireDefault(require("../remote"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _commit = _interopRequireDefault(require("../commit"));

var _operationStates = _interopRequireDefault(require("../operation-states"));

var _reporterProxy = require("../../reporter-proxy");

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state["default"] {
  constructor(repository, history) {
    super(repository);
    this.cache = new Cache();
    this.discardHistory = new _discardHistory["default"](this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), {
      maxHistoryLength: 60
    });
    this.operationStates = new _operationStates["default"]({
      didUpdate: this.didUpdate.bind(this)
    });
    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();
    /* istanbul ignore else */

    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, {
    suppressUpdate
  } = {
    suppressUpdate: false
  }) {
    this.commitMessage = message;

    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      this.commitMessageTemplate = template;
    }

    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  acceptInvalidation(spec) {
    this.cache.invalidate(spec());
    this.didUpdate();
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();

    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(Keys.statusBundle);

        for (const k of Keys.filePatch.eachWithOpts({
          staged: false
        })) {
          keys.add(k);
        }

        continue;
      }

      const includes = (...segments) => fullPath.includes(_path["default"].join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(Keys.stagedChanges);
        keys.add(Keys.filePatch.all);
        keys.add(Keys.index.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(Keys.remotes);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(Keys.remotes);
        keys.add(Keys.config.all);
        keys.add(Keys.statusBundle);
        continue;
      } // File change within the working directory


      const relativePath = _path["default"].relative(this.workdir(), fullPath);

      for (const key of Keys.filePatch.eachWithFileOpts([relativePath], [{
        staged: false
      }])) {
        keys.add(key);
      }

      keys.add(Keys.statusBundle);
    }
    /* istanbul ignore else */


    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }

    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();

        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }

        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init()["catch"](e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone()["catch"](e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Staging and unstaging


  stageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, {
        index: true
      });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  } // Committing


  commit(message, options) {
    return this.invalidate(Keys.headOperationKeys, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _objectSpread2({}, options, {
        coAuthors: coAuthors.map(author => {
          return {
            email: author.getEmail(),
            name: author.getFullName()
          };
        })
      });
      await this.git().commit(message, opts); // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here

      const {
        unstagedFiles,
        mergeConflictFiles
      } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_objectSpread2({}, unstagedFiles, {}, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        "package": 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  } // Merging


  merge(branchName) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, Keys.filePatch.all, Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...Keys.filePatch.eachWithFileOpts([filePath], [{
      staged: false
    }, {
      staged: true
    }]), Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  } // Checkout


  checkout(revision, options = {}) {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.filePatch.allAgainstNonHead, Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...paths.map(fileName => Keys.index.oneWith(fileName)), ...Keys.filePatch.eachWithFileOpts(paths, [{
      staged: true
    }]), ...Keys.filePatch.eachNonHeadWithFiles(paths)], () => this.git().checkoutFiles(paths, revision));
  } // Reset


  undoLastCommit() {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', {
          "package": 'github'
        });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  } // Remote interactions


  fetch(branchName, options = {}) {
    return this.invalidate(() => [Keys.statusBundle, Keys.headDescription], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [Keys.statusBundle, Keys.headDescription];

      if (options.setUpstream) {
        keys.push(Keys.branches);
        keys.push(...Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    }, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  } // Configuration


  setConfig(setting, value, options) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options));
  }

  unsetConfig(setting) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  } // Direct blob interactions


  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  } // Discard history


  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    /* istanbul ignore else */

    if (snapshots) {
      await this.saveDiscardHistory();
    }

    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);

    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [Keys.statusBundle, ...paths.map(filePath => Keys.filePatch.oneWith(filePath, {
      staged: false
    })), ...Keys.filePatch.eachNonHeadWithFiles(paths)], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path["default"].join(this.workdir(), filePath);

        return _fsExtra["default"].remove(absPath);
      }));
    });
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Index queries


  getStatusBundle() {
    return this.cache.getOrSet(Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({
    changedEntries,
    untrackedEntries,
    renamedEntries,
    unmergedEntries
  }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };
    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};
    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }

      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });
    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });
    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }

      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });
    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const {
        stagedStatus,
        unstagedStatus,
        filePath
      } = unmergedEntries[i];

      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({
            target: 'HEAD'
          });
        }

        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  async getStatusesForChangedFiles() {
    const {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    } = await this.getStatusBundle();
    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  getFilePatchForPath(filePath, options) {
    const opts = _objectSpread2({
      staged: false,
      patchBuffer: null,
      builder: {},
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, {
      staged: opts.staged
    }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, {
        staged: opts.staged
      });
      const payload = opts.before();
      const patch = (0, _patch.buildFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  getDiffsForFilePath(filePath, baseCommit) {
    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, {
      baseCommit
    }), () => {
      return this.git().getDiffsForFilePath(filePath, {
        baseCommit
      });
    });
  }

  getStagedChangesPatch(options) {
    const opts = _objectSpread2({
      builder: {},
      patchBuffer: null,
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.stagedChanges, async () => {
      const diffs = await this.git().getStagedChangesPatch();
      const payload = opts.before();
      const patch = (0, _patch.buildMultiFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  } // Commit access


  getLastCommit() {
    return this.cache.getOrSet(Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit["default"].createUnborn() : new _commit["default"](headCommit);
    });
  }

  getCommit(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), async () => {
      const [rawCommit] = await this.git().getCommits({
        max: 1,
        ref: sha,
        includePatch: true
      });
      const commit = new _commit["default"](rawCommit);
      return commit;
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_objectSpread2({
        ref: 'HEAD'
      }, options));
      return commits.map(commit => new _commit["default"](commit));
    });
  }

  async isCommitPushed(sha) {
    const currentBranch = await this.repository.getCurrentBranch();
    const upstream = currentBranch.getPush();

    if (!upstream.isPresent()) {
      return false;
    }

    const contained = await this.git().getBranchesWithCommit(sha, {
      showLocal: false,
      showRemote: true,
      pattern: upstream.getShortRef()
    });
    return contained.some(ref => ref.length > 0);
  } // Author information


  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author["default"](email, authorMap[email]));
    });
  } // Branches


  getBranches() {
    return this.cache.getOrSet(Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet["default"]();

      for (const payload of payloads) {
        let upstream = _branch.nullBranch;

        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch["default"].createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch["default"](payload.upstream.trackingRef);
        }

        let push = upstream;

        if (payload.push) {
          push = payload.push.remoteName ? _branch["default"].createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch["default"](payload.push.trackingRef);
        }

        branches.add(new _branch["default"](payload.name, upstream, push, payload.head, {
          sha: payload.sha
        }));
      }

      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(Keys.headDescription, () => {
      return this.git().describeHead();
    });
  } // Merging and rebasing status


  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  } // Remotes


  getRemotes() {
    return this.cache.getOrSet(Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet["default"](remotesInfo.map(({
        name,
        url
      }) => new _remote["default"](name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [...Keys.config.eachWithSetting(`remote.${name}.url`), ...Keys.config.eachWithSetting(`remote.${name}.fetch`), Keys.remotes], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote["default"](name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, {
    local
  } = {
    local: false
  }) {
    return this.cache.getOrSet(Keys.config.oneWith(option, {
      local
    }), () => {
      return this.git().getConfig(option, {
        local
      });
    });
  }

  directGetConfig(key, options) {
    return this.getConfig(key, options);
  } // Direct blob access


  getBlobContents(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  directGetBlobContents(sha) {
    return this.getBlobContents(sha);
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  } // Cache

  /* istanbul ignore next */


  getCache() {
    return this.cache;
  }

  invalidate(spec, body) {
    return body().then(result => {
      this.acceptInvalidation(spec);
      return result;
    }, err => {
      this.acceptInvalidation(spec);
      return Promise.reject(err);
    });
  }

}

exports["default"] = Present;

_state["default"].register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);

    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();
    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });
    const groups = key.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);

      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }

      groupSet.add(key);
    }

    this.didUpdate();
    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage["delete"](primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet["delete"](key);
    this.didUpdate();
  }
  /* istanbul ignore next */


  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }
  /* istanbul ignore next */


  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }

}

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());
    const groups = this.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];

      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `CacheKey(${this.primary})`;
  }

}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `GroupKey(${this.group})`;
  }

}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),
  stagedChanges: new CacheKey('staged-changes'),
  filePatch: {
    _optKey: ({
      staged
    }) => staged ? 's' : 'u',
    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);

      const baseCommit = options.baseCommit || 'head';
      const extraGroups = [];

      if (options.baseCommit) {
        extraGroups.push(`file-patch:base-nonhead:path-${fileName}`);
        extraGroups.push('file-patch:base-nonhead');
      } else {
        extraGroups.push('file-patch:base-head');
      }

      return new CacheKey(`file-patch:${optKey}:${baseCommit}:${fileName}`, ['file-patch', `file-patch:opt-${optKey}`, `file-patch:opt-${optKey}:path-${fileName}`, ...extraGroups]);
    },
    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];

      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opts[j])}:path-${fileNames[i]}`));
        }
      }

      return keys;
    },
    eachNonHeadWithFiles: fileNames => {
      return fileNames.map(fileName => new GroupKey(`file-patch:base-nonhead:path-${fileName}`));
    },
    allAgainstNonHead: new GroupKey('file-patch:base-nonhead'),
    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opt)}`)),
    all: new GroupKey('file-patch')
  },
  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),
    all: new GroupKey('index')
  },
  lastCommit: new CacheKey('last-commit'),
  recentCommits: new CacheKey('recent-commits'),
  authors: new CacheKey('authors'),
  branches: new CacheKey('branches'),
  headDescription: new CacheKey('head-description'),
  remotes: new CacheKey('remotes'),
  config: {
    _optKey: options => options.local ? 'l' : '',
    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);

      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },
    eachWithSetting: setting => [Keys.config.oneWith(setting, {
      local: true
    }), Keys.config.oneWith(setting, {
      local: false
    })],
    all: new GroupKey('config')
  },
  blob: {
    oneWith: sha => new CacheKey(`blob:${sha}`, ['blob'])
  },
  // Common collections of keys and patterns for use with invalidate().
  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: false
  }])],
  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: true
  }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],
  headOperationKeys: () => [Keys.headDescription, Keys.branches, ...Keys.filePatch.eachWithOpts({
    staged: true
  }), Keys.filePatch.allAgainstNonHead, Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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