"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

class ModelObserver {
  constructor({
    fetchData,
    didUpdate
  }) {
    this.fetchData = fetchData || (() => {});

    this.didUpdate = didUpdate || (() => {});

    this.activeModel = null;
    this.activeModelData = null;
    this.activeModelUpdateSubscription = null;
    this.inProgress = false;
    this.pending = false;
  }

  setActiveModel(model) {
    if (model !== this.activeModel) {
      if (this.activeModelUpdateSubscription) {
        this.activeModelUpdateSubscription.dispose();
        this.activeModelUpdateSubscription = null;
      }

      this.activeModel = model;
      this.activeModelData = null;
      this.inProgress = false;
      this.pending = false;
      this.didUpdate(model);

      if (model) {
        this.activeModelUpdateSubscription = model.onDidUpdate(() => this.refreshModelData(model));
        return this.refreshModelData(model);
      }
    }

    return null;
  }

  refreshModelData(model = this.activeModel) {
    if (this.inProgress) {
      this.pending = true;
      return null;
    }

    this.lastModelDataRefreshPromise = this._refreshModelData(model);
    return this.lastModelDataRefreshPromise;
  }

  async _refreshModelData(model) {
    try {
      this.inProgress = true;
      const fetchDataPromise = this.fetchData(model);
      this.lastFetchDataPromise = fetchDataPromise;
      const modelData = await fetchDataPromise; // Since we re-fetch immediately when the model changes,
      // we need to ensure a fetch for an old active model
      // does not trample the newer fetch for the newer active model.

      if (fetchDataPromise === this.lastFetchDataPromise) {
        this.activeModel = model;
        this.activeModelData = modelData;
        this.didUpdate(model);
      }
    } finally {
      this.inProgress = false;

      if (this.pending) {
        this.pending = false;
        this.refreshModelData();
      }
    }
  }

  getActiveModel() {
    return this.activeModel;
  }

  getActiveModelData() {
    return this.activeModelData;
  }

  getLastModelDataRefreshPromise() {
    return this.lastModelDataRefreshPromise;
  }

  hasPendingUpdate() {
    return this.pending;
  }

  destroy() {
    if (this.activeModelUpdateSubscription) {
      this.activeModelUpdateSubscription.dispose();
    }
  }

}

exports["default"] = ModelObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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