"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _atom = require("atom");

var _path = _interopRequireDefault(require("path"));

var _eventLogger = _interopRequireDefault(require("./event-logger"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

class FileSystemChangeObserver {
  constructor(repository) {
    this.emitter = new _eventKit.Emitter();
    this.repository = repository;
    this.logger = new _eventLogger["default"]('fs watcher');
    this.started = false;
  }

  async start() {
    await this.watchRepository();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.emitter.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchRepository() {
    const allPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const isNonGitFile = event => allPaths(event).some(eventPath => !eventPath.split(_path["default"].sep).includes('.git'));

    const isWatchedGitFile = event => allPaths(event).some(eventPath => {
      return ['config', 'index', 'HEAD', 'MERGE_HEAD'].includes(_path["default"].basename(eventPath)) || _path["default"].dirname(eventPath).includes(_path["default"].join('.git', 'refs'));
    });

    const handleEvents = events => {
      const filteredEvents = events.filter(e => isNonGitFile(e) || isWatchedGitFile(e));

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);
        const workdirOrHeadEvent = filteredEvents.find(event => {
          return allPaths(event).every(eventPath => !['config', 'index'].includes(_path["default"].basename(eventPath)));
        });

        if (workdirOrHeadEvent) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(this.repository.getWorkingDirectoryPath(), {}, handleEvents);
    this.logger.showStarted(this.repository.getWorkingDirectoryPath(), 'Atom watchPath');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

}

exports["default"] = FileSystemChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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