"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _mkdirp = _interopRequireDefault(require("mkdirp"));

var _discardHistoryStores = require("./discard-history-stores");

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const emptyFilePath = _path["default"].join(_os["default"].tmpdir(), 'empty-file.txt');

const emptyFilePromise = _fsExtra["default"].writeFile(emptyFilePath, '');

class DiscardHistory {
  constructor(createBlob, expandBlobToFile, mergeFile, workdirPath, {
    maxHistoryLength
  } = {}) {
    this.createBlob = createBlob;
    this.expandBlobToFile = expandBlobToFile;
    this.mergeFile = mergeFile;
    this.workdirPath = workdirPath;
    this.partialFileHistory = new _discardHistoryStores.PartialFileDiscardHistory(maxHistoryLength);
    this.wholeFileHistory = new _discardHistoryStores.WholeFileDiscardHistory(maxHistoryLength);
  }

  getLastSnapshots(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getLastSnapshotsForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getLastSnapshots();
    }
  }

  getHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getHistory();
    }
  }

  hasHistory(partialDiscardFilePath = null) {
    const history = this.getHistory(partialDiscardFilePath);
    return history.length > 0;
  }

  popHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.popHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.popHistory();
    }
  }

  clearHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      this.partialFileHistory.clearHistoryForPath(partialDiscardFilePath);
    } else {
      this.wholeFileHistory.clearHistory();
    }
  }

  updateHistory(history) {
    this.partialFileHistory.setHistory(history.partialFileHistory || {});
    this.wholeFileHistory.setHistory(history.wholeFileHistory || []);
  }

  async createHistoryBlob() {
    const histories = {
      wholeFileHistory: this.wholeFileHistory.getHistory(),
      partialFileHistory: this.partialFileHistory.getHistory()
    };
    const historySha = await this.createBlob({
      stdin: JSON.stringify(histories)
    });
    return historySha;
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return await this.storeBlobsForPartialFileHistory(partialDiscardFilePath, isSafe, destructiveAction);
    } else {
      return await this.storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction);
    }
  }

  async storeBlobsForPartialFileHistory(filePath, isSafe, destructiveAction) {
    const beforeSha = await this.createBlob({
      filePath
    });
    const isNotSafe = !(await isSafe());

    if (isNotSafe) {
      return null;
    }

    await destructiveAction();
    const afterSha = await this.createBlob({
      filePath
    });
    const snapshots = {
      beforeSha,
      afterSha
    };
    this.partialFileHistory.addHistory(filePath, snapshots);
    return snapshots;
  }

  async storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction) {
    const snapshotsByPath = {};
    const beforePromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath] = {
        beforeSha: await this.createBlob({
          filePath
        })
      };
    });
    await Promise.all(beforePromises);
    const isNotSafe = !(await isSafe());

    if (isNotSafe) {
      return null;
    }

    await destructiveAction();
    const afterPromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath].afterSha = await this.createBlob({
        filePath
      });
    });
    await Promise.all(afterPromises);
    this.wholeFileHistory.addHistory(snapshotsByPath);
    return snapshotsByPath;
  }

  async restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    let lastDiscardSnapshots = this.getLastSnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastDiscardSnapshots = lastDiscardSnapshots ? [lastDiscardSnapshots] : [];
    }

    const tempFolderPaths = await this.expandBlobsToFilesInTempFolder(lastDiscardSnapshots);

    if (!isSafe()) {
      return [];
    }

    return await this.mergeFiles(tempFolderPaths);
  }

  async expandBlobsToFilesInTempFolder(snapshots) {
    const tempFolderPath = await (0, _helpers.getTempDir)({
      prefix: 'github-discard-history-'
    });
    const pathPromises = snapshots.map(async ({
      filePath,
      beforeSha,
      afterSha
    }) => {
      const dir = _path["default"].dirname(_path["default"].join(tempFolderPath, filePath));

      await (0, _mkdirp["default"])(dir);
      const theirsPath = !beforeSha ? null : await this.expandBlobToFile(_path["default"].join(tempFolderPath, `${filePath}-before-discard`), beforeSha);
      const commonBasePath = !afterSha ? null : await this.expandBlobToFile(_path["default"].join(tempFolderPath, `${filePath}-after-discard`), afterSha);

      const resultPath = _path["default"].join(tempFolderPath, `~${_path["default"].basename(filePath)}-merge-result`);

      return {
        filePath,
        commonBasePath,
        theirsPath,
        resultPath,
        theirsSha: beforeSha,
        commonBaseSha: afterSha
      };
    });
    return await Promise.all(pathPromises);
  }

  async mergeFiles(filePaths) {
    const mergeFilePromises = filePaths.map(async (filePathInfo, i) => {
      const {
        filePath,
        commonBasePath,
        theirsPath,
        resultPath,
        theirsSha,
        commonBaseSha
      } = filePathInfo;
      const currentSha = await this.createBlob({
        filePath
      });
      let mergeResult;

      if (theirsPath && commonBasePath) {
        mergeResult = await this.mergeFile(filePath, commonBasePath, theirsPath, resultPath);
      } else if (!theirsPath && commonBasePath) {
        // deleted file
        const oursSha = await this.createBlob({
          filePath
        });

        if (oursSha === commonBaseSha) {
          // no changes since discard, mark file to be deleted
          mergeResult = {
            filePath,
            resultPath: null,
            deleted: true,
            conflict: false
          };
        } else {
          // changes since discard result in conflict
          await _fsExtra["default"].copy(_path["default"].join(this.workdirPath, filePath), resultPath);
          mergeResult = {
            filePath,
            resultPath,
            conflict: true
          };
        }
      } else if (theirsPath && !commonBasePath) {
        // added file
        const fileDoesExist = await (0, _helpers.fileExists)(_path["default"].join(this.workdirPath, filePath));

        if (!fileDoesExist) {
          await _fsExtra["default"].copy(theirsPath, resultPath);
          mergeResult = {
            filePath,
            resultPath,
            conflict: false
          };
        } else {
          await emptyFilePromise;
          mergeResult = await this.mergeFile(filePath, emptyFilePath, theirsPath, resultPath);
        }
      } else {
        throw new Error('One of the following must be defined - theirsPath:' + `${theirsPath} or commonBasePath: ${commonBasePath}`);
      }

      return _objectSpread2({}, mergeResult, {
        theirsSha,
        commonBaseSha,
        currentSha
      });
    });
    return await Promise.all(mergeFilePromises);
  }

}

exports["default"] = DiscardHistory;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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