"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _listSelection = _interopRequireDefault(require("./list-selection"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const COPY = Symbol('COPY');

class CompositeListSelection {
  constructor(options) {
    if (options._copy !== COPY) {
      this.keysBySelection = new Map();
      this.selections = [];

      this.idForItem = options.idForItem || (item => item);

      this.resolveNextUpdatePromise = () => {};

      this.activeSelectionIndex = null;

      for (const [key, items] of options.listsByKey) {
        const selection = new _listSelection["default"]({
          items
        });
        this.keysBySelection.set(selection, key);
        this.selections.push(selection);

        if (this.activeSelectionIndex === null && selection.getItems().length) {
          this.activeSelectionIndex = this.selections.length - 1;
        }
      }

      if (this.activeSelectionIndex === null) {
        this.activeSelectionIndex = 0;
      }
    } else {
      this.keysBySelection = options.keysBySelection;
      this.selections = options.selections;
      this.idForItem = options.idForItem;
      this.activeSelectionIndex = options.activeSelectionIndex;
      this.resolveNextUpdatePromise = options.resolveNextUpdatePromise;
    }
  }

  copy(options = {}) {
    let selections = [];
    let keysBySelection = new Map();

    if (options.keysBySelection || options.selections) {
      if (!options.keysBySelection || !options.selections) {
        throw new Error('keysBySelection and selection must always be updated simultaneously');
      }

      selections = options.selections;
      keysBySelection = options.keysBySelection;
    } else {
      selections = this.selections;
      keysBySelection = this.keysBySelection;
    }

    return new CompositeListSelection({
      keysBySelection,
      selections,
      activeSelectionIndex: options.activeSelectionIndex !== undefined ? options.activeSelectionIndex : this.activeSelectionIndex,
      idForItem: options.idForItem || this.idForItem,
      resolveNextUpdatePromise: options.resolveNextUpdatePromise || this.resolveNextUpdatePromise,
      _copy: COPY
    });
  }

  updateLists(listsByKey) {
    let isDifferent = false;

    if (listsByKey.length === 0) {
      return this;
    }

    const newKeysBySelection = new Map();
    const newSelections = [];

    for (let i = 0; i < listsByKey.length; i++) {
      const [key, newItems] = listsByKey[i];
      let selection = this.selections[i];
      const oldItems = selection.getItems();

      if (!isDifferent) {
        isDifferent = oldItems.length !== newItems.length || oldItems.some((oldItem, j) => oldItem === newItems[j]);
      }

      const oldHeadItem = selection.getHeadItem();
      selection = selection.setItems(newItems);
      let newHeadItem = null;

      if (oldHeadItem) {
        newHeadItem = newItems.find(item => this.idForItem(item) === this.idForItem(oldHeadItem));
      }

      if (newHeadItem) {
        selection = selection.selectItem(newHeadItem);
      }

      newKeysBySelection.set(selection, key);
      newSelections.push(selection);
    }

    let updated = this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });

    if (updated.getActiveSelection().getItems().length === 0) {
      const next = updated.activateNextSelection();
      updated = next !== updated ? next : updated.activatePreviousSelection();
    }

    updated.resolveNextUpdatePromise();
    return updated;
  }

  updateActiveSelection(fn) {
    const oldSelection = this.getActiveSelection();
    const newSelection = fn(oldSelection);

    if (oldSelection === newSelection) {
      return this;
    }

    const key = this.keysBySelection.get(oldSelection);
    const newKeysBySelection = new Map(this.keysBySelection);
    newKeysBySelection["delete"](oldSelection);
    newKeysBySelection.set(newSelection, key);
    const newSelections = this.selections.slice();
    newSelections[this.activeSelectionIndex] = newSelection;
    return this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  selectFirstNonEmptyList() {
    return this.copy({
      activeSelectionIndex: this.selections.findIndex(selection => selection.getItems().length > 0)
    });
  }

  getActiveListKey() {
    return this.keysBySelection.get(this.getActiveSelection());
  }

  getSelectedItems() {
    return this.getActiveSelection().getSelectedItems();
  }

  getHeadItem() {
    return this.getActiveSelection().getHeadItem();
  }

  getActiveSelection() {
    return this.selections[this.activeSelectionIndex];
  }

  activateSelection(selection) {
    const index = this.selections.indexOf(selection);

    if (index === -1) {
      throw new Error('Selection not found');
    }

    return this.copy({
      activeSelectionIndex: index
    });
  }

  activateNextSelection() {
    for (let i = this.activeSelectionIndex + 1; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  activatePreviousSelection() {
    for (let i = this.activeSelectionIndex - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  activateLastSelection() {
    for (let i = this.selections.length - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({
          activeSelectionIndex: i
        });
      }
    }

    return this;
  }

  selectItem(item, preserveTail = false) {
    const selection = this.selectionForItem(item);

    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    let next = this;

    if (!preserveTail) {
      next = next.activateSelection(selection);
    }

    if (selection === next.getActiveSelection()) {
      next = next.updateActiveSelection(s => s.selectItem(item, preserveTail));
    }

    return next;
  }

  addOrSubtractSelection(item) {
    const selection = this.selectionForItem(item);

    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    if (selection === this.getActiveSelection()) {
      return this.updateActiveSelection(s => s.addOrSubtractSelection(item));
    } else {
      return this.activateSelection(selection).updateActiveSelection(s => s.selectItem(item));
    }
  }

  selectAllItems() {
    return this.updateActiveSelection(s => s.selectAllItems());
  }

  selectFirstItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectFirstItem(preserveTail));
  }

  selectLastItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectLastItem(preserveTail));
  }

  coalesce() {
    return this.updateActiveSelection(s => s.coalesce());
  }

  selectionForItem(item) {
    return this.selections.find(selection => selection.getItems().includes(item));
  }

  listKeyForItem(item) {
    return this.keysBySelection.get(this.selectionForItem(item));
  }

  selectNextItem(preserveTail = false) {
    let next = this;

    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getLastItem()) {
      next = next.activateNextSelection();

      if (next !== this) {
        return next.updateActiveSelection(s => s.selectFirstItem());
      } else {
        return next.updateActiveSelection(s => s.selectLastItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectNextItem(preserveTail));
    }
  }

  selectPreviousItem(preserveTail = false) {
    let next = this;

    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getItems()[0]) {
      next = next.activatePreviousSelection();

      if (next !== this) {
        return next.updateActiveSelection(s => s.selectLastItem());
      } else {
        return next.updateActiveSelection(s => s.selectFirstItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectPreviousItem(preserveTail));
    }
  }

  findItem(predicate) {
    for (let i = 0; i < this.selections.length; i++) {
      const selection = this.selections[i];
      const key = this.keysBySelection.get(selection);
      const found = selection.getItems().find(item => predicate(item, key));

      if (found !== undefined) {
        return found;
      }
    }

    return null;
  }

}

exports["default"] = CompositeListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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