"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullCommit = exports["default"] = void 0;

var _patch = require("./patch");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const UNBORN = Symbol('unborn'); // Truncation elipsis styles

const WORD_ELIPSES = '...';
const NEWLINE_ELIPSES = '\n...';
const PARAGRAPH_ELIPSES = '\n\n...';

class Commit {
  static createUnborn() {
    return new Commit({
      unbornRef: UNBORN
    });
  }

  constructor({
    sha,
    authorEmail,
    authorName,
    coAuthors,
    authorDate,
    messageSubject,
    messageBody,
    unbornRef,
    patch
  }) {
    this.sha = sha;
    this.authorEmail = authorEmail;
    this.authorName = authorName;
    this.coAuthors = coAuthors || [];
    this.authorDate = authorDate;
    this.messageSubject = messageSubject;
    this.messageBody = messageBody;
    this.unbornRef = unbornRef === UNBORN;
    this.multiFileDiff = patch ? (0, _patch.buildMultiFilePatch)(patch) : (0, _patch.buildMultiFilePatch)([]);
  }

  getSha() {
    return this.sha;
  }

  getAuthorEmail() {
    return this.authorEmail;
  }

  getAuthorName() {
    return this.authorName;
  }

  getAuthorDate() {
    return this.authorDate;
  }

  getCoAuthors() {
    return this.coAuthors;
  }

  getMessageSubject() {
    return this.messageSubject;
  }

  getMessageBody() {
    return this.messageBody;
  }

  isBodyLong() {
    if (this.getMessageBody().length > this.constructor.LONG_MESSAGE_THRESHOLD) {
      return true;
    }

    if ((this.getMessageBody().match(/\r?\n/g) || []).length > this.constructor.NEWLINE_THRESHOLD) {
      return true;
    }

    return false;
  }

  getFullMessage() {
    return `${this.getMessageSubject()}\n\n${this.getMessageBody()}`.trim();
  }
  /*
   * Return the messageBody truncated to at most LONG_MESSAGE_THRESHOLD characters or NEWLINE_THRESHOLD newlines,
   * whichever comes first.
   *
   * If NEWLINE_THRESHOLD newlines are encountered before LONG_MESSAGE_THRESHOLD characters, the body will be truncated
   * at the last counted newline and elipses added.
   *
   * If a paragraph boundary is found before LONG_MESSAGE_THRESHOLD characters, the message will be truncated at the end
   * of the previous paragraph and an elipses added. If no paragraph boundary is found, but a word boundary is, the text
   * is truncated at the last word boundary and an elipsis added. If neither are found, the text is truncated hard at
   * LONG_MESSAGE_THRESHOLD - 3 characters and an elipsis is added.
   */


  abbreviatedBody() {
    if (!this.isBodyLong()) {
      return this.getMessageBody();
    }

    const {
      LONG_MESSAGE_THRESHOLD,
      NEWLINE_THRESHOLD
    } = this.constructor;
    let lastNewlineCutoff = null;
    let lastParagraphCutoff = null;
    let lastWordCutoff = null;
    const searchText = this.getMessageBody().substring(0, LONG_MESSAGE_THRESHOLD);
    const boundaryRx = /\s+/g;
    let result;
    let lineCount = 0;

    while ((result = boundaryRx.exec(searchText)) !== null) {
      const newlineCount = (result[0].match(/\r?\n/g) || []).length;
      lineCount += newlineCount;

      if (lineCount > NEWLINE_THRESHOLD) {
        lastNewlineCutoff = result.index;
        break;
      }

      if (newlineCount < 2 && result.index <= LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length) {
        lastWordCutoff = result.index;
      } else if (result.index < LONG_MESSAGE_THRESHOLD - PARAGRAPH_ELIPSES.length) {
        lastParagraphCutoff = result.index;
      }
    }

    let elipses = WORD_ELIPSES;
    let cutoffIndex = LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length;

    if (lastNewlineCutoff !== null) {
      elipses = NEWLINE_ELIPSES;
      cutoffIndex = lastNewlineCutoff;
    } else if (lastParagraphCutoff !== null) {
      elipses = PARAGRAPH_ELIPSES;
      cutoffIndex = lastParagraphCutoff;
    } else if (lastWordCutoff !== null) {
      cutoffIndex = lastWordCutoff;
    }

    return this.getMessageBody().substring(0, cutoffIndex) + elipses;
  }

  setMultiFileDiff(multiFileDiff) {
    this.multiFileDiff = multiFileDiff;
  }

  getMultiFileDiff() {
    return this.multiFileDiff;
  }

  isUnbornRef() {
    return this.unbornRef;
  }

  isPresent() {
    return true;
  }

  isEqual(other) {
    // Directly comparable properties
    const properties = ['sha', 'authorEmail', 'authorDate', 'messageSubject', 'messageBody', 'unbornRef', 'authorName'];

    for (const property of properties) {
      if (this[property] !== other[property]) {
        return false;
      }
    } // Co-author array


    if (this.coAuthors.length !== other.coAuthors.length) {
      return false;
    }

    for (let i = 0; i < this.coAuthors.length; i++) {
      const thisCoAuthor = this.coAuthors[i];
      const otherCoAuthor = other.coAuthors[i];

      if (thisCoAuthor.name !== otherCoAuthor.name || thisCoAuthor.email !== otherCoAuthor.email) {
        return false;
      }
    } // Multi-file patch


    if (!this.multiFileDiff.isEqual(other.multiFileDiff)) {
      return false;
    }

    return true;
  }

}

exports["default"] = Commit;

_defineProperty(Commit, "LONG_MESSAGE_THRESHOLD", 400);

_defineProperty(Commit, "NEWLINE_THRESHOLD", 5);

const nullCommit = {
  getSha() {
    return '';
  },

  getMessageSubject() {
    return '';
  },

  isUnbornRef() {
    return false;
  },

  isPresent() {
    return false;
  },

  isBodyLong() {
    return false;
  }

};
exports.nullCommit = nullCommit;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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