"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _react = _interopRequireWildcard(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _uriPattern = _interopRequireWildcard(require("./uri-pattern"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _stubItem = _interopRequireDefault(require("../items/stub-item"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * PaneItem registers an opener with the current Atom workspace as long as this component is mounted. The opener will
 * trigger on URIs that match a specified pattern and render a subtree returned by a render prop.
 *
 * The render prop can receive three arguments:
 *
 * * itemHolder: A RefHolder. If used as the target for a ref, the referenced component will be used as the "item" of
 *   the pane item - its `getTitle()`, `getIconName()`, and other methods will be used by the pane.
 * * params: An object containing the named parameters captured by the URI pattern.
 * * uri: The exact, matched URI used to launch this item.
 *
 * render() {
 *   return (
 *     <PaneItem workspace={this.props.workspace} uriPattern='atom-github://host/{id}'>
 *       {({itemHolder, params}) => (
 *         <ItemComponent ref={itemHolder.setter} id={params.id} />
 *       )}
 *     </PaneItem>
 *   );
 * }
 */
class PaneItem extends _react["default"].Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'opener');
    const uriPattern = new _uriPattern["default"](this.props.uriPattern);
    const currentlyOpen = this.props.workspace.getPaneItems().reduce((arr, item) => {
      const element = item.getElement ? item.getElement() : null;
      const match = item.getURI ? uriPattern.matches(item.getURI()) : _uriPattern.nonURIMatch;
      const stub = item.setRealItem ? item : null;

      if (element && match.ok()) {
        const openItem = new OpenItem(match, element, stub);
        arr.push(openItem);
      }

      return arr;
    }, []);
    this.subs = new _eventKit.CompositeDisposable();
    this.state = {
      uriPattern,
      currentlyOpen
    };
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.uriPattern.getOriginal() === nextProps.uriPattern) {
      return null;
    }

    return {
      uriPattern: new _uriPattern["default"](nextProps.uriPattern)
    };
  }

  componentDidMount() {
    for (const openItem of this.state.currentlyOpen) {
      this.registerCloseListener(openItem.stubItem, openItem);
      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });

      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }
    }

    this.subs.add(this.props.workspace.addOpener(this.opener));
  }

  render() {
    return this.state.currentlyOpen.map(item => {
      return _react["default"].createElement(_react.Fragment, {
        key: item.getKey()
      }, item.renderPortal(this.props.children));
    });
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  opener(uri) {
    const m = this.state.uriPattern.matches(uri);

    if (!m.ok()) {
      return undefined;
    }

    const openItem = new OpenItem(m);

    if (this.props.className) {
      openItem.addClassName(this.props.className);
    }

    return new Promise(resolve => {
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }), () => {
        const paneItem = openItem.create({
          copy: () => this.copyOpenItem(openItem)
        });
        this.registerCloseListener(paneItem, openItem);
        resolve(paneItem);
      });
    });
  }

  copyOpenItem(openItem) {
    const m = this.state.uriPattern.matches(openItem.getURI());

    if (!m.ok()) {
      return null;
    }

    const stub = _stubItem["default"].create('generic', openItem.getStubProps(), openItem.getURI());

    const copiedItem = new OpenItem(m, stub.getElement(), stub);
    this.setState(prevState => ({
      currentlyOpen: [...prevState.currentlyOpen, copiedItem]
    }), () => {
      this.registerCloseListener(stub, copiedItem);
      copiedItem.hydrateStub({
        copy: () => this.copyOpenItem(copiedItem)
      });
    });
    return stub;
  }

  registerCloseListener(paneItem, openItem) {
    const sub = this.props.workspace.onDidDestroyPaneItem(({
      item
    }) => {
      if (item === paneItem) {
        sub.dispose();
        this.subs.remove(sub);
        this.setState(prevState => ({
          currentlyOpen: prevState.currentlyOpen.filter(each => each !== openItem)
        }));
      }
    });
    this.subs.add(sub);
  }

}
/**
 * A subtree rendered through a portal onto a detached DOM node for use as the root as a PaneItem.
 */


exports["default"] = PaneItem;

_defineProperty(PaneItem, "propTypes", {
  workspace: _propTypes["default"].object.isRequired,
  children: _propTypes["default"].func.isRequired,
  uriPattern: _propTypes["default"].string.isRequired,
  className: _propTypes["default"].string
});

class OpenItem {
  constructor(match, element = null, stub = null) {
    this.id = this.constructor.nextID;
    this.constructor.nextID++;
    this.domNode = element || document.createElement('div');
    this.domNode.tabIndex = '-1';
    this.domNode.onfocus = this.onFocus.bind(this);
    this.stubItem = stub;
    this.stubProps = stub ? stub.props : {};
    this.match = match;
    this.itemHolder = new _refHolder["default"]();
  }

  getURI() {
    return this.match.getURI();
  }

  create(extra = {}) {
    const h = this.itemHolder.isEmpty() ? null : this.itemHolder;
    return (0, _helpers.createItem)(this.domNode, h, this.match.getURI(), extra);
  }

  hydrateStub(extra = {}) {
    if (this.stubItem) {
      this.stubItem.setRealItem(this.create(extra));
      this.stubItem = null;
    }
  }

  addClassName(className) {
    this.domNode.classList.add(className);
  }

  getKey() {
    return this.id;
  }

  getStubProps() {
    const itemProps = this.itemHolder.map(item => ({
      title: item.getTitle ? item.getTitle() : null,
      iconName: item.getIconName ? item.getIconName() : null
    }));
    return _objectSpread2({}, this.stubProps, {}, itemProps);
  }

  onFocus() {
    return this.itemHolder.map(item => item.focus && item.focus());
  }

  renderPortal(renderProp) {
    return _reactDom["default"].createPortal(renderProp({
      deserialized: this.stubProps,
      itemHolder: this.itemHolder,
      params: this.match.getParams(),
      uri: this.match.getURI()
    }), this.domNode);
  }

}

_defineProperty(OpenItem, "nextID", 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInBhbmUtaXRlbS5qcyJdLCJuYW1lcyI6WyJQYW5lSXRlbSIsIlJlYWN0IiwiQ29tcG9uZW50IiwiY29uc3RydWN0b3IiLCJwcm9wcyIsInVyaVBhdHRlcm4iLCJVUklQYXR0ZXJuIiwiY3VycmVudGx5T3BlbiIsIndvcmtzcGFjZSIsImdldFBhbmVJdGVtcyIsInJlZHVjZSIsImFyciIsIml0ZW0iLCJlbGVtZW50IiwiZ2V0RWxlbWVudCIsIm1hdGNoIiwiZ2V0VVJJIiwibWF0Y2hlcyIsIm5vblVSSU1hdGNoIiwic3R1YiIsInNldFJlYWxJdGVtIiwib2siLCJvcGVuSXRlbSIsIk9wZW5JdGVtIiwicHVzaCIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwic3RhdGUiLCJnZXREZXJpdmVkU3RhdGVGcm9tUHJvcHMiLCJuZXh0UHJvcHMiLCJwcmV2U3RhdGUiLCJnZXRPcmlnaW5hbCIsImNvbXBvbmVudERpZE1vdW50IiwicmVnaXN0ZXJDbG9zZUxpc3RlbmVyIiwic3R1Ykl0ZW0iLCJoeWRyYXRlU3R1YiIsImNvcHkiLCJjb3B5T3Blbkl0ZW0iLCJjbGFzc05hbWUiLCJhZGRDbGFzc05hbWUiLCJhZGQiLCJhZGRPcGVuZXIiLCJvcGVuZXIiLCJyZW5kZXIiLCJtYXAiLCJnZXRLZXkiLCJyZW5kZXJQb3J0YWwiLCJjaGlsZHJlbiIsImNvbXBvbmVudFdpbGxVbm1vdW50IiwiZGlzcG9zZSIsInVyaSIsIm0iLCJ1bmRlZmluZWQiLCJQcm9taXNlIiwicmVzb2x2ZSIsInNldFN0YXRlIiwicGFuZUl0ZW0iLCJjcmVhdGUiLCJTdHViSXRlbSIsImdldFN0dWJQcm9wcyIsImNvcGllZEl0ZW0iLCJzdWIiLCJvbkRpZERlc3Ryb3lQYW5lSXRlbSIsInJlbW92ZSIsImZpbHRlciIsImVhY2giLCJQcm9wVHlwZXMiLCJvYmplY3QiLCJpc1JlcXVpcmVkIiwiZnVuYyIsInN0cmluZyIsImlkIiwibmV4dElEIiwiZG9tTm9kZSIsImRvY3VtZW50IiwiY3JlYXRlRWxlbWVudCIsInRhYkluZGV4Iiwib25mb2N1cyIsIm9uRm9jdXMiLCJiaW5kIiwic3R1YlByb3BzIiwiaXRlbUhvbGRlciIsIlJlZkhvbGRlciIsImV4dHJhIiwiaCIsImlzRW1wdHkiLCJjbGFzc0xpc3QiLCJpdGVtUHJvcHMiLCJ0aXRsZSIsImdldFRpdGxlIiwiaWNvbk5hbWUiLCJnZXRJY29uTmFtZSIsImZvY3VzIiwicmVuZGVyUHJvcCIsIlJlYWN0RE9NIiwiY3JlYXRlUG9ydGFsIiwiZGVzZXJpYWxpemVkIiwicGFyYW1zIiwiZ2V0UGFyYW1zIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7Ozs7Ozs7Ozs7QUFFQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBcUJlLE1BQU1BLFFBQU4sU0FBdUJDLGtCQUFNQyxTQUE3QixDQUF1QztBQVFwREMsRUFBQUEsV0FBVyxDQUFDQyxLQUFELEVBQVE7QUFDakIsVUFBTUEsS0FBTjtBQUNBLDJCQUFTLElBQVQsRUFBZSxRQUFmO0FBRUEsVUFBTUMsVUFBVSxHQUFHLElBQUlDLHNCQUFKLENBQWUsS0FBS0YsS0FBTCxDQUFXQyxVQUExQixDQUFuQjtBQUNBLFVBQU1FLGFBQWEsR0FBRyxLQUFLSCxLQUFMLENBQVdJLFNBQVgsQ0FBcUJDLFlBQXJCLEdBQ25CQyxNQURtQixDQUNaLENBQUNDLEdBQUQsRUFBTUMsSUFBTixLQUFlO0FBQ3JCLFlBQU1DLE9BQU8sR0FBR0QsSUFBSSxDQUFDRSxVQUFMLEdBQWtCRixJQUFJLENBQUNFLFVBQUwsRUFBbEIsR0FBc0MsSUFBdEQ7QUFDQSxZQUFNQyxLQUFLLEdBQUdILElBQUksQ0FBQ0ksTUFBTCxHQUFjWCxVQUFVLENBQUNZLE9BQVgsQ0FBbUJMLElBQUksQ0FBQ0ksTUFBTCxFQUFuQixDQUFkLEdBQWtERSx1QkFBaEU7QUFDQSxZQUFNQyxJQUFJLEdBQUdQLElBQUksQ0FBQ1EsV0FBTCxHQUFtQlIsSUFBbkIsR0FBMEIsSUFBdkM7O0FBRUEsVUFBSUMsT0FBTyxJQUFJRSxLQUFLLENBQUNNLEVBQU4sRUFBZixFQUEyQjtBQUN6QixjQUFNQyxRQUFRLEdBQUcsSUFBSUMsUUFBSixDQUFhUixLQUFiLEVBQW9CRixPQUFwQixFQUE2Qk0sSUFBN0IsQ0FBakI7QUFDQVIsUUFBQUEsR0FBRyxDQUFDYSxJQUFKLENBQVNGLFFBQVQ7QUFDRDs7QUFFRCxhQUFPWCxHQUFQO0FBQ0QsS0FabUIsRUFZakIsRUFaaUIsQ0FBdEI7QUFjQSxTQUFLYyxJQUFMLEdBQVksSUFBSUMsNkJBQUosRUFBWjtBQUNBLFNBQUtDLEtBQUwsR0FBYTtBQUFDdEIsTUFBQUEsVUFBRDtBQUFhRSxNQUFBQTtBQUFiLEtBQWI7QUFDRDs7QUFFRCxTQUFPcUIsd0JBQVAsQ0FBZ0NDLFNBQWhDLEVBQTJDQyxTQUEzQyxFQUFzRDtBQUNwRCxRQUFJQSxTQUFTLENBQUN6QixVQUFWLENBQXFCMEIsV0FBckIsT0FBdUNGLFNBQVMsQ0FBQ3hCLFVBQXJELEVBQWlFO0FBQy9ELGFBQU8sSUFBUDtBQUNEOztBQUVELFdBQU87QUFDTEEsTUFBQUEsVUFBVSxFQUFFLElBQUlDLHNCQUFKLENBQWV1QixTQUFTLENBQUN4QixVQUF6QjtBQURQLEtBQVA7QUFHRDs7QUFFRDJCLEVBQUFBLGlCQUFpQixHQUFHO0FBQ2xCLFNBQUssTUFBTVYsUUFBWCxJQUF1QixLQUFLSyxLQUFMLENBQVdwQixhQUFsQyxFQUFpRDtBQUMvQyxXQUFLMEIscUJBQUwsQ0FBMkJYLFFBQVEsQ0FBQ1ksUUFBcEMsRUFBOENaLFFBQTlDO0FBRUFBLE1BQUFBLFFBQVEsQ0FBQ2EsV0FBVCxDQUFxQjtBQUNuQkMsUUFBQUEsSUFBSSxFQUFFLE1BQU0sS0FBS0MsWUFBTCxDQUFrQmYsUUFBbEI7QUFETyxPQUFyQjs7QUFHQSxVQUFJLEtBQUtsQixLQUFMLENBQVdrQyxTQUFmLEVBQTBCO0FBQ3hCaEIsUUFBQUEsUUFBUSxDQUFDaUIsWUFBVCxDQUFzQixLQUFLbkMsS0FBTCxDQUFXa0MsU0FBakM7QUFDRDtBQUNGOztBQUVELFNBQUtiLElBQUwsQ0FBVWUsR0FBVixDQUFjLEtBQUtwQyxLQUFMLENBQVdJLFNBQVgsQ0FBcUJpQyxTQUFyQixDQUErQixLQUFLQyxNQUFwQyxDQUFkO0FBQ0Q7O0FBRURDLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQU8sS0FBS2hCLEtBQUwsQ0FBV3BCLGFBQVgsQ0FBeUJxQyxHQUF6QixDQUE2QmhDLElBQUksSUFBSTtBQUMxQyxhQUNFLGdDQUFDLGVBQUQ7QUFBVSxRQUFBLEdBQUcsRUFBRUEsSUFBSSxDQUFDaUMsTUFBTDtBQUFmLFNBQ0dqQyxJQUFJLENBQUNrQyxZQUFMLENBQWtCLEtBQUsxQyxLQUFMLENBQVcyQyxRQUE3QixDQURILENBREY7QUFLRCxLQU5NLENBQVA7QUFPRDs7QUFFREMsRUFBQUEsb0JBQW9CLEdBQUc7QUFDckIsU0FBS3ZCLElBQUwsQ0FBVXdCLE9BQVY7QUFDRDs7QUFFRFAsRUFBQUEsTUFBTSxDQUFDUSxHQUFELEVBQU07QUFDVixVQUFNQyxDQUFDLEdBQUcsS0FBS3hCLEtBQUwsQ0FBV3RCLFVBQVgsQ0FBc0JZLE9BQXRCLENBQThCaUMsR0FBOUIsQ0FBVjs7QUFDQSxRQUFJLENBQUNDLENBQUMsQ0FBQzlCLEVBQUYsRUFBTCxFQUFhO0FBQ1gsYUFBTytCLFNBQVA7QUFDRDs7QUFFRCxVQUFNOUIsUUFBUSxHQUFHLElBQUlDLFFBQUosQ0FBYTRCLENBQWIsQ0FBakI7O0FBQ0EsUUFBSSxLQUFLL0MsS0FBTCxDQUFXa0MsU0FBZixFQUEwQjtBQUN4QmhCLE1BQUFBLFFBQVEsQ0FBQ2lCLFlBQVQsQ0FBc0IsS0FBS25DLEtBQUwsQ0FBV2tDLFNBQWpDO0FBQ0Q7O0FBRUQsV0FBTyxJQUFJZSxPQUFKLENBQVlDLE9BQU8sSUFBSTtBQUM1QixXQUFLQyxRQUFMLENBQWN6QixTQUFTLEtBQUs7QUFDMUJ2QixRQUFBQSxhQUFhLEVBQUUsQ0FBQyxHQUFHdUIsU0FBUyxDQUFDdkIsYUFBZCxFQUE2QmUsUUFBN0I7QUFEVyxPQUFMLENBQXZCLEVBRUksTUFBTTtBQUNSLGNBQU1rQyxRQUFRLEdBQUdsQyxRQUFRLENBQUNtQyxNQUFULENBQWdCO0FBQy9CckIsVUFBQUEsSUFBSSxFQUFFLE1BQU0sS0FBS0MsWUFBTCxDQUFrQmYsUUFBbEI7QUFEbUIsU0FBaEIsQ0FBakI7QUFHQSxhQUFLVyxxQkFBTCxDQUEyQnVCLFFBQTNCLEVBQXFDbEMsUUFBckM7QUFDQWdDLFFBQUFBLE9BQU8sQ0FBQ0UsUUFBRCxDQUFQO0FBQ0QsT0FSRDtBQVNELEtBVk0sQ0FBUDtBQVdEOztBQUVEbkIsRUFBQUEsWUFBWSxDQUFDZixRQUFELEVBQVc7QUFDckIsVUFBTTZCLENBQUMsR0FBRyxLQUFLeEIsS0FBTCxDQUFXdEIsVUFBWCxDQUFzQlksT0FBdEIsQ0FBOEJLLFFBQVEsQ0FBQ04sTUFBVCxFQUE5QixDQUFWOztBQUNBLFFBQUksQ0FBQ21DLENBQUMsQ0FBQzlCLEVBQUYsRUFBTCxFQUFhO0FBQ1gsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQsVUFBTUYsSUFBSSxHQUFHdUMscUJBQVNELE1BQVQsQ0FBZ0IsU0FBaEIsRUFBMkJuQyxRQUFRLENBQUNxQyxZQUFULEVBQTNCLEVBQW9EckMsUUFBUSxDQUFDTixNQUFULEVBQXBELENBQWI7O0FBRUEsVUFBTTRDLFVBQVUsR0FBRyxJQUFJckMsUUFBSixDQUFhNEIsQ0FBYixFQUFnQmhDLElBQUksQ0FBQ0wsVUFBTCxFQUFoQixFQUFtQ0ssSUFBbkMsQ0FBbkI7QUFDQSxTQUFLb0MsUUFBTCxDQUFjekIsU0FBUyxLQUFLO0FBQzFCdkIsTUFBQUEsYUFBYSxFQUFFLENBQUMsR0FBR3VCLFNBQVMsQ0FBQ3ZCLGFBQWQsRUFBNkJxRCxVQUE3QjtBQURXLEtBQUwsQ0FBdkIsRUFFSSxNQUFNO0FBQ1IsV0FBSzNCLHFCQUFMLENBQTJCZCxJQUEzQixFQUFpQ3lDLFVBQWpDO0FBQ0FBLE1BQUFBLFVBQVUsQ0FBQ3pCLFdBQVgsQ0FBdUI7QUFDckJDLFFBQUFBLElBQUksRUFBRSxNQUFNLEtBQUtDLFlBQUwsQ0FBa0J1QixVQUFsQjtBQURTLE9BQXZCO0FBR0QsS0FQRDtBQVNBLFdBQU96QyxJQUFQO0FBQ0Q7O0FBRURjLEVBQUFBLHFCQUFxQixDQUFDdUIsUUFBRCxFQUFXbEMsUUFBWCxFQUFxQjtBQUN4QyxVQUFNdUMsR0FBRyxHQUFHLEtBQUt6RCxLQUFMLENBQVdJLFNBQVgsQ0FBcUJzRCxvQkFBckIsQ0FBMEMsQ0FBQztBQUFDbEQsTUFBQUE7QUFBRCxLQUFELEtBQVk7QUFDaEUsVUFBSUEsSUFBSSxLQUFLNEMsUUFBYixFQUF1QjtBQUNyQkssUUFBQUEsR0FBRyxDQUFDWixPQUFKO0FBQ0EsYUFBS3hCLElBQUwsQ0FBVXNDLE1BQVYsQ0FBaUJGLEdBQWpCO0FBQ0EsYUFBS04sUUFBTCxDQUFjekIsU0FBUyxLQUFLO0FBQzFCdkIsVUFBQUEsYUFBYSxFQUFFdUIsU0FBUyxDQUFDdkIsYUFBVixDQUF3QnlELE1BQXhCLENBQStCQyxJQUFJLElBQUlBLElBQUksS0FBSzNDLFFBQWhEO0FBRFcsU0FBTCxDQUF2QjtBQUdEO0FBQ0YsS0FSVyxDQUFaO0FBVUEsU0FBS0csSUFBTCxDQUFVZSxHQUFWLENBQWNxQixHQUFkO0FBQ0Q7O0FBL0htRDtBQWtJdEQ7Ozs7Ozs7Z0JBbElxQjdELFEsZUFDQTtBQUNqQlEsRUFBQUEsU0FBUyxFQUFFMEQsc0JBQVVDLE1BQVYsQ0FBaUJDLFVBRFg7QUFFakJyQixFQUFBQSxRQUFRLEVBQUVtQixzQkFBVUcsSUFBVixDQUFlRCxVQUZSO0FBR2pCL0QsRUFBQUEsVUFBVSxFQUFFNkQsc0JBQVVJLE1BQVYsQ0FBaUJGLFVBSFo7QUFJakI5QixFQUFBQSxTQUFTLEVBQUU0QixzQkFBVUk7QUFKSixDOztBQW9JckIsTUFBTS9DLFFBQU4sQ0FBZTtBQUdicEIsRUFBQUEsV0FBVyxDQUFDWSxLQUFELEVBQVFGLE9BQU8sR0FBRyxJQUFsQixFQUF3Qk0sSUFBSSxHQUFHLElBQS9CLEVBQXFDO0FBQzlDLFNBQUtvRCxFQUFMLEdBQVUsS0FBS3BFLFdBQUwsQ0FBaUJxRSxNQUEzQjtBQUNBLFNBQUtyRSxXQUFMLENBQWlCcUUsTUFBakI7QUFFQSxTQUFLQyxPQUFMLEdBQWU1RCxPQUFPLElBQUk2RCxRQUFRLENBQUNDLGFBQVQsQ0FBdUIsS0FBdkIsQ0FBMUI7QUFDQSxTQUFLRixPQUFMLENBQWFHLFFBQWIsR0FBd0IsSUFBeEI7QUFDQSxTQUFLSCxPQUFMLENBQWFJLE9BQWIsR0FBdUIsS0FBS0MsT0FBTCxDQUFhQyxJQUFiLENBQWtCLElBQWxCLENBQXZCO0FBQ0EsU0FBSzdDLFFBQUwsR0FBZ0JmLElBQWhCO0FBQ0EsU0FBSzZELFNBQUwsR0FBaUI3RCxJQUFJLEdBQUdBLElBQUksQ0FBQ2YsS0FBUixHQUFnQixFQUFyQztBQUNBLFNBQUtXLEtBQUwsR0FBYUEsS0FBYjtBQUNBLFNBQUtrRSxVQUFMLEdBQWtCLElBQUlDLHFCQUFKLEVBQWxCO0FBQ0Q7O0FBRURsRSxFQUFBQSxNQUFNLEdBQUc7QUFDUCxXQUFPLEtBQUtELEtBQUwsQ0FBV0MsTUFBWCxFQUFQO0FBQ0Q7O0FBRUR5QyxFQUFBQSxNQUFNLENBQUMwQixLQUFLLEdBQUcsRUFBVCxFQUFhO0FBQ2pCLFVBQU1DLENBQUMsR0FBRyxLQUFLSCxVQUFMLENBQWdCSSxPQUFoQixLQUE0QixJQUE1QixHQUFtQyxLQUFLSixVQUFsRDtBQUNBLFdBQU8seUJBQVcsS0FBS1IsT0FBaEIsRUFBeUJXLENBQXpCLEVBQTRCLEtBQUtyRSxLQUFMLENBQVdDLE1BQVgsRUFBNUIsRUFBaURtRSxLQUFqRCxDQUFQO0FBQ0Q7O0FBRURoRCxFQUFBQSxXQUFXLENBQUNnRCxLQUFLLEdBQUcsRUFBVCxFQUFhO0FBQ3RCLFFBQUksS0FBS2pELFFBQVQsRUFBbUI7QUFDakIsV0FBS0EsUUFBTCxDQUFjZCxXQUFkLENBQTBCLEtBQUtxQyxNQUFMLENBQVkwQixLQUFaLENBQTFCO0FBQ0EsV0FBS2pELFFBQUwsR0FBZ0IsSUFBaEI7QUFDRDtBQUNGOztBQUVESyxFQUFBQSxZQUFZLENBQUNELFNBQUQsRUFBWTtBQUN0QixTQUFLbUMsT0FBTCxDQUFhYSxTQUFiLENBQXVCOUMsR0FBdkIsQ0FBMkJGLFNBQTNCO0FBQ0Q7O0FBRURPLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQU8sS0FBSzBCLEVBQVo7QUFDRDs7QUFFRFosRUFBQUEsWUFBWSxHQUFHO0FBQ2IsVUFBTTRCLFNBQVMsR0FBRyxLQUFLTixVQUFMLENBQWdCckMsR0FBaEIsQ0FBb0JoQyxJQUFJLEtBQUs7QUFDN0M0RSxNQUFBQSxLQUFLLEVBQUU1RSxJQUFJLENBQUM2RSxRQUFMLEdBQWdCN0UsSUFBSSxDQUFDNkUsUUFBTCxFQUFoQixHQUFrQyxJQURJO0FBRTdDQyxNQUFBQSxRQUFRLEVBQUU5RSxJQUFJLENBQUMrRSxXQUFMLEdBQW1CL0UsSUFBSSxDQUFDK0UsV0FBTCxFQUFuQixHQUF3QztBQUZMLEtBQUwsQ0FBeEIsQ0FBbEI7QUFLQSw4QkFDSyxLQUFLWCxTQURWLE1BRUtPLFNBRkw7QUFJRDs7QUFFRFQsRUFBQUEsT0FBTyxHQUFHO0FBQ1IsV0FBTyxLQUFLRyxVQUFMLENBQWdCckMsR0FBaEIsQ0FBb0JoQyxJQUFJLElBQUlBLElBQUksQ0FBQ2dGLEtBQUwsSUFBY2hGLElBQUksQ0FBQ2dGLEtBQUwsRUFBMUMsQ0FBUDtBQUNEOztBQUVEOUMsRUFBQUEsWUFBWSxDQUFDK0MsVUFBRCxFQUFhO0FBQ3ZCLFdBQU9DLHFCQUFTQyxZQUFULENBQ0xGLFVBQVUsQ0FBQztBQUNURyxNQUFBQSxZQUFZLEVBQUUsS0FBS2hCLFNBRFY7QUFFVEMsTUFBQUEsVUFBVSxFQUFFLEtBQUtBLFVBRlI7QUFHVGdCLE1BQUFBLE1BQU0sRUFBRSxLQUFLbEYsS0FBTCxDQUFXbUYsU0FBWCxFQUhDO0FBSVRoRCxNQUFBQSxHQUFHLEVBQUUsS0FBS25DLEtBQUwsQ0FBV0MsTUFBWDtBQUpJLEtBQUQsQ0FETCxFQU9MLEtBQUt5RCxPQVBBLENBQVA7QUFTRDs7QUFsRVk7O2dCQUFUbEQsUSxZQUNZLEMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20vb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IFJlYWN0LCB7RnJhZ21lbnR9IGZyb20gJ3JlYWN0JztcbmltcG9ydCBSZWFjdERPTSBmcm9tICdyZWFjdC1kb20nO1xuaW1wb3J0IFByb3BUeXBlcyBmcm9tICdwcm9wLXR5cGVzJztcbmltcG9ydCB7Q29tcG9zaXRlRGlzcG9zYWJsZX0gZnJvbSAnZXZlbnQta2l0JztcblxuaW1wb3J0IFVSSVBhdHRlcm4sIHtub25VUklNYXRjaH0gZnJvbSAnLi91cmktcGF0dGVybic7XG5pbXBvcnQgUmVmSG9sZGVyIGZyb20gJy4uL21vZGVscy9yZWYtaG9sZGVyJztcbmltcG9ydCBTdHViSXRlbSBmcm9tICcuLi9pdGVtcy9zdHViLWl0ZW0nO1xuaW1wb3J0IHtjcmVhdGVJdGVtLCBhdXRvYmluZH0gZnJvbSAnLi4vaGVscGVycyc7XG5cbi8qKlxuICogUGFuZUl0ZW0gcmVnaXN0ZXJzIGFuIG9wZW5lciB3aXRoIHRoZSBjdXJyZW50IEF0b20gd29ya3NwYWNlIGFzIGxvbmcgYXMgdGhpcyBjb21wb25lbnQgaXMgbW91bnRlZC4gVGhlIG9wZW5lciB3aWxsXG4gKiB0cmlnZ2VyIG9uIFVSSXMgdGhhdCBtYXRjaCBhIHNwZWNpZmllZCBwYXR0ZXJuIGFuZCByZW5kZXIgYSBzdWJ0cmVlIHJldHVybmVkIGJ5IGEgcmVuZGVyIHByb3AuXG4gKlxuICogVGhlIHJlbmRlciBwcm9wIGNhbiByZWNlaXZlIHRocmVlIGFyZ3VtZW50czpcbiAqXG4gKiAqIGl0ZW1Ib2xkZXI6IEEgUmVmSG9sZGVyLiBJZiB1c2VkIGFzIHRoZSB0YXJnZXQgZm9yIGEgcmVmLCB0aGUgcmVmZXJlbmNlZCBjb21wb25lbnQgd2lsbCBiZSB1c2VkIGFzIHRoZSBcIml0ZW1cIiBvZlxuICogICB0aGUgcGFuZSBpdGVtIC0gaXRzIGBnZXRUaXRsZSgpYCwgYGdldEljb25OYW1lKClgLCBhbmQgb3RoZXIgbWV0aG9kcyB3aWxsIGJlIHVzZWQgYnkgdGhlIHBhbmUuXG4gKiAqIHBhcmFtczogQW4gb2JqZWN0IGNvbnRhaW5pbmcgdGhlIG5hbWVkIHBhcmFtZXRlcnMgY2FwdHVyZWQgYnkgdGhlIFVSSSBwYXR0ZXJuLlxuICogKiB1cmk6IFRoZSBleGFjdCwgbWF0Y2hlZCBVUkkgdXNlZCB0byBsYXVuY2ggdGhpcyBpdGVtLlxuICpcbiAqIHJlbmRlcigpIHtcbiAqICAgcmV0dXJuIChcbiAqICAgICA8UGFuZUl0ZW0gd29ya3NwYWNlPXt0aGlzLnByb3BzLndvcmtzcGFjZX0gdXJpUGF0dGVybj0nYXRvbS1naXRodWI6Ly9ob3N0L3tpZH0nPlxuICogICAgICAgeyh7aXRlbUhvbGRlciwgcGFyYW1zfSkgPT4gKFxuICogICAgICAgICA8SXRlbUNvbXBvbmVudCByZWY9e2l0ZW1Ib2xkZXIuc2V0dGVyfSBpZD17cGFyYW1zLmlkfSAvPlxuICogICAgICAgKX1cbiAqICAgICA8L1BhbmVJdGVtPlxuICogICApO1xuICogfVxuICovXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBQYW5lSXRlbSBleHRlbmRzIFJlYWN0LkNvbXBvbmVudCB7XG4gIHN0YXRpYyBwcm9wVHlwZXMgPSB7XG4gICAgd29ya3NwYWNlOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgY2hpbGRyZW46IFByb3BUeXBlcy5mdW5jLmlzUmVxdWlyZWQsXG4gICAgdXJpUGF0dGVybjogUHJvcFR5cGVzLnN0cmluZy5pc1JlcXVpcmVkLFxuICAgIGNsYXNzTmFtZTogUHJvcFR5cGVzLnN0cmluZyxcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICAgIGF1dG9iaW5kKHRoaXMsICdvcGVuZXInKTtcblxuICAgIGNvbnN0IHVyaVBhdHRlcm4gPSBuZXcgVVJJUGF0dGVybih0aGlzLnByb3BzLnVyaVBhdHRlcm4pO1xuICAgIGNvbnN0IGN1cnJlbnRseU9wZW4gPSB0aGlzLnByb3BzLndvcmtzcGFjZS5nZXRQYW5lSXRlbXMoKVxuICAgICAgLnJlZHVjZSgoYXJyLCBpdGVtKSA9PiB7XG4gICAgICAgIGNvbnN0IGVsZW1lbnQgPSBpdGVtLmdldEVsZW1lbnQgPyBpdGVtLmdldEVsZW1lbnQoKSA6IG51bGw7XG4gICAgICAgIGNvbnN0IG1hdGNoID0gaXRlbS5nZXRVUkkgPyB1cmlQYXR0ZXJuLm1hdGNoZXMoaXRlbS5nZXRVUkkoKSkgOiBub25VUklNYXRjaDtcbiAgICAgICAgY29uc3Qgc3R1YiA9IGl0ZW0uc2V0UmVhbEl0ZW0gPyBpdGVtIDogbnVsbDtcblxuICAgICAgICBpZiAoZWxlbWVudCAmJiBtYXRjaC5vaygpKSB7XG4gICAgICAgICAgY29uc3Qgb3Blbkl0ZW0gPSBuZXcgT3Blbkl0ZW0obWF0Y2gsIGVsZW1lbnQsIHN0dWIpO1xuICAgICAgICAgIGFyci5wdXNoKG9wZW5JdGVtKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBhcnI7XG4gICAgICB9LCBbXSk7XG5cbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuICAgIHRoaXMuc3RhdGUgPSB7dXJpUGF0dGVybiwgY3VycmVudGx5T3Blbn07XG4gIH1cblxuICBzdGF0aWMgZ2V0RGVyaXZlZFN0YXRlRnJvbVByb3BzKG5leHRQcm9wcywgcHJldlN0YXRlKSB7XG4gICAgaWYgKHByZXZTdGF0ZS51cmlQYXR0ZXJuLmdldE9yaWdpbmFsKCkgPT09IG5leHRQcm9wcy51cmlQYXR0ZXJuKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgdXJpUGF0dGVybjogbmV3IFVSSVBhdHRlcm4obmV4dFByb3BzLnVyaVBhdHRlcm4pLFxuICAgIH07XG4gIH1cblxuICBjb21wb25lbnREaWRNb3VudCgpIHtcbiAgICBmb3IgKGNvbnN0IG9wZW5JdGVtIG9mIHRoaXMuc3RhdGUuY3VycmVudGx5T3Blbikge1xuICAgICAgdGhpcy5yZWdpc3RlckNsb3NlTGlzdGVuZXIob3Blbkl0ZW0uc3R1Ykl0ZW0sIG9wZW5JdGVtKTtcblxuICAgICAgb3Blbkl0ZW0uaHlkcmF0ZVN0dWIoe1xuICAgICAgICBjb3B5OiAoKSA9PiB0aGlzLmNvcHlPcGVuSXRlbShvcGVuSXRlbSksXG4gICAgICB9KTtcbiAgICAgIGlmICh0aGlzLnByb3BzLmNsYXNzTmFtZSkge1xuICAgICAgICBvcGVuSXRlbS5hZGRDbGFzc05hbWUodGhpcy5wcm9wcy5jbGFzc05hbWUpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMuc3Vicy5hZGQodGhpcy5wcm9wcy53b3Jrc3BhY2UuYWRkT3BlbmVyKHRoaXMub3BlbmVyKSk7XG4gIH1cblxuICByZW5kZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuc3RhdGUuY3VycmVudGx5T3Blbi5tYXAoaXRlbSA9PiB7XG4gICAgICByZXR1cm4gKFxuICAgICAgICA8RnJhZ21lbnQga2V5PXtpdGVtLmdldEtleSgpfT5cbiAgICAgICAgICB7aXRlbS5yZW5kZXJQb3J0YWwodGhpcy5wcm9wcy5jaGlsZHJlbil9XG4gICAgICAgIDwvRnJhZ21lbnQ+XG4gICAgICApO1xuICAgIH0pO1xuICB9XG5cbiAgY29tcG9uZW50V2lsbFVubW91bnQoKSB7XG4gICAgdGhpcy5zdWJzLmRpc3Bvc2UoKTtcbiAgfVxuXG4gIG9wZW5lcih1cmkpIHtcbiAgICBjb25zdCBtID0gdGhpcy5zdGF0ZS51cmlQYXR0ZXJuLm1hdGNoZXModXJpKTtcbiAgICBpZiAoIW0ub2soKSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCBvcGVuSXRlbSA9IG5ldyBPcGVuSXRlbShtKTtcbiAgICBpZiAodGhpcy5wcm9wcy5jbGFzc05hbWUpIHtcbiAgICAgIG9wZW5JdGVtLmFkZENsYXNzTmFtZSh0aGlzLnByb3BzLmNsYXNzTmFtZSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKHJlc29sdmUgPT4ge1xuICAgICAgdGhpcy5zZXRTdGF0ZShwcmV2U3RhdGUgPT4gKHtcbiAgICAgICAgY3VycmVudGx5T3BlbjogWy4uLnByZXZTdGF0ZS5jdXJyZW50bHlPcGVuLCBvcGVuSXRlbV0sXG4gICAgICB9KSwgKCkgPT4ge1xuICAgICAgICBjb25zdCBwYW5lSXRlbSA9IG9wZW5JdGVtLmNyZWF0ZSh7XG4gICAgICAgICAgY29weTogKCkgPT4gdGhpcy5jb3B5T3Blbkl0ZW0ob3Blbkl0ZW0pLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5yZWdpc3RlckNsb3NlTGlzdGVuZXIocGFuZUl0ZW0sIG9wZW5JdGVtKTtcbiAgICAgICAgcmVzb2x2ZShwYW5lSXRlbSk7XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxuXG4gIGNvcHlPcGVuSXRlbShvcGVuSXRlbSkge1xuICAgIGNvbnN0IG0gPSB0aGlzLnN0YXRlLnVyaVBhdHRlcm4ubWF0Y2hlcyhvcGVuSXRlbS5nZXRVUkkoKSk7XG4gICAgaWYgKCFtLm9rKCkpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cblxuICAgIGNvbnN0IHN0dWIgPSBTdHViSXRlbS5jcmVhdGUoJ2dlbmVyaWMnLCBvcGVuSXRlbS5nZXRTdHViUHJvcHMoKSwgb3Blbkl0ZW0uZ2V0VVJJKCkpO1xuXG4gICAgY29uc3QgY29waWVkSXRlbSA9IG5ldyBPcGVuSXRlbShtLCBzdHViLmdldEVsZW1lbnQoKSwgc3R1Yik7XG4gICAgdGhpcy5zZXRTdGF0ZShwcmV2U3RhdGUgPT4gKHtcbiAgICAgIGN1cnJlbnRseU9wZW46IFsuLi5wcmV2U3RhdGUuY3VycmVudGx5T3BlbiwgY29waWVkSXRlbV0sXG4gICAgfSksICgpID0+IHtcbiAgICAgIHRoaXMucmVnaXN0ZXJDbG9zZUxpc3RlbmVyKHN0dWIsIGNvcGllZEl0ZW0pO1xuICAgICAgY29waWVkSXRlbS5oeWRyYXRlU3R1Yih7XG4gICAgICAgIGNvcHk6ICgpID0+IHRoaXMuY29weU9wZW5JdGVtKGNvcGllZEl0ZW0pLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICByZXR1cm4gc3R1YjtcbiAgfVxuXG4gIHJlZ2lzdGVyQ2xvc2VMaXN0ZW5lcihwYW5lSXRlbSwgb3Blbkl0ZW0pIHtcbiAgICBjb25zdCBzdWIgPSB0aGlzLnByb3BzLndvcmtzcGFjZS5vbkRpZERlc3Ryb3lQYW5lSXRlbSgoe2l0ZW19KSA9PiB7XG4gICAgICBpZiAoaXRlbSA9PT0gcGFuZUl0ZW0pIHtcbiAgICAgICAgc3ViLmRpc3Bvc2UoKTtcbiAgICAgICAgdGhpcy5zdWJzLnJlbW92ZShzdWIpO1xuICAgICAgICB0aGlzLnNldFN0YXRlKHByZXZTdGF0ZSA9PiAoe1xuICAgICAgICAgIGN1cnJlbnRseU9wZW46IHByZXZTdGF0ZS5jdXJyZW50bHlPcGVuLmZpbHRlcihlYWNoID0+IGVhY2ggIT09IG9wZW5JdGVtKSxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGhpcy5zdWJzLmFkZChzdWIpO1xuICB9XG59XG5cbi8qKlxuICogQSBzdWJ0cmVlIHJlbmRlcmVkIHRocm91Z2ggYSBwb3J0YWwgb250byBhIGRldGFjaGVkIERPTSBub2RlIGZvciB1c2UgYXMgdGhlIHJvb3QgYXMgYSBQYW5lSXRlbS5cbiAqL1xuY2xhc3MgT3Blbkl0ZW0ge1xuICBzdGF0aWMgbmV4dElEID0gMFxuXG4gIGNvbnN0cnVjdG9yKG1hdGNoLCBlbGVtZW50ID0gbnVsbCwgc3R1YiA9IG51bGwpIHtcbiAgICB0aGlzLmlkID0gdGhpcy5jb25zdHJ1Y3Rvci5uZXh0SUQ7XG4gICAgdGhpcy5jb25zdHJ1Y3Rvci5uZXh0SUQrKztcblxuICAgIHRoaXMuZG9tTm9kZSA9IGVsZW1lbnQgfHwgZG9jdW1lbnQuY3JlYXRlRWxlbWVudCgnZGl2Jyk7XG4gICAgdGhpcy5kb21Ob2RlLnRhYkluZGV4ID0gJy0xJztcbiAgICB0aGlzLmRvbU5vZGUub25mb2N1cyA9IHRoaXMub25Gb2N1cy5iaW5kKHRoaXMpO1xuICAgIHRoaXMuc3R1Ykl0ZW0gPSBzdHViO1xuICAgIHRoaXMuc3R1YlByb3BzID0gc3R1YiA/IHN0dWIucHJvcHMgOiB7fTtcbiAgICB0aGlzLm1hdGNoID0gbWF0Y2g7XG4gICAgdGhpcy5pdGVtSG9sZGVyID0gbmV3IFJlZkhvbGRlcigpO1xuICB9XG5cbiAgZ2V0VVJJKCkge1xuICAgIHJldHVybiB0aGlzLm1hdGNoLmdldFVSSSgpO1xuICB9XG5cbiAgY3JlYXRlKGV4dHJhID0ge30pIHtcbiAgICBjb25zdCBoID0gdGhpcy5pdGVtSG9sZGVyLmlzRW1wdHkoKSA/IG51bGwgOiB0aGlzLml0ZW1Ib2xkZXI7XG4gICAgcmV0dXJuIGNyZWF0ZUl0ZW0odGhpcy5kb21Ob2RlLCBoLCB0aGlzLm1hdGNoLmdldFVSSSgpLCBleHRyYSk7XG4gIH1cblxuICBoeWRyYXRlU3R1YihleHRyYSA9IHt9KSB7XG4gICAgaWYgKHRoaXMuc3R1Ykl0ZW0pIHtcbiAgICAgIHRoaXMuc3R1Ykl0ZW0uc2V0UmVhbEl0ZW0odGhpcy5jcmVhdGUoZXh0cmEpKTtcbiAgICAgIHRoaXMuc3R1Ykl0ZW0gPSBudWxsO1xuICAgIH1cbiAgfVxuXG4gIGFkZENsYXNzTmFtZShjbGFzc05hbWUpIHtcbiAgICB0aGlzLmRvbU5vZGUuY2xhc3NMaXN0LmFkZChjbGFzc05hbWUpO1xuICB9XG5cbiAgZ2V0S2V5KCkge1xuICAgIHJldHVybiB0aGlzLmlkO1xuICB9XG5cbiAgZ2V0U3R1YlByb3BzKCkge1xuICAgIGNvbnN0IGl0ZW1Qcm9wcyA9IHRoaXMuaXRlbUhvbGRlci5tYXAoaXRlbSA9PiAoe1xuICAgICAgdGl0bGU6IGl0ZW0uZ2V0VGl0bGUgPyBpdGVtLmdldFRpdGxlKCkgOiBudWxsLFxuICAgICAgaWNvbk5hbWU6IGl0ZW0uZ2V0SWNvbk5hbWUgPyBpdGVtLmdldEljb25OYW1lKCkgOiBudWxsLFxuICAgIH0pKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi50aGlzLnN0dWJQcm9wcyxcbiAgICAgIC4uLml0ZW1Qcm9wcyxcbiAgICB9O1xuICB9XG5cbiAgb25Gb2N1cygpIHtcbiAgICByZXR1cm4gdGhpcy5pdGVtSG9sZGVyLm1hcChpdGVtID0+IGl0ZW0uZm9jdXMgJiYgaXRlbS5mb2N1cygpKTtcbiAgfVxuXG4gIHJlbmRlclBvcnRhbChyZW5kZXJQcm9wKSB7XG4gICAgcmV0dXJuIFJlYWN0RE9NLmNyZWF0ZVBvcnRhbChcbiAgICAgIHJlbmRlclByb3Aoe1xuICAgICAgICBkZXNlcmlhbGl6ZWQ6IHRoaXMuc3R1YlByb3BzLFxuICAgICAgICBpdGVtSG9sZGVyOiB0aGlzLml0ZW1Ib2xkZXIsXG4gICAgICAgIHBhcmFtczogdGhpcy5tYXRjaC5nZXRQYXJhbXMoKSxcbiAgICAgICAgdXJpOiB0aGlzLm1hdGNoLmdldFVSSSgpLFxuICAgICAgfSksXG4gICAgICB0aGlzLmRvbU5vZGUsXG4gICAgKTtcbiAgfVxufVxuIl19