(function() {
  var CSON, Disposable, MenuHelpers, MenuManager, _, fs, ipcRenderer, path, platformMenu, ref, ref1;

  path = require('path');

  _ = require('underscore-plus');

  ipcRenderer = require('electron').ipcRenderer;

  CSON = require('season');

  fs = require('fs-plus');

  Disposable = require('event-kit').Disposable;

  MenuHelpers = require('./menu-helpers');

  platformMenu = (ref = require('../package.json')) != null ? (ref1 = ref._atomMenu) != null ? ref1.menu : void 0 : void 0;

  module.exports = MenuManager = (function() {
    function MenuManager(arg) {
      this.resourcePath = arg.resourcePath, this.keymapManager = arg.keymapManager, this.packageManager = arg.packageManager;
      this.initialized = false;
      this.pendingUpdateOperation = null;
      this.template = [];
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
      this.packageManager.onDidActivateInitialPackages((function(_this) {
        return function() {
          return _this.sortPackagesMenu();
        };
      })(this));
    }

    MenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath;
      this.keymapManager.onDidReloadKeymap((function(_this) {
        return function() {
          return _this.update();
        };
      })(this));
      this.update();
      return this.initialized = true;
    };

    MenuManager.prototype.add = function(items) {
      var i, item, len;
      items = _.deepClone(items);
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        if (item.label == null) {
          continue;
        }
        this.merge(this.template, item);
      }
      this.update();
      return new Disposable((function(_this) {
        return function() {
          return _this.remove(items);
        };
      })(this));
    };

    MenuManager.prototype.remove = function(items) {
      var i, item, len;
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.unmerge(this.template, item);
      }
      return this.update();
    };

    MenuManager.prototype.clear = function() {
      this.template = [];
      return this.update();
    };

    MenuManager.prototype.includeSelector = function(selector) {
      var element, error, ref2, ref3, testBody, testDocument, testWorkspace, workspaceClasses;
      try {
        if (document.body.webkitMatchesSelector(selector)) {
          return true;
        }
      } catch (error1) {
        error = error1;
        return false;
      }
      if (this.testEditor == null) {
        testDocument = document.implementation.createDocument(document.namespaceURI, 'html');
        testBody = testDocument.createElement('body');
        (ref2 = testBody.classList).add.apply(ref2, this.classesForElement(document.body));
        testWorkspace = testDocument.createElement('atom-workspace');
        workspaceClasses = this.classesForElement(document.body.querySelector('atom-workspace'));
        if (workspaceClasses.length === 0) {
          workspaceClasses = ['workspace'];
        }
        (ref3 = testWorkspace.classList).add.apply(ref3, workspaceClasses);
        testBody.appendChild(testWorkspace);
        this.testEditor = testDocument.createElement('atom-text-editor');
        this.testEditor.classList.add('editor');
        testWorkspace.appendChild(this.testEditor);
      }
      element = this.testEditor;
      while (element) {
        if (element.webkitMatchesSelector(selector)) {
          return true;
        }
        element = element.parentElement;
      }
      return false;
    };

    MenuManager.prototype.update = function() {
      if (!this.initialized) {
        return;
      }
      if (this.pendingUpdateOperation != null) {
        clearTimeout(this.pendingUpdateOperation);
      }
      return this.pendingUpdateOperation = setTimeout((function(_this) {
        return function() {
          var binding, i, j, keystrokesByCommand, len, len1, name, ref2, ref3, unsetKeystrokes;
          unsetKeystrokes = new Set;
          ref2 = _this.keymapManager.getKeyBindings();
          for (i = 0, len = ref2.length; i < len; i++) {
            binding = ref2[i];
            if (binding.command === 'unset!') {
              unsetKeystrokes.add(binding.keystrokes);
            }
          }
          keystrokesByCommand = {};
          ref3 = _this.keymapManager.getKeyBindings();
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            binding = ref3[j];
            if (!_this.includeSelector(binding.selector)) {
              continue;
            }
            if (unsetKeystrokes.has(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'darwin' && /^alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'win32' && /^ctrl-alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (keystrokesByCommand[name = binding.command] == null) {
              keystrokesByCommand[name] = [];
            }
            keystrokesByCommand[binding.command].unshift(binding.keystrokes);
          }
          return _this.sendToBrowserProcess(_this.template, keystrokesByCommand);
        };
      })(this), 1);
    };

    MenuManager.prototype.loadPlatformItems = function() {
      var menu, menusDirPath, platformMenuPath;
      if (platformMenu != null) {
        return this.add(platformMenu);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        menu = CSON.readFileSync(platformMenuPath).menu;
        return this.add(menu);
      }
    };

    MenuManager.prototype.merge = function(menu, item) {
      return MenuHelpers.merge(menu, item);
    };

    MenuManager.prototype.unmerge = function(menu, item) {
      return MenuHelpers.unmerge(menu, item);
    };

    MenuManager.prototype.sendToBrowserProcess = function(template, keystrokesByCommand) {
      return ipcRenderer.send('update-application-menu', template, keystrokesByCommand);
    };

    MenuManager.prototype.classesForElement = function(element) {
      var classList;
      if (classList = element != null ? element.classList : void 0) {
        return Array.prototype.slice.apply(classList);
      } else {
        return [];
      }
    };

    MenuManager.prototype.sortPackagesMenu = function() {
      var packagesMenu;
      packagesMenu = _.find(this.template, function(arg) {
        var label;
        label = arg.label;
        return MenuHelpers.normalizeLabel(label) === 'Packages';
      });
      if ((packagesMenu != null ? packagesMenu.submenu : void 0) == null) {
        return;
      }
      packagesMenu.submenu.sort(function(item1, item2) {
        if (item1.label && item2.label) {
          return MenuHelpers.normalizeLabel(item1.label).localeCompare(MenuHelpers.normalizeLabel(item2.label));
        } else {
          return 0;
        }
      });
      return this.update();
    };

    return MenuManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
