(function() {
  var CompositeDisposable, CorrectionsView, SpellCheckTask, SpellCheckView, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SpellCheckTask = require('./spell-check-task');

  CorrectionsView = null;

  module.exports = SpellCheckView = (function() {
    function SpellCheckView(editor, spellCheckModule, manager) {
      this.editor = editor;
      this.spellCheckModule = spellCheckModule;
      this.manager = manager;
      this.makeCorrection = bind(this.makeCorrection, this);
      this.addContextMenuEntries = bind(this.addContextMenuEntries, this);
      this.disposables = new CompositeDisposable;
      this.initializeMarkerLayer();
      this.taskWrapper = new SpellCheckTask(this.manager);
      this.correctMisspellingCommand = atom.commands.add(atom.views.getView(this.editor), 'spell-check:correct-misspelling', (function(_this) {
        return function() {
          var marker, ref;
          if (marker = _this.markerLayer.findMarkers({
            containsBufferPosition: _this.editor.getCursorBufferPosition()
          })[0]) {
            if (CorrectionsView == null) {
              CorrectionsView = require('./corrections-view');
            }
            if ((ref = _this.correctionsView) != null) {
              ref.destroy();
            }
            _this.correctionsView = new CorrectionsView(_this.editor, _this.getCorrections(marker), marker, _this, _this.updateMisspellings);
            return _this.correctionsView.attach();
          }
        };
      })(this));
      atom.views.getView(this.editor).addEventListener('contextmenu', this.addContextMenuEntries);
      this.disposables.add(this.editor.onDidChangePath((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(this.editor.onDidChangeGrammar((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('editor.fontSize', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('spell-check.grammars', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.subscribeToBuffer();
      this.disposables.add(this.editor.onDidDestroy(this.destroy.bind(this)));
    }

    SpellCheckView.prototype.initializeMarkerLayer = function() {
      this.markerLayer = this.editor.addMarkerLayer({
        maintainHistory: false
      });
      return this.markerLayerDecoration = this.editor.decorateMarkerLayer(this.markerLayer, {
        type: 'highlight',
        "class": 'spell-check-misspelling',
        deprecatedRegionClass: 'misspelling'
      });
    };

    SpellCheckView.prototype.destroy = function() {
      var ref;
      this.unsubscribeFromBuffer();
      this.disposables.dispose();
      this.taskWrapper.terminate();
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      this.correctMisspellingCommand.dispose();
      if ((ref = this.correctionsView) != null) {
        ref.destroy();
      }
      return this.clearContextMenuEntries();
    };

    SpellCheckView.prototype.unsubscribeFromBuffer = function() {
      this.destroyMarkers();
      if (this.buffer != null) {
        this.bufferDisposable.dispose();
        return this.buffer = null;
      }
    };

    SpellCheckView.prototype.subscribeToBuffer = function() {
      this.unsubscribeFromBuffer();
      if (this.spellCheckCurrentGrammar()) {
        this.buffer = this.editor.getBuffer();
        this.bufferDisposable = new CompositeDisposable(this.buffer.onDidStopChanging((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this), this.editor.onDidTokenize((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this))));
        return this.updateMisspellings();
      }
    };

    SpellCheckView.prototype.spellCheckCurrentGrammar = function() {
      var grammar;
      grammar = this.editor.getGrammar().scopeName;
      return _.contains(atom.config.get('spell-check.grammars'), grammar);
    };

    SpellCheckView.prototype.destroyMarkers = function() {
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      return this.initializeMarkerLayer();
    };

    SpellCheckView.prototype.addMarkers = function(misspellings) {
      var i, len, misspelling, results, scope;
      results = [];
      for (i = 0, len = misspellings.length; i < len; i++) {
        misspelling = misspellings[i];
        scope = this.editor.scopeDescriptorForBufferPosition(misspelling[0]);
        if (!this.scopeIsExcluded(scope)) {
          results.push(this.markerLayer.markBufferRange(misspelling, {
            invalidate: 'touch'
          }));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    SpellCheckView.prototype.updateMisspellings = function() {
      return this.taskWrapper.start(this.editor, (function(_this) {
        return function(misspellings) {
          _this.destroyMarkers();
          if (_this.buffer != null) {
            return _this.addMarkers(misspellings);
          }
        };
      })(this));
    };

    SpellCheckView.prototype.getCorrections = function(marker) {
      var args, misspelling, projectPath, ref, ref1, ref2, relativePath;
      projectPath = null;
      relativePath = null;
      if ((ref = this.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
        ref2 = atom.project.relativizePath(this.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
      }
      args = {
        projectPath: projectPath,
        relativePath: relativePath
      };
      misspelling = this.editor.getTextInBufferRange(marker.getBufferRange());
      return this.manager.suggest(args, misspelling);
    };

    SpellCheckView.prototype.addContextMenuEntries = function(mouseEvent) {
      var commandName, contextMenuEntry, correction, correctionIndex, corrections, currentBufferPosition, currentScreenPosition, i, len, marker;
      this.clearContextMenuEntries();
      currentScreenPosition = atom.views.getView(this.editor).component.screenPositionForMouseEvent(mouseEvent);
      currentBufferPosition = this.editor.bufferPositionForScreenPosition(currentScreenPosition);
      if (marker = this.markerLayer.findMarkers({
        containsBufferPosition: currentBufferPosition
      })[0]) {
        corrections = this.getCorrections(marker);
        if (corrections.length > 0) {
          this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
          correctionIndex = 0;
          for (i = 0, len = corrections.length; i < len; i++) {
            correction = corrections[i];
            contextMenuEntry = {};
            commandName = 'spell-check:correct-misspelling-' + correctionIndex;
            contextMenuEntry.command = (function(_this) {
              return function(correction, contextMenuEntry) {
                return atom.commands.add(atom.views.getView(_this.editor), commandName, function() {
                  _this.makeCorrection(correction, marker);
                  return _this.clearContextMenuEntries();
                });
              };
            })(this)(correction, contextMenuEntry);
            contextMenuEntry.menuItem = atom.contextMenu.add({
              'atom-text-editor': [
                {
                  label: correction.label,
                  command: commandName
                }
              ]
            });
            this.spellCheckModule.contextMenuEntries.push(contextMenuEntry);
            correctionIndex++;
          }
          return this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
        }
      }
    };

    SpellCheckView.prototype.makeCorrection = function(correction, marker) {
      var args, projectPath, ref, ref1, ref2, relativePath;
      if (correction.isSuggestion) {
        this.editor.setSelectedBufferRange(marker.getBufferRange());
        return this.editor.insertText(correction.suggestion);
      } else {
        projectPath = null;
        relativePath = null;
        if ((ref = this.editor.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
          ref2 = atom.project.relativizePath(this.editor.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
        }
        args = {
          id: this.id,
          projectPath: projectPath,
          relativePath: relativePath
        };
        correction.plugin.add(args, correction);
        return this.updateMisspellings.bind(this)();
      }
    };

    SpellCheckView.prototype.clearContextMenuEntries = function() {
      var entry, i, len, ref, ref1, ref2;
      ref = this.spellCheckModule.contextMenuEntries;
      for (i = 0, len = ref.length; i < len; i++) {
        entry = ref[i];
        if ((ref1 = entry.command) != null) {
          ref1.dispose();
        }
        if ((ref2 = entry.menuItem) != null) {
          ref2.dispose();
        }
      }
      return this.spellCheckModule.contextMenuEntries = [];
    };

    SpellCheckView.prototype.scopeIsExcluded = function(scopeDescriptor, excludedScopes) {
      return this.spellCheckModule.excludedScopeRegexLists.some(function(regexList) {
        return scopeDescriptor.scopes.some(function(scopeName) {
          return regexList.every(function(regex) {
            return regex.test(scopeName);
          });
        });
      });
    };

    return SpellCheckView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
