/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread"));

var _inheritsLoose2 = _interopRequireDefault(require("@babel/runtime/helpers/inheritsLoose"));

var React = require("react");

var ReactRelayContext = require("./ReactRelayContext");

var ReactRelayQueryFetcher = require("./ReactRelayQueryFetcher");

var areEqual = require("fbjs/lib/areEqual");

var _require = require("relay-runtime"),
    deepFreeze = _require.deepFreeze;

/**
 * React may double-fire the constructor, and we call 'fetch' in the
 * constructor. If a request is already in flight from a previous call to the
 * constructor, just reuse the query fetcher and wait for the response.
 */
var requestCache = {};
var NETWORK_ONLY = 'NETWORK_ONLY';
var STORE_THEN_NETWORK = 'STORE_THEN_NETWORK';
var DataFromEnum = {
  NETWORK_ONLY: NETWORK_ONLY,
  STORE_THEN_NETWORK: STORE_THEN_NETWORK
};

/**
 * @public
 *
 * Orchestrates fetching and rendering data for a single view or view hierarchy:
 * - Fetches the query/variables using the given network implementation.
 * - Normalizes the response(s) to that query, publishing them to the given
 *   store.
 * - Renders the pending/fail/success states with the provided render function.
 * - Subscribes for updates to the root data and re-renders with any changes.
 */
var ReactRelayQueryRenderer =
/*#__PURE__*/
function (_React$Component) {
  (0, _inheritsLoose2["default"])(ReactRelayQueryRenderer, _React$Component);

  function ReactRelayQueryRenderer(props) {
    var _this;

    _this = _React$Component.call(this, props) || this; // Callbacks are attached to the current instance and shared with static
    // lifecyles by bundling with state. This is okay to do because the
    // callbacks don't change in reaction to props. However we should not
    // "leak" them before mounting (since we would be unable to clean up). For
    // that reason, we define them as null initially and fill them in after
    // mounting to avoid leaking memory.

    var retryCallbacks = {
      handleDataChange: null,
      handleRetryAfterError: null
    };
    var queryFetcher;
    var requestCacheKey;

    if (props.query) {
      var query = props.query;
      var getRequest = props.environment.unstable_internal.getRequest;
      var request = getRequest(query);
      requestCacheKey = getRequestCacheKey(request.params, props.variables);
      queryFetcher = requestCache[requestCacheKey] ? requestCache[requestCacheKey].queryFetcher : new ReactRelayQueryFetcher();
    } else {
      queryFetcher = new ReactRelayQueryFetcher();
    }

    _this.state = (0, _objectSpread2["default"])({
      prevPropsEnvironment: props.environment,
      prevPropsVariables: props.variables,
      prevQuery: props.query,
      queryFetcher: queryFetcher,
      retryCallbacks: retryCallbacks
    }, fetchQueryAndComputeStateFromProps(props, queryFetcher, retryCallbacks, requestCacheKey));
    return _this;
  }

  ReactRelayQueryRenderer.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.prevQuery !== nextProps.query || prevState.prevPropsEnvironment !== nextProps.environment || !areEqual(prevState.prevPropsVariables, nextProps.variables)) {
      var query = nextProps.query;
      var prevSelectionReferences = prevState.queryFetcher.getSelectionReferences();
      prevState.queryFetcher.disposeRequest();
      var queryFetcher;

      if (query) {
        var getRequest = nextProps.environment.unstable_internal.getRequest;
        var request = getRequest(query);
        var requestCacheKey = getRequestCacheKey(request.params, nextProps.variables);
        queryFetcher = requestCache[requestCacheKey] ? requestCache[requestCacheKey].queryFetcher : new ReactRelayQueryFetcher(prevSelectionReferences);
      } else {
        queryFetcher = new ReactRelayQueryFetcher(prevSelectionReferences);
      }

      return (0, _objectSpread2["default"])({
        prevQuery: nextProps.query,
        prevPropsEnvironment: nextProps.environment,
        prevPropsVariables: nextProps.variables,
        queryFetcher: queryFetcher
      }, fetchQueryAndComputeStateFromProps(nextProps, queryFetcher, prevState.retryCallbacks // passing no requestCacheKey will cause it to be recalculated internally
      // and we want the updated requestCacheKey, since variables may have changed
      ));
    }

    return null;
  };

  var _proto = ReactRelayQueryRenderer.prototype;

  _proto.componentDidMount = function componentDidMount() {
    var _this2 = this;

    var _this$state = this.state,
        retryCallbacks = _this$state.retryCallbacks,
        queryFetcher = _this$state.queryFetcher,
        requestCacheKey = _this$state.requestCacheKey;

    if (requestCacheKey) {
      delete requestCache[requestCacheKey];
    }

    retryCallbacks.handleDataChange = function (params) {
      var error = params.error == null ? null : params.error;
      var snapshot = params.snapshot == null ? null : params.snapshot;

      _this2.setState(function (prevState) {
        var prevRequestCacheKey = prevState.requestCacheKey;

        if (prevRequestCacheKey) {
          delete requestCache[prevRequestCacheKey];
        } // Don't update state if nothing has changed.


        if (snapshot === prevState.snapshot && error === prevState.error) {
          return null;
        }

        return {
          renderProps: getRenderProps(error, snapshot, prevState.queryFetcher, prevState.retryCallbacks),
          snapshot: snapshot,
          requestCacheKey: null
        };
      });
    };

    retryCallbacks.handleRetryAfterError = function (error) {
      return _this2.setState(function (prevState) {
        var prevRequestCacheKey = prevState.requestCacheKey;

        if (prevRequestCacheKey) {
          delete requestCache[prevRequestCacheKey];
        }

        return {
          renderProps: getLoadingRenderProps(),
          requestCacheKey: null
        };
      });
    }; // Re-initialize the ReactRelayQueryFetcher with callbacks.
    // If data has changed since constructions, this will re-render.


    if (this.props.query) {
      queryFetcher.setOnDataChange(retryCallbacks.handleDataChange);
    }
  };

  _proto.componentDidUpdate = function componentDidUpdate() {
    // We don't need to cache the request after the component commits
    var requestCacheKey = this.state.requestCacheKey;

    if (requestCacheKey) {
      delete requestCache[requestCacheKey]; // HACK

      delete this.state.requestCacheKey;
    }
  };

  _proto.componentWillUnmount = function componentWillUnmount() {
    this.state.queryFetcher.dispose();
  };

  _proto.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState) {
    return nextProps.render !== this.props.render || nextState.renderProps !== this.state.renderProps;
  };

  _proto.render = function render() {
    var _this$state2 = this.state,
        renderProps = _this$state2.renderProps,
        relayContext = _this$state2.relayContext; // Note that the root fragment results in `renderProps.props` is already
    // frozen by the store; this call is to freeze the renderProps object and
    // error property if set.

    if (process.env.NODE_ENV !== "production") {
      deepFreeze(renderProps);
    }

    return React.createElement(ReactRelayContext.Provider, {
      value: relayContext
    }, this.props.render(renderProps));
  };

  return ReactRelayQueryRenderer;
}(React.Component);

function getContext(environment, variables) {
  return {
    environment: environment,
    variables: variables
  };
}

function getLoadingRenderProps() {
  return {
    error: null,
    props: null,
    // `props: null` indicates that the data is being fetched (i.e. loading)
    retry: null
  };
}

function getEmptyRenderProps() {
  return {
    error: null,
    props: {},
    // `props: {}` indicates no data available
    retry: null
  };
}

function getRenderProps(error, snapshot, queryFetcher, retryCallbacks) {
  return {
    error: error ? error : null,
    props: snapshot ? snapshot.data : null,
    retry: function retry() {
      var syncSnapshot = queryFetcher.retry();

      if (syncSnapshot && typeof retryCallbacks.handleDataChange === 'function') {
        retryCallbacks.handleDataChange({
          snapshot: syncSnapshot
        });
      } else if (error && typeof retryCallbacks.handleRetryAfterError === 'function') {
        // If retrying after an error and no synchronous result available,
        // reset the render props
        retryCallbacks.handleRetryAfterError(error);
      }
    }
  };
}

function getRequestCacheKey(request, variables) {
  var requestID = request.id || request.text;
  return JSON.stringify({
    id: String(requestID),
    variables: variables
  });
}

function fetchQueryAndComputeStateFromProps(props, queryFetcher, retryCallbacks, requestCacheKey) {
  var environment = props.environment,
      query = props.query,
      variables = props.variables;
  var genericEnvironment = environment;

  if (query) {
    var _genericEnvironment$u = genericEnvironment.unstable_internal,
        createOperationDescriptor = _genericEnvironment$u.createOperationDescriptor,
        getRequest = _genericEnvironment$u.getRequest;
    var request = getRequest(query);
    var operation = createOperationDescriptor(request, variables);
    var relayContext = getContext(genericEnvironment, operation.variables);

    if (typeof requestCacheKey === 'string' && requestCache[requestCacheKey]) {
      // This same request is already in flight.
      var snapshot = requestCache[requestCacheKey].snapshot;

      if (snapshot) {
        // Use the cached response
        return {
          error: null,
          relayContext: relayContext,
          renderProps: getRenderProps(null, snapshot, queryFetcher, retryCallbacks),
          snapshot: snapshot,
          requestCacheKey: requestCacheKey
        };
      } else {
        // Render loading state
        return {
          error: null,
          relayContext: relayContext,
          renderProps: getLoadingRenderProps(),
          snapshot: null,
          requestCacheKey: requestCacheKey
        };
      }
    }

    try {
      var storeSnapshot = props.dataFrom === STORE_THEN_NETWORK ? queryFetcher.lookupInStore(genericEnvironment, operation) : null;
      var querySnapshot = queryFetcher.fetch({
        cacheConfig: props.cacheConfig,
        dataFrom: props.dataFrom,
        environment: genericEnvironment,
        onDataChange: retryCallbacks.handleDataChange,
        operation: operation
      }); // Use network data first, since it may be fresher

      var _snapshot = querySnapshot || storeSnapshot; // cache the request to avoid duplicate requests


      requestCacheKey = requestCacheKey || getRequestCacheKey(request.params, props.variables);
      requestCache[requestCacheKey] = {
        queryFetcher: queryFetcher,
        snapshot: _snapshot
      };

      if (!_snapshot) {
        return {
          error: null,
          relayContext: relayContext,
          renderProps: getLoadingRenderProps(),
          snapshot: null,
          requestCacheKey: requestCacheKey
        };
      }

      return {
        error: null,
        relayContext: relayContext,
        renderProps: getRenderProps(null, _snapshot, queryFetcher, retryCallbacks),
        snapshot: _snapshot,
        requestCacheKey: requestCacheKey
      };
    } catch (error) {
      return {
        error: error,
        relayContext: relayContext,
        renderProps: getRenderProps(error, null, queryFetcher, retryCallbacks),
        snapshot: null,
        requestCacheKey: requestCacheKey
      };
    }
  } else {
    queryFetcher.dispose();

    var _relayContext = getContext(genericEnvironment, variables);

    return {
      error: null,
      relayContext: _relayContext,
      renderProps: getEmptyRenderProps(),
      requestCacheKey: null // if there is an error, don't cache request

    };
  }
}

module.exports = ReactRelayQueryRenderer;