"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports["default"] = void 0;

var _util = _interopRequireDefault(require("util"));

var _relayRuntime = require("relay-runtime");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const LODASH_ISEQUAL = 'lodash.isequal';
let isEqual = null;
const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');

  const resetsIn = _moment["default"].unix(parseInt(resets, 10)).from(); // eslint-disable-next-line no-console


  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const handler = typeof response === 'function' ? response : () => ({
    data: response
  });
  const promise = new Promise((resolve0, reject0) => {
    resolve = resolve0;
    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    handler,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery["delete"](operationPattern.name);

  return {
    promise,
    resolve,
    reject,
    disable
  };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (isEqual === null) {
          // Lazily require lodash.isequal so we can keep it as a dev dependency.
          // Require indirectly to trick electron-link into not following this.
          isEqual = require(LODASH_ISEQUAL);
        }

        return isEqual(expectation.variables, variables);
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util["default"].inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      const responsePromise = match.promise.then(() => {
        return match.handler(operation);
      });

      if (match.trace) {
        // eslint-disable-next-line no-console
        console.log(`[Relay] query "${operation.name}":\n${operation.text}`);
        responsePromise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`[Relay] response "${operation.name}":`, result);
        }, err => {
          // eslint-disable-next-line no-console
          console.error(`[Relay] error "${operation.name}":\n${err.stack || err}`);
          throw err;
        });
      }

      return responsePromise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    let response;

    try {
      response = await fetch(url, {
        method: 'POST',
        headers: {
          'content-type': 'application/json',
          'Authorization': `bearer ${currentToken}`,
          'Accept': 'application/vnd.github.antiope-preview+json'
        },
        body: JSON.stringify({
          query: operation.text,
          variables
        })
      });
    } catch (e) {
      // A network error was encountered. Mark it so that QueryErrorView and ErrorView can distinguish these, because
      // the errors from "fetch" are TypeErrors without much information.
      e.network = true;
      e.rawStack = e.stack;
      throw e;
    }

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {
      /* do nothing */
    }

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);

    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }

    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);

    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }

    return fetch;
  }

}

exports["default"] = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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