"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);

    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }

    this.known.set(startPath, workDir);
    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra["default"].stat(startPath)).isDirectory() ? startPath : _path["default"].dirname(startPath); // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Within a gitdir or outside of a worktree, return an empty string.
      // Throw if startDir does not exist.

      const topLevel = await _compositeGitStrategy["default"].create(startDir).exec(['rev-parse', '--show-toplevel']);

      if (/\S/.test(topLevel)) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      } // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.


      const gitDir = await _compositeGitStrategy["default"].create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path["default"].resolve(gitDir, '..'));
    } catch (e) {
      /* istanbul ignore if */
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }

      return null;
    }
  }

}

exports["default"] = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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