"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _commit = require("../commit");

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _operationStates = require("../operation-states");

var _multiFilePatch = _interopRequireDefault(require("../patch/multi-file-patch"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();
/**
 * Base class for Repository states. Implements default "null" behavior.
 */

class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  } // This state has just been entered. Perform any asynchronous initialization that needs to occur.


  start() {
    return Promise.resolve();
  } // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.


  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  } // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.


  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  } // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.


  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }
  /* istanbul ignore next */


  refresh() {} // No-op

  /* istanbul ignore next */


  observeFilesystemChange(events) {
    this.repository.refresh();
  }
  /* istanbul ignore next */


  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    this.repository.refresh();
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.
  // Staging and unstaging


  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  } // Committing


  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  } // Merging


  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  } // Checkout


  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  } // Reset


  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  } // Remote interactions


  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  } // Configuration


  setConfig(option, value, {
    replaceAll
  } = {}) {
    return unsupportedOperationPromise(this, 'setConfig');
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  } // Direct blob interactions


  createBlob({
    filePath,
    stdin
  } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  } // Discard history


  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.
  // Index queries


  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: {
          ahead: null,
          behind: null
        }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(_multiFilePatch["default"].createNull());
  }

  getDiffsForFilePath(filePath, options = {}) {
    return Promise.resolve([]);
  }

  getStagedChangesPatch() {
    return Promise.resolve(_multiFilePatch["default"].createNull());
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  } // Commit access


  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  isCommitPushed(sha) {
    return false;
  } // Author information


  getAuthors() {
    return Promise.resolve([]);
  } // Branches


  getBranches() {
    return Promise.resolve(new _branchSet["default"]());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  } // Merging and rebasing status


  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  } // Remotes


  getRemotes() {
    return Promise.resolve(new _remoteSet["default"]([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(null);
  }

  getBehindCount(branchName) {
    return Promise.resolve(null);
  }

  getConfig(option, {
    local
  } = {}) {
    return Promise.resolve(null);
  } // Direct blob access


  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  } // Atom repo state


  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  } // Cache


  getCache() {
    return null;
  } // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.


  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  } // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.


  current() {
    return this.repository.state;
  } // pipeline


  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  } // Return a Promise that will resolve once the state transitions from Loading.


  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  } // Initiate a transition to another state.


  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    /* istanbul ignore if */

    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }

    return this.repository.transition(this, StateConstructor, ...payload);
  } // Event broadcast


  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  } // Direct git access
  // Non-delegated git operations for internal use within states.

  /* istanbul ignore next */


  directResolveDotGitDir() {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }
  /* istanbul ignore next */


  directInit() {
    return Promise.resolve();
  }
  /* istanbul ignore next */


  directClone(remoteUrl, options) {
    return Promise.resolve();
  } // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.


  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  } // Parse a DiscardHistory payload from the SHA recorded in config.


  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');

    if (!historySha) {
      return {};
    }

    let blob;

    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  } // Debugging assistance.


  toString() {
    return this.constructor.name;
  }

}

exports["default"] = State;

function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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