"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

let headless = null;

class EventLogger {
  constructor(kind) {
    this.kind = kind;
    this.directory = '<unknown>';
    this.shortDirectory = '<unknown>';

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  showStarted(directory, implementation) {
    this.directory = directory;
    this.shortDirectory = directory.split(_path["default"].sep).slice(-2).join(_path["default"].sep);

    if (!this.isEnabled()) {
      return;
    }

    this.shortLog(`${implementation} watcher started`);
  }

  showEvents(events) {
    if (!this.isEnabled()) {
      return;
    }

    const uniqueRelativeNames = new Set(events.map(event => {
      return _path["default"].relative(this.directory, event.path);
    }));
    const fileNames = Array.from(uniqueRelativeNames).slice(0, 3);
    const elipses = uniqueRelativeNames.size > 3 ? '...' : '';
    const summary = `${this.getShortName()}: ${fileNames.join(', ')}${elipses}`;
    /* eslint-disable no-console */

    if (headless) {
      const eventText = events.map(event => {
        if (event.action === 'renamed') {
          return `  ${event.oldPath} => ${event.path} (${event.action})\n`;
        } else {
          return `  ${event.path} (${event.action})\n`;
        }
      }).join('\n');
      console.log(summary + '\n' + eventText);
    } else {
      console.groupCollapsed(summary);
      console.table(events, ['action', 'path', 'oldPath']);
      console.groupEnd();
    }
    /* eslint-enable no-console */

  }

  showFocusEvent() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('focus triggered');
  }

  showWorkdirOrHeadEvents() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('working directory or HEAD change');
  }

  showStopped() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('stopped');
  }

  isEnabled() {
    return process.env.ATOM_GITHUB_FS_EVENT_LOG || atom.config.get('github.filesystemEventDiagnostics');
  }

  getShortName() {
    return `${this.kind} @ ${this.shortDirectory}`;
  }

  shortLog(line) {
    if (headless) {
      // eslint-disable-next-line no-console
      console.log(`${this.getShortName()}: ${line}`);
      return;
    } // eslint-disable-next-line no-console


    console.log('%c%s%c: %s', 'font-weight: bold; color: blue;', this.getShortName(), 'font-weight: normal; color: black;', line);
  }

}

exports["default"] = EventLogger;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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