"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _statusBar = _interopRequireDefault(require("../atom/status-bar"));

var _paneItem = _interopRequireDefault(require("../atom/pane-item"));

var _openIssueishDialog = require("../views/open-issueish-dialog");

var _openCommitDialog = require("../views/open-commit-dialog");

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _gitTabItem = _interopRequireDefault(require("../items/git-tab-item"));

var _githubTabItem = _interopRequireDefault(require("../items/github-tab-item"));

var _reviewsItem = _interopRequireDefault(require("../items/reviews-item"));

var _commentDecorationsContainer = _interopRequireDefault(require("../containers/comment-decorations-container"));

var _dialogsController = _interopRequireWildcard(require("./dialogs-controller"));

var _statusBarTileController = _interopRequireDefault(require("./status-bar-tile-controller"));

var _repositoryConflictController = _interopRequireDefault(require("./repository-conflict-controller"));

var _gitCacheView = _interopRequireDefault(require("../views/git-cache-view"));

var _gitTimingsView = _interopRequireDefault(require("../views/git-timings-view"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _switchboard = _interopRequireDefault(require("../switchboard"));

var _propTypes2 = require("../prop-types");

var _helpers = require("../helpers");

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RootController extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "closeDialog", () => new Promise(resolve => this.setState({
      dialogRequest: _dialogsController.dialogRequests["null"]
    }, resolve)));

    _defineProperty(this, "openInitializeDialog", async dirPath => {
      if (!dirPath) {
        const activeEditor = this.props.workspace.getActiveTextEditor();

        if (activeEditor) {
          const [projectPath] = this.props.project.relativizePath(activeEditor.getPath());

          if (projectPath) {
            dirPath = projectPath;
          }
        }
      }

      if (!dirPath) {
        const directories = this.props.project.getDirectories();
        const withRepositories = await Promise.all(directories.map(async d => [d, await this.props.project.repositoryForDirectory(d)]));
        const firstUninitialized = withRepositories.find(([d, r]) => !r);

        if (firstUninitialized && firstUninitialized[0]) {
          dirPath = firstUninitialized[0].getPath();
        }
      }

      if (!dirPath) {
        dirPath = this.props.config.get('core.projectHome');
      }

      const dialogRequest = _dialogsController.dialogRequests.init({
        dirPath
      });

      dialogRequest.onProgressingAccept(async chosenPath => {
        await this.props.initialize(chosenPath);
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCloneDialog", opts => {
      const dialogRequest = _dialogsController.dialogRequests.clone(opts);

      dialogRequest.onProgressingAccept(async (url, chosenPath) => {
        await this.props.clone(url, chosenPath);
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCredentialsDialog", query => {
      return new Promise((resolve, reject) => {
        const dialogRequest = _dialogsController.dialogRequests.credential(query);

        dialogRequest.onProgressingAccept(async result => {
          resolve(result);
          await this.closeDialog();
        });
        dialogRequest.onCancel(async () => {
          reject();
          await this.closeDialog();
        });
        this.setState({
          dialogRequest
        });
      });
    });

    _defineProperty(this, "openIssueishDialog", () => {
      const dialogRequest = _dialogsController.dialogRequests.issueish();

      dialogRequest.onProgressingAccept(async url => {
        await (0, _openIssueishDialog.openIssueishItem)(url, {
          workspace: this.props.workspace,
          workdir: this.props.repository.getWorkingDirectoryPath()
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "openCommitDialog", () => {
      const dialogRequest = _dialogsController.dialogRequests.commit();

      dialogRequest.onProgressingAccept(async ref => {
        await (0, _openCommitDialog.openCommitDetailItem)(ref, {
          workspace: this.props.workspace,
          repository: this.props.repository
        });
        await this.closeDialog();
      });
      dialogRequest.onCancel(this.closeDialog);
      return new Promise(resolve => this.setState({
        dialogRequest
      }, resolve));
    });

    _defineProperty(this, "toggleCommitPreviewItem", () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem["default"].buildURI(workdir));
    });

    _defineProperty(this, "surfaceFromFileAtPath", (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    });

    _defineProperty(this, "surfaceToCommitPreviewButton", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    });

    _defineProperty(this, "surfaceToRecentCommit", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    });

    _defineProperty(this, "reportRelayError", (friendlyMessage, err) => {
      const opts = {
        dismissable: true
      };

      if (err.network) {
        // Offline
        opts.icon = 'alignment-unalign';
        opts.description = "It looks like you're offline right now.";
      } else if (err.responseText) {
        // Transient error like a 500 from the API
        opts.description = 'The GitHub API reported a problem.';
        opts.detail = err.responseText;
      } else if (err.errors) {
        // GraphQL errors
        opts.detail = err.errors.map(e => e.message).join('\n');
      } else {
        opts.detail = err.stack;
      }

      this.props.notificationManager.addError(friendlyMessage, opts);
    });

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');
    this.state = {
      dialogRequest: _dialogsController.dialogRequests["null"]
    };
    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));
    this.props.commands.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          "package": 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react["default"].createElement(_react.Fragment, null, this.renderCommands(), this.renderStatusBarTile(), this.renderPaneItems(), this.renderDialogs(), this.renderConflictResolver(), this.renderCommentDecorations());
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();
    return _react["default"].createElement(_commands["default"], {
      registry: this.props.commands,
      target: "atom-workspace"
    }, devMode && _react["default"].createElement(_commands.Command, {
      command: "github:install-react-dev-tools",
      callback: this.installReactDevTools
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-commit-preview",
      callback: this.toggleCommitPreviewItem
    }), _react["default"].createElement(_commands.Command, {
      command: "github:logout",
      callback: this.clearGithubToken
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-waterfall-diagnostics",
      callback: this.showWaterfallDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-cache-diagnostics",
      callback: this.showCacheDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab",
      callback: this.gitTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab-focus",
      callback: this.gitTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab",
      callback: this.githubTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab-focus",
      callback: this.githubTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:initialize",
      callback: () => this.openInitializeDialog()
    }), _react["default"].createElement(_commands.Command, {
      command: "github:clone",
      callback: () => this.openCloneDialog()
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-issue-or-pull-request",
      callback: () => this.openIssueishDialog()
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-commit",
      callback: () => this.openCommitDialog()
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-unstaged-changes-for-current-file",
      callback: this.viewUnstagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-staged-changes-for-current-file",
      callback: this.viewStagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-all-diff-views",
      callback: this.destroyFilePatchPaneItems
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-empty-diff-views",
      callback: this.destroyEmptyFilePatchPaneItems
    }));
  }

  renderStatusBarTile() {
    return _react["default"].createElement(_statusBar["default"], {
      statusBar: this.props.statusBar,
      onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
      className: "github-StatusBarTileController"
    }, _react["default"].createElement(_statusBarTileController["default"], {
      pipelineManager: this.props.pipelineManager,
      workspace: this.props.workspace,
      repository: this.props.repository,
      commands: this.props.commands,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      confirm: this.props.confirm,
      toggleGitTab: this.gitTabTracker.toggle,
      toggleGithubTab: this.githubTabTracker.toggle
    }));
  }

  renderDialogs() {
    return _react["default"].createElement(_dialogsController["default"], {
      config: this.props.config,
      commands: this.props.commands,
      workspace: this.props.workspace,
      request: this.state.dialogRequest
    });
  }

  renderCommentDecorations() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_commentDecorationsContainer["default"], {
      workspace: this.props.workspace,
      commands: this.props.commands,
      localRepository: this.props.repository,
      loginModel: this.props.loginModel,
      reportRelayError: this.reportRelayError
    });
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_repositoryConflictController["default"], {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commands: this.props.commands
    });
  }

  renderPaneItems() {
    return _react["default"].createElement(_react.Fragment, null, _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTabItem["default"].uriPattern,
      className: "github-Git-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTabItem["default"], {
      ref: itemHolder.setter,
      workspace: this.props.workspace,
      commands: this.props.commands,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      grammars: this.props.grammars,
      project: this.props.project,
      confirm: this.props.confirm,
      config: this.props.config,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      openInitializeDialog: this.openInitializeDialog,
      resolutionProgress: this.props.resolutionProgress,
      ensureGitTab: this.gitTabTracker.ensureVisible,
      openFiles: this.openFiles,
      discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
      undoLastDiscard: this.undoLastDiscard,
      refreshResolutionProgress: this.refreshResolutionProgress
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _githubTabItem["default"].uriPattern,
      className: "github-GitHub-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_githubTabItem["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _changedFileItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_changedFileItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      relPath: _path["default"].join(...params.relPath),
      workingDirectory: params.workingDirectory,
      stagingStatus: params.stagingStatus,
      tooltips: this.props.tooltips,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      workspace: this.props.workspace,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceFileAtPath: this.surfaceFromFileAtPath
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitPreviewItem["default"].uriPattern,
      className: "github-CommitPreview-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitPreviewItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitDetailItem["default"].uriPattern,
      className: "github-CommitDetail-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitDetailItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      sha: params.sha,
      surfaceCommit: this.surfaceToRecentCommit
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _issueishDetailItem["default"].uriPattern
    }, ({
      itemHolder,
      params,
      deserialized
    }) => _react["default"].createElement(_issueishDetailItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      issueishNumber: parseInt(params.issueishNumber, 10),
      workingDirectory: params.workingDirectory,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      initSelectedTab: deserialized.initSelectedTab,
      workspace: this.props.workspace,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      reportRelayError: this.reportRelayError
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _reviewsItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_reviewsItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      number: parseInt(params.number, 10),
      workdir: params.workdir,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      commands: this.props.commands,
      confirm: this.props.confirm,
      reportRelayError: this.reportRelayError
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTimingsView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTimingsView["default"], {
      ref: itemHolder.setter
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitCacheView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitCacheView["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository
    })));
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem["default"].buildURI(), _githubTabItem["default"].buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    await Promise.all([this.installExtension(devTools.REACT_DEVELOPER_TOOLS.id), // relay developer tools extension id
    this.installExtension('ncedobpgnmkhcmnnkcimnobpfepidadl')]);
    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React/Relay dev tools!');
  }

  async installExtension(id) {
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';

    const unzip = require(crossUnzipName);

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${id}%26uc&prodversion=32`;

    const extensionFolder = _path["default"].resolve(_electron.remote.app.getPath('userData'), `extensions/${id}`);

    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra["default"].ensureDir(_path["default"].dirname(extensionFile));
    const response = await fetch(url, {
      method: 'GET'
    });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra["default"].writeFile(extensionFile, body);
    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra["default"].exists(_path["default"].join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });
    await _fsExtra["default"].ensureDir(extensionFolder, 0o755);
    await devTools["default"](id);
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView["default"].buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView["default"].buildURI());
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({
      onlyStaged: false
    }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();

    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra["default"].realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();

    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath); // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place

            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }

    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();

      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }

      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem["default"].buildURI(filePath, repoPath, stagingStatus), {
        pending: true,
        activatePane: true,
        activateItem: true
      });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path["default"].join(repository.getWorkingDirectoryPath(), filePath);

      return this.props.workspace.open(absolutePath, {
        pending: filePaths.length === 1
      });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path["default"].join(workdirPath, filePath);

      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');

    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };

    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();

    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };

    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }

    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);

    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);

      if (results.length === 0) {
        return;
      }

      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({
      conflict
    }) => conflict);

    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({
      filePath
    }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });

    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({
        resultPath
      }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const {
        filePath,
        resultPath,
        deleted,
        conflict,
        theirsSha,
        commonBaseSha,
        currentSha
      } = result;

      const absFilePath = _path["default"].join(this.props.repository.getWorkingDirectoryPath(), filePath);

      if (deleted && resultPath === null) {
        await _fsExtra["default"].remove(absFilePath);
      } else {
        await _fsExtra["default"].copy(resultPath, absFilePath);
      }

      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra["default"].createReadStream(fullPath, {
      encoding: 'utf8'
    });

    return new Promise(resolve => {
      _conflict["default"].countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }

}

exports["default"] = RootController;

_defineProperty(RootController, "propTypes", {
  // Atom enviornment
  workspace: _propTypes["default"].object.isRequired,
  commands: _propTypes["default"].object.isRequired,
  deserializers: _propTypes["default"].object.isRequired,
  notificationManager: _propTypes["default"].object.isRequired,
  tooltips: _propTypes["default"].object.isRequired,
  keymaps: _propTypes["default"].object.isRequired,
  grammars: _propTypes["default"].object.isRequired,
  config: _propTypes["default"].object.isRequired,
  project: _propTypes["default"].object.isRequired,
  confirm: _propTypes["default"].func.isRequired,
  // Models
  loginModel: _propTypes["default"].object.isRequired,
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired,
  repository: _propTypes["default"].object.isRequired,
  resolutionProgress: _propTypes["default"].object.isRequired,
  statusBar: _propTypes["default"].object,
  switchboard: _propTypes["default"].instanceOf(_switchboard["default"]),
  pipelineManager: _propTypes["default"].object,
  // Git actions
  initialize: _propTypes["default"].func.isRequired,
  clone: _propTypes["default"].func.isRequired,
  // Control
  startOpen: _propTypes["default"].bool,
  startRevealed: _propTypes["default"].bool
});

_defineProperty(RootController, "defaultProps", {
  switchboard: new _switchboard["default"](),
  startOpen: false,
  startRevealed: false
});

class TabTracker {
  constructor(name, {
    getWorkspace,
    uri
  }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;
    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false; // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.

    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();

      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }

      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }

    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: false,
      activatePane: false
    });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: true,
      activatePane: true
    });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);

    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);

    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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