"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createUrlFormat = createUrlFormat;

var _i18n = require("@kbn/i18n");

var _lodash = require("lodash");

var _common = require("../../../../../../plugins/data/common/");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const templateMatchRE = /{{([\s\S]+?)}}/g;
const whitelistUrlSchemes = ['http://', 'https://'];
const URL_TYPES = [{
  kind: 'a',
  text: _i18n.i18n.translate('kbn.common.fieldFormats.url.types.link', {
    defaultMessage: 'Link'
  })
}, {
  kind: 'img',
  text: _i18n.i18n.translate('kbn.common.fieldFormats.url.types.img', {
    defaultMessage: 'Image'
  })
}, {
  kind: 'audio',
  text: _i18n.i18n.translate('kbn.common.fieldFormats.url.types.audio', {
    defaultMessage: 'Audio'
  })
}];
const DEFAULT_URL_TYPE = 'a';

function createUrlFormat() {
  class UrlFormat extends _common.FieldFormat {
    constructor(params) {
      super(params);

      _defineProperty(this, "textConvert", value => this.formatLabel(value));

      _defineProperty(this, "htmlConvert", (rawValue, field, hit, parsedUrl) => {
        const url = (0, _lodash.escape)(this.formatUrl(rawValue));
        const label = (0, _lodash.escape)(this.formatLabel(rawValue, url));

        switch (this.param('type')) {
          case 'audio':
            return `<audio controls preload="none" src="${url}">`;

          case 'img':
            // If the URL hasn't been formatted to become a meaningful label then the best we can do
            // is tell screen readers where the image comes from.
            const imageLabel = label === url ? `A dynamically-specified image located at ${url}` : label;
            return this.generateImgHtml(url, imageLabel);

          default:
            const inWhitelist = whitelistUrlSchemes.some(scheme => url.indexOf(scheme) === 0);

            if (!inWhitelist && !parsedUrl) {
              return url;
            }

            let prefix = '';
            /**
             * This code attempts to convert a relative url into a kibana absolute url
             *
             * SUPPORTED:
             *  - /app/kibana/
             *  - ../app/kibana
             *  - #/discover
             *
             * UNSUPPORTED
             *  - app/kibana
             */

            if (!inWhitelist) {
              // Handles urls like: `#/discover`
              if (url[0] === '#') {
                prefix = `${parsedUrl.origin}${parsedUrl.pathname}`;
              } // Handle urls like: `/app/kibana` or `/xyz/app/kibana`
              else if (url.indexOf(parsedUrl.basePath || '/') === 0) {
                  prefix = `${parsedUrl.origin}`;
                } // Handle urls like: `../app/kibana`
                else {
                    const prefixEnd = url[0] === '/' ? '' : '/';
                    prefix = `${parsedUrl.origin}${parsedUrl.basePath || ''}/app${prefixEnd}`;
                  }
            }

            let linkLabel;

            if (hit && hit.highlight && hit.highlight[field.name]) {
              linkLabel = (0, _common.getHighlightHtml)(label, hit.highlight[field.name]);
            } else {
              linkLabel = label;
            }

            const linkTarget = this.param('openLinkInCurrentTab') ? '_self' : '_blank';
            return `<a href="${prefix}${url}" target="${linkTarget}" rel="noopener noreferrer">${linkLabel}</a>`;
        }
      });

      this.compileTemplate = (0, _lodash.memoize)(this.compileTemplate);
    }

    getParamDefaults() {
      return {
        type: DEFAULT_URL_TYPE,
        urlTemplate: null,
        labelTemplate: null
      };
    }

    formatLabel(value, url) {
      const template = this.param('labelTemplate');
      if (url == null) url = this.formatUrl(value);
      if (!template) return url;
      return this.compileTemplate(template)({
        value,
        url
      });
    }

    formatUrl(value) {
      const template = this.param('urlTemplate');
      if (!template) return value;
      return this.compileTemplate(template)({
        value: encodeURIComponent(value),
        rawValue: value
      });
    }

    compileTemplate(template) {
      // trim all the odd bits, the variable names
      const parts = template.split(templateMatchRE).map((part, i) => i % 2 ? part.trim() : part);
      return function (locals) {
        // replace all the odd bits with their local var
        let output = '';
        let i = -1;

        while (++i < parts.length) {
          if (i % 2) {
            if (locals.hasOwnProperty(parts[i])) {
              const local = locals[parts[i]];
              output += local == null ? '' : local;
            }
          } else {
            output += parts[i];
          }
        }

        return output;
      };
    }

    generateImgHtml(url, imageLabel) {
      const isValidWidth = !isNaN(parseInt(this.param('width'), 10));
      const isValidHeight = !isNaN(parseInt(this.param('height'), 10));
      const maxWidth = isValidWidth ? `${this.param('width')}px` : 'none';
      const maxHeight = isValidHeight ? `${this.param('height')}px` : 'none';
      return `<img src="${url}" alt="${imageLabel}" style="width:auto; height:auto; max-width:${maxWidth}; max-height:${maxHeight};">`;
    }

  }

  _defineProperty(UrlFormat, "id", 'url');

  _defineProperty(UrlFormat, "title", 'Url');

  _defineProperty(UrlFormat, "fieldType", [_common.KBN_FIELD_TYPES.NUMBER, _common.KBN_FIELD_TYPES.BOOLEAN, _common.KBN_FIELD_TYPES.DATE, _common.KBN_FIELD_TYPES.IP, _common.KBN_FIELD_TYPES.STRING, _common.KBN_FIELD_TYPES.MURMUR3, _common.KBN_FIELD_TYPES.UNKNOWN, _common.KBN_FIELD_TYPES.CONFLICT]);

  _defineProperty(UrlFormat, "urlTypes", URL_TYPES);

  return UrlFormat;
}