(function() {
  var SpellCheckerManager, manager;

  SpellCheckerManager = (function() {
    function SpellCheckerManager() {}

    SpellCheckerManager.prototype.checkers = [];

    SpellCheckerManager.prototype.checkerPaths = [];

    SpellCheckerManager.prototype.locales = [];

    SpellCheckerManager.prototype.localePaths = [];

    SpellCheckerManager.prototype.useLocales = false;

    SpellCheckerManager.prototype.localeCheckers = null;

    SpellCheckerManager.prototype.knownWords = [];

    SpellCheckerManager.prototype.addKnownWords = false;

    SpellCheckerManager.prototype.knownWordsChecker = null;

    SpellCheckerManager.prototype.setGlobalArgs = function(data) {
      var _, checker, checkers, i, len, ref, removeKnownWordsChecker, removeLocaleCheckers;
      _ = require("underscore-plus");
      removeLocaleCheckers = false;
      removeKnownWordsChecker = false;
      if (!_.isEqual(this.locales, data.locales)) {
        if (!this.localeCheckers || ((ref = data.locales) != null ? ref.length : void 0) !== 0) {
          this.locales = data.locales;
          removeLocaleCheckers = true;
        }
      }
      if (!_.isEqual(this.localePaths, data.localePaths)) {
        this.localePaths = data.localePaths;
        removeLocaleCheckers = true;
      }
      if (this.useLocales !== data.useLocales) {
        this.useLocales = data.useLocales;
        removeLocaleCheckers = true;
      }
      if (!_.isEqual(this.knownWords, data.knownWords)) {
        this.knownWords = data.knownWords;
        removeKnownWordsChecker = true;
      }
      if (this.addKnownWords !== data.addKnownWords) {
        this.addKnownWords = data.addKnownWords;
        removeKnownWordsChecker = true;
      }
      if (removeLocaleCheckers && this.localeCheckers) {
        checkers = this.localeCheckers;
        for (i = 0, len = checkers.length; i < len; i++) {
          checker = checkers[i];
          this.removeSpellChecker(checker);
        }
        this.localeCheckers = null;
      }
      if (removeKnownWordsChecker && this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        return this.knownWordsChecker = null;
      }
    };

    SpellCheckerManager.prototype.addCheckerPath = function(checkerPath) {
      var checker;
      checker = require(checkerPath);
      if (checker["default"]) {
        checker = new checker["default"]();
      }
      return this.addPluginChecker(checker);
    };

    SpellCheckerManager.prototype.addPluginChecker = function(checker) {
      return this.addSpellChecker(checker);
    };

    SpellCheckerManager.prototype.addSpellChecker = function(checker) {
      return this.checkers.push(checker);
    };

    SpellCheckerManager.prototype.removeSpellChecker = function(spellChecker) {
      return this.checkers = this.checkers.filter(function(plugin) {
        return plugin !== spellChecker;
      });
    };

    SpellCheckerManager.prototype.check = function(args, text) {
      var checker, correct, i, incorrects, len, multirange, promises, ref;
      this.init();
      multirange = require('multi-integer-range');
      correct = new multirange.MultiRange([]);
      incorrects = [];
      promises = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSpelling(args)) {
          continue;
        }
        promises.push(Promise.resolve(checker.check(args, text)));
      }
      return Promise.all(promises).then((function(_this) {
        return function(allResults) {
          var incorrect, intersection, invertedCorrect, j, k, l, len1, len2, len3, len4, len5, lineBeginIndex, lineEndIndex, lineRange, m, misspellings, n, newIncorrect, range, rangeIndex, rangeRange, ref1, ref2, ref3, removeRange, results, row;
          for (j = 0, len1 = allResults.length; j < len1; j++) {
            results = allResults[j];
            if (results.invertIncorrectAsCorrect && results.incorrect) {
              invertedCorrect = new multirange.MultiRange([[0, text.length]]);
              removeRange = new multirange.MultiRange([]);
              ref1 = results.incorrect;
              for (k = 0, len2 = ref1.length; k < len2; k++) {
                range = ref1[k];
                removeRange.appendRange(range.start, range.end);
              }
              invertedCorrect.subtract(removeRange);
              correct.append(invertedCorrect);
            } else if (results.correct) {
              ref2 = results.correct;
              for (l = 0, len3 = ref2.length; l < len3; l++) {
                range = ref2[l];
                correct.appendRange(range.start, range.end);
              }
            }
            if (results.incorrect) {
              newIncorrect = new multirange.MultiRange([]);
              incorrects.push(newIncorrect);
              ref3 = results.incorrect;
              for (m = 0, len4 = ref3.length; m < len4; m++) {
                range = ref3[m];
                newIncorrect.appendRange(range.start, range.end);
              }
            }
          }
          if (incorrects.length === 0) {
            return {
              misspellings: []
            };
          }
          intersection = null;
          for (n = 0, len5 = incorrects.length; n < len5; n++) {
            incorrect = incorrects[n];
            if (intersection === null) {
              intersection = incorrect;
            } else {
              intersection.append(incorrect);
            }
          }
          if (intersection.length === 0) {
            return {
              misspellings: []
            };
          }
          if (correct.ranges.length > 0) {
            intersection.subtract(correct);
          }
          row = 0;
          rangeIndex = 0;
          lineBeginIndex = 0;
          misspellings = [];
          while (lineBeginIndex < text.length && rangeIndex < intersection.ranges.length) {
            lineEndIndex = text.indexOf('\n', lineBeginIndex);
            if (lineEndIndex === -1) {
              lineEndIndex = 2e308;
            }
            while (true) {
              range = intersection.ranges[rangeIndex];
              if (range && range[0] < lineEndIndex) {
                lineRange = new multirange.MultiRange([]).appendRange(lineBeginIndex, lineEndIndex);
                rangeRange = new multirange.MultiRange([]).appendRange(range[0], range[1]);
                lineRange.intersect(rangeRange);
                _this.addMisspellings(misspellings, row, lineRange.ranges[0], lineBeginIndex, text);
                if (lineEndIndex >= range[1]) {
                  rangeIndex++;
                } else {
                  break;
                }
              } else {
                break;
              }
            }
            lineBeginIndex = lineEndIndex + 1;
            row++;
          }
          return {
            misspellings: misspellings
          };
        };
      })(this));
    };

    SpellCheckerManager.prototype.suggest = function(args, word) {
      var checker, i, index, j, k, key, keys, l, len, len1, len2, len3, len4, m, priority, ref, ref1, results, s, seen, suggestion, suggestions, target, targets, that;
      this.init();
      suggestions = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSuggestions(args)) {
          continue;
        }
        index = 0;
        priority = checker.getPriority();
        ref1 = checker.suggest(args, word);
        for (j = 0, len1 = ref1.length; j < len1; j++) {
          suggestion = ref1[j];
          suggestions.push({
            isSuggestion: true,
            priority: priority,
            index: index++,
            suggestion: suggestion,
            label: suggestion
          });
        }
      }
      keys = Object.keys(suggestions).sort(function(key1, key2) {
        var value1, value2, weight1, weight2;
        value1 = suggestions[key1];
        value2 = suggestions[key2];
        weight1 = value1.priority + value1.index;
        weight2 = value2.priority + value2.index;
        if (weight1 !== weight2) {
          return weight1 - weight2;
        }
        return value1.suggestion.localeCompare(value2.suggestion);
      });
      results = [];
      seen = [];
      for (k = 0, len2 = keys.length; k < len2; k++) {
        key = keys[k];
        s = suggestions[key];
        if (seen.hasOwnProperty(s.suggestion)) {
          continue;
        }
        results.push(s);
        seen[s.suggestion] = 1;
      }
      that = this;
      keys = Object.keys(this.checkers).sort(function(key1, key2) {
        var value1, value2;
        value1 = that.checkers[key1];
        value2 = that.checkers[key2];
        return value1.getPriority() - value2.getPriority();
      });
      for (l = 0, len3 = keys.length; l < len3; l++) {
        key = keys[l];
        checker = this.checkers[key];
        if (!checker.isEnabled() || !checker.providesAdding(args)) {
          continue;
        }
        targets = checker.getAddingTargets(args);
        for (m = 0, len4 = targets.length; m < len4; m++) {
          target = targets[m];
          target.plugin = checker;
          target.word = word;
          target.isSuggestion = false;
          results.push(target);
        }
      }
      return results;
    };

    SpellCheckerManager.prototype.addMisspellings = function(misspellings, row, range, lineBeginIndex, text) {
      var i, len, markBeginIndex, markEndIndex, part, parts, substring, substringIndex;
      substring = text.substring(range[0], range[1]);
      if (/\s+/.test(substring)) {
        parts = substring.split(/(\s+)/);
        substringIndex = 0;
        for (i = 0, len = parts.length; i < len; i++) {
          part = parts[i];
          if (!/\s+/.test(part)) {
            markBeginIndex = range[0] - lineBeginIndex + substringIndex;
            markEndIndex = markBeginIndex + part.length;
            misspellings.push([[row, markBeginIndex], [row, markEndIndex]]);
          }
          substringIndex += part.length;
        }
        return;
      }
      return misspellings.push([[row, range[0] - lineBeginIndex], [row, range[1] - lineBeginIndex]]);
    };

    SpellCheckerManager.prototype.init = function() {
      var KnownWordsChecker, SystemChecker, checker, defaultLocale, i, len, locale, ref, separatorChar;
      if (this.localeCheckers === null) {
        this.localeCheckers = [];
        if (this.useLocales) {
          if (!this.locales.length) {
            defaultLocale = process.env.LANG;
            if (defaultLocale) {
              this.locales = [defaultLocale.split('.')[0]];
            }
          }
          if (!this.locales.length) {
            defaultLocale = navigator.language;
            if (defaultLocale && defaultLocale.length === 5) {
              separatorChar = defaultLocale.charAt(2);
              if (separatorChar === '_' || separatorChar === '-') {
                this.locales = [defaultLocale];
              }
            }
          }
          if (!this.locales.length) {
            this.locales = ['en_US'];
          }
          SystemChecker = require("./system-checker");
          ref = this.locales;
          for (i = 0, len = ref.length; i < len; i++) {
            locale = ref[i];
            checker = new SystemChecker(locale, this.localePaths);
            this.addSpellChecker(checker);
            this.localeCheckers.push(checker);
          }
        }
      }
      if (this.knownWordsChecker === null) {
        KnownWordsChecker = require('./known-words-checker');
        this.knownWordsChecker = new KnownWordsChecker(this.knownWords);
        this.knownWordsChecker.enableAdd = this.addKnownWords;
        return this.addSpellChecker(this.knownWordsChecker);
      }
    };

    SpellCheckerManager.prototype.deactivate = function() {
      this.checkers = [];
      this.locales = [];
      this.localePaths = [];
      this.useLocales = false;
      this.localeCheckers = null;
      this.knownWords = [];
      this.addKnownWords = false;
      return this.knownWordsChecker = null;
    };

    SpellCheckerManager.prototype.reloadLocales = function() {
      var i, len, localeChecker, ref;
      if (this.localeCheckers) {
        ref = this.localeCheckers;
        for (i = 0, len = ref.length; i < len; i++) {
          localeChecker = ref[i];
          this.removeSpellChecker(localeChecker);
        }
        return this.localeCheckers = null;
      }
    };

    SpellCheckerManager.prototype.reloadKnownWords = function() {
      if (this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        return this.knownWordsChecker = null;
      }
    };

    return SpellCheckerManager;

  })();

  manager = new SpellCheckerManager;

  module.exports = manager;

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
