/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var areEqual = require("fbjs/lib/areEqual");

var emptyFunction = require("fbjs/lib/emptyFunction");

var invariant = require("fbjs/lib/invariant");

var warning = require("fbjs/lib/warning");

var _require = require("relay-runtime"),
    Network = _require.Network,
    isRelayModernEnvironment = _require.isRelayModernEnvironment;

/**
 * The next id to return from `generateId()`.
 */
var nextId = 0;

var ReactRelayTestMocker =
/*#__PURE__*/
function () {
  function ReactRelayTestMocker(env) {
    (0, _defineProperty2["default"])(this, "_defaults", {});
    (0, _defineProperty2["default"])(this, "_pendingFetches", []);

    if (isRelayModernEnvironment(env)) {
      this._mockNetworkLayer(env);
    } else {
      process.env.NODE_ENV !== "production" ? warning(false, 'Netork mocking is currently only supported in Relay Modern. ' + 'You will not be able to resolve requests made with Relay ' + 'Classic environments.') : void 0;
    }

    this._environment = env;
  }
  /**
   * Get a unique id number (as a string). Note: will wrap around after 2^32
   * calls, if your test needs that many IDs.
   *
   * @returns a unique id string
   */


  ReactRelayTestMocker.generateId = function generateId() {
    var toRet = nextId.toString();
    nextId++;
    return toRet;
  }
  /**
   * Create a unique identifier for a (query, variables) pair.
   * @param request: the request associated with the query
   * @param variables: the variables associated with this invocation of the
   * query
   *
   * @returns a string which can later be used to uniquely identify this query
   * in the list of pending queries
   */
  ;

  ReactRelayTestMocker.getIdentifier = function getIdentifier(request) {
    return request.name;
  }
  /**
   * Remove variables that we don't need from the query that will make it more
   * annoying to test (e.g. client_mutation_id, actor_id)
   */
  ;

  ReactRelayTestMocker.stripUnused = function stripUnused(variables) {
    if (variables.input) {
      var toRemove = ['client_mutation_id', 'actor_id', 'clientMutationId', 'actorId'];
      toRemove.forEach(function (item) {
        return variables.input[item] = undefined;
      });
    }

    return variables;
  }
  /**
   * Replace the environment's network layer with a mocked out one to allow for
   * better testing. Mocking the network allows testing without using a mocked
   * out QueryRenderer, and will allow for easier testing of components wrapped
   * in refetch containers, for example. It also allows test writers to see how
   * their components behave under error conditions.
   */
  ;

  var _proto = ReactRelayTestMocker.prototype;

  _proto._mockNetworkLayer = function _mockNetworkLayer(env) {
    var _this = this;

    var fetch = function fetch(request, variables, cacheConfig) {
      var resolve = emptyFunction;
      var reject = emptyFunction;
      var promise = new Promise(function (res, rej) {
        resolve = res;
        reject = rej;
      });
      var strippedVars = ReactRelayTestMocker.stripUnused(variables);
      var ident = ReactRelayTestMocker.getIdentifier(request); // there's a default value for this query, use it

      if (_this._defaults[ident]) {
        var payload = _this._defaults[ident];
        return typeof payload === 'function' ? payload(strippedVars) : payload;
      }

      _this._pendingFetches.push({
        ident: ident,
        cacheConfig: cacheConfig,
        deferred: {
          resolve: resolve,
          reject: reject
        },
        request: request,
        variables: variables
      });

      return promise;
    };

    var isLoading = function isLoading(ident) {
      return _this._pendingFetches.some(function (pending) {
        return pending.ident === ident;
      });
    };

    var resolveRawQuery = function resolveRawQuery(toResolve, payload) {
      _this._pendingFetches = _this._pendingFetches.filter(function (pending) {
        return pending !== toResolve;
      });
      var deferred = toResolve.deferred;
      deferred.resolve(payload);
    };

    var rejectQuery = function rejectQuery(toResolve, payload) {
      _this._pendingFetches = _this._pendingFetches.filter(function (pending) {
        return pending !== toResolve;
      });
      var deferred = toResolve.deferred;
      deferred.reject(payload.error);
    };

    env.mock = {
      isLoading: isLoading,
      rejectQuery: rejectQuery,
      resolveRawQuery: resolveRawQuery,
      fetch: fetch
    };
    env.hasMockedNetwork = true;

    env.__setNet(Network.create(fetch));

    return env;
  };

  _proto._getDefaults = function _getDefaults() {
    return this._defaults;
  }
  /**
   * set a default payload for a given query
   */
  ;

  _proto.setDefault = function setDefault(toSet) {
    var query = toSet.query,
        payload = toSet.payload;
    var ident = ReactRelayTestMocker.getIdentifier(query.params);
    this._defaults[ident] = payload;
  }
  /**
   * remove a default payload for a given query
   */
  ;

  _proto.unsetDefault = function unsetDefault(toUnset) {
    var query = toUnset.query;
    var ident = ReactRelayTestMocker.getIdentifier(query.params);
    delete this._defaults[ident];
  }
  /**
   * Write directly to the Relay store instead of trying to resolve a query that
   * was sent via the network.
   *
   * Use this method when testing a component wrapped in a fragment container
   * (via `createFragmentContainer`). The component under test should also be
   * wrapped in a `RelayTestRenderer`.
   */
  ;

  _proto.dataWrite = function dataWrite(config) {
    var query = config.query,
        variables = config.variables,
        payload = config.payload;
    var createOperationDescriptor = this._environment.unstable_internal.createOperationDescriptor;
    var operationDescriptor = createOperationDescriptor(query, variables);
    !(payload.data != null && payload.errors === undefined) ? process.env.NODE_ENV !== "production" ? invariant(false, 'Only `data` can be written when using `writeDirect`. You may need to ' + 'wrap your payload in an object like `{data: payload}`.') : invariant(false) : void 0;

    this._environment.commitPayload(operationDescriptor, payload.data);
  }
  /**
   * Write the data specified in config's payload to the instance's environment.
   * NOTE: callers may need to invoke `jest.runOnlyPendingTimers()` after
   * calling this function.
   *
   * @param config: an object containing the data to write and the query and
   * variables that the payload is simulating a response to
   */
  ;

  _proto.networkWrite = function networkWrite(config) {
    !this._environment.hasMockedNetwork ? process.env.NODE_ENV !== "production" ? invariant(false, 'You cannot resolve queries without a mocked environment. Did you mean ' + 'to use `writeDirect` instead?') : invariant(false) : void 0;
    var query = config.query,
        variables = config.variables,
        payload = config.payload;
    var ident = ReactRelayTestMocker.getIdentifier(query.params);
    var usedVars;

    if (variables) {
      var createOperationDescriptor = this._environment.unstable_internal.createOperationDescriptor;
      var operationDescriptor = createOperationDescriptor(query, variables);
      usedVars = ReactRelayTestMocker.stripUnused(operationDescriptor.variables);
    }

    var toResolve;

    this._pendingFetches.forEach(function (pending) {
      var pendingVars = pending.variables;

      if (pending.ident === ident) {
        !(!toResolve || variables) ? process.env.NODE_ENV !== "production" ? invariant(false, 'Multiple queries with the same name are currently pending. You ' + 'should pass variables to `write` so that it can determine which ' + 'to resolve') : invariant(false) : void 0;

        if (variables) {
          if (areEqual(pendingVars, usedVars)) {
            toResolve = pending;
          }
        } else {
          toResolve = pending;
        }
      }
    });

    var varMessage = usedVars ? ' - variables: ' + JSON.stringify(usedVars) : '';
    !toResolve ? process.env.NODE_ENV !== "production" ? invariant(false, 'You are attempting to resolve a query that has not been fetched ' + '(%s%s).\n\tPlease ensure you passed the correct variables, or use ' + '`writeDirect` instead.', ident, varMessage) : invariant(false) : void 0;
    var realPayload = typeof payload === 'function' ? payload(toResolve.variables) : payload; // if there are errors, reject the query

    if (realPayload.errors != null && realPayload.errors.length > 0) {
      this._environment.mock.rejectQuery(toResolve, {
        error: realPayload.errors[0]
      });
    } else {
      this._environment.mock.resolveRawQuery(toResolve, realPayload);
    }
  };

  return ReactRelayTestMocker;
}();

module.exports = ReactRelayTestMocker;