"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.openIssueishItem = openIssueishItem;
exports["default"] = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _atom = require("atom");

var _atomTextEditor = _interopRequireDefault(require("../atom/atom-text-editor"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _autofocus = _interopRequireDefault(require("../autofocus"));

var _dialogView = _interopRequireDefault(require("./dialog-view"));

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const ISSUEISH_URL_REGEX = /^(?:https?:\/\/)?(github.com)\/([^/]+)\/([^/]+)\/(?:issues|pull)\/(\d+)/;

class OpenIssueishDialog extends _react["default"].Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "accept", () => {
      const issueishURL = this.url.getText();

      if (issueishURL.length === 0) {
        return Promise.resolve();
      }

      return this.props.request.accept(issueishURL);
    });

    _defineProperty(this, "didChangeURL", () => {
      const enabled = !this.url.isEmpty();

      if (this.state.acceptEnabled !== enabled) {
        this.setState({
          acceptEnabled: enabled
        });
      }
    });

    this.url = new _atom.TextBuffer();
    this.state = {
      acceptEnabled: false
    };
    this.sub = this.url.onDidChange(this.didChangeURL);
    this.autofocus = new _autofocus["default"]();
  }

  render() {
    return _react["default"].createElement(_dialogView["default"], {
      acceptEnabled: this.state.acceptEnabled,
      acceptClassName: "icon icon-git-pull-request",
      acceptText: "Open Issue or Pull Request",
      accept: this.accept,
      cancel: this.props.request.cancel,
      autofocus: this.autofocus,
      inProgress: this.props.inProgress,
      error: this.props.error,
      workspace: this.props.workspace,
      commands: this.props.commands
    }, _react["default"].createElement("label", {
      className: "github-DialogLabel"
    }, "Issue or pull request URL:", _react["default"].createElement(_atomTextEditor["default"], {
      ref: this.autofocus.target,
      mini: true,
      className: "github-OpenIssueish-url",
      buffer: this.url
    })));
  }

  componentDidMount() {
    this.autofocus.trigger();
  }

  componentWillUnmount() {
    this.sub.dispose();
  }

  parseUrl() {
    const url = this.getIssueishUrl();
    const matches = url.match(ISSUEISH_URL_REGEX);

    if (!matches) {
      return false;
    }

    const [_full, repoOwner, repoName, issueishNumber] = matches; // eslint-disable-line no-unused-vars

    return {
      repoOwner,
      repoName,
      issueishNumber
    };
  }

}

exports["default"] = OpenIssueishDialog;

_defineProperty(OpenIssueishDialog, "propTypes", {
  // Model
  request: _propTypes["default"].shape({
    getParams: _propTypes["default"].func.isRequired,
    accept: _propTypes["default"].func.isRequired,
    cancel: _propTypes["default"].func.isRequired
  }).isRequired,
  inProgress: _propTypes["default"].bool,
  error: _propTypes["default"].instanceOf(Error),
  // Atom environment
  workspace: _propTypes["default"].object.isRequired,
  commands: _propTypes["default"].object.isRequired
});

async function openIssueishItem(issueishURL, {
  workspace,
  workdir
}) {
  const matches = ISSUEISH_URL_REGEX.exec(issueishURL);

  if (!matches) {
    throw new Error('Not a valid issue or pull request URL');
  }

  const [, host, owner, repo, number] = matches;

  const uri = _issueishDetailItem["default"].buildURI({
    host,
    owner,
    repo,
    number,
    workdir
  });

  const item = await workspace.open(uri, {
    searchAllPanes: true
  });
  (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
    "package": 'github',
    from: 'dialog'
  });
  return item;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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