"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _util = require("util");

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const sortOrders = {
  'by key': (a, b) => a.key.localeCompare(b.key),
  'oldest first': (a, b) => b.age - a.age,
  'newest first': (a, b) => a.age - b.age,
  'most hits': (a, b) => b.hits - a.hits,
  'fewest hits': (a, b) => a.hits - b.hits
};

class GitCacheView extends _react["default"].Component {
  static buildURI() {
    return this.uriPattern;
  }

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'fetchRepositoryData', 'fetchCacheData', 'renderCache', 'didSelectItem', 'clearCache');
    this.state = {
      order: 'by key'
    };
  }

  getURI() {
    return 'atom-github://debug/cache';
  }

  getTitle() {
    return 'GitHub Package Cache View';
  }

  serialize() {
    return null;
  }

  fetchRepositoryData(repository) {
    return repository.getCache();
  }

  fetchCacheData(cache) {
    const cached = {};
    const promises = [];
    const now = performance.now();

    for (const [key, value] of cache) {
      cached[key] = {
        hits: value.hits,
        age: now - value.createdAt
      };
      promises.push(value.promise.then(payload => (0, _util.inspect)(payload, {
        depth: 3,
        breakLength: 30
      }), err => `${err.message}\n${err.stack}`).then(resolved => {
        cached[key].value = resolved;
      }));
    }

    return Promise.all(promises).then(() => cached);
  }

  render() {
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.repository,
      fetchData: this.fetchRepositoryData
    }, cache => _react["default"].createElement(_observeModel["default"], {
      model: cache,
      fetchData: this.fetchCacheData
    }, this.renderCache));
  }

  renderCache(contents) {
    const rows = Object.keys(contents || {}).map(key => {
      return {
        key,
        age: contents[key].age,
        hits: contents[key].hits,
        content: contents[key].value
      };
    });
    rows.sort(sortOrders[this.state.order]);
    const orders = Object.keys(sortOrders);
    return _react["default"].createElement("div", {
      className: "github-CacheView"
    }, _react["default"].createElement("header", null, _react["default"].createElement("h1", null, "Cache contents"), _react["default"].createElement("p", null, _react["default"].createElement("span", {
      className: "badge"
    }, rows.length), " cached items")), _react["default"].createElement("main", null, _react["default"].createElement("p", {
      className: "github-CacheView-Controls"
    }, _react["default"].createElement("span", {
      className: "github-CacheView-Order"
    }, "order", _react["default"].createElement("select", {
      className: "input-select",
      onChange: this.didSelectItem,
      value: this.state.order
    }, orders.map(order => {
      return _react["default"].createElement("option", {
        key: order,
        value: order
      }, order);
    }))), _react["default"].createElement("span", {
      className: "github-CacheView-Clear"
    }, _react["default"].createElement("button", {
      className: "btn icon icon-trashcan",
      onClick: this.clearCache
    }, "Clear"))), _react["default"].createElement("table", null, _react["default"].createElement("thead", null, _react["default"].createElement("tr", null, _react["default"].createElement("td", {
      className: "github-CacheView-Key"
    }, "key"), _react["default"].createElement("td", {
      className: "github-CacheView-Age"
    }, "age"), _react["default"].createElement("td", {
      className: "github-CacheView-Hits"
    }, "hits"), _react["default"].createElement("td", {
      className: "github-CacheView-Content"
    }, "content"))), _react["default"].createElement("tbody", null, rows.map(row => _react["default"].createElement("tr", {
      key: row.key,
      className: "github-CacheView-Row"
    }, _react["default"].createElement("td", {
      className: "github-CacheView-Key"
    }, _react["default"].createElement("button", {
      className: "btn",
      onClick: () => this.didClickKey(row.key)
    }, row.key)), _react["default"].createElement("td", {
      className: "github-CacheView-Age"
    }, this.formatAge(row.age)), _react["default"].createElement("td", {
      className: "github-CacheView-Hits"
    }, row.hits), _react["default"].createElement("td", {
      className: "github-CacheView-Content"
    }, _react["default"].createElement("code", null, row.content))))))));
  }

  formatAge(ageMs) {
    let remaining = ageMs;
    const parts = [];

    if (remaining > 3600000) {
      const hours = Math.floor(remaining / 3600000);
      parts.push(`${hours}h`);
      remaining -= 3600000 * hours;
    }

    if (remaining > 60000) {
      const minutes = Math.floor(remaining / 60000);
      parts.push(`${minutes}m`);
      remaining -= 60000 * minutes;
    }

    if (remaining > 1000) {
      const seconds = Math.floor(remaining / 1000);
      parts.push(`${seconds}s`);
      remaining -= 1000 * seconds;
    }

    parts.push(`${Math.floor(remaining)}ms`);
    return parts.slice(parts.length - 2).join(' ');
  }

  didSelectItem(event) {
    this.setState({
      order: event.target.value
    });
  }

  didClickKey(key) {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.removePrimary(key);
  }

  clearCache() {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.clear();
  }

}

exports["default"] = GitCacheView;

_defineProperty(GitCacheView, "uriPattern", 'atom-github://debug/cache');

_defineProperty(GitCacheView, "propTypes", {
  repository: _propTypes["default"].object.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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