"use strict";

/* eslint comma-dangle: ["error", {
    "arrays": "never",
    "objects": "never",
    "imports": "never",
    "exports": "never",
    "functions": "never"
  }] */
const {
  execFile
} = require('child_process');

const fs = require('fs');

if (typeof atom === 'undefined') {
  global.atom = {
    inSpecMode() {
      return !!process.env.ATOM_GITHUB_SPEC_MODE;
    },

    inDevMode() {
      return false;
    }

  };
} // No token available in your OS keychain.


const UNAUTHENTICATED = Symbol('UNAUTHENTICATED'); // The token in your keychain isn't granted all of the required OAuth scopes.

const INSUFFICIENT = Symbol('INSUFFICIENT'); // The token in your keychain is not accepted by GitHub.

const UNAUTHORIZED = Symbol('UNAUTHORIZED');

class KeytarStrategy {
  static get keytar() {
    return require('keytar');
  }

  static async isValid() {
    // Allow for disabling Keytar on problematic CI environments
    if (process.env.ATOM_GITHUB_DISABLE_KEYTAR) {
      return false;
    }

    const keytar = this.keytar;

    try {
      const rand = Math.floor(Math.random() * 10e20).toString(16);
      await keytar.setPassword('atom-test-service', rand, rand);
      const pass = await keytar.getPassword('atom-test-service', rand);
      const success = pass === rand;
      keytar.deletePassword('atom-test-service', rand);
      return success;
    } catch (err) {
      return false;
    }
  }

  async getPassword(service, account) {
    const password = await this.constructor.keytar.getPassword(service, account);
    return password !== null ? password : UNAUTHENTICATED;
  }

  replacePassword(service, account, password) {
    return this.constructor.keytar.setPassword(service, account, password);
  }

  deletePassword(service, account) {
    return this.constructor.keytar.deletePassword(service, account);
  }

}

class SecurityBinaryStrategy {
  static isValid() {
    return process.platform === 'darwin';
  }

  async getPassword(service, account) {
    try {
      const password = await this.exec(['find-generic-password', '-s', service, '-a', account, '-w']);
      return password.trim() || UNAUTHENTICATED;
    } catch (err) {
      return UNAUTHENTICATED;
    }
  }

  replacePassword(service, account, newPassword) {
    return this.exec(['add-generic-password', '-s', service, '-a', account, '-w', newPassword, '-U']);
  }

  deletePassword(service, account) {
    return this.exec(['delete-generic-password', '-s', service, '-a', account]);
  }

  exec(securityArgs, {
    binary
  } = {
    binary: 'security'
  }) {
    return new Promise((resolve, reject) => {
      execFile(binary, securityArgs, (error, stdout) => {
        if (error) {
          return reject(error);
        }

        return resolve(stdout);
      });
    });
  }

}

class InMemoryStrategy {
  static isValid() {
    return true;
  }

  constructor() {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using an InMemoryStrategy strategy for storing tokens. ' + 'The tokens will only be stored for the current window.');
    }

    this.passwordsByService = new Map();
  }

  getPassword(service, account) {
    const passwords = this.passwordsByService.get(service) || new Map();
    const password = passwords.get(account);
    return password || UNAUTHENTICATED;
  }

  replacePassword(service, account, newPassword) {
    const passwords = this.passwordsByService.get(service) || new Map();
    passwords.set(account, newPassword);
    this.passwordsByService.set(service, passwords);
  }

  deletePassword(service, account) {
    const passwords = this.passwordsByService.get(service);

    if (passwords) {
      passwords["delete"](account);
    }
  }

}

class FileStrategy {
  static isValid() {
    if (!atom.inSpecMode() && !atom.inDevMode()) {
      return false;
    }

    return Boolean(process.env.ATOM_GITHUB_KEYTAR_FILE);
  }

  constructor() {
    this.filePath = process.env.ATOM_GITHUB_KEYTAR_FILE;

    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using a FileStrategy strategy for storing tokens. ' + 'The tokens will be stored %cin the clear%c in a file at %s. ' + "You probably shouldn't use real credentials while this strategy is in use. " + 'Unset ATOM_GITHUB_KEYTAR_FILE_STRATEGY to disable it.', 'color: red; font-weight: bold; font-style: italic', 'color: black; font-weight: normal; font-style: normal', this.filePath);
    }
  }

  async getPassword(service, account) {
    const payload = await this.load();
    const forService = payload[service];

    if (forService === undefined) {
      return UNAUTHENTICATED;
    }

    const passwd = forService[account];

    if (passwd === undefined) {
      return UNAUTHENTICATED;
    }

    return passwd;
  }

  replacePassword(service, account, password) {
    return this.modify(payload => {
      let forService = payload[service];

      if (forService === undefined) {
        forService = {};
        payload[service] = forService;
      }

      forService[account] = password;
    });
  }

  deletePassword(service, account) {
    return this.modify(payload => {
      const forService = payload[service];

      if (forService === undefined) {
        return;
      }

      delete forService[account];

      if (Object.keys(forService).length === 0) {
        delete payload[service];
      }
    });
  }

  load() {
    return new Promise((resolve, reject) => {
      fs.readFile(this.filePath, 'utf8', (err, content) => {
        if (err && err.code === 'ENOENT') {
          return resolve({});
        }

        if (err) {
          return reject(err);
        }

        return resolve(JSON.parse(content));
      });
    });
  }

  save(payload) {
    return new Promise((resolve, reject) => {
      fs.writeFile(this.filePath, JSON.stringify(payload), 'utf8', err => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

  async modify(callback) {
    const payload = await this.load();
    callback(payload);
    await this.save(payload);
  }

}

const strategies = [FileStrategy, KeytarStrategy, SecurityBinaryStrategy, InMemoryStrategy];
let ValidStrategy = null;

async function createStrategy() {
  if (ValidStrategy) {
    return new ValidStrategy();
  }

  for (let i = 0; i < strategies.length; i++) {
    const strat = strategies[i];
    const isValid = await strat.isValid();

    if (isValid) {
      ValidStrategy = strat;
      break;
    }
  }

  if (!ValidStrategy) {
    throw new Error('None of the listed keytar strategies returned true for `isValid`');
  }

  return new ValidStrategy();
}

module.exports = {
  UNAUTHENTICATED,
  INSUFFICIENT,
  UNAUTHORIZED,
  KeytarStrategy,
  SecurityBinaryStrategy,
  InMemoryStrategy,
  FileStrategy,
  createStrategy
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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