"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.incrementCounter = incrementCounter;
exports.addTiming = addTiming;
exports.addEvent = addEvent;
exports.FakeReporter = exports.reporterProxy = exports.FIVE_MINUTES_IN_MILLISECONDS = void 0;

const pjson = require('../package.json');

const FIVE_MINUTES_IN_MILLISECONDS = 1000 * 60 * 5; // this class allows us to call reporter methods
// before the reporter is actually loaded, since we don't want to
// assume that the metrics package will load before the GitHub package.

exports.FIVE_MINUTES_IN_MILLISECONDS = FIVE_MINUTES_IN_MILLISECONDS;

class ReporterProxy {
  constructor() {
    this.reporter = null;
    this.events = [];
    this.timings = [];
    this.counters = [];
    this.gitHubPackageVersion = pjson.version;
    this.timeout = null;
  } // function that is called after the reporter is actually loaded, to
  // set the reporter and send any data that have accumulated while it was loading.


  setReporter(reporter) {
    this.reporter = reporter;
    this.events.forEach(customEvent => {
      this.reporter.addCustomEvent(customEvent.eventType, customEvent.event);
    });
    this.events = [];
    this.timings.forEach(timing => {
      this.reporter.addTiming(timing.eventType, timing.durationInMilliseconds, timing.metadata);
    });
    this.timings = [];
    this.counters.forEach(counterName => {
      this.reporter.incrementCounter(counterName);
    });
    this.counters = [];
  }

  incrementCounter(counterName) {
    if (this.reporter === null) {
      this.startTimer();
      this.counters.push(counterName);
      return;
    }

    this.reporter.incrementCounter(counterName);
  }

  addTiming(eventType, durationInMilliseconds, metadata = {}) {
    if (this.reporter === null) {
      this.startTimer();
      this.timings.push({
        eventType,
        durationInMilliseconds,
        metadata
      });
      return;
    }

    this.reporter.addTiming(eventType, durationInMilliseconds, metadata);
  }

  addEvent(eventType, event) {
    if (this.reporter === null) {
      this.startTimer();
      this.events.push({
        eventType,
        event
      });
      return;
    }

    this.reporter.addCustomEvent(eventType, event);
  }

  startTimer() {
    if (this.timeout !== null) {
      return;
    } // if for some reason a user disables the metrics package, we don't want to
    // just keep accumulating events in memory until the heat death of the universe.
    // Use a no-op class, clear all queues, move on with our lives.


    this.timeout = setTimeout(FIVE_MINUTES_IN_MILLISECONDS, () => {
      if (this.reporter === null) {
        this.setReporter(new FakeReporter());
        this.events = [];
        this.timings = [];
        this.counters = [];
      }
    });
  }

}

const reporterProxy = new ReporterProxy();
exports.reporterProxy = reporterProxy;

class FakeReporter {
  addCustomEvent() {}

  addTiming() {}

  incrementCounter() {}

}

exports.FakeReporter = FakeReporter;

function incrementCounter(counterName) {
  reporterProxy.incrementCounter(counterName);
}

function addTiming(eventType, durationInMilliseconds, metadata = {}) {
  metadata.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addTiming(eventType, durationInMilliseconds, metadata);
}

function addEvent(eventType, event) {
  event.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addEvent(eventType, event);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlcG9ydGVyLXByb3h5LmpzIl0sIm5hbWVzIjpbInBqc29uIiwicmVxdWlyZSIsIkZJVkVfTUlOVVRFU19JTl9NSUxMSVNFQ09ORFMiLCJSZXBvcnRlclByb3h5IiwiY29uc3RydWN0b3IiLCJyZXBvcnRlciIsImV2ZW50cyIsInRpbWluZ3MiLCJjb3VudGVycyIsImdpdEh1YlBhY2thZ2VWZXJzaW9uIiwidmVyc2lvbiIsInRpbWVvdXQiLCJzZXRSZXBvcnRlciIsImZvckVhY2giLCJjdXN0b21FdmVudCIsImFkZEN1c3RvbUV2ZW50IiwiZXZlbnRUeXBlIiwiZXZlbnQiLCJ0aW1pbmciLCJhZGRUaW1pbmciLCJkdXJhdGlvbkluTWlsbGlzZWNvbmRzIiwibWV0YWRhdGEiLCJjb3VudGVyTmFtZSIsImluY3JlbWVudENvdW50ZXIiLCJzdGFydFRpbWVyIiwicHVzaCIsImFkZEV2ZW50Iiwic2V0VGltZW91dCIsIkZha2VSZXBvcnRlciIsInJlcG9ydGVyUHJveHkiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7QUFBQSxNQUFNQSxLQUFLLEdBQUdDLE9BQU8sQ0FBQyxpQkFBRCxDQUFyQjs7QUFFTyxNQUFNQyw0QkFBNEIsR0FBRyxPQUFPLEVBQVAsR0FBWSxDQUFqRCxDLENBRVA7QUFDQTtBQUNBOzs7O0FBQ0EsTUFBTUMsYUFBTixDQUFvQjtBQUNsQkMsRUFBQUEsV0FBVyxHQUFHO0FBQ1osU0FBS0MsUUFBTCxHQUFnQixJQUFoQjtBQUNBLFNBQUtDLE1BQUwsR0FBYyxFQUFkO0FBQ0EsU0FBS0MsT0FBTCxHQUFlLEVBQWY7QUFDQSxTQUFLQyxRQUFMLEdBQWdCLEVBQWhCO0FBQ0EsU0FBS0Msb0JBQUwsR0FBNEJULEtBQUssQ0FBQ1UsT0FBbEM7QUFFQSxTQUFLQyxPQUFMLEdBQWUsSUFBZjtBQUNELEdBVGlCLENBV2xCO0FBQ0E7OztBQUNBQyxFQUFBQSxXQUFXLENBQUNQLFFBQUQsRUFBVztBQUNwQixTQUFLQSxRQUFMLEdBQWdCQSxRQUFoQjtBQUVBLFNBQUtDLE1BQUwsQ0FBWU8sT0FBWixDQUFvQkMsV0FBVyxJQUFJO0FBQ2pDLFdBQUtULFFBQUwsQ0FBY1UsY0FBZCxDQUE2QkQsV0FBVyxDQUFDRSxTQUF6QyxFQUFvREYsV0FBVyxDQUFDRyxLQUFoRTtBQUNELEtBRkQ7QUFHQSxTQUFLWCxNQUFMLEdBQWMsRUFBZDtBQUVBLFNBQUtDLE9BQUwsQ0FBYU0sT0FBYixDQUFxQkssTUFBTSxJQUFJO0FBQzdCLFdBQUtiLFFBQUwsQ0FBY2MsU0FBZCxDQUF3QkQsTUFBTSxDQUFDRixTQUEvQixFQUEwQ0UsTUFBTSxDQUFDRSxzQkFBakQsRUFBeUVGLE1BQU0sQ0FBQ0csUUFBaEY7QUFDRCxLQUZEO0FBR0EsU0FBS2QsT0FBTCxHQUFlLEVBQWY7QUFFQSxTQUFLQyxRQUFMLENBQWNLLE9BQWQsQ0FBc0JTLFdBQVcsSUFBSTtBQUNuQyxXQUFLakIsUUFBTCxDQUFja0IsZ0JBQWQsQ0FBK0JELFdBQS9CO0FBQ0QsS0FGRDtBQUdBLFNBQUtkLFFBQUwsR0FBZ0IsRUFBaEI7QUFDRDs7QUFFRGUsRUFBQUEsZ0JBQWdCLENBQUNELFdBQUQsRUFBYztBQUM1QixRQUFJLEtBQUtqQixRQUFMLEtBQWtCLElBQXRCLEVBQTRCO0FBQzFCLFdBQUttQixVQUFMO0FBQ0EsV0FBS2hCLFFBQUwsQ0FBY2lCLElBQWQsQ0FBbUJILFdBQW5CO0FBQ0E7QUFDRDs7QUFFRCxTQUFLakIsUUFBTCxDQUFja0IsZ0JBQWQsQ0FBK0JELFdBQS9CO0FBQ0Q7O0FBRURILEVBQUFBLFNBQVMsQ0FBQ0gsU0FBRCxFQUFZSSxzQkFBWixFQUFvQ0MsUUFBUSxHQUFHLEVBQS9DLEVBQW1EO0FBQzFELFFBQUksS0FBS2hCLFFBQUwsS0FBa0IsSUFBdEIsRUFBNEI7QUFDMUIsV0FBS21CLFVBQUw7QUFDQSxXQUFLakIsT0FBTCxDQUFha0IsSUFBYixDQUFrQjtBQUFDVCxRQUFBQSxTQUFEO0FBQVlJLFFBQUFBLHNCQUFaO0FBQW9DQyxRQUFBQTtBQUFwQyxPQUFsQjtBQUNBO0FBQ0Q7O0FBRUQsU0FBS2hCLFFBQUwsQ0FBY2MsU0FBZCxDQUF3QkgsU0FBeEIsRUFBbUNJLHNCQUFuQyxFQUEyREMsUUFBM0Q7QUFDRDs7QUFFREssRUFBQUEsUUFBUSxDQUFDVixTQUFELEVBQVlDLEtBQVosRUFBbUI7QUFDekIsUUFBSSxLQUFLWixRQUFMLEtBQWtCLElBQXRCLEVBQTRCO0FBQzFCLFdBQUttQixVQUFMO0FBQ0EsV0FBS2xCLE1BQUwsQ0FBWW1CLElBQVosQ0FBaUI7QUFBQ1QsUUFBQUEsU0FBRDtBQUFZQyxRQUFBQTtBQUFaLE9BQWpCO0FBQ0E7QUFDRDs7QUFFRCxTQUFLWixRQUFMLENBQWNVLGNBQWQsQ0FBNkJDLFNBQTdCLEVBQXdDQyxLQUF4QztBQUNEOztBQUVETyxFQUFBQSxVQUFVLEdBQUc7QUFDWCxRQUFJLEtBQUtiLE9BQUwsS0FBaUIsSUFBckIsRUFBMkI7QUFDekI7QUFDRCxLQUhVLENBS1g7QUFDQTtBQUNBOzs7QUFDQSxTQUFLQSxPQUFMLEdBQWVnQixVQUFVLENBQUN6Qiw0QkFBRCxFQUErQixNQUFNO0FBQzVELFVBQUksS0FBS0csUUFBTCxLQUFrQixJQUF0QixFQUE0QjtBQUMxQixhQUFLTyxXQUFMLENBQWlCLElBQUlnQixZQUFKLEVBQWpCO0FBQ0EsYUFBS3RCLE1BQUwsR0FBYyxFQUFkO0FBQ0EsYUFBS0MsT0FBTCxHQUFlLEVBQWY7QUFDQSxhQUFLQyxRQUFMLEdBQWdCLEVBQWhCO0FBQ0Q7QUFDRixLQVB3QixDQUF6QjtBQVFEOztBQTlFaUI7O0FBaUZiLE1BQU1xQixhQUFhLEdBQUcsSUFBSTFCLGFBQUosRUFBdEI7OztBQUVBLE1BQU15QixZQUFOLENBQW1CO0FBQ3hCYixFQUFBQSxjQUFjLEdBQUcsQ0FBRTs7QUFFbkJJLEVBQUFBLFNBQVMsR0FBRyxDQUFFOztBQUVkSSxFQUFBQSxnQkFBZ0IsR0FBRyxDQUFFOztBQUxHOzs7O0FBUW5CLFNBQVNBLGdCQUFULENBQTBCRCxXQUExQixFQUF1QztBQUM1Q08sRUFBQUEsYUFBYSxDQUFDTixnQkFBZCxDQUErQkQsV0FBL0I7QUFDRDs7QUFFTSxTQUFTSCxTQUFULENBQW1CSCxTQUFuQixFQUE4Qkksc0JBQTlCLEVBQXNEQyxRQUFRLEdBQUcsRUFBakUsRUFBcUU7QUFDMUVBLEVBQUFBLFFBQVEsQ0FBQ1osb0JBQVQsR0FBZ0NvQixhQUFhLENBQUNwQixvQkFBOUM7QUFDQW9CLEVBQUFBLGFBQWEsQ0FBQ1YsU0FBZCxDQUF3QkgsU0FBeEIsRUFBbUNJLHNCQUFuQyxFQUEyREMsUUFBM0Q7QUFDRDs7QUFFTSxTQUFTSyxRQUFULENBQWtCVixTQUFsQixFQUE2QkMsS0FBN0IsRUFBb0M7QUFDekNBLEVBQUFBLEtBQUssQ0FBQ1Isb0JBQU4sR0FBNkJvQixhQUFhLENBQUNwQixvQkFBM0M7QUFDQW9CLEVBQUFBLGFBQWEsQ0FBQ0gsUUFBZCxDQUF1QlYsU0FBdkIsRUFBa0NDLEtBQWxDO0FBQ0QiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20vb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiY29uc3QgcGpzb24gPSByZXF1aXJlKCcuLi9wYWNrYWdlLmpzb24nKTtcblxuZXhwb3J0IGNvbnN0IEZJVkVfTUlOVVRFU19JTl9NSUxMSVNFQ09ORFMgPSAxMDAwICogNjAgKiA1O1xuXG4vLyB0aGlzIGNsYXNzIGFsbG93cyB1cyB0byBjYWxsIHJlcG9ydGVyIG1ldGhvZHNcbi8vIGJlZm9yZSB0aGUgcmVwb3J0ZXIgaXMgYWN0dWFsbHkgbG9hZGVkLCBzaW5jZSB3ZSBkb24ndCB3YW50IHRvXG4vLyBhc3N1bWUgdGhhdCB0aGUgbWV0cmljcyBwYWNrYWdlIHdpbGwgbG9hZCBiZWZvcmUgdGhlIEdpdEh1YiBwYWNrYWdlLlxuY2xhc3MgUmVwb3J0ZXJQcm94eSB7XG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHRoaXMucmVwb3J0ZXIgPSBudWxsO1xuICAgIHRoaXMuZXZlbnRzID0gW107XG4gICAgdGhpcy50aW1pbmdzID0gW107XG4gICAgdGhpcy5jb3VudGVycyA9IFtdO1xuICAgIHRoaXMuZ2l0SHViUGFja2FnZVZlcnNpb24gPSBwanNvbi52ZXJzaW9uO1xuXG4gICAgdGhpcy50aW1lb3V0ID0gbnVsbDtcbiAgfVxuXG4gIC8vIGZ1bmN0aW9uIHRoYXQgaXMgY2FsbGVkIGFmdGVyIHRoZSByZXBvcnRlciBpcyBhY3R1YWxseSBsb2FkZWQsIHRvXG4gIC8vIHNldCB0aGUgcmVwb3J0ZXIgYW5kIHNlbmQgYW55IGRhdGEgdGhhdCBoYXZlIGFjY3VtdWxhdGVkIHdoaWxlIGl0IHdhcyBsb2FkaW5nLlxuICBzZXRSZXBvcnRlcihyZXBvcnRlcikge1xuICAgIHRoaXMucmVwb3J0ZXIgPSByZXBvcnRlcjtcblxuICAgIHRoaXMuZXZlbnRzLmZvckVhY2goY3VzdG9tRXZlbnQgPT4ge1xuICAgICAgdGhpcy5yZXBvcnRlci5hZGRDdXN0b21FdmVudChjdXN0b21FdmVudC5ldmVudFR5cGUsIGN1c3RvbUV2ZW50LmV2ZW50KTtcbiAgICB9KTtcbiAgICB0aGlzLmV2ZW50cyA9IFtdO1xuXG4gICAgdGhpcy50aW1pbmdzLmZvckVhY2godGltaW5nID0+IHtcbiAgICAgIHRoaXMucmVwb3J0ZXIuYWRkVGltaW5nKHRpbWluZy5ldmVudFR5cGUsIHRpbWluZy5kdXJhdGlvbkluTWlsbGlzZWNvbmRzLCB0aW1pbmcubWV0YWRhdGEpO1xuICAgIH0pO1xuICAgIHRoaXMudGltaW5ncyA9IFtdO1xuXG4gICAgdGhpcy5jb3VudGVycy5mb3JFYWNoKGNvdW50ZXJOYW1lID0+IHtcbiAgICAgIHRoaXMucmVwb3J0ZXIuaW5jcmVtZW50Q291bnRlcihjb3VudGVyTmFtZSk7XG4gICAgfSk7XG4gICAgdGhpcy5jb3VudGVycyA9IFtdO1xuICB9XG5cbiAgaW5jcmVtZW50Q291bnRlcihjb3VudGVyTmFtZSkge1xuICAgIGlmICh0aGlzLnJlcG9ydGVyID09PSBudWxsKSB7XG4gICAgICB0aGlzLnN0YXJ0VGltZXIoKTtcbiAgICAgIHRoaXMuY291bnRlcnMucHVzaChjb3VudGVyTmFtZSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5yZXBvcnRlci5pbmNyZW1lbnRDb3VudGVyKGNvdW50ZXJOYW1lKTtcbiAgfVxuXG4gIGFkZFRpbWluZyhldmVudFR5cGUsIGR1cmF0aW9uSW5NaWxsaXNlY29uZHMsIG1ldGFkYXRhID0ge30pIHtcbiAgICBpZiAodGhpcy5yZXBvcnRlciA9PT0gbnVsbCkge1xuICAgICAgdGhpcy5zdGFydFRpbWVyKCk7XG4gICAgICB0aGlzLnRpbWluZ3MucHVzaCh7ZXZlbnRUeXBlLCBkdXJhdGlvbkluTWlsbGlzZWNvbmRzLCBtZXRhZGF0YX0pO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMucmVwb3J0ZXIuYWRkVGltaW5nKGV2ZW50VHlwZSwgZHVyYXRpb25Jbk1pbGxpc2Vjb25kcywgbWV0YWRhdGEpO1xuICB9XG5cbiAgYWRkRXZlbnQoZXZlbnRUeXBlLCBldmVudCkge1xuICAgIGlmICh0aGlzLnJlcG9ydGVyID09PSBudWxsKSB7XG4gICAgICB0aGlzLnN0YXJ0VGltZXIoKTtcbiAgICAgIHRoaXMuZXZlbnRzLnB1c2goe2V2ZW50VHlwZSwgZXZlbnR9KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLnJlcG9ydGVyLmFkZEN1c3RvbUV2ZW50KGV2ZW50VHlwZSwgZXZlbnQpO1xuICB9XG5cbiAgc3RhcnRUaW1lcigpIHtcbiAgICBpZiAodGhpcy50aW1lb3V0ICE9PSBudWxsKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgLy8gaWYgZm9yIHNvbWUgcmVhc29uIGEgdXNlciBkaXNhYmxlcyB0aGUgbWV0cmljcyBwYWNrYWdlLCB3ZSBkb24ndCB3YW50IHRvXG4gICAgLy8ganVzdCBrZWVwIGFjY3VtdWxhdGluZyBldmVudHMgaW4gbWVtb3J5IHVudGlsIHRoZSBoZWF0IGRlYXRoIG9mIHRoZSB1bml2ZXJzZS5cbiAgICAvLyBVc2UgYSBuby1vcCBjbGFzcywgY2xlYXIgYWxsIHF1ZXVlcywgbW92ZSBvbiB3aXRoIG91ciBsaXZlcy5cbiAgICB0aGlzLnRpbWVvdXQgPSBzZXRUaW1lb3V0KEZJVkVfTUlOVVRFU19JTl9NSUxMSVNFQ09ORFMsICgpID0+IHtcbiAgICAgIGlmICh0aGlzLnJlcG9ydGVyID09PSBudWxsKSB7XG4gICAgICAgIHRoaXMuc2V0UmVwb3J0ZXIobmV3IEZha2VSZXBvcnRlcigpKTtcbiAgICAgICAgdGhpcy5ldmVudHMgPSBbXTtcbiAgICAgICAgdGhpcy50aW1pbmdzID0gW107XG4gICAgICAgIHRoaXMuY291bnRlcnMgPSBbXTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxufVxuXG5leHBvcnQgY29uc3QgcmVwb3J0ZXJQcm94eSA9IG5ldyBSZXBvcnRlclByb3h5KCk7XG5cbmV4cG9ydCBjbGFzcyBGYWtlUmVwb3J0ZXIge1xuICBhZGRDdXN0b21FdmVudCgpIHt9XG5cbiAgYWRkVGltaW5nKCkge31cblxuICBpbmNyZW1lbnRDb3VudGVyKCkge31cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGluY3JlbWVudENvdW50ZXIoY291bnRlck5hbWUpIHtcbiAgcmVwb3J0ZXJQcm94eS5pbmNyZW1lbnRDb3VudGVyKGNvdW50ZXJOYW1lKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFkZFRpbWluZyhldmVudFR5cGUsIGR1cmF0aW9uSW5NaWxsaXNlY29uZHMsIG1ldGFkYXRhID0ge30pIHtcbiAgbWV0YWRhdGEuZ2l0SHViUGFja2FnZVZlcnNpb24gPSByZXBvcnRlclByb3h5LmdpdEh1YlBhY2thZ2VWZXJzaW9uO1xuICByZXBvcnRlclByb3h5LmFkZFRpbWluZyhldmVudFR5cGUsIGR1cmF0aW9uSW5NaWxsaXNlY29uZHMsIG1ldGFkYXRhKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFkZEV2ZW50KGV2ZW50VHlwZSwgZXZlbnQpIHtcbiAgZXZlbnQuZ2l0SHViUGFja2FnZVZlcnNpb24gPSByZXBvcnRlclByb3h5LmdpdEh1YlBhY2thZ2VWZXJzaW9uO1xuICByZXBvcnRlclByb3h5LmFkZEV2ZW50KGV2ZW50VHlwZSwgZXZlbnQpO1xufVxuIl19