"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _compareSets = _interopRequireDefault(require("compare-sets"));

var _eventKit = require("event-kit");

var _workdirContext = _interopRequireDefault(require("./workdir-context"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Manage a WorkdirContext for each open directory.
 */
class WorkdirContextPool {
  /**
   * Options will be passed to each `WorkdirContext` as it is created.
   */
  constructor(options = {}) {
    this.options = options;
    this.contexts = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  size() {
    return this.contexts.size;
  }
  /**
   * Access the context mapped to a known directory.
   */


  getContext(directory) {
    const {
      pipelineManager
    } = this.options;
    return this.contexts.get(directory) || _workdirContext["default"].absent({
      pipelineManager
    });
  }
  /**
   * Return a WorkdirContext whose Repository has at least one remote configured to push to the named GitHub repository.
   * Returns a null context if zero or more than one contexts match.
   */


  async getMatchingContext(host, owner, repo) {
    const matches = await Promise.all(this.withResidentContexts(async (_workdir, context) => {
      const match = await context.getRepository().hasGitHubRemote(host, owner, repo);
      return match ? context : null;
    }));
    const filtered = matches.filter(Boolean);
    return filtered.length === 1 ? filtered[0] : _workdirContext["default"].absent(_objectSpread2({}, this.options));
  }

  add(directory, options = {}) {
    if (this.contexts.has(directory)) {
      return this.getContext(directory);
    }

    const context = new _workdirContext["default"](directory, _objectSpread2({}, this.options, {}, options));
    this.contexts.set(directory, context);
    const disposable = new _eventKit.CompositeDisposable();

    const forwardEvent = (subMethod, emitEventName) => {
      const emit = () => this.emitter.emit(emitEventName, context);

      disposable.add(context[subMethod](emit));
    };

    forwardEvent('onDidStartObserver', 'did-start-observer');
    forwardEvent('onDidChangeWorkdirOrHead', 'did-change-workdir-or-head');
    forwardEvent('onDidChangeRepositoryState', 'did-change-repository-state');
    forwardEvent('onDidUpdateRepository', 'did-update-repository');
    forwardEvent('onDidDestroyRepository', 'did-destroy-repository');
    disposable.add(this.onDidRemoveContext(removed => {
      if (removed === context) {
        disposable.dispose();
      }
    }));
    return context;
  }

  replace(directory, options = {}) {
    this.remove(directory);
    this.add(directory, options);
  }

  remove(directory) {
    const existing = this.contexts.get(directory);
    this.contexts["delete"](directory);

    if (existing) {
      this.emitter.emit('did-remove-context', existing);
      existing.destroy();
    }
  }

  set(directories, options = {}) {
    const previous = new Set(this.contexts.keys());
    const {
      added,
      removed
    } = (0, _compareSets["default"])(previous, directories);

    for (const directory of added) {
      this.add(directory, options);
    }

    for (const directory of removed) {
      this.remove(directory);
    }
  }

  withResidentContexts(callback) {
    const results = [];

    for (const [workdir, context] of this.contexts) {
      results.push(callback(workdir, context));
    }

    return results;
  }

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidRemoveContext(callback) {
    return this.emitter.on('did-remove-context', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  clear() {
    this.withResidentContexts(workdir => this.remove(workdir));

    _workdirContext["default"].destroyAbsent();
  }

}

exports["default"] = WorkdirContextPool;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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