"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NoNewline = exports.Unchanged = exports.Deletion = exports.Addition = void 0;

var _atom = require("atom");

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Region {
  constructor(marker) {
    this.marker = marker;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.marker.getRange();
  }

  getStartBufferRow() {
    return this.getRange().start.row;
  }

  getEndBufferRow() {
    return this.getRange().end.row;
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }
  /*
   * intersectRows breaks a Region into runs of rows that are included in
   * rowSet and rows that are not. For example:
   *  @this Region        row 10-20
   *  @param rowSet       row 11, 12, 13, 17, 19
   *  @param includeGaps  true (whether the result will include gaps or not)
   *  @return an array of regions like this:
   *    (10, gap = true) (11, 12, 13, gap = false) (14, 15, 16, gap = true)
   *    (17, gap = false) (18, gap = true) (19, gap = false) (20, gap = true)
   */


  intersectRows(rowSet, includeGaps) {
    const intersections = [];
    let withinIntersection = false;
    let currentRow = this.getRange().start.row;
    let nextStartRow = currentRow;

    const finishRowRange = isGap => {
      if (isGap && !includeGaps) {
        nextStartRow = currentRow;
        return;
      }

      if (currentRow <= this.getRange().start.row) {
        return;
      }

      intersections.push({
        intersection: _atom.Range.fromObject([[nextStartRow, 0], [currentRow - 1, Infinity]]),
        gap: isGap
      });
      nextStartRow = currentRow;
    };

    while (currentRow <= this.getRange().end.row) {
      if (rowSet.has(currentRow) && !withinIntersection) {
        // One row past the end of a gap. Start of intersecting row range.
        finishRowRange(true);
        withinIntersection = true;
      } else if (!rowSet.has(currentRow) && withinIntersection) {
        // One row past the end of intersecting row range. Start of the next gap.
        finishRowRange(false);
        withinIntersection = false;
      }

      currentRow++;
    }

    finishRowRange(!withinIntersection);
    return intersections;
  }

  isAddition() {
    return false;
  }

  isDeletion() {
    return false;
  }

  isUnchanged() {
    return false;
  }

  isNoNewline() {
    return false;
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  when(callbacks) {
    const callback = callbacks[this.constructor.name.toLowerCase()] || callbacks["default"] || (() => undefined);

    return callback();
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;
  }

  destroyMarkers() {
    this.marker.destroy();
  }

  toStringIn(buffer) {
    const raw = buffer.getTextInRange(this.getRange());
    return this.constructor.origin + raw.replace(/\r?\n/g, '$&' + this.constructor.origin) + buffer.lineEndingForRow(this.getRange().end.row);
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread2({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(${this.constructor.name} marker=${this.marker.id})`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    return inspectString + '\n';
  }

  isChange() {
    return true;
  }

}

class Addition extends Region {
  isAddition() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Deletion(nextBuffer.markRange(this.getRange()));
  }

}

exports.Addition = Addition;

_defineProperty(Addition, "origin", '+');

_defineProperty(Addition, "layerName", 'addition');

class Deletion extends Region {
  isDeletion() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Addition(nextBuffer.markRange(this.getRange()));
  }

}

exports.Deletion = Deletion;

_defineProperty(Deletion, "origin", '-');

_defineProperty(Deletion, "layerName", 'deletion');

class Unchanged extends Region {
  isUnchanged() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new Unchanged(nextBuffer.markRange(this.getRange()));
  }

}

exports.Unchanged = Unchanged;

_defineProperty(Unchanged, "origin", ' ');

_defineProperty(Unchanged, "layerName", 'unchanged');

class NoNewline extends Region {
  isNoNewline() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new NoNewline(nextBuffer.markRange(this.getRange()));
  }

}

exports.NoNewline = NoNewline;

_defineProperty(NoNewline, "origin", '\\');

_defineProperty(NoNewline, "layerName", 'nonewline');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlZ2lvbi5qcyJdLCJuYW1lcyI6WyJSZWdpb24iLCJjb25zdHJ1Y3RvciIsIm1hcmtlciIsImdldE1hcmtlciIsImdldFJhbmdlIiwiZ2V0U3RhcnRCdWZmZXJSb3ciLCJzdGFydCIsInJvdyIsImdldEVuZEJ1ZmZlclJvdyIsImVuZCIsImluY2x1ZGVzQnVmZmVyUm93IiwiaW50ZXJzZWN0c1JvdyIsImludGVyc2VjdFJvd3MiLCJyb3dTZXQiLCJpbmNsdWRlR2FwcyIsImludGVyc2VjdGlvbnMiLCJ3aXRoaW5JbnRlcnNlY3Rpb24iLCJjdXJyZW50Um93IiwibmV4dFN0YXJ0Um93IiwiZmluaXNoUm93UmFuZ2UiLCJpc0dhcCIsInB1c2giLCJpbnRlcnNlY3Rpb24iLCJSYW5nZSIsImZyb21PYmplY3QiLCJJbmZpbml0eSIsImdhcCIsImhhcyIsImlzQWRkaXRpb24iLCJpc0RlbGV0aW9uIiwiaXNVbmNoYW5nZWQiLCJpc05vTmV3bGluZSIsImdldEJ1ZmZlclJvd3MiLCJnZXRSb3dzIiwiYnVmZmVyUm93Q291bnQiLCJnZXRSb3dDb3VudCIsIndoZW4iLCJjYWxsYmFja3MiLCJjYWxsYmFjayIsIm5hbWUiLCJ0b0xvd2VyQ2FzZSIsInVuZGVmaW5lZCIsInVwZGF0ZU1hcmtlcnMiLCJtYXAiLCJnZXQiLCJkZXN0cm95TWFya2VycyIsImRlc3Ryb3kiLCJ0b1N0cmluZ0luIiwiYnVmZmVyIiwicmF3IiwiZ2V0VGV4dEluUmFuZ2UiLCJvcmlnaW4iLCJyZXBsYWNlIiwibGluZUVuZGluZ0ZvclJvdyIsImluc3BlY3QiLCJvcHRzIiwib3B0aW9ucyIsImluZGVudCIsImluZGVudGF0aW9uIiwiaSIsImluc3BlY3RTdHJpbmciLCJpZCIsImlzRGVzdHJveWVkIiwiaXNWYWxpZCIsImlzQ2hhbmdlIiwiQWRkaXRpb24iLCJpbnZlcnRJbiIsIm5leHRCdWZmZXIiLCJEZWxldGlvbiIsIm1hcmtSYW5nZSIsIlVuY2hhbmdlZCIsIk5vTmV3bGluZSJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOzs7Ozs7QUFFQSxNQUFNQSxNQUFOLENBQWE7QUFDWEMsRUFBQUEsV0FBVyxDQUFDQyxNQUFELEVBQVM7QUFDbEIsU0FBS0EsTUFBTCxHQUFjQSxNQUFkO0FBQ0Q7O0FBRURDLEVBQUFBLFNBQVMsR0FBRztBQUNWLFdBQU8sS0FBS0QsTUFBWjtBQUNEOztBQUVERSxFQUFBQSxRQUFRLEdBQUc7QUFDVCxXQUFPLEtBQUtGLE1BQUwsQ0FBWUUsUUFBWixFQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLGlCQUFpQixHQUFHO0FBQ2xCLFdBQU8sS0FBS0QsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQTdCO0FBQ0Q7O0FBRURDLEVBQUFBLGVBQWUsR0FBRztBQUNoQixXQUFPLEtBQUtKLFFBQUwsR0FBZ0JLLEdBQWhCLENBQW9CRixHQUEzQjtBQUNEOztBQUVERyxFQUFBQSxpQkFBaUIsQ0FBQ0gsR0FBRCxFQUFNO0FBQ3JCLFdBQU8sS0FBS0gsUUFBTCxHQUFnQk8sYUFBaEIsQ0FBOEJKLEdBQTlCLENBQVA7QUFDRDtBQUVEOzs7Ozs7Ozs7Ozs7QUFVQUssRUFBQUEsYUFBYSxDQUFDQyxNQUFELEVBQVNDLFdBQVQsRUFBc0I7QUFDakMsVUFBTUMsYUFBYSxHQUFHLEVBQXRCO0FBQ0EsUUFBSUMsa0JBQWtCLEdBQUcsS0FBekI7QUFFQSxRQUFJQyxVQUFVLEdBQUcsS0FBS2IsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQXZDO0FBQ0EsUUFBSVcsWUFBWSxHQUFHRCxVQUFuQjs7QUFFQSxVQUFNRSxjQUFjLEdBQUdDLEtBQUssSUFBSTtBQUM5QixVQUFJQSxLQUFLLElBQUksQ0FBQ04sV0FBZCxFQUEyQjtBQUN6QkksUUFBQUEsWUFBWSxHQUFHRCxVQUFmO0FBQ0E7QUFDRDs7QUFFRCxVQUFJQSxVQUFVLElBQUksS0FBS2IsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQXhDLEVBQTZDO0FBQzNDO0FBQ0Q7O0FBRURRLE1BQUFBLGFBQWEsQ0FBQ00sSUFBZCxDQUFtQjtBQUNqQkMsUUFBQUEsWUFBWSxFQUFFQyxZQUFNQyxVQUFOLENBQWlCLENBQUMsQ0FBQ04sWUFBRCxFQUFlLENBQWYsQ0FBRCxFQUFvQixDQUFDRCxVQUFVLEdBQUcsQ0FBZCxFQUFpQlEsUUFBakIsQ0FBcEIsQ0FBakIsQ0FERztBQUVqQkMsUUFBQUEsR0FBRyxFQUFFTjtBQUZZLE9BQW5CO0FBS0FGLE1BQUFBLFlBQVksR0FBR0QsVUFBZjtBQUNELEtBaEJEOztBQWtCQSxXQUFPQSxVQUFVLElBQUksS0FBS2IsUUFBTCxHQUFnQkssR0FBaEIsQ0FBb0JGLEdBQXpDLEVBQThDO0FBQzVDLFVBQUlNLE1BQU0sQ0FBQ2MsR0FBUCxDQUFXVixVQUFYLEtBQTBCLENBQUNELGtCQUEvQixFQUFtRDtBQUNqRDtBQUNBRyxRQUFBQSxjQUFjLENBQUMsSUFBRCxDQUFkO0FBQ0FILFFBQUFBLGtCQUFrQixHQUFHLElBQXJCO0FBQ0QsT0FKRCxNQUlPLElBQUksQ0FBQ0gsTUFBTSxDQUFDYyxHQUFQLENBQVdWLFVBQVgsQ0FBRCxJQUEyQkQsa0JBQS9CLEVBQW1EO0FBQ3hEO0FBQ0FHLFFBQUFBLGNBQWMsQ0FBQyxLQUFELENBQWQ7QUFDQUgsUUFBQUEsa0JBQWtCLEdBQUcsS0FBckI7QUFDRDs7QUFFREMsTUFBQUEsVUFBVTtBQUNYOztBQUVERSxJQUFBQSxjQUFjLENBQUMsQ0FBQ0gsa0JBQUYsQ0FBZDtBQUNBLFdBQU9ELGFBQVA7QUFDRDs7QUFFRGEsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLFVBQVUsR0FBRztBQUNYLFdBQU8sS0FBUDtBQUNEOztBQUVEQyxFQUFBQSxXQUFXLEdBQUc7QUFDWixXQUFPLEtBQVA7QUFDRDs7QUFFREMsRUFBQUEsV0FBVyxHQUFHO0FBQ1osV0FBTyxLQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLGFBQWEsR0FBRztBQUNkLFdBQU8sS0FBSzVCLFFBQUwsR0FBZ0I2QixPQUFoQixFQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLGNBQWMsR0FBRztBQUNmLFdBQU8sS0FBSzlCLFFBQUwsR0FBZ0IrQixXQUFoQixFQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLElBQUksQ0FBQ0MsU0FBRCxFQUFZO0FBQ2QsVUFBTUMsUUFBUSxHQUFHRCxTQUFTLENBQUMsS0FBS3BDLFdBQUwsQ0FBaUJzQyxJQUFqQixDQUFzQkMsV0FBdEIsRUFBRCxDQUFULElBQWtESCxTQUFTLFdBQTNELEtBQXdFLE1BQU1JLFNBQTlFLENBQWpCOztBQUNBLFdBQU9ILFFBQVEsRUFBZjtBQUNEOztBQUVESSxFQUFBQSxhQUFhLENBQUNDLEdBQUQsRUFBTTtBQUNqQixTQUFLekMsTUFBTCxHQUFjeUMsR0FBRyxDQUFDQyxHQUFKLENBQVEsS0FBSzFDLE1BQWIsS0FBd0IsS0FBS0EsTUFBM0M7QUFDRDs7QUFFRDJDLEVBQUFBLGNBQWMsR0FBRztBQUNmLFNBQUszQyxNQUFMLENBQVk0QyxPQUFaO0FBQ0Q7O0FBRURDLEVBQUFBLFVBQVUsQ0FBQ0MsTUFBRCxFQUFTO0FBQ2pCLFVBQU1DLEdBQUcsR0FBR0QsTUFBTSxDQUFDRSxjQUFQLENBQXNCLEtBQUs5QyxRQUFMLEVBQXRCLENBQVo7QUFDQSxXQUFPLEtBQUtILFdBQUwsQ0FBaUJrRCxNQUFqQixHQUEwQkYsR0FBRyxDQUFDRyxPQUFKLENBQVksUUFBWixFQUFzQixPQUFPLEtBQUtuRCxXQUFMLENBQWlCa0QsTUFBOUMsQ0FBMUIsR0FDTEgsTUFBTSxDQUFDSyxnQkFBUCxDQUF3QixLQUFLakQsUUFBTCxHQUFnQkssR0FBaEIsQ0FBb0JGLEdBQTVDLENBREY7QUFFRDtBQUVEOzs7O0FBR0E7OztBQUNBK0MsRUFBQUEsT0FBTyxDQUFDQyxJQUFJLEdBQUcsRUFBUixFQUFZO0FBQ2pCLFVBQU1DLE9BQU87QUFDWEMsTUFBQUEsTUFBTSxFQUFFO0FBREcsT0FFUkYsSUFGUSxDQUFiOztBQUtBLFFBQUlHLFdBQVcsR0FBRyxFQUFsQjs7QUFDQSxTQUFLLElBQUlDLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUdILE9BQU8sQ0FBQ0MsTUFBNUIsRUFBb0NFLENBQUMsRUFBckMsRUFBeUM7QUFDdkNELE1BQUFBLFdBQVcsSUFBSSxHQUFmO0FBQ0Q7O0FBRUQsUUFBSUUsYUFBYSxHQUFJLEdBQUVGLFdBQVksSUFBRyxLQUFLekQsV0FBTCxDQUFpQnNDLElBQUssV0FBVSxLQUFLckMsTUFBTCxDQUFZMkQsRUFBRyxHQUFyRjs7QUFDQSxRQUFJLEtBQUszRCxNQUFMLENBQVk0RCxXQUFaLEVBQUosRUFBK0I7QUFDN0JGLE1BQUFBLGFBQWEsSUFBSSxjQUFqQjtBQUNEOztBQUNELFFBQUksQ0FBQyxLQUFLMUQsTUFBTCxDQUFZNkQsT0FBWixFQUFMLEVBQTRCO0FBQzFCSCxNQUFBQSxhQUFhLElBQUksWUFBakI7QUFDRDs7QUFDRCxXQUFPQSxhQUFhLEdBQUcsSUFBdkI7QUFDRDs7QUFFREksRUFBQUEsUUFBUSxHQUFHO0FBQ1QsV0FBTyxJQUFQO0FBQ0Q7O0FBcEpVOztBQXVKTixNQUFNQyxRQUFOLFNBQXVCakUsTUFBdkIsQ0FBOEI7QUFLbkM0QixFQUFBQSxVQUFVLEdBQUc7QUFDWCxXQUFPLElBQVA7QUFDRDs7QUFFRHNDLEVBQUFBLFFBQVEsQ0FBQ0MsVUFBRCxFQUFhO0FBQ25CLFdBQU8sSUFBSUMsUUFBSixDQUFhRCxVQUFVLENBQUNFLFNBQVgsQ0FBcUIsS0FBS2pFLFFBQUwsRUFBckIsQ0FBYixDQUFQO0FBQ0Q7O0FBWGtDOzs7O2dCQUF4QjZELFEsWUFDSyxHOztnQkFETEEsUSxlQUdRLFU7O0FBV2QsTUFBTUcsUUFBTixTQUF1QnBFLE1BQXZCLENBQThCO0FBS25DNkIsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxJQUFQO0FBQ0Q7O0FBRURxQyxFQUFBQSxRQUFRLENBQUNDLFVBQUQsRUFBYTtBQUNuQixXQUFPLElBQUlGLFFBQUosQ0FBYUUsVUFBVSxDQUFDRSxTQUFYLENBQXFCLEtBQUtqRSxRQUFMLEVBQXJCLENBQWIsQ0FBUDtBQUNEOztBQVhrQzs7OztnQkFBeEJnRSxRLFlBQ0ssRzs7Z0JBRExBLFEsZUFHUSxVOztBQVdkLE1BQU1FLFNBQU4sU0FBd0J0RSxNQUF4QixDQUErQjtBQUtwQzhCLEVBQUFBLFdBQVcsR0FBRztBQUNaLFdBQU8sSUFBUDtBQUNEOztBQUVEa0MsRUFBQUEsUUFBUSxHQUFHO0FBQ1QsV0FBTyxLQUFQO0FBQ0Q7O0FBRURFLEVBQUFBLFFBQVEsQ0FBQ0MsVUFBRCxFQUFhO0FBQ25CLFdBQU8sSUFBSUcsU0FBSixDQUFjSCxVQUFVLENBQUNFLFNBQVgsQ0FBcUIsS0FBS2pFLFFBQUwsRUFBckIsQ0FBZCxDQUFQO0FBQ0Q7O0FBZm1DOzs7O2dCQUF6QmtFLFMsWUFDSyxHOztnQkFETEEsUyxlQUdRLFc7O0FBZWQsTUFBTUMsU0FBTixTQUF3QnZFLE1BQXhCLENBQStCO0FBS3BDK0IsRUFBQUEsV0FBVyxHQUFHO0FBQ1osV0FBTyxJQUFQO0FBQ0Q7O0FBRURpQyxFQUFBQSxRQUFRLEdBQUc7QUFDVCxXQUFPLEtBQVA7QUFDRDs7QUFFREUsRUFBQUEsUUFBUSxDQUFDQyxVQUFELEVBQWE7QUFDbkIsV0FBTyxJQUFJSSxTQUFKLENBQWNKLFVBQVUsQ0FBQ0UsU0FBWCxDQUFxQixLQUFLakUsUUFBTCxFQUFyQixDQUFkLENBQVA7QUFDRDs7QUFmbUM7Ozs7Z0JBQXpCbUUsUyxZQUNLLEk7O2dCQURMQSxTLGVBR1EsVyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1JhbmdlfSBmcm9tICdhdG9tJztcblxuY2xhc3MgUmVnaW9uIHtcbiAgY29uc3RydWN0b3IobWFya2VyKSB7XG4gICAgdGhpcy5tYXJrZXIgPSBtYXJrZXI7XG4gIH1cblxuICBnZXRNYXJrZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMubWFya2VyO1xuICB9XG5cbiAgZ2V0UmFuZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMubWFya2VyLmdldFJhbmdlKCk7XG4gIH1cblxuICBnZXRTdGFydEJ1ZmZlclJvdygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLnN0YXJ0LnJvdztcbiAgfVxuXG4gIGdldEVuZEJ1ZmZlclJvdygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLmVuZC5yb3c7XG4gIH1cblxuICBpbmNsdWRlc0J1ZmZlclJvdyhyb3cpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLmludGVyc2VjdHNSb3cocm93KTtcbiAgfVxuXG4gIC8qXG4gICAqIGludGVyc2VjdFJvd3MgYnJlYWtzIGEgUmVnaW9uIGludG8gcnVucyBvZiByb3dzIHRoYXQgYXJlIGluY2x1ZGVkIGluXG4gICAqIHJvd1NldCBhbmQgcm93cyB0aGF0IGFyZSBub3QuIEZvciBleGFtcGxlOlxuICAgKiAgQHRoaXMgUmVnaW9uICAgICAgICByb3cgMTAtMjBcbiAgICogIEBwYXJhbSByb3dTZXQgICAgICAgcm93IDExLCAxMiwgMTMsIDE3LCAxOVxuICAgKiAgQHBhcmFtIGluY2x1ZGVHYXBzICB0cnVlICh3aGV0aGVyIHRoZSByZXN1bHQgd2lsbCBpbmNsdWRlIGdhcHMgb3Igbm90KVxuICAgKiAgQHJldHVybiBhbiBhcnJheSBvZiByZWdpb25zIGxpa2UgdGhpczpcbiAgICogICAgKDEwLCBnYXAgPSB0cnVlKSAoMTEsIDEyLCAxMywgZ2FwID0gZmFsc2UpICgxNCwgMTUsIDE2LCBnYXAgPSB0cnVlKVxuICAgKiAgICAoMTcsIGdhcCA9IGZhbHNlKSAoMTgsIGdhcCA9IHRydWUpICgxOSwgZ2FwID0gZmFsc2UpICgyMCwgZ2FwID0gdHJ1ZSlcbiAgICovXG4gIGludGVyc2VjdFJvd3Mocm93U2V0LCBpbmNsdWRlR2Fwcykge1xuICAgIGNvbnN0IGludGVyc2VjdGlvbnMgPSBbXTtcbiAgICBsZXQgd2l0aGluSW50ZXJzZWN0aW9uID0gZmFsc2U7XG5cbiAgICBsZXQgY3VycmVudFJvdyA9IHRoaXMuZ2V0UmFuZ2UoKS5zdGFydC5yb3c7XG4gICAgbGV0IG5leHRTdGFydFJvdyA9IGN1cnJlbnRSb3c7XG5cbiAgICBjb25zdCBmaW5pc2hSb3dSYW5nZSA9IGlzR2FwID0+IHtcbiAgICAgIGlmIChpc0dhcCAmJiAhaW5jbHVkZUdhcHMpIHtcbiAgICAgICAgbmV4dFN0YXJ0Um93ID0gY3VycmVudFJvdztcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICBpZiAoY3VycmVudFJvdyA8PSB0aGlzLmdldFJhbmdlKCkuc3RhcnQucm93KSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgaW50ZXJzZWN0aW9ucy5wdXNoKHtcbiAgICAgICAgaW50ZXJzZWN0aW9uOiBSYW5nZS5mcm9tT2JqZWN0KFtbbmV4dFN0YXJ0Um93LCAwXSwgW2N1cnJlbnRSb3cgLSAxLCBJbmZpbml0eV1dKSxcbiAgICAgICAgZ2FwOiBpc0dhcCxcbiAgICAgIH0pO1xuXG4gICAgICBuZXh0U3RhcnRSb3cgPSBjdXJyZW50Um93O1xuICAgIH07XG5cbiAgICB3aGlsZSAoY3VycmVudFJvdyA8PSB0aGlzLmdldFJhbmdlKCkuZW5kLnJvdykge1xuICAgICAgaWYgKHJvd1NldC5oYXMoY3VycmVudFJvdykgJiYgIXdpdGhpbkludGVyc2VjdGlvbikge1xuICAgICAgICAvLyBPbmUgcm93IHBhc3QgdGhlIGVuZCBvZiBhIGdhcC4gU3RhcnQgb2YgaW50ZXJzZWN0aW5nIHJvdyByYW5nZS5cbiAgICAgICAgZmluaXNoUm93UmFuZ2UodHJ1ZSk7XG4gICAgICAgIHdpdGhpbkludGVyc2VjdGlvbiA9IHRydWU7XG4gICAgICB9IGVsc2UgaWYgKCFyb3dTZXQuaGFzKGN1cnJlbnRSb3cpICYmIHdpdGhpbkludGVyc2VjdGlvbikge1xuICAgICAgICAvLyBPbmUgcm93IHBhc3QgdGhlIGVuZCBvZiBpbnRlcnNlY3Rpbmcgcm93IHJhbmdlLiBTdGFydCBvZiB0aGUgbmV4dCBnYXAuXG4gICAgICAgIGZpbmlzaFJvd1JhbmdlKGZhbHNlKTtcbiAgICAgICAgd2l0aGluSW50ZXJzZWN0aW9uID0gZmFsc2U7XG4gICAgICB9XG5cbiAgICAgIGN1cnJlbnRSb3crKztcbiAgICB9XG5cbiAgICBmaW5pc2hSb3dSYW5nZSghd2l0aGluSW50ZXJzZWN0aW9uKTtcbiAgICByZXR1cm4gaW50ZXJzZWN0aW9ucztcbiAgfVxuXG4gIGlzQWRkaXRpb24oKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgaXNEZWxldGlvbigpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBpc1VuY2hhbmdlZCgpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBpc05vTmV3bGluZSgpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBnZXRCdWZmZXJSb3dzKCkge1xuICAgIHJldHVybiB0aGlzLmdldFJhbmdlKCkuZ2V0Um93cygpO1xuICB9XG5cbiAgYnVmZmVyUm93Q291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UmFuZ2UoKS5nZXRSb3dDb3VudCgpO1xuICB9XG5cbiAgd2hlbihjYWxsYmFja3MpIHtcbiAgICBjb25zdCBjYWxsYmFjayA9IGNhbGxiYWNrc1t0aGlzLmNvbnN0cnVjdG9yLm5hbWUudG9Mb3dlckNhc2UoKV0gfHwgY2FsbGJhY2tzLmRlZmF1bHQgfHwgKCgpID0+IHVuZGVmaW5lZCk7XG4gICAgcmV0dXJuIGNhbGxiYWNrKCk7XG4gIH1cblxuICB1cGRhdGVNYXJrZXJzKG1hcCkge1xuICAgIHRoaXMubWFya2VyID0gbWFwLmdldCh0aGlzLm1hcmtlcikgfHwgdGhpcy5tYXJrZXI7XG4gIH1cblxuICBkZXN0cm95TWFya2VycygpIHtcbiAgICB0aGlzLm1hcmtlci5kZXN0cm95KCk7XG4gIH1cblxuICB0b1N0cmluZ0luKGJ1ZmZlcikge1xuICAgIGNvbnN0IHJhdyA9IGJ1ZmZlci5nZXRUZXh0SW5SYW5nZSh0aGlzLmdldFJhbmdlKCkpO1xuICAgIHJldHVybiB0aGlzLmNvbnN0cnVjdG9yLm9yaWdpbiArIHJhdy5yZXBsYWNlKC9cXHI/XFxuL2csICckJicgKyB0aGlzLmNvbnN0cnVjdG9yLm9yaWdpbikgK1xuICAgICAgYnVmZmVyLmxpbmVFbmRpbmdGb3JSb3codGhpcy5nZXRSYW5nZSgpLmVuZC5yb3cpO1xuICB9XG5cbiAgLypcbiAgICogQ29uc3RydWN0IGEgU3RyaW5nIGNvbnRhaW5pbmcgaW50ZXJuYWwgZGlhZ25vc3RpYyBpbmZvcm1hdGlvbi5cbiAgICovXG4gIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gIGluc3BlY3Qob3B0cyA9IHt9KSB7XG4gICAgY29uc3Qgb3B0aW9ucyA9IHtcbiAgICAgIGluZGVudDogMCxcbiAgICAgIC4uLm9wdHMsXG4gICAgfTtcblxuICAgIGxldCBpbmRlbnRhdGlvbiA9ICcnO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgb3B0aW9ucy5pbmRlbnQ7IGkrKykge1xuICAgICAgaW5kZW50YXRpb24gKz0gJyAnO1xuICAgIH1cblxuICAgIGxldCBpbnNwZWN0U3RyaW5nID0gYCR7aW5kZW50YXRpb259KCR7dGhpcy5jb25zdHJ1Y3Rvci5uYW1lfSBtYXJrZXI9JHt0aGlzLm1hcmtlci5pZH0pYDtcbiAgICBpZiAodGhpcy5tYXJrZXIuaXNEZXN0cm95ZWQoKSkge1xuICAgICAgaW5zcGVjdFN0cmluZyArPSAnIFtkZXN0cm95ZWRdJztcbiAgICB9XG4gICAgaWYgKCF0aGlzLm1hcmtlci5pc1ZhbGlkKCkpIHtcbiAgICAgIGluc3BlY3RTdHJpbmcgKz0gJyBbaW52YWxpZF0nO1xuICAgIH1cbiAgICByZXR1cm4gaW5zcGVjdFN0cmluZyArICdcXG4nO1xuICB9XG5cbiAgaXNDaGFuZ2UoKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIEFkZGl0aW9uIGV4dGVuZHMgUmVnaW9uIHtcbiAgc3RhdGljIG9yaWdpbiA9ICcrJztcblxuICBzdGF0aWMgbGF5ZXJOYW1lID0gJ2FkZGl0aW9uJztcblxuICBpc0FkZGl0aW9uKCkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgaW52ZXJ0SW4obmV4dEJ1ZmZlcikge1xuICAgIHJldHVybiBuZXcgRGVsZXRpb24obmV4dEJ1ZmZlci5tYXJrUmFuZ2UodGhpcy5nZXRSYW5nZSgpKSk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIERlbGV0aW9uIGV4dGVuZHMgUmVnaW9uIHtcbiAgc3RhdGljIG9yaWdpbiA9ICctJztcblxuICBzdGF0aWMgbGF5ZXJOYW1lID0gJ2RlbGV0aW9uJztcblxuICBpc0RlbGV0aW9uKCkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgaW52ZXJ0SW4obmV4dEJ1ZmZlcikge1xuICAgIHJldHVybiBuZXcgQWRkaXRpb24obmV4dEJ1ZmZlci5tYXJrUmFuZ2UodGhpcy5nZXRSYW5nZSgpKSk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIFVuY2hhbmdlZCBleHRlbmRzIFJlZ2lvbiB7XG4gIHN0YXRpYyBvcmlnaW4gPSAnICc7XG5cbiAgc3RhdGljIGxheWVyTmFtZSA9ICd1bmNoYW5nZWQnO1xuXG4gIGlzVW5jaGFuZ2VkKCkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgaXNDaGFuZ2UoKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgaW52ZXJ0SW4obmV4dEJ1ZmZlcikge1xuICAgIHJldHVybiBuZXcgVW5jaGFuZ2VkKG5leHRCdWZmZXIubWFya1JhbmdlKHRoaXMuZ2V0UmFuZ2UoKSkpO1xuICB9XG59XG5cbmV4cG9ydCBjbGFzcyBOb05ld2xpbmUgZXh0ZW5kcyBSZWdpb24ge1xuICBzdGF0aWMgb3JpZ2luID0gJ1xcXFwnO1xuXG4gIHN0YXRpYyBsYXllck5hbWUgPSAnbm9uZXdsaW5lJztcblxuICBpc05vTmV3bGluZSgpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIGlzQ2hhbmdlKCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGludmVydEluKG5leHRCdWZmZXIpIHtcbiAgICByZXR1cm4gbmV3IE5vTmV3bGluZShuZXh0QnVmZmVyLm1hcmtSYW5nZSh0aGlzLmdldFJhbmdlKCkpKTtcbiAgfVxufVxuIl19