"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _atom = require("atom");

var _util = require("util");

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LAYER_NAMES = ['unchanged', 'addition', 'deletion', 'nonewline', 'hunk', 'patch'];

class PatchBuffer {
  constructor() {
    this.buffer = new _atom.TextBuffer();
    this.buffer.retain();
    this.layers = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.buffer.addMarkerLayer();
      return map;
    }, {});
  }

  getBuffer() {
    return this.buffer;
  }

  getInsertionPoint() {
    return this.buffer.getEndPosition();
  }

  getLayer(layerName) {
    return this.layers[layerName];
  }

  findMarkers(layerName, ...args) {
    return this.layers[layerName].findMarkers(...args);
  }

  findAllMarkers(...args) {
    return LAYER_NAMES.reduce((arr, layerName) => {
      arr.push(...this.findMarkers(layerName, ...args));
      return arr;
    }, []);
  }

  markPosition(layerName, ...args) {
    return this.layers[layerName].markPosition(...args);
  }

  markRange(layerName, ...args) {
    return this.layers[layerName].markRange(...args);
  }

  clearAllLayers() {
    for (const layerName of LAYER_NAMES) {
      this.layers[layerName].clear();
    }
  }

  createInserterAt(insertionPoint) {
    return new Inserter(this, _atom.Point.fromObject(insertionPoint));
  }

  createInserterAtEnd() {
    return this.createInserterAt(this.getInsertionPoint());
  }

  createSubBuffer(rangeLike, options = {}) {
    const opts = _objectSpread2({
      exclude: new Set()
    }, options);

    const range = _atom.Range.fromObject(rangeLike);

    const baseOffset = range.start.negate();
    const includedMarkersByLayer = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.layers[layerName].findMarkers({
        intersectsRange: range
      }).filter(m => !opts.exclude.has(m));
      return map;
    }, {});
    const markerMap = new Map();
    const subBuffer = new PatchBuffer();
    subBuffer.getBuffer().setText(this.buffer.getTextInRange(range));

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of includedMarkersByLayer[layerName]) {
        const oldRange = oldMarker.getRange();
        const clippedStart = oldRange.start.isLessThanOrEqual(range.start) ? range.start : oldRange.start;
        const clippedEnd = oldRange.end.isGreaterThanOrEqual(range.end) ? range.end : oldRange.end; // Exclude non-empty markers that intersect *only* at the range start or end

        if (clippedStart.isEqual(clippedEnd) && !oldRange.start.isEqual(oldRange.end)) {
          continue;
        }

        const startOffset = clippedStart.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const endOffset = clippedEnd.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const newMarker = subBuffer.markRange(layerName, [clippedStart.translate(startOffset), clippedEnd.translate(endOffset)], oldMarker.getProperties());
        markerMap.set(oldMarker, newMarker);
      }
    }

    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  extractPatchBuffer(rangeLike, options = {}) {
    const {
      patchBuffer: subBuffer,
      markerMap
    } = this.createSubBuffer(rangeLike, options);

    for (const oldMarker of markerMap.keys()) {
      oldMarker.destroy();
    }

    this.buffer.setTextInRange(rangeLike, '');
    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  deleteLastNewline() {
    if (this.buffer.getLastLine() === '') {
      this.buffer.deleteRow(this.buffer.getLastRow());
    }

    return this;
  }

  adopt(original) {
    this.clearAllLayers();
    this.buffer.setText(original.getBuffer().getText());
    const markerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const originalMarker of original.getLayer(layerName).getMarkers()) {
        const newMarker = this.markRange(layerName, originalMarker.getRange(), originalMarker.getProperties());
        markerMap.set(originalMarker, newMarker);
      }
    }

    return markerMap;
  }
  /* istanbul ignore next */


  inspect(opts = {}) {
    /* istanbul ignore next */
    const options = _objectSpread2({
      layerNames: LAYER_NAMES
    }, opts);

    let inspectString = '';
    const increasingMarkers = [];

    for (const layerName of options.layerNames) {
      for (const marker of this.findMarkers(layerName, {})) {
        increasingMarkers.push({
          layerName,
          point: marker.getRange().start,
          start: true,
          id: marker.id
        });
        increasingMarkers.push({
          layerName,
          point: marker.getRange().end,
          end: true,
          id: marker.id
        });
      }
    }

    increasingMarkers.sort((a, b) => {
      const cmp = a.point.compare(b.point);

      if (cmp !== 0) {
        return cmp;
      } else if (a.start && b.start) {
        return 0;
      } else if (a.start && !b.start) {
        return -1;
      } else if (!a.start && b.start) {
        return 1;
      } else {
        return 0;
      }
    });

    let inspectPoint = _atom.Point.fromObject([0, 0]);

    for (const marker of increasingMarkers) {
      if (!marker.point.isEqual(inspectPoint)) {
        inspectString += (0, _util.inspect)(this.buffer.getTextInRange([inspectPoint, marker.point])) + '\n';
      }

      if (marker.start) {
        inspectString += `  start ${marker.layerName}@${marker.id}\n`;
      } else if (marker.end) {
        inspectString += `  end ${marker.layerName}@${marker.id}\n`;
      }

      inspectPoint = marker.point;
    }

    return inspectString;
  }

}

exports["default"] = PatchBuffer;

class Inserter {
  constructor(patchBuffer, insertionPoint) {
    const clipped = patchBuffer.getBuffer().clipPosition(insertionPoint);
    this.patchBuffer = patchBuffer;
    this.startPoint = clipped.copy();
    this.insertionPoint = clipped.copy();
    this.markerBlueprints = [];
    this.markerMapCallbacks = [];
    this.markersBefore = new Set();
    this.markersAfter = new Set();
  }

  keepBefore(markers) {
    for (const marker of markers) {
      if (marker.getRange().end.isEqual(this.startPoint)) {
        this.markersBefore.add(marker);
      }
    }

    return this;
  }

  keepAfter(markers) {
    for (const marker of markers) {
      if (marker.getRange().start.isEqual(this.startPoint)) {
        this.markersAfter.add(marker);
      }
    }

    return this;
  }

  markWhile(layerName, block, markerOpts) {
    const start = this.insertionPoint.copy();
    block();
    const end = this.insertionPoint.copy();
    this.markerBlueprints.push({
      layerName,
      range: new _atom.Range(start, end),
      markerOpts
    });
    return this;
  }

  insert(text) {
    const insertedRange = this.patchBuffer.getBuffer().insert(this.insertionPoint, text);
    this.insertionPoint = insertedRange.end;
    return this;
  }

  insertMarked(text, layerName, markerOpts) {
    return this.markWhile(layerName, () => this.insert(text), markerOpts);
  }

  insertPatchBuffer(subPatchBuffer, opts) {
    const baseOffset = this.insertionPoint.copy();
    this.insert(subPatchBuffer.getBuffer().getText());
    const subMarkerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of subPatchBuffer.findMarkers(layerName, {})) {
        const startOffset = oldMarker.getRange().start.row === 0 ? baseOffset : [baseOffset.row, 0];
        const endOffset = oldMarker.getRange().end.row === 0 ? baseOffset : [baseOffset.row, 0];
        const range = oldMarker.getRange().translate(startOffset, endOffset);

        const markerOpts = _objectSpread2({}, oldMarker.getProperties(), {
          callback: newMarker => {
            subMarkerMap.set(oldMarker, newMarker);
          }
        });

        this.markerBlueprints.push({
          layerName,
          range,
          markerOpts
        });
      }
    }

    this.markerMapCallbacks.push({
      markerMap: subMarkerMap,
      callback: opts.callback
    });
    return this;
  }

  apply() {
    for (const _ref of this.markerBlueprints) {
      const {
        layerName,
        range,
        markerOpts
      } = _ref;
      const callback = markerOpts.callback;
      delete markerOpts.callback;
      const marker = this.patchBuffer.markRange(layerName, range, markerOpts);

      if (callback) {
        callback(marker);
      }
    }

    for (const _ref2 of this.markerMapCallbacks) {
      const {
        markerMap,
        callback
      } = _ref2;
      callback(markerMap);
    }

    for (const beforeMarker of this.markersBefore) {
      const isEmpty = beforeMarker.getRange().isEmpty();

      if (!beforeMarker.isReversed()) {
        beforeMarker.setHeadPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setTailPosition(this.startPoint);
        }
      } else {
        beforeMarker.setTailPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setHeadPosition(this.startPoint);
        }
      }
    }

    for (const afterMarker of this.markersAfter) {
      const isEmpty = afterMarker.getRange().isEmpty();

      if (!afterMarker.isReversed()) {
        afterMarker.setTailPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setHeadPosition(this.insertionPoint);
        }
      } else {
        afterMarker.setHeadPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setTailPosition(this.insertionPoint);
        }
      }
    }
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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