"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildStatusFromStatusContext = buildStatusFromStatusContext;
exports.buildStatusFromCheckResult = buildStatusFromCheckResult;
exports.combineBuildStatuses = combineBuildStatuses;
// Commit or pull request build status, unified from those derived from the Checks API and the Status API.
const DEFAULT = {
  icon: 'unverified',
  classSuffix: 'pending'
};
const PENDING = {
  icon: 'primitive-dot',
  classSuffix: 'pending'
};
const SUCCESS = {
  icon: 'check',
  classSuffix: 'success'
};
const FAILURE = {
  icon: 'x',
  classSuffix: 'failure'
};
const ERROR = {
  icon: 'alert',
  classSuffix: 'failure'
};
const ACTION_REQUIRED = {
  icon: 'bell',
  classSuffix: 'failure'
};
const NEUTRAL = {
  icon: 'dash',
  classSuffix: 'neutral'
};
const STATUS_CONTEXT_MAP = {
  EXPECTED: PENDING,
  PENDING,
  SUCCESS,
  ERROR,
  FAILURE
};

function buildStatusFromStatusContext({
  state
}) {
  return STATUS_CONTEXT_MAP[state] || DEFAULT;
}

const PENDING_CHECK_STATUSES = new Set(['QUEUED', 'IN_PROGRESS', 'REQUESTED']);
const COMPLETED_CHECK_CONCLUSION_MAP = {
  SUCCESS,
  FAILURE,
  TIMED_OUT: ERROR,
  CANCELLED: ERROR,
  ACTION_REQUIRED,
  NEUTRAL
};

function buildStatusFromCheckResult({
  status,
  conclusion
}) {
  if (PENDING_CHECK_STATUSES.has(status)) {
    return PENDING;
  } else if (status === 'COMPLETED') {
    return COMPLETED_CHECK_CONCLUSION_MAP[conclusion] || DEFAULT;
  } else {
    return DEFAULT;
  }
}

const STATUS_PRIORITY = [DEFAULT, NEUTRAL, SUCCESS, PENDING, FAILURE, ERROR, ACTION_REQUIRED];

function combineBuildStatuses(...statuses) {
  let highestPriority = 0;
  let highestPriorityStatus = NEUTRAL;

  for (const status of statuses) {
    const priority = STATUS_PRIORITY.indexOf(status);

    if (priority > highestPriority) {
      highestPriority = priority;
      highestPriorityStatus = status;
    }
  }

  return highestPriorityStatus;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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