"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autobind = autobind;
exports.extractProps = extractProps;
exports.unusedProps = unusedProps;
exports.getPackageRoot = getPackageRoot;
exports.getAtomHelperPath = getAtomHelperPath;
exports.getDugitePath = getDugitePath;
exports.getSharedModulePath = getSharedModulePath;
exports.isBinary = isBinary;
exports.firstImplementer = firstImplementer;
exports.isValidWorkdir = isValidWorkdir;
exports.fileExists = fileExists;
exports.getTempDir = getTempDir;
exports.isFileExecutable = isFileExecutable;
exports.isFileSymlink = isFileSymlink;
exports.shortenSha = shortenSha;
exports.normalizeGitHelperPath = normalizeGitHelperPath;
exports.toNativePathSep = toNativePathSep;
exports.toGitPathSep = toGitPathSep;
exports.filePathEndsWith = filePathEndsWith;
exports.toSentence = toSentence;
exports.pushAtKey = pushAtKey;
exports.getCommitMessagePath = getCommitMessagePath;
exports.getCommitMessageEditors = getCommitMessageEditors;
exports.getFilePatchPaneItems = getFilePatchPaneItems;
exports.destroyFilePatchPaneItems = destroyFilePatchPaneItems;
exports.destroyEmptyFilePatchPaneItems = destroyEmptyFilePatchPaneItems;
exports.extractCoAuthorsAndRawCommitMessage = extractCoAuthorsAndRawCommitMessage;
exports.createItem = createItem;
exports.equalSets = equalSets;
exports.blankLabel = blankLabel;
exports.renderMarkdown = renderMarkdown;
exports.GHOST_USER = exports.reactionTypeToEmoji = exports.NBSP_CHARACTER = exports.classNameForStatus = exports.CHECK_RUN_PAGE_SIZE = exports.CHECK_SUITE_PAGE_SIZE = exports.PAGINATION_WAIT_TIME_MS = exports.PAGE_SIZE = exports.CO_AUTHOR_REGEX = exports.LINE_ENDING_REGEX = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _os = _interopRequireDefault(require("os"));

var _temp = _interopRequireDefault(require("temp"));

var _refHolder = _interopRequireDefault(require("./models/ref-holder"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LINE_ENDING_REGEX = /\r?\n/;
exports.LINE_ENDING_REGEX = LINE_ENDING_REGEX;
const CO_AUTHOR_REGEX = /^co-authored-by. (.+?) <(.+?)>$/i;
exports.CO_AUTHOR_REGEX = CO_AUTHOR_REGEX;
const PAGE_SIZE = 50;
exports.PAGE_SIZE = PAGE_SIZE;
const PAGINATION_WAIT_TIME_MS = 100;
exports.PAGINATION_WAIT_TIME_MS = PAGINATION_WAIT_TIME_MS;
const CHECK_SUITE_PAGE_SIZE = 10;
exports.CHECK_SUITE_PAGE_SIZE = CHECK_SUITE_PAGE_SIZE;
const CHECK_RUN_PAGE_SIZE = 20;
exports.CHECK_RUN_PAGE_SIZE = CHECK_RUN_PAGE_SIZE;

function autobind(self, ...methods) {
  for (const method of methods) {
    if (typeof self[method] !== 'function') {
      throw new Error(`Unable to autobind method ${method}`);
    }

    self[method] = self[method].bind(self);
  }
} // Extract a subset of props chosen from a propTypes object from a component's props to pass to a different API.
//
// Usage:
//
// ```js
// const apiProps = {
//   zero: PropTypes.number.isRequired,
//   one: PropTypes.string,
//   two: PropTypes.object,
// };
//
// class Component extends React.Component {
//   static propTypes = {
//     ...apiProps,
//     extra: PropTypes.func,
//   }
//
//   action() {
//     const options = extractProps(this.props, apiProps);
//     // options contains zero, one, and two, but not extra
//   }
// }
// ```


function extractProps(props, propTypes, nameMap = {}) {
  return Object.keys(propTypes).reduce((opts, propName) => {
    if (props[propName] !== undefined) {
      const destPropName = nameMap[propName] || propName;
      opts[destPropName] = props[propName];
    }

    return opts;
  }, {});
} // The opposite of extractProps. Return a subset of props that do *not* appear in a component's prop types.


function unusedProps(props, propTypes) {
  return Object.keys(props).reduce((opts, propName) => {
    if (propTypes[propName] === undefined) {
      opts[propName] = props[propName];
    }

    return opts;
  }, {});
}

function getPackageRoot() {
  const {
    resourcePath
  } = atom.getLoadSettings();
  const currentFileWasRequiredFromSnapshot = !_path["default"].isAbsolute(__dirname);

  if (currentFileWasRequiredFromSnapshot) {
    return _path["default"].join(resourcePath, 'node_modules', 'github');
  } else {
    const packageRoot = _path["default"].resolve(__dirname, '..');

    if (_path["default"].extname(resourcePath) === '.asar') {
      if (packageRoot.indexOf(resourcePath) === 0) {
        return _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'github');
      }
    }

    return packageRoot;
  }
}

function getAtomAppName() {
  const match = atom.getVersion().match(/-([A-Za-z]+)(\d+|-)/);

  if (match) {
    const channel = match[1];
    return `Atom ${channel.charAt(0).toUpperCase() + channel.slice(1)} Helper`;
  }

  return 'Atom Helper';
}

function getAtomHelperPath() {
  if (process.platform === 'darwin') {
    const appName = getAtomAppName();
    return _path["default"].resolve(process.resourcesPath, '..', 'Frameworks', `${appName}.app`, 'Contents', 'MacOS', appName);
  } else {
    return process.execPath;
  }
}

let DUGITE_PATH;

function getDugitePath() {
  if (!DUGITE_PATH) {
    DUGITE_PATH = require.resolve('dugite');

    if (!_path["default"].isAbsolute(DUGITE_PATH)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();

      if (_path["default"].extname(resourcePath) === '.asar') {
        DUGITE_PATH = _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'dugite');
      } else {
        DUGITE_PATH = _path["default"].join(resourcePath, 'node_modules', 'dugite');
      }
    }
  }

  return DUGITE_PATH;
}

const SHARED_MODULE_PATHS = new Map();

function getSharedModulePath(relPath) {
  let modulePath = SHARED_MODULE_PATHS.get(relPath);

  if (!modulePath) {
    modulePath = require.resolve(_path["default"].join(__dirname, 'shared', relPath));

    if (!_path["default"].isAbsolute(modulePath)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();
      modulePath = _path["default"].join(resourcePath, modulePath);
    }

    SHARED_MODULE_PATHS.set(relPath, modulePath);
  }

  return modulePath;
}

function isBinary(data) {
  for (let i = 0; i < 50; i++) {
    const code = data.charCodeAt(i); // Char code 65533 is the "replacement character";
    // 8 and below are control characters.

    if (code === 65533 || code < 9) {
      return true;
    }
  }

  return false;
}

function descriptorsFromProto(proto) {
  return Object.getOwnPropertyNames(proto).reduce((acc, name) => {
    Object.assign(acc, {
      [name]: Reflect.getOwnPropertyDescriptor(proto, name)
    });
    return acc;
  }, {});
}
/**
 * Takes an array of targets and returns a proxy. The proxy intercepts property accessor calls and
 * returns the value of that property on the first object in `targets` where the target implements that property.
 */


function firstImplementer(...targets) {
  return new Proxy({
    __implementations: targets
  }, {
    get(target, name) {
      if (name === 'getImplementers') {
        return () => targets;
      }

      if (Reflect.has(target, name)) {
        return target[name];
      }

      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        return firstValidTarget[name];
      } else {
        return undefined;
      }
    },

    set(target, name, value) {
      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        // eslint-disable-next-line no-return-assign
        return firstValidTarget[name] = value;
      } else {
        // eslint-disable-next-line no-return-assign
        return target[name] = value;
      }
    },

    // Used by sinon
    has(target, name) {
      if (name === 'getImplementers') {
        return true;
      }

      return targets.some(t => Reflect.has(t, name));
    },

    // Used by sinon
    getOwnPropertyDescriptor(target, name) {
      const firstValidTarget = targets.find(t => Reflect.getOwnPropertyDescriptor(t, name));
      const compositeOwnPropertyDescriptor = Reflect.getOwnPropertyDescriptor(target, name);

      if (firstValidTarget) {
        return Reflect.getOwnPropertyDescriptor(firstValidTarget, name);
      } else if (compositeOwnPropertyDescriptor) {
        return compositeOwnPropertyDescriptor;
      } else {
        return undefined;
      }
    },

    // Used by sinon
    getPrototypeOf(target) {
      return targets.reduceRight((acc, t) => {
        return Object.create(acc, descriptorsFromProto(Object.getPrototypeOf(t)));
      }, Object.prototype);
    }

  });
}

function isRoot(dir) {
  return _path["default"].resolve(dir, '..') === dir;
}

function isValidWorkdir(dir) {
  return dir !== _os["default"].homedir() && !isRoot(dir);
}

async function fileExists(absoluteFilePath) {
  try {
    await _fsExtra["default"].access(absoluteFilePath);
    return true;
  } catch (e) {
    if (e.code === 'ENOENT') {
      return false;
    }

    throw e;
  }
}

function getTempDir(options = {}) {
  _temp["default"].track();

  return new Promise((resolve, reject) => {
    _temp["default"].mkdir(options, (tempError, folder) => {
      if (tempError) {
        reject(tempError);
        return;
      }

      if (options.symlinkOk) {
        resolve(folder);
      } else {
        _fsExtra["default"].realpath(folder, (realError, rpath) => realError ? reject(realError) : resolve(rpath));
      }
    });
  });
}

async function isFileExecutable(absoluteFilePath) {
  const stat = await _fsExtra["default"].stat(absoluteFilePath);
  return stat.mode & _fsExtra["default"].constants.S_IXUSR; // eslint-disable-line no-bitwise
}

async function isFileSymlink(absoluteFilePath) {
  const stat = await _fsExtra["default"].lstat(absoluteFilePath);
  return stat.isSymbolicLink();
}

function shortenSha(sha) {
  return sha.slice(0, 8);
}

const classNameForStatus = {
  added: 'added',
  deleted: 'removed',
  modified: 'modified',
  typechange: 'modified',
  equivalent: 'ignored'
};
/*
 * Apply any platform-specific munging to a path before presenting it as
 * a git environment variable or option.
 *
 * Convert a Windows-style "C:\foo\bar\baz" path to a "/c/foo/bar/baz" UNIX-y
 * path that the sh.exe used to execute git's credential helpers will
 * understand.
 */

exports.classNameForStatus = classNameForStatus;

function normalizeGitHelperPath(inPath) {
  if (process.platform === 'win32') {
    return inPath.replace(/\\/g, '/').replace(/^([^:]+):/, '/$1');
  } else {
    return inPath;
  }
}
/*
 * On Windows, git commands report paths with / delimiters. Convert them to \-delimited paths
 * so that Atom unifromly treats paths with native path separators.
 */


function toNativePathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split('/').join(_path["default"].sep);
  }
}
/*
 * Convert Windows paths back to /-delimited paths to be presented to git.
 */


function toGitPathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split(_path["default"].sep).join('/');
  }
}

function filePathEndsWith(filePath, ...segments) {
  return filePath.endsWith(_path["default"].join(...segments));
}
/**
 * Turns an array of things @kuychaco cannot eat
 * into a sentence containing things @kuychaco cannot eat
 *
 * ['toast'] => 'toast'
 * ['toast', 'eggs'] => 'toast and eggs'
 * ['toast', 'eggs', 'cheese'] => 'toast, eggs, and cheese'
 *
 * Oxford comma included because you're wrong, shut up.
 */


function toSentence(array) {
  const len = array.length;

  if (len === 1) {
    return `${array[0]}`;
  } else if (len === 2) {
    return `${array[0]} and ${array[1]}`;
  }

  return array.reduce((acc, item, idx) => {
    if (idx === 0) {
      return `${item}`;
    } else if (idx === len - 1) {
      return `${acc}, and ${item}`;
    } else {
      return `${acc}, ${item}`;
    }
  }, '');
}

function pushAtKey(map, key, value) {
  let existing = map.get(key);

  if (!existing) {
    existing = [];
    map.set(key, existing);
  }

  existing.push(value);
} // Repository and workspace helpers


function getCommitMessagePath(repository) {
  return _path["default"].join(repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
}

function getCommitMessageEditors(repository, workspace) {
  if (!repository.isPresent()) {
    return [];
  }

  return workspace.getTextEditors().filter(editor => editor.getPath() === getCommitMessagePath(repository));
}

let ChangedFileItem = null;

function getFilePatchPaneItems({
  onlyStaged,
  empty
} = {}, workspace) {
  if (ChangedFileItem === null) {
    ChangedFileItem = require('./items/changed-file-item')["default"];
  }

  return workspace.getPaneItems().filter(item => {
    const isFilePatchItem = item && item.getRealItem && item.getRealItem() instanceof ChangedFileItem;

    if (onlyStaged) {
      return isFilePatchItem && item.stagingStatus === 'staged';
    } else if (empty) {
      return isFilePatchItem ? item.isEmpty() : false;
    } else {
      return isFilePatchItem;
    }
  });
}

function destroyFilePatchPaneItems({
  onlyStaged
} = {}, workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    onlyStaged
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function destroyEmptyFilePatchPaneItems(workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    empty: true
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function extractCoAuthorsAndRawCommitMessage(commitMessage) {
  const messageLines = [];
  const coAuthors = [];

  for (const line of commitMessage.split(LINE_ENDING_REGEX)) {
    const match = line.match(CO_AUTHOR_REGEX);

    if (match) {
      // eslint-disable-next-line no-unused-vars
      const [_, name, email] = match;
      coAuthors.push({
        name,
        email
      });
    } else {
      messageLines.push(line);
    }
  }

  return {
    message: messageLines.join('\n'),
    coAuthors
  };
} // Atom API pane item manipulation


function createItem(node, componentHolder = null, uri = null, extra = {}) {
  const holder = componentHolder || new _refHolder["default"]();

  const override = _objectSpread2({
    getElement: () => node,
    getRealItem: () => holder.getOr(null),
    getRealItemPromise: () => holder.getPromise()
  }, extra);

  if (uri) {
    override.getURI = () => uri;
  }

  if (componentHolder) {
    return new Proxy(override, {
      get(target, name) {
        if (Reflect.has(target, name)) {
          return target[name];
        } // The {value: ...} wrapper prevents .map() from flattening a returned RefHolder.
        // If component[name] is a RefHolder, we want to return that RefHolder as-is.


        const {
          value
        } = holder.map(component => ({
          value: component[name]
        })).getOr({
          value: undefined
        });
        return value;
      },

      set(target, name, value) {
        return holder.map(component => {
          component[name] = value;
          return true;
        }).getOr(true);
      },

      has(target, name) {
        return holder.map(component => Reflect.has(component, name)).getOr(false) || Reflect.has(target, name);
      }

    });
  } else {
    return override;
  }
} // Set functions


function equalSets(left, right) {
  if (left.size !== right.size) {
    return false;
  }

  for (const each of left) {
    if (!right.has(each)) {
      return false;
    }
  }

  return true;
} // Constants


const NBSP_CHARACTER = '\u00a0';
exports.NBSP_CHARACTER = NBSP_CHARACTER;

function blankLabel() {
  return NBSP_CHARACTER;
}

const reactionTypeToEmoji = {
  THUMBS_UP: '👍',
  THUMBS_DOWN: '👎',
  LAUGH: '😆',
  HOORAY: '🎉',
  CONFUSED: '😕',
  HEART: '❤️',
  ROCKET: '🚀',
  EYES: '👀'
}; // Markdown

exports.reactionTypeToEmoji = reactionTypeToEmoji;
let marked = null;
let domPurify = null;

function renderMarkdown(md) {
  if (marked === null) {
    marked = require('marked');

    if (domPurify === null) {
      const createDOMPurify = require('dompurify');

      domPurify = createDOMPurify();
    }

    marked.setOptions({
      silent: true,
      sanitize: true,
      sanitizer: html => domPurify.sanitize(html)
    });
  }

  return marked(md);
}

const GHOST_USER = {
  login: 'ghost',
  avatarUrl: 'https://avatars1.githubusercontent.com/u/10137?v=4',
  url: 'https://github.com/ghost'
};
exports.GHOST_USER = GHOST_USER;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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