"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _helpers = require("./helpers");

var _workdirCache = _interopRequireDefault(require("./models/workdir-cache"));

var _workdirContext = _interopRequireDefault(require("./models/workdir-context"));

var _workdirContextPool = _interopRequireDefault(require("./models/workdir-context-pool"));

var _repository = _interopRequireDefault(require("./models/repository"));

var _styleCalculator = _interopRequireDefault(require("./models/style-calculator"));

var _githubLoginModel = _interopRequireDefault(require("./models/github-login-model"));

var _rootController = _interopRequireDefault(require("./controllers/root-controller"));

var _stubItem = _interopRequireDefault(require("./items/stub-item"));

var _switchboard = _interopRequireDefault(require("./switchboard"));

var _yardstick = _interopRequireDefault(require("./yardstick"));

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _contextMenuInterceptor = _interopRequireDefault(require("./context-menu-interceptor"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

var _getRepoPipelineManager = _interopRequireDefault(require("./get-repo-pipeline-manager"));

var _reporterProxy = require("./reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const defaultState = {
  newProject: true
};

class GithubPackage {
  constructor({
    workspace,
    project,
    commands,
    notificationManager,
    tooltips,
    styles,
    grammars,
    keymaps,
    config,
    deserializers,
    confirm,
    getLoadSettings,
    configDirPath,
    renderFn,
    loginModel
  }) {
    _defineProperty(this, "initialize", async projectPath => {
      await _fsExtra["default"].mkdirs(projectPath);
      const repository = this.contextPool.add(projectPath).getRepository();
      await repository.init();
      this.workdirCache.invalidate();

      if (!this.project.contains(projectPath)) {
        this.project.addPath(projectPath);
      }

      await this.refreshAtomGitRepository(projectPath);
      await this.scheduleActiveContextUpdate();
    });

    _defineProperty(this, "clone", async (remoteUrl, projectPath) => {
      const context = this.contextPool.getContext(projectPath);
      let repository;

      if (context.isPresent()) {
        repository = context.getRepository();
        await repository.clone(remoteUrl);
        repository.destroy();
      } else {
        repository = new _repository["default"](projectPath, null, {
          pipelineManager: this.pipelineManager
        });
        await repository.clone(remoteUrl);
      }

      this.workdirCache.invalidate();
      this.project.addPath(projectPath);
      await this.scheduleActiveContextUpdate();
    });

    (0, _helpers.autobind)(this, 'consumeStatusBar', 'createGitTimingsView', 'createIssueishPaneItemStub', 'createDockItemStub', 'createFilePatchControllerStub', 'destroyGitTabItem', 'destroyGithubTabItem', 'getRepositoryForWorkdir', 'scheduleActiveContextUpdate');
    this.workspace = workspace;
    this.project = project;
    this.commands = commands;
    this.deserializers = deserializers;
    this.notificationManager = notificationManager;
    this.tooltips = tooltips;
    this.config = config;
    this.styles = styles;
    this.grammars = grammars;
    this.keymaps = keymaps;
    this.configPath = _path["default"].join(configDirPath, 'github.cson');
    this.styleCalculator = new _styleCalculator["default"](this.styles, this.config);
    this.confirm = confirm;
    this.startOpen = false;
    this.activated = false;
    const criteria = {
      projectPathCount: this.project.getPaths().length,
      initPathCount: (getLoadSettings().initialPaths || []).length
    };
    this.pipelineManager = (0, _getRepoPipelineManager["default"])({
      confirm,
      notificationManager,
      workspace
    });
    this.activeContextQueue = new _asyncQueue["default"]();
    this.guessedContext = _workdirContext["default"].guess(criteria, this.pipelineManager);
    this.activeContext = this.guessedContext;
    this.workdirCache = new _workdirCache["default"]();
    this.contextPool = new _workdirContextPool["default"]({
      window,
      workspace,
      promptCallback: query => this.controller.openCredentialsDialog(query),
      pipelineManager: this.pipelineManager
    });
    this.switchboard = new _switchboard["default"]();
    this.loginModel = loginModel || new _githubLoginModel["default"]();

    this.renderFn = renderFn || ((component, node, callback) => {
      return _reactDom["default"].render(component, node, callback);
    }); // Handle events from all resident contexts.


    this.subscriptions = new _eventKit.CompositeDisposable(this.contextPool.onDidChangeWorkdirOrHead(context => {
      this.refreshAtomGitRepository(context.getWorkingDirectory());
    }), this.contextPool.onDidUpdateRepository(context => {
      this.switchboard.didUpdateRepository(context.getRepository());
    }), this.contextPool.onDidDestroyRepository(context => {
      if (context === this.activeContext) {
        this.setActiveContext(_workdirContext["default"].absent({
          pipelineManager: this.pipelineManager
        }));
      }
    }), _contextMenuInterceptor["default"]);
    this.setupYardstick();
  }

  setupYardstick() {
    const stagingSeries = ['stageLine', 'stageHunk', 'unstageLine', 'unstageHunk'];
    this.subscriptions.add( // Staging and unstaging operations
    this.switchboard.onDidBeginStageOperation(payload => {
      if (payload.stage && payload.line) {
        _yardstick["default"].begin('stageLine');
      } else if (payload.stage && payload.hunk) {
        _yardstick["default"].begin('stageHunk');
      } else if (payload.stage && payload.file) {
        _yardstick["default"].begin('stageFile');
      } else if (payload.stage && payload.mode) {
        _yardstick["default"].begin('stageMode');
      } else if (payload.stage && payload.symlink) {
        _yardstick["default"].begin('stageSymlink');
      } else if (payload.unstage && payload.line) {
        _yardstick["default"].begin('unstageLine');
      } else if (payload.unstage && payload.hunk) {
        _yardstick["default"].begin('unstageHunk');
      } else if (payload.unstage && payload.file) {
        _yardstick["default"].begin('unstageFile');
      } else if (payload.unstage && payload.mode) {
        _yardstick["default"].begin('unstageMode');
      } else if (payload.unstage && payload.symlink) {
        _yardstick["default"].begin('unstageSymlink');
      }
    }), this.switchboard.onDidUpdateRepository(() => {
      _yardstick["default"].mark(stagingSeries, 'update-repository');
    }), this.switchboard.onDidFinishRender(context => {
      if (context === 'RootController.showFilePatchForPath') {
        _yardstick["default"].finish(stagingSeries);
      }
    }), // Active context changes
    this.switchboard.onDidScheduleActiveContextUpdate(() => {
      _yardstick["default"].begin('activeContextChange');
    }), this.switchboard.onDidBeginActiveContextUpdate(() => {
      _yardstick["default"].mark('activeContextChange', 'queue-wait');
    }), this.switchboard.onDidFinishContextChangeRender(() => {
      _yardstick["default"].mark('activeContextChange', 'render');
    }), this.switchboard.onDidFinishActiveContextUpdate(() => {
      _yardstick["default"].finish('activeContextChange');
    }));
  }

  async activate(state = {}) {
    this.savedState = _objectSpread2({}, defaultState, {}, state);
    const firstRun = !(await (0, _helpers.fileExists)(this.configPath));
    const newProject = this.savedState.firstRun !== undefined ? this.savedState.firstRun : this.savedState.newProject;
    this.startOpen = firstRun || newProject;
    this.startRevealed = firstRun && !this.config.get('welcome.showOnStartup');

    if (firstRun) {
      await _fsExtra["default"].writeFile(this.configPath, '# Store non-visible GitHub package state.\n', {
        encoding: 'utf8'
      });
    }

    const hasSelectedFiles = event => {
      return !!event.target.closest('.github-FilePatchListView').querySelector('.is-selected');
    };

    this.subscriptions.add(this.project.onDidChangePaths(this.scheduleActiveContextUpdate), this.workspace.getCenter().onDidChangeActivePaneItem(this.scheduleActiveContextUpdate), this.styleCalculator.startWatching('github-package-styles', ['editor.fontSize', 'editor.fontFamily', 'editor.lineHeight', 'editor.tabLength'], config => `
          .github-HunkView-line {
            font-family: ${config.get('editor.fontFamily')};
            line-height: ${config.get('editor.lineHeight')};
            tab-size: ${config.get('editor.tabLength')}
          }
        `), atom.contextMenu.add({
      '.github-UnstagedChanges .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Discard Changes',
        command: 'github:discard-changes-in-selected-files',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-StagedChanges .github-FilePatchListView': [{
        label: 'Unstage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-MergeConflictPaths .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Ours',
        command: 'github:resolve-file-as-ours',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Theirs',
        command: 'github:resolve-file-as-theirs',
        shouldDisplay: hasSelectedFiles
      }]
    }));
    this.activated = true;
    this.scheduleActiveContextUpdate(this.savedState);
    this.rerender();
  }

  serialize() {
    const activeRepository = this.getActiveRepository();
    const activeRepositoryPath = activeRepository ? activeRepository.getWorkingDirectoryPath() : null;
    return {
      activeRepositoryPath,
      newProject: false
    };
  }

  rerender(callback) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    if (!this.activated) {
      return;
    }

    if (!this.element) {
      this.element = document.createElement('div');
      this.subscriptions.add(new _eventKit.Disposable(() => {
        _reactDom["default"].unmountComponentAtNode(this.element);

        delete this.element;
      }));
    }

    this.renderFn(_react["default"].createElement(_rootController["default"], {
      ref: c => {
        this.controller = c;
      },
      workspace: this.workspace,
      deserializers: this.deserializers,
      commands: this.commands,
      notificationManager: this.notificationManager,
      tooltips: this.tooltips,
      grammars: this.grammars,
      keymaps: this.keymaps,
      config: this.config,
      project: this.project,
      confirm: this.confirm,
      workdirContextPool: this.contextPool,
      loginModel: this.loginModel,
      repository: this.getActiveRepository(),
      resolutionProgress: this.getActiveResolutionProgress(),
      statusBar: this.statusBar,
      initialize: this.initialize,
      clone: this.clone,
      switchboard: this.switchboard,
      startOpen: this.startOpen,
      startRevealed: this.startRevealed,
      removeFilePatchItem: this.removeFilePatchItem
    }), this.element, callback);
  }

  async deactivate() {
    this.subscriptions.dispose();
    this.contextPool.clear();

    _workerManager["default"].reset(false);

    if (this.guessedContext) {
      this.guessedContext.destroy();
      this.guessedContext = null;
    }

    await _yardstick["default"].flush();
  }

  consumeStatusBar(statusBar) {
    this.statusBar = statusBar;
    this.rerender();
  }

  consumeReporter(reporter) {
    _reporterProxy.reporterProxy.setReporter(reporter);
  }

  createGitTimingsView() {
    return _stubItem["default"].create('git-timings-view', {
      title: 'GitHub Package Timings View'
    }, _gitTimingsView["default"].buildURI());
  }

  createIssueishPaneItemStub({
    uri,
    selectedTab
  }) {
    return _stubItem["default"].create('issueish-detail-item', {
      title: 'Issueish',
      initSelectedTab: selectedTab
    }, uri);
  }

  createDockItemStub({
    uri
  }) {
    let item;

    switch (uri) {
      // always return an empty stub
      // but only set it as the active item for a tab type
      // if it doesn't already exist
      case 'atom-github://dock-item/git':
        item = this.createGitStub(uri);
        this.gitTabStubItem = this.gitTabStubItem || item;
        break;

      case 'atom-github://dock-item/github':
        item = this.createGitHubStub(uri);
        this.githubTabStubItem = this.githubTabStubItem || item;
        break;

      default:
        throw new Error(`Invalid DockItem stub URI: ${uri}`);
    }

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createGitStub(uri) {
    return _stubItem["default"].create('git', {
      title: 'Git'
    }, uri);
  }

  createGitHubStub(uri) {
    return _stubItem["default"].create('github', {
      title: 'GitHub'
    }, uri);
  }

  createFilePatchControllerStub({
    uri
  } = {}) {
    const item = _stubItem["default"].create('git-file-patch-controller', {
      title: 'Diff'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitPreviewStub({
    uri
  }) {
    const item = _stubItem["default"].create('git-commit-preview', {
      title: 'Commit preview'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitDetailStub({
    uri
  }) {
    const item = _stubItem["default"].create('git-commit-detail', {
      title: 'Commit'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createReviewsStub({
    uri
  }) {
    const item = _stubItem["default"].create('github-reviews', {
      title: 'Reviews'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  destroyGitTabItem() {
    if (this.gitTabStubItem) {
      this.gitTabStubItem.destroy();
      this.gitTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  destroyGithubTabItem() {
    if (this.githubTabStubItem) {
      this.githubTabStubItem.destroy();
      this.githubTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  getRepositoryForWorkdir(projectPath) {
    const loadingGuessRepo = _repository["default"].loadingGuess({
      pipelineManager: this.pipelineManager
    });

    return this.guessedContext ? loadingGuessRepo : this.contextPool.getContext(projectPath).getRepository();
  }

  getActiveWorkdir() {
    return this.activeContext.getWorkingDirectory();
  }

  getActiveRepository() {
    return this.activeContext.getRepository();
  }

  getActiveResolutionProgress() {
    return this.activeContext.getResolutionProgress();
  }

  getContextPool() {
    return this.contextPool;
  }

  getSwitchboard() {
    return this.switchboard;
  }

  async scheduleActiveContextUpdate(savedState = {}) {
    this.switchboard.didScheduleActiveContextUpdate();
    await this.activeContextQueue.push(this.updateActiveContext.bind(this, savedState), {
      parallel: false
    });
  }
  /**
   * Derive the git working directory context that should be used for the package's git operations based on the current
   * state of the Atom workspace. In priority, this prefers:
   *
   * - A git working directory that contains the active pane item in the workspace's center.
   * - A git working directory corresponding to a single Project.
   * - When initially activating the package, the working directory that was active when the package was last
   *   serialized.
   * - The current context, unchanged, which may be a `NullWorkdirContext`.
   *
   * First updates the pool of resident contexts to match all git working directories that correspond to open
   * projects and pane items.
   */


  async getNextContext(savedState) {
    const workdirs = new Set((await Promise.all(this.project.getPaths().map(async projectPath => {
      const workdir = await this.workdirCache.find(projectPath);
      return workdir || projectPath;
    }))));

    const fromPaneItem = async maybeItem => {
      const itemPath = pathForPaneItem(maybeItem);

      if (!itemPath) {
        return {};
      }

      const itemWorkdir = await this.workdirCache.find(itemPath);

      if (itemWorkdir && !this.project.contains(itemPath)) {
        workdirs.add(itemWorkdir);
      }

      return {
        itemPath,
        itemWorkdir
      };
    };

    const active = await fromPaneItem(this.workspace.getCenter().getActivePaneItem());
    this.contextPool.set(workdirs, savedState);

    if (active.itemPath) {
      // Prefer an active item
      return this.contextPool.getContext(active.itemWorkdir || active.itemPath);
    }

    if (this.project.getPaths().length === 1) {
      // Single project
      const projectPath = this.project.getPaths()[0];
      const activeWorkingDir = await this.workdirCache.find(projectPath);
      return this.contextPool.getContext(activeWorkingDir || projectPath);
    }

    if (this.project.getPaths().length === 0 && !this.activeContext.getRepository().isUndetermined()) {
      // No projects. Revert to the absent context unless we've guessed that more projects are on the way.
      return _workdirContext["default"].absent({
        pipelineManager: this.pipelineManager
      });
    } // Restore models from saved state. Will return a NullWorkdirContext if this path is not presently
    // resident in the pool.


    const savedWorkingDir = savedState.activeRepositoryPath;

    if (savedWorkingDir) {
      return this.contextPool.getContext(savedWorkingDir);
    }

    return this.activeContext;
  }

  setActiveContext(nextActiveContext) {
    if (nextActiveContext !== this.activeContext) {
      if (this.activeContext === this.guessedContext) {
        this.guessedContext.destroy();
        this.guessedContext = null;
      }

      this.activeContext = nextActiveContext;
      this.rerender(() => {
        this.switchboard.didFinishContextChangeRender();
        this.switchboard.didFinishActiveContextUpdate();
      });
    } else {
      this.switchboard.didFinishActiveContextUpdate();
    }
  }

  async updateActiveContext(savedState = {}) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    this.switchboard.didBeginActiveContextUpdate();
    const nextActiveContext = await this.getNextContext(savedState);
    this.setActiveContext(nextActiveContext);
  }

  async refreshAtomGitRepository(workdir) {
    const directory = this.project.getDirectoryForProjectPath(workdir);

    if (!directory) {
      return;
    }

    const atomGitRepo = await this.project.repositoryForDirectory(directory);

    if (atomGitRepo) {
      await atomGitRepo.refreshStatus();
    }
  }

}

exports["default"] = GithubPackage;

function pathForPaneItem(paneItem) {
  if (!paneItem) {
    return null;
  } // Likely GitHub package provided pane item


  if (typeof paneItem.getWorkingDirectory === 'function') {
    return paneItem.getWorkingDirectory();
  } // TextEditor-like


  if (typeof paneItem.getPath === 'function') {
    return paneItem.getPath();
  } // Oh well


  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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