"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.LargeRepoError = exports.GitError = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _child_process = _interopRequireDefault(require("child_process"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _util = _interopRequireDefault(require("util"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _dugite = require("dugite");

var _whatTheDiff = require("what-the-diff");

var _whatTheStatus = require("what-the-status");

var _gitPromptServer = _interopRequireDefault(require("./git-prompt-server"));

var _gitTempDir = _interopRequireDefault(require("./git-temp-dir"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _reporterProxy = require("./reporter-proxy");

var _helpers = require("./helpers");

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _file = _interopRequireDefault(require("./models/patch/file"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread2(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;
let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

}

exports.GitError = GitError;

class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

} // ignored for the purposes of usage metrics tracking because they're noisy


exports.LargeRepoError = LargeRepoError;
const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];
const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);
/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^\\\\/]*)[\\\\/] captures non-slash characters before first slash
 */

const EXPAND_TILDE_REGEX = new RegExp('^~([^\\\\/]*)[\\\\/]');

class GitShellOutStrategy {
  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;

    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os["default"].cpus().length);
      this.commandQueue = new _asyncQueue["default"]({
        parallelism
      });
    }

    this.prompt = options.prompt || (query => Promise.reject());

    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }
  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */


  setPromptCallback(prompt) {
    this.prompt = prompt;
  } // Execute a command and read the output using the embedded Git environment


  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const {
      stdin,
      useGitPromptServer,
      useGpgWrapper,
      useGpgAtomPrompt,
      writeOperation
    } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');
    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;

    const timingMarker = _gitTimingsView["default"].generateMarker(`git ${args.join(' ')}`);

    timingMarker.mark('queued');
    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise((resolve, reject) => {
        _child_process["default"].exec('git --exec-path', (error, stdout, stderr) => {
          /* istanbul ignore if */
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }

    const execPath = await execPathPromise;
    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;
      const pathParts = [];

      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }

      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _objectSpread2({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path["default"].delimiter)
      });

      const gitTempDir = new _gitTempDir["default"]();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer["default"](gitTempDir);
        await gitPromptServer.start(this.prompt);
        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getSocketPath());
        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy'); // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.

        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';
        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else if (process.env.GIT_SSH_COMMAND) {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        } else {
          env.GIT_SSH = process.env.GIT_SSH;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }
      /* istanbul ignore if */


      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = {
        env
      };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }
      /* istanbul ignore if */


      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }

      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({
            args,
            opts
          });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }

        const {
          promise,
          cancel
        } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;

        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({
            handlerPid
          }) => {
            expectCancel = true;
            await cancel(); // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.

            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                /* istanbul ignore if */
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const {
          stdout,
          stderr,
          exitCode,
          signal,
          timing
        } = await promise["catch"](err => {
          if (err.signal) {
            return {
              signal: err.signal
            };
          }

          reject(err);
          return {};
        });

        if (timing) {
          const {
            execTime,
            spawnTime,
            ipcTime
          } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }

        timingMarker.finalize();
        /* istanbul ignore if */

        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }

        if (gitPromptServer) {
          gitPromptServer.terminate();
        }

        subscriptions.dispose();
        /* istanbul ignore if */

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;

            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }

            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }

            summary += 'stdout:';

            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }

            summary += 'stderr:';

            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';
            console.groupCollapsed(`git:${formattedArgs}`);

            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }

            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util["default"].inspect(args, {
              breakLength: Infinity
            }));

            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }

            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }

        resolve(stdout);
      });
    }, {
      parallel: !writeOperation
    });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _objectSpread2({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _objectSpread2({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager["default"].getInstance().isReady()) {
      marker && marker.mark('nexttick');
      let childPid;

      options.processCallback = child => {
        childPid = child.pid;
        /* istanbul ignore next */

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);

      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          /* istanbul ignore if */
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              /* istanbul ignore if */
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager["default"].getInstance();

      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra["default"].stat(this.workingDir); // fails if folder doesn't exist

      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path["default"].join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      return (0, _helpers.toNativePathSep)(dotGitDir);
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }
  /**
   * Staging/Unstaging files and patches and committing
   */


  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');

    if (!templatePath) {
      return null;
    }

    const homeDir = _os["default"].homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path["default"].join(_path["default"].dirname(homeDir), user) : homeDir}/`;
    });
    templatePath = (0, _helpers.toNativePathSep)(templatePath);

    if (!_path["default"].isAbsolute(templatePath)) {
      templatePath = _path["default"].join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }

    return await _fsExtra["default"].readFile(templatePath, {
      encoding: 'utf8'
    });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({
        args,
        opts
      }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], {
      writeOperation: true
    });
  }

  applyPatch(patch, {
    index
  } = {}) {
    const args = ['apply', '-'];

    if (index) {
      args.splice(1, 0, '--cached');
    }

    return this.exec(args, {
      stdin: patch,
      writeOperation: true
    });
  }

  async commit(rawMessage, {
    allowEmpty,
    amend,
    coAuthors,
    verbatim
  } = {}) {
    const args = ['commit'];
    let msg; // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.

    if (amend && rawMessage.length === 0) {
      const {
        unbornRef,
        messageBody,
        messageSubject
      } = await this.getHeadCommit();

      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    } // if commit template is used, strip commented lines from commit
    // to be consistent with command line git.


    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      // respecting the comment character from user settings or fall back to # as default.
      // https://git-scm.com/docs/git-config#git-config-corecommentChar
      let commentChar = await this.getConfig('core.commentChar');

      if (!commentChar) {
        commentChar = '#';
      }

      msg = msg.split('\n').filter(line => !line.startsWith(commentChar)).join('\n');
    } // Determine the cleanup mode.


    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    } // add co-author commit trailers if necessary


    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }

    if (allowEmpty) {
      args.push('--allow-empty');
    }

    return this.gpgExec(args, {
      writeOperation: true
    });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    }); // Ensure that message ends with newline for git-interpret trailers to work

    const msg = `${message.trim()}\n`;
    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }
  /**
   * File Status and Diffs
   */


  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);

    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }

      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];

    if (options.staged) {
      args.push('--staged');
    }

    if (options.target) {
      args.push(options.target);
    }

    const output = await this.exec(args);
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };
    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });

    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }

    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);

    if (output.trim() === '') {
      return [];
    }

    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, {
    staged,
    baseCommit
  } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];

    if (staged) {
      args.push('--staged');
    }

    if (baseCommit) {
      args.push(baseCommit);
    }

    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);
    let rawDiffs = [];

    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];

        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }

        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path["default"].join(this.workingDir, filePath);

      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra["default"].readFile(absPath, {
        encoding: 'utf8'
      });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;

      if (executable) {
        mode = _file["default"].modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file["default"].modes.SYMLINK;
        realpath = await _fsExtra["default"].realpath(absPath);
      } else {
        mode = _file["default"].modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }

    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }

    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);

    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }

      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }

    return diffs;
  }
  /**
   * Miscellaneous getters
   */


  async getCommit(ref) {
    const [commit] = await this.getCommits({
      max: 1,
      ref,
      includeUnborn: true
    });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({
      max: 1,
      ref: 'HEAD',
      includeUnborn: true
    });
    return headCommit;
  }

  async getCommits(options = {}) {
    const {
      max,
      ref,
      includeUnborn,
      includePatch
    } = _objectSpread2({
      max: 1,
      ref: 'HEAD',
      includeUnborn: false,
      includePatch: false
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %an = author full name
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body


    const args = ['log', '--pretty=format:%H%x00%ae%x00%an%x00%at%x00%s%x00%b%x00', '--no-abbrev-commit', '--no-prefix', '--no-ext-diff', '--no-renames', '-z', '-n', max, ref];

    if (includePatch) {
      args.push('--patch', '-m', '--first-parent');
    }

    const output = await this.exec(args.concat('--'))["catch"](err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{
        sha: '',
        message: '',
        unbornRef: true
      }] : [];
    }

    const fields = output.trim().split('\0');
    const commits = [];

    for (let i = 0; i < fields.length; i += 7) {
      const body = fields[i + 5].trim();
      let patch = [];

      if (includePatch) {
        const diffs = fields[i + 6];
        patch = (0, _whatTheDiff.parse)(diffs.trim());
      }

      const {
        message: messageBody,
        coAuthors
      } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);
      commits.push({
        sha: fields[i] && fields[i].trim(),
        authorEmail: fields[i + 1] && fields[i + 1].trim(),
        authorName: fields[i + 2] && fields[i + 2].trim(),
        authorDate: parseInt(fields[i + 3], 10),
        messageSubject: fields[i + 4],
        messageBody,
        coAuthors,
        unbornRef: false,
        patch
      });
    }

    return commits;
  }

  async getAuthors(options = {}) {
    const {
      max,
      ref
    } = _objectSpread2({
      max: 1,
      ref: 'HEAD'
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).


    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);
      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });
        acc[ae] = an;
        acc[ce] = cn;
        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers) {
    const args = ['interpret-trailers'];

    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }

    return this.exec(args, {
      stdin: commitMessage
    });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }
  /**
   * Merge
   */


  merge(branchName) {
    return this.gpgExec(['merge', branchName], {
      writeOperation: true
    });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'MERGE_HEAD'))["catch"](() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], {
      writeOperation: true
    });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }
  /**
   * Rebase
   */


  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path["default"].join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }
  /**
   * Remote interactions
   */


  clone(remoteUrl, options = {}) {
    const args = ['clone'];

    if (options.noLocal) {
      args.push('--no-local');
    }

    if (options.bare) {
      args.push('--bare');
    }

    if (options.recursive) {
      args.push('--recursive');
    }

    args.push(remoteUrl, this.workingDir);
    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];

    if (options.ffOnly) {
      args.push('--ff-only');
    }

    return this.gpgExec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];

    if (options.setUpstream) {
      args.push('--set-upstream');
    }

    if (options.force) {
      args.push('--force');
    }

    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }
  /**
   * Undo Operations
   */


  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];

    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }

    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }
  /**
   * Branches
   */


  checkout(branchName, options = {}) {
    const args = ['checkout'];

    if (options.createNew) {
      args.push('-b');
    }

    args.push(branchName);

    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }

      args.push(options.startPoint);
    }

    return this.exec(args, {
      writeOperation: true
    });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');
    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');
      const branch = {
        name,
        sha,
        head: head === '*'
      };

      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }

      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }

      return branch;
    });
  }

  async getBranchesWithCommit(sha, option = {}) {
    const args = ['branch', '--format=%(refname)', '--contains', sha];

    if (option.showLocal && option.showRemote) {
      args.splice(1, 0, '--all');
    } else if (option.showRemote) {
      args.splice(1, 0, '--remotes');
    }

    if (option.pattern) {
      args.push(option.pattern);
    }

    return (await this.exec(args)).trim().split(_helpers.LINE_ENDING_REGEX);
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }

    const args = ['checkout'];

    if (revision) {
      args.push(revision);
    }

    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), {
      writeOperation: true
    });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, {
    local
  } = {}) {
    let output;

    try {
      let args = ['config'];

      if (local || atom.inSpecMode()) {
        args.push('--local');
      }

      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1) {
        // No matching config found
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, {
    replaceAll
  } = {}) {
    let args = ['config'];

    if (replaceAll) {
      args.push('--replace-all');
    }

    args = args.concat(option, value);
    return this.exec(args, {
      writeOperation: true
    });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], {
      writeOperation: true
    });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], {
      local: true
    });

    if (output) {
      output = output.trim();

      if (!output.length) {
        return [];
      }

      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({
    filePath,
    stdin
  } = {}) {
    let output;

    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], {
          writeOperation: true
        })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], {
        stdin,
        writeOperation: true
      })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }

    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra["default"].writeFile(absFilePath, output, {
      encoding: 'utf8'
    });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;

    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    } // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.


    const resolvedResultPath = _path["default"].resolve(this.workingDir, resultPath);

    await _fsExtra["default"].writeFile(resolvedResultPath, output, {
      encoding: 'utf8'
    });
    return {
      filePath: oursPath,
      resultPath,
      conflict
    };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;

    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }

    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }

    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }

    return this.exec(['update-index', '--index-info'], {
      stdin: indexInfo,
      writeOperation: true
    });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);

    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path["default"].join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path["default"].join(this.workingDir, filePath));

      if (symlink) {
        return _file["default"].modes.SYMLINK;
      } else if (executable) {
        return _file["default"].modes.EXECUTABLE;
      } else {
        return _file["default"].modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }

}

exports["default"] = GitShellOutStrategy;

_defineProperty(GitShellOutStrategy, "defaultExecArgs", {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
});

function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];

  if (contents) {
    let noNewLine;
    let lines;

    if (mode === _file["default"].modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }

    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }

    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }

  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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