"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNullActionPipelineManager = getNullActionPipelineManager;
exports["default"] = exports.ActionPipeline = void 0;

function partial(fn, ...args) {
  return function wrapped() {
    return fn(...args);
  };
}

class ActionPipeline {
  constructor(actionKey) {
    this.actionKey = actionKey;
    this.middleware = [];
  }

  run(fn, repository, ...args) {
    const pipelineFn = this.middleware.map(middleware => middleware.fn).reduceRight((acc, nextFn) => partial(nextFn, acc, repository, ...args), partial(fn, ...args));
    return pipelineFn(...args);
  }

  addMiddleware(name, fn) {
    if (!name) {
      throw new Error('Middleware must be registered with a unique middleware name');
    }

    if (this.middleware.find(middleware => middleware.name === name)) {
      throw new Error(`A middleware with the name ${name} is already registered`);
    }

    this.middleware.push({
      name,
      fn
    });
  }

}

exports.ActionPipeline = ActionPipeline;

class ActionPipelineManager {
  constructor({
    actionNames
  } = {
    actionNames: []
  }) {
    this.pipelines = new Map();
    this.actionKeys = {};
    actionNames.forEach(actionName => {
      const actionKey = Symbol(actionName);
      this.actionKeys[actionName] = actionKey;
      this.pipelines.set(actionKey, new ActionPipeline(actionKey));
    });
  }

  getPipeline(actionKey) {
    const pipeline = this.pipelines.get(actionKey);

    if (actionKey && pipeline) {
      return pipeline;
    } else {
      throw new Error(`${actionKey} is not a known action`);
    }
  }

}

exports["default"] = ActionPipelineManager;

class NullActionPipeline extends ActionPipeline {
  run(fn, repository, ...args) {
    return fn(...args);
  }

  addMiddleware() {
    throw new Error('Cannot add middleware to a null pipeline');
  }

}

const nullPipeline = new NullActionPipeline();

class NullActionPipelineManager extends ActionPipelineManager {
  getPipeline() {
    return nullPipeline;
  }

}

const nullActionPipelineManager = new NullActionPipelineManager();

function getNullActionPipelineManager() {
  return nullActionPipelineManager;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImFjdGlvbi1waXBlbGluZS5qcyJdLCJuYW1lcyI6WyJwYXJ0aWFsIiwiZm4iLCJhcmdzIiwid3JhcHBlZCIsIkFjdGlvblBpcGVsaW5lIiwiY29uc3RydWN0b3IiLCJhY3Rpb25LZXkiLCJtaWRkbGV3YXJlIiwicnVuIiwicmVwb3NpdG9yeSIsInBpcGVsaW5lRm4iLCJtYXAiLCJyZWR1Y2VSaWdodCIsImFjYyIsIm5leHRGbiIsImFkZE1pZGRsZXdhcmUiLCJuYW1lIiwiRXJyb3IiLCJmaW5kIiwicHVzaCIsIkFjdGlvblBpcGVsaW5lTWFuYWdlciIsImFjdGlvbk5hbWVzIiwicGlwZWxpbmVzIiwiTWFwIiwiYWN0aW9uS2V5cyIsImZvckVhY2giLCJhY3Rpb25OYW1lIiwiU3ltYm9sIiwic2V0IiwiZ2V0UGlwZWxpbmUiLCJwaXBlbGluZSIsImdldCIsIk51bGxBY3Rpb25QaXBlbGluZSIsIm51bGxQaXBlbGluZSIsIk51bGxBY3Rpb25QaXBlbGluZU1hbmFnZXIiLCJudWxsQWN0aW9uUGlwZWxpbmVNYW5hZ2VyIiwiZ2V0TnVsbEFjdGlvblBpcGVsaW5lTWFuYWdlciJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7QUFBQSxTQUFTQSxPQUFULENBQWlCQyxFQUFqQixFQUFxQixHQUFHQyxJQUF4QixFQUE4QjtBQUM1QixTQUFPLFNBQVNDLE9BQVQsR0FBbUI7QUFDeEIsV0FBT0YsRUFBRSxDQUFDLEdBQUdDLElBQUosQ0FBVDtBQUNELEdBRkQ7QUFHRDs7QUFFTSxNQUFNRSxjQUFOLENBQXFCO0FBQzFCQyxFQUFBQSxXQUFXLENBQUNDLFNBQUQsRUFBWTtBQUNyQixTQUFLQSxTQUFMLEdBQWlCQSxTQUFqQjtBQUNBLFNBQUtDLFVBQUwsR0FBa0IsRUFBbEI7QUFDRDs7QUFFREMsRUFBQUEsR0FBRyxDQUFDUCxFQUFELEVBQUtRLFVBQUwsRUFBaUIsR0FBR1AsSUFBcEIsRUFBMEI7QUFDM0IsVUFBTVEsVUFBVSxHQUFHLEtBQUtILFVBQUwsQ0FDaEJJLEdBRGdCLENBQ1pKLFVBQVUsSUFBSUEsVUFBVSxDQUFDTixFQURiLEVBRWhCVyxXQUZnQixDQUVKLENBQUNDLEdBQUQsRUFBTUMsTUFBTixLQUFpQmQsT0FBTyxDQUFDYyxNQUFELEVBQVNELEdBQVQsRUFBY0osVUFBZCxFQUEwQixHQUFHUCxJQUE3QixDQUZwQixFQUV3REYsT0FBTyxDQUFDQyxFQUFELEVBQUssR0FBR0MsSUFBUixDQUYvRCxDQUFuQjtBQUdBLFdBQU9RLFVBQVUsQ0FBQyxHQUFHUixJQUFKLENBQWpCO0FBQ0Q7O0FBRURhLEVBQUFBLGFBQWEsQ0FBQ0MsSUFBRCxFQUFPZixFQUFQLEVBQVc7QUFDdEIsUUFBSSxDQUFDZSxJQUFMLEVBQVc7QUFDVCxZQUFNLElBQUlDLEtBQUosQ0FBVSw2REFBVixDQUFOO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLVixVQUFMLENBQWdCVyxJQUFoQixDQUFxQlgsVUFBVSxJQUFJQSxVQUFVLENBQUNTLElBQVgsS0FBb0JBLElBQXZELENBQUosRUFBa0U7QUFDaEUsWUFBTSxJQUFJQyxLQUFKLENBQVcsOEJBQTZCRCxJQUFLLHdCQUE3QyxDQUFOO0FBQ0Q7O0FBRUQsU0FBS1QsVUFBTCxDQUFnQlksSUFBaEIsQ0FBcUI7QUFBQ0gsTUFBQUEsSUFBRDtBQUFPZixNQUFBQTtBQUFQLEtBQXJCO0FBQ0Q7O0FBdkJ5Qjs7OztBQTBCYixNQUFNbUIscUJBQU4sQ0FBNEI7QUFDekNmLEVBQUFBLFdBQVcsQ0FBQztBQUFDZ0IsSUFBQUE7QUFBRCxNQUFnQjtBQUFDQSxJQUFBQSxXQUFXLEVBQUU7QUFBZCxHQUFqQixFQUFvQztBQUM3QyxTQUFLQyxTQUFMLEdBQWlCLElBQUlDLEdBQUosRUFBakI7QUFDQSxTQUFLQyxVQUFMLEdBQWtCLEVBQWxCO0FBQ0FILElBQUFBLFdBQVcsQ0FBQ0ksT0FBWixDQUFvQkMsVUFBVSxJQUFJO0FBQ2hDLFlBQU1wQixTQUFTLEdBQUdxQixNQUFNLENBQUNELFVBQUQsQ0FBeEI7QUFDQSxXQUFLRixVQUFMLENBQWdCRSxVQUFoQixJQUE4QnBCLFNBQTlCO0FBQ0EsV0FBS2dCLFNBQUwsQ0FBZU0sR0FBZixDQUFtQnRCLFNBQW5CLEVBQThCLElBQUlGLGNBQUosQ0FBbUJFLFNBQW5CLENBQTlCO0FBQ0QsS0FKRDtBQUtEOztBQUVEdUIsRUFBQUEsV0FBVyxDQUFDdkIsU0FBRCxFQUFZO0FBQ3JCLFVBQU13QixRQUFRLEdBQUcsS0FBS1IsU0FBTCxDQUFlUyxHQUFmLENBQW1CekIsU0FBbkIsQ0FBakI7O0FBQ0EsUUFBSUEsU0FBUyxJQUFJd0IsUUFBakIsRUFBMkI7QUFDekIsYUFBT0EsUUFBUDtBQUNELEtBRkQsTUFFTztBQUNMLFlBQU0sSUFBSWIsS0FBSixDQUFXLEdBQUVYLFNBQVUsd0JBQXZCLENBQU47QUFDRDtBQUNGOztBQWxCd0M7Ozs7QUFzQjNDLE1BQU0wQixrQkFBTixTQUFpQzVCLGNBQWpDLENBQWdEO0FBQzlDSSxFQUFBQSxHQUFHLENBQUNQLEVBQUQsRUFBS1EsVUFBTCxFQUFpQixHQUFHUCxJQUFwQixFQUEwQjtBQUMzQixXQUFPRCxFQUFFLENBQUMsR0FBR0MsSUFBSixDQUFUO0FBQ0Q7O0FBRURhLEVBQUFBLGFBQWEsR0FBRztBQUNkLFVBQU0sSUFBSUUsS0FBSixDQUFVLDBDQUFWLENBQU47QUFDRDs7QUFQNkM7O0FBVWhELE1BQU1nQixZQUFZLEdBQUcsSUFBSUQsa0JBQUosRUFBckI7O0FBRUEsTUFBTUUseUJBQU4sU0FBd0NkLHFCQUF4QyxDQUE4RDtBQUM1RFMsRUFBQUEsV0FBVyxHQUFHO0FBQ1osV0FBT0ksWUFBUDtBQUNEOztBQUgyRDs7QUFNOUQsTUFBTUUseUJBQXlCLEdBQUcsSUFBSUQseUJBQUosRUFBbEM7O0FBQ08sU0FBU0UsNEJBQVQsR0FBd0M7QUFDN0MsU0FBT0QseUJBQVA7QUFDRCIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJmdW5jdGlvbiBwYXJ0aWFsKGZuLCAuLi5hcmdzKSB7XG4gIHJldHVybiBmdW5jdGlvbiB3cmFwcGVkKCkge1xuICAgIHJldHVybiBmbiguLi5hcmdzKTtcbiAgfTtcbn1cblxuZXhwb3J0IGNsYXNzIEFjdGlvblBpcGVsaW5lIHtcbiAgY29uc3RydWN0b3IoYWN0aW9uS2V5KSB7XG4gICAgdGhpcy5hY3Rpb25LZXkgPSBhY3Rpb25LZXk7XG4gICAgdGhpcy5taWRkbGV3YXJlID0gW107XG4gIH1cblxuICBydW4oZm4sIHJlcG9zaXRvcnksIC4uLmFyZ3MpIHtcbiAgICBjb25zdCBwaXBlbGluZUZuID0gdGhpcy5taWRkbGV3YXJlXG4gICAgICAubWFwKG1pZGRsZXdhcmUgPT4gbWlkZGxld2FyZS5mbilcbiAgICAgIC5yZWR1Y2VSaWdodCgoYWNjLCBuZXh0Rm4pID0+IHBhcnRpYWwobmV4dEZuLCBhY2MsIHJlcG9zaXRvcnksIC4uLmFyZ3MpLCBwYXJ0aWFsKGZuLCAuLi5hcmdzKSk7XG4gICAgcmV0dXJuIHBpcGVsaW5lRm4oLi4uYXJncyk7XG4gIH1cblxuICBhZGRNaWRkbGV3YXJlKG5hbWUsIGZuKSB7XG4gICAgaWYgKCFuYW1lKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pZGRsZXdhcmUgbXVzdCBiZSByZWdpc3RlcmVkIHdpdGggYSB1bmlxdWUgbWlkZGxld2FyZSBuYW1lJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMubWlkZGxld2FyZS5maW5kKG1pZGRsZXdhcmUgPT4gbWlkZGxld2FyZS5uYW1lID09PSBuYW1lKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBIG1pZGRsZXdhcmUgd2l0aCB0aGUgbmFtZSAke25hbWV9IGlzIGFscmVhZHkgcmVnaXN0ZXJlZGApO1xuICAgIH1cblxuICAgIHRoaXMubWlkZGxld2FyZS5wdXNoKHtuYW1lLCBmbn0pO1xuICB9XG59XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEFjdGlvblBpcGVsaW5lTWFuYWdlciB7XG4gIGNvbnN0cnVjdG9yKHthY3Rpb25OYW1lc30gPSB7YWN0aW9uTmFtZXM6IFtdfSkge1xuICAgIHRoaXMucGlwZWxpbmVzID0gbmV3IE1hcCgpO1xuICAgIHRoaXMuYWN0aW9uS2V5cyA9IHt9O1xuICAgIGFjdGlvbk5hbWVzLmZvckVhY2goYWN0aW9uTmFtZSA9PiB7XG4gICAgICBjb25zdCBhY3Rpb25LZXkgPSBTeW1ib2woYWN0aW9uTmFtZSk7XG4gICAgICB0aGlzLmFjdGlvbktleXNbYWN0aW9uTmFtZV0gPSBhY3Rpb25LZXk7XG4gICAgICB0aGlzLnBpcGVsaW5lcy5zZXQoYWN0aW9uS2V5LCBuZXcgQWN0aW9uUGlwZWxpbmUoYWN0aW9uS2V5KSk7XG4gICAgfSk7XG4gIH1cblxuICBnZXRQaXBlbGluZShhY3Rpb25LZXkpIHtcbiAgICBjb25zdCBwaXBlbGluZSA9IHRoaXMucGlwZWxpbmVzLmdldChhY3Rpb25LZXkpO1xuICAgIGlmIChhY3Rpb25LZXkgJiYgcGlwZWxpbmUpIHtcbiAgICAgIHJldHVybiBwaXBlbGluZTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAke2FjdGlvbktleX0gaXMgbm90IGEga25vd24gYWN0aW9uYCk7XG4gICAgfVxuICB9XG59XG5cblxuY2xhc3MgTnVsbEFjdGlvblBpcGVsaW5lIGV4dGVuZHMgQWN0aW9uUGlwZWxpbmUge1xuICBydW4oZm4sIHJlcG9zaXRvcnksIC4uLmFyZ3MpIHtcbiAgICByZXR1cm4gZm4oLi4uYXJncyk7XG4gIH1cblxuICBhZGRNaWRkbGV3YXJlKCkge1xuICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFkZCBtaWRkbGV3YXJlIHRvIGEgbnVsbCBwaXBlbGluZScpO1xuICB9XG59XG5cbmNvbnN0IG51bGxQaXBlbGluZSA9IG5ldyBOdWxsQWN0aW9uUGlwZWxpbmUoKTtcblxuY2xhc3MgTnVsbEFjdGlvblBpcGVsaW5lTWFuYWdlciBleHRlbmRzIEFjdGlvblBpcGVsaW5lTWFuYWdlciB7XG4gIGdldFBpcGVsaW5lKCkge1xuICAgIHJldHVybiBudWxsUGlwZWxpbmU7XG4gIH1cbn1cblxuY29uc3QgbnVsbEFjdGlvblBpcGVsaW5lTWFuYWdlciA9IG5ldyBOdWxsQWN0aW9uUGlwZWxpbmVNYW5hZ2VyKCk7XG5leHBvcnQgZnVuY3Rpb24gZ2V0TnVsbEFjdGlvblBpcGVsaW5lTWFuYWdlcigpIHtcbiAgcmV0dXJuIG51bGxBY3Rpb25QaXBlbGluZU1hbmFnZXI7XG59XG4iXX0=