"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsService = void 0;

var _operators = require("rxjs/operators");

var _ = require("./");

var _mappings = require("./mappings");

var _migrations = require("./migrations");

var _retry_call_cluster = require("../elasticsearch/retry_call_cluster");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SavedObjectsService {
  constructor(coreContext) {
    this.coreContext = coreContext;

    _defineProperty(this, "migrator", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "clientProvider", void 0);

    this.logger = coreContext.logger.get('savedobjects-service');
  }

  async setup(coreSetup, migrationsRetryDelay) {
    this.logger.debug('Setting up SavedObjects service');
    const {
      savedObjectSchemas,
      savedObjectMappings,
      savedObjectMigrations,
      savedObjectValidations
    } = await coreSetup.legacy.uiExports;
    const adminClient = await coreSetup.elasticsearch.adminClient$.pipe((0, _operators.first)()).toPromise();
    const kibanaConfig = await this.coreContext.configService.atPath('kibana').pipe((0, _operators.first)()).toPromise();
    const savedObjectsConfig = await this.coreContext.configService.atPath('migrations').pipe((0, _operators.first)()).toPromise();
    const migrator = this.migrator = new _migrations.KibanaMigrator({
      savedObjectSchemas,
      savedObjectMappings,
      savedObjectMigrations,
      savedObjectValidations,
      logger: this.coreContext.logger.get('migrations'),
      kibanaVersion: this.coreContext.env.packageInfo.version,
      config: coreSetup.legacy.pluginExtendedConfig,
      savedObjectsConfig,
      kibanaConfig,
      callCluster: (0, _retry_call_cluster.migrationsRetryCallCluster)(adminClient.callAsInternalUser, this.coreContext.logger.get('migrations'), migrationsRetryDelay)
    });
    const mappings = this.migrator.getActiveMappings();
    const allTypes = Object.keys((0, _mappings.getRootPropertiesObjects)(mappings));
    const schema = new _.SavedObjectsSchema(savedObjectSchemas);
    const serializer = new _.SavedObjectsSerializer(schema);
    const visibleTypes = allTypes.filter(type => !schema.isHiddenType(type));
    this.clientProvider = new _.SavedObjectsClientProvider({
      defaultClientFactory({
        request
      }) {
        const repository = new _.SavedObjectsRepository({
          index: kibanaConfig.index,
          config: coreSetup.legacy.pluginExtendedConfig,
          migrator,
          mappings,
          schema,
          serializer,
          allowedTypes: visibleTypes,
          callCluster: (0, _retry_call_cluster.retryCallCluster)(adminClient.asScoped(request).callAsCurrentUser)
        });
        return new _.SavedObjectsClient(repository);
      }

    });
    return {
      clientProvider: this.clientProvider
    };
  }

  async start(core) {
    if (!this.clientProvider) {
      throw new Error('#setup() needs to be run first');
    }

    this.logger.debug('Starting SavedObjects service');
    /**
     * Note: We want to ensure that migrations have completed before
     * continuing with further Core startup steps that might use SavedObjects
     * such as running the legacy server, legacy plugins and allowing incoming
     * HTTP requests.
     *
     * However, our build system optimize step and some tests depend on the
     * HTTP server running without an Elasticsearch server being available.
     * So, when the `migrations.skip` is true, we skip migrations altogether.
     */

    const cliArgs = this.coreContext.env.cliArgs;
    const savedObjectsConfig = await this.coreContext.configService.atPath('migrations').pipe((0, _operators.first)()).toPromise();
    const skipMigrations = cliArgs.optimize || savedObjectsConfig.skip;
    await this.migrator.runMigrations(skipMigrations);
    return {
      migrator: this.migrator,
      clientProvider: this.clientProvider
    };
  }

  async stop() {}

}

exports.SavedObjectsService = SavedObjectsService;