"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@angular-devkit/core");
const node_1 = require("@angular-devkit/core/node");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
/**
 * @deprecated
 */
class ProjectNotFoundException extends core_1.BaseException {
    constructor(projectName) {
        super(`Project '${projectName}' could not be found in Workspace.`);
    }
}
exports.ProjectNotFoundException = ProjectNotFoundException;
/**
 * @deprecated
 */
class TargetNotFoundException extends core_1.BaseException {
    constructor(projectName, targetName) {
        super(`Target '${targetName}' could not be found in project '${projectName}'.`);
    }
}
exports.TargetNotFoundException = TargetNotFoundException;
/**
 * @deprecated
 */
class ConfigurationNotFoundException extends core_1.BaseException {
    constructor(projectName, configurationName) {
        super(`Configuration '${configurationName}' could not be found in project '${projectName}'.`);
    }
}
exports.ConfigurationNotFoundException = ConfigurationNotFoundException;
// TODO: break this exception apart into more granular ones.
/**
 * @deprecated
 */
class BuilderCannotBeResolvedException extends core_1.BaseException {
    constructor(builder) {
        super(`Builder '${builder}' cannot be resolved.`);
    }
}
exports.BuilderCannotBeResolvedException = BuilderCannotBeResolvedException;
/**
 * @deprecated
 */
class ArchitectNotYetLoadedException extends core_1.BaseException {
    constructor() { super(`Architect needs to be loaded before Architect is used.`); }
}
exports.ArchitectNotYetLoadedException = ArchitectNotYetLoadedException;
/**
 * @deprecated
 */
class BuilderNotFoundException extends core_1.BaseException {
    constructor(builder) {
        super(`Builder ${builder} could not be found.`);
    }
}
exports.BuilderNotFoundException = BuilderNotFoundException;
/**
 * @deprecated
 */
class Architect {
    constructor(_workspace) {
        this._workspace = _workspace;
        this._targetsSchemaPath = core_1.join(core_1.normalize(__dirname), 'targets-schema.json');
        this._buildersSchemaPath = core_1.join(core_1.normalize(__dirname), 'builders-schema.json');
        this._architectSchemasLoaded = false;
        this._targetMapMap = new Map();
        this._builderPathsMap = new Map();
        this._builderDescriptionMap = new Map();
        this._builderConstructorMap = new Map();
    }
    loadArchitect() {
        if (this._architectSchemasLoaded) {
            return rxjs_1.of(this);
        }
        else {
            return rxjs_1.forkJoin(this._loadJsonFile(this._targetsSchemaPath), this._loadJsonFile(this._buildersSchemaPath)).pipe(operators_1.concatMap(([targetsSchema, buildersSchema]) => {
                this._targetsSchema = targetsSchema;
                this._buildersSchema = buildersSchema;
                this._architectSchemasLoaded = true;
                // Validate and cache all project target maps.
                return rxjs_1.forkJoin(...this._workspace.listProjectNames().map(projectName => {
                    const unvalidatedTargetMap = this._workspace.getProjectTargets(projectName);
                    return this._workspace.validateAgainstSchema(unvalidatedTargetMap, this._targetsSchema).pipe(operators_1.tap(targetMap => this._targetMapMap.set(projectName, targetMap)));
                }));
            }), operators_1.map(() => this));
        }
    }
    listProjectTargets(projectName) {
        return Object.keys(this._getProjectTargetMap(projectName));
    }
    _getProjectTargetMap(projectName) {
        if (!this._targetMapMap.has(projectName)) {
            throw new ProjectNotFoundException(projectName);
        }
        return this._targetMapMap.get(projectName);
    }
    _getProjectTarget(projectName, targetName) {
        const targetMap = this._getProjectTargetMap(projectName);
        const target = targetMap[targetName];
        if (!target) {
            throw new TargetNotFoundException(projectName, targetName);
        }
        return target;
    }
    getBuilderConfiguration(targetSpec) {
        const { project: projectName, target: targetName, configuration: configurationName, overrides, } = targetSpec;
        const project = this._workspace.getProject(projectName);
        const target = this._getProjectTarget(projectName, targetName);
        const options = target.options;
        let configuration = {};
        if (configurationName) {
            if (!target.configurations) {
                throw new ConfigurationNotFoundException(projectName, configurationName);
            }
            configuration = target.configurations[configurationName];
            if (!configuration) {
                throw new ConfigurationNotFoundException(projectName, configurationName);
            }
        }
        const builderConfiguration = {
            root: project.root,
            sourceRoot: project.sourceRoot,
            projectType: project.projectType,
            builder: target.builder,
            options: Object.assign({}, options, configuration, overrides),
        };
        return builderConfiguration;
    }
    run(builderConfig, partialContext = {}) {
        const context = Object.assign({ logger: new core_1.logging.NullLogger(), architect: this, host: this._workspace.host, workspace: this._workspace }, partialContext);
        let builderDescription;
        return this.getBuilderDescription(builderConfig).pipe(operators_1.tap(description => builderDescription = description), operators_1.concatMap(() => this.validateBuilderOptions(builderConfig, builderDescription)), operators_1.tap(validatedBuilderConfig => builderConfig = validatedBuilderConfig), operators_1.map(() => this.getBuilder(builderDescription, context)), operators_1.concatMap(builder => builder.run(builderConfig)));
    }
    getBuilderDescription(builderConfig) {
        // Check cache for this builder description.
        if (this._builderDescriptionMap.has(builderConfig.builder)) {
            return rxjs_1.of(this._builderDescriptionMap.get(builderConfig.builder));
        }
        return new rxjs_1.Observable((obs) => {
            // TODO: this probably needs to be more like NodeModulesEngineHost.
            const basedir = core_1.getSystemPath(this._workspace.root);
            const [pkg, builderName] = builderConfig.builder.split(':');
            const pkgJsonPath = node_1.resolve(pkg, { basedir, resolvePackageJson: true, checkLocal: true });
            let buildersJsonPath;
            let builderPaths;
            // Read the `builders` entry of package.json.
            return this._loadJsonFile(core_1.normalize(pkgJsonPath)).pipe(operators_1.concatMap((pkgJson) => {
                const pkgJsonBuildersentry = pkgJson['builders'];
                if (!pkgJsonBuildersentry) {
                    return rxjs_1.throwError(new BuilderCannotBeResolvedException(builderConfig.builder));
                }
                buildersJsonPath = core_1.join(core_1.dirname(core_1.normalize(pkgJsonPath)), pkgJsonBuildersentry);
                return this._loadJsonFile(buildersJsonPath);
            }), 
            // Validate builders json.
            operators_1.concatMap((builderPathsMap) => this._workspace.validateAgainstSchema(builderPathsMap, this._buildersSchema)), operators_1.concatMap((builderPathsMap) => {
                builderPaths = builderPathsMap.builders[builderName];
                if (!builderPaths) {
                    return rxjs_1.throwError(new BuilderCannotBeResolvedException(builderConfig.builder));
                }
                // Resolve paths in the builder paths.
                const builderJsonDir = core_1.dirname(buildersJsonPath);
                builderPaths.schema = core_1.join(builderJsonDir, builderPaths.schema);
                builderPaths.class = core_1.join(builderJsonDir, builderPaths.class);
                // Save the builder paths so that we can lazily load the builder.
                this._builderPathsMap.set(builderConfig.builder, builderPaths);
                // Load the schema.
                return this._loadJsonFile(builderPaths.schema);
            }), operators_1.map(builderSchema => {
                const builderDescription = {
                    name: builderConfig.builder,
                    schema: builderSchema,
                    description: builderPaths.description,
                };
                // Save to cache before returning.
                this._builderDescriptionMap.set(builderDescription.name, builderDescription);
                return builderDescription;
            })).subscribe(obs);
        });
    }
    validateBuilderOptions(builderConfig, builderDescription) {
        return this._workspace.validateAgainstSchema(builderConfig.options, builderDescription.schema).pipe(operators_1.map(validatedOptions => {
            builderConfig.options = validatedOptions;
            return builderConfig;
        }));
    }
    getBuilder(builderDescription, context) {
        const name = builderDescription.name;
        let builderConstructor;
        // Check cache for this builder.
        if (this._builderConstructorMap.has(name)) {
            builderConstructor = this._builderConstructorMap.get(name);
        }
        else {
            if (!this._builderPathsMap.has(name)) {
                throw new BuilderNotFoundException(name);
            }
            const builderPaths = this._builderPathsMap.get(name);
            // TODO: support more than the default export, maybe via builder#import-name.
            const builderModule = require(core_1.getSystemPath(builderPaths.class));
            builderConstructor = builderModule['default'];
            // Save builder to cache before returning.
            this._builderConstructorMap.set(builderDescription.name, builderConstructor);
        }
        const builder = new builderConstructor(context);
        return builder;
    }
    _loadJsonFile(path) {
        return this._workspace.host.read(core_1.normalize(path)).pipe(operators_1.map(buffer => core_1.virtualFs.fileBufferToString(buffer)), operators_1.map(str => core_1.parseJson(str, core_1.JsonParseMode.Loose)));
    }
}
exports.Architect = Architect;
//# sourceMappingURL=data:application/json;base64,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