(function() {
  var CompositeDisposable, Disposable, Emitter, NotificationsLog, NotificationsLogItem, ref, typeIcons;

  ref = require('atom'), Emitter = ref.Emitter, CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable;

  NotificationsLogItem = require('./notifications-log-item');

  typeIcons = {
    fatal: 'bug',
    error: 'flame',
    warning: 'alert',
    info: 'info',
    success: 'check'
  };

  module.exports = NotificationsLog = (function() {
    NotificationsLog.prototype.subscriptions = null;

    NotificationsLog.prototype.logItems = [];

    NotificationsLog.prototype.typesHidden = {
      fatal: false,
      error: false,
      warning: false,
      info: false,
      success: false
    };

    function NotificationsLog(duplicateTimeDelay, typesHidden) {
      this.duplicateTimeDelay = duplicateTimeDelay;
      if (typesHidden == null) {
        typesHidden = null;
      }
      if (typesHidden != null) {
        this.typesHidden = typesHidden;
      }
      this.emitter = new Emitter;
      this.subscriptions = new CompositeDisposable;
      this.subscriptions.add(atom.notifications.onDidClearNotifications((function(_this) {
        return function() {
          return _this.clearLogItems();
        };
      })(this)));
      this.render();
      this.subscriptions.add(new Disposable((function(_this) {
        return function() {
          return _this.clearLogItems();
        };
      })(this)));
    }

    NotificationsLog.prototype.render = function() {
      var button, header, i, icon, lastNotification, len, notification, ref1, timeSpan, type;
      this.element = document.createElement('div');
      this.element.classList.add('notifications-log');
      header = document.createElement('header');
      this.element.appendChild(header);
      this.list = document.createElement('ul');
      this.list.classList.add('notifications-log-items');
      this.element.appendChild(this.list);
      for (type in typeIcons) {
        icon = typeIcons[type];
        button = document.createElement('button');
        button.classList.add('notification-type', 'btn', 'icon', "icon-" + icon, type);
        button.classList.toggle('show-type', !this.typesHidden[type]);
        this.list.classList.toggle("hide-" + type, this.typesHidden[type]);
        button.dataset.type = type;
        button.addEventListener('click', (function(_this) {
          return function(e) {
            return _this.toggleType(e.target.dataset.type);
          };
        })(this));
        this.subscriptions.add(atom.tooltips.add(button, {
          title: "Toggle " + type + " notifications"
        }));
        header.appendChild(button);
      }
      button = document.createElement('button');
      button.classList.add('notifications-clear-log', 'btn', 'icon', 'icon-trashcan');
      button.addEventListener('click', function(e) {
        return atom.commands.dispatch(atom.views.getView(atom.workspace), "notifications:clear-log");
      });
      this.subscriptions.add(atom.tooltips.add(button, {
        title: "Clear notifications"
      }));
      header.appendChild(button);
      lastNotification = null;
      ref1 = atom.notifications.getNotifications();
      for (i = 0, len = ref1.length; i < len; i++) {
        notification = ref1[i];
        if (lastNotification != null) {
          timeSpan = notification.getTimestamp() - lastNotification.getTimestamp();
          if (!(timeSpan < this.duplicateTimeDelay && notification.isEqual(lastNotification))) {
            this.addNotification(notification);
          }
        } else {
          this.addNotification(notification);
        }
        lastNotification = notification;
      }
      return this.subscriptions.add(new Disposable((function(_this) {
        return function() {
          return _this.element.remove();
        };
      })(this)));
    };

    NotificationsLog.prototype.destroy = function() {
      this.subscriptions.dispose();
      return this.emitter.emit('did-destroy');
    };

    NotificationsLog.prototype.getElement = function() {
      return this.element;
    };

    NotificationsLog.prototype.getURI = function() {
      return 'atom://notifications/log';
    };

    NotificationsLog.prototype.getTitle = function() {
      return 'Log';
    };

    NotificationsLog.prototype.getLongTitle = function() {
      return 'Notifications Log';
    };

    NotificationsLog.prototype.getIconName = function() {
      return 'alert';
    };

    NotificationsLog.prototype.getDefaultLocation = function() {
      return 'bottom';
    };

    NotificationsLog.prototype.getAllowedLocations = function() {
      return ['left', 'right', 'bottom'];
    };

    NotificationsLog.prototype.serialize = function() {
      return {
        typesHidden: this.typesHidden,
        deserializer: 'notifications/NotificationsLog'
      };
    };

    NotificationsLog.prototype.toggleType = function(type, force) {
      var button, hide;
      button = this.element.querySelector(".notification-type." + type);
      hide = !button.classList.toggle('show-type', force);
      this.list.classList.toggle("hide-" + type, hide);
      return this.typesHidden[type] = hide;
    };

    NotificationsLog.prototype.addNotification = function(notification) {
      var logItem;
      logItem = new NotificationsLogItem(notification);
      logItem.onClick((function(_this) {
        return function() {
          return _this.emitter.emit('item-clicked', notification);
        };
      })(this));
      this.logItems.push(logItem);
      return this.list.insertBefore(logItem.getElement(), this.list.firstChild);
    };

    NotificationsLog.prototype.onItemClick = function(callback) {
      return this.emitter.on('item-clicked', callback);
    };

    NotificationsLog.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    NotificationsLog.prototype.clearLogItems = function() {
      var i, len, logItem, ref1;
      ref1 = this.logItems;
      for (i = 0, len = ref1.length; i < len; i++) {
        logItem = ref1[i];
        logItem.destroy();
      }
      return this.logItems = [];
    };

    return NotificationsLog;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
