"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

// Measure elapsed durations from specific beginning points.
// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
const MAXIMUM_MARKS = 100; // Flush all non-active DurationSets to disk each time that this many marks have been accumulated.

const PERSIST_INTERVAL = 1000; // A sequence of durations measured from a fixed beginning point.

class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }

  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }

    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({
        eventName,
        duration
      });
    }

    this.markCount++;

    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }

  finish() {
    this.mark('finish');

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }

  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }

  getCount() {
    return this.marks.length;
  }

}

let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();

function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });

  if (!anyActive) {
    return false;
  }

  const mask = new RegExp(atom.config.get('github.performanceMask'));

  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }

  return true;
}

const yardstick = {
  async save() {
    const destDir = atom.config.get('github.performanceToDirectory');

    if (destDir === '' || destDir === undefined || destDir === null) {
      return;
    }

    const fileName = _path["default"].join(destDir, `performance-${Date.now()}.json`);

    await new Promise((resolve, reject) => {
      _fsExtra["default"].ensureDir(destDir, err => err ? reject(err) : resolve());
    });
    const payload = JSON.stringify(durationSets.map(set => set.serialize()));
    await _fsExtra["default"].writeFile(fileName, payload, {
      encoding: 'utf8'
    });

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
    }

    durationSets = [];
  },

  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }

    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },

  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }

      return;
    }

    if (!shouldCapture(seriesName, eventName)) {
      return;
    }

    const ds = activeSets.get(seriesName);

    if (ds === undefined) {
      return;
    }

    ds.mark(eventName);
  },

  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }

      return;
    }

    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }

    const ds = activeSets.get(seriesName);

    if (ds === undefined) {
      return;
    }

    ds.finish();
    durationSets.push(ds);
    activeSets["delete"](seriesName);
    totalMarkCount += ds.getCount();

    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },

  async flush() {
    durationSets.push(...activeSets.values());
    activeSets.clear();
    await this.save();
  }

};
var _default = yardstick;
exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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