/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkImageSimpleSource.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkImageSimpleSource
 * @brief   Create a 3-D image with simple-to-compute pixel values.
 *
 * vtkImageSimpleSource produces images with pixel values that can
 * be obtained with relatively low computational load.
 * The output point data contains 1-3 arrays:
 *  - "X": A scalar double array labeled "X" is always generated. Its values
 *    are set to the X coordinate at each point.
 *  - "DistanceSquared": A scalar double array labeled "DistanceSquared" is
 *    generated by
 *    default, but can be omitted by setting EnableDistanceSquaredData false.
 *    The values are set to the Euclidean distance of each point from
 *    the origin (X*X + Y*Y + Z*Z).
 *  - "Swirl": A vector double array labeled "Swirl" is generated by default,
 *    but can be omitted by setting EnableSwirlData false. The values at
 *    each point are set to the <Y, Z, X> coordinates.
*/

#ifndef vtkImageSimpleSource_h
#define vtkImageSimpleSource_h

#include "vtkPVVTKExtensionsDefaultModule.h" //needed for exports
#include "vtkThreadedImageAlgorithm.h"

class VTKPVVTKEXTENSIONSDEFAULT_EXPORT vtkImageSimpleSource : public vtkThreadedImageAlgorithm
{
public:
  static vtkImageSimpleSource* New();
  vtkTypeMacro(vtkImageSimpleSource, vtkThreadedImageAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Set/Get the extent of the whole output image.
   */
  vtkGetVector6Macro(WholeExtent, int);
  void SetWholeExtent(int xMinx, int xMax, int yMin, int yMax, int zMin, int zMax);

  /**
   * Enable/Disable DistanceSquared data array.
   */
  vtkGetMacro(EnableDistanceSquaredData, bool);
  vtkSetMacro(EnableDistanceSquaredData, bool);
  vtkBooleanMacro(EnableDistanceSquaredData, bool);

  /**
   * Enable/Disable Swirl data array.
   */
  vtkGetMacro(EnableSwirlData, bool);
  vtkSetMacro(EnableSwirlData, bool);
  vtkBooleanMacro(EnableSwirlData, bool);

protected:
  vtkImageSimpleSource();
  ~vtkImageSimpleSource() override {}

  bool EnableDistanceSquaredData;
  bool EnableSwirlData;
  int FirstIndex[3];
  int WholeExtent[6];

  void PrepareImageData(vtkInformationVector** inputVector, vtkInformationVector* outputVector,
    vtkImageData*** inDataObjects = nullptr, vtkImageData** outDataObjects = nullptr) override;

  int RequestInformation(vtkInformation* request, vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  void ThreadedRequestData(vtkInformation* request, vtkInformationVector** inputVector,
    vtkInformationVector* outputVector, vtkImageData*** inData, vtkImageData** outData,
    int extent[6], int threadId) override;

private:
  vtkImageSimpleSource(const vtkImageSimpleSource&) = delete;
  void operator=(const vtkImageSimpleSource&) = delete;
};

#endif
