"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiSettingsClient = void 0;

var _lodash = require("lodash");

var _boom = _interopRequireDefault(require("boom"));

var _create_or_upgrade_saved_config = require("./create_or_upgrade_saved_config");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class UiSettingsClient {
  constructor(options) {
    _defineProperty(this, "type", void 0);

    _defineProperty(this, "id", void 0);

    _defineProperty(this, "buildNum", void 0);

    _defineProperty(this, "savedObjectsClient", void 0);

    _defineProperty(this, "overrides", void 0);

    _defineProperty(this, "defaults", void 0);

    _defineProperty(this, "log", void 0);

    const {
      type,
      id,
      buildNum,
      savedObjectsClient,
      log,
      defaults = {},
      overrides = {}
    } = options;
    this.type = type;
    this.id = id;
    this.buildNum = buildNum;
    this.savedObjectsClient = savedObjectsClient;
    this.defaults = defaults;
    this.overrides = overrides;
    this.log = log;
  }

  getDefaults() {
    return this.defaults;
  }

  async get(key) {
    const all = await this.getAll();
    return all[key];
  }

  async getAll() {
    const raw = await this.getRaw();
    return Object.keys(raw).reduce((all, key) => {
      const item = raw[key];
      all[key] = 'userValue' in item ? item.userValue : item.value;
      return all;
    }, {});
  } // NOTE: should be a private method


  async getRaw() {
    const userProvided = await this.getUserProvided();
    return (0, _lodash.defaultsDeep)(userProvided, this.defaults);
  }

  async getUserProvided(options = {}) {
    const userProvided = {}; // write the userValue for each key stored in the saved object that is not overridden

    for (const [key, userValue] of Object.entries((await this.read(options)))) {
      if (userValue !== null && !this.isOverridden(key)) {
        userProvided[key] = {
          userValue
        };
      }
    } // write all overridden keys, dropping the userValue is override is null and
    // adding keys for overrides that are not in saved object


    for (const [key, userValue] of Object.entries(this.overrides)) {
      userProvided[key] = userValue === null ? {
        isOverridden: true
      } : {
        isOverridden: true,
        userValue
      };
    }

    return userProvided;
  }

  async setMany(changes) {
    await this.write({
      changes
    });
  }

  async set(key, value) {
    await this.setMany({
      [key]: value
    });
  }

  async remove(key) {
    await this.set(key, null);
  }

  async removeMany(keys) {
    const changes = {};
    keys.forEach(key => {
      changes[key] = null;
    });
    await this.setMany(changes);
  }

  isOverridden(key) {
    return this.overrides.hasOwnProperty(key);
  } // NOTE: should be private method


  assertUpdateAllowed(key) {
    if (this.isOverridden(key)) {
      throw _boom.default.badRequest(`Unable to update "${key}" because it is overridden`);
    }
  }

  async write({
    changes,
    autoCreateOrUpgradeIfMissing = true
  }) {
    for (const key of Object.keys(changes)) {
      this.assertUpdateAllowed(key);
    }

    try {
      await this.savedObjectsClient.update(this.type, this.id, changes);
    } catch (error) {
      const {
        isNotFoundError
      } = this.savedObjectsClient.errors;

      if (!isNotFoundError(error) || !autoCreateOrUpgradeIfMissing) {
        throw error;
      }

      await (0, _create_or_upgrade_saved_config.createOrUpgradeSavedConfig)({
        savedObjectsClient: this.savedObjectsClient,
        version: this.id,
        buildNum: this.buildNum,
        log: this.log
      });
      await this.write({
        changes,
        autoCreateOrUpgradeIfMissing: false
      });
    }
  }

  async read({
    ignore401Errors = false,
    autoCreateOrUpgradeIfMissing = true
  } = {}) {
    const {
      isConflictError,
      isNotFoundError,
      isForbiddenError,
      isNotAuthorizedError
    } = this.savedObjectsClient.errors;

    try {
      const resp = await this.savedObjectsClient.get(this.type, this.id);
      return resp.attributes;
    } catch (error) {
      if (isNotFoundError(error) && autoCreateOrUpgradeIfMissing) {
        const failedUpgradeAttributes = await (0, _create_or_upgrade_saved_config.createOrUpgradeSavedConfig)({
          savedObjectsClient: this.savedObjectsClient,
          version: this.id,
          buildNum: this.buildNum,
          log: this.log,

          onWriteError(writeError, attributes) {
            if (isConflictError(writeError)) {
              // trigger `!failedUpgradeAttributes` check below, since another
              // request caused the uiSettings object to be created so we can
              // just re-read
              return;
            }

            if (isNotAuthorizedError(writeError) || isForbiddenError(writeError)) {
              return attributes;
            }

            throw writeError;
          }

        });

        if (!failedUpgradeAttributes) {
          return await this.read({
            ignore401Errors,
            autoCreateOrUpgradeIfMissing: false
          });
        }

        return failedUpgradeAttributes;
      }

      if (this.isIgnorableError(error, ignore401Errors)) {
        return {};
      }

      throw error;
    }
  }

  isIgnorableError(error, ignore401Errors) {
    const {
      isForbiddenError,
      isEsUnavailableError,
      isNotAuthorizedError
    } = this.savedObjectsClient.errors;
    return isForbiddenError(error) || isEsUnavailableError(error) || ignore401Errors && isNotAuthorizedError(error);
  }

}

exports.UiSettingsClient = UiSettingsClient;