"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Server = void 0;

var _operators = require("rxjs/operators");

var _config = require("./config");

var _elasticsearch = require("./elasticsearch");

var _http = require("./http");

var _legacy = require("./legacy");

var _ui_settings = require("./ui_settings");

var _plugins = require("./plugins");

var _saved_objects = require("../server/saved_objects");

var _logging = require("./logging");

var _dev = require("./dev");

var _kibana_config = require("./kibana_config");

var _saved_objects2 = require("./saved_objects");

var _utils = require("../utils/");

var _context = require("./context");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const coreId = Symbol('core');

class Server {
  constructor(config$, env, logger) {
    this.config$ = config$;
    this.env = env;
    this.logger = logger;

    _defineProperty(this, "configService", void 0);

    _defineProperty(this, "context", void 0);

    _defineProperty(this, "elasticsearch", void 0);

    _defineProperty(this, "http", void 0);

    _defineProperty(this, "legacy", void 0);

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "plugins", void 0);

    _defineProperty(this, "savedObjects", void 0);

    _defineProperty(this, "uiSettings", void 0);

    this.log = this.logger.get('server');
    this.configService = new _config.ConfigService(config$, env, logger);
    const core = {
      coreId,
      configService: this.configService,
      env,
      logger
    };
    this.context = new _context.ContextService(core);
    this.http = new _http.HttpService(core);
    this.plugins = new _plugins.PluginsService(core);
    this.legacy = new _legacy.LegacyService(core);
    this.elasticsearch = new _elasticsearch.ElasticsearchService(core);
    this.savedObjects = new _saved_objects.SavedObjectsService(core);
    this.uiSettings = new _ui_settings.UiSettingsService(core);
  }

  async setup() {
    this.log.debug('setting up server'); // Discover any plugins before continuing. This allows other systems to utilize the plugin dependency graph.

    const pluginDependencies = await this.plugins.discover();
    const contextServiceSetup = this.context.setup({
      // We inject a fake "legacy plugin" with no dependencies so that legacy plugins can register context providers
      // that will only be available to other legacy plugins and will not leak into New Platform plugins.
      pluginDependencies: new Map([...pluginDependencies, [this.legacy.legacyId, []]])
    });
    const httpSetup = await this.http.setup({
      context: contextServiceSetup
    });
    this.registerDefaultRoute(httpSetup);
    const elasticsearchServiceSetup = await this.elasticsearch.setup({
      http: httpSetup
    });
    const uiSettingsSetup = await this.uiSettings.setup({
      http: httpSetup
    });
    const coreSetup = {
      context: contextServiceSetup,
      elasticsearch: elasticsearchServiceSetup,
      http: httpSetup,
      uiSettings: uiSettingsSetup
    };
    const pluginsSetup = await this.plugins.setup(coreSetup);
    const legacySetup = await this.legacy.setup({
      core: { ...coreSetup,
        plugins: pluginsSetup
      },
      plugins: (0, _utils.mapToObject)(pluginsSetup.contracts)
    });
    const savedObjectsSetup = await this.savedObjects.setup({
      elasticsearch: elasticsearchServiceSetup,
      legacy: legacySetup
    });
    this.registerCoreContext({ ...coreSetup,
      savedObjects: savedObjectsSetup
    });
    return coreSetup;
  }

  async start() {
    this.log.debug('starting server');
    const pluginsStart = await this.plugins.start({});
    const savedObjectsStart = await this.savedObjects.start({});
    const coreStart = {
      savedObjects: savedObjectsStart,
      plugins: pluginsStart
    };
    await this.legacy.start({
      core: coreStart,
      plugins: (0, _utils.mapToObject)(pluginsStart.contracts)
    });
    await this.http.start();
    return coreStart;
  }

  async stop() {
    this.log.debug('stopping server');
    await this.legacy.stop();
    await this.plugins.stop();
    await this.savedObjects.stop();
    await this.elasticsearch.stop();
    await this.http.stop();
  }

  registerDefaultRoute(httpSetup) {
    const router = httpSetup.createRouter('/core');
    router.get({
      path: '/',
      validate: false
    }, async (context, req, res) => res.ok({
      body: {
        version: '0.0.1'
      }
    }));
  }

  registerCoreContext(coreSetup) {
    coreSetup.http.registerRouteHandlerContext(coreId, 'core', async (context, req) => {
      const adminClient = await coreSetup.elasticsearch.adminClient$.pipe((0, _operators.take)(1)).toPromise();
      const dataClient = await coreSetup.elasticsearch.dataClient$.pipe((0, _operators.take)(1)).toPromise();
      return {
        savedObjects: {
          // Note: the client provider doesn't support new ES clients
          // emitted from adminClient$
          client: coreSetup.savedObjects.clientProvider.getClient(req)
        },
        elasticsearch: {
          adminClient: adminClient.asScoped(req),
          dataClient: dataClient.asScoped(req)
        }
      };
    });
  }

  async setupConfigSchemas() {
    const schemas = [[_elasticsearch.config.path, _elasticsearch.config.schema], [_logging.config.path, _logging.config.schema], [_http.config.path, _http.config.schema], [_plugins.config.path, _plugins.config.schema], [_dev.config.path, _dev.config.schema], [_kibana_config.config.path, _kibana_config.config.schema], [_saved_objects2.config.path, _saved_objects2.config.schema], [_ui_settings.config.path, _ui_settings.config.schema]];

    for (const [path, schema] of schemas) {
      await this.configService.setSchema(path, schema);
    }
  }

}

exports.Server = Server;